"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Annotations = void 0;
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const constructs_1 = require("constructs");
const DEPRECATIONS_SYMBOL = Symbol.for('@aws-cdk/core.deprecations');
/**
 * Includes API for attaching annotations such as warning messages to constructs.
 */
class Annotations {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Returns the annotations API for a construct scope.
     * @param scope The scope
     */
    static of(scope) {
        return new Annotations(scope);
    }
    /**
     * Adds a warning metadata entry to this construct.
     *
     * The CLI will display the warning when an app is synthesized, or fail if run
     * in --strict mode.
     *
     * @param message The warning message.
     */
    addWarning(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.WARN, message);
    }
    /**
     * Adds an info metadata entry to this construct.
     *
     * The CLI will display the info message when apps are synthesized.
     *
     * @param message The info message.
     */
    addInfo(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.INFO, message);
    }
    /**
     * Adds an { "error": <message> } metadata entry to this construct.
     * The toolkit will fail synthesis when errors are reported.
     * @param message The error message.
     */
    addError(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.ERROR, message);
    }
    /**
     * Adds a deprecation warning for a specific API.
     *
     * Deprecations will be added only once per construct as a warning and will be
     * deduplicated based on the `api`.
     *
     * If the environment variable `CDK_BLOCK_DEPRECATIONS` is set, this method
     * will throw an error instead with the deprecation message.
     *
     * @param api The API being deprecated in the format `module.Class.property`
     * (e.g. `@aws-cdk/core.Construct.node`).
     * @param message The deprecation message to display, with information about
     * alternatives.
     */
    addDeprecation(api, message) {
        const text = `The API ${api} is deprecated: ${message}. This API will be removed in the next major release`;
        // throw if CDK_BLOCK_DEPRECATIONS is set
        if (process.env.CDK_BLOCK_DEPRECATIONS) {
            throw new Error(`${constructs_1.Node.of(this.scope).path}: ${text}`);
        }
        // de-dup based on api key
        const set = this.deprecationsReported;
        if (set.has(api)) {
            return;
        }
        this.addWarning(text);
        set.add(api);
    }
    /**
     * Adds a message metadata entry to the construct node, to be displayed by the CDK CLI.
     * @param level The message level
     * @param message The message itself
     */
    addMessage(level, message) {
        constructs_1.Node.of(this.scope).addMetadata(level, message);
    }
    /**
     * Returns the set of deprecations reported on this construct.
     */
    get deprecationsReported() {
        let set = this.scope[DEPRECATIONS_SYMBOL];
        if (!set) {
            set = new Set();
            Object.defineProperty(this.scope, DEPRECATIONS_SYMBOL, { value: set });
        }
        return set;
    }
}
exports.Annotations = Annotations;
//# sourceMappingURL=data:application/json;base64,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