"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnCodeDeployBlueGreenHook = exports.CfnTrafficRoutingType = void 0;
const cfn_hook_1 = require("./cfn-hook");
/**
 * The possible types of traffic shifting for the blue-green deployment configuration.
 * The type of the {@link CfnTrafficRoutingConfig.type} property.
 */
var CfnTrafficRoutingType;
(function (CfnTrafficRoutingType) {
    /**
     * Switch from blue to green at once.
     */
    CfnTrafficRoutingType["ALL_AT_ONCE"] = "AllAtOnce";
    /**
     * Specifies a configuration that shifts traffic from blue to green in two increments.
     */
    CfnTrafficRoutingType["TIME_BASED_CANARY"] = "TimeBasedCanary";
    /**
     * Specifies a configuration that shifts traffic from blue to green in equal increments,
     * with an equal number of minutes between each increment.
     */
    CfnTrafficRoutingType["TIME_BASED_LINEAR"] = "TimeBasedLinear";
})(CfnTrafficRoutingType = exports.CfnTrafficRoutingType || (exports.CfnTrafficRoutingType = {}));
/**
 * A CloudFormation Hook for CodeDeploy blue-green ECS deployments.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/blue-green.html#blue-green-template-reference
 */
class CfnCodeDeployBlueGreenHook extends cfn_hook_1.CfnHook {
    /**
     * Creates a new CodeDeploy blue-green ECS Hook.
     *
     * @param scope the scope to create the hook in (usually the containing Stack object)
     * @param id the identifier of the construct - will be used to generate the logical ID of the Hook
     * @param props the properties of the Hook
     */
    constructor(scope, id, props) {
        super(scope, id, {
            type: 'AWS::CodeDeploy::BlueGreen',
        });
        this._serviceRole = props.serviceRole;
        this._applications = props.applications;
        this._trafficRoutingConfig = props.trafficRoutingConfig;
        this._additionalOptions = props.additionalOptions;
        this._lifecycleEventHooks = props.lifecycleEventHooks;
    }
    /**
     * A factory method that creates a new instance of this class from an object
     * containing the CloudFormation properties of this resource.
     * Used in the @aws-cdk/cloudformation-include module.
     *
     * @internal
     */
    static _fromCloudFormation(scope, id, hookAttributes, options) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
        hookAttributes = hookAttributes || {};
        const hookProperties = options.parser.parseValue(hookAttributes.Properties);
        return new CfnCodeDeployBlueGreenHook(scope, id, {
            serviceRole: hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.ServiceRole,
            applications: (_a = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.Applications) === null || _a === void 0 ? void 0 : _a.map(applicationFromCloudFormation),
            trafficRoutingConfig: {
                type: (_b = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _b === void 0 ? void 0 : _b.Type,
                timeBasedCanary: {
                    stepPercentage: (_d = (_c = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _c === void 0 ? void 0 : _c.TimeBasedCanary) === null || _d === void 0 ? void 0 : _d.StepPercentage,
                    bakeTimeMins: (_f = (_e = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _e === void 0 ? void 0 : _e.TimeBasedCanary) === null || _f === void 0 ? void 0 : _f.BakeTimeMins,
                },
                timeBasedLinear: {
                    stepPercentage: (_h = (_g = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _g === void 0 ? void 0 : _g.TimeBasedLinear) === null || _h === void 0 ? void 0 : _h.StepPercentage,
                    bakeTimeMins: (_k = (_j = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _j === void 0 ? void 0 : _j.TimeBasedLinear) === null || _k === void 0 ? void 0 : _k.BakeTimeMins,
                },
            },
            additionalOptions: {
                terminationWaitTimeInMinutes: (_l = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.AdditionalOptions) === null || _l === void 0 ? void 0 : _l.TerminationWaitTimeInMinutes,
            },
            lifecycleEventHooks: {
                beforeInstall: (_m = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _m === void 0 ? void 0 : _m.BeforeInstall,
                afterInstall: (_o = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _o === void 0 ? void 0 : _o.AfterInstall,
                afterAllowTestTraffic: (_p = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _p === void 0 ? void 0 : _p.AfterAllowTestTraffic,
                beforeAllowTraffic: (_q = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _q === void 0 ? void 0 : _q.BeforeAllowTraffic,
                afterAllowTraffic: (_r = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _r === void 0 ? void 0 : _r.AfterAllowTraffic,
            },
        });
        function applicationFromCloudFormation(app) {
            var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w;
            const target = findResource((_a = app === null || app === void 0 ? void 0 : app.Target) === null || _a === void 0 ? void 0 : _a.LogicalID);
            const taskDefinitions = (_c = (_b = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _b === void 0 ? void 0 : _b.TaskDefinitions) === null || _c === void 0 ? void 0 : _c.map((td) => findResource(td));
            const taskSets = (_e = (_d = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _d === void 0 ? void 0 : _d.TaskSets) === null || _e === void 0 ? void 0 : _e.map((ts) => findResource(ts));
            const prodTrafficRoute = findResource((_h = (_g = (_f = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _f === void 0 ? void 0 : _f.TrafficRouting) === null || _g === void 0 ? void 0 : _g.ProdTrafficRoute) === null || _h === void 0 ? void 0 : _h.LogicalID);
            const testTrafficRoute = findResource((_l = (_k = (_j = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _j === void 0 ? void 0 : _j.TrafficRouting) === null || _k === void 0 ? void 0 : _k.TestTrafficRoute) === null || _l === void 0 ? void 0 : _l.LogicalID);
            const targetGroups = (_p = (_o = (_m = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _m === void 0 ? void 0 : _m.TrafficRouting) === null || _o === void 0 ? void 0 : _o.TargetGroups) === null || _p === void 0 ? void 0 : _p.map((tg) => findResource(tg));
            return {
                target: {
                    type: (_q = app === null || app === void 0 ? void 0 : app.Target) === null || _q === void 0 ? void 0 : _q.Type,
                    logicalId: target === null || target === void 0 ? void 0 : target.logicalId,
                },
                ecsAttributes: {
                    taskDefinitions: taskDefinitions === null || taskDefinitions === void 0 ? void 0 : taskDefinitions.map(td => td === null || td === void 0 ? void 0 : td.logicalId),
                    taskSets: taskSets === null || taskSets === void 0 ? void 0 : taskSets.map(ts => ts === null || ts === void 0 ? void 0 : ts.logicalId),
                    trafficRouting: {
                        prodTrafficRoute: {
                            type: (_t = (_s = (_r = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _r === void 0 ? void 0 : _r.TrafficRouting) === null || _s === void 0 ? void 0 : _s.ProdTrafficRoute) === null || _t === void 0 ? void 0 : _t.Type,
                            logicalId: prodTrafficRoute === null || prodTrafficRoute === void 0 ? void 0 : prodTrafficRoute.logicalId,
                        },
                        testTrafficRoute: {
                            type: (_w = (_v = (_u = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _u === void 0 ? void 0 : _u.TrafficRouting) === null || _v === void 0 ? void 0 : _v.TestTrafficRoute) === null || _w === void 0 ? void 0 : _w.Type,
                            logicalId: testTrafficRoute === null || testTrafficRoute === void 0 ? void 0 : testTrafficRoute.logicalId,
                        },
                        targetGroups: targetGroups === null || targetGroups === void 0 ? void 0 : targetGroups.map((tg) => tg === null || tg === void 0 ? void 0 : tg.logicalId),
                    },
                },
            };
        }
        function findResource(logicalId) {
            if (logicalId == null) {
                return undefined;
            }
            const ret = options.parser.finder.findResource(logicalId);
            if (!ret) {
                throw new Error(`Hook '${id}' references resource '${logicalId}' that was not found in the template`);
            }
            return ret;
        }
    }
    /**
     * The IAM Role for CloudFormation to use to perform blue-green deployments.
     */
    get serviceRole() {
        return this._serviceRole;
    }
    set serviceRole(serviceRole) {
        this._serviceRole = serviceRole;
    }
    /**
     * Properties of the Amazon ECS applications being deployed.
     */
    get applications() {
        return this._applications;
    }
    set applications(value) {
        this._applications = value;
    }
    /**
     * Traffic routing configuration settings.
     *
     * @default - time-based canary traffic shifting, with a 15% step percentage and a five minute bake time
     */
    get trafficRoutingConfig() {
        return this._trafficRoutingConfig;
    }
    set trafficRoutingConfig(value) {
        this._trafficRoutingConfig = value;
    }
    /**
     * Additional options for the blue/green deployment.
     *
     * @default - no additional options
     */
    get additionalOptions() {
        return this._additionalOptions;
    }
    set additionalOptions(value) {
        this._additionalOptions = value;
    }
    /**
     * Use lifecycle event hooks to specify a Lambda function that CodeDeploy can call to validate a deployment.
     * You can use the same function or a different one for deployment lifecycle events.
     * Following completion of the validation tests,
     * the Lambda {@link CfnCodeDeployBlueGreenLifecycleEventHooks.afterAllowTraffic}
     * function calls back CodeDeploy and delivers a result of 'Succeeded' or 'Failed'.
     *
     * @default - no lifecycle event hooks
     */
    get lifecycleEventHooks() {
        return this._lifecycleEventHooks;
    }
    set lifecycleEventHooks(value) {
        this._lifecycleEventHooks = value;
    }
    renderProperties(_props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q;
        return {
            ServiceRole: this.serviceRole,
            Applications: this.applications.map((app) => ({
                Target: {
                    Type: app.target.type,
                    LogicalID: app.target.logicalId,
                },
                ECSAttributes: {
                    TaskDefinitions: app.ecsAttributes.taskDefinitions,
                    TaskSets: app.ecsAttributes.taskSets,
                    TrafficRouting: {
                        ProdTrafficRoute: {
                            Type: app.ecsAttributes.trafficRouting.prodTrafficRoute.type,
                            LogicalID: app.ecsAttributes.trafficRouting.prodTrafficRoute.logicalId,
                        },
                        TestTrafficRoute: {
                            Type: app.ecsAttributes.trafficRouting.testTrafficRoute.type,
                            LogicalID: app.ecsAttributes.trafficRouting.testTrafficRoute.logicalId,
                        },
                        TargetGroups: app.ecsAttributes.trafficRouting.targetGroups,
                    },
                },
            })),
            TrafficRoutingConfig: {
                Type: (_a = this.trafficRoutingConfig) === null || _a === void 0 ? void 0 : _a.type,
                TimeBasedCanary: {
                    StepPercentage: (_c = (_b = this.trafficRoutingConfig) === null || _b === void 0 ? void 0 : _b.timeBasedCanary) === null || _c === void 0 ? void 0 : _c.stepPercentage,
                    BakeTimeMins: (_e = (_d = this.trafficRoutingConfig) === null || _d === void 0 ? void 0 : _d.timeBasedCanary) === null || _e === void 0 ? void 0 : _e.bakeTimeMins,
                },
                TimeBasedLinear: {
                    StepPercentage: (_g = (_f = this.trafficRoutingConfig) === null || _f === void 0 ? void 0 : _f.timeBasedLinear) === null || _g === void 0 ? void 0 : _g.stepPercentage,
                    BakeTimeMins: (_j = (_h = this.trafficRoutingConfig) === null || _h === void 0 ? void 0 : _h.timeBasedLinear) === null || _j === void 0 ? void 0 : _j.bakeTimeMins,
                },
            },
            AdditionalOptions: {
                TerminationWaitTimeInMinutes: (_k = this.additionalOptions) === null || _k === void 0 ? void 0 : _k.terminationWaitTimeInMinutes,
            },
            LifecycleEventHooks: {
                BeforeInstall: (_l = this.lifecycleEventHooks) === null || _l === void 0 ? void 0 : _l.beforeInstall,
                AfterInstall: (_m = this.lifecycleEventHooks) === null || _m === void 0 ? void 0 : _m.afterInstall,
                AfterAllowTestTraffic: (_o = this.lifecycleEventHooks) === null || _o === void 0 ? void 0 : _o.afterAllowTestTraffic,
                BeforeAllowTraffic: (_p = this.lifecycleEventHooks) === null || _p === void 0 ? void 0 : _p.beforeAllowTraffic,
                AfterAllowTraffic: (_q = this.lifecycleEventHooks) === null || _q === void 0 ? void 0 : _q.afterAllowTraffic,
            },
        };
    }
}
exports.CfnCodeDeployBlueGreenHook = CfnCodeDeployBlueGreenHook;
//# sourceMappingURL=data:application/json;base64,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