"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScopedAws = exports.Aws = void 0;
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
const AWS_ACCOUNTID = 'AWS::AccountId';
const AWS_URLSUFFIX = 'AWS::URLSuffix';
const AWS_NOTIFICATIONARNS = 'AWS::NotificationARNs';
const AWS_PARTITION = 'AWS::Partition';
const AWS_REGION = 'AWS::Region';
const AWS_STACKID = 'AWS::StackId';
const AWS_STACKNAME = 'AWS::StackName';
const AWS_NOVALUE = 'AWS::NoValue';
/**
 * Accessor for pseudo parameters
 *
 * Since pseudo parameters need to be anchored to a stack somewhere in the
 * construct tree, this class takes an scope parameter; the pseudo parameter
 * values can be obtained as properties from an scoped object.
 */
class Aws {
    constructor() { }
}
exports.Aws = Aws;
Aws.ACCOUNT_ID = pseudoString(AWS_ACCOUNTID);
Aws.URL_SUFFIX = pseudoString(AWS_URLSUFFIX);
Aws.NOTIFICATION_ARNS = token_1.Token.asList({ Ref: AWS_NOTIFICATIONARNS }, { displayHint: AWS_NOTIFICATIONARNS });
Aws.PARTITION = pseudoString(AWS_PARTITION);
Aws.REGION = pseudoString(AWS_REGION);
Aws.STACK_ID = pseudoString(AWS_STACKID);
Aws.STACK_NAME = pseudoString(AWS_STACKNAME);
Aws.NO_VALUE = pseudoString(AWS_NOVALUE);
/**
 * Accessor for scoped pseudo parameters
 *
 * These pseudo parameters are anchored to a stack somewhere in the construct
 * tree, and their values will be exported automatically.
 */
class ScopedAws {
    constructor(scope) {
        this.scope = scope;
    }
    get accountId() {
        return this.asString(AWS_ACCOUNTID);
    }
    get urlSuffix() {
        return this.asString(AWS_URLSUFFIX);
    }
    get notificationArns() {
        return token_1.Token.asList(cfn_reference_1.CfnReference.forPseudo(AWS_NOTIFICATIONARNS, this.scope), {
            displayHint: AWS_NOTIFICATIONARNS,
        });
    }
    get partition() {
        return this.asString(AWS_PARTITION);
    }
    get region() {
        return this.asString(AWS_REGION);
    }
    get stackId() {
        return this.asString(AWS_STACKID);
    }
    get stackName() {
        return this.asString(AWS_STACKNAME);
    }
    asString(name) {
        return token_1.Token.asString(cfn_reference_1.CfnReference.forPseudo(name, this.scope), { displayHint: name });
    }
}
exports.ScopedAws = ScopedAws;
function pseudoString(name) {
    // we don't want any ':' in the serialized form,
    // as ':' is the ARN separator,
    // and so we don't want ARN components
    // (which these CFN references like AWS::Partition certainly can be)
    // to contain ':'s themselves
    return token_1.Token.asString({ Ref: name }, { displayHint: name.replace('::', '.') });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2ZuLXBzZXVkby5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNmbi1wc2V1ZG8udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EsMkRBQXVEO0FBQ3ZELG1DQUFnQztBQUNoQyxNQUFNLGFBQWEsR0FBRyxnQkFBZ0IsQ0FBQztBQUN2QyxNQUFNLGFBQWEsR0FBRyxnQkFBZ0IsQ0FBQztBQUN2QyxNQUFNLG9CQUFvQixHQUFHLHVCQUF1QixDQUFDO0FBQ3JELE1BQU0sYUFBYSxHQUFHLGdCQUFnQixDQUFDO0FBQ3ZDLE1BQU0sVUFBVSxHQUFHLGFBQWEsQ0FBQztBQUNqQyxNQUFNLFdBQVcsR0FBRyxjQUFjLENBQUM7QUFDbkMsTUFBTSxhQUFhLEdBQUcsZ0JBQWdCLENBQUM7QUFDdkMsTUFBTSxXQUFXLEdBQUcsY0FBYyxDQUFDO0FBQ25DOzs7Ozs7R0FNRztBQUNILE1BQWEsR0FBRztJQVNaLGdCQUF3QixDQUFDOztBQVQ3QixrQkFVQztBQVQwQixjQUFVLEdBQUcsWUFBWSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0FBQ3pDLGNBQVUsR0FBRyxZQUFZLENBQUMsYUFBYSxDQUFDLENBQUM7QUFDekMscUJBQWlCLEdBQUcsYUFBSyxDQUFDLE1BQU0sQ0FBQyxFQUFFLEdBQUcsRUFBRSxvQkFBb0IsRUFBRSxFQUFFLEVBQUUsV0FBVyxFQUFFLG9CQUFvQixFQUFFLENBQUMsQ0FBQztBQUN2RyxhQUFTLEdBQUcsWUFBWSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0FBQ3hDLFVBQU0sR0FBRyxZQUFZLENBQUMsVUFBVSxDQUFDLENBQUM7QUFDbEMsWUFBUSxHQUFHLFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQztBQUNyQyxjQUFVLEdBQUcsWUFBWSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0FBQ3pDLFlBQVEsR0FBRyxZQUFZLENBQUMsV0FBVyxDQUFDLENBQUM7QUFHaEU7Ozs7O0dBS0c7QUFDSCxNQUFhLFNBQVM7SUFDbEIsWUFBNkIsS0FBZ0I7UUFBaEIsVUFBSyxHQUFMLEtBQUssQ0FBVztJQUM3QyxDQUFDO0lBQ0QsSUFBVyxTQUFTO1FBQ2hCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBQ0QsSUFBVyxTQUFTO1FBQ2hCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBQ0QsSUFBVyxnQkFBZ0I7UUFDdkIsT0FBTyxhQUFLLENBQUMsTUFBTSxDQUFDLDRCQUFZLENBQUMsU0FBUyxDQUFDLG9CQUFvQixFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUMxRSxXQUFXLEVBQUUsb0JBQW9CO1NBQ3BDLENBQUMsQ0FBQztJQUNQLENBQUM7SUFDRCxJQUFXLFNBQVM7UUFDaEIsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFDRCxJQUFXLE1BQU07UUFDYixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLENBQUM7SUFDckMsQ0FBQztJQUNELElBQVcsT0FBTztRQUNkLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBQ0QsSUFBVyxTQUFTO1FBQ2hCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBQ08sUUFBUSxDQUFDLElBQVk7UUFDekIsT0FBTyxhQUFLLENBQUMsUUFBUSxDQUFDLDRCQUFZLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsRUFBRSxXQUFXLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUMzRixDQUFDO0NBQ0o7QUE3QkQsOEJBNkJDO0FBQ0QsU0FBUyxZQUFZLENBQUMsSUFBWTtJQUM5QixnREFBZ0Q7SUFDaEQsK0JBQStCO0lBQy9CLHNDQUFzQztJQUN0QyxvRUFBb0U7SUFDcEUsNkJBQTZCO0lBQzdCLE9BQU8sYUFBSyxDQUFDLFFBQVEsQ0FBQyxFQUFFLEdBQUcsRUFBRSxJQUFJLEVBQUUsRUFBRSxFQUFFLFdBQVcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUM7QUFDbkYsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJy4vY29uc3RydWN0LWNvbXBhdCc7XG5pbXBvcnQgeyBDZm5SZWZlcmVuY2UgfSBmcm9tICcuL3ByaXZhdGUvY2ZuLXJlZmVyZW5jZSc7XG5pbXBvcnQgeyBUb2tlbiB9IGZyb20gJy4vdG9rZW4nO1xuY29uc3QgQVdTX0FDQ09VTlRJRCA9ICdBV1M6OkFjY291bnRJZCc7XG5jb25zdCBBV1NfVVJMU1VGRklYID0gJ0FXUzo6VVJMU3VmZml4JztcbmNvbnN0IEFXU19OT1RJRklDQVRJT05BUk5TID0gJ0FXUzo6Tm90aWZpY2F0aW9uQVJOcyc7XG5jb25zdCBBV1NfUEFSVElUSU9OID0gJ0FXUzo6UGFydGl0aW9uJztcbmNvbnN0IEFXU19SRUdJT04gPSAnQVdTOjpSZWdpb24nO1xuY29uc3QgQVdTX1NUQUNLSUQgPSAnQVdTOjpTdGFja0lkJztcbmNvbnN0IEFXU19TVEFDS05BTUUgPSAnQVdTOjpTdGFja05hbWUnO1xuY29uc3QgQVdTX05PVkFMVUUgPSAnQVdTOjpOb1ZhbHVlJztcbi8qKlxuICogQWNjZXNzb3IgZm9yIHBzZXVkbyBwYXJhbWV0ZXJzXG4gKlxuICogU2luY2UgcHNldWRvIHBhcmFtZXRlcnMgbmVlZCB0byBiZSBhbmNob3JlZCB0byBhIHN0YWNrIHNvbWV3aGVyZSBpbiB0aGVcbiAqIGNvbnN0cnVjdCB0cmVlLCB0aGlzIGNsYXNzIHRha2VzIGFuIHNjb3BlIHBhcmFtZXRlcjsgdGhlIHBzZXVkbyBwYXJhbWV0ZXJcbiAqIHZhbHVlcyBjYW4gYmUgb2J0YWluZWQgYXMgcHJvcGVydGllcyBmcm9tIGFuIHNjb3BlZCBvYmplY3QuXG4gKi9cbmV4cG9ydCBjbGFzcyBBd3Mge1xuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQUNDT1VOVF9JRCA9IHBzZXVkb1N0cmluZyhBV1NfQUNDT1VOVElEKTtcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFVSTF9TVUZGSVggPSBwc2V1ZG9TdHJpbmcoQVdTX1VSTFNVRkZJWCk7XG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBOT1RJRklDQVRJT05fQVJOUyA9IFRva2VuLmFzTGlzdCh7IFJlZjogQVdTX05PVElGSUNBVElPTkFSTlMgfSwgeyBkaXNwbGF5SGludDogQVdTX05PVElGSUNBVElPTkFSTlMgfSk7XG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBQQVJUSVRJT04gPSBwc2V1ZG9TdHJpbmcoQVdTX1BBUlRJVElPTik7XG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBSRUdJT04gPSBwc2V1ZG9TdHJpbmcoQVdTX1JFR0lPTik7XG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBTVEFDS19JRCA9IHBzZXVkb1N0cmluZyhBV1NfU1RBQ0tJRCk7XG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBTVEFDS19OQU1FID0gcHNldWRvU3RyaW5nKEFXU19TVEFDS05BTUUpO1xuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTk9fVkFMVUUgPSBwc2V1ZG9TdHJpbmcoQVdTX05PVkFMVUUpO1xuICAgIHByaXZhdGUgY29uc3RydWN0b3IoKSB7IH1cbn1cbi8qKlxuICogQWNjZXNzb3IgZm9yIHNjb3BlZCBwc2V1ZG8gcGFyYW1ldGVyc1xuICpcbiAqIFRoZXNlIHBzZXVkbyBwYXJhbWV0ZXJzIGFyZSBhbmNob3JlZCB0byBhIHN0YWNrIHNvbWV3aGVyZSBpbiB0aGUgY29uc3RydWN0XG4gKiB0cmVlLCBhbmQgdGhlaXIgdmFsdWVzIHdpbGwgYmUgZXhwb3J0ZWQgYXV0b21hdGljYWxseS5cbiAqL1xuZXhwb3J0IGNsYXNzIFNjb3BlZEF3cyB7XG4gICAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBzY29wZTogQ29uc3RydWN0KSB7XG4gICAgfVxuICAgIHB1YmxpYyBnZXQgYWNjb3VudElkKCk6IHN0cmluZyB7XG4gICAgICAgIHJldHVybiB0aGlzLmFzU3RyaW5nKEFXU19BQ0NPVU5USUQpO1xuICAgIH1cbiAgICBwdWJsaWMgZ2V0IHVybFN1ZmZpeCgpOiBzdHJpbmcge1xuICAgICAgICByZXR1cm4gdGhpcy5hc1N0cmluZyhBV1NfVVJMU1VGRklYKTtcbiAgICB9XG4gICAgcHVibGljIGdldCBub3RpZmljYXRpb25Bcm5zKCk6IHN0cmluZ1tdIHtcbiAgICAgICAgcmV0dXJuIFRva2VuLmFzTGlzdChDZm5SZWZlcmVuY2UuZm9yUHNldWRvKEFXU19OT1RJRklDQVRJT05BUk5TLCB0aGlzLnNjb3BlKSwge1xuICAgICAgICAgICAgZGlzcGxheUhpbnQ6IEFXU19OT1RJRklDQVRJT05BUk5TLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgcHVibGljIGdldCBwYXJ0aXRpb24oKTogc3RyaW5nIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuYXNTdHJpbmcoQVdTX1BBUlRJVElPTik7XG4gICAgfVxuICAgIHB1YmxpYyBnZXQgcmVnaW9uKCk6IHN0cmluZyB7XG4gICAgICAgIHJldHVybiB0aGlzLmFzU3RyaW5nKEFXU19SRUdJT04pO1xuICAgIH1cbiAgICBwdWJsaWMgZ2V0IHN0YWNrSWQoKTogc3RyaW5nIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuYXNTdHJpbmcoQVdTX1NUQUNLSUQpO1xuICAgIH1cbiAgICBwdWJsaWMgZ2V0IHN0YWNrTmFtZSgpOiBzdHJpbmcge1xuICAgICAgICByZXR1cm4gdGhpcy5hc1N0cmluZyhBV1NfU1RBQ0tOQU1FKTtcbiAgICB9XG4gICAgcHJpdmF0ZSBhc1N0cmluZyhuYW1lOiBzdHJpbmcpIHtcbiAgICAgICAgcmV0dXJuIFRva2VuLmFzU3RyaW5nKENmblJlZmVyZW5jZS5mb3JQc2V1ZG8obmFtZSwgdGhpcy5zY29wZSksIHsgZGlzcGxheUhpbnQ6IG5hbWUgfSk7XG4gICAgfVxufVxuZnVuY3Rpb24gcHNldWRvU3RyaW5nKG5hbWU6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgLy8gd2UgZG9uJ3Qgd2FudCBhbnkgJzonIGluIHRoZSBzZXJpYWxpemVkIGZvcm0sXG4gICAgLy8gYXMgJzonIGlzIHRoZSBBUk4gc2VwYXJhdG9yLFxuICAgIC8vIGFuZCBzbyB3ZSBkb24ndCB3YW50IEFSTiBjb21wb25lbnRzXG4gICAgLy8gKHdoaWNoIHRoZXNlIENGTiByZWZlcmVuY2VzIGxpa2UgQVdTOjpQYXJ0aXRpb24gY2VydGFpbmx5IGNhbiBiZSlcbiAgICAvLyB0byBjb250YWluICc6J3MgdGhlbXNlbHZlc1xuICAgIHJldHVybiBUb2tlbi5hc1N0cmluZyh7IFJlZjogbmFtZSB9LCB7IGRpc3BsYXlIaW50OiBuYW1lLnJlcGxhY2UoJzo6JywgJy4nKSB9KTtcbn1cbiJdfQ==