"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnRule = void 0;
const cfn_element_1 = require("./cfn-element");
const util_1 = require("./util");
/**
 * The Rules that define template constraints in an AWS Service Catalog portfolio describe when
 * end users can use the template and which values they can specify for parameters that are declared
 * in the AWS CloudFormation template used to create the product they are attempting to use. Rules
 * are useful for preventing end users from inadvertently specifying an incorrect value.
 * For example, you can add a rule to verify whether end users specified a valid subnet in a
 * given VPC or used m1.small instance types for test environments. AWS CloudFormation uses
 * rules to validate parameter values before it creates the resources for the product.
 *
 * A rule can include a RuleCondition property and must include an Assertions property.
 * For each rule, you can define only one rule condition; you can define one or more asserts within the Assertions property.
 * You define a rule condition and assertions by using rule-specific intrinsic functions.
 *
 * @link https://docs.aws.amazon.com/servicecatalog/latest/adminguide/reference-template_constraint_rules.html
 */
class CfnRule extends cfn_element_1.CfnRefElement {
    /**
     * Creates and adds a rule.
     * @param scope The parent construct.
     * @param props The rule props.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.ruleCondition = props && props.ruleCondition;
        this.assertions = props && props.assertions;
    }
    /**
     * Adds an assertion to the rule.
     * @param condition The expression to evaluation.
     * @param description The description of the assertion.
     */
    addAssertion(condition, description) {
        if (!this.assertions) {
            this.assertions = [];
        }
        this.assertions.push({
            assert: condition,
            assertDescription: description,
        });
    }
    /**
     * @internal
     */
    _toCloudFormation() {
        return {
            Rules: {
                [this.logicalId]: {
                    RuleCondition: this.ruleCondition,
                    Assertions: util_1.capitalizePropertyNames(this, this.assertions),
                },
            },
        };
    }
}
exports.CfnRule = CfnRule;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2ZuLXJ1bGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjZm4tcnVsZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSwrQ0FBOEM7QUFFOUMsaUNBQWlEO0FBcUNqRDs7Ozs7Ozs7Ozs7Ozs7R0FjRztBQUNILE1BQWEsT0FBUSxTQUFRLDJCQUFhO0lBR3RDOzs7O09BSUc7SUFDSCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW9CO1FBQzFELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDakIsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLElBQUksS0FBSyxDQUFDLGFBQWEsQ0FBQztRQUNsRCxJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssSUFBSSxLQUFLLENBQUMsVUFBVSxDQUFDO0lBQ2hELENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksWUFBWSxDQUFDLFNBQWtDLEVBQUUsV0FBbUI7UUFDdkUsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUU7WUFDbEIsSUFBSSxDQUFDLFVBQVUsR0FBRyxFQUFFLENBQUM7U0FDeEI7UUFDRCxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQztZQUNqQixNQUFNLEVBQUUsU0FBUztZQUNqQixpQkFBaUIsRUFBRSxXQUFXO1NBQ2pDLENBQUMsQ0FBQztJQUNQLENBQUM7SUFDRDs7T0FFRztJQUNJLGlCQUFpQjtRQUNwQixPQUFPO1lBQ0gsS0FBSyxFQUFFO2dCQUNILENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFO29CQUNkLGFBQWEsRUFBRSxJQUFJLENBQUMsYUFBYTtvQkFDakMsVUFBVSxFQUFFLDhCQUF1QixDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDO2lCQUM3RDthQUNKO1NBQ0osQ0FBQztJQUNOLENBQUM7Q0FDSjtBQXhDRCwwQkF3Q0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBJQ2ZuQ29uZGl0aW9uRXhwcmVzc2lvbiB9IGZyb20gJy4vY2ZuLWNvbmRpdGlvbic7XG5pbXBvcnQgeyBDZm5SZWZFbGVtZW50IH0gZnJvbSAnLi9jZm4tZWxlbWVudCc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICcuL2NvbnN0cnVjdC1jb21wYXQnO1xuaW1wb3J0IHsgY2FwaXRhbGl6ZVByb3BlcnR5TmFtZXMgfSBmcm9tICcuL3V0aWwnO1xuLyoqXG4gKiBBIHJ1bGUgY2FuIGluY2x1ZGUgYSBSdWxlQ29uZGl0aW9uIHByb3BlcnR5IGFuZCBtdXN0IGluY2x1ZGUgYW4gQXNzZXJ0aW9ucyBwcm9wZXJ0eS5cbiAqIEZvciBlYWNoIHJ1bGUsIHlvdSBjYW4gZGVmaW5lIG9ubHkgb25lIHJ1bGUgY29uZGl0aW9uOyB5b3UgY2FuIGRlZmluZSBvbmUgb3IgbW9yZSBhc3NlcnRzIHdpdGhpbiB0aGUgQXNzZXJ0aW9ucyBwcm9wZXJ0eS5cbiAqIFlvdSBkZWZpbmUgYSBydWxlIGNvbmRpdGlvbiBhbmQgYXNzZXJ0aW9ucyBieSB1c2luZyBydWxlLXNwZWNpZmljIGludHJpbnNpYyBmdW5jdGlvbnMuXG4gKlxuICogWW91IGNhbiB1c2UgdGhlIGZvbGxvd2luZyBydWxlLXNwZWNpZmljIGludHJpbnNpYyBmdW5jdGlvbnMgdG8gZGVmaW5lIHJ1bGUgY29uZGl0aW9ucyBhbmQgYXNzZXJ0aW9uczpcbiAqXG4gKiAgRm46OkFuZFxuICogIEZuOjpDb250YWluc1xuICogIEZuOjpFYWNoTWVtYmVyRXF1YWxzXG4gKiAgRm46OkVhY2hNZW1iZXJJblxuICogIEZuOjpFcXVhbHNcbiAqICBGbjo6SWZcbiAqICBGbjo6Tm90XG4gKiAgRm46Ok9yXG4gKiAgRm46OlJlZkFsbFxuICogIEZuOjpWYWx1ZU9mXG4gKiAgRm46OlZhbHVlT2ZBbGxcbiAqXG4gKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZWNhdGFsb2cvbGF0ZXN0L2FkbWluZ3VpZGUvcmVmZXJlbmNlLXRlbXBsYXRlX2NvbnN0cmFpbnRfcnVsZXMuaHRtbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIENmblJ1bGVQcm9wcyB7XG4gICAgLyoqXG4gICAgICogSWYgdGhlIHJ1bGUgY29uZGl0aW9uIGV2YWx1YXRlcyB0byBmYWxzZSwgdGhlIHJ1bGUgZG9lc24ndCB0YWtlIGVmZmVjdC5cbiAgICAgKiBJZiB0aGUgZnVuY3Rpb24gaW4gdGhlIHJ1bGUgY29uZGl0aW9uIGV2YWx1YXRlcyB0byB0cnVlLCBleHByZXNzaW9ucyBpbiBlYWNoIGFzc2VydCBhcmUgZXZhbHVhdGVkIGFuZCBhcHBsaWVkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBSdWxlJ3MgYXNzZXJ0aW9ucyB3aWxsIGFsd2F5cyB0YWtlIGVmZmVjdC5cbiAgICAgKi9cbiAgICByZWFkb25seSBydWxlQ29uZGl0aW9uPzogSUNmbkNvbmRpdGlvbkV4cHJlc3Npb247XG4gICAgLyoqXG4gICAgICogQXNzZXJ0aW9ucyB3aGljaCBkZWZpbmUgdGhlIHJ1bGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGFzc2VydGlvbnMgZm9yIHRoZSBydWxlLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFzc2VydGlvbnM/OiBDZm5SdWxlQXNzZXJ0aW9uW107XG59XG4vKipcbiAqIFRoZSBSdWxlcyB0aGF0IGRlZmluZSB0ZW1wbGF0ZSBjb25zdHJhaW50cyBpbiBhbiBBV1MgU2VydmljZSBDYXRhbG9nIHBvcnRmb2xpbyBkZXNjcmliZSB3aGVuXG4gKiBlbmQgdXNlcnMgY2FuIHVzZSB0aGUgdGVtcGxhdGUgYW5kIHdoaWNoIHZhbHVlcyB0aGV5IGNhbiBzcGVjaWZ5IGZvciBwYXJhbWV0ZXJzIHRoYXQgYXJlIGRlY2xhcmVkXG4gKiBpbiB0aGUgQVdTIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlIHVzZWQgdG8gY3JlYXRlIHRoZSBwcm9kdWN0IHRoZXkgYXJlIGF0dGVtcHRpbmcgdG8gdXNlLiBSdWxlc1xuICogYXJlIHVzZWZ1bCBmb3IgcHJldmVudGluZyBlbmQgdXNlcnMgZnJvbSBpbmFkdmVydGVudGx5IHNwZWNpZnlpbmcgYW4gaW5jb3JyZWN0IHZhbHVlLlxuICogRm9yIGV4YW1wbGUsIHlvdSBjYW4gYWRkIGEgcnVsZSB0byB2ZXJpZnkgd2hldGhlciBlbmQgdXNlcnMgc3BlY2lmaWVkIGEgdmFsaWQgc3VibmV0IGluIGFcbiAqIGdpdmVuIFZQQyBvciB1c2VkIG0xLnNtYWxsIGluc3RhbmNlIHR5cGVzIGZvciB0ZXN0IGVudmlyb25tZW50cy4gQVdTIENsb3VkRm9ybWF0aW9uIHVzZXNcbiAqIHJ1bGVzIHRvIHZhbGlkYXRlIHBhcmFtZXRlciB2YWx1ZXMgYmVmb3JlIGl0IGNyZWF0ZXMgdGhlIHJlc291cmNlcyBmb3IgdGhlIHByb2R1Y3QuXG4gKlxuICogQSBydWxlIGNhbiBpbmNsdWRlIGEgUnVsZUNvbmRpdGlvbiBwcm9wZXJ0eSBhbmQgbXVzdCBpbmNsdWRlIGFuIEFzc2VydGlvbnMgcHJvcGVydHkuXG4gKiBGb3IgZWFjaCBydWxlLCB5b3UgY2FuIGRlZmluZSBvbmx5IG9uZSBydWxlIGNvbmRpdGlvbjsgeW91IGNhbiBkZWZpbmUgb25lIG9yIG1vcmUgYXNzZXJ0cyB3aXRoaW4gdGhlIEFzc2VydGlvbnMgcHJvcGVydHkuXG4gKiBZb3UgZGVmaW5lIGEgcnVsZSBjb25kaXRpb24gYW5kIGFzc2VydGlvbnMgYnkgdXNpbmcgcnVsZS1zcGVjaWZpYyBpbnRyaW5zaWMgZnVuY3Rpb25zLlxuICpcbiAqIEBsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlY2F0YWxvZy9sYXRlc3QvYWRtaW5ndWlkZS9yZWZlcmVuY2UtdGVtcGxhdGVfY29uc3RyYWludF9ydWxlcy5odG1sXG4gKi9cbmV4cG9ydCBjbGFzcyBDZm5SdWxlIGV4dGVuZHMgQ2ZuUmVmRWxlbWVudCB7XG4gICAgcHJpdmF0ZSBydWxlQ29uZGl0aW9uPzogSUNmbkNvbmRpdGlvbkV4cHJlc3Npb247XG4gICAgcHJpdmF0ZSBhc3NlcnRpb25zPzogQ2ZuUnVsZUFzc2VydGlvbltdO1xuICAgIC8qKlxuICAgICAqIENyZWF0ZXMgYW5kIGFkZHMgYSBydWxlLlxuICAgICAqIEBwYXJhbSBzY29wZSBUaGUgcGFyZW50IGNvbnN0cnVjdC5cbiAgICAgKiBAcGFyYW0gcHJvcHMgVGhlIHJ1bGUgcHJvcHMuXG4gICAgICovXG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM/OiBDZm5SdWxlUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgdGhpcy5ydWxlQ29uZGl0aW9uID0gcHJvcHMgJiYgcHJvcHMucnVsZUNvbmRpdGlvbjtcbiAgICAgICAgdGhpcy5hc3NlcnRpb25zID0gcHJvcHMgJiYgcHJvcHMuYXNzZXJ0aW9ucztcbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkcyBhbiBhc3NlcnRpb24gdG8gdGhlIHJ1bGUuXG4gICAgICogQHBhcmFtIGNvbmRpdGlvbiBUaGUgZXhwcmVzc2lvbiB0byBldmFsdWF0aW9uLlxuICAgICAqIEBwYXJhbSBkZXNjcmlwdGlvbiBUaGUgZGVzY3JpcHRpb24gb2YgdGhlIGFzc2VydGlvbi5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkQXNzZXJ0aW9uKGNvbmRpdGlvbjogSUNmbkNvbmRpdGlvbkV4cHJlc3Npb24sIGRlc2NyaXB0aW9uOiBzdHJpbmcpIHtcbiAgICAgICAgaWYgKCF0aGlzLmFzc2VydGlvbnMpIHtcbiAgICAgICAgICAgIHRoaXMuYXNzZXJ0aW9ucyA9IFtdO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuYXNzZXJ0aW9ucy5wdXNoKHtcbiAgICAgICAgICAgIGFzc2VydDogY29uZGl0aW9uLFxuICAgICAgICAgICAgYXNzZXJ0RGVzY3JpcHRpb246IGRlc2NyaXB0aW9uLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQGludGVybmFsXG4gICAgICovXG4gICAgcHVibGljIF90b0Nsb3VkRm9ybWF0aW9uKCk6IG9iamVjdCB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBSdWxlczoge1xuICAgICAgICAgICAgICAgIFt0aGlzLmxvZ2ljYWxJZF06IHtcbiAgICAgICAgICAgICAgICAgICAgUnVsZUNvbmRpdGlvbjogdGhpcy5ydWxlQ29uZGl0aW9uLFxuICAgICAgICAgICAgICAgICAgICBBc3NlcnRpb25zOiBjYXBpdGFsaXplUHJvcGVydHlOYW1lcyh0aGlzLCB0aGlzLmFzc2VydGlvbnMpLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICB9O1xuICAgIH1cbn1cbi8qKlxuICogQSBydWxlIGFzc2VydGlvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDZm5SdWxlQXNzZXJ0aW9uIHtcbiAgICAvKipcbiAgICAgKiBUaGUgYXNzZXJ0aW9uLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFzc2VydDogSUNmbkNvbmRpdGlvbkV4cHJlc3Npb247XG4gICAgLyoqXG4gICAgICogVGhlIGFzc2VydGlvbiBkZXNjcmlwdGlvbi5cbiAgICAgKi9cbiAgICByZWFkb25seSBhc3NlcnREZXNjcmlwdGlvbjogc3RyaW5nO1xufVxuIl19