"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContextProvider = void 0;
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const constructs_1 = require("constructs");
const annotations_1 = require("./annotations");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * Base class for the model side of context providers
 *
 * Instances of this class communicate with context provider plugins in the 'cdk
 * toolkit' via context variables (input), outputting specialized queries for
 * more context variables (output).
 *
 * ContextProvider needs access to a Construct to hook into the context mechanism.
 *
 * @experimental
 */
class ContextProvider {
    /**
     * @returns the context key or undefined if a key cannot be rendered (due to tokens used in any of the props)
     */
    static getKey(scope, options) {
        const stack = stack_1.Stack.of(scope);
        const props = {
            account: stack.account,
            region: stack.region,
            ...options.props || {},
        };
        if (Object.values(props).find(x => token_1.Token.isUnresolved(x))) {
            throw new Error(`Cannot determine scope for context provider ${options.provider}.\n` +
                'This usually happens when one or more of the provider props have unresolved tokens');
        }
        const propStrings = propsToArray(props);
        return {
            key: `${options.provider}:${propStrings.join(':')}`,
            props,
        };
    }
    static getValue(scope, options) {
        const stack = stack_1.Stack.of(scope);
        if (token_1.Token.isUnresolved(stack.account) || token_1.Token.isUnresolved(stack.region)) {
            throw new Error(`Cannot retrieve value from context provider ${options.provider} since account/region ` +
                'are not specified at the stack level. Either configure "env" with explicit account and region when ' +
                'you define your stack, or use the environment variables "CDK_DEFAULT_ACCOUNT" and "CDK_DEFAULT_REGION" ' +
                'to inherit environment information from the CLI (not recommended for production stacks)');
        }
        const { key, props } = this.getKey(scope, options);
        const value = constructs_1.Node.of(scope).tryGetContext(key);
        const providerError = extractProviderError(value);
        // if context is missing or an error occurred during context retrieval,
        // report and return a dummy value.
        if (value === undefined || providerError !== undefined) {
            stack.reportMissingContext({
                key,
                provider: options.provider,
                props: props,
            });
            if (providerError !== undefined) {
                annotations_1.Annotations.of(scope).addError(providerError);
            }
            return { value: options.dummyValue };
        }
        return { value };
    }
    constructor() { }
}
exports.ContextProvider = ContextProvider;
/**
 * If the context value represents an error, return the error message
 */
function extractProviderError(value) {
    if (typeof value === 'object' && value !== null) {
        return value[cxapi.PROVIDER_ERROR_KEY];
    }
    return undefined;
}
/**
 * Quote colons in all strings so that we can undo the quoting at a later point
 *
 * We'll use $ as a quoting character, for no particularly good reason other
 * than that \ is going to lead to quoting hell when the keys are stored in JSON.
 */
function colonQuote(xs) {
    return xs.replace('$', '$$').replace(':', '$:');
}
function propsToArray(props, keyPrefix = '') {
    const ret = [];
    for (const key of Object.keys(props)) {
        // skip undefined values
        if (props[key] === undefined) {
            continue;
        }
        switch (typeof props[key]) {
            case 'object': {
                ret.push(...propsToArray(props[key], `${keyPrefix}${key}.`));
                break;
            }
            case 'string': {
                ret.push(`${keyPrefix}${key}=${colonQuote(props[key])}`);
                break;
            }
            default: {
                ret.push(`${keyPrefix}${key}=${JSON.stringify(props[key])}`);
                break;
            }
        }
    }
    ret.sort();
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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