"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TreeMetadata = void 0;
const fs = require("fs");
const path = require("path");
const cloud_assembly_schema_1 = require("../../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const annotations_1 = require("../annotations");
const construct_compat_1 = require("../construct-compat");
const stack_1 = require("../stack");
const tree_1 = require("../tree");
const FILE_PATH = 'tree.json';
/**
 * Construct that is automatically attached to the top-level `App`.
 * This generates, as part of synthesis, a file containing the construct tree and the metadata for each node in the tree.
 * The output is in a tree format so as to preserve the construct hierarchy.
 *
 * @experimental
 */
class TreeMetadata extends construct_compat_1.Construct {
    constructor(scope) {
        super(scope, 'Tree');
    }
    /**
     * Create tree.json
     * @internal
     */
    _synthesizeTree(session) {
        const lookup = {};
        const visit = (construct) => {
            const children = construct.node.children.map((c) => {
                try {
                    return visit(c);
                }
                catch (e) {
                    annotations_1.Annotations.of(this).addWarning(`Failed to render tree metadata for node [${c.node.id}]. Reason: ${e}`);
                    return undefined;
                }
            });
            const childrenMap = children
                .filter((child) => child !== undefined)
                .reduce((map, child) => Object.assign(map, { [child.id]: child }), {});
            const node = {
                id: construct.node.id || 'App',
                path: construct.node.path,
                children: Object.keys(childrenMap).length === 0 ? undefined : childrenMap,
                attributes: this.synthAttributes(construct),
            };
            lookup[node.path] = node;
            return node;
        };
        const tree = {
            version: 'tree-0.1',
            tree: visit(this.node.root),
        };
        const builder = session.assembly;
        fs.writeFileSync(path.join(builder.outdir, FILE_PATH), JSON.stringify(tree, undefined, 2), { encoding: 'utf-8' });
        builder.addArtifact('Tree', {
            type: cloud_assembly_schema_1.ArtifactType.CDK_TREE,
            properties: {
                file: FILE_PATH,
            },
        });
    }
    synthAttributes(construct) {
        // check if a construct implements IInspectable
        function canInspect(inspectable) {
            return inspectable.inspect !== undefined;
        }
        const inspector = new tree_1.TreeInspector();
        // get attributes from the inspector
        if (canInspect(construct)) {
            construct.inspect(inspector);
            return stack_1.Stack.of(construct).resolve(inspector.attributes);
        }
        return undefined;
    }
}
exports.TreeMetadata = TreeMetadata;
//# sourceMappingURL=data:application/json;base64,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