"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultTokenResolver = exports.StringConcat = void 0;
const encoding_1 = require("./private/encoding");
const token_map_1 = require("./private/token-map");
/**
 * Converts all fragments to strings and concats those
 *
 * Drops 'undefined's.
 */
class StringConcat {
    join(left, right) {
        if (left === undefined) {
            return right !== undefined ? `${right}` : undefined;
        }
        if (right === undefined) {
            return `${left}`;
        }
        return `${left}${right}`;
    }
}
exports.StringConcat = StringConcat;
/**
 * Default resolver implementation
 *
 * @experimental
 */
class DefaultTokenResolver {
    constructor(concat) {
        this.concat = concat;
    }
    /**
     * Default Token resolution
     *
     * Resolve the Token, recurse into whatever it returns,
     * then finally post-process it.
     */
    resolveToken(t, context, postProcessor) {
        try {
            let resolved = t.resolve(context);
            // The token might have returned more values that need resolving, recurse
            resolved = context.resolve(resolved);
            resolved = postProcessor.postProcess(resolved, context);
            return resolved;
        }
        catch (e) {
            let message = `Resolution error: ${e.message}.`;
            if (t.creationStack && t.creationStack.length > 0) {
                message += `\nObject creation stack:\n  at ${t.creationStack.join('\n  at ')}`;
            }
            e.message = message;
            throw e;
        }
    }
    /**
     * Resolve string fragments to Tokens
     */
    resolveString(fragments, context) {
        return fragments.mapTokens({ mapToken: context.resolve }).join(this.concat);
    }
    resolveList(xs, context) {
        // Must be a singleton list token, because concatenation is not allowed.
        if (xs.length !== 1) {
            throw new Error(`Cannot add elements to list token, got: ${xs}`);
        }
        const str = encoding_1.TokenString.forListToken(xs[0]);
        const tokenMap = token_map_1.TokenMap.instance();
        const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
        if (fragments.length !== 1) {
            throw new Error(`Cannot concatenate strings in a tokenized string array, got: ${xs[0]}`);
        }
        return fragments.mapTokens({ mapToken: context.resolve }).firstValue;
    }
}
exports.DefaultTokenResolver = DefaultTokenResolver;
//# sourceMappingURL=data:application/json;base64,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