"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.rootPathTo = exports.Stack = void 0;
const fs = require("fs");
const path = require("path");
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const constructs_1 = require("constructs");
const annotations_1 = require("./annotations");
const app_1 = require("./app");
const arn_1 = require("./arn");
const cfn_element_1 = require("./cfn-element");
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_1 = require("./cfn-resource");
const construct_compat_1 = require("./construct-compat");
const context_provider_1 = require("./context-provider");
const feature_flags_1 = require("./feature-flags");
const cloudformation_lang_1 = require("./private/cloudformation-lang");
const logical_id_1 = require("./private/logical-id");
const resolve_1 = require("./private/resolve");
const uniqueid_1 = require("./private/uniqueid");
const STACK_SYMBOL = Symbol.for('@aws-cdk/core.Stack');
const MY_STACK_CACHE = Symbol.for('@aws-cdk/core.Stack.myStack');
const VALID_STACK_NAME_REGEX = /^[A-Za-z][A-Za-z0-9-]*$/;
/**
 * A root construct which represents a single CloudFormation stack.
 */
class Stack extends construct_compat_1.Construct {
    /**
     * Creates a new stack.
     *
     * @param scope Parent of this stack, usually an `App` or a `Stage`, but could be any construct.
     * @param id The construct ID of this stack. If `stackName` is not explicitly
     * defined, this id (and any parent IDs) will be used to determine the
     * physical ID of the stack.
     * @param props Stack properties.
     */
    constructor(scope, id, props = {}) {
        var _a, _b;
        // For unit test scope and id are optional for stacks, but we still want an App
        // as the parent because apps implement much of the synthesis logic.
        scope = scope !== null && scope !== void 0 ? scope : new app_1.App({
            autoSynth: false,
            outdir: fs_1.FileSystem.mkdtemp('cdk-test-app-'),
        });
        // "Default" is a "hidden id" from a `node.uniqueId` perspective
        id = id !== null && id !== void 0 ? id : 'Default';
        super(scope, id);
        this._missingContext = new Array();
        this._stackDependencies = {};
        this.templateOptions = {};
        Object.defineProperty(this, STACK_SYMBOL, { value: true });
        this._logicalIds = new logical_id_1.LogicalIDs();
        const { account, region, environment } = this.parseEnvironment(props.env);
        this.account = account;
        this.region = region;
        this.environment = environment;
        this.terminationProtection = props.terminationProtection;
        if (props.description !== undefined) {
            // Max length 1024 bytes
            // Typically 2 bytes per character, may be more for more exotic characters
            if (props.description.length > 512) {
                throw new Error(`Stack description must be <= 1024 bytes. Received description: '${props.description}'`);
            }
            this.templateOptions.description = props.description;
        }
        this._stackName = props.stackName !== undefined ? props.stackName : this.generateStackName();
        this.tags = new tag_manager_1.TagManager(cfn_resource_1.TagType.KEY_VALUE, 'aws:cdk:stack', props.tags);
        if (!VALID_STACK_NAME_REGEX.test(this.stackName)) {
            throw new Error(`Stack name must match the regular expression: ${VALID_STACK_NAME_REGEX.toString()}, got '${this.stackName}'`);
        }
        // the preferred behavior is to generate a unique id for this stack and use
        // it as the artifact ID in the assembly. this allows multiple stacks to use
        // the same name. however, this behavior is breaking for 1.x so it's only
        // applied under a feature flag which is applied automatically for new
        // projects created using `cdk init`.
        //
        // Also use the new behavior if we are using the new CI/CD-ready synthesizer; that way
        // people only have to flip one flag.
        const featureFlags = feature_flags_1.FeatureFlags.of(this);
        const stackNameDupeContext = featureFlags.isEnabled(cxapi.ENABLE_STACK_NAME_DUPLICATES_CONTEXT);
        const newStyleSynthesisContext = featureFlags.isEnabled(cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT);
        this.artifactId = (stackNameDupeContext || newStyleSynthesisContext)
            ? this.generateStackArtifactId()
            : this.stackName;
        this.templateFile = `${this.artifactId}.template.json`;
        // Not for nested stacks
        this._versionReportingEnabled = ((_a = props.analyticsReporting) !== null && _a !== void 0 ? _a : this.node.tryGetContext(cxapi.ANALYTICS_REPORTING_ENABLED_CONTEXT))
            && !this.nestedStackParent;
        this.synthesizer = (_b = props.synthesizer) !== null && _b !== void 0 ? _b : (newStyleSynthesisContext
            ? new stack_synthesizers_1.DefaultStackSynthesizer()
            : new stack_synthesizers_1.LegacyStackSynthesizer());
        this.synthesizer.bind(this);
    }
    /**
     * Return whether the given object is a Stack.
     *
     * We do attribute detection since we can't reliably use 'instanceof'.
     */
    static isStack(x) {
        return x !== null && typeof (x) === 'object' && STACK_SYMBOL in x;
    }
    /**
     * Looks up the first stack scope in which `construct` is defined. Fails if there is no stack up the tree.
     * @param construct The construct to start the search from.
     */
    static of(construct) {
        // we want this to be as cheap as possible. cache this result by mutating
        // the object. anecdotally, at the time of this writing, @aws-cdk/core unit
        // tests hit this cache 1,112 times, @aws-cdk/aws-cloudformation unit tests
        // hit this 2,435 times).
        const cache = construct[MY_STACK_CACHE];
        if (cache) {
            return cache;
        }
        else {
            const value = _lookup(construct);
            Object.defineProperty(construct, MY_STACK_CACHE, {
                enumerable: false,
                writable: false,
                configurable: false,
                value,
            });
            return value;
        }
        function _lookup(c) {
            var _a, _b;
            if (Stack.isStack(c)) {
                return c;
            }
            const _scope = constructs_1.Node.of(c).scope;
            if (stage_1.Stage.isStage(c) || !_scope) {
                throw new Error(`${(_b = (_a = construct.constructor) === null || _a === void 0 ? void 0 : _a.name) !== null && _b !== void 0 ? _b : 'Construct'} at '${constructs_1.Node.of(construct).path}' should be created in the scope of a Stack, but no Stack found`);
            }
            return _lookup(_scope);
        }
    }
    /**
     * Resolve a tokenized value in the context of the current stack.
     */
    resolve(obj) {
        return resolve_1.resolve(obj, {
            scope: this,
            prefix: [],
            resolver: cloudformation_lang_1.CLOUDFORMATION_TOKEN_RESOLVER,
            preparing: false,
        });
    }
    /**
     * Convert an object, potentially containing tokens, to a JSON string
     */
    toJsonString(obj, space) {
        return cloudformation_lang_1.CloudFormationLang.toJSON(obj, space).toString();
    }
    /**
     * Indicate that a context key was expected
     *
     * Contains instructions which will be emitted into the cloud assembly on how
     * the key should be supplied.
     *
     * @param report The set of parameters needed to obtain the context
     */
    reportMissingContext(report) {
        if (!Object.values(cxschema.ContextProvider).includes(report.provider)) {
            throw new Error(`Unknown context provider requested in: ${JSON.stringify(report)}`);
        }
        this._missingContext.push(report);
    }
    /**
     * Rename a generated logical identities
     *
     * To modify the naming scheme strategy, extend the `Stack` class and
     * override the `allocateLogicalId` method.
     */
    renameLogicalId(oldId, newId) {
        this._logicalIds.addRename(oldId, newId);
    }
    /**
     * Allocates a stack-unique CloudFormation-compatible logical identity for a
     * specific resource.
     *
     * This method is called when a `CfnElement` is created and used to render the
     * initial logical identity of resources. Logical ID renames are applied at
     * this stage.
     *
     * This method uses the protected method `allocateLogicalId` to render the
     * logical ID for an element. To modify the naming scheme, extend the `Stack`
     * class and override this method.
     *
     * @param element The CloudFormation element for which a logical identity is
     * needed.
     */
    getLogicalId(element) {
        const logicalId = this.allocateLogicalId(element);
        return this._logicalIds.applyRename(logicalId);
    }
    /**
     * Add a dependency between this stack and another stack.
     *
     * This can be used to define dependencies between any two stacks within an
     * app, and also supports nested stacks.
     */
    addDependency(target, reason) {
        deps_1.addDependency(this, target, reason);
    }
    /**
     * Return the stacks this stack depends on
     */
    get dependencies() {
        return Object.values(this._stackDependencies).map(x => x.stack);
    }
    /**
     * The concrete CloudFormation physical stack name.
     *
     * This is either the name defined explicitly in the `stackName` prop or
     * allocated based on the stack's location in the construct tree. Stacks that
     * are directly defined under the app use their construct `id` as their stack
     * name. Stacks that are defined deeper within the tree will use a hashed naming
     * scheme based on the construct path to ensure uniqueness.
     *
     * If you wish to obtain the deploy-time AWS::StackName intrinsic,
     * you can use `Aws.stackName` directly.
     */
    get stackName() {
        return this._stackName;
    }
    /**
     * The partition in which this stack is defined
     */
    get partition() {
        // Always return a non-scoped partition intrinsic. These will usually
        // be used to construct an ARN, but there are no cross-partition
        // calls anyway.
        return cfn_pseudo_1.Aws.PARTITION;
    }
    /**
     * The Amazon domain suffix for the region in which this stack is defined
     */
    get urlSuffix() {
        // Since URL Suffix always follows partition, it is unscoped like partition is.
        return cfn_pseudo_1.Aws.URL_SUFFIX;
    }
    /**
     * The ID of the stack
     *
     * @example After resolving, looks like arn:aws:cloudformation:us-west-2:123456789012:stack/teststack/51af3dc0-da77-11e4-872e-1234567db123
     */
    get stackId() {
        return new cfn_pseudo_1.ScopedAws(this).stackId;
    }
    /**
     * Returns the list of notification Amazon Resource Names (ARNs) for the current stack.
     */
    get notificationArns() {
        return new cfn_pseudo_1.ScopedAws(this).notificationArns;
    }
    /**
     * Indicates if this is a nested stack, in which case `parentStack` will include a reference to it's parent.
     */
    get nested() {
        return this.nestedStackResource !== undefined;
    }
    /**
     * Creates an ARN from components.
     *
     * If `partition`, `region` or `account` are not specified, the stack's
     * partition, region and account will be used.
     *
     * If any component is the empty string, an empty string will be inserted
     * into the generated ARN at the location that component corresponds to.
     *
     * The ARN will be formatted as follows:
     *
     *   arn:{partition}:{service}:{region}:{account}:{resource}{sep}}{resource-name}
     *
     * The required ARN pieces that are omitted will be taken from the stack that
     * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
     * can be 'undefined'.
     */
    formatArn(components) {
        return arn_1.Arn.format(components, this);
    }
    /**
     * Given an ARN, parses it and returns components.
     *
     * If the ARN is a concrete string, it will be parsed and validated. The
     * separator (`sep`) will be set to '/' if the 6th component includes a '/',
     * in which case, `resource` will be set to the value before the '/' and
     * `resourceName` will be the rest. In case there is no '/', `resource` will
     * be set to the 6th components and `resourceName` will be set to the rest
     * of the string.
     *
     * If the ARN includes tokens (or is a token), the ARN cannot be validated,
     * since we don't have the actual value yet at the time of this function
     * call. You will have to know the separator and the type of ARN. The
     * resulting `ArnComponents` object will contain tokens for the
     * subexpressions of the ARN, not string literals. In this case this
     * function cannot properly parse the complete final resourceName (path) out
     * of ARNs that use '/' to both separate the 'resource' from the
     * 'resourceName' AND to subdivide the resourceName further. For example, in
     * S3 ARNs:
     *
     *    arn:aws:s3:::my_corporate_bucket/path/to/exampleobject.png
     *
     * After parsing the resourceName will not contain
     * 'path/to/exampleobject.png' but simply 'path'. This is a limitation
     * because there is no slicing functionality in CloudFormation templates.
     *
     * @param arn The ARN string to parse
     * @param sepIfToken The separator used to separate resource from resourceName
     * @param hasName Whether there is a name component in the ARN at all. For
     * example, SNS Topics ARNs have the 'resource' component contain the topic
     * name, and no 'resourceName' component.
     *
     * @returns an ArnComponents object which allows access to the various
     * components of the ARN.
     *
     * @returns an ArnComponents object which allows access to the various
     *      components of the ARN.
     */
    parseArn(arn, sepIfToken = '/', hasName = true) {
        return arn_1.Arn.parse(arn, sepIfToken, hasName);
    }
    /**
     * Returns the list of AZs that are available in the AWS environment
     * (account/region) associated with this stack.
     *
     * If the stack is environment-agnostic (either account and/or region are
     * tokens), this property will return an array with 2 tokens that will resolve
     * at deploy-time to the first two availability zones returned from CloudFormation's
     * `Fn::GetAZs` intrinsic function.
     *
     * If they are not available in the context, returns a set of dummy values and
     * reports them as missing, and let the CLI resolve them by calling EC2
     * `DescribeAvailabilityZones` on the target environment.
     *
     * To specify a different strategy for selecting availability zones override this method.
     */
    get availabilityZones() {
        // if account/region are tokens, we can't obtain AZs through the context
        // provider, so we fallback to use Fn::GetAZs. the current lowest common
        // denominator is 2 AZs across all AWS regions.
        const agnostic = token_1.Token.isUnresolved(this.account) || token_1.Token.isUnresolved(this.region);
        if (agnostic) {
            return this.node.tryGetContext(cxapi.AVAILABILITY_ZONE_FALLBACK_CONTEXT_KEY) || [
                cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.getAzs()),
                cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.getAzs()),
            ];
        }
        const value = context_provider_1.ContextProvider.getValue(this, {
            provider: cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
            dummyValue: ['dummy1a', 'dummy1b', 'dummy1c'],
        }).value;
        if (!Array.isArray(value)) {
            throw new Error(`Provider ${cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER} expects a list`);
        }
        return value;
    }
    /**
     * Register a file asset on this Stack
     *
     * @deprecated Use `stack.synthesizer.addFileAsset()` if you are calling,
     * and a different IStackSynthesizer class if you are implementing.
     */
    addFileAsset(asset) {
        return this.synthesizer.addFileAsset(asset);
    }
    /**
     * Register a docker image asset on this Stack
     *
     * @deprecated Use `stack.synthesizer.addDockerImageAsset()` if you are calling,
     * and a different `IStackSynthesizer` class if you are implementing.
     */
    addDockerImageAsset(asset) {
        return this.synthesizer.addDockerImageAsset(asset);
    }
    /**
     * If this is a nested stack, returns it's parent stack.
     */
    get nestedStackParent() {
        return this.nestedStackResource && Stack.of(this.nestedStackResource);
    }
    /**
     * Returns the parent of a nested stack.
     *
     * @deprecated use `nestedStackParent`
     */
    get parentStack() {
        return this.nestedStackParent;
    }
    /**
     * Add a Transform to this stack. A Transform is a macro that AWS
     * CloudFormation uses to process your template.
     *
     * Duplicate values are removed when stack is synthesized.
     *
     * @example addTransform('AWS::Serverless-2016-10-31')
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-section-structure.html
     *
     * @param transform The transform to add
     */
    addTransform(transform) {
        if (!this.templateOptions.transforms) {
            this.templateOptions.transforms = [];
        }
        this.templateOptions.transforms.push(transform);
    }
    /**
     * Called implicitly by the `addDependency` helper function in order to
     * realize a dependency between two top-level stacks at the assembly level.
     *
     * Use `stack.addDependency` to define the dependency between any two stacks,
     * and take into account nested stack relationships.
     *
     * @internal
     */
    _addAssemblyDependency(target, reason) {
        // defensive: we should never get here for nested stacks
        if (this.nested || target.nested) {
            throw new Error('Cannot add assembly-level dependencies for nested stacks');
        }
        reason = reason || 'dependency added using stack.addDependency()';
        const cycle = target.stackDependencyReasons(this);
        if (cycle !== undefined) {
            // eslint-disable-next-line max-len
            throw new Error(`'${target.node.path}' depends on '${this.node.path}' (${cycle.join(', ')}). Adding this dependency (${reason}) would create a cyclic reference.`);
        }
        let dep = this._stackDependencies[target.node.uniqueId];
        if (!dep) {
            dep = this._stackDependencies[target.node.uniqueId] = {
                stack: target,
                reasons: [],
            };
        }
        dep.reasons.push(reason);
        if (process.env.CDK_DEBUG_DEPS) {
            // eslint-disable-next-line no-console
            console.error(`[CDK_DEBUG_DEPS] stack "${this.node.path}" depends on "${target.node.path}" because: ${reason}`);
        }
    }
    /**
     * Synthesizes the cloudformation template into a cloud assembly.
     * @internal
     */
    _synthesizeTemplate(session) {
        // In principle, stack synthesis is delegated to the
        // StackSynthesis object.
        //
        // However, some parts of synthesis currently use some private
        // methods on Stack, and I don't really see the value in refactoring
        // this right now, so some parts still happen here.
        const builder = session.assembly;
        const template = this._toCloudFormation();
        // write the CloudFormation template as a JSON file
        const outPath = path.join(builder.outdir, this.templateFile);
        fs.writeFileSync(outPath, JSON.stringify(template, undefined, 2));
        for (const ctx of this._missingContext) {
            builder.addMissing(ctx);
        }
    }
    /**
     * Returns the naming scheme used to allocate logical IDs. By default, uses
     * the `HashedAddressingScheme` but this method can be overridden to customize
     * this behavior.
     *
     * In order to make sure logical IDs are unique and stable, we hash the resource
     * construct tree path (i.e. toplevel/secondlevel/.../myresource) and add it as
     * a suffix to the path components joined without a separator (CloudFormation
     * IDs only allow alphanumeric characters).
     *
     * The result will be:
     *
     *   <path.join('')><md5(path.join('/')>
     *     "human"      "hash"
     *
     * If the "human" part of the ID exceeds 240 characters, we simply trim it so
     * the total ID doesn't exceed CloudFormation's 255 character limit.
     *
     * We only take 8 characters from the md5 hash (0.000005 chance of collision).
     *
     * Special cases:
     *
     * - If the path only contains a single component (i.e. it's a top-level
     *   resource), we won't add the hash to it. The hash is not needed for
     *   disamiguation and also, it allows for a more straightforward migration an
     *   existing CloudFormation template to a CDK stack without logical ID changes
     *   (or renames).
     * - For aesthetic reasons, if the last components of the path are the same
     *   (i.e. `L1/L2/Pipeline/Pipeline`), they will be de-duplicated to make the
     *   resulting human portion of the ID more pleasing: `L1L2Pipeline<HASH>`
     *   instead of `L1L2PipelinePipeline<HASH>`
     * - If a component is named "Default" it will be omitted from the path. This
     *   allows refactoring higher level abstractions around constructs without affecting
     *   the IDs of already deployed resources.
     * - If a component is named "Resource" it will be omitted from the user-visible
     *   path, but included in the hash. This reduces visual noise in the human readable
     *   part of the identifier.
     *
     * @param cfnElement The element for which the logical ID is allocated.
     */
    allocateLogicalId(cfnElement) {
        const scopes = cfnElement.node.scopes;
        const stackIndex = scopes.indexOf(cfnElement.stack);
        const pathComponents = scopes.slice(stackIndex + 1).map(x => x.node.id);
        return uniqueid_1.makeUniqueId(pathComponents);
    }
    /**
     * Validate stack name
     *
     * CloudFormation stack names can include dashes in addition to the regular identifier
     * character classes, and we don't allow one of the magic markers.
     *
     * @internal
     */
    _validateId(name) {
        if (name && !VALID_STACK_NAME_REGEX.test(name)) {
            throw new Error(`Stack name must match the regular expression: ${VALID_STACK_NAME_REGEX.toString()}, got '${name}'`);
        }
    }
    /**
     * Returns the CloudFormation template for this stack by traversing
     * the tree and invoking _toCloudFormation() on all Entity objects.
     *
     * @internal
     */
    _toCloudFormation() {
        let transform;
        if (this.templateOptions.transform) {
            // eslint-disable-next-line max-len
            annotations_1.Annotations.of(this).addWarning('This stack is using the deprecated `templateOptions.transform` property. Consider switching to `addTransform()`.');
            this.addTransform(this.templateOptions.transform);
        }
        if (this.templateOptions.transforms) {
            if (this.templateOptions.transforms.length === 1) { // Extract single value
                transform = this.templateOptions.transforms[0];
            }
            else { // Remove duplicate values
                transform = Array.from(new Set(this.templateOptions.transforms));
            }
        }
        const template = {
            Description: this.templateOptions.description,
            Transform: transform,
            AWSTemplateFormatVersion: this.templateOptions.templateFormatVersion,
            Metadata: this.templateOptions.metadata,
        };
        const elements = cfnElements(this);
        const fragments = elements.map(e => this.resolve(e._toCloudFormation()));
        // merge in all CloudFormation fragments collected from the tree
        for (const fragment of fragments) {
            merge(template, fragment);
        }
        // resolve all tokens and remove all empties
        const ret = this.resolve(template) || {};
        this._logicalIds.assertAllRenamesApplied();
        return ret;
    }
    /**
     * Deprecated.
     *
     * @see https://github.com/aws/aws-cdk/pull/7187
     * @returns reference itself without any change
     * @deprecated cross reference handling has been moved to `App.prepare()`.
     */
    prepareCrossReference(_sourceStack, reference) {
        return reference;
    }
    /**
     * Determine the various stack environment attributes.
     *
     */
    parseEnvironment(env = {}) {
        var _a, _b, _c, _d;
        // if an environment property is explicitly specified when the stack is
        // created, it will be used. if not, use tokens for account and region.
        //
        // (They do not need to be anchored to any construct like resource attributes
        // are, because we'll never Export/Fn::ImportValue them -- the only situation
        // in which Export/Fn::ImportValue would work is if the value are the same
        // between producer and consumer anyway, so we can just assume that they are).
        const containingAssembly = stage_1.Stage.of(this);
        const account = (_b = (_a = env.account) !== null && _a !== void 0 ? _a : containingAssembly === null || containingAssembly === void 0 ? void 0 : containingAssembly.account) !== null && _b !== void 0 ? _b : cfn_pseudo_1.Aws.ACCOUNT_ID;
        const region = (_d = (_c = env.region) !== null && _c !== void 0 ? _c : containingAssembly === null || containingAssembly === void 0 ? void 0 : containingAssembly.region) !== null && _d !== void 0 ? _d : cfn_pseudo_1.Aws.REGION;
        // this is the "aws://" env specification that will be written to the cloud assembly
        // manifest. it will use "unknown-account" and "unknown-region" to indicate
        // environment-agnosticness.
        const envAccount = !token_1.Token.isUnresolved(account) ? account : cxapi.UNKNOWN_ACCOUNT;
        const envRegion = !token_1.Token.isUnresolved(region) ? region : cxapi.UNKNOWN_REGION;
        return {
            account,
            region,
            environment: cxapi.EnvironmentUtils.format(envAccount, envRegion),
        };
    }
    /**
     * Check whether this stack has a (transitive) dependency on another stack
     *
     * Returns the list of reasons on the dependency path, or undefined
     * if there is no dependency.
     */
    stackDependencyReasons(other) {
        if (this === other) {
            return [];
        }
        for (const dep of Object.values(this._stackDependencies)) {
            const ret = dep.stack.stackDependencyReasons(other);
            if (ret !== undefined) {
                return [...dep.reasons, ...ret];
            }
        }
        return undefined;
    }
    /**
     * Calculate the stack name based on the construct path
     *
     * The stack name is the name under which we'll deploy the stack,
     * and incorporates containing Stage names by default.
     *
     * Generally this looks a lot like how logical IDs are calculated.
     * The stack name is calculated based on the construct root path,
     * as follows:
     *
     * - Path is calculated with respect to containing App or Stage (if any)
     * - If the path is one component long just use that component, otherwise
     *   combine them with a hash.
     *
     * Since the hash is quite ugly and we'd like to avoid it if possible -- but
     * we can't anymore in the general case since it has been written into legacy
     * stacks. The introduction of Stages makes it possible to make this nicer however.
     * When a Stack is nested inside a Stage, we use the path components below the
     * Stage, and prefix the path components of the Stage before it.
     */
    generateStackName() {
        const assembly = stage_1.Stage.of(this);
        const prefix = (assembly && assembly.stageName) ? `${assembly.stageName}-` : '';
        return `${prefix}${this.generateStackId(assembly)}`;
    }
    /**
     * The artifact ID for this stack
     *
     * Stack artifact ID is unique within the App's Cloud Assembly.
     */
    generateStackArtifactId() {
        return this.generateStackId(this.node.root);
    }
    /**
     * Generate an ID with respect to the given container construct.
     */
    generateStackId(container) {
        const rootPath = rootPathTo(this, container);
        const ids = rootPath.map(c => constructs_1.Node.of(c).id);
        // In unit tests our Stack (which is the only component) may not have an
        // id, so in that case just pretend it's "Stack".
        if (ids.length === 1 && !ids[0]) {
            throw new Error('unexpected: stack id must always be defined');
        }
        return makeStackName(ids);
    }
}
exports.Stack = Stack;
function merge(template, fragment) {
    for (const section of Object.keys(fragment)) {
        const src = fragment[section];
        // create top-level section if it doesn't exist
        const dest = template[section];
        if (!dest) {
            template[section] = src;
        }
        else {
            template[section] = mergeSection(section, dest, src);
        }
    }
}
function mergeSection(section, val1, val2) {
    switch (section) {
        case 'Description':
            return `${val1}\n${val2}`;
        case 'AWSTemplateFormatVersion':
            if (val1 != null && val2 != null && val1 !== val2) {
                throw new Error(`Conflicting CloudFormation template versions provided: '${val1}' and '${val2}`);
            }
            return val1 !== null && val1 !== void 0 ? val1 : val2;
        case 'Transform':
            return mergeSets(val1, val2);
        default:
            return mergeObjectsWithoutDuplicates(section, val1, val2);
    }
}
function mergeSets(val1, val2) {
    const array1 = val1 == null ? [] : (Array.isArray(val1) ? val1 : [val1]);
    const array2 = val2 == null ? [] : (Array.isArray(val2) ? val2 : [val2]);
    for (const value of array2) {
        if (!array1.includes(value)) {
            array1.push(value);
        }
    }
    return array1.length === 1 ? array1[0] : array1;
}
function mergeObjectsWithoutDuplicates(section, dest, src) {
    if (typeof dest !== 'object') {
        throw new Error(`Expecting ${JSON.stringify(dest)} to be an object`);
    }
    if (typeof src !== 'object') {
        throw new Error(`Expecting ${JSON.stringify(src)} to be an object`);
    }
    // add all entities from source section to destination section
    for (const id of Object.keys(src)) {
        if (id in dest) {
            throw new Error(`section '${section}' already contains '${id}'`);
        }
        dest[id] = src[id];
    }
    return dest;
}
/**
 * Collect all CfnElements from a Stack.
 *
 * @param node Root node to collect all CfnElements from
 * @param into Array to append CfnElements to
 * @returns The same array as is being collected into
 */
function cfnElements(node, into = []) {
    if (cfn_element_1.CfnElement.isCfnElement(node)) {
        into.push(node);
    }
    for (const child of constructs_1.Node.of(node).children) {
        // Don't recurse into a substack
        if (Stack.isStack(child)) {
            continue;
        }
        cfnElements(child, into);
    }
    return into;
}
/**
 * Return the construct root path of the given construct relative to the given ancestor
 *
 * If no ancestor is given or the ancestor is not found, return the entire root path.
 */
function rootPathTo(construct, ancestor) {
    const scopes = constructs_1.Node.of(construct).scopes;
    for (let i = scopes.length - 2; i >= 0; i--) {
        if (scopes[i] === ancestor) {
            return scopes.slice(i + 1);
        }
    }
    return scopes;
}
exports.rootPathTo = rootPathTo;
/**
 * makeUniqueId, specialized for Stack names
 *
 * Stack names may contain '-', so we allow that character if the stack name
 * has only one component. Otherwise we fall back to the regular "makeUniqueId"
 * behavior.
 */
function makeStackName(components) {
    if (components.length === 1) {
        return components[0];
    }
    return uniqueid_1.makeUniqueId(components);
}
// These imports have to be at the end to prevent circular imports
const deps_1 = require("./deps");
const stack_synthesizers_1 = require("./stack-synthesizers");
const stage_1 = require("./stage");
const tag_manager_1 = require("./tag-manager");
const token_1 = require("./token");
const fs_1 = require("./fs");
//# sourceMappingURL=data:application/json;base64,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