"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeployCdkStackAction = void 0;
const fs = require("fs");
const path = require("path");
const cfn = require("../../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const cpactions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const cxapi = require("../../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const construct_internals_1 = require("../private/construct-internals");
/**
 * Action to deploy a CDK Stack
 *
 * Adds two CodePipeline Actions to the pipeline: one to create a ChangeSet
 * and one to execute it.
 *
 * You do not need to instantiate this action yourself -- it will automatically
 * be added by the pipeline when you add stack artifacts or entire stages.
 */
class DeployCdkStackAction {
    constructor(props) {
        var _a, _b, _c, _d, _e;
        if (props.output && !props.outputFileName) {
            throw new Error('If \'output\' is set, \'outputFileName\' is also required');
        }
        this.stackArtifactId = props.stackArtifactId;
        this.dependencyStackArtifactIds = (_a = props.dependencyStackArtifactIds) !== null && _a !== void 0 ? _a : [];
        this.prepareRunOrder = (_b = props.prepareRunOrder) !== null && _b !== void 0 ? _b : 1;
        this.executeRunOrder = (_c = props.executeRunOrder) !== null && _c !== void 0 ? _c : this.prepareRunOrder + 1;
        this.stackName = props.stackName;
        const baseActionName = (_d = props.baseActionName) !== null && _d !== void 0 ? _d : this.stackName;
        const changeSetName = (_e = props.changeSetName) !== null && _e !== void 0 ? _e : 'PipelineChange';
        this.prepareChangeSetAction = new cpactions.CloudFormationCreateReplaceChangeSetAction({
            actionName: `${baseActionName}.Prepare`,
            changeSetName,
            runOrder: this.prepareRunOrder,
            stackName: this.stackName,
            templatePath: props.cloudAssemblyInput.atPath(props.templatePath),
            adminPermissions: false,
            role: props.actionRole,
            deploymentRole: props.cloudFormationExecutionRole,
            region: props.region,
            capabilities: [cfn.CloudFormationCapabilities.NAMED_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND],
            templateConfiguration: props.templateConfigurationPath ? props.cloudAssemblyInput.atPath(props.templateConfigurationPath) : undefined,
        });
        this.executeChangeSetAction = new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: `${baseActionName}.Deploy`,
            changeSetName,
            runOrder: this.executeRunOrder,
            stackName: this.stackName,
            role: props.actionRole,
            region: props.region,
            outputFileName: props.outputFileName,
            output: props.output,
        });
    }
    /**
     * Construct a DeployCdkStackAction from a Stack artifact
     */
    static fromStackArtifact(scope, artifact, options) {
        var _a;
        if (!artifact.assumeRoleArn) {
            throw new Error(`Stack '${artifact.stackName}' does not have deployment role information; use the 'DefaultStackSynthesizer' synthesizer, or set the '@aws-cdk/core:newStyleStackSynthesis' context key.`);
        }
        const artRegion = artifact.environment.region;
        const region = artRegion === core_1.Stack.of(scope).region || artRegion === cxapi.UNKNOWN_REGION ? undefined : artRegion;
        const artAccount = artifact.environment.account;
        const account = artAccount === core_1.Stack.of(scope).account || artAccount === cxapi.UNKNOWN_ACCOUNT ? undefined : artAccount;
        const actionRole = roleFromPlaceholderArn(scope, region, account, artifact.assumeRoleArn);
        const cloudFormationExecutionRole = roleFromPlaceholderArn(scope, region, account, artifact.cloudFormationExecutionRoleArn);
        // We need the path of the template relative to the root Cloud Assembly
        // It should be easier to get this, but for now it is what it is.
        const appAsmRoot = construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(scope)).outdir;
        const fullTemplatePath = path.join(artifact.assembly.directory, artifact.templateFile);
        let fullConfigPath;
        if (Object.keys(artifact.tags).length > 0) {
            fullConfigPath = `${fullTemplatePath}.config.json`;
            // Write the template configuration file (for parameters into CreateChangeSet call that
            // cannot be configured any other way). They must come from a file, and there's unfortunately
            // no better hook to write this file (`construct.onSynthesize()` would have been the prime candidate
            // but that is being deprecated--and DeployCdkStackAction isn't even a construct).
            writeTemplateConfiguration(fullConfigPath, {
                Tags: artifact.tags,
            });
        }
        return new DeployCdkStackAction({
            actionRole,
            cloudFormationExecutionRole,
            templatePath: path.relative(appAsmRoot, fullTemplatePath),
            templateConfigurationPath: fullConfigPath ? path.relative(appAsmRoot, fullConfigPath) : undefined,
            region,
            stackArtifactId: artifact.id,
            dependencyStackArtifactIds: artifact.dependencies.filter(isStackArtifact).map(s => s.id),
            stackName: (_a = options.stackName) !== null && _a !== void 0 ? _a : artifact.stackName,
            ...options,
        });
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        stage.addAction(this.prepareChangeSetAction);
        return this.executeChangeSetAction.bind(scope, stage, options);
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        return this.executeChangeSetAction.onStateChange(name, target, options);
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this.executeChangeSetAction.actionProperties;
    }
}
exports.DeployCdkStackAction = DeployCdkStackAction;
function roleFromPlaceholderArn(scope, region, account, arn) {
    if (!arn) {
        return undefined;
    }
    // Use placeholdered arn as construct ID.
    const id = arn;
    // https://github.com/aws/aws-cdk/issues/7255
    let existingRole = scope.node.tryFindChild(`ImmutableRole${id}`);
    if (existingRole) {
        return existingRole;
    }
    // For when #7255 is fixed.
    existingRole = scope.node.tryFindChild(id);
    if (existingRole) {
        return existingRole;
    }
    const arnToImport = cxapi.EnvironmentPlaceholders.replace(arn, {
        region: region !== null && region !== void 0 ? region : core_1.Aws.REGION,
        accountId: account !== null && account !== void 0 ? account : core_1.Aws.ACCOUNT_ID,
        partition: core_1.Aws.PARTITION,
    });
    return iam.Role.fromRoleArn(scope, id, arnToImport, { mutable: false });
}
function isStackArtifact(a) {
    // instanceof is too risky, and we're at a too late stage to properly fix.
    // return a instanceof cxapi.CloudFormationStackArtifact;
    return a.constructor.name === 'CloudFormationStackArtifact';
}
/**
 * Write template configuration to the given file
 */
function writeTemplateConfiguration(filename, config) {
    fs.writeFileSync(filename, JSON.stringify(config, undefined, 2), { encoding: 'utf-8' });
}
//# sourceMappingURL=data:application/json;base64,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