"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PublishAssetsAction = exports.AssetType = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Type of the asset that is being published
 */
var AssetType;
(function (AssetType) {
    /**
     * A file
     */
    AssetType["FILE"] = "file";
    /**
     * A Docker image
     */
    AssetType["DOCKER_IMAGE"] = "docker-image";
})(AssetType = exports.AssetType || (exports.AssetType = {}));
/**
 * Action to publish an asset in the pipeline
 *
 * Creates a CodeBuild project which will use the CDK CLI
 * to prepare and publish the asset.
 *
 * You do not need to instantiate this action -- it will automatically
 * be added by the pipeline when you add stacks that use assets.
 */
class PublishAssetsAction extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.commands = new Array();
        const installSuffix = props.cdkCliVersion ? `@${props.cdkCliVersion}` : '';
        const project = new codebuild.PipelineProject(this, 'Default', {
            projectName: this.props.projectName,
            environment: {
                buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
                privileged: (props.assetType === AssetType.DOCKER_IMAGE) ? true : undefined,
            },
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: `npm install -g cdk-assets${installSuffix}`,
                    },
                    build: {
                        commands: core_1.Lazy.listValue({ produce: () => this.commands }),
                    },
                },
            }),
            role: props.role,
        });
        const rolePattern = props.assetType === AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        project.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        this.action = new codepipeline_actions.CodeBuildAction({
            actionName: props.actionName,
            project,
            input: this.props.cloudAssemblyInput,
            role: props.role,
        });
    }
    /**
     * Add a single publishing command
     *
     * Manifest path should be relative to the root Cloud Assembly.
     */
    addPublishCommand(relativeManifestPath, assetSelector) {
        const command = `cdk-assets --path "${relativeManifestPath}" --verbose publish "${assetSelector}"`;
        if (!this.commands.includes(command)) {
            this.commands.push(command);
        }
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        return this.action.bind(scope, stage, options);
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        return this.action.onStateChange(name, target, options);
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        // FIXME: I have had to make this class a Construct, because:
        //
        // - It needs access to the Construct tree, because it is going to add a `PipelineProject`.
        // - I would have liked to have done that in bind(), however,
        // - `actionProperties` (this method) is called BEFORE bind() is called, and by that point I
        //   don't have the "inner" Action yet to forward the call to.
        //
        // I've therefore had to construct the inner CodeBuildAction in the constructor, which requires making this
        // Action a Construct.
        //
        // Combined with how non-intuitive it is to make the "StackDeployAction", I feel there is something
        // wrong with the Action abstraction here.
        return this.action.actionProperties;
    }
}
exports.PublishAssetsAction = PublishAssetsAction;
//# sourceMappingURL=data:application/json;base64,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