"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackOutput = exports.CdkStage = void 0;
const cpactions = require("../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const actions_1 = require("./actions");
const asset_manifest_1 = require("./private/asset-manifest");
const toposort_1 = require("./private/toposort");
/**
 * Stage in a CdkPipeline
 *
 * You don't need to instantiate this class directly. Use
 * `cdkPipeline.addStage()` instead.
 */
class CdkStage extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._nextSequentialRunOrder = 1; // Must start at 1 eh
        this._manualApprovalCounter = 1;
        this.stacksToDeploy = new Array();
        this._prepared = false;
        this.stageName = props.stageName;
        this.pipelineStage = props.pipelineStage;
        this.cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        this.host = props.host;
        core_1.Aspects.of(this).add({ visit: () => this.prepareStage() });
    }
    /**
     * Add all stacks in the application Stage to this stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplication(appStage, options = {}) {
        const asm = appStage.synth();
        if (asm.stacks.length === 0) {
            // If we don't check here, a more puzzling "stage contains no actions"
            // error will be thrown come deployment time.
            throw new Error(`The given Stage construct ('${appStage.node.path}') should contain at least one Stack`);
        }
        const sortedTranches = toposort_1.topologicalSort(asm.stacks, stack => stack.id, stack => stack.dependencies.map(d => d.id));
        for (const stacks of sortedTranches) {
            const runOrder = this.nextSequentialRunOrder(2); // We need 2 actions
            let executeRunOrder = runOrder + 1;
            // If we need to insert a manual approval action, then what's the executeRunOrder
            // now is where we add a manual approval step, and we allocate 1 more runOrder
            // for the execute.
            if (options.manualApprovals) {
                this.addManualApprovalAction({ runOrder: executeRunOrder });
                executeRunOrder = this.nextSequentialRunOrder();
            }
            // These don't have a dependency on each other, so can all be added in parallel
            for (const stack of stacks) {
                this.addStackArtifactDeployment(stack, { runOrder, executeRunOrder });
            }
        }
    }
    /**
     * Add a deployment action based on a stack artifact
     */
    addStackArtifactDeployment(stackArtifact, options = {}) {
        var _a, _b;
        // Get all assets manifests and add the assets in 'em to the asset publishing stage.
        this.publishAssetDependencies(stackArtifact);
        // Remember for later, see 'prepare()'
        // We know that deploying a stack is going to take up 2 runorder slots later on.
        const runOrder = (_a = options.runOrder) !== null && _a !== void 0 ? _a : this.nextSequentialRunOrder(2);
        const executeRunOrder = (_b = options.executeRunOrder) !== null && _b !== void 0 ? _b : runOrder + 1;
        this.stacksToDeploy.push({
            prepareRunOrder: runOrder,
            executeRunOrder,
            stackArtifact,
        });
        this.advanceRunOrderPast(runOrder);
        this.advanceRunOrderPast(executeRunOrder);
    }
    /**
     * Add a manual approval action
     *
     * If you need more flexibility than what this method offers,
     * use `addAction` with a `ManualApprovalAction`.
     */
    addManualApprovalAction(options = {}) {
        var _a;
        let actionName = options.actionName;
        if (!actionName) {
            actionName = `ManualApproval${this._manualApprovalCounter > 1 ? this._manualApprovalCounter : ''}`;
            this._manualApprovalCounter += 1;
        }
        this.addActions(new cpactions.ManualApprovalAction({
            actionName,
            runOrder: (_a = options.runOrder) !== null && _a !== void 0 ? _a : this.nextSequentialRunOrder(),
        }));
    }
    /**
     * Add one or more CodePipeline Actions
     *
     * You need to make sure it is created with the right runOrder. Call `nextSequentialRunOrder()`
     * for every action to get actions to execute in sequence.
     */
    addActions(...actions) {
        for (const action of actions) {
            this.pipelineStage.addAction(action);
        }
    }
    /**
     * Return the runOrder number necessary to run the next Action in sequence with the rest
     *
     * FIXME: This is here because Actions are immutable and can't be reordered
     * after creation, nor is there a way to specify relative priorities, which
     * is a limitation that we should take away in the base library.
     */
    nextSequentialRunOrder(count = 1) {
        const ret = this._nextSequentialRunOrder;
        this._nextSequentialRunOrder += count;
        return ret;
    }
    /**
     * Whether this Stage contains an action to deploy the given stack, identified by its artifact ID
     */
    deploysStack(artifactId) {
        return this.stacksToDeploy.map(s => s.stackArtifact.id).includes(artifactId);
    }
    /**
     * Actually add all the DeployStack actions to the stage.
     *
     * We do this late because before we can render the actual DeployActions,
     * we need to know whether or not we need to capture the stack outputs.
     *
     * FIXME: This is here because Actions are immutable and can't be reordered
     * after creation, nor is there a way to specify relative priorities, which
     * is a limitation that we should take away in the base library.
     */
    prepareStage() {
        // FIXME: Make sure this only gets run once. There seems to be an issue in the reconciliation
        // loop that may trigger this more than once if it throws an error somewhere, and the exception
        // that gets thrown here will then override the actual failure.
        if (this._prepared) {
            return;
        }
        this._prepared = true;
        for (const { prepareRunOrder, stackArtifact, executeRunOrder } of this.stacksToDeploy) {
            const artifact = this.host.stackOutputArtifact(stackArtifact.id);
            this.pipelineStage.addAction(actions_1.DeployCdkStackAction.fromStackArtifact(this, stackArtifact, {
                baseActionName: this.simplifyStackName(stackArtifact.stackName),
                cloudAssemblyInput: this.cloudAssemblyArtifact,
                output: artifact,
                outputFileName: artifact ? 'outputs.json' : undefined,
                prepareRunOrder,
                executeRunOrder,
            }));
        }
    }
    /**
     * Advance the runorder counter so that the next sequential number is higher than the given one
     */
    advanceRunOrderPast(lastUsed) {
        this._nextSequentialRunOrder = Math.max(lastUsed + 1, this._nextSequentialRunOrder);
    }
    /**
     * Simplify the stack name by removing the `Stage-` prefix if it exists.
     */
    simplifyStackName(s) {
        return stripPrefix(s, `${this.stageName}-`);
    }
    /**
     * Make sure all assets depended on by this stack are published in this pipeline
     *
     * Taking care to exclude the stack template itself -- it is being published
     * as an asset because the CLI needs to know the asset publishing role when
     * pushing the template to S3, but in the case of CodePipeline we always
     * reference the template from the artifact bucket.
     *
     * (NOTE: this is only true for top-level stacks, not nested stacks. Nested
     * Stack templates are always published as assets).
     */
    publishAssetDependencies(stackArtifact) {
        const assetManifests = stackArtifact.dependencies.filter(isAssetManifest);
        for (const manifestArtifact of assetManifests) {
            const manifest = asset_manifest_1.AssetManifestReader.fromFile(manifestArtifact.file);
            for (const entry of manifest.entries) {
                let assetType;
                if (entry instanceof asset_manifest_1.DockerImageManifestEntry) {
                    assetType = actions_1.AssetType.DOCKER_IMAGE;
                }
                else if (entry instanceof asset_manifest_1.FileManifestEntry) {
                    // Don't publishg the template for this stack
                    if (entry.source.packaging === 'file' && entry.source.path === stackArtifact.templateFile) {
                        continue;
                    }
                    assetType = actions_1.AssetType.FILE;
                }
                else {
                    throw new Error(`Unrecognized asset type: ${entry.type}`);
                }
                this.host.publishAsset({
                    assetManifestPath: manifestArtifact.file,
                    assetId: entry.id.assetId,
                    assetSelector: entry.id.toString(),
                    assetType,
                });
            }
        }
    }
}
exports.CdkStage = CdkStage;
/**
 * A single output of a Stack
 */
class StackOutput {
    /**
     * Build a StackOutput from a known artifact and an output name
     */
    constructor(artifactFile, outputName) {
        this.artifactFile = artifactFile;
        this.outputName = outputName;
    }
}
exports.StackOutput = StackOutput;
function stripPrefix(s, prefix) {
    return s.startsWith(prefix) ? s.substr(prefix.length) : s;
}
function isAssetManifest(s) {
    // instanceof is too risky, and we're at a too late stage to properly fix.
    // return s instanceof cxapi.AssetManifestArtifact;
    return s.constructor.name === 'AssetManifestArtifact';
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhZ2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzdGFnZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSw0REFBNEQsQ0FBQyxvRUFBb0U7QUFDakkscUNBQXVELENBQUMsZ0RBQWdEO0FBRXhHLHVDQUE0RDtBQUM1RCw2REFBNEc7QUFDNUcsaURBQXFEO0FBc0JyRDs7Ozs7R0FLRztBQUNILE1BQWEsUUFBUyxTQUFRLGdCQUFTO0lBU25DLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBb0I7UUFDMUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQVRiLDRCQUF1QixHQUFHLENBQUMsQ0FBQyxDQUFDLHFCQUFxQjtRQUNsRCwyQkFBc0IsR0FBRyxDQUFDLENBQUM7UUFHbEIsbUJBQWMsR0FBRyxJQUFJLEtBQUssRUFBc0IsQ0FBQztRQUcxRCxjQUFTLEdBQUcsS0FBSyxDQUFDO1FBR3RCLElBQUksQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUNqQyxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUM7UUFDekMsSUFBSSxDQUFDLHFCQUFxQixHQUFHLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQztRQUN6RCxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFDdkIsY0FBTyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLENBQUMsRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLFlBQVksRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvRCxDQUFDO0lBQ0Q7Ozs7Ozs7Ozs7T0FVRztJQUNJLGNBQWMsQ0FBQyxRQUFlLEVBQUUsVUFBMkIsRUFBRTtRQUNoRSxNQUFNLEdBQUcsR0FBRyxRQUFRLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDN0IsSUFBSSxHQUFHLENBQUMsTUFBTSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDekIsc0VBQXNFO1lBQ3RFLDZDQUE2QztZQUM3QyxNQUFNLElBQUksS0FBSyxDQUFDLCtCQUErQixRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksc0NBQXNDLENBQUMsQ0FBQztTQUM1RztRQUNELE1BQU0sY0FBYyxHQUFHLDBCQUFlLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxFQUFFLEVBQUUsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ2xILEtBQUssTUFBTSxNQUFNLElBQUksY0FBYyxFQUFFO1lBQ2pDLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLG9CQUFvQjtZQUNyRSxJQUFJLGVBQWUsR0FBRyxRQUFRLEdBQUcsQ0FBQyxDQUFDO1lBQ25DLGlGQUFpRjtZQUNqRiw4RUFBOEU7WUFDOUUsbUJBQW1CO1lBQ25CLElBQUksT0FBTyxDQUFDLGVBQWUsRUFBRTtnQkFDekIsSUFBSSxDQUFDLHVCQUF1QixDQUFDLEVBQUUsUUFBUSxFQUFFLGVBQWUsRUFBRSxDQUFDLENBQUM7Z0JBQzVELGVBQWUsR0FBRyxJQUFJLENBQUMsc0JBQXNCLEVBQUUsQ0FBQzthQUNuRDtZQUNELCtFQUErRTtZQUMvRSxLQUFLLE1BQU0sS0FBSyxJQUFJLE1BQU0sRUFBRTtnQkFDeEIsSUFBSSxDQUFDLDBCQUEwQixDQUFDLEtBQUssRUFBRSxFQUFFLFFBQVEsRUFBRSxlQUFlLEVBQUUsQ0FBQyxDQUFDO2FBQ3pFO1NBQ0o7SUFDTCxDQUFDO0lBQ0Q7O09BRUc7SUFDSSwwQkFBMEIsQ0FBQyxhQUFnRCxFQUFFLFVBQTJCLEVBQUU7O1FBQzdHLG9GQUFvRjtRQUNwRixJQUFJLENBQUMsd0JBQXdCLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDN0Msc0NBQXNDO1FBQ3RDLGdGQUFnRjtRQUNoRixNQUFNLFFBQVEsU0FBRyxPQUFPLENBQUMsUUFBUSxtQ0FBSSxJQUFJLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDcEUsTUFBTSxlQUFlLFNBQUcsT0FBTyxDQUFDLGVBQWUsbUNBQUksUUFBUSxHQUFHLENBQUMsQ0FBQztRQUNoRSxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQztZQUNyQixlQUFlLEVBQUUsUUFBUTtZQUN6QixlQUFlO1lBQ2YsYUFBYTtTQUNoQixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsbUJBQW1CLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDbkMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFDRDs7Ozs7T0FLRztJQUNJLHVCQUF1QixDQUFDLFVBQW9DLEVBQUU7O1FBQ2pFLElBQUksVUFBVSxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUM7UUFDcEMsSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNiLFVBQVUsR0FBRyxpQkFBaUIsSUFBSSxDQUFDLHNCQUFzQixHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLHNCQUFzQixDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQztZQUNuRyxJQUFJLENBQUMsc0JBQXNCLElBQUksQ0FBQyxDQUFDO1NBQ3BDO1FBQ0QsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLFNBQVMsQ0FBQyxvQkFBb0IsQ0FBQztZQUMvQyxVQUFVO1lBQ1YsUUFBUSxRQUFFLE9BQU8sQ0FBQyxRQUFRLG1DQUFJLElBQUksQ0FBQyxzQkFBc0IsRUFBRTtTQUM5RCxDQUFDLENBQUMsQ0FBQztJQUNSLENBQUM7SUFDRDs7Ozs7T0FLRztJQUNJLFVBQVUsQ0FBQyxHQUFHLE9BQStCO1FBQ2hELEtBQUssTUFBTSxNQUFNLElBQUksT0FBTyxFQUFFO1lBQzFCLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ3hDO0lBQ0wsQ0FBQztJQUNEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQixDQUFDLFFBQWdCLENBQUM7UUFDM0MsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUFDO1FBQ3pDLElBQUksQ0FBQyx1QkFBdUIsSUFBSSxLQUFLLENBQUM7UUFDdEMsT0FBTyxHQUFHLENBQUM7SUFDZixDQUFDO0lBQ0Q7O09BRUc7SUFDSSxZQUFZLENBQUMsVUFBa0I7UUFDbEMsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxhQUFhLENBQUMsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ2pGLENBQUM7SUFDRDs7Ozs7Ozs7O09BU0c7SUFDSyxZQUFZO1FBQ2hCLDZGQUE2RjtRQUM3RiwrRkFBK0Y7UUFDL0YsK0RBQStEO1FBQy9ELElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNoQixPQUFPO1NBQ1Y7UUFDRCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQztRQUN0QixLQUFLLE1BQU0sRUFBRSxlQUFlLEVBQUUsYUFBYSxFQUFFLGVBQWUsRUFBRSxJQUFJLElBQUksQ0FBQyxjQUFjLEVBQUU7WUFDbkYsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxhQUFhLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDakUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTLENBQUMsOEJBQW9CLENBQUMsaUJBQWlCLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtnQkFDckYsY0FBYyxFQUFFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDO2dCQUMvRCxrQkFBa0IsRUFBRSxJQUFJLENBQUMscUJBQXFCO2dCQUM5QyxNQUFNLEVBQUUsUUFBUTtnQkFDaEIsY0FBYyxFQUFFLFFBQVEsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUNyRCxlQUFlO2dCQUNmLGVBQWU7YUFDbEIsQ0FBQyxDQUFDLENBQUM7U0FDUDtJQUNMLENBQUM7SUFDRDs7T0FFRztJQUNLLG1CQUFtQixDQUFDLFFBQWdCO1FBQ3hDLElBQUksQ0FBQyx1QkFBdUIsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLFFBQVEsR0FBRyxDQUFDLEVBQUUsSUFBSSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDeEYsQ0FBQztJQUNEOztPQUVHO0lBQ0ssaUJBQWlCLENBQUMsQ0FBUztRQUMvQixPQUFPLFdBQVcsQ0FBQyxDQUFDLEVBQUUsR0FBRyxJQUFJLENBQUMsU0FBUyxHQUFHLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBQ0Q7Ozs7Ozs7Ozs7T0FVRztJQUNLLHdCQUF3QixDQUFDLGFBQWdEO1FBQzdFLE1BQU0sY0FBYyxHQUFHLGFBQWEsQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQzFFLEtBQUssTUFBTSxnQkFBZ0IsSUFBSSxjQUFjLEVBQUU7WUFDM0MsTUFBTSxRQUFRLEdBQUcsb0NBQW1CLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3JFLEtBQUssTUFBTSxLQUFLLElBQUksUUFBUSxDQUFDLE9BQU8sRUFBRTtnQkFDbEMsSUFBSSxTQUFvQixDQUFDO2dCQUN6QixJQUFJLEtBQUssWUFBWSx5Q0FBd0IsRUFBRTtvQkFDM0MsU0FBUyxHQUFHLG1CQUFTLENBQUMsWUFBWSxDQUFDO2lCQUN0QztxQkFDSSxJQUFJLEtBQUssWUFBWSxrQ0FBaUIsRUFBRTtvQkFDekMsNkNBQTZDO29CQUM3QyxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxLQUFLLE1BQU0sSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLElBQUksS0FBSyxhQUFhLENBQUMsWUFBWSxFQUFFO3dCQUN2RixTQUFTO3FCQUNaO29CQUNELFNBQVMsR0FBRyxtQkFBUyxDQUFDLElBQUksQ0FBQztpQkFDOUI7cUJBQ0k7b0JBQ0QsTUFBTSxJQUFJLEtBQUssQ0FBQyw0QkFBNEIsS0FBSyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7aUJBQzdEO2dCQUNELElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDO29CQUNuQixpQkFBaUIsRUFBRSxnQkFBZ0IsQ0FBQyxJQUFJO29CQUN4QyxPQUFPLEVBQUUsS0FBSyxDQUFDLEVBQUUsQ0FBQyxPQUFPO29CQUN6QixhQUFhLEVBQUUsS0FBSyxDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUU7b0JBQ2xDLFNBQVM7aUJBQ1osQ0FBQyxDQUFDO2FBQ047U0FDSjtJQUNMLENBQUM7Q0FDSjtBQXJNRCw0QkFxTUM7QUFrQkQ7O0dBRUc7QUFDSCxNQUFhLFdBQVc7SUFTcEI7O09BRUc7SUFDSCxZQUFZLFlBQXVDLEVBQUUsVUFBa0I7UUFDbkUsSUFBSSxDQUFDLFlBQVksR0FBRyxZQUFZLENBQUM7UUFDakMsSUFBSSxDQUFDLFVBQVUsR0FBRyxVQUFVLENBQUM7SUFDakMsQ0FBQztDQUNKO0FBaEJELGtDQWdCQztBQUNELFNBQVMsV0FBVyxDQUFDLENBQVMsRUFBRSxNQUFjO0lBQzFDLE9BQU8sQ0FBQyxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztBQUM5RCxDQUFDO0FBQ0QsU0FBUyxlQUFlLENBQUMsQ0FBc0I7SUFDM0MsMEVBQTBFO0lBQzFFLG1EQUFtRDtJQUNuRCxPQUFPLENBQUMsQ0FBQyxXQUFXLENBQUMsSUFBSSxLQUFLLHVCQUF1QixDQUFDO0FBQzFELENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjb2RlcGlwZWxpbmUgZnJvbSBcIi4uLy4uL2F3cy1jb2RlcGlwZWxpbmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2RlcGlwZWxpbmUnXG5pbXBvcnQgKiBhcyBjcGFjdGlvbnMgZnJvbSBcIi4uLy4uL2F3cy1jb2RlcGlwZWxpbmUtYWN0aW9uc1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVwaXBlbGluZS1hY3Rpb25zJ1xuaW1wb3J0IHsgQ29uc3RydWN0LCBTdGFnZSwgQXNwZWN0cyB9IGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0ICogYXMgY3hhcGkgZnJvbSBcIi4uLy4uL2N4LWFwaVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY3gtYXBpJ1xuaW1wb3J0IHsgQXNzZXRUeXBlLCBEZXBsb3lDZGtTdGFja0FjdGlvbiB9IGZyb20gJy4vYWN0aW9ucyc7XG5pbXBvcnQgeyBBc3NldE1hbmlmZXN0UmVhZGVyLCBEb2NrZXJJbWFnZU1hbmlmZXN0RW50cnksIEZpbGVNYW5pZmVzdEVudHJ5IH0gZnJvbSAnLi9wcml2YXRlL2Fzc2V0LW1hbmlmZXN0JztcbmltcG9ydCB7IHRvcG9sb2dpY2FsU29ydCB9IGZyb20gJy4vcHJpdmF0ZS90b3Bvc29ydCc7XG4vKipcbiAqIENvbnN0cnVjdGlvbiBwcm9wZXJ0aWVzIGZvciBhIENka1N0YWdlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ2RrU3RhZ2VQcm9wcyB7XG4gICAgLyoqXG4gICAgICogTmFtZSBvZiB0aGUgc3RhZ2UgdGhhdCBzaG91bGQgYmUgY3JlYXRlZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHN0YWdlTmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSB1bmRlcmx5aW5nIFBpcGVsaW5lIFN0YWdlIGFzc29jaWF0ZWQgd2l0aCB0aGlzQ2RrU3RhZ2VcbiAgICAgKi9cbiAgICByZWFkb25seSBwaXBlbGluZVN0YWdlOiBjb2RlcGlwZWxpbmUuSVN0YWdlO1xuICAgIC8qKlxuICAgICAqIFRoZSBDb2RlUGlwZWxpbmUgQXJ0aWZhY3Qgd2l0aCB0aGUgQ2xvdWQgQXNzZW1ibHlcbiAgICAgKi9cbiAgICByZWFkb25seSBjbG91ZEFzc2VtYmx5QXJ0aWZhY3Q6IGNvZGVwaXBlbGluZS5BcnRpZmFjdDtcbiAgICAvKipcbiAgICAgKiBGZWF0dXJlcyB0aGUgU3RhZ2UgbmVlZHMgZnJvbSBpdHMgZW52aXJvbm1lbnRcbiAgICAgKi9cbiAgICByZWFkb25seSBob3N0OiBJU3RhZ2VIb3N0O1xufVxuLyoqXG4gKiBTdGFnZSBpbiBhIENka1BpcGVsaW5lXG4gKlxuICogWW91IGRvbid0IG5lZWQgdG8gaW5zdGFudGlhdGUgdGhpcyBjbGFzcyBkaXJlY3RseS4gVXNlXG4gKiBgY2RrUGlwZWxpbmUuYWRkU3RhZ2UoKWAgaW5zdGVhZC5cbiAqL1xuZXhwb3J0IGNsYXNzIENka1N0YWdlIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgICBwcml2YXRlIF9uZXh0U2VxdWVudGlhbFJ1bk9yZGVyID0gMTsgLy8gTXVzdCBzdGFydCBhdCAxIGVoXG4gICAgcHJpdmF0ZSBfbWFudWFsQXBwcm92YWxDb3VudGVyID0gMTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IHBpcGVsaW5lU3RhZ2U6IGNvZGVwaXBlbGluZS5JU3RhZ2U7XG4gICAgcHJpdmF0ZSByZWFkb25seSBjbG91ZEFzc2VtYmx5QXJ0aWZhY3Q6IGNvZGVwaXBlbGluZS5BcnRpZmFjdDtcbiAgICBwcml2YXRlIHJlYWRvbmx5IHN0YWNrc1RvRGVwbG95ID0gbmV3IEFycmF5PERlcGxveVN0YWNrQ29tbWFuZD4oKTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IHN0YWdlTmFtZTogc3RyaW5nO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgaG9zdDogSVN0YWdlSG9zdDtcbiAgICBwcml2YXRlIF9wcmVwYXJlZCA9IGZhbHNlO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDZGtTdGFnZVByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgICAgIHRoaXMuc3RhZ2VOYW1lID0gcHJvcHMuc3RhZ2VOYW1lO1xuICAgICAgICB0aGlzLnBpcGVsaW5lU3RhZ2UgPSBwcm9wcy5waXBlbGluZVN0YWdlO1xuICAgICAgICB0aGlzLmNsb3VkQXNzZW1ibHlBcnRpZmFjdCA9IHByb3BzLmNsb3VkQXNzZW1ibHlBcnRpZmFjdDtcbiAgICAgICAgdGhpcy5ob3N0ID0gcHJvcHMuaG9zdDtcbiAgICAgICAgQXNwZWN0cy5vZih0aGlzKS5hZGQoeyB2aXNpdDogKCkgPT4gdGhpcy5wcmVwYXJlU3RhZ2UoKSB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkIGFsbCBzdGFja3MgaW4gdGhlIGFwcGxpY2F0aW9uIFN0YWdlIHRvIHRoaXMgc3RhZ2VcbiAgICAgKlxuICAgICAqIFRoZSBhcHBsaWNhdGlvbiBjb25zdHJ1Y3Qgc2hvdWxkIHN1YmNsYXNzIGBTdGFnZWAgYW5kIGNhbiBjb250YWluIGFueVxuICAgICAqIG51bWJlciBvZiBgU3RhY2tzYCBpbnNpZGUgaXQgdGhhdCBtYXkgaGF2ZSBkZXBlbmRlbmN5IHJlbGF0aW9uc2hpcHNcbiAgICAgKiBvbiBvbmUgYW5vdGhlci5cbiAgICAgKlxuICAgICAqIEFsbCBzdGFja3MgaW4gdGhlIGFwcGxpY2F0aW9uIHdpbGwgYmUgZGVwbG95ZWQgaW4gdGhlIGFwcHJvcHJpYXRlIG9yZGVyLFxuICAgICAqIGFuZCBhbGwgYXNzZXRzIGZvdW5kIGluIHRoZSBhcHBsaWNhdGlvbiB3aWxsIGJlIGFkZGVkIHRvIHRoZSBhc3NldFxuICAgICAqIHB1Ymxpc2hpbmcgc3RhZ2UuXG4gICAgICovXG4gICAgcHVibGljIGFkZEFwcGxpY2F0aW9uKGFwcFN0YWdlOiBTdGFnZSwgb3B0aW9uczogQWRkU3RhZ2VPcHRpb25zID0ge30pIHtcbiAgICAgICAgY29uc3QgYXNtID0gYXBwU3RhZ2Uuc3ludGgoKTtcbiAgICAgICAgaWYgKGFzbS5zdGFja3MubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICAvLyBJZiB3ZSBkb24ndCBjaGVjayBoZXJlLCBhIG1vcmUgcHV6emxpbmcgXCJzdGFnZSBjb250YWlucyBubyBhY3Rpb25zXCJcbiAgICAgICAgICAgIC8vIGVycm9yIHdpbGwgYmUgdGhyb3duIGNvbWUgZGVwbG95bWVudCB0aW1lLlxuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBUaGUgZ2l2ZW4gU3RhZ2UgY29uc3RydWN0ICgnJHthcHBTdGFnZS5ub2RlLnBhdGh9Jykgc2hvdWxkIGNvbnRhaW4gYXQgbGVhc3Qgb25lIFN0YWNrYCk7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3Qgc29ydGVkVHJhbmNoZXMgPSB0b3BvbG9naWNhbFNvcnQoYXNtLnN0YWNrcywgc3RhY2sgPT4gc3RhY2suaWQsIHN0YWNrID0+IHN0YWNrLmRlcGVuZGVuY2llcy5tYXAoZCA9PiBkLmlkKSk7XG4gICAgICAgIGZvciAoY29uc3Qgc3RhY2tzIG9mIHNvcnRlZFRyYW5jaGVzKSB7XG4gICAgICAgICAgICBjb25zdCBydW5PcmRlciA9IHRoaXMubmV4dFNlcXVlbnRpYWxSdW5PcmRlcigyKTsgLy8gV2UgbmVlZCAyIGFjdGlvbnNcbiAgICAgICAgICAgIGxldCBleGVjdXRlUnVuT3JkZXIgPSBydW5PcmRlciArIDE7XG4gICAgICAgICAgICAvLyBJZiB3ZSBuZWVkIHRvIGluc2VydCBhIG1hbnVhbCBhcHByb3ZhbCBhY3Rpb24sIHRoZW4gd2hhdCdzIHRoZSBleGVjdXRlUnVuT3JkZXJcbiAgICAgICAgICAgIC8vIG5vdyBpcyB3aGVyZSB3ZSBhZGQgYSBtYW51YWwgYXBwcm92YWwgc3RlcCwgYW5kIHdlIGFsbG9jYXRlIDEgbW9yZSBydW5PcmRlclxuICAgICAgICAgICAgLy8gZm9yIHRoZSBleGVjdXRlLlxuICAgICAgICAgICAgaWYgKG9wdGlvbnMubWFudWFsQXBwcm92YWxzKSB7XG4gICAgICAgICAgICAgICAgdGhpcy5hZGRNYW51YWxBcHByb3ZhbEFjdGlvbih7IHJ1bk9yZGVyOiBleGVjdXRlUnVuT3JkZXIgfSk7XG4gICAgICAgICAgICAgICAgZXhlY3V0ZVJ1bk9yZGVyID0gdGhpcy5uZXh0U2VxdWVudGlhbFJ1bk9yZGVyKCk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICAvLyBUaGVzZSBkb24ndCBoYXZlIGEgZGVwZW5kZW5jeSBvbiBlYWNoIG90aGVyLCBzbyBjYW4gYWxsIGJlIGFkZGVkIGluIHBhcmFsbGVsXG4gICAgICAgICAgICBmb3IgKGNvbnN0IHN0YWNrIG9mIHN0YWNrcykge1xuICAgICAgICAgICAgICAgIHRoaXMuYWRkU3RhY2tBcnRpZmFjdERlcGxveW1lbnQoc3RhY2ssIHsgcnVuT3JkZXIsIGV4ZWN1dGVSdW5PcmRlciB9KTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYSBkZXBsb3ltZW50IGFjdGlvbiBiYXNlZCBvbiBhIHN0YWNrIGFydGlmYWN0XG4gICAgICovXG4gICAgcHVibGljIGFkZFN0YWNrQXJ0aWZhY3REZXBsb3ltZW50KHN0YWNrQXJ0aWZhY3Q6IGN4YXBpLkNsb3VkRm9ybWF0aW9uU3RhY2tBcnRpZmFjdCwgb3B0aW9uczogQWRkU3RhY2tPcHRpb25zID0ge30pIHtcbiAgICAgICAgLy8gR2V0IGFsbCBhc3NldHMgbWFuaWZlc3RzIGFuZCBhZGQgdGhlIGFzc2V0cyBpbiAnZW0gdG8gdGhlIGFzc2V0IHB1Ymxpc2hpbmcgc3RhZ2UuXG4gICAgICAgIHRoaXMucHVibGlzaEFzc2V0RGVwZW5kZW5jaWVzKHN0YWNrQXJ0aWZhY3QpO1xuICAgICAgICAvLyBSZW1lbWJlciBmb3IgbGF0ZXIsIHNlZSAncHJlcGFyZSgpJ1xuICAgICAgICAvLyBXZSBrbm93IHRoYXQgZGVwbG95aW5nIGEgc3RhY2sgaXMgZ29pbmcgdG8gdGFrZSB1cCAyIHJ1bm9yZGVyIHNsb3RzIGxhdGVyIG9uLlxuICAgICAgICBjb25zdCBydW5PcmRlciA9IG9wdGlvbnMucnVuT3JkZXIgPz8gdGhpcy5uZXh0U2VxdWVudGlhbFJ1bk9yZGVyKDIpO1xuICAgICAgICBjb25zdCBleGVjdXRlUnVuT3JkZXIgPSBvcHRpb25zLmV4ZWN1dGVSdW5PcmRlciA/PyBydW5PcmRlciArIDE7XG4gICAgICAgIHRoaXMuc3RhY2tzVG9EZXBsb3kucHVzaCh7XG4gICAgICAgICAgICBwcmVwYXJlUnVuT3JkZXI6IHJ1bk9yZGVyLFxuICAgICAgICAgICAgZXhlY3V0ZVJ1bk9yZGVyLFxuICAgICAgICAgICAgc3RhY2tBcnRpZmFjdCxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMuYWR2YW5jZVJ1bk9yZGVyUGFzdChydW5PcmRlcik7XG4gICAgICAgIHRoaXMuYWR2YW5jZVJ1bk9yZGVyUGFzdChleGVjdXRlUnVuT3JkZXIpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYSBtYW51YWwgYXBwcm92YWwgYWN0aW9uXG4gICAgICpcbiAgICAgKiBJZiB5b3UgbmVlZCBtb3JlIGZsZXhpYmlsaXR5IHRoYW4gd2hhdCB0aGlzIG1ldGhvZCBvZmZlcnMsXG4gICAgICogdXNlIGBhZGRBY3Rpb25gIHdpdGggYSBgTWFudWFsQXBwcm92YWxBY3Rpb25gLlxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRNYW51YWxBcHByb3ZhbEFjdGlvbihvcHRpb25zOiBBZGRNYW51YWxBcHByb3ZhbE9wdGlvbnMgPSB7fSkge1xuICAgICAgICBsZXQgYWN0aW9uTmFtZSA9IG9wdGlvbnMuYWN0aW9uTmFtZTtcbiAgICAgICAgaWYgKCFhY3Rpb25OYW1lKSB7XG4gICAgICAgICAgICBhY3Rpb25OYW1lID0gYE1hbnVhbEFwcHJvdmFsJHt0aGlzLl9tYW51YWxBcHByb3ZhbENvdW50ZXIgPiAxID8gdGhpcy5fbWFudWFsQXBwcm92YWxDb3VudGVyIDogJyd9YDtcbiAgICAgICAgICAgIHRoaXMuX21hbnVhbEFwcHJvdmFsQ291bnRlciArPSAxO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuYWRkQWN0aW9ucyhuZXcgY3BhY3Rpb25zLk1hbnVhbEFwcHJvdmFsQWN0aW9uKHtcbiAgICAgICAgICAgIGFjdGlvbk5hbWUsXG4gICAgICAgICAgICBydW5PcmRlcjogb3B0aW9ucy5ydW5PcmRlciA/PyB0aGlzLm5leHRTZXF1ZW50aWFsUnVuT3JkZXIoKSxcbiAgICAgICAgfSkpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgb25lIG9yIG1vcmUgQ29kZVBpcGVsaW5lIEFjdGlvbnNcbiAgICAgKlxuICAgICAqIFlvdSBuZWVkIHRvIG1ha2Ugc3VyZSBpdCBpcyBjcmVhdGVkIHdpdGggdGhlIHJpZ2h0IHJ1bk9yZGVyLiBDYWxsIGBuZXh0U2VxdWVudGlhbFJ1bk9yZGVyKClgXG4gICAgICogZm9yIGV2ZXJ5IGFjdGlvbiB0byBnZXQgYWN0aW9ucyB0byBleGVjdXRlIGluIHNlcXVlbmNlLlxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRBY3Rpb25zKC4uLmFjdGlvbnM6IGNvZGVwaXBlbGluZS5JQWN0aW9uW10pIHtcbiAgICAgICAgZm9yIChjb25zdCBhY3Rpb24gb2YgYWN0aW9ucykge1xuICAgICAgICAgICAgdGhpcy5waXBlbGluZVN0YWdlLmFkZEFjdGlvbihhY3Rpb24pO1xuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJldHVybiB0aGUgcnVuT3JkZXIgbnVtYmVyIG5lY2Vzc2FyeSB0byBydW4gdGhlIG5leHQgQWN0aW9uIGluIHNlcXVlbmNlIHdpdGggdGhlIHJlc3RcbiAgICAgKlxuICAgICAqIEZJWE1FOiBUaGlzIGlzIGhlcmUgYmVjYXVzZSBBY3Rpb25zIGFyZSBpbW11dGFibGUgYW5kIGNhbid0IGJlIHJlb3JkZXJlZFxuICAgICAqIGFmdGVyIGNyZWF0aW9uLCBub3IgaXMgdGhlcmUgYSB3YXkgdG8gc3BlY2lmeSByZWxhdGl2ZSBwcmlvcml0aWVzLCB3aGljaFxuICAgICAqIGlzIGEgbGltaXRhdGlvbiB0aGF0IHdlIHNob3VsZCB0YWtlIGF3YXkgaW4gdGhlIGJhc2UgbGlicmFyeS5cbiAgICAgKi9cbiAgICBwdWJsaWMgbmV4dFNlcXVlbnRpYWxSdW5PcmRlcihjb3VudDogbnVtYmVyID0gMSk6IG51bWJlciB7XG4gICAgICAgIGNvbnN0IHJldCA9IHRoaXMuX25leHRTZXF1ZW50aWFsUnVuT3JkZXI7XG4gICAgICAgIHRoaXMuX25leHRTZXF1ZW50aWFsUnVuT3JkZXIgKz0gY291bnQ7XG4gICAgICAgIHJldHVybiByZXQ7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFdoZXRoZXIgdGhpcyBTdGFnZSBjb250YWlucyBhbiBhY3Rpb24gdG8gZGVwbG95IHRoZSBnaXZlbiBzdGFjaywgaWRlbnRpZmllZCBieSBpdHMgYXJ0aWZhY3QgSURcbiAgICAgKi9cbiAgICBwdWJsaWMgZGVwbG95c1N0YWNrKGFydGlmYWN0SWQ6IHN0cmluZykge1xuICAgICAgICByZXR1cm4gdGhpcy5zdGFja3NUb0RlcGxveS5tYXAocyA9PiBzLnN0YWNrQXJ0aWZhY3QuaWQpLmluY2x1ZGVzKGFydGlmYWN0SWQpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBY3R1YWxseSBhZGQgYWxsIHRoZSBEZXBsb3lTdGFjayBhY3Rpb25zIHRvIHRoZSBzdGFnZS5cbiAgICAgKlxuICAgICAqIFdlIGRvIHRoaXMgbGF0ZSBiZWNhdXNlIGJlZm9yZSB3ZSBjYW4gcmVuZGVyIHRoZSBhY3R1YWwgRGVwbG95QWN0aW9ucyxcbiAgICAgKiB3ZSBuZWVkIHRvIGtub3cgd2hldGhlciBvciBub3Qgd2UgbmVlZCB0byBjYXB0dXJlIHRoZSBzdGFjayBvdXRwdXRzLlxuICAgICAqXG4gICAgICogRklYTUU6IFRoaXMgaXMgaGVyZSBiZWNhdXNlIEFjdGlvbnMgYXJlIGltbXV0YWJsZSBhbmQgY2FuJ3QgYmUgcmVvcmRlcmVkXG4gICAgICogYWZ0ZXIgY3JlYXRpb24sIG5vciBpcyB0aGVyZSBhIHdheSB0byBzcGVjaWZ5IHJlbGF0aXZlIHByaW9yaXRpZXMsIHdoaWNoXG4gICAgICogaXMgYSBsaW1pdGF0aW9uIHRoYXQgd2Ugc2hvdWxkIHRha2UgYXdheSBpbiB0aGUgYmFzZSBsaWJyYXJ5LlxuICAgICAqL1xuICAgIHByaXZhdGUgcHJlcGFyZVN0YWdlKCkge1xuICAgICAgICAvLyBGSVhNRTogTWFrZSBzdXJlIHRoaXMgb25seSBnZXRzIHJ1biBvbmNlLiBUaGVyZSBzZWVtcyB0byBiZSBhbiBpc3N1ZSBpbiB0aGUgcmVjb25jaWxpYXRpb25cbiAgICAgICAgLy8gbG9vcCB0aGF0IG1heSB0cmlnZ2VyIHRoaXMgbW9yZSB0aGFuIG9uY2UgaWYgaXQgdGhyb3dzIGFuIGVycm9yIHNvbWV3aGVyZSwgYW5kIHRoZSBleGNlcHRpb25cbiAgICAgICAgLy8gdGhhdCBnZXRzIHRocm93biBoZXJlIHdpbGwgdGhlbiBvdmVycmlkZSB0aGUgYWN0dWFsIGZhaWx1cmUuXG4gICAgICAgIGlmICh0aGlzLl9wcmVwYXJlZCkge1xuICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuX3ByZXBhcmVkID0gdHJ1ZTtcbiAgICAgICAgZm9yIChjb25zdCB7IHByZXBhcmVSdW5PcmRlciwgc3RhY2tBcnRpZmFjdCwgZXhlY3V0ZVJ1bk9yZGVyIH0gb2YgdGhpcy5zdGFja3NUb0RlcGxveSkge1xuICAgICAgICAgICAgY29uc3QgYXJ0aWZhY3QgPSB0aGlzLmhvc3Quc3RhY2tPdXRwdXRBcnRpZmFjdChzdGFja0FydGlmYWN0LmlkKTtcbiAgICAgICAgICAgIHRoaXMucGlwZWxpbmVTdGFnZS5hZGRBY3Rpb24oRGVwbG95Q2RrU3RhY2tBY3Rpb24uZnJvbVN0YWNrQXJ0aWZhY3QodGhpcywgc3RhY2tBcnRpZmFjdCwge1xuICAgICAgICAgICAgICAgIGJhc2VBY3Rpb25OYW1lOiB0aGlzLnNpbXBsaWZ5U3RhY2tOYW1lKHN0YWNrQXJ0aWZhY3Quc3RhY2tOYW1lKSxcbiAgICAgICAgICAgICAgICBjbG91ZEFzc2VtYmx5SW5wdXQ6IHRoaXMuY2xvdWRBc3NlbWJseUFydGlmYWN0LFxuICAgICAgICAgICAgICAgIG91dHB1dDogYXJ0aWZhY3QsXG4gICAgICAgICAgICAgICAgb3V0cHV0RmlsZU5hbWU6IGFydGlmYWN0ID8gJ291dHB1dHMuanNvbicgOiB1bmRlZmluZWQsXG4gICAgICAgICAgICAgICAgcHJlcGFyZVJ1bk9yZGVyLFxuICAgICAgICAgICAgICAgIGV4ZWN1dGVSdW5PcmRlcixcbiAgICAgICAgICAgIH0pKTtcbiAgICAgICAgfVxuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZHZhbmNlIHRoZSBydW5vcmRlciBjb3VudGVyIHNvIHRoYXQgdGhlIG5leHQgc2VxdWVudGlhbCBudW1iZXIgaXMgaGlnaGVyIHRoYW4gdGhlIGdpdmVuIG9uZVxuICAgICAqL1xuICAgIHByaXZhdGUgYWR2YW5jZVJ1bk9yZGVyUGFzdChsYXN0VXNlZDogbnVtYmVyKSB7XG4gICAgICAgIHRoaXMuX25leHRTZXF1ZW50aWFsUnVuT3JkZXIgPSBNYXRoLm1heChsYXN0VXNlZCArIDEsIHRoaXMuX25leHRTZXF1ZW50aWFsUnVuT3JkZXIpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBTaW1wbGlmeSB0aGUgc3RhY2sgbmFtZSBieSByZW1vdmluZyB0aGUgYFN0YWdlLWAgcHJlZml4IGlmIGl0IGV4aXN0cy5cbiAgICAgKi9cbiAgICBwcml2YXRlIHNpbXBsaWZ5U3RhY2tOYW1lKHM6IHN0cmluZykge1xuICAgICAgICByZXR1cm4gc3RyaXBQcmVmaXgocywgYCR7dGhpcy5zdGFnZU5hbWV9LWApO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBNYWtlIHN1cmUgYWxsIGFzc2V0cyBkZXBlbmRlZCBvbiBieSB0aGlzIHN0YWNrIGFyZSBwdWJsaXNoZWQgaW4gdGhpcyBwaXBlbGluZVxuICAgICAqXG4gICAgICogVGFraW5nIGNhcmUgdG8gZXhjbHVkZSB0aGUgc3RhY2sgdGVtcGxhdGUgaXRzZWxmIC0tIGl0IGlzIGJlaW5nIHB1Ymxpc2hlZFxuICAgICAqIGFzIGFuIGFzc2V0IGJlY2F1c2UgdGhlIENMSSBuZWVkcyB0byBrbm93IHRoZSBhc3NldCBwdWJsaXNoaW5nIHJvbGUgd2hlblxuICAgICAqIHB1c2hpbmcgdGhlIHRlbXBsYXRlIHRvIFMzLCBidXQgaW4gdGhlIGNhc2Ugb2YgQ29kZVBpcGVsaW5lIHdlIGFsd2F5c1xuICAgICAqIHJlZmVyZW5jZSB0aGUgdGVtcGxhdGUgZnJvbSB0aGUgYXJ0aWZhY3QgYnVja2V0LlxuICAgICAqXG4gICAgICogKE5PVEU6IHRoaXMgaXMgb25seSB0cnVlIGZvciB0b3AtbGV2ZWwgc3RhY2tzLCBub3QgbmVzdGVkIHN0YWNrcy4gTmVzdGVkXG4gICAgICogU3RhY2sgdGVtcGxhdGVzIGFyZSBhbHdheXMgcHVibGlzaGVkIGFzIGFzc2V0cykuXG4gICAgICovXG4gICAgcHJpdmF0ZSBwdWJsaXNoQXNzZXREZXBlbmRlbmNpZXMoc3RhY2tBcnRpZmFjdDogY3hhcGkuQ2xvdWRGb3JtYXRpb25TdGFja0FydGlmYWN0KSB7XG4gICAgICAgIGNvbnN0IGFzc2V0TWFuaWZlc3RzID0gc3RhY2tBcnRpZmFjdC5kZXBlbmRlbmNpZXMuZmlsdGVyKGlzQXNzZXRNYW5pZmVzdCk7XG4gICAgICAgIGZvciAoY29uc3QgbWFuaWZlc3RBcnRpZmFjdCBvZiBhc3NldE1hbmlmZXN0cykge1xuICAgICAgICAgICAgY29uc3QgbWFuaWZlc3QgPSBBc3NldE1hbmlmZXN0UmVhZGVyLmZyb21GaWxlKG1hbmlmZXN0QXJ0aWZhY3QuZmlsZSk7XG4gICAgICAgICAgICBmb3IgKGNvbnN0IGVudHJ5IG9mIG1hbmlmZXN0LmVudHJpZXMpIHtcbiAgICAgICAgICAgICAgICBsZXQgYXNzZXRUeXBlOiBBc3NldFR5cGU7XG4gICAgICAgICAgICAgICAgaWYgKGVudHJ5IGluc3RhbmNlb2YgRG9ja2VySW1hZ2VNYW5pZmVzdEVudHJ5KSB7XG4gICAgICAgICAgICAgICAgICAgIGFzc2V0VHlwZSA9IEFzc2V0VHlwZS5ET0NLRVJfSU1BR0U7XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIGVsc2UgaWYgKGVudHJ5IGluc3RhbmNlb2YgRmlsZU1hbmlmZXN0RW50cnkpIHtcbiAgICAgICAgICAgICAgICAgICAgLy8gRG9uJ3QgcHVibGlzaGcgdGhlIHRlbXBsYXRlIGZvciB0aGlzIHN0YWNrXG4gICAgICAgICAgICAgICAgICAgIGlmIChlbnRyeS5zb3VyY2UucGFja2FnaW5nID09PSAnZmlsZScgJiYgZW50cnkuc291cmNlLnBhdGggPT09IHN0YWNrQXJ0aWZhY3QudGVtcGxhdGVGaWxlKSB7XG4gICAgICAgICAgICAgICAgICAgICAgICBjb250aW51ZTtcbiAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICBhc3NldFR5cGUgPSBBc3NldFR5cGUuRklMRTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5yZWNvZ25pemVkIGFzc2V0IHR5cGU6ICR7ZW50cnkudHlwZX1gKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgdGhpcy5ob3N0LnB1Ymxpc2hBc3NldCh7XG4gICAgICAgICAgICAgICAgICAgIGFzc2V0TWFuaWZlc3RQYXRoOiBtYW5pZmVzdEFydGlmYWN0LmZpbGUsXG4gICAgICAgICAgICAgICAgICAgIGFzc2V0SWQ6IGVudHJ5LmlkLmFzc2V0SWQsXG4gICAgICAgICAgICAgICAgICAgIGFzc2V0U2VsZWN0b3I6IGVudHJ5LmlkLnRvU3RyaW5nKCksXG4gICAgICAgICAgICAgICAgICAgIGFzc2V0VHlwZSxcbiAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgIH1cbn1cbi8qKlxuICogQWRkaXRpb25hbCBvcHRpb25zIGZvciBhZGRpbmcgYSBzdGFjayBkZXBsb3ltZW50XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWRkU3RhY2tPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBCYXNlIHJ1bm9yZGVyXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5leHQgc2VxdWVudGlhbCBydW5vcmRlclxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJ1bk9yZGVyPzogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIEJhc2UgcnVub3JkZXJcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gcnVuT3JkZXIgKyAxXG4gICAgICovXG4gICAgcmVhZG9ubHkgZXhlY3V0ZVJ1bk9yZGVyPzogbnVtYmVyO1xufVxuLyoqXG4gKiBBIHNpbmdsZSBvdXRwdXQgb2YgYSBTdGFja1xuICovXG5leHBvcnQgY2xhc3MgU3RhY2tPdXRwdXQge1xuICAgIC8qKlxuICAgICAqIFRoZSBhcnRpZmFjdCBhbmQgZmlsZSB0aGUgb3V0cHV0IGlzIHN0b3JlZCBpblxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBhcnRpZmFjdEZpbGU6IGNvZGVwaXBlbGluZS5BcnRpZmFjdFBhdGg7XG4gICAgLyoqXG4gICAgICogVGhlIG5hbWUgb2YgdGhlIG91dHB1dCBpbiB0aGUgSlNPTiBvYmplY3QgaW4gdGhlIGZpbGVcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgb3V0cHV0TmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEJ1aWxkIGEgU3RhY2tPdXRwdXQgZnJvbSBhIGtub3duIGFydGlmYWN0IGFuZCBhbiBvdXRwdXQgbmFtZVxuICAgICAqL1xuICAgIGNvbnN0cnVjdG9yKGFydGlmYWN0RmlsZTogY29kZXBpcGVsaW5lLkFydGlmYWN0UGF0aCwgb3V0cHV0TmFtZTogc3RyaW5nKSB7XG4gICAgICAgIHRoaXMuYXJ0aWZhY3RGaWxlID0gYXJ0aWZhY3RGaWxlO1xuICAgICAgICB0aGlzLm91dHB1dE5hbWUgPSBvdXRwdXROYW1lO1xuICAgIH1cbn1cbmZ1bmN0aW9uIHN0cmlwUHJlZml4KHM6IHN0cmluZywgcHJlZml4OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gcy5zdGFydHNXaXRoKHByZWZpeCkgPyBzLnN1YnN0cihwcmVmaXgubGVuZ3RoKSA6IHM7XG59XG5mdW5jdGlvbiBpc0Fzc2V0TWFuaWZlc3QoczogY3hhcGkuQ2xvdWRBcnRpZmFjdCk6IHMgaXMgY3hhcGkuQXNzZXRNYW5pZmVzdEFydGlmYWN0IHtcbiAgICAvLyBpbnN0YW5jZW9mIGlzIHRvbyByaXNreSwgYW5kIHdlJ3JlIGF0IGEgdG9vIGxhdGUgc3RhZ2UgdG8gcHJvcGVybHkgZml4LlxuICAgIC8vIHJldHVybiBzIGluc3RhbmNlb2YgY3hhcGkuQXNzZXRNYW5pZmVzdEFydGlmYWN0O1xuICAgIHJldHVybiBzLmNvbnN0cnVjdG9yLm5hbWUgPT09ICdBc3NldE1hbmlmZXN0QXJ0aWZhY3QnO1xufVxuLyoqXG4gKiBGZWF0dXJlcyB0aGF0IHRoZSBTdGFnZSBuZWVkcyBmcm9tIGl0cyBlbnZpcm9ubWVudFxuICovXG5leHBvcnQgaW50ZXJmYWNlIElTdGFnZUhvc3Qge1xuICAgIC8qKlxuICAgICAqIE1ha2Ugc3VyZSBhbGwgdGhlIGFzc2V0cyBmcm9tIHRoZSBnaXZlbiBtYW5pZmVzdCBhcmUgcHVibGlzaGVkXG4gICAgICovXG4gICAgcHVibGlzaEFzc2V0KGNvbW1hbmQ6IEFzc2V0UHVibGlzaGluZ0NvbW1hbmQpOiB2b2lkO1xuICAgIC8qKlxuICAgICAqIFJldHVybiB0aGUgQXJ0aWZhY3QgdGhlIGdpdmVuIHN0YWNrIGhhcyB0byBlbWl0IGl0cyBvdXRwdXRzIGludG8sIGlmIGFueVxuICAgICAqL1xuICAgIHN0YWNrT3V0cHV0QXJ0aWZhY3Qoc3RhY2tBcnRpZmFjdElkOiBzdHJpbmcpOiBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QgfCB1bmRlZmluZWQ7XG59XG4vKipcbiAqIEluc3RydWN0aW9ucyB0byBwdWJsaXNoIGNlcnRhaW4gYXNzZXRzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXNzZXRQdWJsaXNoaW5nQ29tbWFuZCB7XG4gICAgLyoqXG4gICAgICogQXNzZXQgbWFuaWZlc3QgcGF0aFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFzc2V0TWFuaWZlc3RQYXRoOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogQXNzZXQgaWRlbnRpZmllclxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFzc2V0SWQ6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBBc3NldCBzZWxlY3RvciB0byBwYXNzIHRvIGBjZGstYXNzZXRzYC5cbiAgICAgKi9cbiAgICByZWFkb25seSBhc3NldFNlbGVjdG9yOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVHlwZSBvZiBhc3NldCB0byBwdWJsaXNoXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXNzZXRUeXBlOiBBc3NldFR5cGU7XG59XG4vKipcbiAqIE9wdGlvbnMgZm9yIGFkZGluZyBhbiBhcHBsaWNhdGlvbiBzdGFnZSB0byBhIHBpcGVsaW5lXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWRkU3RhZ2VPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBBZGQgbWFudWFsIGFwcHJvdmFscyBiZWZvcmUgZXhlY3V0aW5nIGNoYW5nZSBzZXRzXG4gICAgICpcbiAgICAgKiBUaGlzIGdpdmVzIGh1bWFucyB0aGUgb3Bwb3J0dW5pdHkgdG8gY29uZmlybSB0aGUgY2hhbmdlIHNldCBsb29rcyBhbHJpZ2h0XG4gICAgICogYmVmb3JlIGRlcGxveWluZyBpdC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWFudWFsQXBwcm92YWxzPzogYm9vbGVhbjtcbn1cbi8qKlxuICogT3B0aW9ucyBmb3IgYWRkTWFudWFsQXBwcm92YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBZGRNYW51YWxBcHByb3ZhbE9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSBtYW51YWwgYXBwcm92YWwgYWN0aW9uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAnTWFudWFsQXBwcm92YWwnIHdpdGggYSByb2xsaW5nIGNvdW50ZXJcbiAgICAgKi9cbiAgICByZWFkb25seSBhY3Rpb25OYW1lPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBydW5PcmRlciBmb3IgdGhpcyBhY3Rpb25cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gVGhlIG5leHQgc2VxdWVudGlhbCBydW5PcmRlclxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJ1bk9yZGVyPzogbnVtYmVyO1xufVxuLyoqXG4gKiBRdWV1ZWQgXCJkZXBsb3kgc3RhY2tcIiBjb21tYW5kIHRoYXQgaXMgcmVpZmllZCBkdXJpbmcgcHJlcGFyZSgpXG4gKi9cbmludGVyZmFjZSBEZXBsb3lTdGFja0NvbW1hbmQge1xuICAgIHByZXBhcmVSdW5PcmRlcjogbnVtYmVyO1xuICAgIGV4ZWN1dGVSdW5PcmRlcjogbnVtYmVyO1xuICAgIHN0YWNrQXJ0aWZhY3Q6IGN4YXBpLkNsb3VkRm9ybWF0aW9uU3RhY2tBcnRpZmFjdDtcbn1cbiJdfQ==