"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FactName = exports.Fact = void 0;
/**
 * A database of regional information.
 */
class Fact {
    constructor() {
        throw new Error('Use the static methods of Fact instead!');
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     *          may not be an exhaustive list of all available AWS regions.
     */
    static get regions() {
        return Object.keys(this.database);
    }
    /**
     * Retrieves a fact from this Fact database.
     *
     * @param region the name of the region (e.g: `us-east-1`)
     * @param name   the name of the fact being looked up (see the `FactName` class for details)
     *
     * @returns the fact value if it is known, and `undefined` otherwise.
     */
    static find(region, name) {
        const regionFacts = this.database[region];
        return regionFacts && regionFacts[name];
    }
    /**
     * Retrieve a fact from the Fact database. (retrieval will fail if the specified region or
     * fact name does not exist.)
     *
     * @param region the name of the region (e.g: `us-east-1`)
     * @param name the name of the fact being looked up (see the `FactName` class for details)
     */
    static requireFact(region, name) {
        const foundFact = this.find(region, name);
        if (!foundFact) {
            throw new Error(`No fact ${name} could be found for region: ${region} and name: ${name}`);
        }
        return foundFact;
    }
    /**
     * Registers a new fact in this Fact database.
     *
     * @param fact           the new fact to be registered.
     * @param allowReplacing whether new facts can replace existing facts or not.
     */
    static register(fact, allowReplacing = false) {
        const regionFacts = this.database[fact.region] || (this.database[fact.region] = {});
        if (fact.name in regionFacts && regionFacts[fact.name] !== fact.value && !allowReplacing) {
            throw new Error(`Region ${fact.region} already has a fact ${fact.name}, with value ${regionFacts[fact.name]}`);
        }
        if (fact.value !== undefined) {
            regionFacts[fact.name] = fact.value;
        }
    }
    /**
     * Removes a fact from the database.
     * @param region the region for which the fact is to be removed.
     * @param name   the name of the fact to remove.
     * @param value  the value that should be removed (removal will fail if the value is specified, but does not match the
     *               current stored value).
     */
    static unregister(region, name, value) {
        const regionFacts = this.database[region] || {};
        if (name in regionFacts && value && regionFacts[name] !== value) {
            throw new Error(`Attempted to remove ${name} from ${region} with value ${value}, but the fact's value is ${regionFacts[name]}`);
        }
        delete regionFacts[name];
    }
}
exports.Fact = Fact;
Fact.database = {};
/**
 * All standardized fact names.
 */
class FactName {
    /**
     * The name of the regional service principal for a given service.
     *
     * @param service the service name, either simple (e.g: `s3`, `codedeploy`) or qualified (e.g: `s3.amazonaws.com`).
     *                The `.amazonaws.com` and `.amazonaws.com.cn` domains are stripped from service names, so they are
     *                canonicalized in that respect.
     */
    static servicePrincipal(service) {
        return `service-principal:${service.replace(/\.amazonaws\.com(\.cn)?$/, '')}`;
    }
}
exports.FactName = FactName;
/**
 * The name of the partition for a region (e.g: 'aws', 'aws-cn', ...)
 */
FactName.PARTITION = 'partition';
/**
 * The domain suffix for a region (e.g: 'amazonaws.com`)
 */
FactName.DOMAIN_SUFFIX = 'domainSuffix';
/**
 * Whether the AWS::CDK::Metadata CloudFormation Resource is available in-region or not. The value is a boolean
 * modelled as `YES` or `NO`.
 */
FactName.CDK_METADATA_RESOURCE_AVAILABLE = 'cdk:metadata-resource:available';
/**
 * The endpoint used for hosting S3 static websites
 */
FactName.S3_STATIC_WEBSITE_ENDPOINT = 's3-static-website:endpoint';
/**
 * The endpoint used for aliasing S3 static websites in Route 53
 */
FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID = 's3-static-website:route-53-hosted-zone-id';
/**
 * The prefix for VPC Endpoint Service names,
 * cn.com.amazonaws.vpce for China regions,
 * com.amazonaws.vpce otherwise.
 */
FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'vpcEndpointServiceNamePrefix';
/**
 * The account for ELBv2 in this region
 */
FactName.ELBV2_ACCOUNT = 'elbv2Account';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the
 * AWS Deep Learning Containers images in a given region.
 */
FactName.DLC_REPOSITORY_ACCOUNT = 'dlcRepositoryAccount';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the
 * AWS App Mesh Envoy Proxy images in a given region.
 */
FactName.APPMESH_ECR_ACCOUNT = 'appMeshRepositoryAccount';
//# sourceMappingURL=data:application/json;base64,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