"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RegionInfo = void 0;
const fact_1 = require("./fact");
/**
 * Information pertaining to an AWS region.
 */
class RegionInfo {
    constructor(name) {
        this.name = name;
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     *          may not be an exaustive list of all available AWS regions.
     */
    static get regions() {
        return fact_1.Fact.regions.map(RegionInfo.get);
    }
    /**
     * Retrieves a collection of all fact values for all regions that fact is defined in.
     *
     * @param factName the name of the fact to retrieve values for.
     *   For a list of common fact names, see the FactName class
     * @returns a mapping with AWS region codes as the keys,
     *   and the fact in the given region as the value for that key
     */
    static regionMap(factName) {
        const ret = {};
        for (const regionInfo of RegionInfo.regions) {
            const fact = fact_1.Fact.find(regionInfo.name, factName);
            if (fact) {
                ret[regionInfo.name] = fact;
            }
        }
        return ret;
    }
    /**
     * Obtain region info for a given region name.
     *
     * @param name the name of the region (e.g: us-east-1)
     */
    static get(name) {
        return new RegionInfo(name);
    }
    /**
     * Whether the `AWS::CDK::Metadata` CloudFormation Resource is available in this region or not.
     */
    get cdkMetadataResourceAvailable() {
        return fact_1.Fact.find(this.name, fact_1.FactName.CDK_METADATA_RESOURCE_AVAILABLE) === 'YES';
    }
    /**
     * The domain name suffix (e.g: amazonaws.com) for this region.
     */
    get domainSuffix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.DOMAIN_SUFFIX);
    }
    /**
     * The name of the ARN partition for this region (e.g: aws).
     */
    get partition() {
        return fact_1.Fact.find(this.name, fact_1.FactName.PARTITION);
    }
    /**
     * The endpoint used by S3 static website hosting in this region (e.g: s3-static-website-us-east-1.amazonaws.com)
     */
    get s3StaticWebsiteEndpoint() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ENDPOINT);
    }
    /**
     * The hosted zone ID used by Route 53 to alias a S3 static website in this region (e.g: Z2O1EMRO9K5GLX)
     */
    get s3StaticWebsiteHostedZoneId() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID);
    }
    /**
     * The prefix for VPC Endpoint Service names,
     * cn.com.amazonaws.vpce for China regions,
     * com.amazonaws.vpce otherwise.
     */
    get vpcEndpointServiceNamePrefix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX);
    }
    /**
     * The name of the service principal for a given service in this region.
     * @param service the service name (e.g: s3.amazonaws.com)
     */
    servicePrincipal(service) {
        return fact_1.Fact.find(this.name, fact_1.FactName.servicePrincipal(service));
    }
    /**
     * The account ID for ELBv2 in this region
     *
     */
    get elbv2Account() {
        return fact_1.Fact.find(this.name, fact_1.FactName.ELBV2_ACCOUNT);
    }
    /**
     * The ID of the AWS account that owns the public ECR repository containing the
     * AWS Deep Learning Containers images in this region.
     */
    get dlcRepositoryAccount() {
        return fact_1.Fact.find(this.name, fact_1.FactName.DLC_REPOSITORY_ACCOUNT);
    }
    /**
     * The ID of the AWS account that owns the public ECR repository that contains the
     * AWS App Mesh Envoy Proxy images in a given region.
     */
    get appMeshRepositoryAccount() {
        return fact_1.Fact.find(this.name, fact_1.FactName.APPMESH_ECR_ACCOUNT);
    }
}
exports.RegionInfo = RegionInfo;
//# sourceMappingURL=data:application/json;base64,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