"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TableSortStyle = exports.TableDistStyle = exports.Table = exports.TableAction = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const database_query_1 = require("./private/database-query");
const handler_name_1 = require("./private/database-query-provider/handler-name");
const util_1 = require("./private/database-query-provider/util");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * An action that a Redshift user can be granted privilege to perform on a table.
 */
var TableAction;
(function (TableAction) {
    /**
     * Grants privilege to select data from a table or view using a SELECT statement.
     */
    TableAction[TableAction["SELECT"] = 0] = "SELECT";
    /**
     * Grants privilege to load data into a table using an INSERT statement or a COPY statement.
     */
    TableAction[TableAction["INSERT"] = 1] = "INSERT";
    /**
     * Grants privilege to update a table column using an UPDATE statement.
     */
    TableAction[TableAction["UPDATE"] = 2] = "UPDATE";
    /**
     * Grants privilege to delete a data row from a table.
     */
    TableAction[TableAction["DELETE"] = 3] = "DELETE";
    /**
     * Grants privilege to drop a table.
     */
    TableAction[TableAction["DROP"] = 4] = "DROP";
    /**
     * Grants privilege to create a foreign key constraint.
     *
     * You need to grant this privilege on both the referenced table and the referencing table; otherwise, the user can't create the constraint.
     */
    TableAction[TableAction["REFERENCES"] = 5] = "REFERENCES";
    /**
     * Grants all available privileges at once to the specified user or user group.
     */
    TableAction[TableAction["ALL"] = 6] = "ALL";
})(TableAction = exports.TableAction || (exports.TableAction = {}));
class TableBase extends core_1.Construct {
    grant(user, ...actions) {
        user.addTablePrivileges(this, ...actions);
    }
}
/**
 * A table in a Redshift cluster.
 */
class Table extends TableBase {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_TableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.validateDistKeyColumns(props.tableColumns);
        if (props.distStyle) {
            this.validateDistStyle(props.distStyle, props.tableColumns);
        }
        if (props.sortStyle) {
            this.validateSortStyle(props.sortStyle, props.tableColumns);
        }
        this.tableColumns = props.tableColumns;
        this.cluster = props.cluster;
        this.databaseName = props.databaseName;
        this.resource = new database_query_1.DatabaseQuery(this, 'Resource', {
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            ...props,
            handler: handler_name_1.HandlerName.Table,
            properties: {
                tableName: {
                    prefix: (_b = props.tableName) !== null && _b !== void 0 ? _b : cdk.Names.uniqueId(this),
                    generateSuffix: !props.tableName ? 'true' : 'false',
                },
                tableColumns: this.tableColumns,
                distStyle: props.distStyle,
                sortStyle: (_c = props.sortStyle) !== null && _c !== void 0 ? _c : this.getDefaultSortStyle(props.tableColumns),
            },
        });
        this.tableName = this.resource.ref;
    }
    /**
     * Specify a Redshift table using a table name and schema that already exists.
     */
    static fromTableAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_TableAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTableAttributes);
            }
            throw error;
        }
        return new class extends TableBase {
            constructor() {
                super(...arguments);
                this.tableName = attrs.tableName;
                this.tableColumns = attrs.tableColumns;
                this.cluster = attrs.cluster;
                this.databaseName = attrs.databaseName;
            }
        }(scope, id);
    }
    /**
     * Apply the given removal policy to this resource
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be destroyed (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * This resource is retained by default.
     */
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    validateDistKeyColumns(columns) {
        try {
            util_1.getDistKeyColumn(columns);
        }
        catch (err) {
            throw new Error('Only one column can be configured as distKey.');
        }
    }
    validateDistStyle(distStyle, columns) {
        const distKeyColumn = util_1.getDistKeyColumn(columns);
        if (distKeyColumn && distStyle !== TableDistStyle.KEY) {
            throw new Error(`Only 'TableDistStyle.KEY' can be configured when distKey is also configured. Found ${distStyle}`);
        }
        if (!distKeyColumn && distStyle === TableDistStyle.KEY) {
            throw new Error('distStyle of "TableDistStyle.KEY" can only be configured when distKey is also configured.');
        }
    }
    validateSortStyle(sortStyle, columns) {
        const sortKeyColumns = util_1.getSortKeyColumns(columns);
        if (sortKeyColumns.length === 0 && sortStyle !== TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${sortStyle}' can only be configured when sortKey is also configured.`);
        }
        if (sortKeyColumns.length > 0 && sortStyle === TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${TableSortStyle.AUTO}' cannot be configured when sortKey is also configured.`);
        }
    }
    getDefaultSortStyle(columns) {
        const sortKeyColumns = util_1.getSortKeyColumns(columns);
        return (sortKeyColumns.length === 0) ? TableSortStyle.AUTO : TableSortStyle.COMPOUND;
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-redshift.Table", version: "1.157.0" };
/**
 * The data distribution style of a table.
 */
var TableDistStyle;
(function (TableDistStyle) {
    /**
     *  Amazon Redshift assigns an optimal distribution style based on the table data
     */
    TableDistStyle["AUTO"] = "AUTO";
    /**
     * The data in the table is spread evenly across the nodes in a cluster in a round-robin distribution.
     */
    TableDistStyle["EVEN"] = "EVEN";
    /**
     * The data is distributed by the values in the DISTKEY column.
     */
    TableDistStyle["KEY"] = "KEY";
    /**
     * A copy of the entire table is distributed to every node.
     */
    TableDistStyle["ALL"] = "ALL";
})(TableDistStyle = exports.TableDistStyle || (exports.TableDistStyle = {}));
/**
 * The sort style of a table.
 */
var TableSortStyle;
(function (TableSortStyle) {
    /**
     * Amazon Redshift assigns an optimal sort key based on the table data.
     */
    TableSortStyle["AUTO"] = "AUTO";
    /**
     * Specifies that the data is sorted using a compound key made up of all of the listed columns,
     * in the order they are listed.
     */
    TableSortStyle["COMPOUND"] = "COMPOUND";
    /**
     * Specifies that the data is sorted using an interleaved sort key.
     */
    TableSortStyle["INTERLEAVED"] = "INTERLEAVED";
})(TableSortStyle = exports.TableSortStyle || (exports.TableSortStyle = {}));
//# sourceMappingURL=data:application/json;base64,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