"""BaseGetBHsimple class code
DOCSTRINGS V4 READY
"""
from magpylib._src.fields.field_wrap_BH_level2 import getBH_level2
from magpylib._src.utility import format_star_input


class BaseGetBH:
    """provides getB and getH methods for source objects"""

    def getB(self, *observers, squeeze=True, pixel_agg=None):
        """Compute the B-field in units of [mT] generated by the source.

        Parameters
        ----------
        observers: array_like or (list of) `Sensor` objects
            Can be array_like positions of shape (n1, n2, ..., 3) where the field
            should be evaluated, a `Sensor` object with pixel shape (n1, n2, ..., 3) or a list
            of such sensor objects (must all have similar pixel shapes). All positions
            are given in units of [mm].

        squeeze: bool, default=`True`
            If `True`, the output is squeezed, i.e. all axes of length 1 in the output (e.g.
            only a single source) are eliminated.

        pixel_agg: str, default=`None`
            Reference to a compatible numpy aggregator function like `'min'` or `'mean'`,
            which is applied to observer output values, e.g. mean of all sensor pixel outputs.
            With this option, observers input with different (pixel) shapes is allowed.

        Returns
        -------
        B-field: ndarray, shape squeeze(m, k, n1, n2, ..., 3)
            B-field at each path position (m) for each sensor (k) and each sensor pixel
            position (n1,n2,...) in units of [mT]. Sensor pixel positions are equivalent
            to simple observer positions. Paths of objects that are shorter than m will be
            considered as static beyond their end.

        Examples
        --------
        Compute the B-field of a spherical magnet at three positions:

        >>> import magpylib as magpy
        >>> src = magpy.magnet.Sphere((0,0,1000), 1)
        >>> B = src.getB(((0,0,0), (1,0,0), (2,0,0)))
        >>> print(B)
        [[  0.           0.         666.66666667]
         [  0.           0.         -41.66666667]
         [  0.           0.          -5.20833333]]

        Compute the B-field at two sensors, each one with two pixels

        >>> sens1 = magpy.Sensor(position=(1,0,0), pixel=((0,0,.1), (0,0,-.1)))
        >>> sens2 = sens1.copy(position=(2,0,0))
        >>> B = src.getB(sens1, sens2)
        >>> print(B)
        [[[ 12.19288783   0.         -39.83010025]
          [-12.19288783   0.         -39.83010025]]
        <BLANKLINE>
         [[  0.77638847   0.          -5.15004352]
          [ -0.77638847   0.          -5.15004352]]]
        """
        observers = format_star_input(observers)
        return getBH_level2(
            self,
            observers,
            sumup=False,
            squeeze=squeeze,
            pixel_agg=pixel_agg,
            field="B",
        )

    def getH(self, *observers, squeeze=True, pixel_agg=None):
        """Compute the H-field in units of [kA/m] generated by the source.

        Parameters
        ----------
        observers: array_like or (list of) `Sensor` objects
            Can be array_like positions of shape (n1, n2, ..., 3) where the field
            should be evaluated, a `Sensor` object with pixel shape (n1, n2, ..., 3) or a list
            of such sensor objects (must all have similar pixel shapes). All positions
            are given in units of [mm].

        squeeze: bool, default=`True`
            If `True`, the output is squeezed, i.e. all axes of length 1 in the output (e.g.
            only a single source) are eliminated.

        pixel_agg: str, default=`None`
            Reference to a compatible numpy aggregator function like `'min'` or `'mean'`,
            which is applied to observer output values, e.g. mean of all sensor pixel outputs.
            With this option, observers input with different (pixel) shapes is allowed.

        Returns
        -------
        H-field: ndarray, shape squeeze(m, k, n1, n2, ..., 3)
            H-field at each path position (m) for each sensor (k) and each sensor pixel
            position (n1,n2,...) in units of [kA/m]. Sensor pixel positions are equivalent
            to simple observer positions. Paths of objects that are shorter than m will be
            considered as static beyond their end.

        Examples
        --------
        Compute the H-field of a spherical magnet at three positions:

        >>> import magpylib as magpy

        >>> src = magpy.magnet.Sphere((0,0,1000), 1)
        >>> H = src.getH(((0,0,0), (1,0,0), (2,0,0)))
        >>> print(H)
        [[   0.            0.         -265.25823849]
         [   0.            0.          -33.15727981]
         [   0.            0.           -4.14465998]]

        Compute the H-field at two sensors, each one with two pixels

        >>> sens1 = magpy.Sensor(position=(1,0,0), pixel=((0,0,.1), (0,0,-.1)))
        >>> sens2 = sens1.copy(position=(2,0,0))
        >>> H = src.getH(sens1, sens2)
        >>> print(H)
        [[[  9.70279185   0.         -31.69578669]
          [ -9.70279185   0.         -31.69578669]]
        <BLANKLINE>
         [[  0.61783031   0.          -4.09827441]
          [ -0.61783031   0.          -4.09827441]]]
        """
        observers = format_star_input(observers)
        return getBH_level2(
            self,
            observers,
            sumup=False,
            squeeze=squeeze,
            pixel_agg=pixel_agg,
            field="H",
        )
