"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnInclude = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core = require("@aws-cdk/core");
const cfn_parse = require("@aws-cdk/core/lib/cfn-parse");
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_1 = require("@aws-cdk/core");
/**
 * Construct to import an existing CloudFormation template file into a CDK application.
 *
 * All resources defined in the template file can be retrieved by calling the {@link getResource} method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 *
 * @stability stable
 */
class CfnInclude extends core.CfnElement {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        this.parameters = {};
        this.mappings = {};
        this.rules = {};
        this.hooks = {};
        this.outputs = {};
        this.nestedStacks = {};
        this.parametersToReplace = props.parameters || {};
        // read the template into a JS object
        this.template = futils.readYamlSync(props.templateFile);
        this.preserveLogicalIds = (_b = props.preserveLogicalIds) !== null && _b !== void 0 ? _b : true;
        // check if all user specified parameter values exist in the template
        for (const logicalId of Object.keys(this.parametersToReplace)) {
            if (!(logicalId in (this.template.Parameters || {}))) {
                throw new Error(`Parameter with logical ID '${logicalId}' was not found in the template`);
            }
        }
        // instantiate the Mappings
        this.mappingsScope = new core_1.Construct(this, '$Mappings');
        for (const mappingName of Object.keys(this.template.Mappings || {})) {
            this.createMapping(mappingName);
        }
        // instantiate all parameters
        for (const logicalId of Object.keys(this.template.Parameters || {})) {
            this.createParameter(logicalId);
        }
        // instantiate the conditions
        this.conditionsScope = new core_1.Construct(this, '$Conditions');
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.getOrCreateCondition(conditionName);
        }
        // instantiate the rules
        this.rulesScope = new core_1.Construct(this, '$Rules');
        for (const ruleName of Object.keys(this.template.Rules || {})) {
            this.createRule(ruleName);
        }
        this.nestedStacksToInclude = props.loadNestedStacks || {};
        // instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
        // verify that all nestedStacks have been instantiated
        for (const nestedStackId of Object.keys(props.loadNestedStacks || {})) {
            if (!(nestedStackId in this.resources)) {
                throw new Error(`Nested Stack with logical ID '${nestedStackId}' was not found in the template`);
            }
        }
        // instantiate the Hooks
        this.hooksScope = new core_1.Construct(this, '$Hooks');
        for (const hookName of Object.keys(this.template.Hooks || {})) {
            this.createHook(hookName);
        }
        const outputScope = new core_1.Construct(this, '$Ouputs');
        for (const logicalId of Object.keys(this.template.Outputs || {})) {
            this.createOutput(logicalId, outputScope);
        }
    }
    /**
     * Returns the low-level CfnResource from the template with the given logical ID.
     *
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *      // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *      const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *      // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file.
     * @stability stable
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnCondition object from the 'Conditions' section of the CloudFormation template with the given name.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file.
     * @stability stable
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnParameter object from the 'Parameters' section of the included template.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Parameter with the given name is not present in the template,
     * throws an exception.
     *
     * @param parameterName the name of the parameter to retrieve.
     * @stability stable
     */
    getParameter(parameterName) {
        const ret = this.parameters[parameterName];
        if (!ret) {
            throw new Error(`Parameter with name '${parameterName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnMapping object from the 'Mappings' section of the included template.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Mapping with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param mappingName the name of the Mapping in the template to retrieve.
     * @stability stable
     */
    getMapping(mappingName) {
        const ret = this.mappings[mappingName];
        if (!ret) {
            throw new Error(`Mapping with name '${mappingName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnOutput object from the 'Outputs' section of the included template.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If an Output with the given name is not present in the template,
     * throws an exception.
     *
     * @param logicalId the name of the output to retrieve.
     * @stability stable
     */
    getOutput(logicalId) {
        const ret = this.outputs[logicalId];
        if (!ret) {
            throw new Error(`Output with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnRule object from the 'Rules' section of the CloudFormation template with the given name.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Rule with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param ruleName the name of the Rule in the CloudFormation template.
     * @stability stable
     */
    getRule(ruleName) {
        const ret = this.rules[ruleName];
        if (!ret) {
            throw new Error(`Rule with name '${ruleName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnHook object from the 'Hooks' section of the included CloudFormation template with the given logical ID.
     *
     * Any modifications performed on the returned object will be reflected in the resulting CDK template.
     *
     * If a Hook with the given logical ID is not present in the template,
     * an exception will be thrown.
     *
     * @param hookLogicalId the logical ID of the Hook in the included CloudFormation template's 'Hooks' section.
     * @stability stable
     */
    getHook(hookLogicalId) {
        const ret = this.hooks[hookLogicalId];
        if (!ret) {
            throw new Error(`Hook with logical ID '${hookLogicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns a loaded NestedStack with name logicalId.
     *
     * For a nested stack to be returned by this method,
     * it must be specified either in the {@link CfnIncludeProps.loadNestedStacks} property,
     * or through the {@link loadNestedStack} method.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template.
     * @stability stable
     */
    getNestedStack(logicalId) {
        if (!this.nestedStacks[logicalId]) {
            if (!this.template.Resources[logicalId]) {
                throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
            }
            else if (this.template.Resources[logicalId].Type !== 'AWS::CloudFormation::Stack') {
                throw new Error(`Resource with logical ID '${logicalId}' is not a CloudFormation Stack`);
            }
            else {
                throw new Error(`Nested Stack '${logicalId}' was not included in the parent template. ` +
                    'To retrieve an included nested stack, it must be specified either in the `loadNestedStacks` property, or through the `loadNestedStack` method');
            }
        }
        return this.nestedStacks[logicalId];
    }
    /**
     * Includes a template for a child stack inside of this parent template.
     *
     * A child with this logical ID must exist in the template,
     * and be of type AWS::CloudFormation::Stack.
     * This is equivalent to specifying the value in the {@link CfnIncludeProps.loadNestedStacks}
     * property on object construction.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template.
     * @param nestedStackProps the properties of the included child Stack.
     * @returns the same {@link IncludedNestedStack} object that {@link getNestedStack} returns for this logical ID
     * @stability stable
     */
    loadNestedStack(logicalId, nestedStackProps) {
        if (logicalId in this.nestedStacks) {
            throw new Error(`Nested Stack '${logicalId}' was already included in its parent template`);
        }
        const cfnStack = this.resources[logicalId];
        if (!cfnStack) {
            throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
        }
        if (cfnStack instanceof core.CfnStack) {
            // delete the old CfnStack child - one will be created by the NestedStack object
            this.node.tryRemoveChild(logicalId);
            // remove the previously created CfnStack resource from the resources map
            delete this.resources[logicalId];
            // createNestedStack() (called by getOrCreateResource()) expects this to be filled
            this.nestedStacksToInclude[logicalId] = nestedStackProps;
            this.getOrCreateResource(logicalId);
            return this.nestedStacks[logicalId];
        }
        else {
            throw new Error(`Nested Stack with logical ID '${logicalId}' is not an AWS::CloudFormation::Stack resource`);
        }
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            const self = this;
            const finder = {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _b;
                    return (_b = self.resources[elementName]) !== null && _b !== void 0 ? _b : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            };
            const cfnParser = new cfn_parse.CfnParser({
                finder,
                parameters: this.parametersToReplace,
            });
            switch (section) {
                case 'Conditions':
                case 'Mappings':
                case 'Resources':
                case 'Parameters':
                case 'Rules':
                case 'Hooks':
                case 'Outputs':
                    // these are rendered as a side effect of instantiating the L1s
                    break;
                default:
                    ret[section] = cfnParser.parseValue(this.template[section]);
            }
        }
        return ret;
    }
    createMapping(mappingName) {
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findCondition() { throw new Error('Referring to Conditions in Mapping definitions is not allowed'); },
                findMapping() { throw new Error('Referring to other Mappings in Mapping definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Mapping definitions is not allowed'); },
                findResource() { throw new Error('Using GetAtt expressions in Mapping definitions is not allowed'); },
            },
            parameters: {},
        });
        const cfnMapping = new core.CfnMapping(this.mappingsScope, mappingName, {
            mapping: cfnParser.parseValue(this.template.Mappings[mappingName]),
        });
        this.mappings[mappingName] = cfnMapping;
        this.overrideLogicalIdIfNeeded(cfnMapping, mappingName);
    }
    createParameter(logicalId) {
        if (logicalId in this.parametersToReplace) {
            return;
        }
        const expression = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt expressions in Parameter definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Parameter definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Parameter definitions is not allowed'); },
                findMapping() { throw new Error('Referring to Mappings in Parameter definitions is not allowed'); },
            },
            parameters: {},
        }).parseValue(this.template.Parameters[logicalId]);
        const cfnParameter = new core.CfnParameter(this, logicalId, {
            type: expression.Type,
            default: expression.Default,
            allowedPattern: expression.AllowedPattern,
            allowedValues: expression.AllowedValues,
            constraintDescription: expression.ConstraintDescription,
            description: expression.Description,
            maxLength: expression.MaxLength,
            maxValue: expression.MaxValue,
            minLength: expression.MinLength,
            minValue: expression.MinValue,
            noEcho: expression.NoEcho,
        });
        this.overrideLogicalIdIfNeeded(cfnParameter, logicalId);
        this.parameters[logicalId] = cfnParameter;
    }
    createRule(ruleName) {
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findRefTarget(refTarget) {
                    // only parameters can be referenced in Rules
                    return self.parameters[refTarget];
                },
                findResource() { throw new Error('Using GetAtt expressions in Rule definitions is not allowed'); },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
            context: cfn_parse.CfnParsingContext.RULES,
        });
        const ruleProperties = cfnParser.parseValue(this.template.Rules[ruleName]);
        const rule = new core.CfnRule(this.rulesScope, ruleName, {
            ruleCondition: ruleProperties.RuleCondition,
            assertions: ruleProperties.Assertions,
        });
        this.rules[ruleName] = rule;
        this.overrideLogicalIdIfNeeded(rule, ruleName);
    }
    createHook(hookName) {
        var _b;
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _b;
                    return (_b = self.resources[elementName]) !== null && _b !== void 0 ? _b : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        });
        const hookAttributes = this.template.Hooks[hookName];
        let hook;
        switch (hookAttributes.Type) {
            case 'AWS::CodeDeploy::BlueGreen':
                hook = core.CfnCodeDeployBlueGreenHook._fromCloudFormation(this.hooksScope, hookName, hookAttributes, {
                    parser: cfnParser,
                });
                break;
            default: {
                const hookProperties = (_b = cfnParser.parseValue(hookAttributes.Properties)) !== null && _b !== void 0 ? _b : {};
                hook = new core.CfnHook(this.hooksScope, hookName, {
                    type: hookAttributes.Type,
                    properties: hookProperties,
                });
            }
        }
        this.hooks[hookName] = hook;
        this.overrideLogicalIdIfNeeded(hook, hookName);
    }
    createOutput(logicalId, scope) {
        const self = this;
        const outputAttributes = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _b;
                    return (_b = self.resources[elementName]) !== null && _b !== void 0 ? _b : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        }).parseValue(this.template.Outputs[logicalId]);
        const cfnOutput = new core.CfnOutput(scope, logicalId, {
            value: outputAttributes.Value,
            description: outputAttributes.Description,
            exportName: outputAttributes.Export ? outputAttributes.Export.Name : undefined,
            condition: (() => {
                if (!outputAttributes.Condition) {
                    return undefined;
                }
                else if (this.conditions[outputAttributes.Condition]) {
                    return self.getCondition(outputAttributes.Condition);
                }
                throw new Error(`Output with name '${logicalId}' refers to a Condition with name ` +
                    `'${outputAttributes.Condition}' which was not found in this template`);
            })(),
        });
        this.overrideLogicalIdIfNeeded(cfnOutput, logicalId);
        this.outputs[logicalId] = cfnOutput;
    }
    getOrCreateCondition(conditionName) {
        if (conditionName in this.conditions) {
            return this.conditions[conditionName];
        }
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt in Condition definitions is not allowed'); },
                findRefTarget(elementName) {
                    // only Parameters can be referenced in the 'Conditions' section
                    return self.parameters[elementName];
                },
                findCondition(cName) {
                    return cName in (self.template.Conditions || {})
                        ? self.getOrCreateCondition(cName)
                        : undefined;
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            context: cfn_parse.CfnParsingContext.CONDITIONS,
            parameters: this.parametersToReplace,
        });
        const cfnCondition = new core.CfnCondition(this.conditionsScope, conditionName, {
            expression: cfnParser.parseValue(this.template.Conditions[conditionName]),
        });
        this.overrideLogicalIdIfNeeded(cfnCondition, conditionName);
        this.conditions[conditionName] = cfnCondition;
        return cfnCondition;
    }
    getOrCreateResource(logicalId) {
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const resourceAttributes = this.template.Resources[logicalId];
        // fail early for resource attributes we don't support yet
        const knownAttributes = [
            'Condition', 'DependsOn', 'Description', 'Metadata', 'Properties', 'Type', 'Version',
            'CreationPolicy', 'DeletionPolicy', 'UpdatePolicy', 'UpdateReplacePolicy',
        ];
        for (const attribute of Object.keys(resourceAttributes)) {
            if (!knownAttributes.includes(attribute)) {
                throw new Error(`The '${attribute}' resource attribute is not supported by cloudformation-include yet. ` +
                    'Either remove it from the template, or use the CdkInclude class from the core package instead.');
            }
        }
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findMapping(mappingName) {
                return self.mappings[mappingName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId);
            },
            findRefTarget(elementName) {
                if (elementName in self.parameters) {
                    return self.parameters[elementName];
                }
                return this.findResource(elementName);
            },
        };
        const cfnParser = new cfn_parse.CfnParser({
            finder,
            parameters: this.parametersToReplace,
        });
        let l1Instance;
        if (this.nestedStacksToInclude[logicalId]) {
            l1Instance = this.createNestedStack(logicalId, cfnParser);
        }
        else {
            const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
            // The AWS::CloudFormation::CustomResource type corresponds to the CfnCustomResource class.
            // Unfortunately, it's quite useless; it only has a single property, ServiceToken.
            // For that reason, even the CustomResource class from @core doesn't use it!
            // So, special-case the handling of this one resource type
            if (l1ClassFqn && resourceAttributes.Type !== 'AWS::CloudFormation::CustomResource') {
                const options = {
                    parser: cfnParser,
                };
                const [moduleName, ...className] = l1ClassFqn.split('.');
                const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
                const jsClassFromModule = module[className.join('.')];
                l1Instance = jsClassFromModule._fromCloudFormation(this, logicalId, resourceAttributes, options);
            }
            else {
                l1Instance = new core.CfnResource(this, logicalId, {
                    type: resourceAttributes.Type,
                    properties: cfnParser.parseValue(resourceAttributes.Properties),
                });
                cfnParser.handleAttributes(l1Instance, resourceAttributes, logicalId);
            }
        }
        this.overrideLogicalIdIfNeeded(l1Instance, logicalId);
        this.resources[logicalId] = l1Instance;
        return l1Instance;
    }
    createNestedStack(nestedStackId, cfnParser) {
        const templateResources = this.template.Resources || {};
        const nestedStackAttributes = templateResources[nestedStackId] || {};
        if (nestedStackAttributes.Type !== 'AWS::CloudFormation::Stack') {
            throw new Error(`Nested Stack with logical ID '${nestedStackId}' is not an AWS::CloudFormation::Stack resource`);
        }
        if (nestedStackAttributes.CreationPolicy) {
            throw new Error('CreationPolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        if (nestedStackAttributes.UpdatePolicy) {
            throw new Error('UpdatePolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        const nestedStackProps = cfnParser.parseValue(nestedStackAttributes.Properties);
        const nestedStack = new core.NestedStack(this, nestedStackId, {
            parameters: this.parametersForNestedStack(nestedStackProps.Parameters, nestedStackId),
            notificationArns: nestedStackProps.NotificationArns,
            timeout: nestedStackProps.Timeout,
        });
        const template = new CfnInclude(nestedStack, nestedStackId, this.nestedStacksToInclude[nestedStackId]);
        this.nestedStacks[nestedStackId] = { stack: nestedStack, includedTemplate: template };
        // we know this is never undefined for nested stacks
        const nestedStackResource = nestedStack.nestedStackResource;
        cfnParser.handleAttributes(nestedStackResource, nestedStackAttributes, nestedStackId);
        return nestedStackResource;
    }
    parametersForNestedStack(parameters, nestedStackId) {
        var _b;
        if (parameters == null) {
            return undefined;
        }
        const parametersToReplace = (_b = this.nestedStacksToInclude[nestedStackId].parameters) !== null && _b !== void 0 ? _b : {};
        const ret = {};
        for (const paramName of Object.keys(parameters)) {
            if (!(paramName in parametersToReplace)) {
                ret[paramName] = parameters[paramName];
            }
        }
        return ret;
    }
    overrideLogicalIdIfNeeded(element, id) {
        if (this.preserveLogicalIds) {
            element.overrideLogicalId(id);
        }
    }
}
exports.CfnInclude = CfnInclude;
_a = JSII_RTTI_SYMBOL_1;
CfnInclude[_a] = { fqn: "@aws-cdk/cloudformation-include.CfnInclude", version: "1.96.0" };
//# sourceMappingURL=data:application/json;base64,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