from ....Internal.Core import Core
from ....Internal.CommandsGroup import CommandsGroup
from ....Internal import Conversions


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class Drx:
	"""Drx commands group definition. 3 total commands, 0 Sub-groups, 3 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._base = CommandsGroup("drx", core, parent)

	def get_enable(self) -> bool:
		"""SCPI: CONFigure:GSM:SIGNaling<instance>:CBS:DRX:ENABle \n
		Snippet: value: bool = driver.configure.cbs.drx.get_enable() \n
		Enables DRX for CBS. \n
			:return: enable: OFF | ON
		"""
		response = self._core.io.query_str('CONFigure:GSM:SIGNaling<Instance>:CBS:DRX:ENABle?')
		return Conversions.str_to_bool(response)

	def set_enable(self, enable: bool) -> None:
		"""SCPI: CONFigure:GSM:SIGNaling<instance>:CBS:DRX:ENABle \n
		Snippet: driver.configure.cbs.drx.set_enable(enable = False) \n
		Enables DRX for CBS. \n
			:param enable: OFF | ON
		"""
		param = Conversions.bool_to_str(enable)
		self._core.io.write(f'CONFigure:GSM:SIGNaling<Instance>:CBS:DRX:ENABle {param}')

	def get_length(self) -> int:
		"""SCPI: CONFigure:GSM:SIGNaling<instance>:CBS:DRX:LENGth \n
		Snippet: value: int = driver.configure.cbs.drx.get_length() \n
		Specifies the length of DRX (L) that the MS can use for the processing of particular CB message. Define the value
		matching with the position of the specific CB message within the CBS scheduling period. \n
			:return: length_of_period: Range: 1 to 40
		"""
		response = self._core.io.query_str('CONFigure:GSM:SIGNaling<Instance>:CBS:DRX:LENGth?')
		return Conversions.str_to_int(response)

	def set_length(self, length_of_period: int) -> None:
		"""SCPI: CONFigure:GSM:SIGNaling<instance>:CBS:DRX:LENGth \n
		Snippet: driver.configure.cbs.drx.set_length(length_of_period = 1) \n
		Specifies the length of DRX (L) that the MS can use for the processing of particular CB message. Define the value
		matching with the position of the specific CB message within the CBS scheduling period. \n
			:param length_of_period: Range: 1 to 40
		"""
		param = Conversions.decimal_value_to_str(length_of_period)
		self._core.io.write(f'CONFigure:GSM:SIGNaling<Instance>:CBS:DRX:LENGth {param}')

	def get_offset(self) -> int:
		"""SCPI: CONFigure:GSM:SIGNaling<instance>:CBS:DRX:OFFSet \n
		Snippet: value: int = driver.configure.cbs.drx.get_offset() \n
		Offset (O) within period of scheduling message. This offset is used for the transmission of a scheduling message. \n
			:return: offset: Range: 1 to 39
		"""
		response = self._core.io.query_str('CONFigure:GSM:SIGNaling<Instance>:CBS:DRX:OFFSet?')
		return Conversions.str_to_int(response)

	def set_offset(self, offset: int) -> None:
		"""SCPI: CONFigure:GSM:SIGNaling<instance>:CBS:DRX:OFFSet \n
		Snippet: driver.configure.cbs.drx.set_offset(offset = 1) \n
		Offset (O) within period of scheduling message. This offset is used for the transmission of a scheduling message. \n
			:param offset: Range: 1 to 39
		"""
		param = Conversions.decimal_value_to_str(offset)
		self._core.io.write(f'CONFigure:GSM:SIGNaling<Instance>:CBS:DRX:OFFSet {param}')
