"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const queue_base_1 = require("./queue-base");
const sqs_generated_1 = require("./sqs.generated");
const validate_props_1 = require("./validate-props");
/**
 * What kind of encryption to apply to this queue
 */
var QueueEncryption;
(function (QueueEncryption) {
    /**
     * Messages in the queue are not encrypted
     */
    QueueEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side KMS encryption with a master key managed by SQS.
     */
    QueueEncryption["KMS_MANAGED"] = "MANAGED";
    /**
     * Server-side encryption with a KMS key managed by the user.
     *
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    QueueEncryption["KMS"] = "KMS";
})(QueueEncryption = exports.QueueEncryption || (exports.QueueEncryption = {}));
/**
 * A new Amazon SQS queue
 */
class Queue extends queue_base_1.QueueBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.queueName,
        });
        this.autoCreatePolicy = true;
        validate_props_1.validateProps(props);
        const redrivePolicy = props.deadLetterQueue
            ? {
                deadLetterTargetArn: props.deadLetterQueue.queue.queueArn,
                maxReceiveCount: props.deadLetterQueue.maxReceiveCount
            }
            : undefined;
        const { encryptionMasterKey, encryptionProps } = _determineEncryptionProps.call(this);
        const fifoProps = this.determineFifoProps(props);
        this.fifo = fifoProps.fifoQueue || false;
        const queue = new sqs_generated_1.CfnQueue(this, 'Resource', Object.assign(Object.assign(Object.assign({ queueName: this.physicalName }, fifoProps), encryptionProps), { redrivePolicy, delaySeconds: props.deliveryDelay && props.deliveryDelay.toSeconds(), maximumMessageSize: props.maxMessageSizeBytes, messageRetentionPeriod: props.retentionPeriod && props.retentionPeriod.toSeconds(), receiveMessageWaitTimeSeconds: props.receiveMessageWaitTime && props.receiveMessageWaitTime.toSeconds(), visibilityTimeout: props.visibilityTimeout && props.visibilityTimeout.toSeconds() }));
        this.queueArn = this.getResourceArnAttribute(queue.attrArn, {
            service: 'sqs',
            resource: this.physicalName,
        });
        this.queueName = this.getResourceNameAttribute(queue.attrQueueName);
        this.encryptionMasterKey = encryptionMasterKey;
        this.queueUrl = queue.ref;
        function _determineEncryptionProps() {
            let encryption = props.encryption || QueueEncryption.UNENCRYPTED;
            if (encryption !== QueueEncryption.KMS && props.encryptionMasterKey) {
                encryption = QueueEncryption.KMS; // KMS is implied by specifying an encryption key
            }
            if (encryption === QueueEncryption.UNENCRYPTED) {
                return { encryptionProps: {} };
            }
            if (encryption === QueueEncryption.KMS_MANAGED) {
                return {
                    encryptionProps: {
                        kmsMasterKeyId: 'alias/aws/sqs',
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds()
                    }
                };
            }
            if (encryption === QueueEncryption.KMS) {
                const masterKey = props.encryptionMasterKey || new kms.Key(this, 'Key', {
                    description: `Created by ${this.node.path}`
                });
                return {
                    encryptionMasterKey: masterKey,
                    encryptionProps: {
                        kmsMasterKeyId: masterKey.keyArn,
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds()
                    }
                };
            }
            throw new Error(`Unexpected 'encryptionType': ${encryption}`);
        }
    }
    static fromQueueArn(scope, id, queueArn) {
        return Queue.fromQueueAttributes(scope, id, { queueArn });
    }
    /**
     * Import an existing queue
     */
    static fromQueueAttributes(scope, id, attrs) {
        const stack = core_1.Stack.of(scope);
        const queueName = attrs.queueName || stack.parseArn(attrs.queueArn).resource;
        const queueUrl = attrs.queueUrl || `https://sqs.${stack.region}.${stack.urlSuffix}/${stack.account}/${queueName}`;
        class Import extends queue_base_1.QueueBase {
            constructor() {
                super(...arguments);
                this.queueArn = attrs.queueArn; // arn:aws:sqs:us-east-1:123456789012:queue1
                this.queueUrl = queueUrl;
                this.queueName = queueName;
                this.encryptionMasterKey = attrs.keyArn
                    ? kms.Key.fromKeyArn(this, 'Key', attrs.keyArn)
                    : undefined;
                this.fifo = queueName.endsWith('.fifo') ? true : false;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Look at the props, see if the FIFO props agree, and return the correct subset of props
     */
    determineFifoProps(props) {
        // Check if any of the signals that we have say that this is a FIFO queue.
        let fifoQueue = props.fifo;
        const queueName = props.queueName;
        if (typeof fifoQueue === 'undefined' && queueName && !core_1.Token.isUnresolved(queueName) && queueName.endsWith('.fifo')) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.contentBasedDeduplication) {
            fifoQueue = true;
        }
        // If we have a name, see that it agrees with the FIFO setting
        if (typeof queueName === 'string') {
            if (fifoQueue && !queueName.endsWith('.fifo')) {
                throw new Error("FIFO queue names must end in '.fifo'");
            }
            if (!fifoQueue && queueName.endsWith('.fifo')) {
                throw new Error("Non-FIFO queue name may not end in '.fifo'");
            }
        }
        if (props.contentBasedDeduplication && !fifoQueue) {
            throw new Error('Content-based deduplication can only be defined for FIFO queues');
        }
        return {
            contentBasedDeduplication: props.contentBasedDeduplication,
            fifoQueue,
        };
    }
}
exports.Queue = Queue;
//# sourceMappingURL=data:application/json;base64,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