"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const sqs = require("../lib");
const lib_1 = require("../lib");
function testGrant(action, ...expectedActions) {
    const stack = new core_1.Stack();
    const queue = new lib_1.Queue(stack, 'MyQueue');
    const principal = new iam.User(stack, 'User');
    action(queue, principal);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        "PolicyDocument": {
            "Statement": [
                {
                    "Action": expectedActions,
                    "Effect": "Allow",
                    "Resource": {
                        "Fn::GetAtt": [
                            "MyQueueE6CA6235",
                            "Arn"
                        ]
                    }
                }
            ],
            "Version": "2012-10-17"
        }
    }));
}
module.exports = {
    'default properties'(test) {
        const stack = new core_1.Stack();
        const q = new sqs.Queue(stack, 'Queue');
        test.deepEqual(q.fifo, false);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Queue4A7E3555": {
                    "Type": "AWS::SQS::Queue"
                }
            }
        });
        test.done();
    },
    'with a dead letter queue'(test) {
        const stack = new core_1.Stack();
        const dlq = new sqs.Queue(stack, 'DLQ');
        new sqs.Queue(stack, 'Queue', { deadLetterQueue: { queue: dlq, maxReceiveCount: 3 } });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "DLQ581697C4": {
                    "Type": "AWS::SQS::Queue"
                },
                "Queue4A7E3555": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "RedrivePolicy": {
                            "deadLetterTargetArn": {
                                "Fn::GetAtt": [
                                    "DLQ581697C4",
                                    "Arn"
                                ]
                            },
                            "maxReceiveCount": 3
                        }
                    }
                }
            }
        });
        test.done();
    },
    'addToPolicy will automatically create a policy for this queue'(test) {
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'MyQueue');
        queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['sqs:*'],
            principals: [new iam.ArnPrincipal('arn')]
        }));
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyQueueE6CA6235": {
                    "Type": "AWS::SQS::Queue"
                },
                "MyQueuePolicy6BBEDDAC": {
                    "Type": "AWS::SQS::QueuePolicy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:*",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": "arn"
                                    },
                                    "Resource": "*"
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "Queues": [
                            {
                                "Ref": "MyQueueE6CA6235"
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'export and import': {
        'importing works correctly'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            const imports = sqs.Queue.fromQueueArn(stack, 'Imported', 'arn:aws:sqs:us-east-1:123456789012:queue1');
            // THEN
            // "import" returns an IQueue bound to `Fn::ImportValue`s.
            test.deepEqual(stack.resolve(imports.queueArn), 'arn:aws:sqs:us-east-1:123456789012:queue1');
            test.deepEqual(stack.resolve(imports.queueUrl), { 'Fn::Join': ['', ['https://sqs.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/', { Ref: 'AWS::AccountId' }, '/queue1']] });
            test.deepEqual(stack.resolve(imports.queueName), 'queue1');
            test.done();
        },
        'importing fifo and standard queues are detected correctly'(test) {
            const stack = new core_1.Stack();
            const stdQueue = sqs.Queue.fromQueueArn(stack, 'StdQueue', 'arn:aws:sqs:us-east-1:123456789012:queue1');
            const fifoQueue = sqs.Queue.fromQueueArn(stack, 'FifoQueue', 'arn:aws:sqs:us-east-1:123456789012:queue2.fifo');
            test.deepEqual(stdQueue.fifo, false);
            test.deepEqual(fifoQueue.fifo, true);
            test.done();
        },
    },
    'grants': {
        'grantConsumeMessages'(test) {
            testGrant((q, p) => q.grantConsumeMessages(p), 'sqs:ReceiveMessage', 'sqs:ChangeMessageVisibility', 'sqs:GetQueueUrl', 'sqs:DeleteMessage', 'sqs:GetQueueAttributes');
            test.done();
        },
        'grantSendMessages'(test) {
            testGrant((q, p) => q.grantSendMessages(p), 'sqs:SendMessage', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
            test.done();
        },
        'grantPurge'(test) {
            testGrant((q, p) => q.grantPurge(p), 'sqs:PurgeQueue', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
            test.done();
        },
        'grant() is general purpose'(test) {
            testGrant((q, p) => q.grant(p, 'service:hello', 'service:world'), 'service:hello', 'service:world');
            test.done();
        },
        'grants also work on imported queues'(test) {
            const stack = new core_1.Stack();
            const queue = lib_1.Queue.fromQueueAttributes(stack, 'Import', {
                queueArn: 'arn:aws:sqs:us-east-1:123456789012:queue1',
                queueUrl: 'https://queue-url'
            });
            const user = new iam.User(stack, 'User');
            queue.grantPurge(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                "sqs:PurgeQueue",
                                "sqs:GetQueueAttributes",
                                "sqs:GetQueueUrl"
                            ],
                            "Effect": "Allow",
                            "Resource": "arn:aws:sqs:us-east-1:123456789012:queue1"
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        }
    },
    'queue encryption': {
        'encryptionMasterKey can be set to a custom KMS key'(test) {
            const stack = new core_1.Stack();
            const key = new kms.Key(stack, 'CustomKey');
            const queue = new sqs.Queue(stack, 'Queue', { encryptionMasterKey: key });
            test.same(queue.encryptionMasterKey, key);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
                "KmsMasterKeyId": { "Fn::GetAtt": ["CustomKey1E6D0D07", "Arn"] }
            }));
            test.done();
        },
        'a kms key will be allocated if encryption = kms but a master key is not specified'(test) {
            const stack = new core_1.Stack();
            new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.KMS });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
                "KmsMasterKeyId": {
                    "Fn::GetAtt": [
                        "QueueKey39FCBAE6",
                        "Arn"
                    ]
                }
            }));
            test.done();
        },
        'it is possible to use a managed kms key'(test) {
            const stack = new core_1.Stack();
            new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.KMS_MANAGED });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "Queue4A7E3555": {
                        "Type": "AWS::SQS::Queue",
                        "Properties": {
                            "KmsMasterKeyId": "alias/aws/sqs"
                        }
                    }
                }
            });
            test.done();
        },
        'grant also affects key on encrypted queue'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const queue = new sqs.Queue(stack, 'Queue', {
                encryption: sqs.QueueEncryption.KMS
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('someone')
            });
            // WHEN
            queue.grantSendMessages(role);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                "sqs:SendMessage",
                                "sqs:GetQueueAttributes",
                                "sqs:GetQueueUrl"
                            ],
                            "Effect": "Allow",
                            "Resource": { "Fn::GetAtt": ["Queue4A7E3555", "Arn"] }
                        },
                        {
                            "Action": [
                                "kms:Encrypt",
                                "kms:ReEncrypt*",
                                "kms:GenerateDataKey*"
                            ],
                            "Effect": "Allow",
                            "Resource": { "Fn::GetAtt": ["QueueKey39FCBAE6", "Arn"] }
                        }
                    ],
                    "Version": "2012-10-17"
                },
            }));
            test.done();
        },
    },
    'test ".fifo" suffixed queues register as fifo'(test) {
        const stack = new core_1.Stack();
        const queue = new lib_1.Queue(stack, 'Queue', {
            queueName: 'MyQueue.fifo'
        });
        test.deepEqual(queue.fifo, true);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Queue4A7E3555": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "QueueName": "MyQueue.fifo",
                        "FifoQueue": true
                    }
                }
            }
        });
        test.done();
    },
    'test a fifo queue is observed when the "fifo" property is specified'(test) {
        const stack = new core_1.Stack();
        const queue = new lib_1.Queue(stack, 'Queue', {
            fifo: true
        });
        test.deepEqual(queue.fifo, true);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Queue4A7E3555": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "FifoQueue": true
                    }
                }
            }
        });
        test.done();
    },
    'test metrics'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const queue = new lib_1.Queue(stack, 'Queue');
        // THEN
        test.deepEqual(stack.resolve(queue.metricNumberOfMessagesSent()), {
            dimensions: { QueueName: { 'Fn::GetAtt': ['Queue4A7E3555', 'QueueName'] } },
            namespace: 'AWS/SQS',
            metricName: 'NumberOfMessagesSent',
            period: core_1.Duration.minutes(5),
            statistic: 'Sum'
        });
        test.deepEqual(stack.resolve(queue.metricSentMessageSize()), {
            dimensions: { QueueName: { 'Fn::GetAtt': ['Queue4A7E3555', 'QueueName'] } },
            namespace: 'AWS/SQS',
            metricName: 'SentMessageSize',
            period: core_1.Duration.minutes(5),
            statistic: 'Average'
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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