# MELTS for Excel Google App Engine Project #

**MELTS** is a software package designed to facilitate thermodynamic modeling of phase equilibria in magmatic systems. It provides the ability to compute equilibrium phase relations for igneous systems over the temperature range 500-2000 °C and the pressure range 0-4 GPa.

This project contains source and configuration files for a Google App Engine implementation of the MELTS for Excel web services. The project is forked from xMelts, which contains the original Objective-C version that utilizes the BomBax webservice engine. That code is completely rewritten into Python, but still uses the underlying C-code MELTS engine. Cython is used to bridge Python and C. While the app is designed for Google App Engine, it may be implemented on any server that supports an internet connection, Python and a C compiler. 

The App maintains state by passing a session ID via browser cookies and/or XML I/O, which identifies the contact request and stores results in a datastore in Google Cloud. The store expires after 1 hour if subsequent requests from that session ID are not made. Expired session IDs are ignored. All input XML is obtained from webservice POST requests and that XML is verified against a schema. All output is delivered as XML. I/O XML schema are documented at the default service endpoint.


## Cloud Deployment

The GitLab CI Yaml file in the source repository installs the App into Google Cloud using credentials specified in Settings->CI/CD->Variables.

## PyPI package

An installable Python package (e.g., *pip install ...*) is available at https://pypi.org/project/melts-cython-framework/. The package installs as:
    ```
    pip install melts-cython-framework
    ```
and requirs a C-compiler along with a Python 3.6+ installation. This package is deployed from the GitLab repository using the pipeline generated by the  GitLab CI Yaml file.

## Local Development/Deployment

The following configuration commands should be executed in a terminal window.

1. Install the Google Coud SDK, **gcloud**, if not already installed:
    ```
    curl https://sdk.cloud.google.com | bash
    ```
    and initialize the SDK against a Google Cloud project ID that has been previously established at https://console.cloud.google.com. 
    ```
    gcloud init
    ```

1. Install the Google Cloud Datastore Python API and the datastore emulator 

    Python API docs: https://googleapis.dev/python/datastore/latest/entities.html
    ```
    pip install google-cloud-datastore
    gcloud components install cloud-datastore-emulator
    ```

1. Execute the following datastore emulator configuration commands
    
    (authentication docs: https://googleapis.dev/python/google-api-core/latest/auth.html)
    ```
    gcloud auth application-default login
    gcloud auth login
    gcloud beta emulators datastore start
    (place in background)
    (gcloud beta emulators datastore env-init)
    ```

1. Clone the repository, navigate to the project root, and build and install the MELTS Python framework
    ```
    make install
    ```
    
1. Either, start the server for local (single-user) testing and debugging
    ```
    python main.py
    ```
    
1. Or, start a **gunicorn** web server for production access
    ```
    gunicorn -w 4 —-reload -b xxx.xxx.xxx.xxx:8080 -—timeout 5 main:app
    ```
    where
    - xxx.xxx.xxx.xxx is the IP address of the machine that will serve the requests (the machine the above command is executed on)
    - -w n starts n worker threads (4 here) to process remote requests. Two workers per compute core is recommended
    - --reload reloads workers if a failure occurs
    - 8080 is the port number that exposes the service
    - --timeout n is the time a worker thread is allowed to respond to a particular request (in this case 5 seconds)
    - and, main:app is the endpoint service (app) in the file main.py

1. To monitor local datastore emulation execute this command in a separate terminal window:
    ```
    docker run -p 8080:8080 ghcr.io/remko/dsadmin:latest --project=pelagic-script-244221 --datastore-emulator-host=host.docker.internal:8081
    ```
    where *project* points to the Google Cloud Project associated with your *auth login* account. 


**There are four endpoints available for testing.**

- Test the default service endpoint using a browser at
    ```
    http://127.0.0.1:8080/
    ```
    which should retrieve a genrated HTML page with links to the webservice XML schema documentation.

To test endpoints that retrieve XML data, open another terminal window and execute the following **curl** commands:
- Get a list of oxides known to MELTS
    ```
    curl -X POST -H "Content-Type: text/xml" -i http://127.0.0.1:8080/Oxides
    ```
    which should return a list of oxides known to MELTS:
    ```
    HTTP/1.0 200 OK
    Content-Type: application/xml; charset=utf-8
    Content-Length: 388
    Server: Werkzeug/1.0.1 Python/3.8.8
    Date: Thu, 11 Nov 2021 17:00:16 GMT

    <MELTSWSoxides><Oxide>SiO2</Oxide><Oxide>TiO2</Oxide><Oxide>Al2O3</Oxide><Oxide>Fe2O3</Oxide><Oxide>Cr2O3</Oxide><Oxide>FeO</Oxide><Oxide>MnO</Oxide><Oxide>MgO</Oxide><Oxide>NiO</Oxide><Oxide>CoO</Oxide><Oxide>CaO</Oxide><Oxide>Na2O</Oxide><Oxide>K2O</Oxide><Oxide>P2O5</Oxide><Oxide>H2O</Oxide><Oxide>CO2</Oxide><Oxide>SO3</Oxide><Oxide>Cl2O-1</Oxide><Oxide>F2O-1</Oxide></MELTSWSoxides>
    ```
- Get a list of phases known to MELTS for the modelType specified:
    ```
    curl -X POST -H "Content-Type: text/xml" -d 'MELTS_v1.0.x' -i http://127.0.0.1:8080/Phases
    ```
    which should return a list of phases known to MELTS_v1.0.x (rhyolite-MELTS)
    ```
    HTTP/1.0 200 OK
    Content-Type: application/xml; charset=utf-8
    Content-Length: 931
    Server: Werkzeug/1.0.1 Python/3.8.8
    Date: Thu, 11 Nov 2021 17:04:48 GMT

    <MELTSWSphases><Phase>olivine</Phase><Phase>fayalite</Phase><Phase>sphene</Phase><Phase>garnet</Phase><Phase>melilite</Phase><Phase>orthopyroxene</Phase><Phase>clinopyroxene</Phase><Phase>aegirine</Phase><Phase>aenigmatite</Phase><Phase>cummingtonite</Phase><Phase>amphibole</Phase><Phase>hornblende</Phase><Phase>biotite</Phase><Phase>muscovite</Phase><Phase>feldspar</Phase><Phase>quartz</Phase><Phase>tridymite</Phase><Phase>cristobalite</Phase><Phase>nepheline</Phase><Phase>kalsilite</Phase><Phase>leucite</Phase><Phase>corundum</Phase><Phase>sillimanite</Phase><Phase>rutile</Phase><Phase>perovskite</Phase><Phase>spinel</Phase><Phase>rhm-oxide</Phase><Phase>ortho-oxide</Phase><Phase>whitlockite</Phase><Phase>apatite</Phase><Phase>water</Phase><Phase>alloy-solid</Phase><Phase>alloy-liquid</Phase><Phase>olivine</Phase><Phase>fayalite</Phase><Phase>sphene</Phase><Phase>garnet</Phase><Phase>melilite</Phase></MELTSWSphases>
    ```
    Similarly, 
    ```
    curl -X POST -H "Content-Type: text/xml" -d 'MELTS_v1.1.x' -i http://127.0.0.1:8080/Phases
    ```
    generates a list of known phases for MELTS_v1.1.x, 
    ```
    curl -X POST -H "Content-Type: text/xml" -d 'MELTS_v1.2.x' -i http://127.0.0.1:8080/Phases
    ```
    generates a list of known phases for MELTS_v1.2.x, and 
    ```
    curl -X POST -H "Content-Type: text/xml" -d 'MELTS_v5.6.1' -i http://127.0.0.1:8080/Phases
    ```
    generates a list of known phases for MELTS_v5.6.1 (pMELTS).

- Perform a MELTS calculation for a specified input composition at fixed T and P:

    The **Test** endpoint specified here generates DEBUG printout to the terminal running the main app. Substitute **Compute** for **Test** to eliminate this debuging information
    ```
    curl -X POST -H "Content-Type: text/xml" -d '<MELTSinput><initialize><modelSelection>MELTS_v1.2.x</modelSelection><SiO2>77.7</SiO2><TiO2>0.08</TiO2><Al2O3>12.5</Al2O3><Fe2O3>0.1924</Fe2O3><FeO>0.4869</FeO><MgO>0.03</MgO><CaO>0.43</CaO><Na2O>3.99</Na2O><K2O>4.89</K2O><H2O>5.47</H2O></initialize><calculationMode>equilibrate</calculationMode><title>MELTS Web Service called from Excel. [created 2014Apr23] (version 1.0.3)</title><sessionID></sessionID><constraints><setTP><initialT>757</initialT><initialP>1750</initialP><fo2Path>nno</fo2Path><fo2Offset>0</fo2Offset></setTP></constraints><suppressPhase>amphibole</suppressPhase><suppressPhase>leucite</suppressPhase><fractionationMode>fractionateNone</fractionationMode></MELTSinput>' -i http://127.0.0.1:8080/Test
    ```
    which will generate output like (note that the XML string is long and continues on one line):
    ```
    HTTP/1.0 200 OK
    Content-Type: application/xml; charset=utf-8
    Content-Length: 30742
    Server: Werkzeug/1.0.1 Python/3.8.8
    Date: Thu, 11 Nov 2021 00:53:30 GMT

    <?xml version="1.0" encoding="UTF-8"?><MELTSoutput><status>Success: Equilibrate</status><sessionID>1636592010.112680.9746532025001503</sessionID><title>MELTS Web Service called from Excel. [created 2014Apr23] (version 1.0.3)</title><time>Wed Nov 10 16:53:30 2021</time><release>MELTS Web Services, MELTS v.1.2.x</release><buildDate>Nov  8 2021</buildDate><buildTime>13:34:31</buildTime><temperature>757.00000000000011369</temperature><pressure>1750</pressure><log_fO2>-14.84035917099451396</log_fO2><deltaHM>-5.1707207688200789164</deltaHM><deltaNNO>0</deltaNNO><deltaFMQ>0.40942629714119505024</deltaFMQ><deltaCOH>1.8933933165073035809</deltaCOH><deltaIW>4.7379540358200245009</deltaIW><liquid><mass>95.257878539150439678</mass><density>2.1684053000488523466</density><viscosity>5.9923403370279624269</viscosity><gibbsFreeEnergy>-1554251.1308688730933</gibbsFreeEnergy><enthalpy>-1300871.1365733931307</enthalpy><entropy>245.96417443622732435</entropy><volume>43.929923311386644968</volume><dvdt>0.0045999280015925087667</dvdt><dvdp>-0.00044198957092661072914</dvdp><d2vdt2>0</d2vdt2><d2vdtdp>7.3878782721609045846e-08</d2vdtdp><d2vdp2>5.5817298977780984587e-09</d2vdp2><heatCapacity>127.88288035137705378</heatCapacity><SiO2>74.341358532970559736</SiO2><TiO2>0.083982554752269206189</TiO2><Al2O3>10.88611714542891562</Al2O3><Fe2O3>0.21607004335321830513</Fe2O3><FeO>0.49845868783029528837</FeO><MgO>0.031493458032104112987</MgO><CaO>0.29280491120025436969</CaO><Na2O>3.1096193928343995871</Na2O><K2O>4.9738561246276438155</K2O><H2O>5.5662391489703306746</H2O></liquid><solid><name>feldspar</name><formula>K0.08Na0.85Ca0.07Al1.07Si2.93O8</formula><mass>10.345047699436058508</mass><density>2.562169247797019711</density><gibbsFreeEnergy>-168016.33122939470923</gibbsFreeEnergy><enthalpy>-145522.16198166285176</enthalpy><entropy>21.835819295958692976</entropy><volume>4.037612936120765994</volume><dvdt>0.00013041451146846285369</dvdt><dvdp>-7.459424660115498221e-06</dvdp><d2vdt2>-1.5680651273945810772e-07</d2vdt2><d2vdtdp>5.4296495658622212108e-10</d2vdtdp><d2vdp2>3.610457577397163922e-11</d2vdp2><heatCapacity>12.68528739053502008</heatCapacity><SiO2>66.543907619369150552</SiO2><Al2O3>20.590680815261102055</Al2O3><CaO>1.4604113748114235793</CaO><Na2O>9.9355997730193781337</Na2O><K2O>1.4694004175389612232</K2O><component><name>albite</name><formula>NaAlSi3O8</formula><moleFraction>0.84851623873020143307</moleFraction></component><component><name>anorthite</name><formula>CaAl2Si2O8</formula><moleFraction>0.068921078772056484918</moleFraction></component><component><name>sanidine</name><formula>KAlSi3O8</formula><moleFraction>0.082562682497742193033</moleFraction></component></solid><solid><name>fluid</name><formula>H2O1.00CO20.00</formula><mass>0.16771867227517700538</mass><density>0.41398884832569243786</density><gibbsFreeEnergy>-3667.8008291880564684</gibbsFreeEnergy><enthalpy>-2118.1634029211199959</enthalpy><entropy>1.5042832852176251102</entropy><volume>0.40512847858942740986</volume><dvdt>0.00068382079960127096011</dvdt><dvdp>-0.00017260896481820765217</dvdp><d2vdt2>8.4631419340172041256e-08</d2vdt2><d2vdtdp>-5.2005852520757845594e-07</d2vdtdp><d2vdp2>2.4927007813864158492e-07</d2vdp2><heatCapacity>0.68897982980606031767</heatCapacity><H2O>100</H2O><component><name>h2oduan</name><formula>H2O</formula><moleFraction>1</moleFraction></component><component><name>co2duan</name><formula>CO2</formula><moleFraction>0</moleFraction></component></solid><totalSolids><mass>10.51276637171123518</mass><density>2.3662791484786418472</density><gibbsFreeEnergy>-171684.13205858276342</gibbsFreeEnergy><enthalpy>-147640.32538458396448</enthalpy><entropy>23.340102581176317642</entropy><volume>4.4427414147101940145</volume><dvdt>0.00081423531106973373248</dvdt><dvdp>-0.00018006838947832315293</dvdp><d2vdt2>-7.2175093399286079694e-08</d2vdt2><d2vdtdp>-5.1951556025099213797e-07</d2vdtdp><d2vdp2>2.4930618271441561351e-07</d2vdp2><heatCapacity>13.374267220341080176</heatCapacity></totalSolids><system><mass>105.77064491086167664</mass><density>2.1865788355835369394</density><viscosity>6.1686280808412492505</viscosity><gibbsFreeEnergy>-1725935.2629274558276</gibbsFreeEnergy><enthalpy>-1448511.4619579771534</enthalpy><entropy>269.3042770174036491</entropy><volume>48.372664726096843424</volume><dvdt>43.930737546697713469</dvdt><dvdp>43.929743242997162156</dvdp><d2vdt2>43.929923239211554176</d2vdt2><d2vdtdp>43.929922791871085508</d2vdtdp><d2vdp2>43.929923560692827778</d2vdp2><heatCapacity>141.25714757171812153</heatCapacity></system><oxygen><moles>1.5456642644947953258</moles><mass>49.459401666716054535</mass><gibbsFreeEnergy>-352754.05785415176069</gibbsFreeEnergy><enthalpy>36713.579631822474767</enthalpy><entropy>378.06886131725883615</entropy><volume>132385.7755357689457</volume><dvdt>128.51116394289076084</dvdt><dvdp>-132385.7755357689457</dvdp><d2vdt2>0</d2vdt2><d2vdtdp>-128.51116394289076084</d2vdtdp><d2vdp2>264771.5510715378914</d2vdp2><heatCapacity>54.031508313935177057</heatCapacity></oxygen><potentialSolid><name>olivine</name><formula>(Ca0.00Mg1.00Fe''0.00Mn0.00Co0.00Ni0.00)2SiO4</formula><affinity>7579.1312451820931528</affinity><density>3.1440662472830926966</density><gibbsFreeEnergy>-5.4230202876351501464e+83</gibbsFreeEnergy><enthalpy>-4.7525493617563887443e+83</enthalpy><entropy>6.5084786281489260325e+79</entropy><volume>1.0363107332329559566e+79</volume><dvdt>4.2910806216235009185e+74</dvdt><dvdp>-7.9499911193997256282e+72</dvdp><d2vdt2>1.792330362851599423e+71</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>2.7319828057411014978e+67</d2vdp2><heatCapacity>4.0768209595892479772e+79</heatCapacity><SiO2>42.705932273864149806</SiO2><FeO>3.5685136644920424962e-76</FeO><MnO>-4.354352439886222311e-76</MnO><MgO>57.294067726135857299</MgO><NiO>-4.5858892230676042129e-76</NiO><CoO>-4.5995899150098373967e-76</CoO><CaO>1.7961988338484700512e-79</CaO><component><name>tephroite</name><formula>Mn2SiO4</formula><moleFraction>-1</moleFraction></component><component><name>fayalite</name><formula>Fe2SiO4</formula><moleFraction>0.8091593206278746564</moleFraction></component><component><name>co-olivine</name><formula>Co2SiO4</formula><moleFraction>-1</moleFraction></component><component><name>ni-olivine</name><formula>Ni2SiO4</formula><moleFraction>-0.99999999999999988898</moleFraction></component><component><name>monticellite</name><formula>CaMgSiO4</formula><moleFraction>0.0010435967939094979717</moleFraction></component><component><name>forsterite</name><formula>Mg2SiO4</formula><moleFraction>2.3158417847463239085e+77</moleFraction></component></potentialSolid><potentialSolid><name>fayalite</name><formula>Fe2SiO4</formula><affinity>9108.6775469835847616</affinity><density>4.3084842135945784491</density><gibbsFreeEnergy>-1712246.0826108332258</gibbsFreeEnergy><enthalpy>-1346921.4463664137293</enthalpy><entropy>354.6324673537052945</entropy><volume>47.296703410684727942</volume><dvdt>0.0015528307947007889515</dvdt><dvdp>-4.1211945089610977212e-05</dvdp><d2vdt2>3.1626099977907446008e-07</d2vdt2><d2vdtdp>-8.3935294182468328446e-09</d2vdtdp><d2vdp2>2.2276327509300604124e-10</d2vdp2><heatCapacity>188.80306934745576086</heatCapacity><SiO2>29.485305267373497884</SiO2><FeO>70.514694732626509222</FeO><component><name>fayalite</name><formula>Fe2SiO4</formula><moleFraction>1</moleFraction></component></potentialSolid><potentialSolid><name>sphene</name><formula>CaTiSiO5</formula><affinity>26897.596473163226619</affinity><density>3.4628322124270947135</density><gibbsFreeEnergy>-2810815.8750947774388</gibbsFreeEnergy><enthalpy>-2455258.4388061463833</enthalpy><entropy>345.15112972735136054</entropy><volume>56.619116368500002068</volume><dvdt>0.001402379999999999946</dvdt><dvdp>-3.2833500000000003362e-05</dvdp><d2vdt2>0</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>0</d2vdp2><heatCapacity>197.9257603423638443</heatCapacity><SiO2>30.645482945489366244</SiO2><TiO2>40.751699075549950635</TiO2><CaO>28.602817978960676015</CaO><component><name>sphene</name><formula>CaTiSiO5</formula><moleFraction>1</moleFraction></component></potentialSolid><potentialSolid><name>garnet</name><formula>(Ca0.06Fe''0.94Mg-0.0)3Al2Si3O12</formula><affinity>16038.279174851631979</affinity><density>4.2022376834114734123</density><gibbsFreeEnergy>-5073166.9336785133928</gibbsFreeEnergy><enthalpy>-4291936.595664572902</enthalpy><entropy>758.36561472983623844</entropy><volume>101.06248500401342483</volume><dvdt>0.0029410756273935611746</dvdt><dvdp>-5.5940741258933679284e-05</dvdp><d2vdt2>1.488115550226177773e-06</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>8.1455428790234609529e-11</d2vdp2><heatCapacity>432.39197148916929336</heatCapacity><SiO2>36.429620714298700079</SiO2><Al2O3>20.60666296045392798</Al2O3><FeO>40.839032348352311885</FeO><MgO>1.9734789331580166192e-308</MgO><CaO>2.1246839768950609439</CaO><component><name>almandine</name><formula>Fe3Al2Si3O12</formula><moleFraction>0.80467361730591602242</moleFraction></component><component><name>grossular</name><formula>Ca3Al2Si3O12</formula><moleFraction>0.053634015769271632668</moleFraction></component><component><name>pyrope</name><formula>Mg3Al2Si3O12</formula><moleFraction>6.9315341206167105426e-310</moleFraction></component></potentialSolid><potentialSolid><name>melilite</name><formula>Na0.00Ca2.00Al0.44Mg0.07Fe0.71Si1.78O7</formula><affinity>52142.519817109096039</affinity><density>3.117639262105719844</density><gibbsFreeEnergy>-3849564.5342419226654</gibbsFreeEnergy><enthalpy>-3283138.4230818878859</enthalpy><entropy>549.84818828329343887</entropy><volume>91.198481088691096375</volume><dvdt>0.0028836338045721853808</dvdt><dvdp>-7.3203179138586326608e-05</dvdp><d2vdt2>9.4250650739754747119e-07</d2vdt2><d2vdtdp>2.2694263156378066386e-317</d2vdtdp><d2vdp2>5.4541544011991649602e-11</d2vdp2><heatCapacity>295.13232099760455185</heatCapacity><SiO2>36.199620407333121364</SiO2><Al2O3>7.5828619161485102396</Al2O3><FeO>17.35127948136941356</FeO><MgO>0.90886621358362984502</MgO><CaO>37.957371981565316332</CaO><Na2O>1.5109759084320126578e-308</Na2O><component><name>akermanite</name><formula>Ca2MgSi2O7</formula><moleFraction>0.064115194807769004814</moleFraction></component><component><name>gehlenite</name><formula>Ca2Al2SiO7</formula><moleFraction>0.21145177544057200403</moleFraction></component><component><name>iron-akermanite</name><formula>Ca2FeSi2O7</formula><moleFraction>0.68665717209745003036</moleFraction></component><component><name>soda-melilite</name><formula>Na2Si3O7</formula><moleFraction>6.9314941775022545732e-310</moleFraction></component></potentialSolid><potentialSolid><name>orthopyroxene</name><formula>opx Na0.00Ca0.97Fe''-0.0Mg0.66Fe'''0.39Ti0.02Al0.43Si1.57O6</formula><affinity>7136.3917030187440105</affinity><density>3.3537909899844033923</density><gibbsFreeEnergy>-8444254.421733180061</gibbsFreeEnergy><enthalpy>-7321416.976698409766</enthalpy><entropy>1089.9747076006110547</entropy><volume>171.41513309363276107</volume><dvdt>0.0059357575037292057557</dvdt><dvdp>-0.00014373637332567071843</dvdp><d2vdt2>2.3387857782379923605e-06</d2vdt2><d2vdtdp>8.6315104769863390569e-25</d2vdtdp><d2vdp2>6.2959636348342847057e-10</d2vdp2><heatCapacity>636.97121670997307774</heatCapacity><SiO2>41.49330801895345644</SiO2><TiO2>0.72314557458329731254</TiO2><Al2O3>9.5849997496861085722</Al2O3><Fe2O3>13.729902497216519563</Fe2O3><FeO>-1.010696726001766077</FeO><MgO>11.611500941950932742</MgO><CaO>23.86783994361145389</CaO><component><name>diopside</name><formula>CaMgSi2O6</formula><moleFraction>1.4409120130395085546</moleFraction></component><component><name>clinoenstatite</name><formula>Mg2Si2O6</formula><moleFraction>0.081643765544061969175</moleFraction></component><component><name>hedenbergite</name><formula>CaFeSi2O6</formula><moleFraction>-0.080872524633844103392</moleFraction></component><component><name>alumino-buffonite</name><formula>CaTi0.5Mg0.5AlSiO6</formula><moleFraction>0.098189869630887313678</moleFraction></component><component><name>buffonite</name><formula>CaTi0.5Mg0.5FeSiO6</formula><moleFraction>0.0058741568265152574457</moleFraction></component><component><name>essenite</name><formula>CaFeAlSiO6</formula><moleFraction>0.98267643094134193849</moleFraction></component><component><name>jadeite</name><formula>NaAlSi2O6</formula><moleFraction>0</moleFraction></component></potentialSolid><potentialSolid><name>clinopyroxene</name><formula>cpx Na0.00Ca0.97Fe''-0.0Mg0.66Fe'''0.38Ti0.02Al0.41Si1.58O6</formula><affinity>7488.0011667207727442</affinity><density>3.3551056739978859511</density><gibbsFreeEnergy>-8678083.5045379046351</gibbsFreeEnergy><enthalpy>-7528479.5876581231132</enthalpy><entropy>1115.9577895255838484</entropy><volume>175.58390166526507414</volume><dvdt>0.0061187509852519503945</dvdt><dvdp>-0.00014765099067143221492</dvdp><d2vdt2>2.4144231395628537059e-06</d2vdt2><d2vdtdp>2.8680772617920914301e-27</d2vdtdp><d2vdp2>6.4541507202953518601e-10</d2vdp2><heatCapacity>652.74240889317457004</heatCapacity><SiO2>41.873317922865282981</SiO2><TiO2>0.79520882240876233915</TiO2><Al2O3>9.2312937553921852185</Al2O3><Fe2O3>13.208277426086272754</Fe2O3><FeO>-0.81430307939688251206</FeO><MgO>11.675634775865828985</MgO><CaO>24.03057037677853458</CaO><component><name>diopside</name><formula>CaMgSi2O6</formula><moleFraction>1.5118899057052779256</moleFraction></component><component><name>clinoenstatite</name><formula>Mg2Si2O6</formula><moleFraction>0.068014089492988899499</moleFraction></component><component><name>hedenbergite</name><formula>CaFeSi2O6</formula><moleFraction>-0.066768552949379722516</moleFraction></component><component><name>alumino-buffonite</name><formula>CaTi0.5Mg0.5AlSiO6</formula><moleFraction>0.10473713136846775351</moleFraction></component><component><name>buffonite</name><formula>CaTi0.5Mg0.5FeSiO6</formula><moleFraction>0.012526091829012726137</moleFraction></component><component><name>essenite</name><formula>CaFeAlSiO6</formula><moleFraction>0.96197733956445097281</moleFraction></component><component><name>jadeite</name><formula>NaAlSi2O6</formula><moleFraction>0</moleFraction></component></potentialSolid><potentialSolid><name>aegirine</name><formula>NaFeSi2O6</formula><affinity>142324.7343922778964</affinity><density>0.36096093566885945059</density><gibbsFreeEnergy>-2754290.7588493428193</gibbsFreeEnergy><enthalpy>-2299451.5739982943051</enthalpy><entropy>441.52714153380435391</entropy><volume>639.97000000000002728</volume><dvdt>0</dvdt><dvdp>0</dvdp><d2vdt2>0</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>0</d2vdp2><heatCapacity>250.70810879854303721</heatCapacity><SiO2>52.020099896901264458</SiO2><Fe2O3>34.564787293662405432</Fe2O3><Na2O>13.415112809436321228</Na2O><component><name>aegirine</name><formula>NaFeSi2O6</formula><moleFraction>1</moleFraction></component></potentialSolid><potentialSolid><name>aenigmatite</name><formula>Na2Fe5TiSi6O20</formula><affinity>53356.245067204814404</affinity><density>3.7699873985980936197</density><gibbsFreeEnergy>-9592507.641901794821</gibbsFreeEnergy><enthalpy>-7852964.1758291339502</enthalpy><entropy>1688.6312343568038159</entropy><volume>228.54600000000002069</volume><dvdt>0</dvdt><dvdp>0</dvdp><d2vdt2>0</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>0</d2vdp2><heatCapacity>881.91689549011994131</heatCapacity><SiO2>41.840679893032586278</SiO2><TiO2>9.273138225896678577</TiO2><FeO>41.692841333846637042</FeO><Na2O>7.1933405472240972145</Na2O><component><name>aenigmatite</name><formula>Na2Fe5TiSi6O20</formula><moleFraction>1</moleFraction></component></potentialSolid><potentialSolid><name>cummingtonite</name><formula>(Fe''0.00Mg1.00)7Si8O22(OH)2</formula><affinity>38578.273130620247684</affinity><density>2.901821284928485678</density><gibbsFreeEnergy>-4461510.1099180169404</gibbsFreeEnergy><enthalpy>-3896415.5357236228883</enthalpy><entropy>548.55562218550130638</entropy><volume>92.200056057828319922</volume><dvdt>0.0033663243728671733007</dvdt><dvdp>-0.00010279498751643591204</dvdp><d2vdt2>1.1348560227039594026e-06</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>0</d2vdp2><heatCapacity>339.96852024327318986</heatCapacity><SiO2>61.560174401181079418</SiO2><MgO>36.132611289356717066</MgO><H2O>2.307214309462201296</H2O><component><name>cummingtonite</name><formula>Mg7Si8O22(OH)2</formula><moleFraction>0.34264996808511383186</moleFraction></component><component><name>grunerite</name><formula>Fe7Si8O22(OH)2</formula><moleFraction>0</moleFraction></component></potentialSolid><potentialSolid><name>hornblende</name><formula>NaCa2Mg3.99Fe2+0.01Al1.00Fe3+0.00Al2Si6O22(OH)2</formula><affinity>59810.870715129334712</affinity><density>2.9949782704662619714</density><gibbsFreeEnergy>nan</gibbsFreeEnergy><enthalpy>nan</enthalpy><entropy>nan</entropy><volume>215.95239792378922061</volume><dvdt>0.0082029280786800311959</dvdt><dvdp>-0.00029193688854214688757</dvdp><d2vdt2>4.1611830116072696432e-06</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>1.4729888815518829277e-09</d2vdp2><heatCapacity>799.82146882916595132</heatCapacity><SiO2>43.116702924290990495</SiO2><Al2O3>18.291942402343185137</Al2O3><FeO>0.078337188503887766888</FeO><MgO>19.237772930577648367</MgO><CaO>13.414257579059707837</CaO><Na2O>3.7063585988664509152</Na2O><H2O>2.1546283763581262072</H2O><component><name>pargasite</name><formula>NaCa2Mg4AlAl2Si6O22(OH)2</formula><moleFraction>0.77178094048825129914</moleFraction></component><component><name>ferropargasite</name><formula>NaCa2Fe4AlAl2Si6O22(OH)2</formula><moleFraction>0.001763009628623676394</moleFraction></component><component><name>magnesiohastingsite</name><formula>NaCa2Mg4FeAl2Si6O22(OH)2</formula><moleFraction>0</moleFraction></component></potentialSolid><potentialSolid><name>biotite</name><formula>K(Fe''1.00Mg0.00)3AlSi3O10(OH)2</formula><affinity>20131.375346434655512</affinity><density>3.249930277627445907</density><gibbsFreeEnergy>-5329842.5235565872863</gibbsFreeEnergy><enthalpy>-4364056.2455128310248</enthalpy><entropy>937.52004857909651037</entropy><volume>144.78593588508152834</volume><dvdt>0.0048789444005018682907</dvdt><dvdp>-0.00024035140534224162714</dvdp><d2vdt2>0</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>0</d2vdp2><heatCapacity>521.51433507044623639</heatCapacity><SiO2>35.213234633935783791</SiO2><Al2O3>9.9593029466278135686</Al2O3><FeO>42.106575940858064655</FeO><K2O>9.2015341431802433192</K2O><H2O>3.5193523353980986634</H2O><component><name>annite</name><formula>KFe3Si3AlO10(OH)2</formula><moleFraction>0.91922977149440376898</moleFraction></component><component><name>phlogopite</name><formula>KMg3Si3AlO10(OH)2</formula><moleFraction>0</moleFraction></component></potentialSolid><potentialSolid><name>muscovite</name><formula>KAl2Si3AlO10(OH)2</formula><affinity>24291.121584265143611</affinity><density>2.7678601810788245885</density><gibbsFreeEnergy>-6476444.6749562164769</gibbsFreeEnergy><enthalpy>-5620377.7302846498787</enthalpy><entropy>831.01193483625377212</entropy><volume>143.90601184368736654</volume><dvdt>0.0047229484900000000269</dvdt><dvdp>-0.0002397573717982999942</dvdp><d2vdt2>0</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>1.2100732999999999556e-09</d2vdp2><heatCapacity>515.86818217857239688</heatCapacity><SiO2>45.2542295265627601</SiO2><Al2O3>38.397544516139241466</Al2O3><K2O>11.825336196484517615</K2O><H2O>4.5228897608134719377</H2O><component><name>muscovite</name><formula>KAl2Si3AlO10(OH)2</formula><moleFraction>1</moleFraction></component></potentialSolid><potentialSolid><name>quartz</name><formula>SiO2</formula><affinity>289.49326263437978923</affinity><density>2.5406983011149990581</density><gibbsFreeEnergy>-982126.45394465106074</gibbsFreeEnergy><enthalpy>-860256.27974386734422</enthalpy><entropy>118.30332883636721419</entropy><volume>23.648734670162010474</volume><dvdt>0</dvdt><dvdp>-2.928184692738000633e-05</dvdp><d2vdt2>0</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>3.3592380000000001099e-11</d2vdp2><heatCapacity>69.630602356070241399</heatCapacity><SiO2>100</SiO2><component><name>quartz</name><formula>SiO2</formula><moleFraction>1</moleFraction></component></potentialSolid><potentialSolid><name>tridymite</name><formula>SiO2</formula><affinity>2435.6401096405461431</affinity><density>2.1903286293800392492</density><gibbsFreeEnergy>-979980.30709764489438</gibbsFreeEnergy><enthalpy>-856358.57167525147088</enthalpy><entropy>120.00362609561071281</entropy><volume>27.4316370585022824</volume><dvdt>0.00013216972999999999808</dvdt><dvdp>-1.9896210128899998425e-05</dvdp><d2vdt2>0</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>2.044539000000000047e-10</d2vdp2><heatCapacity>70.632106188108579659</heatCapacity><SiO2>100</SiO2><component><name>tridymite</name><formula>SiO2</formula><moleFraction>1</moleFraction></component></potentialSolid><potentialSolid><name>cristobalite</name><formula>SiO2</formula><affinity>2500.149465067894198</affinity><density>2.1999498962482526565</density><gibbsFreeEnergy>-979915.79774221754633</gibbsFreeEnergy><enthalpy>-856003.37577358039562</enthalpy><entropy>120.28580494941233781</entropy><volume>27.311667462275607932</volume><dvdt>8.7059699999999989113e-05</dvdt><dvdp>-2.9501432960999999751e-05</dvdp><d2vdt2>0</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>3.0221100000000004341e-10</d2vdp2><heatCapacity>69.77804558183012773</heatCapacity><SiO2>100</SiO2><component><name>cristobalite</name><formula>SiO2</formula><moleFraction>1</moleFraction></component></potentialSolid><potentialSolid><name>nepheline</name><formula>neph Na0.04K3.95Ca0.00[]0.01Al3.99Si4.01O16</formula><affinity>40329.650202323282429</affinity><density>2.5313002197959697526</density><gibbsFreeEnergy>-9174427.767888866365</gibbsFreeEnergy><enthalpy>-7843581.2699366491288</enthalpy><entropy>1291.8958384237403152</entropy><volume>247.18467898345653566</volume><dvdt>0.015108321449141917311</dvdt><dvdp>-0.00049562929616468731891</dvdp><d2vdt2>9.9571369194453518349e-06</d2vdt2><d2vdtdp>9.1016097281298821564e-09</d2vdtdp><d2vdp2>2.4078384928296578256e-09</d2vdp2><heatCapacity>693.62072918052888326</heatCapacity><SiO2>38.139457468806597262</SiO2><Al2O3>32.208019571979143336</Al2O3><Na2O>0.20175980762641002042</Na2O><K2O>29.450763151587860733</K2O><component><name>na-nepheline</name><formula>Na4Al4Si4O16</formula><moleFraction>0.0031540762550949863821</moleFraction></component><component><name>k-nepheline</name><formula>K4Al4Si4O16</formula><moleFraction>0.97805929628881349558</moleFraction></component><component><name>vc-nepheline</name><formula>Na3Al3Si5O16</formula><moleFraction>0.0093734608432116308524</moleFraction></component><component><name>ca-nepheline</name><formula>CaNa2Al4Si4O16</formula><moleFraction>0</moleFraction></component></potentialSolid><potentialSolid><name>corundum</name><formula>Al2O3</formula><affinity>11528.370642236666754</affinity><density>3.917505814381032625</density><gibbsFreeEnergy>-1778689.214159975294</gibbsFreeEnergy><enthalpy>-1589465.0822674646042</enthalpy><entropy>183.68599902199747476</entropy><volume>26.027090917313650209</volume><dvdt>0.00072261330816000010971</dvdt><dvdp>-9.814745434999999987e-06</dvdp><d2vdt2>2.4137287999999996192e-07</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>1.9184999999999999536e-11</d2vdp2><heatCapacity>125.90192367180661392</heatCapacity><Al2O3>100</Al2O3><component><name>corundum</name><formula>Al2O3</formula><moleFraction>1</moleFraction></component></potentialSolid><potentialSolid><name>sillimanite</name><formula>Al2SiO5</formula><affinity>11433.606580409221351</affinity><density>3.224513190753354408</density><gibbsFreeEnergy>-2761199.9254291681573</gibbsFreeEnergy><enthalpy>-2451262.9769789176062</enthalpy><entropy>300.86584327549422824</entropy><volume>50.254277285850001533</volume><dvdt>0.00066926672999999996857</dvdt><dvdp>-3.7521989999999996821e-05</dvdp><d2vdt2>0</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>0</d2vdp2><heatCapacity>195.35519985755365724</heatCapacity><SiO2>37.07864170068702947</SiO2><Al2O3>62.92135829931297053</Al2O3><component><name>sillimanite</name><formula>Al2SiO5</formula><moleFraction>1</moleFraction></component></potentialSolid><potentialSolid><name>rutile</name><formula>TiO2</formula><affinity>10858.690792850218713</affinity><density>4.1668434996680909777</density><gibbsFreeEnergy>-1028982.6365359509364</gibbsFreeEnergy><enthalpy>-890460.8276075880276</enthalpy><entropy>134.46761047261361455</entropy><volume>19.174898218847026499</volume><dvdt>0.00052643073643199994865</dvdt><dvdp>-8.5058339017600011542e-06</dvdp><d2vdt2>5.7999475999999994981e-08</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>2.1981759999999999425e-11</d2vdp2><heatCapacity>75.024586070281429784</heatCapacity><TiO2>100</TiO2><component><name>rutile</name><formula>TiO2</formula><moleFraction>1</moleFraction></component></potentialSolid><potentialSolid><name>perovskite</name><formula>CaTiO3</formula><affinity>41076.980730806477368</affinity><density>4.0438410753583537982</density><gibbsFreeEnergy>-1814220.5436300120782</gibbsFreeEnergy><enthalpy>-1565233.1030539311469</enthalpy><entropy>241.70018014471773427</entropy><volume>33.625999999999997669</volume><dvdt>0</dvdt><dvdp>0</dvdp><d2vdt2>0</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>0</d2vdp2><heatCapacity>130.73898692558270795</heatCapacity><TiO2>58.758536294788399346</TiO2><CaO>41.24146370521160776</CaO><component><name>perovskite</name><formula>CaTiO3</formula><moleFraction>1</moleFraction></component></potentialSolid><potentialSolid><name>spinel</name><formula>Fe''0.21Mg0.79Fe'''0.38Al1.58Cr0.05Ti0.00O4</formula><affinity>2347.6561662182575674</affinity><density>3.8737869915323317471</density><gibbsFreeEnergy>-2163148.8047988335602</gibbsFreeEnergy><enthalpy>-1852542.7239671209827</enthalpy><entropy>301.51539176985164659</entropy><volume>40.162852667387987537</volume><dvdt>0.0012915154867950970948</dvdt><dvdp>-1.9486824187344321536e-05</dvdp><d2vdt2>5.3087791809156936149e-07</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>2.8419954524585720719e-11</d2vdp2><heatCapacity>178.55905430503142384</heatCapacity><Al2O3>49.918923097558817403</Al2O3><Fe2O3>18.710073636287383181</Fe2O3><Cr2O3>2.1869467013353580676</Cr2O3><FeO>9.4515439473498545908</FeO><MgO>19.732512617468596972</MgO><component><name>chromite</name><formula>FeCr2O4</formula><moleFraction>0.022386329970697019243</moleFraction></component><component><name>hercynite</name><formula>FeAl2O4</formula><moleFraction>0</moleFraction></component><component><name>magnetite</name><formula>Fe3O4</formula><moleFraction>0.18228548212846235055</moleFraction></component><component><name>spinel</name><formula>MgAl2O4</formula><moleFraction>0.76171098248213542359</moleFraction></component><component><name>ulvospinel</name><formula>Fe2TiO4</formula><moleFraction>0</moleFraction></component></potentialSolid><potentialSolid><name>rhm-oxide</name><formula>Mn0.06Fe''0.00Mg0.90Fe'''0.09Al0.00Ti0.96O3</formula><affinity>4779.437786784535092</affinity><density>3.9531443012033413709</density><gibbsFreeEnergy>-1427171.687593522016</gibbsFreeEnergy><enthalpy>-1226014.4649961174</enthalpy><entropy>195.26983701150760453</entropy><volume>26.772965101862524762</volume><dvdt>0.00080284292459480426519</dvdt><dvdp>-1.5036698989987208429e-05</dvdp><d2vdt2>1.4305724684225811744e-07</d2vdt2><d2vdtdp>1.2035157788272214824e-10</d2vdtdp><d2vdp2>6.2058235674926258398e-11</d2vdp2><heatCapacity>110.15712468715067018</heatCapacity><TiO2>61.757413350638323379</TiO2><Fe2O3>5.6886081143930500303</Fe2O3><MnO>3.2440347760839176594</MnO><MgO>29.309943758884710263</MgO><component><name>geikielite</name><formula>MgTiO3</formula><moleFraction>0.76966486984057747023</moleFraction></component><component><name>hematite</name><formula>Fe2O3</formula><moleFraction>0.037701745025539259992</moleFraction></component><component><name>ilmenite</name><formula>FeTiO3</formula><moleFraction>0</moleFraction></component><component><name>pyrophanite</name><formula>MnTiO3</formula><moleFraction>0.048400447169548478643</moleFraction></component><component><name>corundum</name><formula>Al2O3</formula><moleFraction>0</moleFraction></component></potentialSolid><potentialSolid><name>ortho-oxide</name><formula>Fe''0.01Mg0.00Fe'''1.98Ti1.01O5</formula><affinity>25368.11141322444746</affinity><density>4.3613531629354351793</density><gibbsFreeEnergy>-1419952.5353238880634</gibbsFreeEnergy><enthalpy>-1131667.1352471299469</enthalpy><entropy>279.84798337791397671</entropy><volume>38.755880918457030759</volume><dvdt>0</dvdt><dvdp>0</dvdp><d2vdt2>0</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>0</d2vdp2><heatCapacity>150.63605951048592146</heatCapacity><TiO2>33.66823460854457295</TiO2><Fe2O3>66.053047130131687936</Fe2O3><FeO>0.27871826132373916929</FeO><component><name>pseudobrookite</name><formula>Fe2TiO5</formula><moleFraction>0.69914623176827928042</moleFraction></component><component><name>ferropseudobrookite</name><formula>FeTi2O5</formula><moleFraction>0.0065572128372323569903</moleFraction></component><component><name>karrooite</name><formula>MgTi2O5</formula><moleFraction>0</moleFraction></component></potentialSolid><potentialSolid><name>alloy-solid</name><formula>solid Fe0.00Ni1.00</formula><affinity>66140.73553546750918</affinity><density>8.5923398323585633563</density><gibbsFreeEnergy>-7.2225944386702790971e-11</gibbsFreeEnergy><enthalpy>-5.5828675016300621792e-12</enthalpy><entropy>6.4692595141554847787e-14</entropy><volume>4.7361723189178825793e-309</volume><dvdt>2.501327756619959267e-313</dvdt><dvdp>-2.4874057070678645759e-315</dvdp><d2vdt2>0</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>0</d2vdp2><heatCapacity>2.1619932685798888403e-308</heatCapacity><Fe2O3>-272.00170328735214298</Fe2O3><FeO>244.7501277465576095</FeO><NiO>127.25157554079451927</NiO><component><name>Fe-metal</name><formula>Fe</formula><moleFraction>0</moleFraction></component><component><name>Ni-metal</name><formula>Ni</formula><moleFraction>6.9314941353691268009e-310</moleFraction></component></potentialSolid><potentialSolid><name>alloy-liquid</name><formula>liquid Fe0.00Ni1.00</formula><affinity>70745.11709904205054</affinity><density>8.5549210757165887742</density><gibbsFreeEnergy>-7.2889374619674995611e-11</gibbsFreeEnergy><enthalpy>-6.2552185653430569801e-12</enthalpy><entropy>6.4683935401962761771e-14</entropy><volume>4.756915518205023887e-309</volume><dvdt>5.647404646056447732e-313</dvdt><dvdp>-3.8482860110128361385e-315</dvdp><d2vdt2>2.8173302949063738187e-317</d2vdt2><d2vdtdp>0</d2vdtdp><d2vdp2>1.1363509854348670516e-320</d2vdp2><heatCapacity>2.6966216339174516907e-308</heatCapacity><Fe2O3>-272.00170328735208614</Fe2O3><FeO>244.75012774655758108</FeO><NiO>127.25157554079449085</NiO><component><name>Fe-liquid</name><formula>Fe</formula><moleFraction>0</moleFraction></component><component><name>Ni-liquid</name><formula>Ni</formula><moleFraction>6.9315341206090031185e-310</moleFraction></component></potentialSolid></MELTSoutput>
    ```

