"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValheimChart = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const cdk8s_plus_26_1 = require("cdk8s-plus-26");
/**
 * A chart to deploy a Valheim server
 * Uses the container by @lloesche
 * @see https://github.com/lloesche/valheim-server-docker
 */
class ValheimChart extends cdk8s_1.Chart {
    constructor(scope, name, props) {
        super(scope, name, props);
        const serverVol = this.formServerPersistance(props?.persistence?.server);
        const configVol = this.formConfigPersistance(props?.persistence?.config);
        const ports = this.formPortRecords(props);
        const deployment = new cdk8s_plus_26_1.Deployment(this, 'deployment', {
            replicas: 1,
            strategy: cdk8s_plus_26_1.DeploymentStrategy.recreate(),
            containers: [
                {
                    image: this.formImage(props?.imageTag),
                    ports: Object.keys(ports).map((key) => ports[key]),
                    envVariables: this.formEnvironment(props),
                    volumeMounts: [
                        {
                            path: '/config',
                            volume: configVol,
                        },
                        {
                            path: '/opt/valheim',
                            volume: serverVol,
                        },
                    ],
                    securityContext: {
                        ensureNonRoot: false,
                        readOnlyRootFilesystem: props?.security?.readOnlyRootFilesystem ?? false,
                        privileged: props?.security?.privileged,
                        allowPrivilegeEscalation: props?.security?.allowPrivilegeEscalation,
                        group: props?.security?.group,
                        user: props?.security?.user,
                    },
                    resources: {
                        memory: props?.resourceLimits?.memory ?? {
                            request: cdk8s_1.Size.gibibytes(4),
                        },
                        cpu: props?.resourceLimits?.cpu ?? {
                            request: cdk8s_plus_26_1.Cpu.millis(2000),
                        },
                    },
                },
            ],
            volumes: [configVol, serverVol],
        });
        new cdk8s_plus_26_1.Service(this, 'service', {
            type: props?.server?.serviceType ?? cdk8s_plus_26_1.ServiceType.LOAD_BALANCER,
            selector: deployment,
            ports: [
                {
                    name: ports.game.name,
                    port: ports.game.number,
                    targetPort: ports.game.number,
                    protocol: cdk8s_plus_26_1.Protocol.UDP,
                },
                {
                    name: ports.query.name,
                    port: ports.query.number,
                    targetPort: ports.query.number,
                    protocol: cdk8s_plus_26_1.Protocol.UDP,
                },
            ],
        });
        if (props?.supervisorHttp?.enabled) {
            this.formService(deployment, ports.supervisor, props?.supervisorHttp.serviceType);
        }
        if (props?.statusHttp?.enabled) {
            this.formService(deployment, ports.status, props?.statusHttp.serviceType);
        }
    }
    formImage(tag) {
        return `ghcr.io/lloesche/valheim-server:${tag ?? 'latest'}`;
    }
    formPortRecords(props) {
        const gamePort = props?.server?.port ?? 2456;
        const queryPort = gamePort + 1;
        let portRecords = {
            game: {
                number: gamePort,
                name: 'server',
                protocol: cdk8s_plus_26_1.Protocol.UDP,
            },
            query: {
                number: queryPort,
                name: 'query',
                protocol: cdk8s_plus_26_1.Protocol.UDP,
            },
        };
        if (props?.supervisorHttp?.enabled) {
            portRecords.supervisor = {
                number: props.supervisorHttp.port ?? 9001,
                name: 'supervisor',
                protocol: cdk8s_plus_26_1.Protocol.TCP,
            };
        }
        if (props?.statusHttp?.enabled) {
            portRecords.status = {
                number: props.statusHttp.port ?? 80,
                name: 'status',
                protocol: cdk8s_plus_26_1.Protocol.TCP,
            };
        }
        return portRecords;
    }
    formService(scope, port, type) {
        let name = port.name;
        return new cdk8s_plus_26_1.Service(this, `${name}-service`, {
            type: type ?? cdk8s_plus_26_1.ServiceType.CLUSTER_IP,
            selector: scope,
            ports: [
                {
                    port: port.number,
                    targetPort: port.number,
                    protocol: port.protocol,
                },
            ],
        });
    }
    formHostVolume(name, path) {
        return cdk8s_plus_26_1.Volume.fromHostPath(this, `${name}-volume`, path, {
            path: `/data/valheim/${path}`,
            type: cdk8s_plus_26_1.HostPathVolumeType.DIRECTORY_OR_CREATE,
        });
    }
    formPersistanceVolume(name, props) {
        let pvc = new cdk8s_plus_26_1.PersistentVolumeClaim(this, `${name}-pvc`, props);
        return cdk8s_plus_26_1.Volume.fromPersistentVolumeClaim(this, `${name}-volume`, pvc);
    }
    formServerPersistance(props) {
        let volume;
        if (props) {
            let pvcProps = {
                storageClassName: props?.storageClass ?? 'local-path',
                accessModes: props?.accessModes ?? [cdk8s_plus_26_1.PersistentVolumeAccessMode.READ_WRITE_ONCE],
                storage: props?.storage ?? cdk8s_1.Size.gibibytes(10),
            };
            volume = this.formPersistanceVolume('server', pvcProps);
        }
        else {
            volume = this.formHostVolume('server', 'valheim-server');
        }
        return volume;
    }
    formConfigPersistance(props) {
        let volume;
        if (props) {
            let pvcProps = {
                storageClassName: props.storageClass,
                accessModes: props?.accessModes ?? [cdk8s_plus_26_1.PersistentVolumeAccessMode.READ_WRITE_ONCE],
                storage: props?.storage ?? cdk8s_1.Size.gibibytes(1),
            };
            volume = this.formPersistanceVolume('config', pvcProps);
        }
        else {
            volume = this.formHostVolume('config', 'valheim-config');
        }
        return volume;
    }
    formPasswordEnvValue(name, props) {
        let envValue;
        if (props.secret) {
            const secret = cdk8s_plus_26_1.Secret.fromSecretName(this, `${name}-password`, props.secret);
            envValue = cdk8s_plus_26_1.EnvValue.fromSecretValue({
                secret: secret,
                key: 'password',
            });
        }
        else if (props.raw) {
            envValue = cdk8s_plus_26_1.EnvValue.fromValue(props.raw);
        }
        else {
            throw new Error('Invalid password configuration');
        }
        return envValue;
    }
    formEnvironment(props) {
        let env = {};
        let launchArgs = '';
        if (props?.server?.launchArgs) {
            launchArgs = props.server.launchArgs;
        }
        if (props?.server?.crossplay) {
            launchArgs += ' -crossplay';
        }
        if (launchArgs !== '') {
            env.SERVER_ARGS = cdk8s_plus_26_1.EnvValue.fromValue(launchArgs);
        }
        if (props?.server) {
            if (props.server.port) {
                env.SERVER_PORT = cdk8s_plus_26_1.EnvValue.fromValue(props.server.port.toString());
            }
            if (props.server.name) {
                env.SERVER_NAME = cdk8s_plus_26_1.EnvValue.fromValue(props.server.name);
            }
            if (props.server.worldName) {
                env.WORLD_NAME = cdk8s_plus_26_1.EnvValue.fromValue(props.server.worldName);
            }
            if (props.server.password) {
                env.SERVER_PASS = this.formPasswordEnvValue('server', props.server.password);
            }
            if (props.server.public !== undefined) {
                let isPublic = 'false';
                if (props.server.public) {
                    isPublic = 'true';
                }
                env.SERVER_PUBLIC = cdk8s_plus_26_1.EnvValue.fromValue(isPublic);
            }
            if (props.server.updateCron) {
                env.UPDATE_CRON = cdk8s_plus_26_1.EnvValue.fromValue(props.server.updateCron);
            }
            if (props.server.idleDatagramWindow) {
                env.IDLE_DATAGRAM_WINDOW = cdk8s_plus_26_1.EnvValue.fromValue(props.server.idleDatagramWindow.toString());
            }
            if (props.server.idleDatagramMaxCount) {
                env.IDLE_DATAGRAM_MAX_COUNT = cdk8s_plus_26_1.EnvValue.fromValue(props.server.idleDatagramMaxCount.toString());
            }
            if (props.server.updateWhenIdle !== undefined) {
                let updateWhenIdle = 'false';
                if (props.server.updateWhenIdle) {
                    updateWhenIdle = 'true';
                }
                env.UPDATE_WHEN_IDLE = cdk8s_plus_26_1.EnvValue.fromValue(updateWhenIdle);
            }
            if (props.server.restartIfIdle !== undefined) {
                let restartIfIdle = 'false';
                if (props.server.restartIfIdle) {
                    restartIfIdle = 'true';
                }
                env.RESTART_IF_IDLE = cdk8s_plus_26_1.EnvValue.fromValue(restartIfIdle);
            }
            if (props.server.timezone) {
                env.TZ = cdk8s_plus_26_1.EnvValue.fromValue(props.server.timezone);
            }
            if (props.server.adminList) {
                env.ADMINLIST_IDS = cdk8s_plus_26_1.EnvValue.fromValue(props.server.adminList.join(' '));
            }
            if (props.server.blockList) {
                env.BANNEDLIST_IDS = cdk8s_plus_26_1.EnvValue.fromValue(props.server.blockList.join(' '));
            }
            if (props.server.allowList) {
                env.PERMITTEDLIST_IDS = cdk8s_plus_26_1.EnvValue.fromValue(props.server.allowList.join(' '));
            }
            if (props.server.publicBeta !== undefined) {
                let publicBeta = 'false';
                if (props.server.publicBeta) {
                    publicBeta = 'true';
                }
                env.PUBLIC_BETA = cdk8s_plus_26_1.EnvValue.fromValue(publicBeta);
            }
            if (props.server.steamCmdArgs) {
                env.STEAMCMD_ARGS = cdk8s_plus_26_1.EnvValue.fromValue(props.server.steamCmdArgs);
            }
            if (props.server.valheimPlus) {
                if (props.server.valheimPlus.enabled !== undefined) {
                    let valheimPlusEnabled = 'false';
                    if (props.server.valheimPlus.enabled) {
                        valheimPlusEnabled = 'true';
                    }
                    env.VALHEIM_PLUS = cdk8s_plus_26_1.EnvValue.fromValue(valheimPlusEnabled);
                }
                if (props.server.valheimPlus.release) {
                    env.VALHEIM_PLUS_RELEASE = cdk8s_plus_26_1.EnvValue.fromValue(props.server.valheimPlus.release);
                }
            }
        }
        if (props?.backup) {
            if (props.backup.enabled !== undefined) {
                let doBackups = 'false';
                if (props.backup.enabled) {
                    doBackups = 'true';
                }
                env.BACKUPS_ENABLED = cdk8s_plus_26_1.EnvValue.fromValue(doBackups);
            }
            if (props.backup.scheduleCron) {
                env.BACKUPS_CRON = cdk8s_plus_26_1.EnvValue.fromValue(props.backup.scheduleCron);
            }
            if (props.backup.directory) {
                env.BACKUPS_DIRECTORY = cdk8s_plus_26_1.EnvValue.fromValue(props.backup.directory);
            }
            if (props.backup.retentionAge) {
                env.BACKUPS_MAX_AGE = cdk8s_plus_26_1.EnvValue.fromValue(props.backup.retentionAge.toString());
            }
            if (props.backup.maxBackups) {
                env.BACKUPS_MAX_COUNT = cdk8s_plus_26_1.EnvValue.fromValue(props.backup.maxBackups.toString());
            }
            if (props.backup.performIfIdle) {
                env.BACKUPS_IF_IDLE = cdk8s_plus_26_1.EnvValue.fromValue(props.backup.performIfIdle.toString());
            }
            if (props.backup.idleGracePeriod) {
                env.BACKUPS_IDLE_GRACE_PERIOD = cdk8s_plus_26_1.EnvValue.fromValue(props.backup.idleGracePeriod.toString());
            }
            if (props.backup.zip !== undefined) {
                let zipBackups = 'false';
                if (props.backup.zip) {
                    zipBackups = 'true';
                }
                env.BACKUPS_ZIP = cdk8s_plus_26_1.EnvValue.fromValue(zipBackups);
            }
            if (props.backup.permissionUmask) {
                env.PERMISSIONS_UMASK = cdk8s_plus_26_1.EnvValue.fromValue(props.backup.permissionUmask.toString());
            }
        }
        if (props?.supervisorHttp) {
            if (props.supervisorHttp.enabled !== undefined) {
                let supervisorHttpEnabled = 'false';
                if (props.supervisorHttp.enabled) {
                    supervisorHttpEnabled = 'true';
                }
                env.SUPERVISOR_HTTP = cdk8s_plus_26_1.EnvValue.fromValue(supervisorHttpEnabled);
            }
            if (props.supervisorHttp.port) {
                env.SUPERVISOR_HTTP_PORT = cdk8s_plus_26_1.EnvValue.fromValue(props.supervisorHttp.port.toString());
            }
            if (props.supervisorHttp.username) {
                env.SUPERVISOR_HTTP_USER = cdk8s_plus_26_1.EnvValue.fromValue(props.supervisorHttp.username);
            }
            if (props.supervisorHttp.password) {
                env.SUPERVISOR_HTTP_PASS = this.formPasswordEnvValue('supervisor', props.supervisorHttp.password);
                ;
            }
        }
        if (props?.statusHttp) {
            if (props.statusHttp.enabled !== undefined) {
                let statusHttpEnabled = 'false';
                if (props.statusHttp.enabled) {
                    statusHttpEnabled = 'true';
                }
                env.STATUS_HTTP = cdk8s_plus_26_1.EnvValue.fromValue(statusHttpEnabled);
            }
            if (props.statusHttp.port) {
                env.STATUS_HTTP_PORT = cdk8s_plus_26_1.EnvValue.fromValue(props.statusHttp.port.toString());
            }
            if (props.statusHttp.configPath) {
                env.STATUS_HTTP_CONF = cdk8s_plus_26_1.EnvValue.fromValue(props.statusHttp.configPath);
            }
            if (props.statusHttp.htdocLocation) {
                env.STATUS_HTTP_HTDOCS = cdk8s_plus_26_1.EnvValue.fromValue(props.statusHttp.htdocLocation);
            }
        }
        if (props?.sysLog) {
            if (props.sysLog.remoteHost) {
                env.SYSLOG_HOST = cdk8s_plus_26_1.EnvValue.fromValue(props.sysLog.remoteHost);
            }
            if (props.sysLog.remotePort) {
                env.SYSLOG_PORT = cdk8s_plus_26_1.EnvValue.fromValue(props.sysLog.remotePort.toString());
            }
            if (props.sysLog.logLocal !== undefined) {
                let logLocal = 'false';
                if (props.sysLog.logLocal) {
                    logLocal = 'true';
                }
                env.SYSLOG_LOCAL = cdk8s_plus_26_1.EnvValue.fromValue(logLocal);
            }
        }
        if (props?.security?.user) {
            env.PGID = cdk8s_plus_26_1.EnvValue.fromValue(props.security?.user.toString());
        }
        if (props?.security?.group) {
            env.PUID = cdk8s_plus_26_1.EnvValue.fromValue(props.security?.group.toString());
        }
        return env;
    }
}
exports.ValheimChart = ValheimChart;
_a = JSII_RTTI_SYMBOL_1;
ValheimChart[_a] = { fqn: "@awlsring/cdk8s-valheim.ValheimChart", version: "0.0.11" };
//# sourceMappingURL=data:application/json;base64,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