from collections import OrderedDict
from typing import Any
from typing import Dict

import dict_tools.differ as differ

__contracts__ = ["resource"]

AWS_Org_Not_In_Use_Exception = "AWSOrganizationsNotInUseException"
ALREADY_EXISTS = "already exists"


async def present(hub, ctx, name: str, feature_set: str) -> Dict[str, Any]:
    r"""
    **Autogenerated function**

    Creates an AWS organization. The account whose user is calling the CreateOrganization operation
    automatically becomes the management account of the new organization.This operation must be called using
    credentials from the account that is to become the new organization's management account. The principal
    must also have the relevant IAM permissions.By default (or if you set the FeatureSet parameter to ALL ),
    the new organization is created with all features enabled and service control policies automatically
    enabled in the root. If you instead choose to create the organization supporting only the
    consolidated billing features by setting the FeatureSet parameter to CONSOLIDATED_BILLING" ,
    no policy types are enabled by default, and you can't use organization policies

    Args:
        hub:
        ctx:
        name(Text): A name or ID to identify the resource.
        feature_set(str, Default: 'ALL'): optional values: 'ALL'|'CONSOLIDATED_BILLING'

    Request Syntax:
        [org-id]:
          aws.organizations.organization.present:
          - feature_set: 'string'

    Returns:
        Dict[str, Any]

    Examples:

        .. code-block:: sls

            org_is_present:
              aws.organizations.organization.present:
                - feature_set: 'ALL'
    """

    result = dict(comment="", old_state=None, new_state=None, name=name, result=True)

    before = await hub.exec.boto3.client.organizations.describe_organization(ctx)

    if ctx.get("test", False):
        if before:
            result["comment"] = f"aws.organizations.organization already exists {name}."
            result["result"] = True
        else:
            result["comment"] = f"Would create aws.oranizations.organization {name}."
            result["result"] = True
        return result

    if not before:
        # Org not present , create
        hub.log.debug("Creating organization for this account")

        try:
            ret = await hub.exec.boto3.client.organizations.create_organization(
                ctx, FeatureSet=feature_set
            )

            result["result"] = ret["result"]
            if not result["result"]:
                result["comment"] = ret["comment"]
                return result
            result["comment"] = f"Created aws.organizations.organization {name}."
        except hub.tool.boto3.exception.ClientError as e:
            result["comment"] = f"{e.__class__.__name__}: {e}"
            result["result"] = False
    else:
        # Org present
        before["ret"].pop("ResponseMetadata", None)
        result["old_state"] = before["ret"]["Organization"]
        result["result"] = True
        result["comment"] = f"aws.organizations.organization {name} {ALREADY_EXISTS}. "

    try:
        after = await hub.exec.boto3.client.organizations.describe_organization(ctx)
        if after and after.get("ret"):
            after["ret"].pop("ResponseMetadata", None)
            result["new_state"] = after["ret"]["Organization"]

            roots_resp = await hub.exec.boto3.client.organizations.list_roots(ctx)

            if roots_resp and roots_resp["ret"]["Roots"]:
                result["new_state"]["Roots"] = roots_resp["ret"]["Roots"]

    except Exception as e:
        result["comment"] = str(e)
        result["result"] = False

    return result


async def absent(hub, ctx, name: str) -> Dict[str, Any]:
    r"""
    **Autogenerated function**

    Deletes the organization. You can delete an organization only by using credentials from the
    management account. The organization must be empty of member accounts.

    Args:
        name(Text): A name or ID to identify the resource.

    Returns:
        Dict[str, Any]

    Examples:

        .. code-block:: sls

            organization_is_absent:
              aws.organizations.organization.absent:
                - name: value
    """

    result = dict(comment="", old_state=None, new_state=None, name=name, result=True)

    before = await hub.exec.boto3.client.organizations.describe_organization(ctx)

    if ctx.get("test", False):
        if before:
            result["comment"] = f"aws.organizations.organization already absent {name}."
            result["result"] = True
        else:
            result["comment"] = f"Would delete aws.oranizations.organization {name}."
            result["result"] = True
        return result

    if before:
        before["ret"].pop("ResponseMetadata", None)
        result["old_state"] = before["ret"]

    if not before and AWS_Org_Not_In_Use_Exception in str(before["comment"]):
        result["comment"] = f"aws.organizations.organization {name} is already absent"
        return result
    else:
        try:
            ret = await hub.exec.boto3.client.organizations.delete_organization(ctx)

            if not ret:
                result["comment"] = ret["comment"]
                result["result"] = False
                return result
            result["comment"] = f"aws.organizations.organization {name} deleted."
            result["result"] = ret["result"]
        except hub.tool.boto3.exception.ClientError as e:
            result["comment"] = f"{e.__class__.__name__}: {e}"

    try:
        after = await hub.exec.boto3.client.organizations.describe_organization(ctx)
        if after.get("ret"):
            after["ret"].pop("ResponseMetadata", None)
            result["new_state"] = after["ret"]
    except Exception as e:
        result["comment"] = str(e)
        result["result"] = False
    return result


async def describe(hub, ctx) -> Dict[str, Dict[str, Any]]:
    r"""
    **Autogenerated function**

    Retrieves information about the organization that the user's account belongs to.
    This operation can be called from any account in the organization..


    Returns:
        Dict[str, Any]

    Examples:

        .. code-block:: bash

            $ idem describe aws.organizations.organization
    """

    result = {}
    ret = await hub.exec.boto3.client.organizations.describe_organization(ctx)
    if not ret["result"]:
        hub.log.debug(f"Could not describe organization {ret['comment']}")
        return {}

    describe_parameters = OrderedDict(
        {
            "Id": "id",
            "Arn": "arn",
            "MasterAccountArn": "master_account_arn",
            "MasterAccountId": "master_account_id",
            "MasterAccountEmail": "master_account_email",
            "FeatureSet": "feature_set",
            "AvailablePolicyTypes": "available_policy_types",
        }
    )

    resource = ret["ret"]["Organization"]

    if not "FeatureSet" in resource:
        resource["FeatureSet"] = "ALL"

    translated_resource = []
    for camel_case_key, snake_case_key in describe_parameters.items():
        if resource.get(camel_case_key) is not None:
            translated_resource.append({snake_case_key: resource.get(camel_case_key)})

    result[resource["Id"]] = {
        "aws.organizations.organization.present": translated_resource
    }

    return result
