"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Table = exports.TableEncryption = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const s3 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
/**
 * (experimental) Encryption options for a Table.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/encryption.html
 * @experimental
 */
var TableEncryption;
(function (TableEncryption) {
    TableEncryption["UNENCRYPTED"] = "Unencrypted";
    TableEncryption["S3_MANAGED"] = "SSE-S3";
    TableEncryption["KMS"] = "SSE-KMS";
    TableEncryption["KMS_MANAGED"] = "SSE-KMS-MANAGED";
    TableEncryption["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
/**
 * (experimental) A Glue table.
 *
 * @experimental
 */
class Table extends aws_cdk_lib_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.tableName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_TableProps(props);
        this.database = props.database;
        this.dataFormat = props.dataFormat;
        this.s3Prefix = (_b = props.s3Prefix) !== null && _b !== void 0 ? _b : '';
        validateSchema(props.columns, props.partitionKeys);
        this.columns = props.columns;
        this.partitionKeys = props.partitionKeys;
        this.compressed = (_c = props.compressed) !== null && _c !== void 0 ? _c : false;
        const { bucket, encryption, encryptionKey } = createBucket(this, props);
        this.bucket = bucket;
        this.encryption = encryption;
        this.encryptionKey = encryptionKey;
        const tableResource = new aws_glue_1.CfnTable(this, 'Table', {
            catalogId: props.database.catalogId,
            databaseName: props.database.databaseName,
            tableInput: {
                name: this.physicalName,
                description: props.description || `${props.tableName} generated by CDK`,
                partitionKeys: renderColumns(props.partitionKeys),
                parameters: {
                    classification: (_d = props.dataFormat.classificationString) === null || _d === void 0 ? void 0 : _d.value,
                    has_encrypted_data: this.encryption !== TableEncryption.UNENCRYPTED,
                },
                storageDescriptor: {
                    location: `s3://${this.bucket.bucketName}/${this.s3Prefix}`,
                    compressed: this.compressed,
                    storedAsSubDirectories: (_e = props.storedAsSubDirectories) !== null && _e !== void 0 ? _e : false,
                    columns: renderColumns(props.columns),
                    inputFormat: props.dataFormat.inputFormat.className,
                    outputFormat: props.dataFormat.outputFormat.className,
                    serdeInfo: {
                        serializationLibrary: props.dataFormat.serializationLibrary.className,
                    },
                },
                tableType: 'EXTERNAL_TABLE',
            },
        });
        this.tableName = this.getResourceNameAttribute(tableResource.ref);
        this.tableArn = this.stack.formatArn({
            service: 'glue',
            resource: 'table',
            resourceName: `${this.database.databaseName}/${this.tableName}`,
        });
        this.node.defaultChild = tableResource;
    }
    /**
     * @experimental
     */
    static fromTableArn(scope, id, tableArn) {
        const tableName = aws_cdk_lib_1.Fn.select(1, aws_cdk_lib_1.Fn.split('/', aws_cdk_lib_1.Stack.of(scope).splitArn(tableArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName));
        return Table.fromTableAttributes(scope, id, {
            tableArn,
            tableName,
        });
    }
    /**
     * (experimental) Creates a Table construct that represents an external table.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes.
     * @experimental
     */
    static fromTableAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_TableAttributes(attrs);
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.tableArn = attrs.tableArn;
                this.tableName = attrs.tableName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Grant read permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal.
     * @experimental
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, readPermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        this.bucket.grantRead(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * (experimental) Grant write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal.
     * @experimental
     */
    grantWrite(grantee) {
        const ret = this.grant(grantee, writePermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        this.bucket.grantWrite(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * (experimental) Grant read and write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal.
     * @experimental
     */
    grantReadWrite(grantee) {
        const ret = this.grant(grantee, [...readPermissions, ...writePermissions]);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncryptDecrypt(grantee);
        }
        this.bucket.grantReadWrite(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    grant(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.tableArn],
            actions,
        });
    }
    getS3PrefixForGrant() {
        return this.s3Prefix + '*';
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-glue-alpha.Table", version: "2.0.0-alpha.11" };
function validateSchema(columns, partitionKeys) {
    if (columns.length === 0) {
        throw new Error('you must specify at least one column for the table');
    }
    // Check there is at least one column and no duplicated column names or partition keys.
    const names = new Set();
    (columns.concat(partitionKeys || [])).forEach(column => {
        if (names.has(column.name)) {
            throw new Error(`column names and partition keys must be unique, but \'${column.name}\' is duplicated`);
        }
        names.add(column.name);
    });
}
// map TableEncryption to bucket's SSE configuration (s3.BucketEncryption)
const encryptionMappings = {
    [TableEncryption.S3_MANAGED]: s3.BucketEncryption.S3_MANAGED,
    [TableEncryption.KMS_MANAGED]: s3.BucketEncryption.KMS_MANAGED,
    [TableEncryption.KMS]: s3.BucketEncryption.KMS,
    [TableEncryption.CLIENT_SIDE_KMS]: s3.BucketEncryption.UNENCRYPTED,
    [TableEncryption.UNENCRYPTED]: s3.BucketEncryption.UNENCRYPTED,
};
// create the bucket to store a table's data depending on the `encryption` and `encryptionKey` properties.
function createBucket(table, props) {
    const encryption = props.encryption || TableEncryption.UNENCRYPTED;
    let bucket = props.bucket;
    if (bucket && (encryption !== TableEncryption.UNENCRYPTED && encryption !== TableEncryption.CLIENT_SIDE_KMS)) {
        throw new Error('you can not specify encryption settings if you also provide a bucket');
    }
    let encryptionKey;
    if (encryption === TableEncryption.CLIENT_SIDE_KMS && props.encryptionKey === undefined) {
        // CSE-KMS should behave the same as SSE-KMS - use the provided key or create one automatically
        // Since Bucket only knows about SSE, we repeat the logic for CSE-KMS at the Table level.
        encryptionKey = new kms.Key(table, 'Key');
    }
    else {
        encryptionKey = props.encryptionKey;
    }
    // create the bucket if none was provided
    if (!bucket) {
        if (encryption === TableEncryption.CLIENT_SIDE_KMS) {
            bucket = new s3.Bucket(table, 'Bucket');
        }
        else {
            bucket = new s3.Bucket(table, 'Bucket', {
                encryption: encryptionMappings[encryption],
                encryptionKey,
            });
            encryptionKey = bucket.encryptionKey;
        }
    }
    return {
        bucket,
        encryption,
        encryptionKey,
    };
}
const readPermissions = [
    'glue:BatchDeletePartition',
    'glue:BatchGetPartition',
    'glue:GetPartition',
    'glue:GetPartitions',
    'glue:GetTable',
    'glue:GetTables',
    'glue:GetTableVersion',
    'glue:GetTableVersions',
];
const writePermissions = [
    'glue:BatchCreatePartition',
    'glue:BatchDeletePartition',
    'glue:CreatePartition',
    'glue:DeletePartition',
    'glue:UpdatePartition',
];
function renderColumns(columns) {
    if (columns === undefined) {
        return undefined;
    }
    return columns.map(column => {
        return {
            name: column.name,
            type: column.type.inputString,
            comment: column.comment,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFibGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0YWJsZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSwyQ0FBMkM7QUFDM0MsMkNBQTJDO0FBQzNDLHlDQUF5QztBQUN6Qyw2Q0FBd0U7QUFJeEUsbURBQWdEOzs7Ozs7O0FBWWhELElBQVksZUFjWDtBQWRELFdBQVksZUFBZTtJQUN6Qiw4Q0FBMkIsQ0FBQTtJQUczQix3Q0FBcUIsQ0FBQTtJQUdyQixrQ0FBZSxDQUFBO0lBR2Ysa0RBQStCLENBQUE7SUFHL0IsOENBQTJCLENBQUE7QUFDN0IsQ0FBQyxFQWRXLGVBQWUsR0FBZix1QkFBZSxLQUFmLHVCQUFlLFFBYzFCOzs7Ozs7QUE4Q0QsTUFBYSxLQUFNLFNBQVEsc0JBQVE7Ozs7SUFzRGpDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBaUI7O1FBQ3pELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxTQUFTO1NBQzlCLENBQUMsQ0FBQzs7UUFFSCxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDL0IsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO1FBQ25DLElBQUksQ0FBQyxRQUFRLFNBQUcsS0FBSyxDQUFDLFFBQVEsbUNBQUksRUFBRSxDQUFDO1FBRXJDLGNBQWMsQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUNuRCxJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDN0IsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO1FBRXpDLElBQUksQ0FBQyxVQUFVLFNBQUcsS0FBSyxDQUFDLFVBQVUsbUNBQUksS0FBSyxDQUFDO1FBQzVDLE1BQU0sRUFBRSxNQUFNLEVBQUUsVUFBVSxFQUFFLGFBQWEsRUFBRSxHQUFHLFlBQVksQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDeEUsSUFBSSxDQUFDLE1BQU0sR0FBRyxNQUFNLENBQUM7UUFDckIsSUFBSSxDQUFDLFVBQVUsR0FBRyxVQUFVLENBQUM7UUFDN0IsSUFBSSxDQUFDLGFBQWEsR0FBRyxhQUFhLENBQUM7UUFFbkMsTUFBTSxhQUFhLEdBQUcsSUFBSSxtQkFBUSxDQUFDLElBQUksRUFBRSxPQUFPLEVBQUU7WUFDaEQsU0FBUyxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsU0FBUztZQUVuQyxZQUFZLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQyxZQUFZO1lBRXpDLFVBQVUsRUFBRTtnQkFDVixJQUFJLEVBQUUsSUFBSSxDQUFDLFlBQVk7Z0JBQ3ZCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVyxJQUFJLEdBQUcsS0FBSyxDQUFDLFNBQVMsbUJBQW1CO2dCQUV2RSxhQUFhLEVBQUUsYUFBYSxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUM7Z0JBRWpELFVBQVUsRUFBRTtvQkFDVixjQUFjLFFBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxvQkFBb0IsMENBQUUsS0FBSztvQkFDNUQsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLFVBQVUsS0FBSyxlQUFlLENBQUMsV0FBVztpQkFDcEU7Z0JBQ0QsaUJBQWlCLEVBQUU7b0JBQ2pCLFFBQVEsRUFBRSxRQUFRLElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxJQUFJLElBQUksQ0FBQyxRQUFRLEVBQUU7b0JBQzNELFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTtvQkFDM0Isc0JBQXNCLFFBQUUsS0FBSyxDQUFDLHNCQUFzQixtQ0FBSSxLQUFLO29CQUM3RCxPQUFPLEVBQUUsYUFBYSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUM7b0JBQ3JDLFdBQVcsRUFBRSxLQUFLLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxTQUFTO29CQUNuRCxZQUFZLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUMsU0FBUztvQkFDckQsU0FBUyxFQUFFO3dCQUNULG9CQUFvQixFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsb0JBQW9CLENBQUMsU0FBUztxQkFDdEU7aUJBQ0Y7Z0JBRUQsU0FBUyxFQUFFLGdCQUFnQjthQUM1QjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNsRSxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDO1lBQ25DLE9BQU8sRUFBRSxNQUFNO1lBQ2YsUUFBUSxFQUFFLE9BQU87WUFDakIsWUFBWSxFQUFFLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRTtTQUNoRSxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksR0FBRyxhQUFhLENBQUM7S0FDeEM7Ozs7SUE3R00sTUFBTSxDQUFDLFlBQVksQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFnQjtRQUN2RSxNQUFNLFNBQVMsR0FBRyxnQkFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsZ0JBQUUsQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLG1CQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUUsdUJBQVMsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLFlBQWEsQ0FBQyxDQUFDLENBQUM7UUFFL0gsT0FBTyxLQUFLLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUMxQyxRQUFRO1lBQ1IsU0FBUztTQUNWLENBQUMsQ0FBQztLQUNKOzs7Ozs7Ozs7SUFHTSxNQUFNLENBQUMsbUJBQW1CLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBc0I7O1FBQ3BGLE1BQU0sTUFBTyxTQUFRLHNCQUFRO1lBQTdCOztnQkFDa0IsYUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7Z0JBQzFCLGNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1lBQzlDLENBQUM7U0FBQTtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0tBQzlCOzs7Ozs7O0lBK0ZNLFNBQVMsQ0FBQyxPQUF1QjtRQUN0QyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxlQUFlLENBQUMsQ0FBQztRQUNqRCxJQUFJLElBQUksQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLFVBQVUsS0FBSyxlQUFlLENBQUMsZUFBZSxFQUFFO1lBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUM7U0FBRTtRQUM1SCxJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUMsQ0FBQztRQUMzRCxPQUFPLEdBQUcsQ0FBQztLQUNaOzs7Ozs7O0lBR00sVUFBVSxDQUFDLE9BQXVCO1FBQ3ZDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLGdCQUFnQixDQUFDLENBQUM7UUFDbEQsSUFBSSxJQUFJLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQyxVQUFVLEtBQUssZUFBZSxDQUFDLGVBQWUsRUFBRTtZQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQUU7UUFDNUgsSUFBSSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLENBQUM7UUFDNUQsT0FBTyxHQUFHLENBQUM7S0FDWjs7Ozs7OztJQUdNLGNBQWMsQ0FBQyxPQUF1QjtRQUMzQyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDLEdBQUcsZUFBZSxFQUFFLEdBQUcsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDO1FBQzNFLElBQUksSUFBSSxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsVUFBVSxLQUFLLGVBQWUsQ0FBQyxlQUFlLEVBQUU7WUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLG1CQUFtQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQUU7UUFDbkksSUFBSSxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLENBQUM7UUFDaEUsT0FBTyxHQUFHLENBQUM7S0FDWjtJQUVPLEtBQUssQ0FBQyxPQUF1QixFQUFFLE9BQWlCO1FBQ3RELE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7WUFDOUIsT0FBTztZQUNQLFlBQVksRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUM7WUFDN0IsT0FBTztTQUNSLENBQUMsQ0FBQztLQUNKO0lBRU8sbUJBQW1CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLFFBQVEsR0FBRyxHQUFHLENBQUM7S0FDNUI7O0FBbkpILHNCQW9KQzs7O0FBRUQsU0FBUyxjQUFjLENBQUMsT0FBaUIsRUFBRSxhQUF3QjtJQUNqRSxJQUFJLE9BQU8sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1FBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMsb0RBQW9ELENBQUMsQ0FBQztLQUN2RTtJQUNELHVGQUF1RjtJQUN2RixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsRUFBVSxDQUFDO0lBQ2hDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxhQUFhLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUU7UUFDckQsSUFBSSxLQUFLLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUMxQixNQUFNLElBQUksS0FBSyxDQUFDLHlEQUF5RCxNQUFNLENBQUMsSUFBSSxrQkFBa0IsQ0FBQyxDQUFDO1NBQ3pHO1FBQ0QsS0FBSyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDekIsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDO0FBRUQsMEVBQTBFO0FBQzFFLE1BQU0sa0JBQWtCLEdBQUc7SUFDekIsQ0FBQyxlQUFlLENBQUMsVUFBVSxDQUFDLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFVBQVU7SUFDNUQsQ0FBQyxlQUFlLENBQUMsV0FBVyxDQUFDLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFdBQVc7SUFDOUQsQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLEdBQUc7SUFDOUMsQ0FBQyxlQUFlLENBQUMsZUFBZSxDQUFDLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFdBQVc7SUFDbEUsQ0FBQyxlQUFlLENBQUMsV0FBVyxDQUFDLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFdBQVc7Q0FDL0QsQ0FBQztBQUVGLDBHQUEwRztBQUMxRyxTQUFTLFlBQVksQ0FBQyxLQUFZLEVBQUUsS0FBaUI7SUFDbkQsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsSUFBSSxlQUFlLENBQUMsV0FBVyxDQUFDO0lBQ25FLElBQUksTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUM7SUFFMUIsSUFBSSxNQUFNLElBQUksQ0FBQyxVQUFVLEtBQUssZUFBZSxDQUFDLFdBQVcsSUFBSSxVQUFVLEtBQUssZUFBZSxDQUFDLGVBQWUsQ0FBQyxFQUFFO1FBQzVHLE1BQU0sSUFBSSxLQUFLLENBQUMsc0VBQXNFLENBQUMsQ0FBQztLQUN6RjtJQUVELElBQUksYUFBbUMsQ0FBQztJQUN4QyxJQUFJLFVBQVUsS0FBSyxlQUFlLENBQUMsZUFBZSxJQUFJLEtBQUssQ0FBQyxhQUFhLEtBQUssU0FBUyxFQUFFO1FBQ3ZGLCtGQUErRjtRQUMvRix5RkFBeUY7UUFDekYsYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7S0FDM0M7U0FBTTtRQUNMLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO0tBQ3JDO0lBRUQseUNBQXlDO0lBQ3pDLElBQUksQ0FBQyxNQUFNLEVBQUU7UUFDWCxJQUFJLFVBQVUsS0FBSyxlQUFlLENBQUMsZUFBZSxFQUFFO1lBQ2xELE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1NBQ3pDO2FBQU07WUFDTCxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7Z0JBQ3RDLFVBQVUsRUFBRSxrQkFBa0IsQ0FBQyxVQUFVLENBQUM7Z0JBQzFDLGFBQWE7YUFDZCxDQUFDLENBQUM7WUFDSCxhQUFhLEdBQUcsTUFBTSxDQUFDLGFBQWEsQ0FBQztTQUN0QztLQUNGO0lBRUQsT0FBTztRQUNMLE1BQU07UUFDTixVQUFVO1FBQ1YsYUFBYTtLQUNkLENBQUM7QUFDSixDQUFDO0FBRUQsTUFBTSxlQUFlLEdBQUc7SUFDdEIsMkJBQTJCO0lBQzNCLHdCQUF3QjtJQUN4QixtQkFBbUI7SUFDbkIsb0JBQW9CO0lBQ3BCLGVBQWU7SUFDZixnQkFBZ0I7SUFDaEIsc0JBQXNCO0lBQ3RCLHVCQUF1QjtDQUN4QixDQUFDO0FBRUYsTUFBTSxnQkFBZ0IsR0FBRztJQUN2QiwyQkFBMkI7SUFDM0IsMkJBQTJCO0lBQzNCLHNCQUFzQjtJQUN0QixzQkFBc0I7SUFDdEIsc0JBQXNCO0NBQ3ZCLENBQUM7QUFFRixTQUFTLGFBQWEsQ0FBQyxPQUFnQztJQUNyRCxJQUFJLE9BQU8sS0FBSyxTQUFTLEVBQUU7UUFDekIsT0FBTyxTQUFTLENBQUM7S0FDbEI7SUFDRCxPQUFPLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLEVBQUU7UUFDMUIsT0FBTztZQUNMLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSTtZQUNqQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxXQUFXO1lBQzdCLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTztTQUN4QixDQUFDO0lBQ0osQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgaWFtIGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMga21zIGZyb20gJ2F3cy1jZGstbGliL2F3cy1rbXMnO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnYXdzLWNkay1saWIvYXdzLXMzJztcbmltcG9ydCB7IEFybkZvcm1hdCwgRm4sIElSZXNvdXJjZSwgUmVzb3VyY2UsIFN0YWNrIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBEYXRhRm9ybWF0IH0gZnJvbSAnLi9kYXRhLWZvcm1hdCc7XG5pbXBvcnQgeyBJRGF0YWJhc2UgfSBmcm9tICcuL2RhdGFiYXNlJztcbmltcG9ydCB7IENmblRhYmxlIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWdsdWUnO1xuaW1wb3J0IHsgQ29sdW1uIH0gZnJvbSAnLi9zY2hlbWEnO1xuXG5leHBvcnQgaW50ZXJmYWNlIElUYWJsZSBleHRlbmRzIElSZXNvdXJjZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdGFibGVBcm46IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHRhYmxlTmFtZTogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIFRhYmxlRW5jcnlwdGlvbiB7XG4gIFVORU5DUllQVEVEID0gJ1VuZW5jcnlwdGVkJyxcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgUzNfTUFOQUdFRCA9ICdTU0UtUzMnLFxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBLTVMgPSAnU1NFLUtNUycsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBLTVNfTUFOQUdFRCA9ICdTU0UtS01TLU1BTkFHRUQnLFxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgQ0xJRU5UX1NJREVfS01TID0gJ0NTRS1LTVMnXG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgVGFibGVBdHRyaWJ1dGVzIHtcbiAgcmVhZG9ubHkgdGFibGVBcm46IHN0cmluZztcbiAgcmVhZG9ubHkgdGFibGVOYW1lOiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgVGFibGVQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0YWJsZU5hbWU6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkYXRhYmFzZTogSURhdGFiYXNlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBidWNrZXQ/OiBzMy5JQnVja2V0O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgczNQcmVmaXg/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNvbHVtbnM6IENvbHVtbltdO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwYXJ0aXRpb25LZXlzPzogQ29sdW1uW11cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGF0YUZvcm1hdDogRGF0YUZvcm1hdDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY29tcHJlc3NlZD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbmNyeXB0aW9uPzogVGFibGVFbmNyeXB0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbmNyeXB0aW9uS2V5Pzoga21zLklLZXk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc3RvcmVkQXNTdWJEaXJlY3Rvcmllcz86IGJvb2xlYW47XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIFRhYmxlIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJVGFibGUge1xuXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVRhYmxlQXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHRhYmxlQXJuOiBzdHJpbmcpOiBJVGFibGUge1xuICAgIGNvbnN0IHRhYmxlTmFtZSA9IEZuLnNlbGVjdCgxLCBGbi5zcGxpdCgnLycsIFN0YWNrLm9mKHNjb3BlKS5zcGxpdEFybih0YWJsZUFybiwgQXJuRm9ybWF0LlNMQVNIX1JFU09VUkNFX05BTUUpLnJlc291cmNlTmFtZSEpKTtcblxuICAgIHJldHVybiBUYWJsZS5mcm9tVGFibGVBdHRyaWJ1dGVzKHNjb3BlLCBpZCwge1xuICAgICAgdGFibGVBcm4sXG4gICAgICB0YWJsZU5hbWUsXG4gICAgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tVGFibGVBdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBUYWJsZUF0dHJpYnV0ZXMpOiBJVGFibGUge1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVRhYmxlIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSB0YWJsZUFybiA9IGF0dHJzLnRhYmxlQXJuO1xuICAgICAgcHVibGljIHJlYWRvbmx5IHRhYmxlTmFtZSA9IGF0dHJzLnRhYmxlTmFtZTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgZGF0YWJhc2U6IElEYXRhYmFzZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgY29tcHJlc3NlZDogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGVuY3J5cHRpb246IFRhYmxlRW5jcnlwdGlvbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBlbmNyeXB0aW9uS2V5Pzoga21zLklLZXk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGJ1Y2tldDogczMuSUJ1Y2tldDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgczNQcmVmaXg6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSB0YWJsZU5hbWU6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHRhYmxlQXJuOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgZGF0YUZvcm1hdDogRGF0YUZvcm1hdDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGNvbHVtbnM6IENvbHVtbltdO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBwYXJ0aXRpb25LZXlzPzogQ29sdW1uW107XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFRhYmxlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMudGFibGVOYW1lLFxuICAgIH0pO1xuXG4gICAgdGhpcy5kYXRhYmFzZSA9IHByb3BzLmRhdGFiYXNlO1xuICAgIHRoaXMuZGF0YUZvcm1hdCA9IHByb3BzLmRhdGFGb3JtYXQ7XG4gICAgdGhpcy5zM1ByZWZpeCA9IHByb3BzLnMzUHJlZml4ID8/ICcnO1xuXG4gICAgdmFsaWRhdGVTY2hlbWEocHJvcHMuY29sdW1ucywgcHJvcHMucGFydGl0aW9uS2V5cyk7XG4gICAgdGhpcy5jb2x1bW5zID0gcHJvcHMuY29sdW1ucztcbiAgICB0aGlzLnBhcnRpdGlvbktleXMgPSBwcm9wcy5wYXJ0aXRpb25LZXlzO1xuXG4gICAgdGhpcy5jb21wcmVzc2VkID0gcHJvcHMuY29tcHJlc3NlZCA/PyBmYWxzZTtcbiAgICBjb25zdCB7IGJ1Y2tldCwgZW5jcnlwdGlvbiwgZW5jcnlwdGlvbktleSB9ID0gY3JlYXRlQnVja2V0KHRoaXMsIHByb3BzKTtcbiAgICB0aGlzLmJ1Y2tldCA9IGJ1Y2tldDtcbiAgICB0aGlzLmVuY3J5cHRpb24gPSBlbmNyeXB0aW9uO1xuICAgIHRoaXMuZW5jcnlwdGlvbktleSA9IGVuY3J5cHRpb25LZXk7XG5cbiAgICBjb25zdCB0YWJsZVJlc291cmNlID0gbmV3IENmblRhYmxlKHRoaXMsICdUYWJsZScsIHtcbiAgICAgIGNhdGFsb2dJZDogcHJvcHMuZGF0YWJhc2UuY2F0YWxvZ0lkLFxuXG4gICAgICBkYXRhYmFzZU5hbWU6IHByb3BzLmRhdGFiYXNlLmRhdGFiYXNlTmFtZSxcblxuICAgICAgdGFibGVJbnB1dDoge1xuICAgICAgICBuYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgICAgZGVzY3JpcHRpb246IHByb3BzLmRlc2NyaXB0aW9uIHx8IGAke3Byb3BzLnRhYmxlTmFtZX0gZ2VuZXJhdGVkIGJ5IENES2AsXG5cbiAgICAgICAgcGFydGl0aW9uS2V5czogcmVuZGVyQ29sdW1ucyhwcm9wcy5wYXJ0aXRpb25LZXlzKSxcblxuICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgY2xhc3NpZmljYXRpb246IHByb3BzLmRhdGFGb3JtYXQuY2xhc3NpZmljYXRpb25TdHJpbmc/LnZhbHVlLFxuICAgICAgICAgIGhhc19lbmNyeXB0ZWRfZGF0YTogdGhpcy5lbmNyeXB0aW9uICE9PSBUYWJsZUVuY3J5cHRpb24uVU5FTkNSWVBURUQsXG4gICAgICAgIH0sXG4gICAgICAgIHN0b3JhZ2VEZXNjcmlwdG9yOiB7XG4gICAgICAgICAgbG9jYXRpb246IGBzMzovLyR7dGhpcy5idWNrZXQuYnVja2V0TmFtZX0vJHt0aGlzLnMzUHJlZml4fWAsXG4gICAgICAgICAgY29tcHJlc3NlZDogdGhpcy5jb21wcmVzc2VkLFxuICAgICAgICAgIHN0b3JlZEFzU3ViRGlyZWN0b3JpZXM6IHByb3BzLnN0b3JlZEFzU3ViRGlyZWN0b3JpZXMgPz8gZmFsc2UsXG4gICAgICAgICAgY29sdW1uczogcmVuZGVyQ29sdW1ucyhwcm9wcy5jb2x1bW5zKSxcbiAgICAgICAgICBpbnB1dEZvcm1hdDogcHJvcHMuZGF0YUZvcm1hdC5pbnB1dEZvcm1hdC5jbGFzc05hbWUsXG4gICAgICAgICAgb3V0cHV0Rm9ybWF0OiBwcm9wcy5kYXRhRm9ybWF0Lm91dHB1dEZvcm1hdC5jbGFzc05hbWUsXG4gICAgICAgICAgc2VyZGVJbmZvOiB7XG4gICAgICAgICAgICBzZXJpYWxpemF0aW9uTGlicmFyeTogcHJvcHMuZGF0YUZvcm1hdC5zZXJpYWxpemF0aW9uTGlicmFyeS5jbGFzc05hbWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcblxuICAgICAgICB0YWJsZVR5cGU6ICdFWFRFUk5BTF9UQUJMRScsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGhpcy50YWJsZU5hbWUgPSB0aGlzLmdldFJlc291cmNlTmFtZUF0dHJpYnV0ZSh0YWJsZVJlc291cmNlLnJlZik7XG4gICAgdGhpcy50YWJsZUFybiA9IHRoaXMuc3RhY2suZm9ybWF0QXJuKHtcbiAgICAgIHNlcnZpY2U6ICdnbHVlJyxcbiAgICAgIHJlc291cmNlOiAndGFibGUnLFxuICAgICAgcmVzb3VyY2VOYW1lOiBgJHt0aGlzLmRhdGFiYXNlLmRhdGFiYXNlTmFtZX0vJHt0aGlzLnRhYmxlTmFtZX1gLFxuICAgIH0pO1xuICAgIHRoaXMubm9kZS5kZWZhdWx0Q2hpbGQgPSB0YWJsZVJlc291cmNlO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBncmFudFJlYWQoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQge1xuICAgIGNvbnN0IHJldCA9IHRoaXMuZ3JhbnQoZ3JhbnRlZSwgcmVhZFBlcm1pc3Npb25zKTtcbiAgICBpZiAodGhpcy5lbmNyeXB0aW9uS2V5ICYmIHRoaXMuZW5jcnlwdGlvbiA9PT0gVGFibGVFbmNyeXB0aW9uLkNMSUVOVF9TSURFX0tNUykgeyB0aGlzLmVuY3J5cHRpb25LZXkuZ3JhbnREZWNyeXB0KGdyYW50ZWUpOyB9XG4gICAgdGhpcy5idWNrZXQuZ3JhbnRSZWFkKGdyYW50ZWUsIHRoaXMuZ2V0UzNQcmVmaXhGb3JHcmFudCgpKTtcbiAgICByZXR1cm4gcmV0O1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgZ3JhbnRXcml0ZShncmFudGVlOiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudCB7XG4gICAgY29uc3QgcmV0ID0gdGhpcy5ncmFudChncmFudGVlLCB3cml0ZVBlcm1pc3Npb25zKTtcbiAgICBpZiAodGhpcy5lbmNyeXB0aW9uS2V5ICYmIHRoaXMuZW5jcnlwdGlvbiA9PT0gVGFibGVFbmNyeXB0aW9uLkNMSUVOVF9TSURFX0tNUykgeyB0aGlzLmVuY3J5cHRpb25LZXkuZ3JhbnRFbmNyeXB0KGdyYW50ZWUpOyB9XG4gICAgdGhpcy5idWNrZXQuZ3JhbnRXcml0ZShncmFudGVlLCB0aGlzLmdldFMzUHJlZml4Rm9yR3JhbnQoKSk7XG4gICAgcmV0dXJuIHJldDtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGdyYW50UmVhZFdyaXRlKGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICBjb25zdCByZXQgPSB0aGlzLmdyYW50KGdyYW50ZWUsIFsuLi5yZWFkUGVybWlzc2lvbnMsIC4uLndyaXRlUGVybWlzc2lvbnNdKTtcbiAgICBpZiAodGhpcy5lbmNyeXB0aW9uS2V5ICYmIHRoaXMuZW5jcnlwdGlvbiA9PT0gVGFibGVFbmNyeXB0aW9uLkNMSUVOVF9TSURFX0tNUykgeyB0aGlzLmVuY3J5cHRpb25LZXkuZ3JhbnRFbmNyeXB0RGVjcnlwdChncmFudGVlKTsgfVxuICAgIHRoaXMuYnVja2V0LmdyYW50UmVhZFdyaXRlKGdyYW50ZWUsIHRoaXMuZ2V0UzNQcmVmaXhGb3JHcmFudCgpKTtcbiAgICByZXR1cm4gcmV0O1xuICB9XG5cbiAgcHJpdmF0ZSBncmFudChncmFudGVlOiBpYW0uSUdyYW50YWJsZSwgYWN0aW9uczogc3RyaW5nW10pIHtcbiAgICByZXR1cm4gaWFtLkdyYW50LmFkZFRvUHJpbmNpcGFsKHtcbiAgICAgIGdyYW50ZWUsXG4gICAgICByZXNvdXJjZUFybnM6IFt0aGlzLnRhYmxlQXJuXSxcbiAgICAgIGFjdGlvbnMsXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIGdldFMzUHJlZml4Rm9yR3JhbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMuczNQcmVmaXggKyAnKic7XG4gIH1cbn1cblxuZnVuY3Rpb24gdmFsaWRhdGVTY2hlbWEoY29sdW1uczogQ29sdW1uW10sIHBhcnRpdGlvbktleXM/OiBDb2x1bW5bXSk6IHZvaWQge1xuICBpZiAoY29sdW1ucy5sZW5ndGggPT09IDApIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ3lvdSBtdXN0IHNwZWNpZnkgYXQgbGVhc3Qgb25lIGNvbHVtbiBmb3IgdGhlIHRhYmxlJyk7XG4gIH1cbiAgLy8gQ2hlY2sgdGhlcmUgaXMgYXQgbGVhc3Qgb25lIGNvbHVtbiBhbmQgbm8gZHVwbGljYXRlZCBjb2x1bW4gbmFtZXMgb3IgcGFydGl0aW9uIGtleXMuXG4gIGNvbnN0IG5hbWVzID0gbmV3IFNldDxzdHJpbmc+KCk7XG4gIChjb2x1bW5zLmNvbmNhdChwYXJ0aXRpb25LZXlzIHx8IFtdKSkuZm9yRWFjaChjb2x1bW4gPT4ge1xuICAgIGlmIChuYW1lcy5oYXMoY29sdW1uLm5hbWUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYGNvbHVtbiBuYW1lcyBhbmQgcGFydGl0aW9uIGtleXMgbXVzdCBiZSB1bmlxdWUsIGJ1dCBcXCcke2NvbHVtbi5uYW1lfVxcJyBpcyBkdXBsaWNhdGVkYCk7XG4gICAgfVxuICAgIG5hbWVzLmFkZChjb2x1bW4ubmFtZSk7XG4gIH0pO1xufVxuXG4vLyBtYXAgVGFibGVFbmNyeXB0aW9uIHRvIGJ1Y2tldCdzIFNTRSBjb25maWd1cmF0aW9uIChzMy5CdWNrZXRFbmNyeXB0aW9uKVxuY29uc3QgZW5jcnlwdGlvbk1hcHBpbmdzID0ge1xuICBbVGFibGVFbmNyeXB0aW9uLlMzX01BTkFHRURdOiBzMy5CdWNrZXRFbmNyeXB0aW9uLlMzX01BTkFHRUQsXG4gIFtUYWJsZUVuY3J5cHRpb24uS01TX01BTkFHRURdOiBzMy5CdWNrZXRFbmNyeXB0aW9uLktNU19NQU5BR0VELFxuICBbVGFibGVFbmNyeXB0aW9uLktNU106IHMzLkJ1Y2tldEVuY3J5cHRpb24uS01TLFxuICBbVGFibGVFbmNyeXB0aW9uLkNMSUVOVF9TSURFX0tNU106IHMzLkJ1Y2tldEVuY3J5cHRpb24uVU5FTkNSWVBURUQsXG4gIFtUYWJsZUVuY3J5cHRpb24uVU5FTkNSWVBURURdOiBzMy5CdWNrZXRFbmNyeXB0aW9uLlVORU5DUllQVEVELFxufTtcblxuLy8gY3JlYXRlIHRoZSBidWNrZXQgdG8gc3RvcmUgYSB0YWJsZSdzIGRhdGEgZGVwZW5kaW5nIG9uIHRoZSBgZW5jcnlwdGlvbmAgYW5kIGBlbmNyeXB0aW9uS2V5YCBwcm9wZXJ0aWVzLlxuZnVuY3Rpb24gY3JlYXRlQnVja2V0KHRhYmxlOiBUYWJsZSwgcHJvcHM6IFRhYmxlUHJvcHMpIHtcbiAgY29uc3QgZW5jcnlwdGlvbiA9IHByb3BzLmVuY3J5cHRpb24gfHwgVGFibGVFbmNyeXB0aW9uLlVORU5DUllQVEVEO1xuICBsZXQgYnVja2V0ID0gcHJvcHMuYnVja2V0O1xuXG4gIGlmIChidWNrZXQgJiYgKGVuY3J5cHRpb24gIT09IFRhYmxlRW5jcnlwdGlvbi5VTkVOQ1JZUFRFRCAmJiBlbmNyeXB0aW9uICE9PSBUYWJsZUVuY3J5cHRpb24uQ0xJRU5UX1NJREVfS01TKSkge1xuICAgIHRocm93IG5ldyBFcnJvcigneW91IGNhbiBub3Qgc3BlY2lmeSBlbmNyeXB0aW9uIHNldHRpbmdzIGlmIHlvdSBhbHNvIHByb3ZpZGUgYSBidWNrZXQnKTtcbiAgfVxuXG4gIGxldCBlbmNyeXB0aW9uS2V5OiBrbXMuSUtleSB8IHVuZGVmaW5lZDtcbiAgaWYgKGVuY3J5cHRpb24gPT09IFRhYmxlRW5jcnlwdGlvbi5DTElFTlRfU0lERV9LTVMgJiYgcHJvcHMuZW5jcnlwdGlvbktleSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgLy8gQ1NFLUtNUyBzaG91bGQgYmVoYXZlIHRoZSBzYW1lIGFzIFNTRS1LTVMgLSB1c2UgdGhlIHByb3ZpZGVkIGtleSBvciBjcmVhdGUgb25lIGF1dG9tYXRpY2FsbHlcbiAgICAvLyBTaW5jZSBCdWNrZXQgb25seSBrbm93cyBhYm91dCBTU0UsIHdlIHJlcGVhdCB0aGUgbG9naWMgZm9yIENTRS1LTVMgYXQgdGhlIFRhYmxlIGxldmVsLlxuICAgIGVuY3J5cHRpb25LZXkgPSBuZXcga21zLktleSh0YWJsZSwgJ0tleScpO1xuICB9IGVsc2Uge1xuICAgIGVuY3J5cHRpb25LZXkgPSBwcm9wcy5lbmNyeXB0aW9uS2V5O1xuICB9XG5cbiAgLy8gY3JlYXRlIHRoZSBidWNrZXQgaWYgbm9uZSB3YXMgcHJvdmlkZWRcbiAgaWYgKCFidWNrZXQpIHtcbiAgICBpZiAoZW5jcnlwdGlvbiA9PT0gVGFibGVFbmNyeXB0aW9uLkNMSUVOVF9TSURFX0tNUykge1xuICAgICAgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldCh0YWJsZSwgJ0J1Y2tldCcpO1xuICAgIH0gZWxzZSB7XG4gICAgICBidWNrZXQgPSBuZXcgczMuQnVja2V0KHRhYmxlLCAnQnVja2V0Jywge1xuICAgICAgICBlbmNyeXB0aW9uOiBlbmNyeXB0aW9uTWFwcGluZ3NbZW5jcnlwdGlvbl0sXG4gICAgICAgIGVuY3J5cHRpb25LZXksXG4gICAgICB9KTtcbiAgICAgIGVuY3J5cHRpb25LZXkgPSBidWNrZXQuZW5jcnlwdGlvbktleTtcbiAgICB9XG4gIH1cblxuICByZXR1cm4ge1xuICAgIGJ1Y2tldCxcbiAgICBlbmNyeXB0aW9uLFxuICAgIGVuY3J5cHRpb25LZXksXG4gIH07XG59XG5cbmNvbnN0IHJlYWRQZXJtaXNzaW9ucyA9IFtcbiAgJ2dsdWU6QmF0Y2hEZWxldGVQYXJ0aXRpb24nLFxuICAnZ2x1ZTpCYXRjaEdldFBhcnRpdGlvbicsXG4gICdnbHVlOkdldFBhcnRpdGlvbicsXG4gICdnbHVlOkdldFBhcnRpdGlvbnMnLFxuICAnZ2x1ZTpHZXRUYWJsZScsXG4gICdnbHVlOkdldFRhYmxlcycsXG4gICdnbHVlOkdldFRhYmxlVmVyc2lvbicsXG4gICdnbHVlOkdldFRhYmxlVmVyc2lvbnMnLFxuXTtcblxuY29uc3Qgd3JpdGVQZXJtaXNzaW9ucyA9IFtcbiAgJ2dsdWU6QmF0Y2hDcmVhdGVQYXJ0aXRpb24nLFxuICAnZ2x1ZTpCYXRjaERlbGV0ZVBhcnRpdGlvbicsXG4gICdnbHVlOkNyZWF0ZVBhcnRpdGlvbicsXG4gICdnbHVlOkRlbGV0ZVBhcnRpdGlvbicsXG4gICdnbHVlOlVwZGF0ZVBhcnRpdGlvbicsXG5dO1xuXG5mdW5jdGlvbiByZW5kZXJDb2x1bW5zKGNvbHVtbnM/OiBBcnJheTxDb2x1bW4gfCBDb2x1bW4+KSB7XG4gIGlmIChjb2x1bW5zID09PSB1bmRlZmluZWQpIHtcbiAgICByZXR1cm4gdW5kZWZpbmVkO1xuICB9XG4gIHJldHVybiBjb2x1bW5zLm1hcChjb2x1bW4gPT4ge1xuICAgIHJldHVybiB7XG4gICAgICBuYW1lOiBjb2x1bW4ubmFtZSxcbiAgICAgIHR5cGU6IGNvbHVtbi50eXBlLmlucHV0U3RyaW5nLFxuICAgICAgY29tbWVudDogY29sdW1uLmNvbW1lbnQsXG4gICAgfTtcbiAgfSk7XG59XG4iXX0=