# Copyright 2019 The Google Research Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from moai.supervision.losses.regression.robust.adaptive import cubic_spline
from moai.supervision.losses.regression.robust.adaptive import generic
from moai.supervision.losses.regression.robust.adaptive import util

import numpy as np
import torch

__all__ = [
    'Distribution',
]

def partition_spline_curve(alpha: torch.Tensor) -> torch.Tensor:
    """Applies a curve to alpha >= 0 to compress its range before interpolation.

    This is a weird hand-crafted function designed to take in alpha values and
    curve them to occupy a short finite range that works well when using spline
    interpolation to model the partition function Z(alpha). Because Z(alpha)
    is only varied in [0, 4] and is especially interesting around alpha=2, this
    curve is roughly linear in [0, 4] with a slope of ~1 at alpha=0 and alpha=4
    but a slope of ~10 at alpha=2. When alpha > 4 the curve becomes logarithmic.
    Some (input, output) pairs for this function are:
        [(0, 0), (1, ~1.2), (2, 4), (3, ~6.8), (4, 8), (8, ~8.8), (400000, ~12)]
    This function is continuously differentiable.

    Args:
        alpha: A numpy array or tensor (float32 or float64) with values >= 0.

    Returns:
        An array/tensor of curved values >= 0 with the same type as `alpha`, to be
        used as input x-coordinates for spline interpolation.
    """
    return torch.where(
        alpha < 4, 
        (2.25 * alpha - 4.5) / (torch.abs(alpha - 2) + 0.25) + alpha + 2,
        5. / 18. * util.log_safe(4 * alpha - 15) + 8
    )

class Distribution(torch.nn.Module):
    # values, tangents, and x-coordinate scaling of a spline that approximates the partition function. 
    VALUES = np.array([1.491303391769255526e+00,1.490550577878432925e+00,1.489803187879397584e+00,1.489061146962997295e+00,1.488324362700598869e+00,1.487592731043221317e+00,1.486866151716329298e+00,1.486144528008113808e+00,1.485427766574330732e+00,1.484715777258027725e+00,1.484008472922677901e+00,1.483305769297434740e+00,1.482607584833359082e+00,1.481913840569622165e+00,1.481224460008776056e+00,1.480539369000307914e+00,1.479858495631762860e+00,1.479181770126800188e+00,1.478509124749616710e+00,1.477840493715223635e+00,1.477175813105115809e+00,1.476515020787919408e+00,1.475858056344635738e+00,1.475204860998147405e+00,1.474555377546668655e+00,1.473909550300859461e+00,1.473267325024348873e+00,1.472628648877428281e+00,1.471993470363698098e+00,1.471361739279472003e+00,1.470733406665758025e+00,1.470108424762648136e+00,1.469486746965959600e+00,1.468868327785996408e+00,1.468253122808283129e+00,1.467641088656169490e+00,1.467032182955177566e+00,1.466426364298997642e+00,1.465823592217031957e+00,1.465223827143399049e+00,1.464627030387314344e+00,1.464033164104770135e+00,1.463442191271440596e+00,1.462854075656750963e+00,1.462268781799039852e+00,1.461686274981762734e+00,1.461106521210675968e+00,1.460529487191958520e+00,1.459955140311211874e+00,1.459383448613302159e+00,1.458814380783002296e+00,1.458247906126383331e+00,1.457683994552935403e+00,1.457122616558364836e+00,1.456563743208050044e+00,1.456007346121113821e+00,1.455453397455088949e+00,1.454901869891148003e+00,1.454352736619869413e+00,1.453805971327517987e+00,1.453261548182817275e+00,1.452719441824183999e+00,1.452179627347418567e+00,1.451642080293812143e+00,1.451106776638677154e+00,1.450573692780251278e+00,1.450042805528994450e+00,1.449514092097230034e+00,1.448987530089137499e+00,1.448463097491072604e+00,1.447940772662202891e+00,1.447420534325447594e+00,1.446902361558704886e+00,1.446386233786362885e+00,1.445872130771075126e+00,1.445360032605794709e+00,1.444849919706054253e+00,1.444341772802487212e+00,1.443835572933572342e+00,1.443331301438603331e+00,1.442828939950866696e+00,1.442328470391023076e+00,1.441829874960687263e+00,1.441333136136194515e+00,1.440838236662548066e+00,1.440345159547546050e+00,1.439853888056071618e+00,1.439364405704550709e+00,1.438876696255563692e+00,1.438390743712610442e+00,1.437906532315020414e+00,1.437424046533003485e+00,1.436943271062840255e+00,1.436464190822198450e+00,1.435986790945581904e+00,1.435511056779897432e+00,1.435036973880145617e+00,1.434564528005219941e+00,
        1.434093705113824502e+00,1.433624491360491193e+00,1.433156873091706140e+00,1.432690836842134718e+00,1.432226369330942273e+00,1.431763457458212985e+00,1.431302088301454667e+00,1.430842249112195486e+00,1.430383927312664172e+00,1.429927110492554831e+00,1.429471786405870803e+00,1.429017942967843791e+00,1.428565568251934481e+00,1.428114650486900228e+00,1.427665178053936668e+00,1.427217139483887509e+00,1.426770523454520712e+00,1.426325318787869278e+00,1.425881514447635778e+00,1.425439099536656151e+00,1.424998063294425243e+00,1.424558395094675856e+00,1.424120084443018408e+00,1.423683120974632121e+00,1.423247494452006823e+00,1.422813194762741817e+00,1.422380211917388060e+00,1.421948536047342948e+00,1.421518157402791971e+00,1.421089066350691654e+00,1.420661253372803579e+00,1.420234709063766143e+00,1.419809424129212072e+00,1.419385389383924112e+00,1.418962595750031364e+00,1.418541034255246913e+00,1.418120696031139660e+00,1.417701572311443226e+00,1.417283654430403717e+00,1.416866933821159247e+00,1.416451402014154537e+00,1.416037050635588157e+00,1.415623871405893075e+00,1.415211856138246738e+00,1.414800996737113570e+00,1.414391285196815895e+00,1.413982713600134389e+00,1.413575274116937397e+00,1.413168959002837122e+00,1.412763760597872453e+00,1.412359671325218224e+00,1.411956683689923109e+00,1.411554790277665505e+00,1.411153983753541397e+00,1.410754256860871525e+00,1.410355602420032550e+00,1.409958013327310855e+00,1.409561482553780332e+00,1.409166003144198598e+00,1.408771568215926751e+00,1.408378170957868214e+00,1.407985804629429127e+00,1.407594462559496273e+00,1.407204138145435879e+00,1.406814824852111512e+00,1.406426516210918187e+00,1.406039205818834681e+00,1.405652887337497825e+00,1.405267554492285020e+00,1.404883201071423615e+00,1.404499820925106945e+00,1.404117407964635023e+00,1.403735956161563214e+00,1.403355459546871575e+00,1.402975912210147724e+00,1.402597308298780820e+00,1.402219642017175749e+00,1.401842907625976187e+00,1.401467099441303210e+00,1.401092211834009449e+00,1.400718239228940787e+00,1.400345176104216938e+00,1.399973016990520236e+00,1.399601756470398195e+00,1.399231389177578722e+00,1.398861909796295544e+00,1.398493313060625631e+00,1.398125593753839713e+00,1.397758746707759459e+00,1.397392766802131092e+00,1.397027648964004554e+00,1.396663388167123987e+00,1.396299979431333549e+00,1.395937417821983884e+00,1.395575698449356583e+00,1.395214816468094643e+00,1.394854767076640467e+00,1.394495545516686974e+00,1.394137147072635363e+00,1.393779567071060654e+00,1.393422800880190326e+00,1.393066843909383623e+00,
        1.392711691608627511e+00,1.392357339468034638e+00,1.392003783017353946e+00,1.391651017825483061e+00,1.391299039499995116e+00,1.390947843686668461e+00,1.390597426069023701e+00,1.390247782367871832e+00,1.389898908340865935e+00,1.389550799782059975e+00,1.389203452521474924e+00,1.388856862424673100e+00,1.388511025392337173e+00,1.388165937359856050e+00,1.387821594296916095e+00,1.387477992207101662e+00,1.387135127127498091e+00,1.386792995128302008e+00,1.386451592312437642e+00,1.386110914815180228e+00,1.385770958803782316e+00,1.385431720477106721e+00,1.385093196065265486e+00,1.384755381829263943e+00,1.384418274060649656e+00,1.384081869081167149e+00,1.383746163242416172e+00,1.383411152925516197e+00,1.383076834540776456e+00,1.382743204527368430e+00,1.382410259353005877e+00,1.382077995513626201e+00,1.381746409533080477e+00,1.381415497962823258e+00,1.381085257381610809e+00,1.380755684395200911e+00,1.380426775636057313e+00,1.380098527763061300e+00,1.379770937461222591e+00,1.379444001441398004e+00,1.379117716440011243e+00,1.378792079218779332e+00,1.378467086564440391e+00,1.378142735288487408e+00,1.377819022226903778e+00,1.377495944239902848e+00,1.377173498211671898e+00,1.376851681050119014e+00,1.376530489686623726e+00,1.376209921075789655e+00,1.375889972195203148e+00,1.375570640045191695e+00,1.375251921648590780e+00,1.374933814050504965e+00,1.374616314318084287e+00,1.374299419540292444e+00,1.373983126827684087e+00,1.373667433312184549e+00,1.373352336146870023e+00,1.373037832505753286e+00,1.372723919583570540e+00,1.372410594595572242e+00,1.372097854777315273e+00,1.371785697384459102e+00,1.371474119692563942e+00,1.371163118996890695e+00,1.370852692612207102e+00,1.370542837872589459e+00,1.370233552131235211e+00,1.369924832760270217e+00,1.369616677150565343e+00,1.369309082711549719e+00,1.369002046871029332e+00,1.368695567075007169e+00,1.368389640787506911e+00,1.368084265490395968e+00,1.367779438683212279e+00,1.367475157882996006e+00,1.367171420624117228e+00,1.366868224458112291e+00,1.366565566953515942e+00,1.366263445695702572e+00,1.365961858286722119e+00,1.365660802345142866e+00,1.365360275505895338e+00,1.365060275420115765e+00,1.364760799754995091e+00,1.364461846193625538e+00,1.364163412434854061e+00,1.363865496193133131e+00,1.363568095198375074e+00,1.363271207195808188e+00,1.362974829945835076e+00,1.362678961223891649e+00,1.362383598820309016e+00,1.362088740540174703e+00,1.361794384203197872e+00,1.361500527643575431e+00,1.361207168709860360e+00,1.360914305264829371e+00,1.360621935185353903e+00,1.360330056362272666e+00,1.360038666700265297e+00,1.359747764117726909e+00,1.359457346546645740e+00,1.359167411932479697e+00,
        1.358877958234036454e+00,1.358588983423355101e+00,1.358300485485586462e+00,1.358012462418876964e+00,1.357724912234255399e+00,1.357437832955517010e+00,1.357151222619110253e+00,1.356865079274028218e+00,1.356579400981696271e+00,1.356294185815863917e+00,1.356009431862497339e+00,1.355725137219672582e+00,1.355441299997470983e+00,1.355157918317873689e+00,1.354874990314661298e+00,1.354592514133309278e+00,1.354310487930889151e+00,1.354028909875968578e+00,1.353747778148511882e+00,1.353467090939784789e+00,1.353186846452255843e+00,1.352907042899503143e+00,1.352627678506117981e+00,1.352348751507614466e+00,1.352070260150335157e+00,1.351792202691359357e+00,1.351514577398416517e+00,1.351237382549791866e+00,1.350960616434243144e+00,1.350684277350909790e+00,1.350408363609227669e+00,1.350132873528844701e+00,1.349857805439534042e+00,1.349583157681112144e+00,1.349308928603356161e+00,1.349035116565920012e+00,1.348761719938256443e+00,1.348488737099533319e+00,1.348216166438556796e+00,1.347944006353692714e+00,1.347672255252786888e+00,1.347400911553090719e+00,1.347129973681183035e+00,1.346859440072896374e+00,1.346589309173240823e+00,1.346319579436331626e+00,1.346050249325316361e+00,1.345781317312300773e+00,1.345512781878279496e+00,1.345244641513063444e+00,1.344976894715211424e+00,1.344709539991959302e+00,1.344442575859152278e+00,1.344176000841176277e+00,1.343909813470891113e+00,1.343644012289564538e+00,1.343378595846804302e+00,1.343113562700494867e+00,1.342848911416732127e+00,1.342584640569759680e+00,1.342320748741904657e+00,1.342057234523515108e+00,1.341794096512900047e+00,1.341531333316264618e+00,1.341268943547652581e+00,1.341006925828882590e+00,1.340745278789492234e+00,1.340484001066675646e+00,1.340223091305227543e+00,1.339962548157483502e+00,1.339702370283263555e+00,1.339442556349815128e+00,1.339183105031756638e+00,1.338924015011022206e+00,1.338665284976806369e+00,1.338406913625508787e+00,1.338148899660680513e+00,1.337891241792971364e+00,1.337633938740074857e+00,1.337376989226677804e+00,1.337120391984406353e+00,1.336864145751776478e+00,1.336608249274139348e+00,1.336352701303635593e+00,1.336097500599141341e+00,1.335842645926219596e+00,1.335588136057072273e+00,1.335333969770489126e+00,1.335080145851802458e+00,1.334826663092836707e+00,1.334573520291862492e+00,1.334320716253548422e+00,1.334068249788916694e+00,1.333816119715295345e+00,1.333564324856272520e+00,1.333312864041652279e+00,1.333061736107410189e+00,1.332810939895646030e+00,1.332560474254544047e+00,1.332310338038326103e+00,1.332060530107209484e+00,1.331811049327365382e+00,1.331561894570873816e+00,1.331313064715684780e+00,1.331064558645574269e+00,1.330816375250103656e+00,1.330568513424579713e+00,
        1.330320972070012653e+00,1.330073750093076601e+00,1.329826846406071406e+00,1.329580259926880448e+00,1.329333989578933117e+00,1.329088034291166398e+00,1.328842392997985122e+00,1.328597064639225556e+00,1.328352048160116983e+00,1.328107342511243516e+00,1.327862946648509235e+00,1.327618859533098439e+00,1.327375080131442120e+00,1.327131607415181103e+00,1.326888440361128740e+00,1.326645577951237165e+00,1.326403019172561981e+00,1.326160763017226962e+00,1.325918808482390521e+00,1.325677154570209071e+00,1.325435800287806165e+00,1.325194744647237188e+00,1.324953986665456274e+00,1.324713525364283218e+00,1.324473359770371061e+00,1.324233488915173451e+00,1.323993911834911774e+00,1.323754627570544962e+00,1.323515635167735738e+00,1.323276933676821088e+00,1.323038522152779395e+00,1.322800399655201353e+00,1.322562565248258437e+00,1.322325018000672037e+00,1.322087756985684592e+00,1.321850781281029397e+00,1.321614089968900618e+00,1.321377682135924436e+00,1.321141556873129508e+00,1.320905713275918325e+00,1.320670150444039459e+00,1.320434867481557806e+00,1.320199863496827497e+00,1.319965137602463257e+00,1.319730688915313754e+00,1.319496516556433408e+00,1.319262619651055291e+00,1.319028997328564712e+00,1.318795648722471459e+00,1.318562572970384483e+00,1.318329769213984148e+00,1.318097236598998467e+00,1.317864974275174461e+00,1.317632981396253733e+00,1.317401257119948266e+00,1.317169800607913110e+00,1.316938611025721517e+00,1.316707687542842953e+00,1.316477029332613791e+00,1.316246635572217549e+00,1.316016505442656914e+00,1.315786638128732200e+00,1.315557032819016259e+00,1.315327688705831166e+00,1.315098604985224462e+00,1.314869780856946724e+00,1.314641215524427142e+00,1.314412908194751539e+00,1.314184858078640383e+00,1.313957064390423701e+00,1.313729526348021537e+00,1.313502243172919526e+00,1.313275214090148690e+00,1.313048438328262568e+00,1.312821915119315230e+00,1.312595643698841075e+00,1.312369623305831512e+00,1.312143853182716313e+00,1.311918332575340740e+00,1.311693060732944671e+00,1.311468036908143064e+00,1.311243260356905305e+00,1.311018730338533222e+00,1.310794446115643552e+00,1.310570406954144618e+00,1.310346612123220122e+00,1.310123060895306057e+00,1.309899752546073604e+00,1.309676686354408037e+00,1.309453861602390745e+00,1.309231277575278352e+00,1.309008933561486288e+00,1.308786828852567918e+00,1.308564962743195226e+00,1.308343334531143709e+00,1.308121943517269514e+00,1.307900789005495001e+00,1.307679870302787872e+00,1.307459186719144517e+00,1.307238737567572695e+00,1.307018522164071994e+00,1.306798539827616956e+00,1.306578789880141089e+00,1.306359271646517550e+00,1.306139984454542935e+00,1.305920927634919959e+00,1.305702100521240361e+00,
        1.305483502449968247e+00,1.305265132760424107e+00,1.305046990794766382e+00,1.304829075897977697e+00,1.304611387417845103e+00,1.304393924704947416e+00,1.304176687112636568e+00,1.303959673997022950e+00,1.303742884716959205e+00,1.303526318634024239e+00,1.303309975112507901e+00,1.303093853519395440e+00,1.302877953224352181e+00,1.302662273599707543e+00,1.302446814020441490e+00,1.302231573864167880e+00,1.302016552511119363e+00,1.301801749344134729e+00,1.301587163748641363e+00,1.301372795112643477e+00,1.301158642826704126e+00,1.300944706283934327e+00,1.300730984879976626e+00,1.300517478012991335e+00,1.300304185083642761e+00,1.300091105495084554e+00,1.299878238652946161e+00,1.299665583965320170e+00,1.299453140842745880e+00,1.299240908698198416e+00,1.299028886947074080e+00,1.298817075007176136e+00,1.298605472298702823e+00,1.298394078244234029e+00,1.298182892268716859e+00,1.297971913799454535e+00,1.297761142266090850e+00,1.297550577100600400e+00,1.297340217737273260e+00,1.297130063612704109e+00,1.296920114165777793e+00,1.296710368837659333e+00,1.296500827071777939e+00,1.296291488313819018e+00,1.296082352011707739e+00,1.295873417615600376e+00,1.295664684577869652e+00,1.295456152353094748e+00,1.295247820398047089e+00,1.295039688171681025e+00,1.294831755135120499e+00,1.294624020751647731e+00,1.294416484486691887e+00,1.294209145807816874e+00,1.294002004184711785e+00,1.293795059089176247e+00,1.293588309995112873e+00,1.293381756378513714e+00,1.293175397717449382e+00,1.292969233492058612e+00,1.292763263184536937e+00,1.292557486279126477e+00,1.292351902262104169e+00,1.292146510621771327e+00,1.291941310848442992e+00,1.291736302434437711e+00,1.291531484874067104e+00,1.291326857663623873e+00,1.291122420301373364e+00,1.290918172287542465e+00,1.290714113124308948e+00,1.290510242315792366e+00,1.290306559368042283e+00,1.290103063789029836e+00,1.289899755088637079e+00,1.289696632778647656e+00,1.289493696372735032e+00,1.289290945386455611e+00,1.289088379337236079e+00,1.288885997744366740e+00,1.288683800128989310e+00,1.288481786014088915e+00,1.288279954924483883e+00,1.288078306386817085e+00,1.287876839929546380e+00,1.287675555082933521e+00,1.287474451379038598e+00,1.287273528351708274e+00,1.287072785536566233e+00,1.286872222471006078e+00,1.286671838694181558e+00,1.286471633746997245e+00,1.286271607172100095e+00,1.286071758513869678e+00,1.285872087318412182e+00,1.285672593133548203e+00,1.285473275508806079e+00,1.285274133995413237e+00,1.285075168146286639e+00,1.284876377516027235e+00,1.284677761660906414e+00,1.284479320138863123e+00,1.284281052509491428e+00,1.284082958334035185e+00,1.283885037175377164e+00,1.283687288598033494e+00,1.283489712168144115e+00,
        1.283292307453464565e+00,1.283095074023358650e+00,1.282898011448790676e+00,1.282701119302316783e+00,1.282504397158077181e+00,1.282307844591789481e+00,1.282111461180739598e+00,1.281915246503774419e+00,1.281719200141294479e+00,1.281523321675246851e+00,1.281327610689116270e+00,1.281132066767917799e+00,1.280936689498191949e+00,1.280741478467993355e+00,1.280546433266884998e+00,1.280351553485933103e+00,1.280156838717696699e+00,1.279962288556221628e+00,1.279767902597033657e+00,1.279573680437130712e+00,1.279379621674977097e+00,1.279185725910494176e+00,1.278991992745055928e+00,1.278798421781479400e+00,1.278605012624020709e+00,1.278411764878365720e+00,1.278218678151623378e+00,1.278025752052321495e+00,1.277832986190396314e+00,1.277640380177188950e+00,1.277447933625436072e+00,1.277255646149266344e+00,1.277063517364190215e+00,1.276871546887096809e+00,1.276679734336245042e+00,1.276488079331258296e+00,1.276296581493117532e+00,1.276105240444154854e+00,1.275914055808048175e+00,1.275723027209813010e+00,1.275532154275798025e+00,1.275341436633677938e+00,1.275150873912447302e+00,1.274960465742414506e+00,1.274770211755196003e+00,1.274580111583709430e+00,1.274390164862167607e+00,1.274200371226074324e+00,1.274010730312215456e+00,1.273821241758655409e+00,1.273631905204730241e+00,1.273442720291040997e+00,1.273253686659450379e+00,1.273064803953073643e+00,1.272876071816275267e+00,1.272687489894662738e+00,1.272499057835080105e+00,1.272310775285603102e+00,1.272122641895532702e+00,1.271934657315391126e+00,1.271746821196913402e+00,1.271559133193045144e+00,1.271371592957934782e+00,1.271184200146929344e+00,1.270996954416567792e+00,1.270809855424576806e+00,1.270622902829864787e+00,1.270436096292516526e+00,1.270249435473788546e+00,1.270062920036102438e+00,1.269876549643041974e+00,1.269690323959344225e+00,1.269504242650898229e+00,1.269318305384737666e+00,1.269132511829036414e+00,1.268946861653102998e+00,1.268761354527375484e+00,1.268575990123417263e+00,1.268390768113911715e+00,1.268205688172656220e+00,1.268020749974558381e+00,1.267835953195631138e+00,1.267651297512986996e+00,1.267466782604833364e+00,1.267282408150468997e+00,1.267098173830277563e+00,1.266914079325722975e+00,1.266730124319346062e+00,1.266546308494758355e+00,1.266362631536637640e+00,1.266179093130724409e+00,1.265995692963815644e+00,1.265812430723761262e+00,1.265629306099459228e+00,1.265446318780851120e+00,1.265263468458916574e+00,1.265080754825670395e+00,1.264898177574157012e+00,1.264715736398446033e+00,1.264533430993628027e+00,1.264351261055811193e+00,1.264169226282114256e+00,1.263987326370664688e+00,1.263805561020592938e+00,1.263623929932028878e+00,1.263442432806097138e+00,1.263261069344912890e+00,
        1.263079839251577630e+00,1.262898742230174287e+00,1.262717777985765011e+00,1.262536946224384282e+00,1.262356246653036917e+00,1.262175678979692517e+00,1.261995242913283022e+00,1.261814938163696276e+00,1.261634764441774026e+00,1.261454721459307260e+00,1.261274808929031543e+00,1.261095026564624133e+00,1.260915374080698204e+00,1.260735851192801515e+00,1.260556457617410198e+00,1.260377193071926083e+00,1.260198057274671823e+00,1.260019049944888891e+00,1.259840170802731141e+00,1.259661419569262808e+00,1.259482795966454072e+00,1.259304299717177722e+00,1.259125930545205385e+00,1.258947688175202195e+00,1.258769572332725906e+00,1.258591582744220005e+00,1.258413719137013498e+00,1.258235981239314238e+00,1.258058368780206271e+00,1.257880881489646940e+00,1.257703519098462897e+00,1.257526281338345875e+00,1.257349167941850032e+00,1.257172178642387506e+00,1.256995313174225526e+00,1.256818571272483087e+00,1.256641952673127172e+00,1.256465457112968087e+00,1.256289084329658357e+00,1.256112834061688277e+00,1.255936706048380813e+00,1.255760700029890931e+00,1.255584815747201155e+00,1.255409052942116910e+00,1.255233411357264961e+00,1.255057890736090087e+00,1.254882490822849528e+00,1.254707211362612540e+00,1.254532052101255513e+00,1.254357012785459524e+00,1.254182093162705458e+00,1.254007292981272892e+00,1.253832611990235879e+00,1.253658049939459618e+00,1.253483606579597343e+00,1.253309281662088104e+00,1.253135074939152105e+00,1.252960986163788704e+00,1.252787015089772416e+00,1.252613161471651138e+00,1.252439425064741707e+00,1.252265805625128126e+00,1.252092302909656674e+00,1.251918916675935023e+00,1.251745646682327795e+00,1.251572492687954119e+00,1.251399454452684745e+00,1.251226531737138048e+00,1.251053724302678916e+00,1.250881031911414532e+00,1.250708454326191488e+00,1.250535991310593120e+00,1.250363642628936844e+00,1.250191408046271269e+00,1.250019287328371975e+00,1.249847280241741521e+00,1.249675386553603884e+00,1.249503606031902692e+00,1.249331938445299217e+00,1.249160383563167498e+00,1.248988941155594112e+00,1.248817610993373517e+00,1.248646392848005826e+00,1.248475286491694591e+00,1.248304291697344137e+00,1.248133408238554898e+00,1.247962635889624305e+00,1.247791974425540573e+00,1.247621423621981585e+00,1.247450983255313561e+00,1.247280653102585291e+00,1.247110432941528346e+00,1.246940322550553315e+00,1.246770321708747131e+00,1.246600430195870413e+00,1.246430647792355906e+00,1.246260974279304268e+00,1.246091409438483399e+00,1.245921953052324005e+00,1.245752604903918481e+00,1.245583364777017810e+00,1.245414232456029557e+00,1.245245207726014325e+00,1.245076290372685079e+00,1.244907480182402049e+00,1.244738776942173386e+00,1.244570180439650953e+00,
        1.244401690463126986e+00,1.244233306801534100e+00,1.244065029244441067e+00,1.243896857582051041e+00,1.243728791605199113e+00,1.243560831105350539e+00,1.243392975874597184e+00,1.243225225705656189e+00,1.243057580391866868e+00,1.242890039727190032e+00,1.242722603506202450e+00,1.242555271524099059e+00,1.242388043576685863e+00,1.242220919460381712e+00,1.242053898972214077e+00,1.241886981909816168e+00,1.241720168071426267e+00,1.241553457255885284e+00,1.241386849262633207e+00,1.241220343891708433e+00,1.241053940943744882e+00,1.240887640219969557e+00,1.240721441522200985e+00,1.240555344652847003e+00,1.240389349414901421e+00,1.240223455611944026e+00,1.240057663048136805e+00,1.239891971528221948e+00,1.239726380857520516e+00,1.239560890841929552e+00,1.239395501287920753e+00,1.239230212002537801e+00,1.239065022793393922e+00,1.238899933468672110e+00,1.238734943837118907e+00,1.238570053708046848e+00,1.238405262891329572e+00,1.238240571197400275e+00,1.238075978437250813e+00,1.237911484422428599e+00,1.237747088965035047e+00,1.237582791877722688e+00,1.237418592973695830e+00,1.237254492066705236e+00,1.237090488971047897e+00,1.236926583501565480e+00,1.236762775473641662e+00,1.236599064703199913e+00,1.236435451006702602e+00,1.236271934201148337e+00,1.236108514104069744e+00,1.235945190533532800e+00,1.235781963308134168e+00,1.235618832246998977e+00,1.235455797169779268e+00,1.235292857896653329e+00,1.235130014248321695e+00,1.234967266046006484e+00,1.234804613111449845e+00,1.234642055266911731e+00,1.234479592335167686e+00,1.234317224139508617e+00,1.234154950503735915e+00,1.233992771252163889e+00,1.233830686209614225e+00,1.233668695201415977e+00,1.233506798053403797e+00,1.233344994591915267e+00,1.233183284643791344e+00,1.233021668036371032e+00,1.232860144597493823e+00,1.232698714155494368e+00,1.232537376539202700e+00,1.232376131577942457e+00,1.232214979101528218e+00,1.232053918940264836e+00,1.231892950924945218e+00,1.231732074886848993e+00,1.231571290657740070e+00,1.231410598069866191e+00,1.231249996955956716e+00,1.231089487149219952e+00,1.230929068483343380e+00,1.230768740792491212e+00,1.230608503911301277e+00,1.230448357674886584e+00,1.230288301918830873e+00,1.230128336479188400e+00,1.229968461192482154e+00,1.229808675895701864e+00,1.229648980426303106e+00,1.229489374622204423e+00,1.229329858321787983e+00,1.229170431363895810e+00,1.229011093587829340e+00,1.228851844833347418e+00,1.228692684940665636e+00,1.228533613750454112e+00,1.228374631103835490e+00,1.228215736842384720e+00,1.228056930808125946e+00,1.227898212843532511e+00,1.227739582791525397e+00,1.227581040495469455e+00,1.227422585799175403e+00,1.227264218546895602e+00,1.227105938583324063e+00,
        1.226947745753593999e+00,1.226789639903278051e+00,1.226631620878383622e+00,1.226473688525355765e+00,1.226315842691071634e+00,1.226158083222842476e+00,1.226000409968409199e+00,1.225842822775943253e+00,1.225685321494044189e+00,1.225527905971738329e+00,1.225370576058477434e+00,1.225213331604137590e+00,1.225056172459017434e+00,1.224899098473836601e+00,1.224742109499735054e+00,1.224585205388271758e+00,1.224428385991422008e+00,1.224271651161577434e+00,1.224115000751544002e+00,1.223958434614541124e+00,1.223801952604199661e+00,1.223645554574561922e+00,1.223489240380077892e+00,1.223333009875607003e+00,1.223176862916413921e+00,1.223020799358169208e+00,1.222864819056947772e+00,1.222708921869226417e+00,1.222553107651883630e+00,1.222397376262198243e+00,1.222241727557846991e+00,1.222086161396905624e+00,1.221930677637844687e+00,1.221775276139530630e+00,1.221619956761223147e+00,1.221464719362574947e+00,1.221309563803629539e+00,1.221154489944820565e+00,1.220999497646970466e+00,1.220844586771289153e+00,1.220689757179373336e+00,1.220535008733203419e+00,1.220380341295145943e+00,1.220225754727948031e+00,1.220071248894739391e+00,1.219916823659029426e+00,1.219762478884707235e+00,1.219608214436040061e+00,1.219454030177670179e+00,1.219299925974617116e+00,1.219145901692273881e+00,1.218991957196407405e+00,1.218838092353155877e+00,1.218684307029028746e+00,1.218530601090904719e+00,1.218376974406031987e+00,1.218223426842025559e+00,1.218069958266866815e+00,1.217916568548901735e+00,1.217763257556841783e+00,1.217610025159759690e+00,1.217456871227091009e+00,1.217303795628631446e+00,1.217150798234536424e+00,1.216997878915319964e+00,1.216845037541853358e+00,1.216692273985364281e+00,1.216539588117435233e+00,1.216386979810002433e+00,1.216234448935356260e+00,1.216081995366138369e+00,1.215929618975341020e+00,1.215777319636306419e+00,1.215625097222725826e+00,1.215472951608637553e+00,1.215320882668426972e+00,1.215168890276824509e+00,1.215016974308905873e+00,1.214865134640089828e+00,1.214713371146136867e+00,1.214561683703150319e+00,1.214410072187572132e+00,1.214258536476185091e+00,1.214107076446109490e+00,1.213955691974802908e+00,1.213804382940059767e+00,1.213653149220009109e+00,1.213501990693114596e+00,1.213350907238172738e+00,1.213199898734313109e+00,1.213048965060995688e+00,1.212898106098011741e+00,1.212747321725480276e+00,1.212596611823849813e+00,1.212445976273896830e+00,1.212295414956722439e+00,1.212144927753754153e+00,1.211994514546744117e+00,1.211844175217767328e+00,1.211693909649221856e+00,1.211543717723827074e+00,1.211393599324622983e+00,1.211243554334969108e+00,1.211093582638544497e+00,1.210943684119344610e+00,1.210793858661683542e+00,1.210644106150189359e+00,
        1.210494426469806539e+00,1.210344819505793534e+00,1.210195285143721655e+00,1.210045823269473964e+00,1.209896433769245938e+00,1.209747116529542810e+00,1.209597871437180006e+00,1.209448698379280041e+00,1.209299597243275182e+00,1.209150567916903007e+00,1.209001610288208406e+00,1.208852724245539356e+00,1.208703909677550259e+00,1.208555166473197717e+00,1.208406494521740759e+00,1.208257893712739506e+00,1.208109363936056058e+00,1.207960905081851610e+00,1.207812517040586009e+00,1.207664199703017749e+00,1.207515952960202421e+00,1.207367776703492046e+00,1.207219670824534630e+00,1.207071635215271499e+00,1.206923669767939300e+00,1.206775774375067334e+00,1.206627948929476890e+00,1.206480193324280359e+00,1.206332507452881675e+00,1.206184891208973209e+00,1.206037344486537322e+00,1.205889867179843922e+00,1.205742459183450022e+00,1.205595120392199737e+00,1.205447850701221846e+00,1.205300650005931340e+00,1.205153518202025653e+00,1.205006455185486214e+00,1.204859460852577779e+00,1.204712535099845105e+00,1.204565677824114722e+00,1.204418888922493380e+00,1.204272168292366496e+00,1.204125515831398374e+00,1.203978931437531097e+00,1.203832415008983858e+00,1.203685966444250743e+00,1.203539585642103615e+00,1.203393272501586564e+00,1.203247026922019014e+00,1.203100848802993506e+00,1.202954738044373473e+00,1.202808694546295909e+00,1.202662718209167814e+00,1.202516808933665748e+00,1.202370966620736725e+00,1.202225191171595320e+00,1.202079482487724782e+00,1.201933840470874593e+00,1.201788265023062241e+00,1.201642756046569005e+00,1.201497313443942394e+00,1.201351937117993263e+00,1.201206626971797586e+00,1.201061382908691577e+00,1.200916204832275458e+00,1.200771092646410354e+00,1.200626046255217405e+00,1.200481065563078875e+00,1.200336150474634822e+00,1.200191300894784430e+00,1.200046516728685564e+00,1.199901797881751442e+00,1.199757144259653074e+00,1.199612555768316824e+00,1.199468032313923738e+00,1.199323573802909770e+00,1.199179180141963341e+00,1.199034851238027777e+00,1.198890586998296648e+00,1.198746387330216656e+00,1.198602252141484525e+00,1.198458181340047890e+00,1.198314174834103074e+00,1.198170232532095980e+00,1.198026354342720978e+00,1.197882540174919352e+00,1.197738789937879744e+00,1.197595103541036821e+00,1.197451480894070830e+00,1.197307921906907158e+00,1.197164426489715439e+00,1.197020994552909112e+00,1.196877626007143869e+00,1.196734320763318760e+00,1.196591078732574198e+00,1.196447899826291073e+00,1.196304783956091411e+00,1.196161731033837050e+00,1.196018740971628747e+00,1.195875813681805289e+00,1.195732949076943941e+00,1.195590147069859777e+00,1.195447407573603682e+00,1.195304730501462576e+00,1.195162115766958966e+00,1.195019563283850506e+00,
        1.194877072966128440e+00,1.194734644728018269e+00,1.194592278483977754e+00,1.194449974148697136e+00,1.194307731637098913e+00,1.194165550864336289e+00,1.194023431745792951e+00,1.193881374197082623e+00,1.193739378134048623e+00,1.193597443472762087e+00,1.193455570129523968e+00,1.193313758020860593e+00,1.193172007063527440e+00,1.193030317174504917e+00,1.192888688270999475e+00,1.192747120270443162e+00,1.192605613090492733e+00,1.192464166649028101e+00,1.192322780864153442e+00,1.192181455654195643e+00,1.192040190937704081e+00,1.191898986633449287e+00,1.191757842660423838e+00,1.191616758937840137e+00,1.191475735385130630e+00,1.191334771921948477e+00,1.191193868468163997e+00,1.191053024943867111e+00,1.190912241269365346e+00,1.190771517365182497e+00,1.190630853152060631e+00,1.190490248550956975e+00,1.190349703483044363e+00,1.190209217869711011e+00,1.190068791632559186e+00,1.189928424693405873e+00,1.189788116974280552e+00,1.189647868397426533e+00,1.189507678885298292e+00,1.189367548360563243e+00,1.189227476746099521e+00,1.189087463964995539e+00,1.188947509940551539e+00,1.188807614596275375e+00,1.188667777855884733e+00,1.188527999643306909e+00,1.188388279882676146e+00,1.188248618498334963e+00,1.188109015414831937e+00,1.187969470556923257e+00,1.187829983849570947e+00,1.187690555217941535e+00,1.187551184587408049e+00,1.187411871883546688e+00,1.187272617032138156e+00,1.187133419959166769e+00,1.186994280590819795e+00,1.186855198853486115e+00,1.186716174673757562e+00,1.186577207978426696e+00,1.186438298694487692e+00,1.186299446749134567e+00,1.186160652069761401e+00,1.186021914583961667e+00,1.185883234219528459e+00,1.185744610904451823e+00,1.185606044566921646e+00,1.185467535135324102e+00,1.185329082538241874e+00,1.185190686704455931e+00,1.185052347562941533e+00,1.184914065042870224e+00,1.184775839073608728e+00,1.184637669584717834e+00,1.184499556505952844e+00,1.184361499767262238e+00,1.184223499298788562e+00,1.184085555030865988e+00,1.183947666894021422e+00,1.183809834818973394e+00,1.183672058736631838e+00,1.183534338578097200e+00,1.183396674274659999e+00,1.183259065757801265e+00,1.183121512959191435e+00,1.182984015810689238e+00,1.182846574244342364e+00,1.182709188192386351e+00,1.182571857587244368e+00,1.182434582361526765e+00,1.182297362448030409e+00,1.182160197779738242e+00,1.182023088289819057e+00,1.181886033911627054e+00,1.181749034578701396e+00,1.181612090224765765e+00,1.181475200783726587e+00,1.181338366189675249e+00,1.181201586376885437e+00,1.181064861279814027e+00,1.180928190833099745e+00,1.180791574971562730e+00,1.180655013630205197e+00,1.180518506744209217e+00,1.180382054248938495e+00,1.180245656079935479e+00,1.180109312172923586e+00,1.179973022463804089e+00,1.179836786888657674e+00,1.179700605383742884e+00,1.179564477885497009e+00,1.179428404330533198e+00,1.179292384655643566e+00,1.179156418797794981e+00,1.179020506694131942e+00,1.178884648281973258e+00,1.178748843498814924e+00,1.178613092282325470e+00,1.178477394570349723e+00,1.178341750300906599e+00,1.178206159412187315e+00,1.178070621842557619e+00,1.177935137530555343e+00,1.177799706414891068e+00,1.177664328434447238e+00,1.177529003528278384e+00,1.177393731635609120e+00,1.177258512695836368e+00,1.177123346648526026e+00,1.176988233433414965e+00,1.176853172990408591e+00,1.176718165259582616e+00,1.176583210181181505e+00,1.176448307695616702e+00,1.176313457743469293e+00,1.176178660265487119e+00,1.176043915202585000e+00,1.175909222495845174e+00,1.175774582086516196e+00,1.175639993916012038e+00,1.175505457925912767e+00,1.175370974057964091e+00,1.175236542254075145e+00,1.175102162456321375e+00,1.174967834606940986e+00,1.174833558648335829e+00,1.174699334523071848e+00,1.174565162173877964e+00,1.174431041543644749e+00,1.174296972575425535e+00,1.174162955212434856e+00,1.174028989398049339e+00,1.173895075075806815e+00,1.173761212189404546e+00,1.173627400682700994e+00,1.173493640499714941e+00,1.173359931584623261e+00,1.173226273881762927e+00,1.173092667335629891e+00,1.172959111890877981e+00,1.172825607492319122e+00,1.172692154084923111e+00,1.172558751613816952e+00,1.172425400024284414e+00,1.172292099261766252e+00,1.172158849271859093e+00,1.172025650000315222e+00,1.171892501393042574e+00,1.171759403396105181e+00,1.171626355955719623e+00,1.171493359018259017e+00,1.171360412530249473e+00,1.171227516438370975e+00,1.171094670689456718e+00,1.170961875230492888e+00,1.170829130008618657e+00,1.170696434971124855e+00,1.170563790065454857e+00,1.170431195239203026e+00,1.170298650440115162e+00,1.170166155616087833e+00,1.170033710715168374e+00,1.169901315685554000e+00,1.169768970475592029e+00,1.169636675033778550e+00,1.169504429308759530e+00,1.169372233249329707e+00,1.169240086804431478e+00,1.169107989923155788e+00,1.168975942554741687e+00,1.168843944648574995e+00,1.168711996154189192e+00,1.168580097021263864e+00,1.168448247199625367e+00,1.168316446639245942e+00,1.168184695290243935e+00,1.168052993102882242e+00,1.167921340027569421e+00,1.167789736014859026e+00,1.167658181015448049e+00,1.167526674980178258e+00,1.167395217860034418e+00,1.167263809606146063e+00,1.167132450169784175e+00,1.167001139502362950e+00,1.166869877555440027e+00,1.166738664280713378e+00,1.166607499630023970e+00,1.166476383555353324e+00,1.166345316008825073e+00,1.166214296942702511e+00,1.166083326309390156e+00,1.165952404061432413e+00,1.165821530151512908e+00,1.165690704532455380e+00,1.165559927157223008e+00,1.165429197978917086e+00,1.165298516950777463e+00,1.165167884026182543e+00,1.165037299158648398e+00,1.164906762301828991e+00,1.164776273409514618e+00,1.164645832435633688e+00,1.164515439334250502e+00,1.164385094059565473e+00,1.164254796565915573e+00,1.164124546807773664e+00,1.163994344739746500e+00,1.163864190316577618e+00,1.163734083493144222e+00,1.163604024224458522e+00,1.163474012465666396e+00,1.163344048172047618e+00,1.163214131299015852e+00,1.163084261802117547e+00,1.162954439637032156e+00,1.162824664759571913e+00,1.162694937125680950e+00,1.162565256691435733e+00,1.162435623413043961e+00,1.162306037246845003e+00,1.162176498149310344e+00,1.162047006077039812e+00,1.161917560986766018e+00,1.161788162835350136e+00,1.161658811579784123e+00,1.161529507177189613e+00,1.161400249584817024e+00,1.161271038760045116e+00,1.161141874660383433e+00,1.161012757243467641e+00,1.160883686467062637e+00,1.160754662289061212e+00,1.160625684667482949e+00,1.160496753560475103e+00,1.160367868926312163e+00,1.160239030723394515e+00,1.160110238910249114e+00,1.159981493445529255e+00,1.159852794288014133e+00,1.159724141396607733e+00,1.159595534730339494e+00,1.159466974248363647e+00,1.159338459909959429e+00,1.159209991674530427e+00,1.159081569501603237e+00,1.158953193350829247e+00,1.158824863181983078e+00,1.158696578954962364e+00,1.158568340629787530e+00,1.158440148166601569e+00,1.158312001525670043e+00,1.158183900667380861e+00,1.158055845552242724e+00,1.157927836140886457e+00,1.157799872394064122e+00,1.157671954272648795e+00,1.157544081737633235e+00,1.157416254750131657e+00,1.157288473271377516e+00,1.157160737262725503e+00,1.157033046685647770e+00,1.156905401501737041e+00,1.156777801672705053e+00,1.156650247160381451e+00,1.156522737926714450e+00,1.156395273933771728e+00,1.156267855143736867e+00,1.156140481518912022e+00,1.156013153021717033e+00,1.155885869614688088e+00,1.155758631260478397e+00,1.155631437921857962e+00,1.155504289561712028e+00,1.155377186143043300e+00,1.155250127628969281e+00,1.155123113982722494e+00,1.154996145167651367e+00,1.154869221147219127e+00,1.154742341885003354e+00,1.154615507344696201e+00,1.154488717490103289e+00,1.154361972285145255e+00,1.154235271693855092e+00,1.154108615680379701e+00,1.153982004208978784e+00,1.153855437244025062e+00,1.153728914750003387e+00,1.153602436691510746e+00,1.153476003033257369e+00,1.153349613740062951e+00,1.153223268776860433e+00,1.153096968108693554e+00,1.152970711700716411e+00,1.152844499518194121e+00,1.152718331526502160e+00,1.152592207691126136e+00,1.152466127977662236e+00,1.152340092351814782e+00,1.152214100779398454e+00,1.152088153226337175e+00,1.151962249658662785e+00,1.151836390042516811e+00,1.151710574344148252e+00,1.151584802529914908e+00,1.151459074566281382e+00,1.151333390419821301e+00,1.151207750057214430e+00,1.151082153445247336e+00,1.150956600550814946e+00,1.150831091340916990e+00,1.150705625782660890e+00,1.150580203843259097e+00,1.150454825490029975e+00,1.150329490690398471e+00,1.150204199411893224e+00,1.150078951622148793e+00,1.149953747288904982e+00,1.149828586380005291e+00,1.149703468863397138e+00,1.149578394707133411e+00,1.149453363879369805e+00,1.149328376348365710e+00,1.149203432082483545e+00,1.149078531050189422e+00,1.148953673220051375e+00,1.148828858560740906e+00,1.148704087041031219e+00,1.148579358629797209e+00,1.148454673296017026e+00,1.148330031008768515e+00,1.148205431737232107e+00,1.148080875450689486e+00,1.147956362118521589e+00,1.147831891710211716e+00,1.147707464195341975e+00,1.147583079543596174e+00,1.147458737724756261e+00,1.147334438708705218e+00,1.147210182465424610e+00,1.147085968964995706e+00,1.146961798177597691e+00,1.146837670073509230e+00,
        1.146713584623106907e+00,1.146589541796866341e+00,1.146465541565359514e+00,1.146341583899257444e+00,1.146217668769327735e+00,1.146093796146436361e+00,1.145969966001544771e+00,1.145846178305712115e+00,1.145722433030093246e+00,1.145598730145940491e+00,1.145475069624600994e+00,1.145351451437518042e+00,1.145227875556231067e+00,1.145104341952373428e+00,1.144980850597675071e+00,1.144857401463959423e+00,1.144733994523146059e+00,1.144610629747246922e+00,1.144487307108370100e+00,1.144364026578715832e+00,1.144240788130579389e+00,1.144117591736348638e+00,1.143994437368504924e+00,1.143871324999621963e+00,1.143748254602367398e+00,1.143625226149500129e+00,1.143502239613871874e+00,1.143379294968426496e+00,1.143256392186199788e+00,1.143133531240318135e+00,1.143010712104000071e+00,1.142887934750555390e+00,1.142765199153384037e+00,1.142642505285976995e+00,1.142519853121915618e+00,1.142397242634872079e+00,1.142274673798606699e+00,1.142152146586971506e+00,1.142029660973906902e+00,1.141907216933442770e+00,1.141784814439698481e+00,1.141662453466881111e+00,1.141540133989287664e+00,1.141417855981302187e+00,1.141295619417397988e+00,1.141173424272135417e+00,1.141051270520163419e+00,1.140929158136217536e+00,1.140807087095120353e+00,1.140685057371782163e+00,1.140563068941199854e+00,1.140441121778457134e+00,1.140319215858722757e+00,1.140197351157253181e+00,1.140075527649389686e+00,1.139953745310559707e+00,1.139832004116275721e+00,1.139710304042135469e+00,1.139588645063821737e+00,1.139467027157102574e+00,1.139345450297829743e+00,1.139223914461939602e+00,1.139102419625452889e+00,1.138980965764473607e+00,1.138859552855189694e+00,1.138738180873872796e+00,1.138616849796877606e+00,1.138495559600641416e+00,1.138374310261684563e+00,1.138253101756609986e+00,1.138131934062102779e+00,1.138010807154930193e+00,1.137889721011941191e+00,1.137768675610067115e+00,1.137647670926319465e+00,1.137526706937792120e+00,1.137405783621659339e+00,1.137284900955176203e+00,1.137164058915678844e+00,1.137043257480582659e+00,1.136922496627384538e+00,1.136801776333660197e+00,1.136681096577066175e+00,1.136560457335336727e+00,1.136439858586287155e+00,1.136319300307810920e+00,1.136198782477880087e+00,1.136078305074545325e+00,1.135957868075937016e+00,1.135837471460262149e+00,1.135717115205806538e+00,1.135596799290932823e+00,1.135476523694082918e+00,1.135356288393773561e+00,1.135236093368601207e+00,1.135115938597236696e+00,1.134995824058429914e+00,1.134875749731005357e+00,1.134755715593865011e+00,1.134635721625986138e+00,1.134515767806421715e+00,1.134395854114301327e+00,1.134275980528828276e+00,1.134156147029282913e+00,1.134036353595019087e+00,1.133916600205466585e+00,1.133796886840129137e+00,1.133677213478583523e+00,1.133557580100483131e+00,1.133437986685553511e+00,1.133318433213595045e+00,1.133198919664480497e+00,1.133079446018155689e+00,1.132960012254641491e+00,1.132840618354029605e+00,1.132721264296485009e+00,1.132601950062245511e+00,1.132482675631620639e+00,1.132363440984991865e+00,1.132244246102813046e+00,1.132125090965609759e+00,1.132005975553977306e+00,1.131886899848584038e+00,1.131767863830168919e+00,1.131648867479540632e+00,1.131529910777579140e+00,1.131410993705235235e+00,1.131292116243528767e+00,1.131173278373550417e+00,1.131054480076460145e+00,1.130935721333486965e+00,1.130817002125930726e+00,1.130698322435158776e+00,1.130579682242607742e+00,1.130461081529783751e+00,1.130342520278260210e+00,1.130223998469679358e+00,1.130105516085751827e+00,1.129987073108255524e+00,1.129868669519036306e+00,1.129750305300007751e+00,1.129631980433150051e+00,1.129513694900510901e+00,1.129395448684204162e+00,1.129277241766411199e+00,1.129159074129379770e+00,1.129040945755423131e+00,1.128922856626920490e+00,1.128804806726318333e+00,1.128686796036126649e+00,1.128568824538922266e+00,1.128450892217347068e+00,1.128332999054107333e+00,1.128215145031974620e+00,1.128097330133784881e+00,1.127979554342438240e+00,1.127861817640900099e+00,1.127744120012198037e+00,1.127626461439424910e+00,1.127508841905736414e+00,1.127391261394351973e+00,1.127273719888554071e+00,1.127156217371687585e+00,1.127038753827161122e+00,1.126921329238445235e+00,1.126803943589072876e+00,1.126686596862639833e+00,1.126569289042802957e+00,1.126452020113281050e+00,1.126334790057855084e+00,1.126217598860366431e+00,1.126100446504718411e+00,1.125983332974874962e+00,1.125866258254861085e+00,1.125749222328761956e+00,1.125632225180722923e+00,1.125515266794950175e+00,1.125398347155709411e+00,1.125281466247326501e+00,1.125164624054186158e+00,1.125047820560733269e+00,1.124931055751471121e+00,1.124814329610962504e+00,1.124697642123828834e+00,1.124580993274749918e+00,1.124464383048464411e+00,1.124347811429768695e+00,1.124231278403516887e+00,1.124114783954621721e+00,1.123998328068052777e+00,1.123881910728837807e+00,1.123765531922060745e+00,1.123649191632863031e+00,1.123532889846442728e+00,1.123416626548055186e+00,1.123300401723011488e+00,1.123184215356678672e+00,1.123068067434480399e+00,1.122951957941896284e+00,1.122835886864460120e+00,1.122719854187763655e+00,1.122603859897451040e+00,1.122487903979223711e+00,1.122371986418836398e+00,1.122256107202099784e+00,1.122140266314877843e+00,1.122024463743089617e+00,1.121908699472707660e+00,1.121792973489758705e+00,1.121677285780322553e+00,1.121561636330533851e+00,1.121446025126579205e+00,1.121330452154698731e+00,1.121214917401185396e+00,1.121099420852385453e+00,1.120983962494696229e+00,1.120868542314569005e+00,1.120753160298506135e+00,1.120637816433062373e+00,1.120522510704843988e+00,1.120407243100508987e+00,1.120292013606766446e+00,1.120176822210376510e+00,1.120061668898151286e+00,1.119946553656951727e+00,1.119831476473691634e+00,1.119716437335333437e+00,1.119601436228890634e+00,1.119486473141426464e+00,1.119371548060054122e+00,1.119256660971936546e+00,1.119141811864285074e+00,1.119027000724361898e+00,1.118912227539477389e+00,1.118797492296990104e+00,1.118682794984309226e+00,1.118568135588890122e+00,1.118453514098238344e+00,1.118338930499906292e+00,1.118224384781494996e+00,1.118109876930652558e+00,1.117995406935075264e+00,1.117880974782506476e+00,1.117766580460736847e+00,1.117652223957603441e+00,1.117537905260990616e+00,1.117423624358829359e+00,1.117309381239096844e+00,1.117195175889816872e+00,1.117081008299058098e+00,1.116966878454936474e+00,1.116852786345612802e+00,1.116738731959292963e+00,1.116624715284229685e+00,1.116510736308718776e+00,1.116396795021102228e+00,1.116282891409766220e+00,1.116169025463141784e+00,1.116055197169704361e+00,1.115941406517972689e+00,1.115827653496510585e+00,1.115713938093924940e+00,1.115600260298866386e+00,1.115486620100029302e+00,1.115373017486150475e+00,1.115259452446010435e+00,1.115145924968432345e+00,1.115032435042282444e+00,1.114918982656467827e+00,1.114805567799940667e+00,1.114692190461692434e+00,1.114578850630758122e+00,1.114465548296214026e+00,1.114352283447178182e+00,1.114239056072809708e+00,1.114125866162309020e+00,1.114012713704917168e+00,1.113899598689916948e+00,1.113786521106630900e+00,1.113673480944422423e+00,1.113560478192695102e+00,1.113447512840892273e+00,1.113334584878497679e+00,1.113221694295034592e+00,1.113108841080065581e+00,1.112996025223193186e+00,1.112883246714057917e+00,1.112770505542340693e+00,1.112657801697759963e+00,1.112545135170073918e+00,1.112432505949078276e+00,1.112319914024607392e+00,1.112207359386534034e+00,1.112094842024768049e+00,1.111982361929257479e+00,1.111869919089987890e+00,1.111757513496982153e+00,1.111645145140300217e+00,1.111532814010039338e+00,1.111420520096333187e+00,1.111308263389351625e+00,1.111196043879302486e+00,1.111083861556428243e+00,1.110971716411008003e+00,1.110859608433357071e+00,1.110747537613826497e+00,1.110635503942802194e+00,1.110523507410706046e+00,1.110411548007995242e+00,1.110299625725160944e+00,1.110187740552730506e+00,1.110075892481265258e+00,1.109964081501360500e+00,1.109852307603647281e+00,1.109740570778788626e+00,1.109628871017483753e+00,1.109517208310463854e+00,1.109405582648494759e+00,1.109293994022375607e+00,1.109182442422937953e+00,1.109070927841047105e+00,1.108959450267601454e+00,1.108848009693530923e+00,1.108736606109799405e+00,1.108625239507402327e+00,1.108513909877367309e+00,1.108402617210753727e+00,1.108291361498654037e+00,1.108180142732190232e+00,1.108068960902517830e+00,1.107957816000821660e+00,1.107846708018319637e+00,1.107735636946259650e+00,1.107624602775919564e+00,1.107513605498609222e+00,1.107402645105667993e+00,1.107291721588465894e+00,1.107180834938403136e+00,1.107069985146908575e+00,1.106959172205442599e+00,1.106848396105494015e+00,1.106737656838580941e+00,1.106626954396251694e+00,1.106516288770081680e+00,1.106405659951677167e+00,1.106295067932672405e+00,1.106184512704729617e+00,1.106073994259539672e+00,1.105963512588821640e+00,1.105853067684322566e+00,1.105742659537817696e+00,1.105632288141108921e+00,1.105521953486026332e+00,1.105411655564426887e+00,1.105301394368195744e+00,1.105191169889243596e+00,1.105080982119508226e+00,1.104970831050954727e+00,1.104860716675573951e+00,1.104750638985383393e+00,1.104640597972426308e+00,1.104530593628772150e+00,1.104420625946515910e+00,1.104310694917778557e+00,1.104200800534705929e+00,1.104090942789469176e+00,1.103981121674265209e+00,1.103871337181314694e+00,1.103761589302864055e+00,1.103651878031183919e+00,1.103542203358569340e+00,1.103432565277339350e+00,1.103322963779836963e+00,1.103213398858430061e+00,1.103103870505508954e+00,1.102994378713488155e+00,
        1.102884923474805934e+00,1.102775504781922766e+00,1.102666122627323109e+00,1.102556777003514066e+00,1.102447467903024947e+00,1.102338195318408598e+00,1.102228959242239403e+00,1.102119759667113730e+00,1.102010596585651259e+00,1.101901469990492544e+00,1.101792379874300343e+00,1.101683326229758064e+00,1.101574309049572209e+00,1.101465328326468818e+00,1.101356384053196136e+00,1.101247476222522392e+00,1.101138604827237133e+00,1.101029769860150553e+00,1.100920971314093055e+00,1.100812209181915025e+00,1.100703483456487275e+00,1.100594794130700604e+00,1.100486141197465351e+00,1.100377524649711836e+00,1.100268944480388589e+00,1.100160400682464790e+00,1.100051893248928492e+00,1.099943422172786178e+00,1.099834987447063206e+00,1.099726589064803806e+00,1.099618227019070416e+00,1.099509901302943682e+00,1.099401611909523124e+00,1.099293358831925582e+00,1.099185142063285436e+00,1.099076961596755497e+00,1.098968817425505229e+00,1.098860709542722303e+00,1.098752637941610599e+00,1.098644602615392207e+00,1.098536603557304980e+00,1.098428640760603647e+00,1.098320714218560035e+00,1.098212823924461734e+00,1.098104969871612546e+00,1.097997152053333370e+00,1.097889370462959757e+00,1.097781625093843250e+00,1.097673915939351375e+00,1.097566242992866981e+00,1.097458606247787571e+00,1.097351005697526860e+00,1.097243441335512548e+00,1.097135913155187881e+00,1.097028421150009647e+00,1.096920965313450402e+00,1.096813545638996024e+00,1.096706162120147710e+00,1.096598814750419093e+00,1.096491503523338684e+00,1.096384228432448982e+00,1.096276989471305141e+00,1.096169786633476306e+00,1.096062619912544500e+00,1.095955489302105068e+00,1.095848394795766678e+00,1.095741336387149323e+00,1.095634314069887871e+00,1.095527327837627629e+00,1.095420377684026780e+00,1.095313463602756610e+00,1.095206585587499282e+00,1.095099743631948952e+00,1.094992937729812432e+00,1.094886167874806748e+00,1.094779434060661583e+00,1.094672736281116832e+00,1.094566074529923716e+00,1.094459448800845003e+00,1.094352859087654117e+00,1.094246305384134033e+00,1.094139787684079934e+00,1.094033305981295667e+00,1.093926860269596624e+00,1.093820450542807743e+00,1.093714076794763512e+00,1.093607739019308189e+00,1.093501437210296690e+00,1.093395171361591478e+00,1.093288941467066566e+00,1.093182747520603071e+00,1.093076589516092101e+00,1.092970467447434313e+00,1.092864381308537247e+00,1.092758331093318658e+00,1.092652316795703848e+00,1.092546338409626561e+00,1.092440395929028529e+00,1.092334489347860371e+00,1.092228618660079364e+00,1.092122783859650337e+00,1.092016984940546998e+00,1.091911221896749051e+00,1.091805494722244418e+00,1.091699803411027903e+00,1.091594147957100747e+00,1.091488528354471299e+00,1.091382944597155014e+00,1.091277396679172895e+00,1.091171884594553498e+00,1.091066408337330484e+00,1.090960967901544398e+00,1.090855563281241558e+00,1.090750194470473833e+00,1.090644861463299087e+00,1.090539564253779847e+00,1.090434302835985747e+00,1.090329077203989749e+00,1.090223887351871035e+00,1.090118733273713669e+00,1.090013614963605715e+00,1.089908532415641451e+00,1.089803485623917600e+00,1.089698474582537324e+00,1.089593499285606670e+00,1.089488559727236572e+00,1.089383655901540848e+00,1.089278787802639092e+00,1.089173955424652007e+00,1.089069158761706513e+00,1.088964397807930862e+00,1.088859672557457969e+00,1.088754983004422527e+00,1.088650329142963669e+00,1.088545710967222524e+00,1.088441128471343111e+00,1.088336581649471890e+00,1.088232070495758430e+00,1.088127595004353854e+00,1.088023155169411726e+00,1.087918750985087613e+00,1.087814382445539296e+00,1.087710049544926116e+00,1.087605752277408966e+00,1.087501490637150070e+00,1.087397264618313431e+00,1.087293074215063937e+00,1.087188919421568256e+00,1.087084800231992832e+00,1.086980716640506106e+00,1.086876668641276522e+00,1.086772656228473188e+00,1.086668679396266324e+00,1.086564738138825037e+00,1.086460832450320213e+00,1.086356962324921849e+00,1.086253127756799497e+00,1.086149328740123599e+00,1.086045565269063484e+00,1.085941837337788263e+00,1.085838144940465932e+00,1.085734488071264492e+00,1.085630866724350607e+00,1.085527280893889834e+00,1.085423730574047507e+00,1.085320215758986295e+00,1.085216736442868424e+00,1.085113292619854120e+00,1.085009884284102055e+00,1.084906511429770015e+00,1.084803174051012231e+00,1.084699872141982491e+00,1.084596605696831029e+00,1.084493374709707192e+00,1.084390179174756552e+00,1.084287019086123349e+00,1.084183894437948048e+00,1.084080805224368893e+00,1.083977751439521464e+00,1.083874733077538233e+00,1.083771750132547895e+00,1.083668802598676484e+00,1.083565890470046034e+00,1.083463013740775693e+00,1.083360172404980615e+00,1.083257366456771731e+00,1.083154595890257310e+00,1.083051860699540070e+00,1.082949160878720063e+00,1.082846496421891791e+00,1.082743867323145537e+00,1.082641273576568031e+00,1.082538715176240451e+00,1.082436192116239537e+00,1.082333704390637141e+00,1.082231251993500010e+00,1.082128834918889781e+00,1.082026453160862767e+00,1.081924106713470390e+00,1.081821795570757638e+00,1.081719519726765055e+00,1.081617279175526747e+00,1.081515073911070823e+00,1.081412903927420066e+00,1.081310769218590595e+00,1.081208669778592979e+00,1.081106605601430459e+00,1.081004576681101170e+00,1.080902583011595253e+00,1.080800624586897518e+00,1.080698701400985451e+00,1.080596813447829208e+00,1.080494960721392506e+00,1.080393143215631735e+00,1.080291360924496180e+00,1.080189613841927354e+00,1.080087901961860108e+00,1.079986225278221079e+00,1.079884583784929131e+00,1.079782977475895800e+00,1.079681406345023964e+00,1.079579870386209617e+00,1.079478369593339204e+00,1.079376903960292067e+00,1.079275473480938885e+00,1.079174078149141236e+00,1.079072717958752925e+00,1.078971392903618431e+00,1.078870102977574019e+00,1.078768848174446182e+00,1.078667628488053420e+00,1.078566443912204242e+00,1.078465294440698274e+00,1.078364180067326261e+00,1.078263100785867623e+00,1.078162056590094231e+00,1.078061047473767964e+00,1.077960073430639820e+00,1.077859134454451695e+00,1.077758230538935269e+00,1.077657361677812231e+00,1.077556527864794056e+00,1.077455729093581560e+00,1.077354965357865124e+00,1.077254236651325581e+00,1.077153542967631328e+00,1.077052884300441882e+00,1.076952260643404546e+00,1.076851671990155968e+00,1.076751118334322355e+00,1.076650599669517705e+00,1.076550115989345580e+00,1.076449667287397993e+00,1.076349253557255192e+00,1.076248874792485655e+00,1.076148530986647200e+00,1.076048222133284549e+00,1.075947948225931095e+00,1.075847709258109131e+00,1.075747505223327183e+00,1.075647336115082675e+00,1.075547201926860374e+00,1.075447102652132614e+00,1.075347038284359291e+00,1.075247008816988092e+00,1.075147014243453381e+00,1.075047054557177306e+00,1.074947129751568475e+00,1.074847239820022615e+00,1.074747384755923241e+00,1.074647564552639212e+00,1.074547779203527620e+00,1.074448028701931568e+00,1.074348313041179503e+00,1.074248632214588772e+00,1.074148986215460289e+00,1.074049375037083198e+00,1.073949798672732436e+00,1.073850257115667839e+00,1.073750750359136585e+00,1.073651278396370978e+00,1.073551841220589109e+00,1.073452438824995081e+00,1.073353071202777897e+00,1.073253738347113018e+00,1.073154440251160580e+00,1.073055176908065622e+00,1.072955948310959418e+00,1.072856754452957251e+00,1.072757595327160862e+00,1.072658470926655561e+00,1.072559381244512000e+00,1.072460326273785736e+00,1.072361306007516335e+00,1.072262320438728711e+00,1.072163369560431789e+00,1.072064453365618730e+00,1.071965571847267595e+00,1.071866724998340015e+00,1.071767912811782297e+00,1.071669135280524099e+00,1.071570392397479754e+00,1.071471684155546500e+00,1.071373010547606253e+00,1.071274371566524497e+00,1.071175767205149842e+00,1.071077197456314245e+00,1.070978662312833674e+00,1.070880161767507666e+00,1.070781695813117551e+00,1.070683264442429117e+00,1.070584867648191496e+00,1.070486505423135837e+00,1.070388177759976189e+00,1.070289884651410173e+00,1.070191626090117420e+00,1.070093402068760691e+00,1.069995212579984978e+00,1.069897057616418623e+00,1.069798937170670872e+00,1.069700851235334982e+00,1.069602799802984450e+00,1.069504782866177228e+00,1.069406800417450842e+00,1.069308852449327274e+00,1.069210938954308965e+00,1.069113059924880371e+00,1.069015215353508408e+00,1.068917405232640228e+00,1.068819629554706108e+00,1.068721888312116564e+00,1.068624181497265235e+00,1.068526509102525113e+00,1.068428871120251866e+00,1.068331267542781848e+00,1.068233698362432760e+00,1.068136163571502983e+00,1.068038663162272472e+00,1.067941197127001196e+00,1.067843765457931138e+00,1.067746368147283853e+00,1.067649005187262246e+00,1.067551676570049901e+00,1.067454382287809977e+00,1.067357122332687647e+00,1.067259896696807209e+00,1.067162705372273424e+00,1.067065548351171955e+00,1.066968425625567818e+00,1.066871337187506485e+00,1.066774283029013670e+00,1.066677263142093990e+00,1.066580277518734077e+00,1.066483326150897692e+00,1.066386409030530169e+00,1.066289526149556188e+00,1.066192677499878894e+00,1.066095863073382333e+00,1.065999082861929459e+00,1.065902336857362354e+00,1.065805625051503114e+00,1.065708947436152076e+00,1.065612304003089594e+00,1.065515694744074260e+00,1.065419119650844682e+00,1.065322578715117707e+00,1.065226071928589091e+00,1.065129599282934381e+00,1.065033160769806253e+00,1.064936756380837179e+00,1.064840386107638537e+00,1.064744049941799053e+00,1.064647747874887029e+00,1.064551479898448560e+00,1.064455246004008204e+00,1.064359046183070090e+00,1.064262880427113478e+00,1.064166748727599421e+00,1.064070651075964546e+00,1.063974587463624610e+00,1.063878557881973386e+00,1.063782562322381997e+00,1.063686600776199587e+00,1.063590673234753092e+00,1.063494779689347913e+00,1.063398920131266356e+00,1.063303094551768302e+00,1.063207302942091204e+00,1.063111545293450089e+00,1.063015821597037780e+00,1.062920131844024008e+00,1.062824476025556075e+00,1.062728854132758194e+00,1.062633266156731926e+00,1.062537712088556408e+00,1.062442191919286572e+00,1.062346705639956257e+00,1.062251253241574656e+00,1.062155834715128089e+00,
        1.062060450051580673e+00,1.061965099241872101e+00,1.061869782276920304e+00,1.061774499147617679e+00,1.061679249844835082e+00,1.061584034359418949e+00,1.061488852682193285e+00,1.061393704803957228e+00,1.061298590715487267e+00,1.061203510407535688e+00,1.061108463870830798e+00,1.061013451096077809e+00,1.060918472073958396e+00,1.060823526795128924e+00,1.060728615250223550e+00,1.060633737429850898e+00,1.060538893324596943e+00,1.060444082925021680e+00,1.060349306221663346e+00,1.060254563205034195e+00,1.060159853865623170e+00,1.060065178193894120e+00,1.059970536180286915e+00,1.059875927815217445e+00,1.059781353089076505e+00,1.059686811992231359e+00,1.059592304515023065e+00,1.059497830647769590e+00,1.059403390380763366e+00,1.059308983704272844e+00,1.059214610608541163e+00,1.059120271083787035e+00,1.059025965120203860e+00,1.058931692707960615e+00,1.058837453837200959e+00,1.058743248498044354e+00,1.058649076680583612e+00,1.058554938374888232e+00,1.058460833571001958e+00,1.058366762258943217e+00,1.058272724428704459e+00,1.058178720070254597e+00,1.058084749173536121e+00,1.057990811728465763e+00,1.057896907724936275e+00,1.057803037152814207e+00,1.057709200001939465e+00,1.057615396262128860e+00,1.057521625923171671e+00,1.057427888974832308e+00,1.057334185406849203e+00,1.057240515208935694e+00,1.057146878370779142e+00,1.057053274882040261e+00,1.056959704732355565e+00,1.056866167911333809e+00,1.056772664408559548e+00,1.056679194213590689e+00,1.056585757315959384e+00,1.056492353705170695e+00,1.056398983370704814e+00,1.056305646302015955e+00,1.056212342488531020e+00,1.056119071919651597e+00,1.056025834584753076e+00,1.055932630473184197e+00,1.055839459574267281e+00,1.055746321877299332e+00,1.055653217371549379e+00,1.055560146046261583e+00,1.055467107890652789e+00,1.055374102893913424e+00,1.055281131045207710e+00,1.055188192333672781e+00,1.055095286748419348e+00,1.055002414278532807e+00,1.054909574913069248e+00,1.054816768641060332e+00,1.054723995451509966e+00,1.054631255333395856e+00,1.054538548275668397e+00,1.054445874267250893e+00,1.054353233297040671e+00,1.054260625353907299e+00,1.054168050426694148e+00,1.054075508504217051e+00,1.053982999575264756e+00,1.053890523628599585e+00,1.053798080652956326e+00,1.053705670637042457e+00,1.053613293569539033e+00,1.053520949439099796e+00,1.053428638234350290e+00,1.053336359943890299e+00,1.053244114556290967e+00,1.053151902060097234e+00,1.053059722443826285e+00,1.052967575695967772e+00,1.052875461804984258e+00,1.052783380759310328e+00,1.052691332547354364e+00,1.052599317157496328e+00,1.052507334578088205e+00,1.052415384797455999e+00,1.052323467803896628e+00,1.052231583585680363e+00,1.052139732131049277e+00,1.052047913428218351e+00,1.051956127465375035e+00,1.051864374230677912e+00,1.051772653712259364e+00,1.051680965898222464e+00,1.051589310776644082e+00,1.051497688335572223e+00,1.051406098563027580e+00,1.051314541447002870e+00,1.051223016975462610e+00,1.051131525136344003e+00,1.051040065917555610e+00,1.050948639306979127e+00,1.050857245292467157e+00,1.050765883861844552e+00,1.050674555002909072e+00,1.050583258703428946e+00,1.050491994951145980e+00,1.050400763733772225e+00,1.050309565038992865e+00,1.050218398854464219e+00,1.050127265167814850e+00,1.050036163966644898e+00,1.049945095238527193e+00,1.049854058971004367e+00,1.049763055151593072e+00,1.049672083767779984e+00,1.049581144807024469e+00,1.049490238256757024e+00,1.049399364104380394e+00,1.049308522337268235e+00,1.049217712942766889e+00,1.049126935908192726e+00,1.049036191220835468e+00,1.048945478867955305e+00,1.048854798836784230e+00,1.048764151114525589e+00,1.048673535688354752e+00,1.048582952545418667e+00,1.048492401672834973e+00,1.048401883057693107e+00,1.048311396687054087e+00,1.048220942547950729e+00,1.048130520627385875e+00,1.048040130912335721e+00,1.047949773389746486e+00,1.047859448046535968e+00,1.047769154869593766e+00,1.047678893845780168e+00,1.047588664961927041e+00,1.047498468204837829e+00,1.047408303561287113e+00,1.047318171018020161e+00,1.047228070561754487e+00,1.047138002179177851e+00,1.047047965856950036e+00,1.046957961581701735e+00,1.046867989340034555e+00,1.046778049118521459e+00,1.046688140903706765e+00,1.046598264682105706e+00,1.046508420440204867e+00,1.046418608164461972e+00,1.046328827841305209e+00,1.046239079457134791e+00,1.046149362998321397e+00,1.046059678451206842e+00,1.045970025802104297e+00,1.045880405037297622e+00,1.045790816143042257e+00,1.045701259105563663e+00,1.045611733911059327e+00,1.045522240545697201e+00,1.045432778995616596e+00,1.045343349246927511e+00,1.045253951285710636e+00,1.045164585098018240e+00,1.045075250669872835e+00,1.044985947987269181e+00,1.044896677036170729e+00,1.044807437802514505e+00,1.044718230272206005e+00,1.044629054431123416e+00,1.044539910265114946e+00,1.044450797759999716e+00,1.044361716901567538e+00,1.044272667675580024e+00,1.044183650067768809e+00,1.044094664063836442e+00,1.044005709649456159e+00,1.043916786810272779e+00,1.043827895531901140e+00,1.043739035799927439e+00,1.043650207599908342e+00,1.043561410917371202e+00,1.043472645737814508e+00,1.043383912046707884e+00,1.043295209829490311e+00,1.043206539071573236e+00,1.043117899758337463e+00,1.043029291875135822e+00,1.042940715407290941e+00,1.042852170340096363e+00,1.042763656658816540e+00,1.042675174348687062e+00,1.042586723394913317e+00,1.042498303782672719e+00,1.042409915497111594e+00,1.042321558523348513e+00,1.042233232846471846e+00,1.042144938451541769e+00,1.042056675323587811e+00,1.041968443447611081e+00,1.041880242808583601e+00,1.041792073391446527e+00,1.041703935181113483e+00,1.041615828162467894e+00,1.041527752320364542e+00,1.041439707639627343e+00,1.041351694105052905e+00,1.041263711701406969e+00,1.041175760413426632e+00,1.041087840225819683e+00,1.040999951123264156e+00,1.040912093090409440e+00,1.040824266111874508e+00,1.040736470172249684e+00,1.040648705256096429e+00,1.040560971347945785e+00,1.040473268432300369e+00,1.040385596493632381e+00,1.040297955516385819e+00,1.040210345484974708e+00,1.040122766383784203e+00,1.040035218197169264e+00,1.039947700909455763e+00,1.039860214504940927e+00,1.039772758967892008e+00,1.039685334282546725e+00,1.039597940433113710e+00,1.039510577403772507e+00,1.039423245178673127e+00,1.039335943741935830e+00,1.039248673077651786e+00,1.039161433169883075e+00,1.039074224002662472e+00,1.038987045559992106e+00,1.038899897825846574e+00,1.038812780784170720e+00,1.038725694418878520e+00,1.038638638713855977e+00,1.038551613652959782e+00,1.038464619220016871e+00,1.038377655398824873e+00,1.038290722173152325e+00,1.038203819526738014e+00,1.038116947443291416e+00,1.038030105906492917e+00,1.037943294899993596e+00,1.037856514407415220e+00,1.037769764412349804e+00,1.037683044898360718e+00,1.037596355848981133e+00,1.037509697247715801e+00,1.037423069078039717e+00,1.037336471323398124e+00,1.037249903967207398e+00,1.037163366992855273e+00,1.037076860383699062e+00,1.036990384123067210e+00,1.036903938194258856e+00,1.036817522580544271e+00,1.036731137265163527e+00,1.036644782231328277e+00,1.036558457462220195e+00,1.036472162940992092e+00,1.036385898650767690e+00,1.036299664574640733e+00,1.036213460695676103e+00,1.036127286996910257e+00,1.036041143461349012e+00,1.035955030071969318e+00,1.035868946811719038e+00,1.035782893663517390e+00,1.035696870610253173e+00,1.035610877634787208e+00,1.035524914719949896e+00,1.035438981848543216e+00,1.035353079003339838e+00,1.035267206167082676e+00,1.035181363322486225e+00,1.035095550452235003e+00,1.035009767538985104e+00,1.034924014565362871e+00,1.034838291513965558e+00,1.034752598367361553e+00,1.034666935108090158e+00,1.034581301718660473e+00,1.034495698181553625e+00,1.034410124479220983e+00,1.034324580594085052e+00,1.034239066508539473e+00,1.034153582204947686e+00,1.034068127665645154e+00,1.033982702872937809e+00,1.033897307809101829e+00,1.033811942456385635e+00,1.033726606797007674e+00,1.033641300813157082e+00,1.033556024486994573e+00,1.033470777800651552e+00,1.033385560736230335e+00,1.033300373275803707e+00,1.033215215401416254e+00,1.033130087095082805e+00,1.033044988338790438e+00,1.032959919114494918e+00,1.032874879404125146e+00,1.032789869189579379e+00,1.032704888452728786e+00,1.032619937175413227e+00,1.032535015339445472e+00,1.032450122926608538e+00,1.032365259918656353e+00,1.032280426297314424e+00,1.032195622044278061e+00,1.032110847141215260e+00,1.032026101569764265e+00,1.031941385311534454e+00,1.031856698348105894e+00,1.031772040661030676e+00,1.031687412231830914e+00,1.031602813042000300e+00,1.031518243073004104e+00,1.031433702306278288e+00,1.031349190723229947e+00,1.031264708305236422e+00,1.031180255033648407e+00,1.031095830889785736e+00,1.031011435854940261e+00,1.030927069910374749e+00,1.030842733037323100e+00,1.030758425216991236e+00,1.030674146430555105e+00,1.030589896659162452e+00,1.030505675883931938e+00,1.030421484085954242e+00,1.030337321246290516e+00,1.030253187345972821e+00,1.030169082366005906e+00,1.030085006287364324e+00,1.030000959090994650e+00,1.029916940757814814e+00,1.029832951268712993e+00,1.029748990604550496e+00,1.029665058746158213e+00,1.029581155674339499e+00,1.029497281369868178e+00,1.029413435813490540e+00,1.029329618985922901e+00,1.029245830867854039e+00,1.029162071439943649e+00,1.029078340682821890e+00,1.028994638577092280e+00,1.028910965103328357e+00,1.028827320242075682e+00,1.028743703973850288e+00,1.028660116279140890e+00,1.028576557138406899e+00,1.028493026532078858e+00,1.028409524440559997e+00,1.028326050844224016e+00,1.028242605723415970e+00,1.028159189058453382e+00,1.028075800829624020e+00,1.027992441017188785e+00,1.027909109601378157e+00,1.027825806562395305e+00,1.027742531880414756e+00,1.027659285535582834e+00,1.027576067508017443e+00,1.027492877777806957e+00,1.027409716325012878e+00,1.027326583129666959e+00,1.027243478171773861e+00,1.027160401431308268e+00,1.027077352888217554e+00,1.026994332522421338e+00,1.026911340313809706e+00,1.026828376242243879e+00,1.026745440287558875e+00,1.026662532429559294e+00,
        1.026579652648022423e+00,1.026496800922697572e+00,1.026413977233304298e+00,1.026331181559535288e+00,1.026248413881054589e+00,1.026165674177497600e+00,1.026082962428471745e+00,1.026000278613556249e+00,1.025917622712301913e+00,1.025834994704231340e+00,1.025752394568839154e+00,1.025669822285591115e+00,1.025587277833925892e+00,1.025504761193252401e+00,1.025422272342953134e+00,1.025339811262381051e+00,1.025257377930861580e+00,1.025174972327691503e+00,1.025092594432140292e+00,1.025010244223448774e+00,1.024927921680829357e+00,1.024845626783466690e+00,1.024763359510517224e+00,1.024681119841109656e+00,1.024598907754344035e+00,1.024516723229292436e+00,1.024434566244999179e+00,1.024352436780480602e+00,1.024270334814724182e+00,1.024188260326690525e+00,1.024106213295311152e+00,1.024024193699490048e+00,1.023942201518103001e+00,1.023860236729998485e+00,1.023778299313995888e+00,1.023696389248886840e+00,1.023614506513436107e+00,1.023532651086379364e+00,1.023450822946424532e+00,1.023369022072251333e+00,1.023287248442512398e+00,1.023205502035831715e+00,1.023123782830805073e+00,1.023042090806002058e+00,1.022960425939962059e+00,1.022878788211197598e+00,1.022797177598193441e+00,1.022715594079406598e+00,1.022634037633265880e+00,1.022552508238172342e+00,1.022471005872498839e+00,1.022389530514590916e+00,1.022308082142765695e+00,1.022226660735313208e+00,1.022145266270495068e+00,1.022063898726545128e+00,1.021982558081669934e+00,1.021901244314047608e+00,1.021819957401828738e+00,1.021738697323135714e+00,1.021657464056064279e+00,1.021576257578680869e+00,1.021495077869025492e+00,1.021413924905109960e+00,1.021332798664917885e+00,1.021251699126405788e+00,1.021170626267501547e+00,1.021089580066107061e+00,1.021008560500093809e+00,1.020927567547308401e+00,1.020846601185568137e+00,1.020765661392662782e+00,1.020684748146354348e+00,1.020603861424377534e+00,1.020523001204439728e+00,1.020442167464219230e+00,1.020361360181368138e+00,1.020280579333510129e+00,1.020199824898241570e+00,1.020119096853130625e+00,1.020038395175718593e+00,1.019957719843519017e+00,1.019877070834016575e+00,1.019796448124670185e+00,1.019715851692909903e+00,1.019635281516138914e+00,1.019554737571731762e+00,1.019474219837036122e+00,1.019393728289372358e+00,1.019313262906032858e+00,1.019232823664282250e+00,1.019152410541357856e+00,1.019072023514469461e+00,1.018991662560799316e+00,1.018911327657501031e+00,1.018831018781702458e+00,1.018750735910503025e+00,1.018670479020974184e+00,1.018590248090160078e+00,1.018510043095078199e+00,1.018429864012717401e+00,1.018349710820039222e+00,1.018269583493977892e+00,1.018189482011440328e+00,1.018109406349305246e+00,1.018029356484424497e+00,1.017949332393622619e+00,1.017869334053695729e+00,1.017789361441412410e+00,1.017709414533515044e+00,1.017629493306717370e+00,1.017549597737705813e+00,1.017469727803139490e+00,1.017389883479650203e+00,1.017310064743842002e+00,1.017230271572290956e+00,1.017150503941547157e+00,1.017070761828131165e+00,1.016991045208537781e+00,1.016911354059233608e+00,1.016831688356657493e+00,1.016752048077221415e+00,1.016672433197310044e+00,1.016592843693279402e+00,1.016513279541458648e+00,1.016433740718150291e+00,1.016354227199628424e+00,1.016274738962140045e+00,1.016195275981904400e+00,1.016115838235113200e+00,1.016036425697931289e+00,1.015957038346495533e+00,1.015877676156915932e+00,1.015798339105273840e+00,1.015719027167624189e+00,1.015639740319994599e+00,1.015560478538383826e+00,1.015481241798764867e+00,1.015402030077082074e+00,1.015322843349253157e+00,1.015243681591167624e+00,1.015164544778687894e+00,1.015085432887648631e+00,1.015006345893857631e+00,1.014927283773094935e+00,1.014848246501112383e+00,1.014769234053635394e+00,1.014690246406361629e+00,1.014611283534960773e+00,1.014532345415075643e+00,1.014453432022321078e+00,1.014374543332284828e+00,1.014295679320526888e+00,1.014216839962579941e+00,1.014138025233948914e+00,1.014059235110111867e+00,1.013980469566518661e+00,1.013901728578591399e+00,1.013823012121725764e+00,1.013744320171289015e+00,1.013665652702621545e+00,1.013587009691035989e+00,1.013508391111816564e+00,1.013429796940221284e+00,1.013351227151480627e+00,1.013272681720795765e+00,1.013194160623342555e+00,1.013115663834267766e+00,1.013037191328691744e+00,1.012958743081706414e+00,1.012880319068375723e+00,1.012801919263737638e+00,1.012723543642801260e+00,1.012645192180548825e+00,1.012566864851934145e+00,1.012488561631883943e+00,1.012410282495297853e+00,1.012332027417046643e+00,1.012253796371975323e+00,1.012175589334898484e+00,1.012097406280606293e+00,1.012019247183858939e+00,1.011941112019389521e+00,1.011863000761904496e+00,1.011784913386081231e+00,1.011706849866570668e+00,1.011628810177994442e+00,1.011550794294948208e+00,1.011472802191999198e+00,1.011394833843687113e+00,1.011316889224523674e+00,1.011238968308992181e+00,1.011161071071549733e+00,1.011083197486625007e+00,1.011005347528618481e+00,1.010927521171903765e+00,1.010849718390825380e+00,1.010771939159701427e+00,1.010694183452820916e+00,1.010616451244446656e+00,1.010538742508812593e+00,1.010461057220123804e+00,1.010383395352559610e+00,1.010305756880270689e+00,1.010228141777379296e+00,1.010150550017980153e+00,1.010072981576140005e+00,1.009995436425897841e+00,1.009917914541265116e+00,1.009840415896224419e+00,1.009762940464730807e+00,1.009685488220711358e+00,1.009608059138065617e+00,1.009530653190664928e+00,1.009453270352351550e+00,1.009375910596941317e+00,1.009298573898221196e+00,1.009221260229949957e+00,1.009143969565858834e+00,1.009066701879650640e+00,1.008989457144999990e+00,1.008912235335553742e+00,1.008835036424930331e+00,1.008757860386719774e+00,1.008680707194484549e+00,1.008603576821758940e+00,1.008526469242048140e+00,1.008449384428830253e+00,1.008372322355554074e+00,1.008295282995641085e+00,1.008218266322483680e+00,1.008141272309446945e+00,1.008064300929866430e+00,1.007987352157050598e+00,1.007910425964278156e+00,1.007833522324800724e+00,1.007756641211841053e+00,1.007679782598593254e+00,1.007602946458223236e+00,1.007526132763868487e+00,1.007449341488638073e+00,1.007372572605611971e+00,1.007295826087842627e+00,1.007219101908352954e+00,1.007142400040138330e+00,1.007065720456164382e+00,1.006989063129368756e+00,1.006912428032661344e+00,1.006835815138921175e+00,1.006759224421001075e+00,1.006682655851723229e+00,1.006606109403882288e+00,1.006529585050243591e+00,1.006453082763543616e+00,1.006376602516490415e+00,1.006300144281763620e+00,1.006223708032012665e+00,1.006147293739859672e+00,1.006070901377896343e+00,1.005994530918686847e+00,1.005918182334765376e+00,1.005841855598637702e+00,1.005765550682780507e+00,1.005689267559640943e+00,1.005613006201637960e+00,1.005536766581160979e+00,1.005460548670570109e+00,1.005384352442196372e+00,1.005308177868341923e+00,1.005232024921279832e+00,1.005155893573253190e+00,1.005079783796476001e+00,1.005003695563134070e+00,1.004927628845382337e+00,1.004851583615347765e+00,1.004775559845126676e+00,1.004699557506786745e+00,1.004623576572366117e+00,1.004547617013873184e+00,1.004471678803287471e+00,1.004395761912557639e+00,1.004319866313604370e+00,1.004243991978317929e+00,1.004168138878558381e+00,1.004092306986157590e+00,1.004016496272915893e+00,1.003940706710605646e+00,1.003864938270968343e+00,1.003789190925715724e+00,1.003713464646530218e+00,1.003637759405064056e+00,1.003562075172938828e+00,1.003486411921747701e+00,1.003410769623052534e+00,1.003335148248385877e+00,1.003259547769249416e+00,1.003183968157115968e+00,1.003108409383427269e+00,1.003032871419594851e+00,1.002957354237000498e+00,1.002881857806995569e+00,1.002806382100900784e+00,1.002730927090006885e+00,1.002655492745573973e+00,1.002580079038832617e+00,1.002504685940981410e+00,1.002429313423189638e+00,1.002353961456595277e+00,1.002278630012306548e+00,1.002203319061400366e+00,1.002128028574922558e+00,1.002052758523889864e+00,1.001977508879286383e+00,1.001902279612067348e+00,1.001827070693154909e+00,1.001751882093441903e+00,1.001676713783789863e+00,1.001601565735029453e+00,1.001526437917960033e+00,1.001451330303349652e+00,1.001376242861936383e+00,1.001301175564425661e+00,1.001226128381492497e+00,1.001151101283780376e+00,1.001076094241901915e+00,1.001001107226438647e+00,1.000926140207939019e+00,1.000851193156921948e+00,1.000776266043873708e+00,1.000701358839249266e+00,1.000626471513472060e+00,1.000551604036934217e+00,1.000476756379995669e+00,1.000401928512984817e+00,1.000327120406198311e+00,1.000252332029900382e+00,1.000177563354324395e+00,1.000102814349670188e+00,1.000028084986107846e+00,9.999533752337729231e-01,9.998786850627703338e-01,9.998040144431725729e-01,9.997293633450202721e-01,9.996547317383202014e-01,9.995801195930485994e-01,9.995055268791485092e-01,9.994309535665304445e-01,9.993563996250725001e-01,9.992818650246205747e-01,9.992073497349869271e-01,9.991328537259522857e-01,9.990583769672641834e-01,9.989839194286367352e-01,9.989094810797521928e-01,9.988350618902596123e-01,9.987606618297741878e-01,9.986862808678789172e-01,9.986119189741239355e-01,9.985375761180257381e-01,9.984632522690667367e-01,9.983889473966979233e-01,9.983146614703352073e-01,9.982403944593623013e-01,9.981661463331286122e-01,9.980919170609504620e-01,9.980177066121102003e-01,9.979435149558566476e-01,9.978693420614049847e-01,9.977951878979361977e-01,9.977210524345975218e-01,9.976469356405029965e-01,9.975728374847319113e-01,9.974987579363288059e-01,9.974246969643059124e-01,9.973506545376388255e-01,9.972766306252713875e-01,9.972026251961111365e-01,9.971286382190317488e-01,9.970546696628730388e-01,9.969807194964392938e-01,9.969067876885004953e-01,9.968328742077923188e-01,9.967589790230151348e-01,9.966851021028337865e-01,9.966112434158801436e-01,9.965374029307489945e-01,9.964635806160012654e-01,9.963897764401625778e-01,9.963159903717222488e-01,9.962422223791356224e-01,9.961684724308216277e-01,9.960947404951645545e-01,9.960210265405124996e-01,9.959473305351784767e-01,9.958736524474387508e-01,9.957999922455349484e-01,9.957263498976721694e-01,9.956527253720200976e-01,9.955791186367113355e-01,
        9.955055296598432912e-01,9.954319584094767359e-01,9.953584048536364692e-01,9.952848689603105425e-01,9.952113506974508139e-01,9.951378500329723931e-01,9.950643669347538633e-01,9.949909013706371708e-01,9.949174533084277350e-01,9.948440227158928950e-01,9.947706095607641297e-01,9.946972138107358363e-01,9.946238354334643317e-01,9.945504743965698502e-01,9.944771306676347677e-01,9.944038042142037126e-01,9.943304950037838985e-01,9.942572030038454578e-01,9.941839281818204421e-01,9.941106705051030445e-01,9.940374299410498216e-01,9.939642064569789159e-01,9.938910000201707229e-01,9.938178105978674459e-01,9.937446381572728749e-01,9.936714826655528299e-01,9.935983440898336072e-01,9.935252223972045327e-01,9.934521175547146310e-01,9.933790295293755124e-01,9.933059582881589300e-01,9.932329037979988895e-01,9.931598660257884292e-01,9.930868449383835062e-01,9.930138405025986659e-01,9.929408526852120387e-01,9.928678814529589003e-01,9.927949267725372229e-01,9.927219886106045665e-01,9.926490669337788564e-01,9.925761617086379385e-01,9.925032729017199129e-01,9.924304004795219125e-01,9.923575444085022124e-01,9.922847046550782313e-01,9.922118811856265319e-01,9.921390739664831537e-01,9.920662829639442792e-01,9.919935081442643465e-01,9.919207494736577146e-01,9.918480069182974423e-01,9.917752804443148440e-01,9.917025700178009329e-01,9.916298756048055330e-01,9.915571971713359467e-01,9.914845346833585094e-01,9.914118881067986999e-01,9.913392574075383656e-01,9.912666425514187196e-01,9.911940435042390085e-01,9.911214602317552913e-01,9.910488926996825487e-01,9.909763408736926849e-01,9.909038047194157484e-01,9.908312842024374900e-01,9.907587792883029154e-01,9.906862899425129543e-01,9.906138161305256817e-01,9.905413578177564293e-01,9.904689149695768968e-01,9.903964875513152633e-01,9.903240755282564090e-01,9.902516788656418045e-01,9.901792975286685117e-01,9.901069314824902934e-01,9.900345806922161707e-01,9.899622451229123099e-01,9.898899247395984702e-01,9.898176195072518890e-01,9.897453293908045069e-01,9.896730543551431891e-01,9.896007943651106142e-01,9.895285493855036085e-01,9.894563193810751445e-01,9.893841043165316762e-01,9.893119041565352489e-01,9.892397188657012785e-01,9.891675484086001058e-01,9.890953927497572185e-01,9.890232518536500317e-01,9.889511256847113296e-01,9.888790142073274891e-01,9.888069173858378136e-01,9.887348351845361982e-01,9.886627675676689098e-01,9.885907144994352524e-01,9.885186759439887894e-01,9.884466518654335676e-01,9.883746422278295585e-01,9.883026469951863291e-01,9.882306661314674834e-01,9.881586996005884416e-01,9.880867473664173284e-01,9.880148093927729747e-01,9.879428856434265827e-01,9.878709760821017261e-01,9.877990806724722406e-01,9.877271993781643333e-01,9.876553321627544735e-01,9.875834789897707244e-01,9.875116398226913006e-01,9.874398146249465658e-01,9.873680033599153694e-01,9.872962059909284882e-01,9.872244224812667390e-01,9.871526527941593132e-01,9.870808968927875515e-01,9.870091547402807253e-01,9.869374262997187008e-01,9.868657115341303854e-01,9.867940104064931717e-01,9.867223228797346035e-01,9.866506489167300442e-01,9.865789884803038978e-01,9.865073415332292761e-01,9.864357080382269993e-01,9.863640879579672616e-01,9.862924812550666331e-01,9.862208878920900590e-01,9.861493078315506366e-01,9.860777410359076178e-01,9.860061874675689619e-01,9.859346470888890046e-01,9.858631198621681246e-01,9.857916057496547424e-01,9.857201047135425442e-01,9.856486167159728140e-01,9.855771417190322126e-01,9.855056796847525558e-01,9.854342305751124798e-01,9.853627943520363308e-01,9.852913709773928330e-01,9.852199604129961985e-01,9.851485626206060164e-01,9.850771775619262538e-01,9.850058051986055885e-01,9.849344454922364100e-01,9.848630984043567071e-01,9.847917638964472919e-01,9.847204419299323552e-01,9.846491324661807987e-01,9.845778354665044585e-01,9.845065508921578834e-01,9.844352787043392228e-01,9.843640188641885613e-01,9.842927713327890293e-01,9.842215360711662475e-01,9.841503130402874389e-01,9.840791022010618727e-01,9.840079035143409758e-01,9.839367169409166669e-01,9.838655424415228001e-01,9.837943799768342767e-01,9.837232295074662680e-01,9.836520909939751034e-01,9.835809643968570493e-01,9.835098496765487530e-01,9.834387467934267990e-01,9.833676557078071534e-01,9.832965763799458303e-01,9.832255087700371154e-01,9.831544528382150094e-01,9.830834085445525616e-01,9.830123758490605379e-01,9.829413547116885308e-01,9.828703450923238494e-01,9.827993469507919633e-01,9.827283602468562806e-01,9.826573849402164829e-01,9.825864209905101898e-01,9.825154683573117387e-01,9.824445270001321839e-01,9.823735968784185202e-01,9.823026779515549034e-01,9.822317701788598754e-01,9.821608735195888062e-01,9.820899879329322291e-01,9.820191133780155068e-01,9.819482498138989435e-01,9.818773971995775618e-01,9.818065554939813255e-01,9.817357246559733630e-01,9.816649046443509663e-01,9.815940954178455913e-01,9.815232969351215253e-01,9.814525091547756652e-01,9.813817320353389606e-01,9.813109655352738603e-01,9.812402096129749784e-01,9.811694642267705380e-01,9.810987293349183735e-01,9.810280048956091514e-01,9.809572908669643709e-01,9.808865872070360314e-01,9.808158938738075205e-01,9.807452108251921707e-01,9.806745380190331485e-01,9.806038754131040092e-01,9.805332229651071430e-01,9.804625806326746629e-01,9.803919483733677387e-01,9.803213261446750426e-01,9.802507139040153028e-01,9.801801116087344168e-01,9.801095192161055625e-01,9.800389366833300864e-01,9.799683639675365043e-01,9.798978010257799465e-01,9.798272478150423792e-01,9.797567042922313840e-01,9.796861704141816007e-01,9.796156461376520630e-01,9.795451314193280856e-01,9.794746262158199324e-01,9.794041304836620387e-01,9.793336441793136782e-01,9.792631672591579628e-01,9.791926996795025095e-01,9.791222413965775528e-01,9.790517923665363886e-01,9.789813525454560406e-01,9.789109218893348174e-01,9.788405003540946447e-01,9.787700878955778450e-01,9.786996844695492470e-01,9.786292900316944099e-01,9.785589045376194006e-01,9.784885279428513494e-01,9.784181602028376723e-01,9.783478012729452944e-01,9.782774511084602054e-01,9.782071096645880148e-01,9.781367768964530640e-01,9.780664527590985369e-01,9.779961372074844617e-01,9.779258301964902644e-01,9.778555316809107723e-01,9.777852416154599879e-01,9.777149599547669823e-01,9.776446866533774482e-01,9.775744216657539232e-01,9.775041649462731241e-01,9.774339164492277243e-01,9.773636761288253538e-01,9.772934439391878225e-01,9.772232198343506759e-01,9.771530037682640835e-01,9.770827956947906179e-01,9.770125955677062546e-01,9.769424033406993724e-01,9.768722189673710865e-01,9.768020424012333613e-01,9.767318735957095654e-01,9.766617125041351377e-01,9.765915590797551449e-01,9.765214132757253918e-01,9.764512750451112000e-01,9.763811443408867419e-01,9.763110211159371499e-01,9.762409053230537426e-01,9.761707969149369113e-01,9.761006958441956760e-01,9.760306020633452428e-01,9.759605155248084474e-01,9.758904361809138672e-01,9.758203639838973764e-01,9.757502988858993698e-01,9.756802408389659842e-01,9.756101897950480994e-01,9.755401457060011161e-01,9.754701085235839564e-01,9.754000781994595082e-01,9.753300546851935149e-01,9.752600379322541313e-01,9.751900278920121456e-01,9.751200245157390922e-01,9.750500277546096939e-01,9.749800375596968660e-01,9.749100538819760464e-01,9.748400766723215316e-01,9.747701058815070319e-01,9.747001414602061153e-01,9.746301833589897656e-01,9.745602315283271588e-01,9.744902859185858857e-01,9.744203464800296199e-01,9.743504131628188958e-01,9.742804859170099974e-01,9.742105646925561802e-01,9.741406494393045623e-01,9.740707401069967908e-01,9.740008366452694855e-01,9.739309390036525738e-01,9.738610471315685135e-01,9.737911609783338474e-01,9.737212804931554277e-01,9.736514056251326377e-01,9.735815363232568354e-01,9.735116725364080237e-01,9.734418142133576257e-01,9.733719613027661532e-01,9.733021137531837619e-01,9.732322715130475865e-01,9.731624345306846280e-01,9.730926027543075341e-01,9.730227761320169311e-01,9.729529546117989813e-01,9.728831381415267154e-01,9.728133266689570346e-01,9.727435201417330424e-01,9.726737185073798253e-01,9.726039217133081172e-01,9.725341297068104129e-01,9.724643424350621901e-01,9.723945598451203542e-01,9.723247818839230172e-01,9.722550084982894969e-01,9.721852396349186520e-01,9.721154752403888821e-01,9.720457152611582385e-01,9.719759596435620930e-01,9.719062083338136926e-01,9.718364612780042711e-01,9.717667184221007171e-01,9.716969797119459074e-01,9.716272450932582627e-01,9.715575145116308597e-01,9.714877879125300986e-01,9.714180652412972572e-01,9.713483464431450498e-01,9.712786314631586260e-01,9.712089202462950155e-01,9.711392127373820182e-01,9.710695088811174269e-01,9.709998086220689162e-01,9.709301119046725992e-01,9.708604186732333607e-01,9.707907288719235250e-01,9.707210424447816344e-01,9.706513593357135594e-01,9.705816794884899457e-01,9.705120028467469906e-01,9.704423293539843343e-01,9.703726589535661695e-01,9.703029915887180223e-01,9.702333272025288613e-01,9.701636657379477668e-01,9.700940071377865959e-01,9.700243513447148747e-01,9.699546983012627965e-01,9.698850479498182242e-01,9.698154002326275780e-01,9.697457550917945035e-01,9.696761124692780953e-01,9.696064723068933411e-01,9.695368345463104554e-01,9.694671991290536583e-01,9.693975659965003988e-01,9.693279350898802438e-01,9.692583063502752116e-01,9.691886797186179958e-01,9.691190551356921867e-01,9.690494325421298294e-01,9.689798118784120895e-01,9.689101930848685873e-01,9.688405761016748441e-01,9.687709608688540586e-01,9.687013473262732211e-01,9.686317354136456670e-01,9.685621250705275243e-01,9.684925162363181572e-01,9.684229088502596117e-01,9.683533028514338392e-01,9.682836981787651398e-01,9.682140947710158319e-01,9.681444925667878065e-01,9.680748915045213066e-01,9.680052915224925947e-01,9.679356925588146199e-01,9.678660945514356850e-01,9.677964974381382257e-01,9.677269011565384771e-01,9.676573056440853637e-01,9.675877108380589453e-01,9.675181166755704165e-01,9.674485230935611080e-01,9.673789300288011539e-01,9.673093374178883819e-01,9.672397451972482019e-01,
        9.671701533031316078e-01,9.671005616716150666e-01,9.670309702385997408e-01,9.669613789398092685e-01,9.668917877107902070e-01,9.668221964869099239e-01,9.667526052033567074e-01,9.666830137951374358e-01,9.666134221970782425e-01,9.665438303438217416e-01,9.664742381698273599e-01,9.664046456093697834e-01,9.663350525965381799e-01,9.662654590652345332e-01,9.661958649491731999e-01,9.661262701818800203e-01,9.660566746966900986e-01,9.659870784267485799e-01,9.659174813050083186e-01,9.658478832642283241e-01,9.657782842369740939e-01,9.657086841556162815e-01,9.656390829523279207e-01,9.655694805590850915e-01,9.654998769076656995e-01,9.654302719296478097e-01,9.653606655564079819e-01,9.652910577191206043e-01,9.652214483487582264e-01,9.651518373760873404e-01,9.650822247316703795e-01,9.650126103458619431e-01,9.649429941488096851e-01,9.648733760704514273e-01,9.648037560405153812e-01,9.647341339885178169e-01,9.646645098437623966e-01,9.645948835353387318e-01,9.645252549921209395e-01,9.644556241427679755e-01,9.643859909157197485e-01,9.643163552391974536e-01,9.642467170412029054e-01,9.641770762495156522e-01,9.641074327916921982e-01,9.640377865950652270e-01,9.639681375867423796e-01,9.638984856936041457e-01,9.638288308423026418e-01,9.637591729592606127e-01,9.636895119706708757e-01,9.636198478024928793e-01,9.635501803804530363e-01,9.634805096300429472e-01,9.634108354765174020e-01,9.633411578448943802e-01,9.632714766599512757e-01,9.632017918462262296e-01,9.631321033280145771e-01,9.630624110293685147e-01,9.629927148740951015e-01,9.629230147857543720e-01,9.628533106876603354e-01,9.627836025028749800e-01,9.627138901542112714e-01,9.626441735642295994e-01,9.625744526552350022e-01,9.625047273492787214e-01,9.624349975681535385e-01,9.623652632333942192e-01,9.622955242662755149e-01,9.622257805878096093e-01,9.621560321187460074e-01,9.620862787795688709e-01,9.620165204904954637e-01,9.619467571714748200e-01,9.618769887421861897e-01,9.618072151220369292e-01,9.617374362301611690e-01,9.616676519854179261e-01,9.615978623063897723e-01,9.615280671113801692e-01,9.614582663184128020e-01,9.613884598452292485e-01,9.613186476092873134e-01,9.612488295277598072e-01,9.611790055175315484e-01,9.611091754951975874e-01,9.610393393770640946e-01,9.609694970791428092e-01,9.608996485171511504e-01,9.608297936065102185e-01,9.607599322623425753e-01,9.606900643994703559e-01,9.606201899324138260e-01,9.605503087753890501e-01,9.604804208423056711e-01,9.604105260467653560e-01,9.603406243020601307e-01,9.602707155211692713e-01,9.602007996167590820e-01,9.601308765011787871e-01,9.600609460864595324e-01,9.599910082843132741e-01,9.599210630061284499e-01,9.598511101629700892e-01,9.597811496655761498e-01,9.597111814243560746e-01,9.596412053493892369e-01,9.595712213504206112e-01,9.595012293368615497e-01,9.594312292177855639e-01,9.593612209019263259e-01,9.592912042976758924e-01,9.592211793130823727e-01,9.591511458558475978e-01,9.590811038333243443e-01,9.590110531525147808e-01,9.589409937200678025e-01,9.588709254422767003e-01,9.588008482250768294e-01,9.587307619740419451e-01,9.586606665943846473e-01,9.585905619909511621e-01,9.585204480682205652e-01,9.584503247303006734e-01,9.583801918809279341e-01,9.583100494234617628e-01,9.582398972608860976e-01,9.581697352958020719e-01,9.580995634304292352e-01,9.580293815666011126e-01,9.579591896057624290e-01,9.578889874489684431e-01,9.578187749968785081e-01,9.577485521497579590e-01,9.576783188074718955e-01,9.576080748694838496e-01,9.575378202348517886e-01,9.574675548022275606e-01,9.573972784698522309e-01,9.573269911355533068e-01,9.572566926967427392e-01,9.571863830504138138e-01,9.571160620931378205e-01,9.570457297210599457e-01,9.569753858298989391e-01,9.569050303149427839e-01,9.568346630710444778e-01,9.567642839926209231e-01,9.566938929736482633e-01,9.566234899076599962e-01,9.565530746877429769e-01,9.564826472065343088e-01,9.564122073562187909e-01,9.563417550285246982e-01,9.562712901147211175e-01,9.562008125056147279e-01,9.561303220915455814e-01,9.560598187623852162e-01,9.559893024075318824e-01,9.559187729159078772e-01,9.558482301759556599e-01,9.557776740756341871e-01,9.557071045024171374e-01,9.556365213432856942e-01,9.555659244847291012e-01,9.554953138127384449e-01,9.554246892128034352e-01,9.553540505699086305e-01,9.552833977685311062e-01,9.552127306926342376e-01,9.551420492256662564e-01,9.550713532505542558e-01,9.550006426497025247e-01,9.549299173049869971e-01,9.548591770977514770e-01,9.547884219088036417e-01,9.547176516184131545e-01,9.546468661063031158e-01,9.545760652516503963e-01,9.545052489330794199e-01,9.544344170286573892e-01,9.543635694158915106e-01,9.542927059717235538e-01,9.542218265725264104e-01,9.541509310940996524e-01,9.540800194116634270e-01,9.540090913998566791e-01,9.539381469327312679e-01,9.538671858837469708e-01,9.537962081257669311e-01,9.537252135310546608e-01,9.536542019712676010e-01,9.535831733174529035e-01,9.535121274400431002e-01,9.534410642088499976e-01,9.533699834930621231e-01,9.532988851612372860e-01,9.532277690812988036e-01,9.531566351205309484e-01,9.530854831455727316e-01,9.530143130224131287e-01,9.529431246163866387e-01,9.528719177921670669e-01,9.528006924137633060e-01,9.527294483445126749e-01,9.526581854470761446e-01,9.525869035834328979e-01,9.525156026148752231e-01,9.524442824020026288e-01,9.523729428047155166e-01,9.523015836822098512e-01,9.522302048929728313e-01,9.521588062947744513e-01,9.520873877446639488e-01,9.520159490989631434e-01,9.519444902132589981e-01,9.518730109424001773e-01,9.518015111404886097e-01,9.517299906608746030e-01,9.516584493561501823e-01,9.515868870781424294e-01,9.515153036779071538e-01,9.514436990057236754e-01,9.513720729110869412e-01,9.513004252426999763e-01,9.512287558484698868e-01,9.511570645754993114e-01,9.510853512700794266e-01,9.510136157776839516e-01,9.509418579429618212e-01,9.508700776097296359e-01,9.507982746209645564e-01,9.507264488187986418e-01,9.506546000445091904e-01,9.505827281385131888e-01,9.505108329403587630e-01,9.504389142887179620e-01,9.503669720213790972e-01,9.502950059752390821e-01,9.502230159862959935e-01,9.501510018896400789e-01,9.500789635194472060e-01,9.500069007089686490e-01,9.499348132905256481e-01,9.498627010954993066e-01,9.497905639543217093e-01,9.497184016964699271e-01,9.496462141504545817e-01,9.495740011438129624e-01,9.495017625030999220e-01,9.494294980538785511e-01,9.493572076207119625e-01,9.492848910271529661e-01,9.492125480957362971e-01,9.491401786479679581e-01,9.490677825043172255e-01,9.489953594842057694e-01,9.489229094059992153e-01,9.488504320869960429e-01,9.487779273434195915e-01,9.487053949904056260e-01,9.486328348419945655e-01,9.485602467111191594e-01,9.484876304095961608e-01,9.484149857481144474e-01,9.483423125362246964e-01,9.482696105823290589e-01,9.481968796936690591e-01,9.481241196763163792e-01,9.480513303351607579e-01,9.479785114738978891e-01,9.479056628950194296e-01,9.478327843998011204e-01,9.477598757882897962e-01,9.476869368592935050e-01,9.476139674103676303e-01,9.475409672378032333e-01,9.474679361366161734e-01,9.473948739005327857e-01,9.473217803219780020e-01,9.472486551920616948e-01,9.471754983005676864e-01,9.471023094359388717e-01,9.470290883852638952e-01,9.469558349342651615e-01,9.468825488672818480e-01,9.468092299672624668e-01,9.467358780157426601e-01,9.466624927928372069e-01,9.465890740772233691e-01,9.465156216461257932e-01,9.464421352753028538e-01,9.463686147390309999e-01,9.462950598100899890e-01,9.462214702597468996e-01,9.461478458577413653e-01,9.460741863722692546e-01,9.460004915699663508e-01,9.459267612158928085e-01,9.458529950735160563e-01,9.457791929046955870e-01,9.457053544696634173e-01,9.456314795270088780e-01,9.455575678336618495e-01,9.454836191448726668e-01,9.454096332141965764e-01,9.453356097934740854e-01,9.452615486328143080e-01,9.451874494805739824e-01,9.451133120833408174e-01,9.450391361859127315e-01,9.449649215312796446e-01,9.448906678606024956e-01,9.448163749131950340e-01,9.447420424265012828e-01,9.446676701360767758e-01,9.445932577755673520e-01,9.445188050766875065e-01,9.444443117691980749e-01,9.443697775808872485e-01,9.442952022375459276e-01,9.442205854629458495e-01,9.441459269788168296e-01,9.440712265048246676e-01,9.439964837585468338e-01,9.439216984554475998e-01,9.438468703088567224e-01,9.437719990299417994e-01,9.436970843276859533e-01,9.436221259088607427e-01,9.435471234780007377e-01,9.434720767373796502e-01,9.433969853869803579e-01,9.433218491244712567e-01,9.432466676451771725e-01,9.431714406420518282e-01,9.430961678056511976e-01,9.430208488241019760e-01,9.429454833830761551e-01,9.428700711657591604e-01,9.427946118528209851e-01,9.427191051223853258e-01,9.426435506499984962e-01,9.425679481085990075e-01,9.424922971684844830e-01,9.424165974972805726e-01,9.423408487599070904e-01,9.422650506185451524e-01,9.421892027326024266e-01,9.421133047586792708e-01,9.420373563505337611e-01,9.419613571590456091e-01,9.418853068321795252e-01,9.418092050149498018e-01,9.417330513493811228e-01,9.416568454744710381e-01,9.415805870261521049e-01,9.415042756372508093e-01,9.414279109374492638e-01,9.413514925532432409e-01,9.412750201079013168e-01,9.411984932214215727e-01,9.411219115104902944e-01,9.410452745884381187e-01,9.409685820651944033e-01,9.408918335472433725e-01,9.408150286375768223e-01,9.407381669356489340e-01,9.406612480373275353e-01,9.405842715348461391e-01,9.405072370167545381e-01,9.404301440678688451e-01,9.403529922692193121e-01,9.402757811980004821e-01,9.401985104275160099e-01,9.401211795271257055e-01,9.400437880621911324e-01,9.399663355940189868e-01,9.398888216798042539e-01,9.398112458725720320e-01,9.397336077211195793e-01,9.396559067699540302e-01,9.395781425592341085e-01,9.395003146247041803e-01,9.394224224976324145e-01,9.393444657047468338e-01,9.392664437681679246e-01,9.391883562053404688e-01,9.391102025289668198e-01,9.390319822469356259e-01,9.389536948622505541e-01,9.388753398729583477e-01,9.387969167720725538e-01,9.387184250475011371e-01,9.386398641819654332e-01,9.385612336529245425e-01,9.384825329324933962e-01,
        9.384037614873609323e-01,9.383249187787078283e-01,9.382460042621181273e-01,9.381670173874963048e-01,9.380879575989745645e-01,9.380088243348251309e-01,9.379296170273658806e-01,9.378503351028659729e-01,9.377709779814499269e-01,9.376915450769993665e-01,9.376120357970516572e-01,9.375324495426963223e-01,9.374527857084730131e-01,9.373730436822610423e-01,9.372932228451709147e-01,9.372133225714337490e-01,9.371333422282843717e-01,9.370532811758469638e-01,9.369731387670129363e-01,9.368929143473214705e-01,9.368126072548320638e-01,9.367322168199988530e-01,9.366517423655381647e-01,9.365711832062963982e-01,9.364905386491128025e-01,9.364098079926798102e-01,9.363289905273991520e-01,9.362480855352367515e-01,9.361670922895728442e-01,9.360860100550468799e-01,9.360048380874037566e-01,9.359235756333292855e-01,9.358422219302885425e-01,9.357607762063564483e-01,9.356792376800453503e-01,9.355976055601273877e-01,9.355158790454548567e-01,9.354340573247730273e-01,9.353521395765306279e-01,9.352701249686852236e-01,9.351880126585030428e-01,9.351058017923533638e-01,9.350234915054997931e-01,9.349410809218827723e-01,9.348585691539010867e-01,9.347759553021826040e-01,9.346932384553526818e-01,9.346104176897952476e-01,9.345274920694073284e-01,9.344444606453481406e-01,9.343613224557795194e-01,9.342780765256005759e-01,9.341947218661766916e-01,9.341112574750574105e-01,9.340276823356904456e-01,9.339439954171232516e-01,9.338601956737035970e-01,9.337762820447615963e-01,9.336922534542929641e-01,9.336081088106256143e-01,9.335238470060807092e-01,9.334394669166236058e-01,9.333549674015017006e-01,9.332703473028766128e-01,9.331856054454404914e-01,9.331007406360255496e-01,9.330157516631961689e-01,9.329306372968357852e-01,9.328453962877151229e-01,9.327600273670506592e-01,9.326745292460481007e-01,9.325889006154320926e-01,9.325031401449613844e-01,9.324172464829274531e-01,9.323312182556400263e-01,9.322450540668928420e-01,9.321587524974133121e-01,9.320723121042955306e-01,9.319857314204132992e-01,9.318990089538138344e-01,9.318121431870919347e-01,9.317251325767421655e-01,9.316379755524901718e-01,9.315506705165994550e-01,9.314632158431567222e-01,9.313756098773297021e-01,9.312878509346008693e-01,9.311999372999728575e-01,9.311118672271466723e-01,9.310236389376683741e-01,9.309352506200474497e-01,9.308467004288401014e-01,9.307579864837002281e-01,9.306691068683958790e-01,9.305800596297847394e-01,9.304908427767556445e-01,9.304014542791257947e-01,9.303118920664971148e-01,9.302221540270662059e-01,9.301322380063897777e-01,9.300421418060975665e-01,9.299518631825560711e-01,9.298613998454766660e-01,9.297707494564665387e-01,9.296799096275203400e-01,9.295888779194444451e-01,9.294976518402237042e-01,9.294062288433049268e-01,9.293146063258187484e-01,9.292227816267148999e-01,9.291307520248193175e-01,9.290385147368024343e-01,9.289460669150564343e-01,9.288534056454750276e-01,9.287605279451288665e-01,9.286674307598360434e-01,9.285741109616126865e-01,9.284805653460056485e-01,9.283867906292911920e-01,9.282927834455421134e-01,9.281985403435417670e-01,9.281040577835503180e-01,9.280093321339001333e-01,9.279143596674179761e-01,9.278191365576613503e-01,9.277236588749536716e-01,9.276279225822061658e-01,9.275319235305149457e-01,9.274356574545129517e-01,9.273391199674643204e-01,9.272423065560784217e-01,9.271452125750274664e-01,9.270478332411437039e-01,9.269501636272735601e-01,9.268521986557636261e-01,9.267539330915475215e-01,9.266553615348063211e-01,9.265564784131685716e-01,9.264572779734117081e-01,9.263577542726226799e-01,9.262579011687812613e-01,9.261577123107074261e-01,9.260571811273249354e-01,9.259563008161839637e-01,9.258550643311681538e-01,9.257534643693253740e-01,9.256514933567320247e-01,9.255491434333068534e-01,9.254464064364738007e-01,9.253432738835590810e-01,9.252397369528057025e-01,9.251357864628596550e-01,9.250314128505724431e-01,9.249266061469503253e-01,9.248213559510420900e-01,9.247156514015515416e-01,9.246094811459127927e-01,9.245028333065419268e-01,9.243956954439382923e-01,9.242880545162571737e-01,9.241798968349199672e-01,9.240712080157715835e-01,9.239619729252033231e-01,9.238521756205881896e-01,9.237417992842477110e-01,9.236308261500560945e-01,9.235192374216243394e-01,9.234070131808227444e-01,9.232941322851729860e-01,9.231805722523723778e-01,9.230663091298587641e-01,9.229513173469240384e-01,9.228355695463544839e-01,9.227190363919384186e-01,9.226016863473494034e-01,9.224834854208948665e-01,9.223643968692734063e-01,9.222443808517855057e-01,9.221233940241856386e-01,9.220013890584162208e-01,9.218783140705190116e-01,9.217541119336872812e-01,9.216287194461245669e-01,9.215020663132181822e-01,9.213740738891905702e-01,9.212446536027322219e-01,9.211137049607637239e-01,9.209811129788488815e-01,9.208467448164366909e-01,9.207104452833825281e-01,9.205720307007321246e-01,9.204312802855237896e-01,9.202879236693105591e-01,9.201416221023939190e-01,9.199919387592918607e-01,9.198382888674625502e-01,9.196798488830962848e-01,9.195153712616384034e-01,9.193425581483318787e-01,9.191571611954472143e-01,9.189385305678897664e-01,9.187199104842026509e-01,9.185346694200446915e-01,9.183618586907462689e-01,9.181974972032072024e-01,9.180392012686213743e-01,9.178857223107437235e-01,9.177362359519520840e-01,9.175901567213451404e-01,9.174470471937010707e-01,9.173065680946139722e-01,9.171684485899338490e-01,9.170324674786135555e-01,9.168984407028462069e-01,9.167662127265074901e-01,9.166356503910149112e-01,9.165066384179424430e-01,9.163790760411236747e-01,9.162528744345007636e-01,9.161279547137589629e-01,9.160042463601855145e-01,9.158816859608219785e-01,9.157602161893534376e-01,9.156397849728693528e-01,9.155203448039641589e-01,9.154018521678137787e-01,9.152842670611894826e-01,9.151675525856618476e-01,9.150516746012455904e-01,9.149366014296507199e-01,9.148223035985789675e-01,9.147087536202026525e-01,9.145959257983032886e-01,9.144837960595828541e-01,9.143723418054803265e-01,9.142615417814593526e-01,9.141513759612865941e-01,9.140418254441950996e-01,9.139328723631995333e-01,9.138244998030857769e-01,9.137166917268422228e-01,9.136094329094581745e-01,9.135027088782144977e-01,9.133965058586611674e-01,9.132908107256465513e-01,9.131856109588096793e-01,9.130808946020341210e-01,9.129766502264496930e-01,9.128728668965832016e-01,9.127695341393414763e-01,9.126666419155411436e-01,9.125641805937088069e-01,9.124621409259614513e-01,9.123605140257318169e-01,9.122592913471907483e-01,9.121584646661970996e-01,9.120580260626325320e-01,9.119579679040115128e-01,9.118582828302382870e-01,9.117589637394226676e-01,9.116600037746603880e-01,9.115613963116947494e-01,9.114631349473983901e-01,9.113652134889919099e-01,9.112676259439549398e-01,9.111703665105703731e-01,9.110734295690489093e-01,9.109768096731996057e-01,9.108805015425938123e-01,9.107845000551962888e-01,9.106888002404220916e-01,9.105933972725961389e-01,9.104982864647771512e-01,9.104034632629315338e-01,9.103089232404246722e-01,9.102146620928124321e-01,9.101206756329112135e-01,9.100269597861304627e-01,9.099335105860453243e-01,9.098403241702027744e-01,9.097473967761319225e-01,9.096547247375676992e-01,9.095623044808492930e-01,9.094701325215045484e-01,9.093782054609931276e-01,9.092865199836099865e-01,9.091950728535332926e-01,9.091038609120106750e-01,9.090128810746761490e-01,9.089221303289880538e-01,9.088316057317847863e-01,9.087413044069482249e-01,9.086512235431718487e-01,9.085613603918236691e-01,9.084717122649046406e-01,9.083822765330946680e-01,9.082930506238773294e-01,9.082040320197468652e-01,9.081152182564889985e-01,9.080266069215315872e-01,9.079381956523608910e-01,9.078499821350070054e-01,9.077619641025804764e-01,9.076741393338765285e-01,9.075865056520268093e-01,9.074990609232084227e-01,9.074118030553950387e-01,9.073247299971655133e-01,9.072378397365477021e-01,9.071511302999136772e-01,9.070645997509080294e-01,9.069782461894215775e-01,9.068920677505983852e-01,9.068060626038807470e-01,9.067202289520867042e-01,9.066345650305225323e-01,9.065490691061215411e-01,9.064637394766190681e-01,9.063785744697511193e-01,9.062935724424803219e-01,9.062087317802529629e-01,9.061240508962749018e-01,9.060395282308157938e-01,9.059551622505339630e-01,9.058709514478244795e-01,9.057868943401893302e-01,9.057029894696244643e-01,9.056192354020303759e-01,9.055356307266391180e-01,9.054521740554608566e-01,9.053688640227446349e-01,9.052856992844610096e-01,9.052026785177962331e-01,9.051198004206616465e-01,9.050370637112216121e-01,9.049544671274345475e-01,9.048720094266036185e-01,9.047896893849463051e-01,9.047075057971732948e-01,9.046254574760804745e-01,9.045435432521539143e-01,9.044617619731833980e-01,9.043801125038903876e-01,9.042985937255634266e-01,9.042172045357080856e-01,9.041359438477007959e-01,9.040548105904561149e-01,9.039738037081054278e-01,9.038929221596769814e-01,9.038121649187946804e-01,9.037315309733736646e-01,9.036510193253349810e-01,9.035706289903203681e-01,9.034903589974179194e-01,9.034102083888939649e-01,9.033301762199327234e-01,9.032502615583815064e-01,9.031704634845041380e-01,9.030907810907383704e-01,9.030112134814637370e-01,9.029317597727696265e-01,9.028524190922324610e-01,9.027731905787009792e-01,9.026940733820804086e-01,9.026150666631277408e-01,9.025361695932485606e-01,9.024573813543012024e-01,9.023787011384053480e-01,9.023001281477515123e-01,9.022216615944218532e-01,9.021433007002084281e-01,9.020650446964415536e-01,9.019868928238170547e-01,9.019088443322317294e-01,9.018308984806210349e-01,9.017530545367992145e-01,9.016753117773058657e-01,9.015976694872536168e-01,9.015201269601804679e-01,9.014426834979052394e-01,9.013653384103862409e-01,9.012880910155840475e-01,9.012109406393246092e-01,9.011338866151691329e-01,9.010569282842839645e-01,9.009800649953138008e-01,9.009032961042600096e-01,9.008266209743573949e-01,9.007500389759587334e-01,9.006735494864165359e-01,9.005971518899705819e-01,9.005208455776386733e-01,9.004446299471058346e-01,9.003685044026205064e-01,9.002924683548871876e-01,9.002165212209698453e-01,9.001406624241868881e-01,9.000648913940165752e-01,8.999892075660010926e-01,
        8.999136103816531840e-01,8.998380992883610041e-01,8.997626737393048524e-01,8.996873331933620266e-01,8.996120771150258877e-01,8.995369049743179302e-01,8.994618162467064026e-01,8.993868104130251506e-01,8.993118869593917930e-01,8.992370453771335592e-01,8.991622851627064650e-01,8.990876058176233698e-01,8.990130068483774828e-01,8.989384877663730844e-01,8.988640480878524741e-01,8.987896873338274695e-01,8.987154050300106833e-01,8.986412007067489105e-01,8.985670738989568473e-01,8.984930241460536982e-01,8.984190509918987821e-01,8.983451539847299161e-01,8.982713326771026852e-01,8.981975866258317120e-01,8.981239153919283735e-01,8.980503185405462885e-01,8.979767956409242524e-01,8.979033462663302823e-01,8.978299699940061052e-01,8.977566664051142009e-01,8.976834350846867316e-01,8.976102756215704748e-01,8.975371876083791944e-01,8.974641706414409059e-01,8.973912243207513573e-01,8.973183482499241803e-01,8.972455420361437062e-01,8.971728052901193351e-01,8.971001376260376858e-01,8.970275386615211843e-01,8.969550080175788809e-01,8.968825453185678143e-01,8.968101501921480478e-01,8.967378222692391487e-01,8.966655611839827733e-01,8.965933665736978142e-01,8.965212380788436519e-01,8.964491753429791876e-01,8.963771780127244293e-01,8.963052457377234106e-01,8.962333781706049995e-01,8.961615749669479269e-01,8.960898357852441487e-01,8.960181602868622086e-01,8.959465481360142647e-01,8.958749989997178975e-01,8.958035125477676885e-01,8.957320884526964733e-01,8.956607263897462534e-01,8.955894260368332249e-01,8.955181870745186901e-01,8.954470091859750847e-01,8.953758920569570012e-01,8.953048353757691036e-01,8.952338388332385932e-01,8.951629021226831240e-01,8.950920249398853779e-01,8.950212069830597583e-01,8.949504479528284095e-01,8.948797475521919065e-01,8.948091054865017213e-01,8.947385214634342443e-01,8.946679951929638053e-01,8.945975263873373606e-01,8.945271147610461826e-01,8.944567600308058752e-01,8.943864619155255102e-01,8.943162201362878649e-01,8.942460344163217778e-01,8.941759044809811652e-01,8.941058300577188200e-01,8.940358108760662059e-01,8.939658466676085879e-01,8.938959371659629394e-01,8.938260821067562922e-01,8.937562812276044211e-01,8.936865342680881952e-01,8.936168409697351489e-01,8.935472010759970551e-01,8.934776143322286091e-01,8.934080804856694424e-01,8.933385992854214752e-01,8.932691704824318180e-01,8.931997938294703454e-01,8.931304690811141533e-01,8.930611959937237998e-01,8.929919743254298714e-01,8.929228038361116671e-01,8.928536842873785462e-01,8.927846154425536085e-01,8.927155970666557083e-01,8.926466289263806919e-01,8.925777107900868534e-01,8.925088424277749510e-01,8.924400236110738849e-01,8.923712541132228226e-01,8.923025337090553233e-01,8.922338621749832388e-01,8.921652392889815042e-01,8.920966648305717062e-01,8.920281385808058738e-01,8.919596603222537112e-01,8.918912298389851667e-01,8.918228469165573324e-01,8.917545113419975689e-01,8.916862229037928467e-01,8.916179813918707620e-01,8.915497865975894332e-01,8.914816383137218470e-01,8.914135363344408702e-01,8.913454804553091471e-01,8.912774704732626674e-01,8.912095061865991097e-01,8.911415873949637412e-01,8.910737138993377604e-01,8.910058855020250856e-01,8.909381020066390322e-01,8.908703632180914322e-01,8.908026689425786460e-01,8.907350189875716806e-01,8.906674131618022017e-01,8.905998512752510976e-01,8.905323331391388209e-01,8.904648585659110660e-01,8.903974273692292218e-01,8.903300393639587140e-01,8.902626943661577918e-01,8.901953921930669811e-01,8.901281326630973156e-01,8.900609155958209007e-01,8.899937408119596993e-01,8.899266081333753187e-01,8.898595173830587957e-01,8.897924683851190508e-01,8.897254609647756718e-01,8.896584949483461457e-01,8.895915701632378658e-01,8.895246864379375840e-01,8.894578436020014189e-01,8.893910414860471958e-01,8.893242799217430106e-01,8.892575587417990146e-01,8.891908777799588659e-01,8.891242368709891819e-01,8.890576358506705468e-01,8.889910745557918492e-01,8.889245528241368488e-01,8.888580704944792910e-01,8.887916274065730260e-01,8.887252234011417951e-01,8.886588583198756774e-01,8.885925320054173238e-01,8.885262443013567379e-01,8.884599950522237277e-01,8.883937841034782457e-01,8.883276113015037279e-01,8.882614764935986562e-01,8.881953795279687869e-01,8.881293202537204889e-01,8.880632985208518626e-01,8.879973141802462999e-01,8.879313670836641581e-01,8.878654570837378746e-01,8.877995840339614197e-01,8.877337477886843020e-01,8.876679482031072377e-01,8.876021851332704937e-01,8.875364584360515563e-01,8.874707679691543616e-01,8.874051135911056321e-01,8.873394951612465498e-01,8.872739125397263171e-01,8.872083655874961616e-01,8.871428541663030076e-01,8.870773781386820378e-01,8.870119373679512531e-01,8.869465317182052555e-01,8.868811610543083646e-01,8.868158252418889553e-01,8.867505241473344624e-01,8.866852576377824979e-01,8.866200255811178543e-01,8.865548278459659537e-01,8.864896643016857425e-01,8.864245348183645845e-01,8.863594392668142641e-01,8.862943775185621043e-01,8.862293494458486354e-01,8.861643549216202675e-01,8.860993938195237396e-01,8.860344660139016781e-01,8.859695713797860472e-01,8.859047097928952619e-01,8.858398811296256392e-01,8.857750852670487340e-01,8.857103220829050105e-01,8.856455914555989573e-01,8.855808932641950904e-01,8.855162273884112922e-01,8.854515937086148147e-01,8.853869921058172832e-01,8.853224224616708105e-01,8.852578846584604477e-01,8.851933785791032960e-01,8.851289041071402908e-01,8.850644611267343143e-01,8.850000495226638675e-01,8.849356691803186292e-01,8.848713199856963474e-01,8.848070018253966218e-01,8.847427145866173515e-01,8.846784581571514039e-01,8.846142324253797318e-01,8.845500372802694855e-01,8.844858726113683511e-01,8.844217383088019968e-01,8.843576342632664122e-01,8.842935603660286858e-01,8.842295165089189002e-01,8.841655025843286886e-01,8.841015184852044628e-01,8.840375641050465250e-01,8.839736393379031831e-01,8.839097440783675319e-01,8.838458782215734555e-01,8.837820416631919640e-01,8.837182342994269746e-01,8.836544560270123139e-01,8.835907067432068329e-01,8.835269863457922979e-01,8.834632947330686159e-01,8.833996318038502826e-01,8.833359974574633844e-01,8.832723915937423786e-01,8.832088141130239878e-01,8.831452649161477542e-01,8.830817439044500450e-01,8.830182509797602775e-01,8.829547860443988094e-01,8.828913490011743859e-01,8.828279397533776995e-01,8.827645582047817241e-01,8.827012042596352748e-01,8.826378778226625643e-01,8.825745787990579849e-01,8.825113070944837768e-01,8.824480626150675855e-01,8.823848452673972442e-01,8.823216549585201074e-01,8.822584915959381657e-01,8.821953550876061589e-01,8.821322453419280230e-01,8.820691622677541144e-01,8.820061057743786570e-01,8.819430757715355229e-01,8.818800721693967892e-01,8.818170948785691854e-01,8.817541438100910955e-01,8.816912188754303381e-01,8.816283199864812792e-01,8.815654470555606137e-01,8.815025999954079206e-01,8.814397787191782241e-01,8.813769831404438815e-01,8.813142131731889206e-01,8.812514687318084849e-01,8.811887497311045037e-01,8.811260560862828051e-01,8.810633877129534497e-01,8.810007445271245130e-01,8.809381264452017524e-01,8.808755333839862756e-01,8.808129652606695448e-01,8.807504219928344869e-01,8.806879034984509413e-01,8.806254096958727740e-01,8.805629405038364332e-01,8.805004958414586191e-01,8.804380756282341736e-01,8.803756797840323056e-01,8.803133082290950373e-01,8.802509608840360933e-01,8.801886376698366821e-01,8.801263385078444967e-01,8.800640633197707174e-01,8.800018120276891231e-01,8.799395845540312067e-01,8.798773808215872849e-01,8.798152007535013919e-01,8.797530442732713896e-01,8.796909113047454154e-01,8.796288017721202168e-01,8.795667155999390419e-01,8.795046527130898628e-01,8.794426130368022676e-01,8.793805964966471267e-01,8.793186030185327073e-01,8.792566325287038964e-01,8.791946849537394248e-01,8.791327602205513125e-01,8.790708582563804274e-01,8.790089789887967076e-01,8.789471223456967186e-01,8.788852882553012114e-01,8.788234766461527903e-01,8.787616874471152473e-01,8.786999205873724517e-01,8.786381759964229099e-01,8.785764536040816530e-01,8.785147533404762399e-01,8.784530751360464240e-01,8.783914189215408230e-01,8.783297846280163634e-01,8.782681721868352831e-01,8.782065815296651312e-01,8.781450125884754376e-01,8.780834652955370467e-01,8.780219395834187868e-01,8.779604353849876919e-01,8.778989526334071147e-01,8.778374912621329518e-01,8.777760512049147534e-01,8.777146323957916163e-01,8.776532347690927383e-01,8.775918582594343098e-01,8.775305028017181819e-01,8.774691683311307555e-01,8.774078547831407615e-01,8.773465620934982612e-01,8.772852901982332030e-01,8.772240390336519811e-01,8.771628085363391003e-01,8.771015986431531797e-01,8.770404092912257310e-01,8.769792404179614920e-01,8.769180919610335412e-01,8.768569638583866288e-01,8.767958560482297381e-01,8.767347684690397491e-01,8.766737010595582191e-01,8.766126537587886070e-01,8.765516265059964951e-01,8.764906192407083685e-01,8.764296319027081728e-01,8.763686644320388686e-01,8.763077167689976577e-01,8.762467888541382033e-01,8.761858806282660783e-01,8.761249920324397644e-01,8.760641230079672104e-01,8.760032734964076084e-01,8.759424434395655101e-01,8.758816327794941570e-01,8.758208414584912616e-01,8.757600694190990076e-01,8.756993166041016075e-01,8.756385829565250800e-01,8.755778684196362516e-01,8.755171729369402023e-01,8.754564964521802661e-01,8.753958389093351444e-01,8.753352002526204600e-01,8.752745804264846496e-01,8.752139793756091857e-01,8.751533970449075772e-01,8.750928333795228165e-01,8.750322883248283778e-01,8.749717618264254426e-01,8.749112538301413444e-01,8.748507642820301244e-01,8.747902931283689787e-01,8.747298403156604785e-01,8.746694057906283515e-01,8.746089895002179260e-01,8.745485913915937992e-01,8.744882114121402816e-01,8.744278495094599535e-01,8.743675056313710003e-01,8.743071797259082123e-01,8.742468717413209855e-01,8.741865816260715460e-01,8.741263093288346164e-01,8.740660547984974160e-01,8.740058179841566632e-01,8.739455988351183535e-01,8.738853973008973153e-01,8.738252133312147674e-01,8.737650468759996514e-01,8.737048978853845238e-01,
        8.736447663097073324e-01,8.735846520995083075e-01,8.735245552055310725e-01,8.734644755787196457e-01,8.734044131702189961e-01,8.733443679313723784e-01,8.732843398137223323e-01,8.732243287690092393e-01,8.731643347491683249e-01,8.731043577063313244e-01,8.730443975928254829e-01,8.729844543611695595e-01,8.729245279640766020e-01,8.728646183544509496e-01,8.728047254853884551e-01,8.727448493101739313e-01,8.726849897822820390e-01,8.726251468553756219e-01,8.725653204833034859e-01,8.725055106201037303e-01,8.724457172199969746e-01,8.723859402373910221e-01,8.723261796268756418e-01,8.722664353432241224e-01,8.722067073413924954e-01,8.721469955765182025e-01,8.720873000039177647e-01,8.720276205790883362e-01,8.719679572577059279e-01,8.719083099956244087e-01,8.718486787488742840e-01,8.717890634736628064e-01,8.717294641263730881e-01,8.716698806635623242e-01,8.716103130419622369e-01,8.715507612184774100e-01,8.714912251501845120e-01,8.714317047943324068e-01,8.713722001083407109e-01,8.713127110497977945e-01,8.712532375764637793e-01,8.711937796462649874e-01,8.711343372172968280e-01,8.710749102478212436e-01,8.710154986962665991e-01,8.709561025212267937e-01,8.708967216814603729e-01,8.708373561358905279e-01,8.707780058436025428e-01,8.707186707638456813e-01,8.706593508560312999e-01,8.706000460797301832e-01,8.705407563946752081e-01,8.704814817607590127e-01,8.704222221380328861e-01,8.703629774867055469e-01,8.703037477671458078e-01,8.702445329398774687e-01,8.701853329655820923e-01,8.701261478050956732e-01,8.700669774194104145e-01,8.700078217696722849e-01,8.699486808171805752e-01,8.698895545233886750e-01,8.698304428499015195e-01,8.697713457584762553e-01,8.697122632110206863e-01,8.696531951695939400e-01,8.695941415964031362e-01,8.695351024538073847e-01,8.694760777043120115e-01,8.694170673105716673e-01,8.693580712353871087e-01,8.692990894417068626e-01,8.692401218926254503e-01,8.691811685513820551e-01,8.691222293813620769e-01,8.690633043460935792e-01,8.690043934092502864e-01,8.689454965346469217e-01,8.688866136862426481e-01,8.688277448281371829e-01,8.687688899245721297e-01,8.687100489399298686e-01,8.686512218387330009e-01,8.685924085856437937e-01,8.685336091454631813e-01,8.684748234831312086e-01,8.684160515637253663e-01,8.683572933524608128e-01,8.682985488146892639e-01,8.682398179158994367e-01,8.681811006217144966e-01,8.681223968978940553e-01,8.680637067103316173e-01,8.680050300250551354e-01,8.679463668082260108e-01,8.678877170261388718e-01,8.678290806452207962e-01,8.677704576320306451e-01,8.677118479532593964e-01,8.676532515757279240e-01,8.675946684663883302e-01,8.675360985923232793e-01,8.674775419207431115e-01,8.674189984189890623e-01,8.673604680545289325e-01,8.673019507949603080e-01,8.672434466080068960e-01,8.671849554615197464e-01,8.671264773234761414e-01,8.670680121619802616e-01,8.670095599452609658e-01,8.669511206416720128e-01,8.668926942196922836e-01,8.668342806479240048e-01,8.667758798950938592e-01,8.667174919300514313e-01,8.666591167217683189e-01,8.666007542393385776e-01,8.665424044519789426e-01,8.664840673290262751e-01,8.664257428399383398e-01,8.663674309542941376e-01,8.663091316417924626e-01,8.662508448722501253e-01,8.661925706156049509e-01,8.661343088419123371e-01,8.660760595213453650e-01,8.660178226241962429e-01,8.659595981208733084e-01,8.659013859819023606e-01,8.658431861779251060e-01,8.657849986797001574e-01,8.657268234581004807e-01,8.656686604841151711e-01,8.656105097288477879e-01,8.655523711635162432e-01,8.654942447594522470e-01,8.654361304881008632e-01,8.653780283210210644e-01,8.653199382298838449e-01,8.652618601864722203e-01,8.652037941626818940e-01,8.651457401305191475e-01,8.650876980621023948e-01,8.650296679296598512e-01,8.649716497055309761e-01,8.649136433621640307e-01,8.648556488721176327e-01,8.647976662080590904e-01,8.647396953427651800e-01,8.646817362491202585e-01,8.646237889001171517e-01,8.645658532688563769e-01,8.645079293285450328e-01,8.644500170524980209e-01,8.643921164141361579e-01,8.643342273869865089e-01,8.642763499446824982e-01,8.642184840609622443e-01,8.641606297096693368e-01,8.641027868647516152e-01,8.640449555002620574e-01,8.639871355903564476e-01,8.639293271092960413e-01,8.638715300314434575e-01,8.638137443312655650e-01,8.637559699833312621e-01,8.636982069623111435e-01,8.636404552429793879e-01,8.635827148002102049e-01,8.635249856089797227e-01,8.634672676443642114e-01,8.634095608815419709e-01,8.633518652957902217e-01,8.632941808624868818e-01,8.632365075571082347e-01,8.631788453552317053e-01,8.631211942325315301e-01,8.630635541647824205e-01,8.630059251278561216e-01,8.629483070977227444e-01,8.628907000504495439e-01,8.628331039622028076e-01,8.627755188092429695e-01,8.627179445679302727e-01,8.626603812147185524e-01,8.626028287261595651e-01,8.625452870788999915e-01,8.624877562496825467e-01,8.624302362153445367e-01,8.623727269528180805e-01,8.623152284391306654e-01,8.622577406514025933e-01,8.622002635668498671e-01,8.621427971627807496e-01,8.620853414165973172e-01,8.620278963057950161e-01,8.619704618079616631e-01,8.619130379007775566e-01,8.618556245620155876e-01,8.617982217695403513e-01,8.617408295013074815e-01,8.616834477353650934e-01,8.616260764498515634e-01,8.615687156229963062e-01,8.615113652331193306e-01,8.614540252586303515e-01,8.613966956780295670e-01,8.613393764699069921e-01,8.612820676129416819e-01,8.612247690859018423e-01,8.611674808676442749e-01,8.611102029371153765e-01,8.610529352733489183e-01,8.609956778554670453e-01,8.609384306626800543e-01,8.608811936742849502e-01,8.608239668696667790e-01,8.607667502282974059e-01,8.607095437297348495e-01,8.606523473536251689e-01,8.605951610796991336e-01,8.605379848877741100e-01,8.604808187577535072e-01,8.604236626696253332e-01,8.603665166034638601e-01,8.603093805394278482e-01,8.602522544577609898e-01,8.601951383387911321e-01,8.601380321629311654e-01,8.600809359106769136e-01,8.600238495626086888e-01,8.599667730993902914e-01,8.599097065017691222e-01,8.598526497505746269e-01,8.597956028267199624e-01,8.597385657112006641e-01,8.596815383850946457e-01,8.596245208295617557e-01,8.595675130258436658e-01,8.595105149552647594e-01,8.594535265992293560e-01,8.593965479392234874e-01,8.593395789568147869e-01,8.592826196336506017e-01,8.592256699514599916e-01,8.591687298920517302e-01,8.591117994373138611e-01,8.590548785692160294e-01,8.589979672698061508e-01,8.589410655212122991e-01,8.588841733056410410e-01,8.588272906053787681e-01,8.587704174027902537e-01,8.587135536803184310e-01,8.586566994204856140e-01,8.585998546058911662e-01,8.585430192192131660e-01,8.584861932432067411e-01,8.584293766607055121e-01,8.583725694546192608e-01,8.583157716079358179e-01,8.582589831037192862e-01,8.582022039251107071e-01,8.581454340553278382e-01,8.580886734776641545e-01,8.580319221754899583e-01,8.579751801322508253e-01,8.579184473314678261e-01,8.578617237567384146e-01,8.578050093917349850e-01,8.577483042202044272e-01,8.576916082259693486e-01,8.576349213929266302e-01,8.575782437050482043e-01,8.575215751463792779e-01,8.574649157010401090e-01,8.574082653532245635e-01,8.573516240872004479e-01,8.572949918873088437e-01,8.572383687379647732e-01,8.571817546236556451e-01,8.571251495289423650e-01,8.570685534384591131e-01,8.570119663369120122e-01,8.569553882090795716e-01,8.568988190398130200e-01,8.568422588140354179e-01,8.567857075167423231e-01,8.567291651330005697e-01,8.566726316479483794e-01,8.566161070467955829e-01,8.565595913148232876e-01,8.565030844373837660e-01,8.564465863998997897e-01,8.563900971878651847e-01,8.563336167868441651e-01,8.562771451824711111e-01,8.562206823604509021e-01,8.561642283065584724e-01,8.561077830066379235e-01,8.560513464466039668e-01,8.559949186124402587e-01,8.559384994902001775e-01,8.558820890660053804e-01,8.558256873260479125e-01,8.557692942565874317e-01,8.557129098439533177e-01,8.556565340745425630e-01,8.556001669348208827e-01,8.555438084113226038e-01,8.554874584906496660e-01,8.554311171594719543e-01,8.553747844045266335e-01,8.553184602126193692e-01,8.552621445706228842e-01,8.552058374654767370e-01,8.551495388841880985e-01,8.550932488138306420e-01,8.550369672415450983e-01,8.549806941545394778e-01,8.549244295400866278e-01,8.548681733855276743e-01,8.548119256782682474e-01,8.547556864057813675e-01,8.546994555556047812e-01,8.546432331153431816e-01,8.545870190726660987e-01,8.545308134153080104e-01,8.544746161310698973e-01,8.544184272078173548e-01,8.543622466334804821e-01,8.543060743960555481e-01,8.542499104836018819e-01,8.541937548842442052e-01,8.541376075861722983e-01,8.540814685776397797e-01,8.540253378469632173e-01,8.539692153825255705e-01,8.539131011727714160e-01,8.538569952062103896e-01,8.538008974714158539e-01,8.537448079570235659e-01,8.536887266517330097e-01,8.536326535443072849e-01,8.535765886235728850e-01,8.535205318784179207e-01,8.534644832977943407e-01,8.534084428707165992e-01,8.533524105862612119e-01,8.532963864335673110e-01,8.532403704018367563e-01,8.531843624803329140e-01,8.531283626583812119e-01,8.530723709253693610e-01,8.530163872707468009e-01,8.529604116840237005e-01,8.529044441547727340e-01,8.528484846726275270e-01,8.527925332272828784e-01,8.527365898084948714e-01,8.526806544060806514e-01,8.526247270099172049e-01,8.525688076099439128e-01,8.525128961961596641e-01,8.524569927586240770e-01,8.524010972874570546e-01,8.523452097728388965e-01,8.522893302050101871e-01,8.522334585742706858e-01,8.521775948709805482e-01,8.521217390855599927e-01,8.520658912084889680e-01,8.520100512303060425e-01,8.519542191416099586e-01,8.518983949330585226e-01,8.518425785953682716e-01,8.517867701193156948e-01,8.517309694957352351e-01,8.516751767155210651e-01,8.516193917696249782e-01,8.515636146490590530e-01,8.515078453448917672e-01,8.514520838482513287e-01,8.513963301503238990e-01,8.513405842423535930e-01,8.512848461156425905e-01,8.512291157615508030e-01,8.511733931714966506e-01,8.511176783369555077e-01,8.510619712494604805e-01,8.510062719006022958e-01,8.509505802820293008e-01,8.508948963854461311e-01,8.508392202026157092e-01,8.507835517253572455e-01,8.507278909455475713e-01,
        8.506722378551190289e-01,8.506165924460619143e-01,8.505609547104230339e-01,8.505053246403052603e-01,8.504497022278676432e-01,8.503940874653266313e-01,8.503384803449534068e-01,8.502828808590762177e-01,8.502272890000793781e-01,8.501717047604020472e-01,8.501161281325403385e-01,8.500605591090450996e-01,8.500049976825239106e-01,8.499494438456388634e-01,8.498938975911074500e-01,8.498383589117028958e-01,8.497828278002531599e-01,8.497273042496413797e-01,8.496717882528067589e-01,8.496162798027415697e-01,8.495607788924937065e-01,8.495052855151659088e-01,8.494497996639152060e-01,8.493943213319536945e-01,8.493388505125466503e-01,8.492833871990154160e-01,8.492279313847340694e-01,8.491724830631307563e-01,8.491170422276890228e-01,8.490616088719447063e-01,8.490061829894890444e-01,8.489507645739655661e-01,8.488953536190718685e-01,8.488399501185597273e-01,8.487845540662340982e-01,8.487291654559528942e-01,8.486737842816274302e-01,8.486184105372223119e-01,8.485630442167553245e-01,8.485076853142972109e-01,8.484523338239714496e-01,8.483969897399544768e-01,8.483416530564753533e-01,8.482863237678159862e-01,8.482310018683103525e-01,8.481756873523460527e-01,8.481203802143616466e-01,8.480650804488480965e-01,8.480097880503503216e-01,8.479545030134629791e-01,8.478992253328344608e-01,8.478439550031640071e-01,8.477886920192034825e-01,8.477334363757563773e-01,8.476781880676772518e-01,8.476229470898732909e-01,8.475677134373023058e-01,8.475124871049737330e-01,8.474572680879489672e-01,8.474020563813396967e-01,8.473468519803095678e-01,8.472916548800728531e-01,8.472364650758951177e-01,8.471812825630924415e-01,8.471261073370328631e-01,8.470709393931332709e-01,8.470157787268631777e-01,8.469606253337410573e-01,8.469054792093377859e-01,8.468503403492729786e-01,8.467952087492174318e-01,8.467400844048916797e-01,8.466849673120672159e-01,8.466298574665651611e-01,8.465747548642570397e-01,8.465196595010633374e-01,8.464645713729553878e-01,8.464094904759549287e-01,8.463544168061316597e-01,8.462993503596062395e-01,8.462442911325487316e-01,8.461892391211778275e-01,8.461341943217633998e-01,8.460791567306222838e-01,8.460241263441230508e-01,8.459691031586814569e-01,8.459140871707637732e-01,8.458590783768846766e-01,8.458040767736075827e-01,8.457490823575454231e-01,8.456940951253592020e-01,8.456391150737594398e-01,8.455841421995049512e-01,8.455291764994028458e-01,8.454742179703094163e-01,8.454192666091292496e-01,8.453643224128142286e-01,8.453093853783663070e-01,8.452544555028345119e-01,8.451995327833158322e-01,8.451446172169562177e-01,8.450897088009489133e-01,8.450348075325361252e-01,8.449799134090061337e-01,8.449250264276968458e-01,8.448701465859929094e-01,8.448152738813266005e-01,8.447604083111784901e-01,8.447055498730756673e-01,8.446506985645932941e-01,8.445958543833536059e-01,8.445410173270271326e-01,8.444861873933301455e-01,8.444313645800265444e-01,8.443765488849281908e-01,8.443217403058926873e-01,8.442669388408258202e-01,8.442121444876792280e-01,8.441573572444519558e-01,8.441025771091895669e-01,8.440478040799843651e-01,8.439930381549753946e-01,8.439382793323483289e-01,8.438835276103349159e-01,8.438287829872133106e-01,8.437740454613088525e-01,8.437193150309922896e-01,8.436645916946805546e-01,8.436098754508374320e-01,8.435551662979720033e-01,8.435004642346400905e-01,8.434457692594431455e-01,8.433910813710281396e-01,8.433364005680884512e-01,8.432817268493627560e-01,8.432270602136362481e-01,8.431724006597386412e-01,8.431177481865452794e-01,8.430631027929784693e-01,8.430084644780040382e-01,8.429538332406344425e-01,8.428992090799266590e-01,8.428445919949839604e-01,8.427899819849535845e-01,8.427353790490285101e-01,8.426807831864467913e-01,8.426261943964915568e-01,8.425716126784901228e-01,8.425170380318157681e-01,8.424624704558855148e-01,8.424079099501619039e-01,8.423533565141516632e-01,8.422988101474067069e-01,8.422442708495228025e-01,8.421897386201402380e-01,8.421352134589447092e-01,8.420806953656647664e-01,8.420261843400750346e-01,8.419716803819927708e-01,8.419171834912799746e-01,8.418626936678434980e-01,8.418082109116326039e-01,8.417537352226425185e-01,8.416992666009108781e-01,8.416448050465199504e-01,8.415903505595956346e-01,8.415359031403076839e-01,8.414814627888692611e-01,8.414270295055374937e-01,8.413726032906128083e-01,8.413181841444391518e-01,8.412637720674044362e-01,8.412093670599395390e-01,8.411549691225185255e-01,8.411005782556590926e-01,8.410461944599219031e-01,8.409918177359112512e-01,8.409374480842736199e-01,8.408830855056996789e-01,8.408287300009221754e-01,8.407743815707172663e-01,8.407200402159038521e-01,8.406657059373437990e-01,8.406113787359407175e-01,8.405570586126427379e-01,8.405027455684392912e-01,8.404484396043627736e-01,8.403941407214882142e-01,8.403398489209326083e-01,8.402855642038561390e-01,8.402312865714605117e-01,8.401770160249905084e-01,8.401227525657329886e-01,8.400684961950161123e-01,8.400142469142111157e-01,8.399600047247313128e-01,8.399057696280317620e-01,8.398515416256089328e-01,8.397973207190022604e-01,8.397431069097922585e-01,8.396889001996015178e-01,8.396347005900942628e-01,8.395805080829761291e-01,8.395263226799951628e-01,8.394721443829401553e-01,8.394179731936419753e-01,8.393638091139722368e-01,8.393096521458449644e-01,8.392555022912145946e-01,8.392013595520773084e-01,8.391472239304703651e-01,8.390930954284722132e-01,8.390389740482022685e-01,8.389848597918216910e-01,8.389307526615316091e-01,8.388766526595751172e-01,8.388225597882351670e-01,8.387684740498365654e-01,8.387143954467445317e-01,8.386603239813644750e-01,8.386062596561432159e-01,8.385522024735678759e-01,8.384981524361662109e-01,8.384441095465067217e-01,8.383900738071977665e-01,8.383360452208885594e-01,8.382820237902687266e-01,8.382280095180679735e-01,8.381740024070559736e-01,8.381200024600432563e-01,8.380660096798804304e-01,8.380120240694574063e-01,8.379580456317049508e-01,8.379040743695931326e-01,8.378501102861328764e-01,8.377961533843739650e-01,8.377422036674059269e-01,8.376882611383598132e-01,8.376343258004039782e-01,8.375803976567478548e-01,8.375264767106403996e-01,8.374725629653698711e-01,8.374186564242639408e-01,8.373647570906898041e-01,8.373108649680536253e-01,8.372569800598020917e-01,8.372031023694198604e-01,8.371492319004315563e-01,8.370953686564008844e-01,8.370415126409299633e-01,8.369876638576614347e-01,8.369338223102752439e-01,8.368799880024916371e-01,8.368261609380693855e-01,8.367723411208054518e-01,8.367185285545366558e-01,8.366647232431377867e-01,8.366109251905227140e-01,8.365571344006432764e-01,8.365033508774909476e-01,8.364495746250956154e-01,8.363958056475246927e-01,8.363420439488847835e-01,8.362882895333207944e-01,8.362345424050154907e-01,8.361808025681900514e-01,8.361270700271055123e-01,8.360733447860581036e-01,8.360196268493841343e-01,8.359659162214582162e-01,8.359122129066917095e-01,8.358585169095344991e-01,8.358048282344748836e-01,8.357511468860383541e-01,8.356974728687883713e-01,8.356438061873255885e-01,8.355901468462899606e-01,8.355364948503571920e-01,8.354828502042421778e-01,8.354292129126963395e-01,8.353755829805084021e-01,8.353219604125058373e-01,8.352683452135518660e-01,8.352147373885482340e-01,8.351611369424335463e-01,8.351075438801833783e-01,8.350539582068108313e-01,8.350003799273665317e-01,8.349468090469366333e-01,8.348932455706458144e-01,8.348396895036558352e-01,8.347861408511643155e-01,8.347325996184058461e-01,8.346790658106524319e-01,8.346255394332127153e-01,8.345720204914316431e-01,8.345185089906911324e-01,8.344650049364095157e-01,8.344115083340418737e-01,8.343580191890798137e-01,8.343045375070509140e-01,8.342510632935196124e-01,8.341975965540865401e-01,8.341441372943886323e-01,8.340906855200990178e-01,8.340372412369265742e-01,8.339838044506173720e-01,8.339303751669525644e-01,8.338769533917498311e-01,8.338235391308621569e-01,8.337701323901797190e-01,8.337167331756268895e-01,8.336633414931658992e-01,8.336099573487923964e-01,8.335565807485394441e-01,8.335032116984755213e-01,8.334498502047035240e-01,8.333964962733634296e-01,8.333431499106300766e-01,8.332898111227136084e-01,8.332364799158592517e-01,8.331831562963483151e-01,8.331298402704974126e-01,8.330765318446571310e-01,8.330232310252153605e-01,8.329699378185927428e-01,8.329166522312464460e-01,8.328633742696687214e-01,8.328101039403859041e-01,8.327568412499600781e-01,8.327035862049879666e-01,8.326503388121005989e-01,8.325970990779641978e-01,8.325438670092794036e-01,8.324906426127826053e-01,8.324374258952431660e-01,8.323842168634660865e-01,8.323310155242900077e-01,8.322778218845887643e-01,8.322246359512707192e-01,8.321714577312774308e-01,8.321182872315858736e-01,8.320651244592066620e-01,8.320119694211844941e-01,8.319588221245990400e-01,8.319056825765630547e-01,8.318525507842238209e-01,8.317994267547615950e-01,8.317463104953924935e-01,8.316932020133643855e-01,8.316401013159606670e-01,8.315870084104970417e-01,8.315339233043239631e-01,8.314808460048246364e-01,8.314277765194166836e-01,8.313747148555507005e-01,8.313216610207105894e-01,8.312686150224145587e-01,8.312155768682133461e-01,8.311625465656907741e-01,8.311095241224650820e-01,8.310565095461869278e-01,8.310035028445398320e-01,8.309505040252410657e-01,8.308975130960400968e-01,8.308445300647205878e-01,8.307915549390980647e-01,8.307385877270215824e-01,8.306856284363726139e-01,8.306326770750650512e-01,8.305797336510467588e-01,8.305267981722967985e-01,8.304738706468277609e-01,8.304209510826845442e-01,8.303680394879441318e-01,8.303151358707169249e-01,8.302622402391446332e-01,8.302093526014014957e-01,8.301564729656945030e-01,8.301036013402626201e-01,8.300507377333766756e-01,8.299978821533403606e-01,8.299450346084887853e-01,8.298921951071888126e-01,8.298393636578396126e-01,8.297865402688725522e-01,8.297337249487501953e-01,8.296809177059670803e-01,8.296281185490501642e-01,8.295753274865563798e-01,8.295225445270764109e-01,8.294697696792306951e-01,8.294170029516718667e-01,8.293642443530840902e-01,8.293114938921829493e-01,8.292587515777146701e-01,8.292060174184574528e-01,8.291532914232206952e-01,8.291005736008444371e-01,
        8.290478639602000266e-01,8.289951625101903421e-01,8.289424692597484601e-01,8.288897842178386544e-01,8.288371073934563960e-01,8.287844387956272429e-01,8.287317784334085058e-01,8.286791263158874710e-01,8.286264824521817340e-01,8.285738468514404209e-01,8.285212195228426335e-01,8.284686004755978939e-01,8.284159897189462551e-01,8.283633872621581906e-01,8.283107931145341496e-01,8.282582072854053346e-01,8.282056297841323689e-01,8.281530606201067402e-01,8.281004998027499120e-01,8.280479473415121028e-01,8.279954032458759494e-01,8.279428675253512893e-01,8.278903401894794900e-01,8.278378212478313403e-01,8.277853107100068275e-01,8.277328085856359152e-01,8.276803148843784319e-01,8.276278296159232939e-01,8.275753527899893935e-01,8.275228844163242670e-01,8.274704245047059814e-01,8.274179730649399156e-01,8.273655301068627566e-01,8.273130956403395020e-01,8.272606696752642375e-01,8.272082522215599143e-01,8.271558432891791268e-01,8.271034428881026690e-01,8.270510510283407557e-01,8.269986677199320235e-01,8.269462929729440859e-01,8.268939267974731999e-01,8.268415692036445996e-01,8.267892202016110526e-01,8.267368798015550801e-01,8.266845480136874036e-01,8.266322248482462776e-01,8.265799103154986005e-01,8.265276044257409138e-01,8.264753071892960712e-01,8.264230186165157921e-01,8.263707387177804398e-01,8.263184675034974669e-01,8.262662049841028589e-01,8.262139511700608008e-01,8.261617060718622341e-01,8.261094697000270770e-01,8.260572420651026704e-01,8.260050231776632224e-01,8.259528130483112518e-01,8.259006116876768111e-01,8.258484191064175972e-01,8.257962353152176194e-01,8.257440603247897526e-01,8.256918941458731842e-01,8.256397367892344130e-01,8.255875882656673603e-01,8.255354485859932590e-01,8.254833177610594319e-01,8.254311958017414019e-01,8.253790827189405599e-01,8.253269785235860523e-01,8.252748832266326717e-01,8.252227968390631885e-01,8.251707193718863520e-01,8.251186508361374461e-01,8.250665912428785109e-01,8.250145406031980100e-01,8.249624989282104970e-01,8.249104662290575041e-01,8.248584425169059875e-01,8.248064278029504370e-01,8.247544220984099894e-01,8.247024254145305378e-01,8.246504377625842874e-01,8.245984591538690900e-01,8.245464895997081101e-01,8.244945291114519348e-01,8.244425777004749101e-01,8.243906353781784713e-01,8.243387021559893668e-01,8.242867780453595472e-01,8.242348630577670532e-01,8.241829572047145724e-01,8.241310604977309939e-01,8.240791729483700756e-01,8.240272945682105554e-01,8.239754253688568175e-01,8.239235653619385591e-01,8.238717145591096802e-01,8.238198729720495050e-01,8.237680406124628929e-01,8.237162174920780178e-01,8.236644036226491439e-01,8.236125990159551824e-01,8.235608036837986923e-01,8.235090176380081006e-01,8.234572408904351493e-01,8.234054734529567821e-01,8.233537153374738127e-01,8.233019665559120348e-01,8.232502271202212230e-01,8.231984970423744663e-01,8.231467763343703892e-01,8.230950650082302644e-01,8.230433630760009001e-01,8.229916705497511975e-01,8.229399874415753713e-01,8.228883137635907286e-01,8.228366495279381132e-01,8.227849947467824610e-01,8.227333494323119112e-01,8.226817135967383621e-01,8.226300872522970264e-01,8.225784704112466539e-01,8.225268630858684205e-01,8.224752652884679271e-01,8.224236770313734235e-01,8.223720983269360296e-01,8.223205291875294032e-01,8.222689696255519598e-01,8.222174196534227653e-01,8.221658792835850882e-01,8.221143485285046237e-01,8.220628274006694935e-01,8.220113159125906899e-01,8.219598140768014094e-01,8.219083219058576084e-01,8.218568394123373366e-01,8.218053666088410703e-01,8.217539035079918230e-01,8.217024501224342581e-01,8.216510064648353540e-01,8.215995725478845157e-01,8.215481483842922428e-01,8.214967339867916829e-01,8.214453293681374113e-01,8.213939345411055415e-01,8.213425495184946135e-01,8.212911743131242615e-01,8.212398089378354360e-01,8.211884534054907370e-01,8.211371077289746356e-01,8.210857719211920314e-01,8.210344459950693619e-01,8.209831299635550472e-01,8.209318238396172696e-01,8.208805276362464154e-01,8.208292413664528553e-01,8.207779650432682761e-01,8.207266986797455699e-01,8.206754422889573908e-01,8.206241958839978201e-01,8.205729594779810343e-01,8.205217330840428591e-01,8.204705167153374390e-01,8.204193103850411228e-01,8.203681141063500215e-01,8.203169278924802299e-01,8.202657517566682710e-01,8.202145857121704298e-01,8.201634297722629752e-01,8.201122839502423822e-01,8.200611482594249990e-01,8.200100227131469355e-01,8.199589073247631754e-01,8.199078021076494638e-01,8.198567070752004193e-01,8.198056222408306448e-01,8.197545476179735058e-01,8.197034832200822407e-01,8.196524290606284069e-01,8.196013851531045447e-01,8.195503515110204029e-01,8.194993281479059366e-01,8.194483150773089752e-01,8.193973123127968883e-01,8.193463198679564741e-01,8.192953377563919615e-01,8.192443659917274523e-01,8.191934045876041459e-01,8.191424535576831145e-01,8.190915129156428609e-01,8.190405826751809837e-01,8.189896628500129561e-01,8.189387534538724589e-01,8.188878545005112697e-01,8.188369660036987074e-01,8.187860879772232980e-01,8.187352204348899987e-01,8.186843633905228623e-01,8.186335168579622623e-01,8.185826808510675567e-01,8.185318553837149791e-01,8.184810404697983044e-01,8.184302361232288492e-01,8.183794423579346944e-01,8.183286591878617955e-01,8.182778866269733165e-01,8.182271246892492966e-01,8.181763733886860956e-01,8.181256327392990579e-01,8.180749027551180719e-01,8.180241834501906784e-01,8.179734748385816268e-01,8.179227769343715426e-01,8.178720897516577049e-01,8.178214133045550449e-01,8.177707476071927051e-01,8.177200926737177022e-01,8.176694485182930405e-01,8.176188151550977112e-01,8.175681925983259157e-01,8.175175808621899520e-01,8.174669799609158849e-01,8.174163899087469876e-01,8.173658107199409661e-01,8.173152424087726242e-01,8.172646849895315313e-01,8.172141384765228000e-01,8.171636028840667532e-01,8.171130782265000336e-01,8.170625645181732732e-01,8.170120617734535351e-01,8.169615700067214270e-01,8.169110892323743212e-01,8.168606194648231345e-01,8.168101607184943269e-01,8.167597130078292356e-01,8.167092763472828532e-01,8.166588507513264927e-01,8.166084362344443459e-01,8.165580328111362585e-01,8.165076404959153988e-01,8.164572593033099235e-01,8.164068892478619777e-01,8.163565303441281396e-01,8.163061826066786431e-01,8.162558460500973778e-01,8.162055206889828884e-01,8.161552065379468202e-01,8.161049036116148070e-01,8.160546119246262498e-01,8.160043314916338719e-01,8.159540623273040527e-01,8.159038044463157169e-01,8.158535578633629992e-01,8.158033225931504706e-01,8.157530986503984671e-01,8.157028860498388712e-01,8.156526848062163326e-01,8.156024949342899344e-01,8.155523164488298615e-01,8.155021493646197328e-01,8.154519936964559346e-01,8.154018494591467325e-01,8.153517166675136041e-01,8.153015953363897950e-01,8.152514854806216515e-01,8.152013871150668445e-01,8.151513002545948128e-01,8.151012249140883181e-01,8.150511611084417796e-01,8.150011088525603853e-01,8.149510681613620910e-01,8.149010390497762879e-01,8.148510215327442463e-01,8.148010156252173397e-01,8.147510213421610414e-01,8.147010386985493735e-01,8.146510677093694586e-01,8.146011083896189664e-01,8.145511607543063359e-01,8.145012248184518855e-01,8.144513005970857034e-01,8.144013881052495352e-01,8.143514873579960067e-01,8.143015983703874028e-01,8.142517211574978875e-01,8.142018557344112839e-01,8.141520021162218512e-01,8.141021603180338406e-01,8.140523303549627165e-01,8.140025122421338244e-01,8.139527059946818355e-01,8.139029116277517462e-01,8.138531291564988779e-01,8.138033585960876559e-01,8.137535999616930527e-01,8.137038532684984782e-01,8.136541185316980007e-01,8.136043957664947923e-01,8.135546849881006848e-01,8.135049862117377240e-01,8.134552994526369485e-01,8.134056247260386119e-01,8.133559620471907392e-01,8.133063114313520137e-01,8.132566728937885570e-01,8.132070464497765938e-01,8.131574321145992323e-01,8.131078299035501278e-01,8.130582398319301518e-01,8.130086619150486138e-01,8.129590961682233718e-01,8.129095426067811658e-01,8.128600012460553970e-01,8.128104721013885703e-01,8.127609551881314065e-01,8.127114505216411766e-01,8.126619581172841444e-01,8.126124779904337903e-01,8.125630101564710328e-01,8.125135546307846734e-01,8.124641114287706190e-01,8.124146805658324366e-01,8.123652620573805772e-01,8.123158559188325967e-01,8.122664621656138229e-01,8.122170808131555786e-01,8.121677118768968473e-01,8.121183553722829407e-01,8.120690113147660538e-01,8.120196797198047101e-01,8.119703606028647602e-01,8.119210539794173842e-01,8.118717598649409783e-01,8.118224782749196011e-01,8.117732092248445275e-01,8.117239527302115842e-01,8.116747088065237037e-01,8.116254774692891472e-01,8.115762587340219492e-01,8.115270526162428055e-01,8.114778591314766309e-01,8.114286782952555566e-01,8.113795101231150442e-01,8.113303546305975500e-01,8.112812118332503042e-01,8.112320817466255329e-01,8.111829643862812356e-01,8.111338597677794082e-01,8.110847679066877092e-01,8.110356888185780155e-01,8.109866225190275335e-01,8.109375690236176881e-01,8.108885283479346784e-01,8.108395005075688111e-01,8.107904855181151671e-01,8.107414833951729349e-01,8.106924941543455221e-01,8.106435178112403328e-01,8.105945543814685461e-01,8.105456038806458929e-01,8.104966663243911018e-01,8.104477417283272311e-01,8.103988301080811141e-01,8.103499314792826924e-01,8.103010458575649055e-01,8.102521732585652448e-01,8.102033136979236438e-01,8.101544671912834783e-01,8.101056337542907881e-01,8.100568134025951661e-01,8.100080061518489805e-01,8.099592120177072641e-01,8.099104310158278253e-01,8.098616631618712480e-01,8.098129084715003367e-01,8.097641669603802272e-01,8.097154386441791640e-01,8.096667235385668349e-01,8.096180216592158141e-01,8.095693330218001194e-01,8.095206576419962108e-01,8.094719955354815477e-01,8.094233467179373642e-01,8.093747112050441173e-01,8.093260890124855944e-01,8.092774801559466935e-01,8.092288846511135336e-01,8.091803025136738992e-01,8.091317337593161296e-01,8.090831784037305630e-01,8.090346364626082032e-01,8.089861079516413866e-01,8.089375928865225607e-01,8.088890912829458379e-01,8.088406031566052201e-01,
        8.087921285231960411e-01,8.087436673984134128e-01,8.086952197979543344e-01,8.086467857375135848e-01,8.085983652327890514e-01,8.085499582994765122e-01,8.085015649532729665e-01,8.084531852098753024e-01,8.084048190849797422e-01,8.083564665942828409e-01,8.083081277534802656e-01,8.082598025782683493e-01,8.082114910843416489e-01,8.081631932873947211e-01,8.081149092031213454e-01,8.080666388472147466e-01,8.080183822353671497e-01,8.079701393832695588e-01,8.079219103066124230e-01,8.078736950210847478e-01,8.078254935423739846e-01,8.077773058861670297e-01,8.077291320681484477e-01,8.076809721040022483e-01,8.076328260094099987e-01,8.075846938000519337e-01,8.075365754916070671e-01,8.074884710997508597e-01,8.074403806401591055e-01,8.073923041285033797e-01,8.073442415804547023e-01,8.072961930116805407e-01,8.072481584378470298e-01,8.072001378746174183e-01,8.071521313376520679e-01,8.071041388426098973e-01,8.070561604051462723e-01,8.070081960409131172e-01,8.069602457655606909e-01,8.069123095947355884e-01,8.068643875440815183e-01,8.068164796292395247e-01,8.067685858658458775e-01,8.067207062695351816e-01,8.066728408559374897e-01,8.066249896406798570e-01,8.065771526393855639e-01,8.065293298676737832e-01,8.064815213411609118e-01,8.064337270754583509e-01,8.063859470861736156e-01,8.063381813889108907e-01,8.062904299992692536e-01,8.062426929328445624e-01,8.061949702052267908e-01,8.061472618320034700e-01,8.060995678287555810e-01,8.060518882110605521e-01,8.060042229944909264e-01,8.059565721946144734e-01,8.059089358269935222e-01,8.058613139071862941e-01,8.058137064507453484e-01,8.057661134732175823e-01,8.057185349901454519e-01,8.056709710170658623e-01,8.056234215695100565e-01,8.055758866630033932e-01,8.055283663130666794e-01,8.054808605352132833e-01,8.054333693449526876e-01,8.053858927577866034e-01,8.053384307892120786e-01,8.052909834547196111e-01,8.052435507697931483e-01,8.051961327499101984e-01,8.051487294105439396e-01,8.051013407671578914e-01,8.050539668352110212e-01,8.050066076301556350e-01,8.049592631674364895e-01,8.049119334624917910e-01,8.048646185307531953e-01,8.048173183876448089e-01,8.047700330485842990e-01,8.047227625289814501e-01,8.046755068442388303e-01,8.046282660097520134e-01,8.045810400409088015e-01,8.045338289530897802e-01,8.044866327616672086e-01,8.044394514820063513e-01,8.043922851294644794e-01,8.043451337193908701e-01,8.042979972671258082e-01,8.042508757880035830e-01,8.042037692973482699e-01,8.041566778104768387e-01,8.041096013426973776e-01,8.040625399093094261e-01,8.040154935256049740e-01,8.039684622068656861e-01,8.039214459683661218e-01,8.038744448253711816e-01,8.038274587931364401e-01,8.037804878869100333e-01,8.037335321219284401e-01,8.036865915134224769e-01,8.036396660766100819e-01,8.035927558267023096e-01,8.035458607788996677e-01,8.034989809483936707e-01,8.034521163503656194e-01,8.034052669999878216e-01,8.033584329124219270e-01,8.033116141028203705e-01,8.032648105863255950e-01,8.032180223780694961e-01,8.031712494931745328e-01,8.031244919467513954e-01,8.030777497539030030e-01,8.030310229297193958e-01,8.029843114892813993e-01,8.029376154476590699e-01,8.028909348199114726e-01,8.028442696210867924e-01,8.027976198662233331e-01,8.027509855703471864e-01,8.027043667484741185e-01,8.026577634156086827e-01,8.026111755867443298e-01,8.025646032768626315e-01,8.025180465009347230e-01,8.024715052739194165e-01,8.024249796107639776e-01,8.023784695264052358e-01,8.023319750357666980e-01,8.022854961537608798e-01,8.022390328952883065e-01,8.021925852752376240e-01,8.021461533084850437e-01,8.020997370098947865e-01,8.020533363943187499e-01,8.020069514765972851e-01,8.019605822715564214e-01,8.019142287940118630e-01,8.018678910587653252e-01,8.018215690806062002e-01,8.017752628743115562e-01,8.017289724546443619e-01,8.016826978363564837e-01,8.016364390341853552e-01,8.015901960628557532e-01,8.015439689370790211e-01,8.014977576715535124e-01,8.014515622809649242e-01,8.014053827799835217e-01,8.013592191832676903e-01,8.013130715054617159e-01,8.012669397611965616e-01,8.012208239650889796e-01,8.011747241317412893e-01,8.011286402757431535e-01,8.010825724116696911e-01,8.010365205540808109e-01,8.009904847175238762e-01,8.009444649165309293e-01,8.008984611656202457e-01,8.008524734792952238e-01,8.008065018720443851e-01,8.007605463583428174e-01,8.007146069526496213e-01,8.006686836694101306e-01,8.006227765230538029e-01,8.005768855279956631e-01,8.005310106986363028e-01,8.004851520493603267e-01,8.004393095945366854e-01,8.003934833485206735e-01,8.003476733256509323e-01,8.003018795402513375e-01,8.002561020066294439e-01,8.002103407390780410e-01,8.001645957518738195e-01,8.001188670592774832e-01,8.000731546755348589e-01,8.000274586148740097e-01,7.999817788915088990e-01,7.999361155196363926e-01,7.998904685134374803e-01,7.998448378870766096e-01,7.997992236547021294e-01,7.997536258304460688e-01,7.997080444284233591e-01,7.996624794627333888e-01,7.996169309474577824e-01,7.995713988966622887e-01,7.995258833243952257e-01,7.994803842446884801e-01,7.994349016715566192e-01,7.993894356189972239e-01,7.993439861009907776e-01,7.992985531315008885e-01,7.992531367244737339e-01,7.992077368938370618e-01,7.991623536535030770e-01,7.991169870173645551e-01,7.990716369992982848e-01,7.990263036131619589e-01,7.989809868727969500e-01,7.989356867920253125e-01,7.988904033846522257e-01,7.988451366644649942e-01,7.987998866452321600e-01,7.987546533407037241e-01,7.987094367646132564e-01,7.986642369306738987e-01,7.986190538525816951e-01,7.985738875440141493e-01,7.985287380186300021e-01,7.984836052900697867e-01,7.984384893719543852e-01,7.983933902778866942e-01,7.983483080214505145e-01,7.983032426162113282e-01,7.982581940757150774e-01,7.982131624134881642e-01,7.981681476430390054e-01,7.981231497778559225e-01,7.980781688314084743e-01,7.980332048171465686e-01,7.979882577485007955e-01,7.979433276388822049e-01,7.978984145016824181e-01,7.978535183502730721e-01,7.978086391980068193e-01,7.977637770582149956e-01,7.977189319442109516e-01,7.976741038692869434e-01,7.976292928467154653e-01,7.975844988897488053e-01,7.975397220116194896e-01,7.974949622255393944e-01,7.974502195446999675e-01,7.974054939822730059e-01,7.973607855514093234e-01,7.973160942652394167e-01,7.972714201368725773e-01,7.972267631793986675e-01,7.971821234058854566e-01,7.971375008293808406e-01,7.970928954629112884e-01,7.970483073194835066e-01,7.970037364120808876e-01,7.969591827536682827e-01,7.969146463571873396e-01,7.968701272355601661e-01,7.968256254016864437e-01,7.967811408684448704e-01,7.967366736486923839e-01,7.966922237552651609e-01,7.966477912009773954e-01,7.966033759986210772e-01,7.965589781609675457e-01,7.965145977007661582e-01,7.964702346307430680e-01,7.964258889636044447e-01,7.963815607120338091e-01,7.963372498886917006e-01,7.962929565062177861e-01,7.962486805772290843e-01,7.962044221143198541e-01,7.961601811300632603e-01,7.961159576370090418e-01,7.960717516476848443e-01,7.960275631745956648e-01,7.959833922302247400e-01,7.959392388270311036e-01,7.958951029774526953e-01,7.958509846939036958e-01,7.958068839887753043e-01,7.957628008744367376e-01,7.957187353632337867e-01,7.956746874674888170e-01,7.956306571995018784e-01,7.955866445715493729e-01,7.955426495958841659e-01,7.954986722847368075e-01,7.954547126503136445e-01,7.954107707047979314e-01,7.953668464603494970e-01,7.953229399291046331e-01,7.952790511231755399e-01,7.952351800546515470e-01,7.951913267355978920e-01,7.951474911780562760e-01,7.951036733940437529e-01,7.950598733955541730e-01,7.950160911945579612e-01,7.949723268029998957e-01,7.949285802328025508e-01,7.948848514958628542e-01,7.948411406040541971e-01,7.947974475692254348e-01,7.947537724032014417e-01,7.947101151177822231e-01,7.946664757247443589e-01,7.946228542358385605e-01,7.945792506627915586e-01,7.945356650173061031e-01,7.944920973110591866e-01,7.944485475557035992e-01,7.944050157628673725e-01,7.943615019441534475e-01,7.943180061111401180e-01,7.942745282753809200e-01,7.942310684484032990e-01,7.941876266417107200e-01,7.941442028667813346e-01,7.941007971350674266e-01,7.940574094579968545e-01,7.940140398469714977e-01,7.939706883133679227e-01,7.939273548685387150e-01,7.938840395238082603e-01,7.938407422904776300e-01,7.937974631798218050e-01,7.937542022030897870e-01,7.937109593715048206e-01,7.936677346962645041e-01,7.936245281885411229e-01,7.935813398594803170e-01,7.935381697202024132e-01,7.934950177818017591e-01,7.934518840553458352e-01,7.934087685518770305e-01,7.933656712824116441e-01,7.933225922579386635e-01,7.932795314894222072e-01,7.932364889877993042e-01,7.931934647639810043e-01,7.931504588288514901e-01,7.931074711932692978e-01,7.930645018680652081e-01,7.930215508640452438e-01,7.929786181919875609e-01,7.929357038626434484e-01,7.928928078867386597e-01,7.928499302749714150e-01,7.928070710380134001e-01,7.927642301865087671e-01,7.927214077310763551e-01,7.926786036823063597e-01,7.926358180507633300e-01,7.925930508469839486e-01,7.925503020814778088e-01,7.925075717647279694e-01,7.924648599071897337e-01,7.924221665192914266e-01,7.923794916114342834e-01,7.923368351939920062e-01,7.922941972773107633e-01,7.922515778717098556e-01,7.922089769874803844e-01,7.921663946348861396e-01,7.921238308241642656e-01,7.920812855655230411e-01,7.920387588691439884e-01,7.919962507451805411e-01,7.919537612037584884e-01,7.919112902549755306e-01,7.918688379089023899e-01,7.918264041755807003e-01,7.917839890650255619e-01,7.917415925872230975e-01,7.916992147521317857e-01,7.916568555696820164e-01,7.916145150497764238e-01,7.915721932022887763e-01,7.915298900370657531e-01,7.914876055639243901e-01,7.914453397926549671e-01,7.914030927330186760e-01,7.913608643947482868e-01,7.913186547875484811e-01,7.912764639210957407e-01,7.912342918050373486e-01,7.911921384489924991e-01,7.911500038625524089e-01,7.911078880552789849e-01,7.910657910367054901e-01,7.910237128163368769e-01,7.909816534036495650e-01,7.909396128080908861e-01,7.908975910390793063e-01,7.908555881060046477e-01,7.908136040182285331e-01,7.907716387850820539e-01,7.907296924158689899e-01,
        7.906877649198637004e-01,7.906458563063112344e-01,7.906039665844277753e-01,7.905620957633999746e-01,7.905202438523869501e-01,7.904784108605166226e-01,7.904365967968884910e-01,7.903948016705744095e-01,7.903530254906142583e-01,7.903112682660200505e-01,7.902695300057751560e-01,7.902278107188320799e-01,7.901861104141151282e-01,7.901444291005181864e-01,7.901027667869073845e-01,7.900611234821165452e-01,7.900194991949531786e-01,7.899778939341923767e-01,7.899363077085821416e-01,7.898947405268386124e-01,7.898531923976502833e-01,7.898116633296740074e-01,7.897701533315385491e-01,7.897286624118415865e-01,7.896871905791524870e-01,7.896457378420095319e-01,7.896043042089218034e-01,7.895628896883681858e-01,7.895214942887979204e-01,7.894801180186296063e-01,7.894387608862531991e-01,7.893974229000277898e-01,7.893561040682823826e-01,7.893148043993158947e-01,7.892735239013974891e-01,7.892322625827667970e-01,7.891910204516318084e-01,7.891497975161710920e-01,7.891085937845331300e-01,7.890674092648364280e-01,7.890262439651690718e-01,7.889850978935877279e-01,7.889439710581208631e-01,7.889028634667651918e-01,7.888617751274866752e-01,7.888207060482221866e-01,7.887796562368776243e-01,7.887386257013282442e-01,7.886976144494189933e-01,7.886566224889643983e-01,7.886156498277482330e-01,7.885746964735241837e-01,7.885337624340146290e-01,7.884928477169124150e-01,7.884519523298783028e-01,7.884110762805444095e-01,7.883702195765105447e-01,7.883293822253463201e-01,7.882885642345904831e-01,7.882477656117521381e-01,7.882069863643079710e-01,7.881662264997050249e-01,7.881254860253590344e-01,7.880847649486556472e-01,7.880440632769484255e-01,7.880033810175609554e-01,7.879627181777862921e-01,7.879220747648859602e-01,7.878814507860903982e-01,7.878408462485990693e-01,7.878002611595816829e-01,7.877596955261756406e-01,7.877191493554879242e-01,7.876786226545939851e-01,7.876381154305392984e-01,7.875976276903372542e-01,7.875571594409707110e-01,7.875167106893907754e-01,7.874762814425190216e-01,7.874358717072434954e-01,7.873954814904234878e-01,7.873551107988856490e-01,7.873147596394257652e-01,7.872744280188088695e-01,7.872341159437680203e-01,7.871938234210057450e-01,7.871535504571935959e-01,7.871132970589703737e-01,7.870730632329445697e-01,7.870328489856946996e-01,7.869926543237655281e-01,7.869524792536722879e-01,7.869123237818975714e-01,7.868721879148937726e-01,7.868320716590813113e-01,7.867919750208495211e-01,7.867518980065562051e-01,7.867118406225277472e-01,7.866718028750590008e-01,7.866317847704138444e-01,7.865917863148244038e-01,7.865518075144911636e-01,7.865118483755834111e-01,7.864719089042386813e-01,7.864319891065638668e-01,7.863920889886332199e-01,7.863522085564906838e-01,7.863123478161471169e-01,7.862725067735837348e-01,7.862326854347487792e-01,7.861928838055589619e-01,7.861531018919007963e-01,7.861133396996279332e-01,7.860735972345627154e-01,7.860338745024963991e-01,7.859941715091876002e-01,7.859544882603650695e-01,7.859148247617236960e-01,7.858751810189292808e-01,7.858355570376134303e-01,7.857959528233785518e-01,7.857563683817931910e-01,7.857168037183963616e-01,7.856772588386934375e-01,7.856377337481595946e-01,7.855982284522372572e-01,7.855587429563384294e-01,7.855192772658426970e-01,7.854798313860974490e-01,7.854404053224195437e-01,7.854009990800931984e-01,7.853616126643714335e-01,7.853222460804756277e-01,7.852828993335949637e-01,7.852435724288872043e-01,7.852042653714785825e-01,7.851649781664632455e-01,7.851257108189042544e-01,7.850864633338323628e-01,7.850472357162464609e-01,7.850080279711139086e-01,7.849688401033707574e-01,7.849296721179214176e-01,7.848905240196373256e-01,7.848513958133592761e-01,7.848122875038959778e-01,7.847731990960251647e-01,7.847341305944912637e-01,7.846950820040082819e-01,7.846560533292579187e-01,7.846170445748904543e-01,7.845780557455235282e-01,7.845390868457446931e-01,7.845001378801085279e-01,7.844612088531381922e-01,7.844222997693245381e-01,7.843834106331277756e-01,7.843445414489753631e-01,7.843056922212641169e-01,7.842668629543579906e-01,7.842280536525897405e-01,7.841892643202607038e-01,7.841504949616396880e-01,7.841117455809646364e-01,7.840730161824409628e-01,7.840343067702431057e-01,7.839956173485134183e-01,7.839569479213621683e-01,7.839182984928688702e-01,7.838796690670803979e-01,7.838410596480127612e-01,7.838024702396493293e-01,7.837639008459424961e-01,7.837253514708131252e-01,7.836868221181494398e-01,7.836483127918090208e-01,7.836098234956171416e-01,7.835713542333675452e-01,7.835329050088228886e-01,7.834944758257132991e-01,7.834560666877379287e-01,7.834176775985636221e-01,7.833793085618261376e-01,7.833409595811297033e-01,7.833026306600465727e-01,7.832643218021172471e-01,7.832260330108515856e-01,7.831877642897265845e-01,7.831495156421884873e-01,7.831112870716514518e-01,7.830730785814991046e-01,7.830348901750817658e-01,7.829967218557200015e-01,7.829585736267016260e-01,7.829204454912835898e-01,7.828823374526909795e-01,7.828442495141172408e-01,7.828061816787249549e-01,7.827681339496445068e-01,7.827301063299753059e-01,7.826920988227851206e-01,7.826541114311104108e-01,7.826161441579556621e-01,7.825781970062943849e-01,7.825402699790692251e-01,7.825023630791898555e-01,7.824644763095357503e-01,7.824266096729552977e-01,7.823887631722646896e-01,7.823509368102483652e-01,7.823131305896607879e-01,7.822753445132244465e-01,7.822375785836301887e-01,7.821998328035381087e-01,7.821621071755761045e-01,7.821244017023420980e-01,7.820867163864014815e-01,7.820490512302895603e-01,7.820114062365094432e-01,7.819737814075338189e-01,7.819361767458034018e-01,7.818985922537279309e-01,7.818610279336866142e-01,7.818234837880267962e-01,7.817859598190646242e-01,7.817484560290860474e-01,7.817109724203445964e-01,7.816735089950636040e-01,7.816360657554355384e-01,7.815986427036208939e-01,7.815612398417495221e-01,7.815238571719207439e-01,7.814864946962019054e-01,7.814491524166304881e-01,7.814118303352122208e-01,7.813745284539219682e-01,7.813372467747036199e-01,7.812999852994705341e-01,7.812627440301052051e-01,7.812255229684587077e-01,7.811883221163514746e-01,7.811511414755735183e-01,7.811139810478833212e-01,7.810768408350089453e-01,7.810397208386478107e-01,7.810026210604665842e-01,7.809655415021004021e-01,7.809284821651545361e-01,7.808914430512038374e-01,7.808544241617914050e-01,7.808174254984305840e-01,7.807804470626034110e-01,7.807434888557617247e-01,7.807065508793268327e-01,7.806696331346890672e-01,7.806327356232084513e-01,7.805958583462146994e-01,7.805590013050065501e-01,7.805221645008525444e-01,7.804853479349910250e-01,7.804485516086289154e-01,7.804117755229440512e-01,7.803750196790827376e-01,7.803382840781619700e-01,7.803015687212671025e-01,7.802648736094542903e-01,7.802281987437486022e-01,7.801915441251459082e-01,7.801549097546105482e-01,7.801182956330768858e-01,7.800817017614506410e-01,7.800451281406044490e-01,7.800085747713841888e-01,7.799720416546025437e-01,7.799355287910437751e-01,7.798990361814619465e-01,7.798625638265808124e-01,7.798261117270940401e-01,7.797896798836655430e-01,7.797532682969288143e-01,7.797168769674874822e-01,7.796805058959163093e-01,7.796441550827583056e-01,7.796078245285281705e-01,7.795715142337105164e-01,7.795352241987590913e-01,7.794989544240994439e-01,7.794627049101257033e-01,7.794264756572031327e-01,7.793902666656681300e-01,7.793540779358252291e-01,7.793179094679515417e-01,7.792817612622934265e-01,7.792456333190674878e-01,7.792095256384615753e-01,7.791734382206334519e-01,7.791373710657112373e-01,7.791013241737938522e-01,7.790652975449511297e-01,7.790292911792225938e-01,7.789933050766191247e-01,7.789573392371218485e-01,7.789213936606825817e-01,7.788854683472247187e-01,7.788495632966401239e-01,7.788136785087943492e-01,7.787778139835217495e-01,7.787419697206280356e-01,7.787061457198897196e-01,7.786703419810541149e-01,7.786345585038403350e-01,7.785987952879368512e-01,7.785630523330043795e-01,7.785273296386739927e-01,7.784916272045480090e-01,7.784559450302004358e-01,7.784202831151751933e-01,7.783846414589882245e-01,7.783490200611261622e-01,7.783134189210469955e-01,7.782778380381799588e-01,7.782422774119255315e-01,7.782067370416556606e-01,7.781712169267134271e-01,7.781357170664133793e-01,7.781002374600414218e-01,7.780647781068545932e-01,7.780293390060820657e-01,7.779939201569240348e-01,7.779585215585521629e-01,7.779231432101096910e-01,7.778877851107117714e-01,7.778524472594450234e-01,7.778171296553679781e-01,7.777818322975097454e-01,7.777465551848730119e-01,7.777112983164309323e-01,7.776760616911283508e-01,7.776408453078826888e-01,7.776056491655830571e-01,7.775704732630901450e-01,7.775353175992372190e-01,7.775001821728282358e-01,7.774650669826409510e-01,7.774299720274233660e-01,7.773948973058976142e-01,7.773598428167558527e-01,7.773248085586637046e-01,7.772897945302585931e-01,7.772548007301498529e-01,7.772198271569198402e-01,7.771848738091230446e-01,7.771499406852854230e-01,7.771150277839065090e-01,7.770801351034573035e-01,7.770452626423820508e-01,7.770104103990964628e-01,7.769755783719899389e-01,7.769407665594240120e-01,7.769059749597323483e-01,7.768712035712218578e-01,7.768364523921714726e-01,7.768017214208340349e-01,7.767670106554332987e-01,7.767323200941680383e-01,7.766976497352079400e-01,7.766629995766965999e-01,7.766283696167500805e-01,7.765937598534574660e-01,7.765591702848807509e-01,7.765246009090559509e-01,7.764900517239908817e-01,7.764555227276662697e-01,7.764210139180370840e-01,7.763865252930313154e-01,7.763520568505491992e-01,7.763176085884652133e-01,7.762831805046267464e-01,7.762487725968545416e-01,7.762143848629426968e-01,7.761800173006587755e-01,7.761456699077441401e-01,7.761113426819125083e-01,7.760770356208527287e-01,7.760427487222260057e-01,7.760084819836676751e-01,7.759742354027866496e-01,7.759400089771655296e-01,7.759058027043608252e-01,7.758716165819026234e-01,7.758374506072944765e-01,7.758033047780149571e-01,7.757691790915148822e-01,7.757350735452203105e-01,7.757009881365313220e-01,7.756669228628211288e-01,7.756328777214375192e-01,7.755988527097023022e-01,7.755648478249119737e-01,
        7.755308630643358292e-01,7.754968984252189612e-01,7.754629539047801501e-01,7.754290295002123079e-01,7.753951252086822565e-01,7.753612410273329481e-01,7.753273769532795789e-01,7.752935329836134759e-01,7.752597091154000974e-01,7.752259053456784788e-01,7.751921216714638962e-01,7.751583580897452030e-01,7.751246145974861612e-01,7.750908911916255528e-01,7.750571878690760697e-01,7.750235046267268668e-01,7.749898414614402320e-01,7.749561983700549161e-01,7.749225753493834690e-01,7.748889723962134601e-01,7.748553895073085895e-01,7.748218266794067999e-01,7.747882839092207208e-01,7.747547611934400003e-01,7.747212585287268638e-01,7.746877759117211104e-01,7.746543133390370039e-01,7.746208708072633842e-01,7.745874483129661092e-01,7.745540458526848360e-01,7.745206634229361287e-01,7.744873010202110164e-01,7.744539586409768805e-01,7.744206362816757894e-01,7.743873339387266075e-01,7.743540516085229974e-01,7.743207892874349740e-01,7.742875469718080161e-01,7.742543246579635108e-01,7.742211223421987532e-01,7.741879400207870576e-01,7.741547776899775357e-01,7.741216353459955402e-01,7.740885129850422208e-01,7.740554106032955239e-01,7.740223281969084157e-01,7.739892657620111027e-01,7.739562232947092557e-01,7.739232007910858968e-01,7.738901982471992902e-01,7.738572156590842743e-01,7.738242530227522620e-01,7.737913103341922394e-01,7.737583875893679908e-01,7.737254847842205407e-01,7.736926019146678213e-01,7.736597389766042276e-01,7.736268959659007294e-01,7.735940728784053144e-01,7.735612697099423229e-01,7.735284864563138907e-01,7.734957231132977284e-01,7.734629796766492316e-01,7.734302561421013689e-01,7.733975525053626843e-01,7.733648687621199613e-01,7.733322049080367799e-01,7.732995609387538494e-01,7.732669368498892304e-01,7.732343326370378911e-01,7.732017482957722621e-01,7.731691838216429025e-01,7.731366392101760576e-01,7.731041144568775447e-01,7.730716095572290891e-01,7.730391245066903227e-01,7.730066593006991171e-01,7.729742139346701402e-01,7.729417884039962994e-01,7.729093827040478537e-01,7.728769968301733018e-01,7.728446307776986046e-01,7.728122845419276299e-01,7.727799581181423738e-01,7.727476515016026282e-01,7.727153646875465354e-01,7.726830976711890342e-01,7.726508504477255235e-01,7.726186230123274212e-01,7.725864153601458284e-01,7.725542274863084202e-01,7.725220593859229989e-01,7.724899110540747182e-01,7.724577824858273045e-01,7.724256736762229458e-01,7.723935846202820699e-01,7.723615153130041211e-01,7.723294657493673387e-01,7.722974359243278686e-01,7.722654258328204291e-01,7.722334354697594216e-01,7.722014648300374873e-01,7.721695139085262838e-01,7.721375827000757086e-01,7.721056711995157862e-01,7.720737794016537814e-01,7.720419073012781963e-01,7.720100548931547735e-01,7.719782221720289384e-01,7.719464091326259103e-01,7.719146157696491484e-01,7.718828420777819055e-01,7.718510880516870065e-01,7.718193536860057380e-01,7.717876389753598465e-01,7.717559439143498734e-01,7.717242684975559319e-01,7.716926127195381513e-01,7.716609765748357885e-01,7.716293600579674505e-01,7.715977631634325373e-01,7.715661858857091326e-01,7.715346282192558913e-01,7.715030901585108181e-01,7.714715716978918225e-01,7.714400728317971634e-01,7.714085935546045603e-01,7.713771338606721928e-01,7.713456937443383676e-01,7.713142731999208523e-01,7.712828722217185407e-01,7.712514908040100092e-01,7.712201289410542948e-01,7.711887866270905612e-01,7.711574638563388762e-01,7.711261606229986576e-01,7.710948769212512266e-01,7.710636127452576982e-01,7.710323680891594256e-01,7.710011429470785549e-01,7.709699373131191358e-01,7.709387511813639016e-01,7.709075845458780440e-01,7.708764374007062159e-01,7.708453097398755283e-01,7.708142015573924422e-01,7.707831128472454330e-01,7.707520436034036582e-01,7.707209938198174015e-01,7.706899634904176288e-01,7.706589526091170983e-01,7.706279611698098053e-01,7.705969891663699833e-01,7.705660365926546573e-01,7.705351034425009793e-01,7.705041897097287817e-01,7.704732953881376911e-01,7.704424204715105695e-01,7.704115649536102950e-01,7.703807288281826482e-01,7.703499120889539808e-01,7.703191147296334362e-01,7.702883367439109508e-01,7.702575781254584752e-01,7.702268388679309741e-01,7.701961189649636497e-01,7.701654184101737188e-01,7.701347371971625222e-01,7.701040753195108612e-01,7.700734327707829951e-01,7.700428095445251975e-01,7.700122056342658672e-01,7.699816210335151956e-01,7.699510557357664986e-01,7.699205097344947735e-01,7.698899830231580310e-01,7.698594755951956303e-01,7.698289874440310543e-01,7.697985185630688010e-01,7.697680689456962710e-01,7.697376385852844338e-01,7.697072274751859400e-01,7.696768356087368979e-01,7.696464629792553191e-01,7.696161095800431173e-01,7.695857754043842203e-01,7.695554604455456804e-01,7.695251646967782300e-01,7.694948881513146155e-01,7.694646308023712633e-01,7.694343926431473912e-01,7.694041736668258968e-01,7.693739738665721362e-01,7.693437932355362552e-01,7.693136317668494151e-01,7.692834894536279000e-01,7.692533662889710078e-01,7.692232622659614938e-01,7.691931773776650161e-01,7.691631116171321336e-01,7.691330649773957528e-01,7.691030374514730150e-01,7.690730290323641860e-01,7.690430397130546547e-01,7.690130694865123795e-01,7.689831183456895536e-01,7.689531862835222720e-01,7.689232732929306424e-01,7.688933793668184524e-01,7.688635044980743904e-01,7.688336486795702696e-01,7.688038119041625817e-01,7.687739941646919428e-01,7.687441954539832034e-01,7.687144157648452270e-01,7.686846550900723329e-01,7.686549134224416324e-01,7.686251907547152484e-01,7.685954870796408711e-01,7.685658023899489821e-01,7.685361366783560744e-01,7.685064899375627645e-01,7.684768621602535710e-01,7.684472533390999116e-01,7.684176634667552186e-01,7.683880925358598235e-01,7.683585405390382927e-01,7.683290074688992055e-01,7.682994933180379293e-01,7.682699980790337335e-01,7.682405217444506773e-01,7.682110643068388312e-01,7.681816257587325003e-01,7.681522060926523343e-01,7.681228053011031065e-01,7.680934233765756014e-01,7.680640603115457266e-01,7.680347160984751786e-01,7.680053907298102223e-01,7.679760841979832442e-01,7.679467964954125314e-01,7.679175276145011608e-01,7.678882775476387756e-01,7.678590462871998090e-01,7.678298338255448163e-01,7.678006401550201421e-01,7.677714652679583640e-01,7.677423091566769608e-01,7.677131718134805327e-01,7.676840532306590248e-01,7.676549534004878383e-01,7.676258723152298291e-01,7.675968099671330869e-01,7.675677663484319346e-01,7.675387414513472617e-01,7.675097352680847473e-01,7.674807477908395237e-01,7.674517790117900695e-01,7.674228289231025402e-01,7.673938975169294352e-01,7.673649847854097095e-01,7.673360907206689951e-01,7.673072153148193797e-01,7.672783585599592948e-01,7.672495204481747377e-01,7.672207009715377168e-01,7.671919001221070289e-01,7.671631178919283700e-01,7.671343542730348908e-01,7.671056092574463081e-01,7.670768828371684611e-01,7.670481750041956426e-01,7.670194857505084896e-01,7.669908150680746495e-01,7.669621629488494463e-01,7.669335293847746593e-01,7.669049143677797442e-01,7.668763178897817223e-01,7.668477399426846253e-01,7.668191805183798282e-01,7.667906396087466048e-01,7.667621172056510170e-01,7.667336133009471366e-01,7.667051278864766006e-01,7.666766609540687227e-01,7.666482124955401600e-01,7.666197825026952462e-01,7.665913709673268794e-01,7.665629778812148576e-01,7.665346032361273210e-01,7.665062470238198644e-01,7.664779092360372026e-01,7.664495898645103944e-01,7.664212889009597296e-01,7.663930063370932855e-01,7.663647421646068159e-01,7.663364963751848613e-01,7.663082689605000830e-01,7.662800599122132628e-01,7.662518692219734140e-01,7.662236968814183369e-01,7.661955428821736191e-01,7.661674072158541904e-01,7.661392898740619906e-01,7.661111908483890787e-01,7.660831101304154123e-01,7.660550477117096246e-01,7.660270035838294689e-01,7.659989777383198195e-01,7.659709701667167803e-01,7.659429808605429102e-01,7.659150098113117755e-01,7.658870570105242859e-01,7.658591224496705818e-01,7.658312061202303678e-01,7.658033080136716908e-01,7.657754281214531611e-01,7.657475664350200661e-01,7.657197229458088117e-01,7.656918976452443681e-01,7.656640905247411588e-01,7.656363015757027268e-01,7.656085307895215131e-01,7.655807781575809656e-01,7.655530436712523201e-01,7.655253273218967092e-01,7.654976291008648293e-01,7.654699489994978290e-01,7.654422870091251996e-01,7.654146431210666623e-01,7.653870173266318355e-01,7.653594096171195682e-01,7.653318199838193836e-01,7.653042484180094807e-01,7.652766949109591765e-01,7.652491594539262421e-01,7.652216420381601214e-01,7.651941426548990455e-01,7.651666612953713642e-01,7.651391979507966568e-01,7.651117526123831780e-01,7.650843252713304121e-01,7.650569159188271851e-01,7.650295245460539961e-01,7.650021511441803534e-01,7.649747957043664393e-01,7.649474582177627768e-01,7.649201386755112297e-01,7.648928370687430034e-01,7.648655533885804214e-01,7.648382876261363705e-01,7.648110397725140785e-01,7.647838098188080025e-01,7.647565977561023853e-01,7.647294035754733654e-01,7.647022272679866450e-01,7.646750688246999328e-01,7.646479282366615005e-01,7.646208054949092947e-01,7.645937005904743788e-01,7.645666135143772690e-01,7.645395442576298217e-01,7.645124928112352336e-01,7.644854591661877086e-01,7.644584433134730128e-01,7.644314452440676977e-01,7.644044649489389887e-01,7.643775024190468947e-01,7.643505576453415440e-01,7.643236306187647378e-01,7.642967213302501728e-01,7.642698297707229971e-01,7.642429559310985887e-01,7.642160998022857754e-01,7.641892613751831709e-01,7.641624406406828385e-01,7.641356375896668496e-01,7.641088522130102811e-01,7.640820845015791063e-01,7.640553344462313046e-01,7.640286020378170839e-01,7.640018872671779926e-01,7.639751901251482513e-01,7.639485106025529770e-01,7.639218486902101812e-01,7.638952043789296598e-01,7.638685776595134369e-01,7.638419685227552103e-01,7.638153769594412390e-01,7.637888029603498996e-01,7.637622465162519081e-01,7.637357076179100979e-01,7.637091862560797528e-01,7.636826824215087184e-01,7.636561961049362912e-01,7.636297272970961059e-01,7.636032759887120269e-01,7.635768421705022568e-01,7.635504258331765604e-01,
        7.635240269674380409e-01,7.634976455639816972e-01,7.634712816134956448e-01,7.634449351066612266e-01,7.634186060341510149e-01,7.633922943866322530e-01,7.633660001547640794e-01,7.633397233291981943e-01,7.633134639005803024e-01,7.632872218595476710e-01,7.632609971967319051e-01,7.632347899027568383e-01,7.632085999682399757e-01,7.631824273837914951e-01,7.631562721400144689e-01,7.631301342275065291e-01,7.631040136368565374e-01,7.630779103586485812e-01,7.630518243834587544e-01,7.630257557018569337e-01,7.629997043044071114e-01,7.629736701816652866e-01,7.629476533241816849e-01,7.629216537225008699e-01,7.628956713671595224e-01,7.628697062486886615e-01,7.628437583576125336e-01,7.628178276844497230e-01,7.627919142197122637e-01,7.627660179539054175e-01,7.627401388775290059e-01,7.627142769810763001e-01,7.626884322550340212e-01,7.626626046898834499e-01,7.626367942760994278e-01,7.626110010041514675e-01,7.625852248645018649e-01,7.625594658476076981e-01,7.625337239439204939e-01,7.625079991438850069e-01,7.624822914379407734e-01,7.624566008165215569e-01,7.624309272700550144e-01,7.624052707889632519e-01,7.623796313636624911e-01,7.623540089845636247e-01,7.623284036420718834e-01,7.623028153265865026e-01,7.622772440285018325e-01,7.622516897382061174e-01,7.622261524460827165e-01,7.622006321425091047e-01,7.621751288178572059e-01,7.621496424624940591e-01,7.621241730667813741e-01,7.620987206210748655e-01,7.620732851157265841e-01,7.620478665410809205e-01,7.620224648874798223e-01,7.619970801452572440e-01,7.619717123047449192e-01,7.619463613562674764e-01,7.619210272901451031e-01,7.618957100966934348e-01,7.618704097662222230e-01,7.618451262890373332e-01,7.618198596554389690e-01,7.617946098557230039e-01,7.617693768801794274e-01,7.617441607190954533e-01,7.617189613627516342e-01,7.616937788014245259e-01,7.616686130253861320e-01,7.616434640249033494e-01,7.616183317902391892e-01,7.615932163116513332e-01,7.615681175793940216e-01,7.615430355837153886e-01,7.615179703148601265e-01,7.614929217630688196e-01,7.614678899185763905e-01,7.614428747716147639e-01,7.614178763124107574e-01,7.613928945311871921e-01,7.613679294181621149e-01,7.613429809635501311e-01,7.613180491575609610e-01,7.612931339904005501e-01,7.612682354522707362e-01,7.612433535333685830e-01,7.612184882238884898e-01,7.611936395140197487e-01,7.611688073939473220e-01,7.611439918538536187e-01,7.611191928839156073e-01,7.610944104743078142e-01,7.610696446151989925e-01,7.610448952967563407e-01,7.610201625091420619e-01,7.609954462425143618e-01,7.609707464870281157e-01,7.609460632328343133e-01,7.609213964700808353e-01,7.608967461889111217e-01,7.608721123794658370e-01,7.608474950318813157e-01,7.608228941362907838e-01,7.607983096828240255e-01,7.607737416616077164e-01,7.607491900627637582e-01,7.607246548764121652e-01,7.607001360926685107e-01,7.606756337016461478e-01,7.606511476934538774e-01,7.606266780581983911e-01,7.606022247859822727e-01,7.605777878669046643e-01,7.605533672910630427e-01,7.605289630485502217e-01,7.605045751294569056e-01,7.604802035238701352e-01,7.604558482218738424e-01,7.604315092135496279e-01,7.604071864889748733e-01,7.603828800382260722e-01,7.603585898513740560e-01,7.603343159184891009e-01,7.603100582296380416e-01,7.602858167748838270e-01,7.602615915442878514e-01,7.602373825279087338e-01,7.602131897158007634e-01,7.601890130980181182e-01,7.601648526646094250e-01,7.601407084056233110e-01,7.601165803111041841e-01,7.600924683710944540e-01,7.600683725756334219e-01,7.600442929147585014e-01,7.600202293785047747e-01,7.599961819569042154e-01,7.599721506399864657e-01,7.599481354177791692e-01,7.599241362803078603e-01,7.599001532175944096e-01,7.598761862196601324e-01,7.598522352765226806e-01,7.598283003781980405e-01,7.598043815147000890e-01,7.597804786760398166e-01,7.597565918522275474e-01,7.597327210332698311e-01,7.597088662091721067e-01,7.596850273699373712e-01,7.596612045055666229e-01,7.596373976060586397e-01,7.596136066614112003e-01,7.595898316616187529e-01,7.595660725966748572e-01,7.595423294565706307e-01,7.595186022312958585e-01,7.594948909108378832e-01,7.594711954851820490e-01,7.594475159443131451e-01,7.594238522782129630e-01,7.594002044768624060e-01,7.593765725302401570e-01,7.593529564283230116e-01,7.593293561610874320e-01,7.593057717185068833e-01,7.592822030905537201e-01,7.592586502671987425e-01,7.592351132384114187e-01,7.592115919941597735e-01,7.591880865244093890e-01,7.591645968191260696e-01,7.591411228682729551e-01,7.591176646618124080e-01,7.590942221897049036e-01,7.590707954419096959e-01,7.590473844083855948e-01,7.590239890790888566e-01,7.590006094439758488e-01,7.589772454930002743e-01,7.589538972161156138e-01,7.589305646032741270e-01,7.589072476444265192e-01,7.588839463295224963e-01,7.588606606485114314e-01,7.588373905913404771e-01,7.588141361479563418e-01,7.587908973083047348e-01,7.587676740623308103e-01,7.587444663999776129e-01,7.587212743111885205e-01,7.586980977859052455e-01,7.586749368140687233e-01,7.586517913856191120e-01,7.586286614904963477e-01,7.586055471186387011e-01,7.585824482599839991e-01,7.585593649044696241e-01,7.585362970420315154e-01,7.585132446626060565e-01,7.584902077561274103e-01,7.584671863125311830e-01,7.584441803217503164e-01,7.584211897737184183e-01,7.583982146583680972e-01,7.583752549656318509e-01,7.583523106854408447e-01,7.583293818077263548e-01,7.583064683224194358e-01,7.582835702194496985e-01,7.582606874887481974e-01,7.582378201202434331e-01,7.582149681038650169e-01,7.581921314295420045e-01,7.581693100872023416e-01,7.581465040667745292e-01,7.581237133581868459e-01,7.581009379513667934e-01,7.580781778362420953e-01,7.580554330027399201e-01,7.580327034407877695e-01,7.580099891403128121e-01,7.579872900912416611e-01,7.579646062835015963e-01,7.579419377070192310e-01,7.579192843517218448e-01,7.578966462075358290e-01,7.578740232643879082e-01,7.578514155122053619e-01,7.578288229409149146e-01,7.578062455404437348e-01,7.577836833007191020e-01,7.577611362116676297e-01,7.577386042632174856e-01,7.577160874452958383e-01,7.576935857478308556e-01,7.576710991607501500e-01,7.576486276739822223e-01,7.576261712774557955e-01,7.576037299610995923e-01,7.575813037148430018e-01,7.575588925286154129e-01,7.575364963923464368e-01,7.575141152959667945e-01,7.574917492294073185e-01,7.574693981825985079e-01,7.574470621454724162e-01,7.574247411079612080e-01,7.574024350599973809e-01,7.573801439915140987e-01,7.573578678924447471e-01,7.573356067527240443e-01,7.573133605622864861e-01,7.572911293110674569e-01,7.572689129890032289e-01,7.572467115860308517e-01,7.572245250920872639e-01,7.572023534971107361e-01,7.571801967910406495e-01,7.571580549638158297e-01,7.571359280053771013e-01,7.571138159056657324e-01,7.570917186546239908e-01,7.570696362421939218e-01,7.570475686583195696e-01,7.570255158929461992e-01,7.570034779360182986e-01,7.569814547774826874e-01,7.569594464072867401e-01,7.569374528153786086e-01,7.569154739917084429e-01,7.568935099262255051e-01,7.568715606088814996e-01,7.568496260296287970e-01,7.568277061784212112e-01,7.568058010452126672e-01,7.567839106199590882e-01,7.567620348926179519e-01,7.567401738531464028e-01,7.567183274915038060e-01,7.566964957976507478e-01,7.566746787615483694e-01,7.566528763731596996e-01,7.566310886224488774e-01,7.566093154993810410e-01,7.565875569939229939e-01,7.565658130960425387e-01,7.565440837957086995e-01,7.565223690828928316e-01,7.565006689475668455e-01,7.564789833797034291e-01,7.564573123692776013e-01,7.564356559062664909e-01,7.564140139806468932e-01,7.563923865823987125e-01,7.563707737015022969e-01,7.563491753279397711e-01,7.563275914516952581e-01,7.563060220627539909e-01,7.562844671511024242e-01,7.562629267067294547e-01,7.562414007196253118e-01,7.562198891797814460e-01,7.561983920771918610e-01,7.561769094018507831e-01,7.561554411437554357e-01,7.561339872929041528e-01,7.561125478392974886e-01,7.560911227729368855e-01,7.560697120838265617e-01,7.560483157619716232e-01,7.560269337973796189e-01,7.560055661800596516e-01,7.559842129000224897e-01,7.559628739472813441e-01,7.559415493118505358e-01,7.559202389837469394e-01,7.558989429529893167e-01,7.558776612095977621e-01,7.558563937435943680e-01,7.558351405450038918e-01,7.558139016038535329e-01,7.557926769101701581e-01,7.557714664539855187e-01,7.557502702253311444e-01,7.557290882142420063e-01,7.557079204107542969e-01,7.556867668049072062e-01,7.556656273867412565e-01,7.556445021462995237e-01,7.556233910736265269e-01,7.556022941587704489e-01,7.555812113917798056e-01,7.555601427627073319e-01,7.555390882616058734e-01,7.555180478785314957e-01,7.554970216035433728e-01,7.554760094267011228e-01,7.554550113380688048e-01,7.554340273277104778e-01,7.554130573856944197e-01,7.553921015020901297e-01,7.553711596669699935e-01,7.553502318704085061e-01,7.553293181024828273e-01,7.553084183532721152e-01,7.552875326128579703e-01,7.552666608713253238e-01,7.552458031187603282e-01,7.552249593452521337e-01,7.552041295408929988e-01,7.551833136957761816e-01,7.551625117999989367e-01,7.551417238436605173e-01,7.551209498168626189e-01,7.551001897097096016e-01,7.550794435123080461e-01,7.550587112147679747e-01,7.550379928072015190e-01,7.550172882797228091e-01,7.549965976224505271e-01,7.549759208255032439e-01,7.549552578790047486e-01,7.549346087730808286e-01,7.549139734978586036e-01,7.548933520434697453e-01,7.548727444000475906e-01,7.548521505577288071e-01,7.548315705066519499e-01,7.548110042369599038e-01,7.547904517387972190e-01,7.547699130023111103e-01,7.547493880176522341e-01,7.547288767749739113e-01,7.547083792644323497e-01,7.546878954761867542e-01,7.546674254003985505e-01,7.546469690272330499e-01,7.546265263468580065e-01,7.546060973494437274e-01,7.545856820251642949e-01,7.545652803641964557e-01,7.545448923567192878e-01,7.545245179929158663e-01,7.545041572629718196e-01,7.544838101570756628e-01,7.544634766654192415e-01,7.544431567781967329e-01,7.544228504856067552e-01,7.544025577778499247e-01,7.543822786451304108e-01,7.543620130776552690e-01,7.543417610656345529e-01,7.543215225992820905e-01,7.543012976688140414e-01,
        7.542810862644502290e-01,7.542608883764135852e-01,7.542407039949305947e-01,7.542205331102299626e-01,7.542003757125447239e-01,7.541802317921106891e-01,7.541601013391666664e-01,7.541399843439551276e-01,7.541198807967214313e-01,7.540997906877147106e-01,7.540797140071872073e-01,7.540596507453942721e-01,7.540396008925944749e-01,7.540195644390508267e-01,7.539995413750281150e-01,7.539795316907954570e-01,7.539595353766256336e-01,7.539395524227936463e-01,7.539195828195792703e-01,7.538996265572648348e-01,7.538796836261364431e-01,7.538597540164828636e-01,7.538398377185979715e-01,7.538199347227776403e-01,7.538000450193219626e-01,7.537801685985341393e-01,7.537603054507213685e-01,7.537404555661934014e-01,7.537206189352652075e-01,7.537007955482533106e-01,7.536809853954796745e-01,7.536611884672685946e-01,7.536414047539480299e-01,7.536216342458500472e-01,7.536018769333103773e-01,7.535821328066680813e-01,7.535624018562657733e-01,7.535426840724495090e-01,7.535229794455701180e-01,7.535032879659808724e-01,7.534836096240387082e-01,7.534639444101057792e-01,7.534442923145464599e-01,7.534246533277291213e-01,7.534050274400259095e-01,7.533854146418134112e-01,7.533658149234715440e-01,7.533462282753831118e-01,7.533266546879354708e-01,7.533070941515205288e-01,7.532875466565330802e-01,7.532680121933710282e-01,7.532484907524381601e-01,7.532289823241402615e-01,7.532094868988874481e-01,7.531900044670941652e-01,7.531705350191782999e-01,7.531510785455618473e-01,7.531316350366707990e-01,7.531122044829341444e-01,7.530927868747860909e-01,7.530733822026641766e-01,7.530539904570098253e-01,7.530346116282681246e-01,7.530152457068888250e-01,7.529958926833250077e-01,7.529765525480346389e-01,7.529572252914784602e-01,7.529379109041219875e-01,7.529186093764342891e-01,7.528993206988890963e-01,7.528800448619639152e-01,7.528607818561400267e-01,7.528415316719023753e-01,7.528222942997415679e-01,7.528030697301508756e-01,7.527838579536277885e-01,7.527646589606742378e-01,7.527454727417962621e-01,7.527262992875037861e-01,7.527071385883110644e-01,7.526879906347369031e-01,7.526688554173033285e-01,7.526497329265368075e-01,7.526306231529683588e-01,7.526115260871328871e-01,7.525924417195698490e-01,7.525733700408222537e-01,7.525543110414376624e-01,7.525352647119679661e-01,7.525162310429693857e-01,7.524972100250018059e-01,7.524782016486301073e-01,7.524592059044229453e-01,7.524402227829526391e-01,7.524212522747978360e-01,7.524022943705385158e-01,7.523833490607619856e-01,7.523644163360575510e-01,7.523454961870199575e-01,7.523265886042478368e-01,7.523076935783445940e-01,7.522888110999171873e-01,7.522699411595784591e-01,7.522510837479433610e-01,7.522322388556329509e-01,7.522134064732723946e-01,7.521945865914907436e-01,7.521757792009216015e-01,7.521569842922033455e-01,7.521382018559776839e-01,7.521194318828928749e-01,7.521006743635992864e-01,7.520819292887529484e-01,7.520631966490138876e-01,7.520444764350470157e-01,7.520257686375216855e-01,7.520070732471111352e-01,7.519883902544937104e-01,7.519697196503517533e-01,7.519510614253722691e-01,7.519324155702478141e-01,7.519137820756731649e-01,7.518951609323495378e-01,7.518765521309820343e-01,7.518579556622804194e-01,7.518393715169588987e-01,7.518207996857363407e-01,7.518022401593360549e-01,7.517836929284855696e-01,7.517651579839179643e-01,7.517466353163703152e-01,7.517281249165836954e-01,7.517096267753050620e-01,7.516911408832848140e-01,7.516726672312785684e-01,7.516542058100467161e-01,7.516357566103541998e-01,7.516173196229697373e-01,7.515988948386678192e-01,7.515804822482270442e-01,7.515620818424307847e-01,7.515436936120672984e-01,7.515253175479292835e-01,7.515069536408142126e-01,7.514886018815234436e-01,7.514702622608651073e-01,7.514519347696494433e-01,7.514336193986934642e-01,7.514153161388180679e-01,7.513970249808483715e-01,7.513787459156155979e-01,7.513604789339543011e-01,7.513422240267049190e-01,7.513239811847117755e-01,7.513057503988246344e-01,7.512875316598972564e-01,7.512693249587889532e-01,7.512511302863634777e-01,7.512329476334895784e-01,7.512147769910403339e-01,7.511966183498942629e-01,7.511784717009337697e-01,7.511603370350472542e-01,7.511422143431273346e-01,7.511241036160715145e-01,7.511060048447818493e-01,7.510879180201658345e-01,7.510698431331350733e-01,7.510517801746071642e-01,7.510337291355030365e-01,7.510156900067499475e-01,7.509976627792791515e-01,7.509796474440272318e-01,7.509616439919354347e-01,7.509436524139498914e-01,7.509256727010215071e-01,7.509077048441064051e-01,7.508897488341661486e-01,7.508718046621657427e-01,7.508538723190766317e-01,7.508359517958737017e-01,7.508180430835382779e-01,7.508001461730561266e-01,7.507822610554175657e-01,7.507643877216180206e-01,7.507465261626580233e-01,7.507286763695433240e-01,7.507108383332837809e-01,7.506930120448955801e-01,7.506751974953984607e-01,7.506573946758184901e-01,7.506396035771857322e-01,7.506218241905356914e-01,7.506040565069085346e-01,7.505863005173504243e-01,7.505685562129111865e-01,7.505508235846467535e-01,7.505331026236170544e-01,7.505153933208885686e-01,7.504976956675313282e-01,7.504800096546211385e-01,7.504623352732392449e-01,7.504446725144707786e-01,7.504270213694068659e-01,7.504093818291434070e-01,7.503917538847815205e-01,7.503741375274276537e-01,7.503565327481925840e-01,7.503389395381926397e-01,7.503213578885491453e-01,7.503037877903889763e-01,7.502862292348436712e-01,7.502686822130495425e-01,7.502511467161493419e-01,7.502336227352887077e-01,7.502161102616210497e-01,7.501986092863028865e-01,7.501811198004965098e-01,7.501636417953702063e-01,7.501461752620959267e-01,7.501287201918513947e-01,7.501112765758207734e-01,7.500938444051906684e-01,7.500764236711553457e-01,7.500590143649130681e-01,7.500416164776677608e-01,7.500242300006275675e-01,7.500068549250075156e-01,7.499894912420256299e-01,7.499721389429072627e-01,7.499547980188819851e-01,7.499374684611844755e-01,7.499201502610545189e-01,7.499028434097380069e-01,7.498855478984843836e-01,7.498682637185504207e-01,7.498509908611965535e-01,7.498337293176887686e-01,7.498164790792990475e-01,7.497992401373030358e-01,7.497820124829838173e-01,7.497647961076275847e-01,7.497475910025268586e-01,7.497303971589799332e-01,7.497132145682892101e-01,7.496960432217625314e-01,7.496788831107136231e-01,7.496617342264609851e-01,7.496445965603292239e-01,7.496274701036468313e-01,7.496103548477488498e-01,7.495932507839748737e-01,7.495761579036699374e-01,7.495590761981848482e-01,7.495420056588748547e-01,7.495249462771010895e-01,7.495078980442299033e-01,7.494908609516333087e-01,7.494738349906876484e-01,7.494568201527754825e-01,7.494398164292840336e-01,7.494228238116069640e-01,7.494058422911418216e-01,7.493888718592924825e-01,7.493719125074679299e-01,7.493549642270821431e-01,7.493380270095550966e-01,7.493211008463108724e-01,7.493041857287808805e-01,7.492872816484000831e-01,7.492703885966095489e-01,7.492535065648553427e-01,7.492366355445901904e-01,7.492197755272702597e-01,7.492029265043580466e-01,7.491860884673217091e-01,7.491692614076342904e-01,7.491524453167743847e-01,7.491356401862260261e-01,7.491188460074783562e-01,7.491020627720262892e-01,7.490852904713698468e-01,7.490685290970144905e-01,7.490517786404714551e-01,7.490350390932566382e-01,7.490183104468921549e-01,7.490015926929046719e-01,7.489848858228272954e-01,7.489681898281971284e-01,7.489515047005584902e-01,7.489348304314593641e-01,7.489181670124545054e-01,7.489015144351032216e-01,7.488848726909711484e-01,7.488682417716276962e-01,7.488516216686491589e-01,7.488350123736176034e-01,7.488184138781190935e-01,7.488018261737460213e-01,7.487852492520956638e-01,7.487686831047719593e-01,7.487521277233827321e-01,7.487355830995423567e-01,7.487190492248703144e-01,7.487025260909908608e-01,7.486860136895352458e-01,7.486695120121386049e-01,7.486530210504427352e-01,7.486365407960939855e-01,7.486200712407447000e-01,7.486036123760527738e-01,7.485871641936812093e-01,7.485707266852983377e-01,7.485542998425788186e-01,7.485378836572014194e-01,7.485214781208520129e-01,7.485050832252209130e-01,7.484886989620039843e-01,7.484723253229028650e-01,7.484559622996245221e-01,7.484396098838809186e-01,7.484232680673909011e-01,7.484069368418775348e-01,7.483906161990696582e-01,7.483743061307022160e-01,7.483580066285148158e-01,7.483417176842528384e-01,7.483254392896674378e-01,7.483091714365153191e-01,7.482929141165579612e-01,7.482766673215629494e-01,7.482604310433039752e-01,7.482442052735585047e-01,7.482279900041116649e-01,7.482117852267522462e-01,7.481955909332759225e-01,7.481794071154825865e-01,7.481632337651789033e-01,7.481470708741762010e-01,7.481309184342916918e-01,7.481147764373486941e-01,7.480986448751746343e-01,7.480825237396035998e-01,7.480664130224750075e-01,7.480503127156333809e-01,7.480342228109295721e-01,7.480181433002187630e-01,7.480020741753629077e-01,7.479860154282287343e-01,7.479699670506891884e-01,7.479539290346219893e-01,7.479379013719104075e-01,7.479218840544442637e-01,7.479058770741179307e-01,7.478898804228316655e-01,7.478738940924913869e-01,7.478579180750082323e-01,7.478419523622994447e-01,7.478259969462871526e-01,7.478100518188994794e-01,7.477941169720697667e-01,7.477781923977374623e-01,7.477622780878470099e-01,7.477463740343489595e-01,7.477304802291988572e-01,7.477145966643580222e-01,7.476987233317936576e-01,7.476828602234777410e-01,7.476670073313885778e-01,7.476511646475100248e-01,7.476353321638312677e-01,7.476195098723463772e-01,7.476036977650568627e-01,7.475878958339673419e-01,7.475721040710899823e-01,7.475563224684413921e-01,7.475405510180449520e-01,7.475247897119281504e-01,7.475090385421250261e-01,7.474932975006747249e-01,7.474775665796221658e-01,7.474618457710177077e-01,7.474461350669178161e-01,7.474304344593838412e-01,7.474147439404831283e-01,7.473990635022882412e-01,7.473833931368777383e-01,7.473677328363355077e-01,7.473520825927509881e-01,7.473364423982198357e-01,7.473208122448419255e-01,7.473051921247241269e-01,7.472895820299778613e-01,7.472739819527213223e-01,7.472583918850770335e-01,7.472428118191731805e-01,7.472272417471446104e-01,7.472116816611312773e-01,
        7.471961315532776871e-01,7.471805914157355621e-01,7.471650612406611769e-01,7.471495410202169118e-01,7.471340307465699215e-01,7.471185304118945769e-01,7.471030400083689127e-01,7.470875595281777359e-01,7.470720889635110717e-01,7.470566283065642743e-01,7.470411775495394702e-01,7.470257366846426716e-01,7.470103057040866634e-01,7.469948846000898923e-01,7.469794733648753571e-01,7.469640719906726067e-01,7.469486804697162974e-01,7.469332987942474134e-01,7.469179269565111579e-01,7.469025649487597285e-01,7.468872127632496527e-01,7.468718703922451185e-01,7.468565378280129785e-01,7.468412150628276347e-01,7.468259020889690403e-01,7.468105988987224775e-01,7.467953054843780025e-01,7.467800218382324440e-01,7.467647479525875154e-01,7.467494838197509255e-01,7.467342294320359342e-01,7.467189847817615744e-01,7.467037498612509872e-01,7.466885246628354178e-01,7.466733091788491095e-01,7.466581034016344098e-01,7.466429073235372194e-01,7.466277209369100998e-01,7.466125442341106089e-01,7.465973772075022996e-01,7.465822198494552753e-01,7.465670721523425257e-01,7.465519341085454785e-01,7.465368057104495580e-01,7.465216869504458508e-01,7.465065778209323266e-01,7.464914783143107302e-01,7.464763884229896895e-01,7.464613081393826066e-01,7.464462374559093227e-01,7.464311763649945641e-01,7.464161248590687192e-01,7.464010829305687267e-01,7.463860505719355221e-01,7.463710277756165912e-01,7.463560145340647489e-01,7.463410108397389164e-01,7.463260166851030109e-01,7.463110320626263894e-01,7.462960569647848486e-01,7.462810913840587368e-01,7.462661353129348418e-01,7.462511887439048364e-01,7.462362516694666104e-01,7.462213240821229387e-01,7.462064059743834799e-01,7.461914973387614447e-01,7.461765981677773718e-01,7.461617084539567957e-01,7.461468281898308019e-01,7.461319573679358053e-01,7.461170959808144376e-01,7.461022440210141049e-01,7.460874014810885413e-01,7.460725683535961439e-01,7.460577446311023042e-01,7.460429303061766326e-01,7.460281253713948457e-01,7.460133298193383222e-01,7.459985436425938810e-01,7.459837668337540029e-01,7.459689993854163870e-01,7.459542412901849495e-01,7.459394925406689358e-01,7.459247531294826983e-01,7.459100230492465844e-01,7.458953022925866039e-01,7.458805908521342065e-01,7.458658887205256161e-01,7.458511958904043837e-01,7.458365123544180575e-01,7.458218381052204027e-01,7.458071731354708467e-01,7.457925174378338129e-01,7.457778710049796089e-01,7.457632338295845376e-01,7.457486059043298976e-01,7.457339872219022059e-01,7.457193777749949737e-01,7.457047775563059311e-01,7.456901865585382483e-01,7.456756047744019789e-01,7.456610321966111732e-01,7.456464688178868760e-01,7.456319146309542401e-01,7.456173696285453012e-01,7.456028338033967584e-01,7.455883071482508617e-01,7.455737896558565225e-01,7.455592813189665380e-01,7.455447821303404776e-01,7.455302920827427959e-01,7.455158111689441647e-01,7.455013393817198075e-01,7.454868767138516095e-01,7.454724231581256744e-01,7.454579787073352115e-01,7.454435433542776490e-01,7.454291170917562992e-01,7.454146999125809137e-01,7.454002918095652408e-01,7.453858927755300234e-01,7.453715028033002232e-01,7.453571218857069081e-01,7.453427500155871410e-01,7.453283871857828702e-01,7.453140333891420388e-01,7.452996886185174752e-01,7.452853528667678917e-01,7.452710261267576630e-01,7.452567083913567147e-01,7.452423996534400796e-01,7.452280999058886746e-01,7.452138091415888566e-01,7.451995273534317565e-01,7.451852545343153889e-01,7.451709906771428749e-01,7.451567357748217768e-01,7.451424898202660962e-01,7.451282528063957189e-01,7.451140247261349714e-01,7.450998055724143976e-01,7.450855953381695373e-01,7.450713940163421478e-01,7.450572015998787601e-01,7.450430180817315673e-01,7.450288434548592020e-01,7.450146777122242936e-01,7.450005208467956885e-01,7.449863728515481176e-01,7.449722337194607524e-01,7.449581034435187599e-01,7.449439820167137460e-01,7.449298694320418690e-01,7.449157656825037277e-01,7.449016707611079147e-01,7.448875846608660201e-01,7.448735073747967395e-01,7.448594388959233203e-01,7.448453792172752275e-01,7.448313283318871436e-01,7.448172862327988586e-01,7.448032529130557133e-01,7.447892283657085999e-01,7.447752125838146275e-01,7.447612055604345693e-01,7.447472072886369698e-01,7.447332177614940374e-01,7.447192369720837535e-01,7.447052649134902058e-01,7.446913015788029222e-01,7.446773469611159824e-01,7.446634010535292392e-01,7.446494638491488738e-01,7.446355353410855082e-01,7.446216155224555378e-01,7.446077043863806866e-01,7.445938019259882301e-01,7.445799081344114390e-01,7.445660230047878025e-01,7.445521465302615827e-01,7.445382787039810379e-01,7.445244195191013103e-01,7.445105689687816497e-01,7.444967270461879671e-01,7.444828937444903927e-01,7.444690690568653846e-01,7.444552529764948412e-01,7.444414454965655459e-01,7.444276466102693890e-01,7.444138563108049222e-01,7.444000745913749162e-01,7.443863014451881366e-01,7.443725368654590113e-01,7.443587808454061872e-01,7.443450333782557493e-01,7.443312944572366696e-01,7.443175640755854694e-01,7.443038422265427778e-01,7.442901289033554413e-01,7.442764240992750802e-01,7.442627278075587549e-01,7.442490400214694102e-01,7.442353607342750976e-01,7.442216899392491980e-01,7.442080276296699770e-01,7.441943737988224727e-01,7.441807284399958311e-01,7.441670915464851932e-01,7.441534631115905851e-01,7.441398431286176951e-01,7.441262315908774294e-01,7.441126284916868006e-01,7.440990338243671509e-01,7.440854475822459291e-01,7.440718697586552466e-01,7.440583003469334322e-01,7.440447393404234777e-01,7.440311867324737038e-01,7.440176425164384266e-01,7.440041066856769580e-01,7.439905792335540502e-01,7.439770601534395622e-01,7.439635494387083492e-01,7.439500470827417056e-01,7.439365530789253667e-01,7.439230674206509519e-01,7.439095901013148548e-01,7.438961211143191310e-01,7.438826604530712761e-01,7.438692081109840037e-01,7.438557640814752459e-01,7.438423283579682632e-01,7.438289009338914237e-01,7.438154818026795345e-01,7.438020709577710665e-01,7.437886683926109299e-01,7.437752741006494750e-01,7.437618880753412709e-01,7.437485103101468820e-01,7.437351407985325347e-01,7.437217795339688964e-01,7.437084265099330738e-01,7.436950817199065034e-01,7.436817451573757287e-01,7.436684168158337327e-01,7.436550966887777170e-01,7.436417847697109895e-01,7.436284810521414101e-01,7.436151855295829449e-01,7.436018981955532237e-01,7.435886190435773146e-01,7.435753480671842830e-01,7.435620852599085229e-01,7.435488306152902016e-01,7.435355841268742605e-01,7.435223457882109699e-01,7.435091155928559292e-01,7.434958935343705111e-01,7.434826796063205290e-01,7.434694738022773475e-01,7.434562761158179933e-01,7.434430865405238231e-01,7.434299050699824107e-01,7.434167316977862150e-01,7.434035664175330238e-01,7.433904092228252880e-01,7.433772601072716757e-01,7.433641190644851848e-01,7.433509860880844755e-01,7.433378611716934259e-01,7.433247443089414652e-01,7.433116354934623526e-01,7.432985347188958425e-01,7.432854419788867961e-01,7.432723572670851819e-01,7.432592805771457423e-01,7.432462119027293257e-01,7.432331512375016658e-01,7.432200985751332700e-01,7.432070539093001971e-01,7.431940172336831685e-01,7.431809885419693451e-01,7.431679678278502177e-01,7.431549550850222730e-01,7.431419503071879928e-01,7.431289534880539671e-01,7.431159646213328918e-01,7.431029837007423478e-01,7.430900107200051341e-01,7.430770456728488238e-01,7.430640885530066519e-01,7.430511393542171827e-01,7.430381980702235323e-01,7.430252646947740347e-01,7.430123392216234635e-01,7.429994216445297006e-01,7.429865119572570675e-01,7.429736101535751036e-01,7.429607162272580112e-01,7.429478301720854327e-01,7.429349519818418957e-01,7.429220816503174785e-01,7.429092191713072557e-01,7.428963645386107428e-01,7.428835177460338945e-01,7.428706787873866624e-01,7.428578255848355960e-01,7.428449360148873248e-01,7.428320100028886763e-01,7.428190474742122351e-01,7.428060483542618941e-01,7.427930125684724105e-01,7.427799400423091836e-01,7.427668307012728066e-01,7.427536844708990671e-01,7.427405012767591685e-01,7.427272810444637274e-01,7.427140236996621070e-01,7.427007291680450818e-01,7.426873973753453928e-01,7.426740282473399679e-01,7.426606217098515872e-01,7.426471776887495491e-01,7.426336961099516687e-01,7.426201768994262764e-01,7.426066199831923287e-01,7.425930252873222948e-01,7.425793927379438220e-01,7.425657222612395136e-01,7.425520137834507040e-01,7.425382672308779020e-01,7.425244825298822349e-01,7.425106596068872244e-01,7.424967983883800082e-01,7.424828988009146702e-01,7.424689607711120187e-01,7.424549842256610299e-01,7.424409690913219562e-01,7.424269152949274364e-01,7.424128227633828292e-01,7.423986914236700985e-01,7.423845212028478135e-01,7.423703120280532586e-01,7.423560638265044309e-01,7.423417765255015954e-01,7.423274500524285058e-01,7.423130843347548469e-01,7.422986793000370120e-01,7.422842348759215447e-01,7.422697509901443613e-01,7.422552275705344149e-01,7.422406645450152496e-01,7.422260618416064437e-01,7.422114193884240541e-01,7.421967371136851677e-01,7.421820149457071247e-01,7.421672528129110713e-01,7.421524506438226254e-01,7.421376083670740975e-01,7.421227259114063779e-01,7.421078032056703799e-01,7.420928401788299267e-01,7.420778367599618619e-01,7.420627928782590477e-01,7.420477084630332509e-01,7.420325834437144774e-01,7.420174177498541912e-01,7.420022113111275353e-01,7.419869640573349967e-01,7.419716759184036281e-01,7.419563468243899340e-01,7.419409767054813143e-01,7.419255654919976184e-01,7.419101131143934769e-01,7.418946195032609658e-01,7.418790845893296071e-01,7.418635083034705868e-01,7.418478905766966447e-01,7.418322313401662926e-01,7.418165305251834818e-01,7.418007880632004891e-01,7.417850038858210260e-01,7.417691779248009043e-01,7.417533101120499239e-01,7.417374003796354254e-01,7.417214486597821788e-01,7.417054548848756035e-01,7.416894189874647658e-01,7.416733409002629340e-01,7.416572205561492437e-01,7.416410578881723614e-01,7.416248528295518172e-01,7.416086053136801137e-01,7.415923152741248359e-01,7.415759826446296499e-01,7.415596073591187443e-01,7.415431893516972739e-01,7.415267285566539135e-01,
        7.415102249084624120e-01,7.414936783417848121e-01,7.414770887914726716e-01,7.414604561925693949e-01,7.414437804803133414e-01,7.414270615901382699e-01,7.414102994576775574e-01,7.413934940187637546e-01,7.413766452094338044e-01,7.413597529659283758e-01,7.413428172246968595e-01,7.413258379223971462e-01,7.413088149958988460e-01,7.412917483822853981e-01,7.412746380188571793e-01,7.412574838431320590e-01,7.412402857928495070e-01,7.412230438059709270e-01,7.412057578206833197e-01,7.411884277754010597e-01,7.411710536087683376e-01,7.411536352596612698e-01,7.411361726671898964e-01,7.411186657707014014e-01,7.411011145097812225e-01,7.410835188242562710e-01,7.410658786541969301e-01,7.410481939399200524e-01,7.410304646219891822e-01,7.410126906412192183e-01,7.409948719386779681e-01,7.409770084556878134e-01,7.409591001338291516e-01,7.409411469149425056e-01,7.409231487411297445e-01,7.409051055547576370e-01,7.408870172984605151e-01,7.408688839151418293e-01,7.408507053479767013e-01,7.408324815404133679e-01,7.408142124361789538e-01,7.407958979792772514e-01,7.407775381139949378e-01,7.407591327849023521e-01,7.407406819368546058e-01,7.407221855149975775e-01,7.407036434647668033e-01,7.406850557318918060e-01,7.406664222623987603e-01,7.406477430026108255e-01,7.406290178991532525e-01,7.406102468989546050e-01,7.405914299492483144e-01,7.405725669975780079e-01,7.405536579917962881e-01,7.405347028800697284e-01,7.405157016108810941e-01,7.404966541330305629e-01,7.404775603956403884e-01,7.404584203481544558e-01,7.404392339403440548e-01,7.404200011223074362e-01,7.404007218444750293e-01,7.403813960576096642e-01,7.403620237128100134e-01,7.403426047615139227e-01,7.403231391554999652e-01,7.403036268468893288e-01,7.402840677881509235e-01,7.402644619321003816e-01,7.402448092319060535e-01,7.402251096410893405e-01,7.402053631135279144e-01,7.401855696034580490e-01,7.401657290654786170e-01,7.401458414545506459e-01,7.401259067260033131e-01,7.401059248355339459e-01,7.400858957392124626e-01,7.400658193934818163e-01,7.400456957551635462e-01,7.400255247814575554e-01,7.400053064299458860e-01,7.399850406585957163e-01,7.399647274257616925e-01,7.399443666901870387e-01,7.399239584110092194e-01,7.399035025477597172e-01,7.398829990603681406e-01,7.398624479091647776e-01,7.398418490548817061e-01,7.398212024586581226e-01,7.398005080820400092e-01,7.397797658869851301e-01,7.397589758358644740e-01,7.397381378914650307e-01,7.397172520169927878e-01,7.396963181760745076e-01,7.396753363327608355e-01,7.396543064515299637e-01,7.396332284972887416e-01,7.396121024353758955e-01,7.395909282315644706e-01,7.395697058520652734e-01,7.395484352635280922e-01,7.395271164330458058e-01,7.395057493281559369e-01,7.394843339168439833e-01,7.394628701675457494e-01,7.394413580491501214e-01,7.394197975310013993e-01,7.393981885829025158e-01,7.393765311751162583e-01,7.393548252783703756e-01,7.393330708638582438e-01,7.393112679032417534e-01,7.392894163686549724e-01,7.392675162327054794e-01,7.392455674684778044e-01,7.392235700495357609e-01,7.392015239499258872e-01,7.391794291441782239e-01,7.391572856073111986e-01,7.391350933148327362e-01,7.391128522427430347e-01,7.390905623675378955e-01,7.390682236662114990e-01,7.390458361162569600e-01,7.390233996956722118e-01,7.390009143829597837e-01,7.389783801571309096e-01,7.389557969977076368e-01,7.389331648847259348e-01,7.389104837987376939e-01,7.388877537208137225e-01,7.388649746325459677e-01,7.388421465160514012e-01,7.388192693539715750e-01,7.387963431294797267e-01,7.387733678262792258e-01,7.387503434286089021e-01,7.387272699212429350e-01,7.387041472894972927e-01,7.386809755192289551e-01,7.386577545968399106e-01,7.386344845092790434e-01,7.386111652440456865e-01,7.385877967891911755e-01,7.385643791333228458e-01,7.385409122656043657e-01,7.385173961757601768e-01,7.384938308540776042e-01,7.384702162914086321e-01,7.384465524791740121e-01,7.384228394093639292e-01,7.383990770745413323e-01,7.383752654678451544e-01,7.383514045829916439e-01,7.383274944142775853e-01,7.383035349565832961e-01,7.382795262053736263e-01,7.382554681567016219e-01,7.382313608072108568e-01,7.382072041541373197e-01,7.381829981953125230e-01,7.381587429291660563e-01,7.381344383547271404e-01,7.381100844716282916e-01,7.380856812801068756e-01,7.380612287810076610e-01,7.380367269757857063e-01,7.380121758665081355e-01,7.379875754558569145e-01,7.379629257471322923e-01,7.379382267442521348e-01,7.379134784517583645e-01,7.378886808748159609e-01,7.378638340192171796e-01,7.378389378913829955e-01,7.378139924983664333e-01,7.377889978478540112e-01,7.377639539481678499e-01,7.377388608082704469e-01,7.377137184377623447e-01,7.376885268468896806e-01,7.376632860465418551e-01,7.376379960482576381e-01,7.376126568642248360e-01,7.375872685072840662e-01,7.375618309909293124e-01,7.375363443293122545e-01,7.375108085372439337e-01,7.374852236301948638e-01,7.374595896243006932e-01,7.374339065363613166e-01,7.374081743838452052e-01,7.373823931848901836e-01,7.373565629583066494e-01,7.373306837235795719e-01,7.373047555008688247e-01,7.372787783110140714e-01,7.372527521755357638e-01,7.372266771166359201e-01,7.372005531572023429e-01,7.371743803208096191e-01,7.371481586317203405e-01,7.371218881148897673e-01,7.370955687959644953e-01,7.370692007012871194e-01,7.370427838578974544e-01,7.370163182935339785e-01,7.369898040366358316e-01,7.369632411163460350e-01,7.369366295625119356e-01,7.369099694056877592e-01,7.368832606771374971e-01,7.368565034088344623e-01,7.368296976334656190e-01,7.368028433844313607e-01,7.367759406958500623e-01,7.367489896025568585e-01,7.367219901401079740e-01,7.366949423447801681e-01,7.366678462535746208e-01,7.366407019042188198e-01,7.366135093351660057e-01,7.365862685855992797e-01,7.365589796954311597e-01,7.365316427053085757e-01,7.365042576566110943e-01,7.364768245914538047e-01,7.364493435526905385e-01,7.364218145839130925e-01,7.363942377294546704e-01,7.363666130343904381e-01,7.363389405445392999e-01,7.363112203064660077e-01,7.362834523674828269e-01,7.362556367756495357e-01,7.362277735797765343e-01,7.361998628294257330e-01,7.361719045749123280e-01,7.361438988673061345e-01,7.361158457584328074e-01,7.360877453008751736e-01,7.360595975479754527e-01,7.360314025538349236e-01,7.360031603733184769e-01,7.359748710620521717e-01,7.359465346764264559e-01,7.359181512735979425e-01,7.358897209114904081e-01,7.358612436487945718e-01,7.358327195449710922e-01,7.358041486602510117e-01,7.357755310556381989e-01,7.357468667929079054e-01,7.357181559346102073e-01,7.356893985440700057e-01,7.356605946853895794e-01,7.356317444234475866e-01,7.356028478239013957e-01,7.355739049531883067e-01,7.355449158785259955e-01,7.355158806679136241e-01,7.354867993901332834e-01,7.354576721147495499e-01,7.354284989121130378e-01,7.353992798533588449e-01,7.353700150104077737e-01,7.353407044559689965e-01,7.353113482635385001e-01,7.352819465074018623e-01,7.352524992626339184e-01,7.352230066050997603e-01,7.351934686114558470e-01,7.351638853591503375e-01,7.351342569264237570e-01,7.351045833923097739e-01,7.350748648366360882e-01,7.350451013400249867e-01,7.350152929838933424e-01,7.349854398504541697e-01,7.349555420227162905e-01,7.349255995844858891e-01,7.348956126203648465e-01,7.348655812157548484e-01,7.348355054568537215e-01,7.348053854306583199e-01,7.347752212249657466e-01,7.347450129283710218e-01,7.347147606302693035e-01,7.346844644208557762e-01,7.346541243911255403e-01,7.346237406328740560e-01,7.345933132386983644e-01,7.345628423019959774e-01,7.345323279169655439e-01,7.345017701786071829e-01,7.344711691827220390e-01,7.344405250259130602e-01,7.344098378055848864e-01,7.343791076199441825e-01,7.343483345679979735e-01,7.343175187495568634e-01,7.342866602652311503e-01,7.342557592164340452e-01,7.342248157053796742e-01,7.341938298350837444e-01,7.341628017093632108e-01,7.341317314328354993e-01,7.341006191109197276e-01,7.340694648498348185e-01,7.340382687566002762e-01,7.340070309390365200e-01,7.339757515057618864e-01,7.339444305661957380e-01,7.339130682305557984e-01,7.338816646098579310e-01,7.338502198159171375e-01,7.338187339613452265e-01,7.337872071595517021e-01,7.337556395247424312e-01,7.337240311719198660e-01,7.336923822168814890e-01,7.336606927762200359e-01,7.336289629673224955e-01,7.335971929083690002e-01,7.335653827183336029e-01,7.335335325169810572e-01,7.335016424248691491e-01,7.334697125633450332e-01,7.334377430545460097e-01,7.334057340213995246e-01,7.333736855876186178e-01,7.333415978777060307e-01,7.333094710169494324e-01,7.332773051314214197e-01,7.332451003479796281e-01,7.332128567942646225e-01,7.331805745986987866e-01,7.331482538904862123e-01,7.331158947996105901e-01,7.330834974568340989e-01,7.330510619936966288e-01,7.330185885425147818e-01,7.329860772363798738e-01,7.329535282091572679e-01,7.329209415954843765e-01,7.328883175307706610e-01,7.328556561511947454e-01,7.328229575937034168e-01,7.327902219960111818e-01,7.327574494965978236e-01,7.327246402347061816e-01,7.326917943503433728e-01,7.326589119842756848e-01,7.326259932780291306e-01,7.325930383738883389e-01,7.325600474148928898e-01,7.325270205448377592e-01,7.324939579082695440e-01,7.324608596504856850e-01,7.324277259175343557e-01,7.323945568562089115e-01,7.323613526140495544e-01,7.323281133393396702e-01,7.322948391811042734e-01,7.322615302891083422e-01,7.322281868138540428e-01,7.321948089065801746e-01,7.321613967192587280e-01,7.321279504045937747e-01,7.320944701160182477e-01,7.320609560076934974e-01,7.320274082345058497e-01,7.319938269520642748e-01,7.319602123166994989e-01,7.319265644854606734e-01,7.318928836161127105e-01,7.318591698671355061e-01,7.318254233977202761e-01,7.317916443677677796e-01,7.317578329378849888e-01,7.317239892693845338e-01,7.316901135242805942e-01,7.316562058652872347e-01,7.316222664558151845e-01,7.315882954599699506e-01,7.315542930425481538e-01,7.315202593690373067e-01,7.314861946056099296e-01,7.314520989191237721e-01,7.314179724771171509e-01,7.313838154478066178e-01,7.313496280000851835e-01,7.313154103035185427e-01,7.312811625283426320e-01,7.312468848454599657e-01,
        7.312125774264381928e-01,7.311782404435062110e-01,7.311438740695510585e-01,7.311094784781159150e-01,7.310750538433953283e-01,7.310406003402345476e-01,7.310061181441246392e-01,7.309716074311998213e-01,7.309370683782344669e-01,7.309025011626404389e-01,7.308679059624625385e-01,7.308332829563769506e-01,7.307986323236864701e-01,7.307639542443187253e-01,7.307292488988212931e-01,7.306945164683600336e-01,7.306597571347139830e-01,7.306249710802731334e-01,7.305901584880348798e-01,7.305553195416008005e-01,7.305204544251713283e-01,7.304855633235453061e-01,7.304506464221139916e-01,7.304157039068590596e-01,7.303807359643471608e-01,7.303457427817284797e-01,7.303107245467315156e-01,7.302756814476598635e-01,7.302406136733894382e-01,7.302055214133632566e-01,7.301704048575875516e-01,7.301352641966306622e-01,7.301000996216163719e-01,7.300649113242203558e-01,7.300296994966677389e-01,7.299944643317292092e-01,7.299592060227150236e-01,7.299239247634731198e-01,7.298886207483845645e-01,7.298532941723590017e-01,7.298179452308312110e-01,7.297825741197577765e-01,7.297471810356109811e-01,7.297117661753770301e-01,7.296763297365496115e-01,7.296408719171282309e-01,7.296053929156119944e-01,7.295698929309971659e-01,7.295343721627702838e-01,7.294988308109071617e-01,7.294632690758666715e-01,7.294276871585870792e-01,7.293920852604808269e-01,7.293564635834320908e-01,7.293208223297903414e-01,7.292851617023671240e-01,7.292494819044317289e-01,7.292137831397060843e-01,7.291780656123608706e-01,7.291423295270107463e-01,7.291065750887101293e-01,7.290708025029485340e-01,7.290350119756462410e-01,7.289992037131490799e-01,7.289633779222243204e-01,7.289275348100568985e-01,7.288916745842425327e-01,7.288557974527865024e-01,7.288199036240949891e-01,7.287839933069734100e-01,7.287480667106209786e-01,7.287121240446249315e-01,7.286761655189576414e-01,7.286401913439699563e-01,7.286042017303875351e-01,7.285681968893060745e-01,7.285321770321857571e-01,7.284961423708468109e-01,7.284600931174658456e-01,7.284240294845680808e-01,7.283879516850260138e-01,7.283518599320517595e-01,7.283157544391934968e-01,7.282796354203294742e-01,7.282435030896642347e-01,7.282073576617238420e-01,7.281711993513491077e-01,7.281350283736920392e-01,7.280988449442108434e-01,7.280626492786638204e-01,7.280264415931058108e-01,7.279902221038816457e-01,7.279539910276218162e-01,7.279177485812377002e-01,7.278814949819162328e-01,7.278452304471136891e-01,7.278089551945520208e-01,7.277726694422137488e-01,7.277363734083348579e-01,7.277000673114023543e-01,7.276637513701463833e-01,7.276274258035380083e-01,7.275910908307805514e-01,7.275547466713073730e-01,7.275183935447748773e-01,7.274820316710577384e-01,7.274456612702441261e-01,7.274092825626292669e-01,7.273728957687117802e-01,7.273365011091871279e-01,7.273000988049421744e-01,7.272636890770511897e-01,7.272272721467692991e-01,7.271908482355278203e-01,7.271544175649281572e-01,7.271179803567374700e-01,7.270815368328825690e-01,7.270450872154450295e-01,7.270086317266551967e-01,7.269721705888880781e-01,7.269357040246555712e-01,7.268992322566039110e-01,7.268627555075066748e-01,7.268262740002588984e-01,7.267897879578728570e-01,7.267532976034721814e-01,7.267168031602869727e-01,7.266803048516461416e-01,7.266438029009757438e-01,7.266072975317894311e-01,7.265707889676867870e-01,7.265342774323446662e-01,7.264977631495143084e-01,7.264612463430136780e-01,7.264247272367235775e-01,7.263882060545817643e-01,7.263516830205770658e-01,7.263151583587441618e-01,7.262786322931583660e-01,7.262421050479302975e-01,7.262055768471989969e-01,7.261690479151283739e-01,7.261325184759004348e-01,7.260959887537106194e-01,7.260594589727614734e-01,7.260229293572573184e-01,7.259864001313998116e-01,7.259498715193806184e-01,7.259133437453786364e-01,7.258768170335511138e-01,7.258402916080310963e-01,7.258037676929197657e-01,7.257672455122832211e-01,7.257307252901444850e-01,7.256942072504802832e-01,7.256576916172137182e-01,7.256211786142102715e-01,7.255846684652715872e-01,7.255481613941299202e-01,7.255116576244432514e-01,7.254751573797888486e-01,7.254386608836589367e-01,7.254021683594545911e-01,7.253656800304806307e-01,7.253291961199397342e-01,7.252927168509275546e-01,7.252562424464261692e-01,7.252197731293009708e-01,7.251833091222923411e-01,7.251468506480125420e-01,7.251103979289389434e-01,7.250739511874096932e-01,7.250375106456167229e-01,7.250010765256023060e-01,7.249646490492530626e-01,7.249282284382931874e-01,7.248918149142807854e-01,7.248554086986022105e-01,7.248190100124660695e-01,7.247826190768986709e-01,7.247462361127374741e-01,7.247098613406275369e-01,7.246734949810146320e-01,7.246371372541418054e-01,7.246007883800406058e-01,7.245644485785304179e-01,7.245281180692101364e-01,7.244917970714526145e-01,7.244554858044024437e-01,7.244191844869666275e-01,7.243828933378136936e-01,7.243466125753642570e-01,7.243103424177890215e-01,7.242740830830023402e-01,7.242378347886572199e-01,7.242015977521398806e-01,7.241653721905644270e-01,7.241291583207692950e-01,7.240929563593104801e-01,7.240567665224562077e-01,7.240205890261838251e-01,7.239844240861731395e-01,7.239482719178007564e-01,7.239121327361373037e-01,7.238760067559411038e-01,7.238398941916523999e-01,7.238037952573895817e-01,7.237677101669441893e-01,7.237316391337749177e-01,7.236955823710038427e-01,7.236595400914108689e-01,7.236235125074287344e-01,7.235874998311385697e-01,7.235515022742651237e-01,7.235155200481713234e-01,7.234795533638531673e-01,7.234436024319362835e-01,7.234076674626699344e-01,7.233717486659227980e-01,7.233358462511773057e-01,7.232999604275276440e-01,7.232640914036700952e-01,7.232282393879038151e-01,7.231924045881223950e-01,7.231565872118103089e-01,7.231207874660386947e-01,7.230850055574608026e-01,7.230492416923064436e-01,7.230134960763778817e-01,7.229777689150462816e-01,7.229420604132453798e-01,7.229063707754679324e-01,7.228707002057618292e-01,7.228350489077243202e-01,7.227994170844981303e-01,7.227638049387680175e-01,7.227282126727545553e-01,7.226926404882106914e-01,7.226570885864178617e-01,7.226215571681812166e-01,7.225860464338247358e-01,7.225505565831878974e-01,7.225150878156210155e-01,7.224796403299799108e-01,7.224442143246246895e-01,7.224088099974117494e-01,7.223734275456932252e-01,7.223380671663094388e-01,7.223027290555874558e-01,7.222674134093356457e-01,7.222321204228405733e-01,7.221968502908618914e-01,7.221616032076279001e-01,7.221263793668345476e-01,7.220911789616374366e-01,7.220560021846506027e-01,7.220208492279420742e-01,7.219857202830293197e-01,7.219506155408751402e-01,7.219155351918863373e-01,7.218804794259068291e-01,7.218454484322147646e-01,7.218104423995199692e-01,7.217754615159583942e-01,7.217405059690911173e-01,7.217055759458973485e-01,7.216706716327722093e-01,7.216357932155251786e-01,7.216009408793720992e-01,7.215661148089366206e-01,7.215313151882423170e-01,7.214965422007121321e-01,7.214617960291624943e-01,7.214270768558036506e-01,7.213923848622310064e-01,7.213577202294263468e-01,7.213230831377521746e-01,7.212884737669488233e-01,7.212538922961304610e-01,7.212193389037842017e-01,7.211848137677632220e-01,7.211503170652868722e-01,7.211158489729360133e-01,7.210814096666496864e-01,7.210469993217224483e-01,7.210126181128002631e-01,7.209782662138795040e-01,7.209439437983025112e-01,7.209096510387539292e-01,7.208753881072591518e-01,7.208411551751805479e-01,7.208069524132149075e-01,7.207727799913904443e-01,7.207386380790636871e-01,7.207045268449172593e-01,7.206704464569557711e-01,7.206363970825054865e-01,7.206023788882088832e-01,7.205683920400229869e-01,7.205344367032177066e-01,7.205005130423719484e-01,7.204666212213711729e-01,7.204327614034048421e-01,7.203989337509648649e-01,7.203651384258411561e-01,7.203313755891209702e-01,7.202976454011854601e-01,7.202639480217075674e-01,7.202302836096490246e-01,7.201966523232596895e-01,7.201630543200729928e-01,7.201294897569040510e-01,7.200959587898499992e-01,7.200624615742839962e-01,7.200289982648554465e-01,7.199955690154872245e-01,7.199621739793730102e-01,7.199288133089760677e-01,7.198954871560260260e-01,7.198621956715185455e-01,7.198289390057107662e-01,7.197957173081219739e-01,7.197625307275302697e-01,7.197293794119703492e-01,7.196962635087319482e-01,7.196631831643590660e-01,7.196301385246459681e-01,7.195971297346375195e-01,7.195641569386257430e-01,7.195312202801489310e-01,7.194983199019904241e-01,7.194654559461755028e-01,7.194326285539707211e-01,7.193998378658822412e-01,7.193670840216541684e-01,7.193343671602668854e-01,7.193016874199357202e-01,7.192690449381081708e-01,7.192364398514656809e-01,7.192038722959186448e-01,7.191713424066062954e-01,7.191388503178963720e-01,7.191063961633826773e-01,7.190739800758833011e-01,7.190416021874415087e-01,7.190092626293215217e-01,7.189769615320102947e-01,7.189446990252132963e-01,7.189124752378559524e-01,7.188802902980817588e-01,7.188481443332499499e-01,7.188160374699363864e-01,7.187839698339315575e-01,7.187519415502379161e-01,7.187199527430730983e-01,7.186880035358650387e-01,7.186560940512525253e-01,7.186242244110844224e-01,7.185923947364192266e-01,7.185606051475234013e-01,7.185288557638703777e-01,7.184971467041416648e-01,7.184654780862230750e-01,7.184338500272073880e-01,7.184022626433911318e-01,7.183707160502748046e-01,7.183392103625635405e-01,7.183077456941633354e-01,7.182763221581848212e-01,7.182449398669380480e-01,7.182135989319365921e-01,7.181822994638931146e-01,7.181510415727219154e-01,7.181198253675369347e-01,7.180886509566515308e-01,7.180575184475787021e-01,7.180264279470299771e-01,7.179953795609161915e-01,7.179643733943466000e-01,7.179334095516283210e-01,7.179024881362670030e-01,7.178716092509653812e-01,7.178407729976250540e-01,7.178099794773440401e-01,7.177792287904189994e-01,7.177485210363433454e-01,7.177178563138085776e-01,7.176872347207025049e-01,7.176566563541117993e-01,7.176261213103201086e-01,7.175956296848085003e-01,7.175651815722555726e-01,7.175347770665387870e-01,7.175044162607328024e-01,7.174740992471110301e-01,7.174438261171441900e-01,7.174135969615036412e-01,7.173834118700577189e-01,7.173532709318756195e-01,
        7.173231742352247364e-01,7.172931218675735465e-01,7.172631139155907221e-01,7.172331504651447975e-01,7.172032316013058351e-01,7.171733574083460905e-01,7.171435279697396803e-01,7.171137433681625817e-01,7.170840036854951860e-01,7.170543090028200783e-01,7.170246594004252572e-01,7.169950549578032462e-01,7.169654957536520934e-01,7.169359818658759265e-01,7.169065133715859517e-01,7.168770903471003431e-01,7.168477128679460186e-01,7.168183810088598618e-01,7.167890948437865006e-01,7.167598544458828602e-01,7.167306598875166079e-01,7.167015112402683741e-01,7.166724085749308637e-01,7.166433519615120762e-01,7.166143414692341951e-01,7.165853771665351424e-01,7.165564591210712431e-01,7.165275873997150047e-01,7.164987620685587810e-01,7.164699831929141061e-01,7.164412508373149135e-01,7.164125650655159827e-01,7.163839259404957138e-01,7.163553335244566833e-01,7.163267878788269760e-01,7.162982890642616285e-01,7.162698371406430731e-01,7.162414321670833584e-01,7.162130742019233720e-01,7.161847633027375037e-01,7.161564995263316469e-01,7.161282829287459739e-01,7.161001135652557137e-01,7.160719914903742600e-01,7.160439167578511732e-01,7.160158894206767322e-01,7.159879095310818231e-01,7.159599771405392721e-01,7.159320922997655101e-01,7.159042550587237930e-01,7.158764654666209815e-01,7.158487235719147579e-01,7.158210294223111836e-01,7.157933830647671414e-01,7.157657845454937773e-01,7.157382339099545021e-01,7.157107312028705426e-01,7.156832764682191650e-01,7.156558697492370058e-01,7.156285110884221812e-01,7.156012005275341759e-01,7.155739381075973959e-01,7.155467238689012799e-01,7.155195578510034071e-01,7.154924400927299422e-01,7.154653706321780771e-01,7.154383495067181409e-01,7.154113767529942658e-01,7.153844524069279398e-01,7.153575765037173406e-01,7.153307490778409994e-01,7.153039701630600211e-01,7.152772397924179737e-01,7.152505579982446626e-01,7.152239248121565751e-01,7.151973402650596556e-01,7.151708043871518594e-01,7.151443172079228194e-01,7.151178787561582872e-01,7.150914890599407991e-01,7.150651481466508974e-01,7.150388560429719043e-01,7.150126127748890337e-01,7.149864183676921670e-01,7.149602728459788503e-01,7.149341762336560713e-01,7.149081285539405917e-01,7.148821298293640547e-01,7.148561800817715417e-01,7.148302793323272342e-01,7.148044276015141918e-01,7.147786249091369060e-01,7.147528712743235202e-01,7.147271667155287167e-01,7.147015112505347156e-01,7.146759048964541616e-01,7.146503476697320112e-01,7.146248395861483083e-01,7.145993806608192944e-01,7.145739709082006286e-01,7.145486103420890522e-01,7.145232989756258313e-01,7.144980368212960897e-01,7.144728238909346940e-01,7.144476601957259199e-01,7.144225457462070050e-01,7.143974805522704807e-01,7.143724646231652819e-01,7.143474979674999670e-01,7.143225805932457151e-01,7.142977125077371037e-01,7.142728937176755499e-01,7.142481242291307542e-01,7.142234040475451406e-01,7.141987331777330805e-01,7.141741116238854437e-01,7.141495393895729293e-01,7.141250164777439569e-01,7.141005428907332142e-01,7.140761186302589936e-01,7.140517436974285204e-01,7.140274180927390635e-01,7.140031418160807108e-01,7.139789148667394780e-01,7.139547372433986405e-01,7.139306089441412873e-01,7.139065299664544284e-01,7.138825003072286624e-01,7.138585199627638378e-01,7.138345889287690538e-01,7.138107072003657683e-01,7.137868747720915730e-01,7.137630916379005264e-01,7.137393577911677056e-01,7.137156732246906499e-01,7.136920379306919138e-01,7.136684519008217320e-01,7.136449151261609058e-01,7.136214275972231347e-01,7.135979893039564592e-01,7.135746002357478135e-01,7.135512603814246901e-01,7.135279697292568057e-01,7.135047282669599866e-01,7.134815359816985003e-01,7.134583928600861658e-01,7.134352988881914603e-01,7.134122540515377420e-01,7.133892583351076899e-01,7.133663117233441930e-01,7.133434142001544576e-01,7.133205657489118945e-01,7.132977663524581180e-01,7.132750159931068312e-01,7.132523146526460467e-01,7.132296623123396406e-01,7.132070589529312388e-01,7.131845045546467698e-01,7.131619990971959089e-01,7.131395425597764071e-01,7.131171349210743138e-01,7.130947761592700829e-01,7.130724662520374624e-01,7.130502051765492677e-01,7.130279929094771596e-01,7.130058294269973063e-01,7.129837147047899393e-01,7.129616487180449047e-01,7.129396314414622182e-01,7.129176628492547296e-01,7.128957429151531189e-01,7.128738716124053409e-01,7.128520489137813998e-01,7.128302747915750137e-01,7.128085492176076121e-01,7.127868721632290017e-01,7.127652435993204749e-01,7.127436634963001394e-01,7.127221318241214743e-01,7.127006485522781043e-01,7.126792136498074637e-01,7.126578270852907959e-01,7.126364888268588160e-01,7.126151988421907113e-01,7.125939570985212468e-01,7.125727635626388778e-01,7.125516182008923005e-01,7.125305209791902294e-01,7.125094718630057278e-01,7.124884708173775394e-01,7.124675178069144188e-01,7.124466127957961303e-01,7.124257557477777780e-01,7.124049466261902497e-01,7.123841853939442137e-01,7.123634720135335607e-01,7.123428064470362919e-01,7.123221886561178495e-01,7.123016186020344476e-01,7.122810962456346262e-01,7.122606215473624713e-01,7.122401944672601681e-01,7.122198149649699994e-01,7.121994829997385645e-01,7.121791985304176675e-01,7.121589615154676478e-01,7.121387719129601557e-01,7.121186296805804838e-01,7.120985347756300099e-01,7.120784871550295270e-01,7.120584867753214642e-01,7.120385335926718851e-01,7.120186275628737071e-01,7.119987686413490335e-01,7.119789567831523724e-01,7.119591919429727467e-01,7.119394740751352479e-01,7.119198031336063659e-01,7.119001790719932110e-01,7.118806018435477334e-01,7.118610714011708307e-01,7.118415876974115708e-01,7.118221506844725210e-01,7.118027603142111914e-01,7.117834165381420330e-01,7.117641193074404349e-01,7.117448685729443891e-01,7.117256642851560455e-01,7.117065063942467074e-01,7.116873948500573865e-01,7.116683296021013572e-01,7.116493105995684854e-01,7.116303377913246742e-01,7.116114111259178587e-01,7.115925305515778954e-01,7.115736960162203362e-01,7.115549074674483165e-01,7.115361648525556637e-01,7.115174681185286731e-01,7.114988172120489951e-01,7.114802120794961882e-01,7.114616526669497176e-01,7.114431389201922862e-01,7.114246707847112772e-01,7.114062482057021963e-01,7.113878711280696709e-01,7.113695394964317797e-01,7.113512532551209411e-01,7.113330123481876877e-01,7.113148167194009996e-01,7.112966663122530786e-01,7.112785610699610128e-01,7.112605009354677765e-01,7.112424858514466708e-01,7.112245157603022117e-01,7.112065906041736829e-01,7.111887103249365794e-01,7.111708748642054934e-01,7.111530841633357802e-01,7.111353381634275550e-01,7.111176368053260255e-01,7.110999800296251561e-01,7.110823677766699991e-01,7.110647999865579161e-01,7.110472765991414645e-01,7.110297975540325055e-01,7.110123627906009824e-01,7.109949722479805834e-01,7.109776258650692959e-01,7.109603235805310728e-01,7.109430653328012717e-01,7.109258510600841019e-01,7.109086807003590636e-01,7.108915541913820579e-01,7.108744714706860535e-01,7.108574324755855267e-01,7.108404371431775726e-01,7.108234854103439027e-01,7.108065772137543981e-01,7.107897124898671093e-01,7.107728911749332523e-01,7.107561132049970976e-01,7.107393785158989674e-01,7.107226870432782340e-01,7.107060387225742071e-01,7.106894334890286880e-01,7.106728712776894108e-01,7.106563520234097098e-01,7.106398756608529599e-01,7.106234421244941313e-01,7.106070513486210105e-01,7.105907032673374202e-01,7.105743978145647732e-01,7.105581349240445155e-01,7.105419145293394578e-01,7.105257365638374401e-01,7.105096009607517749e-01,7.104935076531245786e-01,7.104774565738274372e-01,7.104614476555657365e-01,7.104454808308778846e-01,7.104295560321401970e-01,7.104136731915660086e-01,7.103978322412107804e-01,7.103820331129719889e-01,7.103662757385916793e-01,7.103505600496583527e-01,7.103348859776107416e-01,7.103192534537360325e-01,7.103036624091759732e-01,7.102881127749260948e-01,7.102726044818391538e-01,7.102571374606256871e-01,7.102417116418576759e-01,7.102263269559693226e-01,7.102109833332594935e-01,7.101956807038928288e-01,7.101804189979032955e-01,7.101651981451949647e-01,7.101500180755432323e-01,7.101348787185984834e-01,7.101197800038867580e-01,7.101047218608117495e-01,7.100897042186579133e-01,7.100747270065899119e-01,7.100597901536569445e-01,7.100448935887929691e-01,7.100300372408195893e-01,7.100152210384467200e-01,7.100004449102760296e-01,7.099857087848012727e-01,7.099710125904106217e-01,7.099563562553891094e-01,7.099417397079194059e-01,7.099271628760840391e-01,7.099126256878668384e-01,7.098981280711561537e-01,7.098836699537443007e-01,7.098692512633316687e-01,7.098548719275261654e-01,7.098405318738467695e-01,7.098262310297245303e-01,7.098119693225043436e-01,7.097977466794463952e-01,7.097835630277283814e-01,7.097694182944476182e-01,7.097553124066205976e-01,7.097412452911870950e-01,7.097272168750108357e-01,7.097132270848809377e-01,7.096992758475145768e-01,7.096853630895562093e-01,7.096714887375831227e-01,7.096576527181026606e-01,7.096438549575576626e-01,7.096300953823252433e-01,7.096163739187203445e-01,7.096026904929962908e-01,7.095890450313461217e-01,7.095754374599060332e-01,7.095618677047534906e-01,7.095483356919130014e-01,7.095348413473542282e-01,7.095213845969952082e-01,7.095079653667041297e-01,7.094945835822986657e-01,7.094812391695510811e-01,7.094679320541862344e-01,7.094546621618854632e-01,7.094414294182869174e-01,7.094282337489868917e-01,7.094150750795423788e-01,7.094019533354719576e-01,7.093888684422565705e-01,7.093758203253417438e-01,7.093628089101396972e-01,7.093498341220288994e-01,7.093368958863569551e-01,7.093239941284418260e-01,7.093111287735730519e-01,7.092982997470129725e-01,7.092855069739980589e-01,7.092727503797413569e-01,7.092600298894321531e-01,7.092473454282381962e-01,7.092346969213081387e-01,7.092220842937705383e-01,7.092095074707375213e-01,7.091969663773045607e-01,7.091844609385528075e-01,7.091719910795496462e-01,7.091595567253499155e-01,7.091471578009981291e-01,7.091347942315294750e-01,7.091224659419700371e-01,7.091101728573401264e-01,7.090979149026533923e-01,7.090856920029190436e-01,7.090735040831436242e-01,
        7.090613510683315690e-01,7.090492328834858693e-01,7.090371494536117369e-01,7.090251007037147168e-01,7.090130865588034625e-01,7.090011069438912905e-01,7.089891617839965132e-01,7.089772510041444376e-01,7.089653745293675868e-01,7.089535322847081433e-01,7.089417241952176152e-01,7.089299501859595010e-01,7.089182101820089565e-01,7.089065041084554597e-01,7.088948318904022550e-01,7.088831934529692402e-01,7.088715887212936329e-01,7.088600176205298586e-01,7.088484800758512172e-01,7.088369760124524355e-01,7.088255053555487795e-01,7.088140680303787189e-01,7.088026639622032610e-01,7.087912930763088371e-01,7.087799552980070805e-01,7.087686505526366032e-01,7.087573787655641055e-01,7.087461398621835995e-01,7.087349337679212935e-01,7.087237604082324838e-01,7.087126197086051071e-01,7.087015115945594079e-01,7.086904359916502694e-01,7.086793928254667696e-01,7.086683820216345131e-01,7.086574035058155197e-01,7.086464572037097787e-01,7.086355430410564704e-01,7.086246609436332999e-01,7.086138108372604938e-01,7.086029926477981356e-01,7.085922063011507177e-01,7.085814517232641441e-01,7.085707288401311699e-01,7.085600375777877380e-01,7.085493778623175309e-01,7.085387496198509716e-01,7.085281527765661114e-01,7.085175872586905177e-01,7.085070529925014959e-01,7.084965499043267556e-01,7.084860779205462977e-01,7.084756369675917487e-01,7.084652269719485806e-01,7.084548478601564447e-01,7.084444995588091709e-01,7.084341819945577656e-01,7.084238950941085244e-01,7.084136387842262517e-01,7.084034129917335942e-01,7.083932176435121519e-01,7.083830526665035876e-01,7.083729179877102933e-01,7.083628135341963894e-01,7.083527392330875028e-01,7.083426950115727649e-01,7.083326807969053673e-01,7.083226965164025613e-01,7.083127420974472122e-01,7.083028174674875777e-01,7.082929225540400830e-01,7.082830572846872119e-01,7.082732215870809478e-01,7.082634153889408868e-01,7.082536386180574572e-01,7.082438912022909205e-01,7.082341730695738136e-01,7.082244841479081732e-01,7.082148243653703101e-01,7.082051936501098099e-01,7.081955919303486446e-01,7.081860191343848365e-01,7.081764751905914590e-01,7.081669600274159704e-01,7.081574735733838777e-01,7.081480157570974043e-01,7.081385865072364894e-01,7.081291857525592315e-01,7.081198134219034435e-01,7.081104694441856529e-01,7.081011537484040996e-01,7.080918662636367378e-01,7.080826069190431227e-01,7.080733756438657434e-01,7.080641723674291343e-01,7.080549970191415410e-01,7.080458495284945863e-01,7.080367298250644925e-01,7.080276378385131908e-01,7.080185734985872115e-01,7.080095367351197932e-01,7.080005274780313274e-01,7.079915456573285804e-01,7.079825912031064705e-01,7.079736640455486230e-01,7.079647641149267034e-01,7.079558913416029720e-01,7.079470456560283953e-01,7.079382269887458667e-01,7.079294352703869864e-01,7.079206704316773902e-01,7.079119324034324201e-01,7.079032211165618982e-01,7.078945365020663516e-01,7.078858784910422308e-01,7.078772470146778018e-01,7.078686420042566985e-01,7.078600633911575901e-01,7.078515111068531818e-01,7.078429850829139891e-01,7.078344852510045637e-01,7.078260115428881560e-01,7.078175638904230516e-01,7.078091422255666787e-01,7.078007464803743876e-01,7.077923765869984507e-01,7.077840324776916159e-01,7.077757140848052186e-01,7.077674213407902926e-01,7.077591541781979023e-01,7.077509125296792547e-01,7.077426963279870309e-01,7.077345055059751644e-01,7.077263399965982860e-01,7.077181997329141661e-01,7.077100846480822716e-01,7.077019946753652091e-01,7.076939297481289470e-01,7.076858897998421494e-01,7.076778747640779521e-01,7.076698845745140742e-01,7.076619191649317075e-01,7.076539784692182922e-01,7.076460624213655182e-01,7.076381709554707689e-01,7.076303040057377869e-01,7.076224615064767853e-01,7.076146433921040035e-01,7.076068495971428174e-01,7.075990800562231842e-01,7.075913347040840851e-01,7.075836134755706386e-01,7.075759163056374312e-01,7.075682431293467411e-01,7.075605938818699814e-01,7.075529684984874779e-01,7.075453669145879143e-01,7.075377890656714408e-01,7.075302348873470093e-01,7.075227043153331508e-01,7.075151972854594185e-01,7.075077137336663879e-01,7.075002535960047689e-01,7.074928168086371816e-01,7.074854033078374904e-01,7.074780130299906933e-01,7.074706459115943646e-01,7.074633018892575453e-01,7.074559808997027410e-01,7.074486828797643678e-01,7.074414077663891964e-01,7.074341554966385726e-01,7.074269260076855304e-01,7.074197192368172349e-01,7.074125351214354263e-01,7.074053735990547542e-01,7.073982346073041105e-01,7.073911180839274060e-01,7.073840239667821272e-01,7.073769521938411131e-01,7.073699027031924436e-01,7.073628754330383295e-01,7.073558703216974441e-01,7.073488873076030359e-01,7.073419263293040382e-01,7.073349873254662912e-01,7.073280702348704319e-01,7.073211749964133377e-01,7.073143015491092367e-01,7.073074498320878201e-01,7.073006197845957965e-01,7.072938113459966702e-01,7.072870244557708519e-01,7.072802590535161027e-01,7.072735150789467573e-01,7.072667924718953891e-01,7.072600911723111450e-01,7.072534111202620766e-01,7.072467522559323649e-01,7.072401145196247629e-01,7.072334978517610393e-01,7.072269021928798693e-01,7.072203274836386111e-01,7.072137736648129724e-01,7.072072406772969000e-01,7.072007284621031342e-01,7.071942369603632095e-01,7.071877661133276760e-01,7.071813158623649898e-01,7.071748861489636218e-01,7.071684769147300598e-01,7.071620881013909177e-01,7.071557196507924914e-01,7.071493715048985385e-01,7.071430436057939417e-01,7.071367358956814897e-01,7.071304483168855404e-01,7.071241808118482464e-01,7.071179333231327746e-01,7.071117057934211969e-01,7.071054981655152671e-01,7.070993103823376424e-01,7.070931423869301069e-01,7.070869941224543487e-01,7.070808655321930702e-01,7.070747565595481010e-01,7.070686671480412855e-01,7.070625972413154825e-01,7.070565467831331219e-01,7.070505157173773148e-01,7.070445039880516314e-01,7.070385115392791020e-01,7.070325383153041043e-01,7.070265842604904760e-01,7.070206493193231800e-01,7.070147334364079716e-01,7.070088365564698440e-01,7.070029586243550268e-01,7.069970995850308748e-01,7.069912593835834258e-01,7.069854379652213971e-01,7.069796352752720781e-01,7.069738512591853263e-01,7.069680858625295716e-01,7.069623390309951461e-01,7.069566107103923969e-01,7.069509008466520195e-01,7.069452093858258346e-01,7.069395362740855671e-01,7.069338814577237340e-01,7.069282448831537558e-01,7.069226264969090678e-01,7.069170262456434539e-01,7.069114440761312679e-01,7.069058799352682110e-01,7.069003337700685563e-01,7.068948055276692566e-01,7.068892951553251702e-01,7.068838026004137243e-01,7.068783278104312506e-01,7.068728707329948735e-01,7.068674313158418432e-01,7.068620095068297582e-01,7.068566052539360101e-01,7.068512185052584496e-01,7.068458492090148315e-01,7.068404973135432590e-01,7.068351627673017390e-01,7.068298455188676277e-01,7.068245455169389624e-01,7.068192627103329073e-01,7.068139970479868639e-01,7.068087484789580266e-01,7.068035169524234940e-01,7.067983024176789364e-01,7.067931048241407055e-01,7.067879241213437247e-01,7.067827602589429326e-01,7.067776131867127276e-01,7.067724828545466353e-01,7.067673692124561979e-01,7.067622722105743049e-01,7.067571917991514185e-01,7.067521279285570168e-01,7.067470805492804820e-01,7.067420496119288797e-01,7.067370350672288470e-01,7.067320368660252594e-01,7.067270549592817863e-01,7.067220892980804470e-01,7.067171398336219434e-01,7.067122065172247725e-01,7.067072893003260026e-01,7.067023881344812741e-01,7.066975029713639112e-01,7.066926337627650323e-01,7.066877804605934399e-01,7.066829430168759529e-01,7.066781213837579623e-01,7.066733155135010991e-01,7.066685253584849002e-01,7.066637508712065863e-01,7.066589920042800621e-01,7.066542487104364723e-01,7.066495209425252000e-01,7.066448086535108697e-01,7.066401117964756784e-01,7.066354303246188406e-01,7.066307641912557003e-01,7.066261133498182856e-01,7.066214777538553093e-01,7.066168573570307254e-01,7.066122521131261713e-01,7.066076619760378597e-01,7.066030868997784653e-01,7.065985268384767926e-01,7.065939817463768868e-01,7.065894515778385898e-01,7.065849362873367623e-01,7.065804358294617282e-01,7.065759501589191638e-01,7.065714792305296532e-01,7.065670229992285778e-01,7.065625814200662269e-01,7.065581544482077980e-01,7.065537420389316203e-01,7.065493441476325964e-01,7.065449607298178725e-01,7.065405917411097247e-01,7.065362371372442274e-01,7.065318968740713634e-01,7.065275709075541366e-01,7.065232591937700146e-01,7.065189616889095969e-01,7.065146783492761706e-01,7.065104091312865986e-01,7.065061539914705424e-01,7.065019128864711284e-01,7.064976857730429494e-01,7.064934726080540628e-01,7.064892733484847698e-01,7.064850879514275039e-01,7.064809163740861653e-01,7.064767585737776745e-01,7.064726145079304187e-01,7.064684841340835852e-01,7.064643674098887161e-01,7.064602642931084864e-01,7.064561747416160387e-01,7.064520987133968699e-01,7.064480361665460562e-01,7.064439870592701398e-01,7.064399513498860195e-01,7.064359289968203948e-01,7.064319199586110987e-01,7.064279241939053211e-01,7.064239416614598310e-01,7.064199723201425307e-01,7.064160161289294582e-01,7.064120730469066745e-01,7.064081430332692646e-01,7.064042260473218926e-01,7.064003220484769141e-01,7.063964309962572630e-01,7.063925528502924545e-01,7.063886875703218049e-01,7.063848351161919892e-01,7.063809954478582620e-01,7.063771685253839028e-01,7.063733543089391054e-01,7.063695527588024214e-01,7.063657638353595392e-01,7.063619874991027281e-01,7.063582237106321715e-01,7.063544724306548561e-01,7.063507336199833508e-01,7.063470072395379162e-01,7.063432932503447281e-01,7.063395916135353225e-01,7.063359022903491491e-01,7.063322252421295744e-01,7.063285604303263243e-01,7.063249078164940409e-01,7.063212673622935034e-01,7.063176390294902962e-01,7.063140227799542537e-01,7.063104185756606812e-01,7.063068263786888012e-01,7.063032461512223081e-01,7.062996778555499233e-01,7.062961214540627308e-01,7.062925769092572859e-01,7.062890441837326172e-01,7.062855232401913375e-01,7.062820140414399761e-01,7.062785165503879803e-01,7.062750307300464936e-01,7.062715565435307985e-01,7.062680939540577629e-01,
        7.062646429249469504e-01,7.062612034196203981e-01,7.062577754016007292e-01,7.062543588345141510e-01,7.062509536820873457e-01,7.062475599081484701e-01,7.062441774766271552e-01,7.062408063515536183e-01,7.062374464970587740e-01,7.062340978773752331e-01,7.062307604568348607e-01,7.062274341998697746e-01,7.062241190710133454e-01,7.062208150348980862e-01,7.062175220562546540e-01,7.062142400999160685e-01,7.062109691308122716e-01,7.062077091139740137e-01,7.062044600145290785e-01,7.062012217977053918e-01,7.061979944288289124e-01,7.061947778733241865e-01,7.061915720967130161e-01,7.061883770646162350e-01,7.061851927427517106e-01,7.061820190969352318e-01,7.061788560930791769e-01,7.061757036971939572e-01,7.061725618753865730e-01,7.061694305938609473e-01,7.061663098189170373e-01,7.061631995169517229e-01,7.061600996544580289e-01,7.061570101980241265e-01,7.061539311143351094e-01,7.061508623701709952e-01,7.061478039324070588e-01,7.061447557680140541e-01,7.061417178440577702e-01,7.061386901276989203e-01,7.061356725861920314e-01,7.061326651868869986e-01,7.061296678972268648e-01,7.061266806847498190e-01,7.061237035170868648e-01,7.061207363619631527e-01,7.061177791871970921e-01,7.061148319607002399e-01,7.061118946504769678e-01,7.061089672246247950e-01,7.061060496513340556e-01,7.061031418988870101e-01,7.061002439356579563e-01,7.060973557301135628e-01,7.060944772508120915e-01,7.060916084664045078e-01,7.060887493456311503e-01,7.060858998573251721e-01,7.060830599704098764e-01,7.060802296539000489e-01,7.060774088769009582e-01,7.060745976086074682e-01,7.060717958183055920e-01,7.060690034753709377e-01,7.060662205492690413e-01,7.060634470095553672e-01,7.060606828258738643e-01,7.060579279679584097e-01,7.060551824056320314e-01,7.060524461088062420e-01,7.060497190474810392e-01,7.060470011917452382e-01,7.060442925117750290e-01,7.060415929778360855e-01,7.060389025602806790e-01,7.060362212295487883e-01,7.060335489561687661e-01,7.060308857107542302e-01,7.060282314640086154e-01,7.060255861867192895e-01,7.060229498497624379e-01,7.060203224240994002e-01,7.060177038807784466e-01,7.060150941909328903e-01,7.060124933257833080e-01,7.060099012566340981e-01,7.060073179548771449e-01,7.060047433919878213e-01,7.060021775395274313e-01,7.059996203691414340e-01,7.059970718525608868e-01,7.059945319616004467e-01,7.059920006681592586e-01,7.059894779442205115e-01,7.059869637618508831e-01,7.059844580932015390e-01,7.059819609105056903e-01,7.059794721860813693e-01,7.059769918923284315e-01,7.059745200017304434e-01,7.059720564868530168e-01,7.059696013203438092e-01,7.059671544749338556e-01,7.059647159234352376e-01,7.059622856387425260e-01,7.059598635938317823e-01,7.059574497617604472e-01,7.059550441156663414e-01,7.059526466287697755e-01,7.059502572743712179e-01,7.059478760258522945e-01,7.059455028566744561e-01,7.059431377403792007e-01,7.059407806505891836e-01,7.059384315610057747e-01,7.059360904454106134e-01,7.059337572776647196e-01,7.059314320317089386e-01,7.059291146815621643e-01,7.059268052013224493e-01,7.059245035651671163e-01,7.059222097473517588e-01,7.059199237222101297e-01,7.059176454641539200e-01,7.059153749476727580e-01,7.059131121473346537e-01,7.059108570377846670e-01,7.059086095937447958e-01,7.059063697900150869e-01,7.059041376014711933e-01,7.059019130030672606e-01,7.058996959698320417e-01,7.058974864768726709e-01,7.058952844993708897e-01,7.058930900125852670e-01,7.058909029918495337e-01,7.058887234125734711e-01,7.058865512502421335e-01,7.058843864804158486e-01,7.058822290787302167e-01,7.058800790208944465e-01,7.058779362826940185e-01,7.058758008399881323e-01,7.058736726687095953e-01,7.058715517448664878e-01,7.058694380445396099e-01,7.058673315438843687e-01,7.058652322191291129e-01,7.058631400465753547e-01,7.058610550025985475e-01,7.058589770636459759e-01,7.058569062062385324e-01,7.058548424069692739e-01,7.058527856425035329e-01,7.058507358895788064e-01,7.058486931250050889e-01,7.058466573256627630e-01,7.058446284685057082e-01,7.058426065305577479e-01,7.058405914889149813e-01,7.058385833207433402e-01,7.058365820032805882e-01,7.058345875138348768e-01,7.058325998297847459e-01,7.058306189285792343e-01,7.058286447877369918e-01,7.058266773848469455e-01,7.058247166975678555e-01,7.058227627036276486e-01,7.058208153808241958e-01,7.058188747070238689e-01,7.058169406601626505e-01,7.058150132182448022e-01,7.058130923593431971e-01,7.058111780615999864e-01,7.058092703032248227e-01,7.058073690624948604e-01,7.058054743177568646e-01,7.058035860474241030e-01,7.058017042299770116e-01,7.057998288439641943e-01,7.057979598680013122e-01,7.057960972807711952e-01,7.057942410610228423e-01,7.057923911875716438e-01,7.057905476393007138e-01,7.057887103951584473e-01,7.057868794341592977e-01,7.057850547353837767e-01,7.057832362779785651e-01,7.057814240411555140e-01,7.057796180041914225e-01,7.057778181464291478e-01,7.057760244472760514e-01,7.057742368862039983e-01,7.057724554427501351e-01,7.057706800965158900e-01,7.057689108271667511e-01,7.057671476144322664e-01,7.057653904381063770e-01,7.057636392780468615e-01,7.057618941141743374e-01,7.057601549264730378e-01,7.057584216949917000e-01,7.057566943998400122e-01,7.057549730211922778e-01,7.057532575392850838e-01,7.057515479344164122e-01,7.057498441869483052e-01,7.057481462773042002e-01,7.057464541859697071e-01,7.057447678934917201e-01,7.057430873804796390e-01,7.057414126276042587e-01,7.057397436155968817e-01,7.057380803252502055e-01,7.057364227374194332e-01,7.057347708330191649e-01,7.057331245930241748e-01,7.057314839984706323e-01,7.057298490304552141e-01,7.057282196701343269e-01,7.057265958987241072e-01,7.057249776975008659e-01,7.057233650478004217e-01,7.057217579310177680e-01,7.057201563286082946e-01,7.057185602220850118e-01,7.057169695930204378e-01,7.057153844230468209e-01,7.057138046938535858e-01,7.057122303871894431e-01,7.057106614848612791e-01,7.057090979687348220e-01,7.057075398207316441e-01,7.057059870228332699e-01,7.057044395570780670e-01,7.057028974055619130e-01,7.057013605504376397e-01,7.056998289739154773e-01,7.056983026582633878e-01,7.056967815858047333e-01,7.056952657389208294e-01,7.056937551000485032e-01,7.056922496516813137e-01,7.056907493763688866e-01,7.056892542567173576e-01,7.056877642753879298e-01,7.056862794150982054e-01,7.056847996586205207e-01,7.056833249887831672e-01,7.056818553884692813e-01,7.056803908406180659e-01,7.056789313282215703e-01,7.056774768343284654e-01,7.056760273420416008e-01,7.056745828345168947e-01,7.056731432949666649e-01,7.056717087066554095e-01,7.056702790529029157e-01,7.056688543170819283e-01,7.056674344826187051e-01,7.056660195329940155e-01,7.056646094517405876e-01,7.056632042224457724e-01,7.056618038287484351e-01,7.056604082543411760e-01,7.056590174829692197e-01,7.056576314984300824e-01,7.056562502845737939e-01,7.056548738253025643e-01,7.056535021045706735e-01,7.056521351063843595e-01,7.056507728148015968e-01,7.056494152139322074e-01,7.056480622879370834e-01,7.056467140210286315e-01,7.056453703974701064e-01,7.056440314015770543e-01,7.056426970177137603e-01,7.056413672302968010e-01,7.056400420237930460e-01,7.056387213827193250e-01,7.056374052916435380e-01,7.056360937351828788e-01,7.056347866980048344e-01,7.056334841648260747e-01,7.056321861204144508e-01,7.056308925495862194e-01,7.056296034372069315e-01,7.056283187681922087e-01,7.056270385275053014e-01,7.056257627001603083e-01,7.056244912712185124e-01,7.056232242257906018e-01,7.056219615490358921e-01,7.056207032261613277e-01,7.056194492424227027e-01,7.056181995831235509e-01,7.056169542336158118e-01,7.056157131792989423e-01,7.056144764056191399e-01,7.056132438980718957e-01,7.056120156421987755e-01,7.056107916235887512e-01,7.056095718278775353e-01,7.056083562407489129e-01,7.056071448479322994e-01,7.056059376352044055e-01,7.056047345883886823e-01,7.056035356933533231e-01,7.056023409360151488e-01,7.056011503023351672e-01,7.055999637783210154e-01,7.055987813500265160e-01,7.055976030035503443e-01,7.055964287250373612e-01,7.055952585006781685e-01,7.055940923167070000e-01,7.055929301594049408e-01,7.055917720150970407e-01,7.055906178701542020e-01,7.055894677109905144e-01,7.055883215240666972e-01,7.055871792958854360e-01,7.055860410129958238e-01,7.055849066619899190e-01,7.055837762295047444e-01,7.055826497022200661e-01,7.055815270668606143e-01,7.055804083101935298e-01,7.055792934190312504e-01,7.055781823802268482e-01,7.055770751806796914e-01,7.055759718073304487e-01,7.055748722471625323e-01,7.055737764872033191e-01,7.055726845145218196e-01,7.055715963162311199e-01,7.055705118794850517e-01,7.055694311914809669e-01,7.055683542394574070e-01,7.055672810106961013e-01,7.055662114925196349e-01,7.055651456722933368e-01,7.055640835374235031e-01,7.055630250753585075e-01,7.055619702735878018e-01,7.055609191196421381e-01,7.055598716010935689e-01,7.055588277055551139e-01,7.055577874206806488e-01,7.055567507341651279e-01,7.055557176337436953e-01,7.055546881071927956e-01,7.055536621423278421e-01,7.055526397270065475e-01,7.055516208491245944e-01,7.055506054966197427e-01,7.055495936574687210e-01,7.055485853196874491e-01,7.055475804713324806e-01,7.055465791004993381e-01,7.055455811953231793e-01,7.055445867439787966e-01,7.055435957346796183e-01,7.055426081556784856e-01,7.055416239952672086e-01,7.055406432417752338e-01,7.055396658835726420e-01,7.055386919090669284e-01,7.055377213067041131e-01,7.055367540649689628e-01,7.055357901723837699e-01,7.055348296175099065e-01,7.055338723889454933e-01,7.055329184753275085e-01,7.055319678653302340e-01,7.055310205476656993e-01,7.055300765110831263e-01,7.055291357443697065e-01,7.055281982363494908e-01,7.055272639758840558e-01,7.055263329518712823e-01,7.055254051532464654e-01,7.055244805689824261e-01,7.055235591880866242e-01,7.055226409996051551e-01,7.055217259926197526e-01,7.055208141562484547e-01,7.055199054796457148e-01,7.055189999520019573e-01,7.055180975625439110e-01,7.055171983005333880e-01,7.055163021552688374e-01,7.055154091160845686e-01,7.055145191723490861e-01,7.055136323134678644e-01,7.055127485288804623e-01,
        7.055118678080629646e-01,7.055109901405252071e-01,7.055101155158134407e-01,7.055092439235068902e-01,7.055083753532218616e-01,7.055075097946071905e-01,7.055066472373476838e-01,7.055057876711626763e-01,7.055049310858046985e-01,7.055040774710605866e-01,7.055032268167529264e-01,7.055023791127367216e-01,7.055015343489010604e-01,7.055006925151704467e-01,7.054998536015004706e-01,7.054990175978823608e-01,7.054981844943395419e-01,7.054973542809297449e-01,7.054965269477436740e-01,7.054957024849051184e-01,7.054948808825709516e-01,7.054940621309303550e-01,7.054932462202069265e-01,7.054924331406553506e-01,7.054916228825641733e-01,7.054908154362533601e-01,7.054900107920761831e-01,7.054892089404182220e-01,7.054884098716970309e-01,7.054876135763619160e-01,7.054868200448946025e-01,7.054860292678090117e-01,7.054852412356502622e-01,7.054844559389955583e-01,7.054836733684538563e-01,7.054828935146651991e-01,7.054821163683013818e-01,7.054813419200651747e-01,7.054805701606912116e-01,7.054798010809445463e-01,7.054790346716215410e-01,7.054782709235498661e-01,7.054775098275866130e-01,7.054767513746210694e-01,7.054759955555723883e-01,7.054752423613911416e-01,7.054744917830572115e-01,7.054737438115809001e-01,7.054729984380040397e-01,7.054722556533965516e-01,7.054715154488601092e-01,7.054707778155254738e-01,7.054700427445536048e-01,7.054693102271347716e-01,7.054685802544896633e-01,7.054678528178675023e-01,7.054671279085478197e-01,7.054664055178393456e-01,7.054656856370792317e-01,7.054649682576351610e-01,7.054642533709034602e-01,7.054635409683087666e-01,7.054628310413045833e-01,7.054621235813746116e-01,7.054614185800299753e-01,7.054607160288105527e-01,7.054600159192853104e-01,7.054593182430511922e-01,7.054586229917337858e-01,7.054579301569859906e-01,7.054572397304910147e-01,7.054565517039573797e-01,7.054558660691235827e-01,7.054551828177549888e-01,7.054545019416457174e-01,7.054538234326159785e-01,7.054531472825158467e-01,7.054524734832211541e-01,7.054518020266350442e-01,7.054511329046896373e-01,7.054504661093426998e-01,7.054498016325805310e-01,7.054491394664147430e-01,7.054484796028854809e-01,7.054478220340600902e-01,7.054471667520303413e-01,7.054465137489177584e-01,7.054458630168685129e-01,7.054452145480557546e-01,7.054445683346797225e-01,7.054439243689656358e-01,7.054432826431672465e-01,7.054426431495625094e-01,7.054420058804562466e-01,7.054413708281795925e-01,7.054407379850894388e-01,7.054401073435678793e-01,7.054394788960242080e-01,7.054388526348914779e-01,7.054382285526302754e-01,7.054376066417262781e-01,7.054369868946900324e-01,7.054363693040565098e-01,7.054357538623893253e-01,7.054351405622737436e-01,7.054345293963218966e-01,7.054339203571705630e-01,7.054333134374812797e-01,7.054327086299416738e-01,7.054321059272626870e-01,7.054315053221799081e-01,7.054309068074554601e-01,7.054303103758734483e-01,7.054297160202446237e-01,7.054291237334036069e-01,7.054285335082074448e-01,7.054279453375404962e-01,7.054273592143088800e-01,7.054267751314441393e-01,7.054261930819013537e-01,7.054256130586592510e-01,7.054250350547202064e-01,7.054244590631114642e-01,7.054238850768834723e-01,7.054233130891097714e-01,7.054227430928877718e-01,7.054221750813385317e-01,7.054216090476064238e-01,7.054210449848583586e-01,7.054204828862855603e-01,7.054199227451021237e-01,7.054193645545444591e-01,7.054188083078731797e-01,7.054182539983711031e-01,7.054177016193436955e-01,7.054171511641197378e-01,7.054166026260499933e-01,7.054160559985090950e-01,7.054155112748927703e-01,7.054149684486201721e-01,7.054144275131323250e-01,7.054138884618932348e-01,7.054133512883883350e-01,7.054128159861259295e-01,7.054122825486356385e-01,7.054117509694697308e-01,7.054112212422031236e-01,7.054106933604307184e-01,7.054101673177710641e-01,7.054096431078632490e-01,7.054091207243687878e-01,7.054086001609699563e-01,7.054080814113717901e-01,7.054075644692995306e-01,7.054070493285006238e-01,7.054065359827438320e-01,7.054060244258183454e-01,7.054055146515353369e-01,7.054050066537266295e-01,7.054045004262459173e-01,7.054039959629662127e-01,7.054034932577830652e-01,7.054029923046123418e-01,7.054024930973895602e-01,7.054019956300732197e-01,7.054014998966400274e-01,7.054010058910893388e-01,7.054005136074390503e-01,7.054000230397287075e-01,7.053995341820177289e-01,7.053990470283870717e-01,7.053985615729354564e-01,7.053980778097835858e-01,7.053975957330720359e-01,7.053971153369607006e-01,7.053966366156300127e-01,7.053961595632801673e-01,7.053956841741315653e-01,7.053952104424225933e-01,7.053947383624139533e-01,7.053942679283847772e-01,7.053937991346326264e-01,7.053933319754760456e-01,7.053928664452525643e-01,7.053924025383185858e-01,7.053919402490509416e-01,7.053914795718444486e-01,7.053910205011136858e-01,7.053905630312917729e-01,7.053901071568329240e-01,7.053896528722075621e-01,7.053892001719062055e-01,7.053887490504388014e-01,7.053882995023325053e-01,7.053878515221357892e-01,7.053874051044128901e-01,7.053869602437488062e-01,7.053865169347454112e-01,7.053860751720240074e-01,7.053856349502252154e-01,7.053851962640058648e-01,7.053847591080426582e-01,7.053843234770297288e-01,7.053838893656798614e-01,7.053834567687242707e-01,7.053830256809107135e-01,7.053825960970067088e-01,7.053821680117967619e-01,7.053817414200834746e-01,7.053813163166869904e-01,7.053808926964459936e-01,7.053804705542155995e-01,7.053800498848695755e-01,7.053796306832988972e-01,7.053792129444119707e-01,7.053787966631350770e-01,7.053783818344109280e-01,7.053779684532007765e-01,7.053775565144823068e-01,7.053771460132515214e-01,7.053767369445196334e-01,7.053763293033169512e-01,7.053759230846891048e-01,7.053755182837004867e-01,7.053751148954314765e-01,7.053747129149789963e-01,7.053743123374569546e-01,7.053739131579966903e-01,7.053735153717459738e-01,7.053731189738692287e-01,7.053727239595463105e-01,7.053723303239756159e-01,7.053719380623704183e-01,7.053715471699618655e-01,7.053711576419949836e-01,7.053707694737351153e-01,7.053703826604598159e-01,7.053699971974655147e-01,7.053696130800630737e-01,7.053692303035808964e-01,7.053688488633619302e-01,7.053684687547673304e-01,7.053680899731715748e-01,7.053677125139675708e-01,7.053673363725612155e-01,7.053669615443768359e-01,7.053665880248534137e-01,7.053662158094445855e-01,7.053658448936219738e-01,7.053654752728707455e-01,7.053651069426922771e-01,7.053647398986032657e-01,7.053643741361357300e-01,7.053640096508381196e-01,7.053636464382724292e-01,7.053632844940176394e-01,7.053629238136667201e-01,7.053625643928277400e-01,7.053622062271248661e-01,7.053618493121965871e-01,7.053614936436967131e-01,7.053611392172943750e-01,7.053607860286720266e-01,7.053604340735286637e-01,7.053600833475777154e-01,7.053597338465458222e-01,7.053593855661776102e-01,7.053590385022288078e-01,7.053586926504723520e-01,7.053583480066940581e-01,7.053580045666947296e-01,7.053576623262904910e-01,7.053573212813099014e-01,7.053569814275988392e-01,7.053566427610146183e-01,7.053563052774305397e-01,7.053559689727331161e-01,7.053556338428235151e-01,7.053552998836174481e-01,7.053549670910439495e-01,7.053546354610461533e-01,7.053543049895819594e-01,7.053539756726212584e-01,7.053536475061504829e-01,7.053533204861687222e-01,7.053529946086878333e-01,7.053526698697342168e-01,7.053523462653492615e-01,7.053520237915850144e-01,7.053517024445109529e-01,7.053513822202065464e-01,7.053510631147665855e-01,7.053507451242994053e-01,7.053504282449258866e-01,7.053501124727811211e-01,7.053497978040121907e-01,7.053494842347824978e-01,7.053491717612652145e-01,7.053488603796475021e-01,7.053485500861319535e-01,7.053482408769309320e-01,7.053479327482732320e-01,7.053476256963979729e-01,7.053473197175580411e-01,7.053470148080197566e-01,7.053467109640617627e-01,7.053464081819765807e-01,7.053461064580675011e-01,7.053458057886524690e-01,7.053455061700620865e-01,7.053452075986377245e-01,7.053449100707350761e-01,7.053446135827224905e-01,7.053443181309801968e-01,7.053440237119006362e-01,7.053437303218895726e-01,7.053434379573645385e-01,7.053431466147557227e-01,7.053428562905054156e-01,7.053425669810684528e-01,7.053422786829113278e-01,7.053419913925146334e-01,7.053417051063680665e-01,7.053414198209762009e-01,7.053411355328540466e-01,7.053408522385287149e-01,7.053405699345409730e-01,7.053402886174405806e-01,7.053400082837926188e-01,7.053397289301713835e-01,7.053394505531646042e-01,7.053391731493711125e-01,7.053388967154013978e-01,7.053386212478770512e-01,7.053383467434337639e-01,7.053380731987162200e-01,7.053378006103814268e-01,7.053375289750991595e-01,7.053372582895495180e-01,7.053369885504234826e-01,7.053367197544244682e-01,7.053364518982683240e-01,7.053361849786798921e-01,7.053359189923968930e-01,7.053356539361679278e-01,7.053353898067533656e-01,7.053351266009239007e-01,7.053348643154614406e-01,7.053346029471589951e-01,7.053343424928222305e-01,7.053340829492653619e-01,7.053338243133163710e-01,7.053335665818117883e-01,7.053333097515999128e-01,7.053330538195403676e-01,7.053327987825036560e-01,7.053325446373702734e-01,7.053322913810323724e-01,7.053320390103925419e-01,7.053317875223636957e-01,7.053315369138711821e-01,7.053312871818472329e-01,7.053310383232392899e-01,7.053307903350019004e-01,7.053305432141019349e-01,7.053302969575160342e-01,7.053300515622310529e-01,7.053298070252460583e-01,7.053295633435683332e-01,7.053293205142155964e-01,7.053290785342184455e-01,7.053288374006153605e-01,7.053285971104548135e-01,7.053283576607971561e-01,7.053281190487106223e-01,7.053278812712778789e-01,7.053276443255864780e-01,7.053274082087381824e-01,7.053271729178428595e-01,7.053269384500194805e-01,7.053267048023994512e-01,7.053264719721218379e-01,7.053262399563370311e-01,7.053260087522057464e-01,7.053257783568965822e-01,7.053255487675889057e-01,7.053253199814724095e-01,7.053250919957470000e-01,7.053248648076192451e-01,7.053246384143097014e-01,7.053244128130451429e-01,7.053241880010630016e-01,7.053239639756119228e-01,7.053237407339472131e-01,7.053235182733356146e-01,7.053232965910529728e-01,7.053230756843842375e-01,7.053228555506241282e-01,7.053226361870770234e-01,
        7.053224175910552951e-01,7.053221997598824178e-01,7.053219826908895262e-01,7.053217663814177474e-01,7.053215508288188662e-01,7.053213360304517732e-01,7.053211219836841295e-01,7.053209086858942545e-01,7.053206961344701265e-01,7.053204843268066071e-01,7.053202732603093272e-01,7.053200629323921333e-01,7.053198533404781978e-01,7.053196444819986866e-01,7.053194363543959788e-01,7.053192289551181160e-01,7.053190222816244637e-01,7.053188163313822701e-01,7.053186111018688864e-01,7.053184065905671041e-01,7.053182027949714827e-01,7.053179997125859080e-01,7.053177973409195944e-01,7.053175956774918598e-01,7.053173947198326799e-01,7.053171944654783587e-01,7.053169949119726390e-01,7.053167960568712536e-01,7.053165978977364858e-01,7.053164004321391678e-01,7.053162036576574589e-01,7.053160075718799549e-01,7.053158121724025786e-01,7.053156174568308012e-01,7.053154234227754227e-01,7.053152300678591224e-01,7.053150373897104641e-01,7.053148453859665601e-01,7.053146540542728493e-01,7.053144633922850959e-01,7.053142733976639489e-01,7.053140840680793833e-01,7.053138954012113659e-01,7.053137073947435276e-01,7.053135200463728216e-01,7.053133333537996430e-01,7.053131473147348229e-01,7.053129619268972972e-01,7.053127771880129959e-01,7.053125930958161760e-01,7.053124096480472005e-01,7.053122268424593111e-01,7.053120446768080809e-01,7.053118631488577428e-01,7.053116822563828547e-01,7.053115019971645250e-01,7.053113223689910782e-01,7.053111433696590549e-01,7.053109649969724337e-01,7.053107872487425212e-01,7.053106101227885061e-01,7.053104336169382371e-01,7.053102577290237818e-01,7.053100824568899752e-01,7.053099077983839837e-01,7.053097337513634102e-01,7.053095603136919634e-01,7.053093874832430110e-01,7.053092152578940288e-01,7.053090436355317072e-01,7.053088726140515075e-01,7.053087021913521104e-01,7.053085323653441874e-01,7.053083631339414072e-01,7.053081944950688742e-01,7.053080264466552451e-01,7.053078589866388359e-01,7.053076921129634025e-01,7.053075258235802503e-01,7.053073601164503437e-01,7.053071949895379777e-01,7.053070304408163294e-01,7.053068664682662359e-01,7.053067030698728646e-01,7.053065402436331510e-01,7.053063779875461403e-01,7.053062162996200923e-01,7.053060551778697063e-01,7.053058946203188961e-01,7.053057346249927972e-01,7.053055751899302006e-01,7.053054163131721177e-01,7.053052579927676646e-01,7.053051002267729519e-01,7.053049430132517505e-01,7.053047863502720505e-01,7.053046302359113895e-01,7.053044746682520794e-01,7.053043196453828712e-01,7.053041651654015087e-01,7.053040112264101769e-01,7.053038578265184988e-01,7.053037049638427591e-01,7.053035526365044605e-01,7.053034008426337653e-01,7.053032495803661650e-01,7.053030988478441454e-01,7.053029486432147444e-01,7.053027989646347695e-01,7.053026498102644704e-01,7.053025011782727560e-01,7.053023530668331986e-01,7.053022054741259206e-01,7.053020583983400371e-01,7.053019118376659957e-01,7.053017657903046800e-01,7.053016202544619695e-01,7.053014752283494060e-01,7.053013307101856366e-01,7.053011866981954148e-01,7.053010431906087119e-01,7.053009001856640481e-01,7.053007576816006097e-01,7.053006156766689072e-01,7.053004741691284440e-01,7.053003331572355039e-01,7.053001926392582499e-01,7.053000526134722836e-01,7.052999130781547610e-01,7.052997740315908315e-01,7.052996354720728611e-01,7.052994973978977677e-01,7.052993598073690196e-01,7.052992226987947477e-01,7.052990860704924092e-01,7.052989499207789059e-01,7.052988142479839073e-01,7.052986790504395254e-01,7.052985443264845333e-01,7.052984100744598139e-01,7.052982762927189064e-01,7.052981429796173485e-01,7.052980101335147856e-01,7.052978777527785237e-01,7.052977458357818641e-01,7.052976143809029930e-01,7.052974833865263138e-01,7.052973528510406709e-01,7.052972227728437904e-01,7.052970931503328433e-01,7.052969639819186565e-01,7.052968352660087259e-01,7.052967070010234263e-01,7.052965791853852418e-01,7.052964518175230957e-01,7.052963248958703524e-01,7.052961984188683697e-01,7.052960723849593938e-01,7.052959467925945525e-01,7.052958216402316349e-01,7.052956969263289855e-01,7.052955726493554955e-01,7.052954488077813888e-01,7.052953254000847716e-01,7.052952024247467477e-01,7.052950798802571919e-01,7.052949577651076440e-01,7.052948360777978598e-01,7.052947148168298153e-01,7.052945939807124809e-01,7.052944735679603783e-01,7.052943535770924699e-01,7.052942340066332694e-01,7.052941148551129524e-01,7.052939961210634712e-01,7.052938778030277689e-01,7.052937598995477897e-01,7.052936424091758028e-01,7.052935253304649654e-01,7.052934086619758736e-01,7.052932924022738970e-01,7.052931765499289574e-01,7.052930611035141961e-01,7.052929460616115254e-01,7.052928314228064099e-01,7.052927171856868682e-01,7.052926033488480240e-01,7.052924899108916623e-01,7.052923768704181251e-01,7.052922642260390784e-01,7.052921519763700742e-01,7.052920401200283296e-01,7.052919286556379452e-01,7.052918175818280178e-01,7.052917068972323067e-01,7.052915966004884574e-01,7.052914866902386670e-01,7.052913771651345698e-01,7.052912680238240251e-01,7.052911592649654393e-01,7.052910508872221040e-01,7.052909428892587540e-01,7.052908352697467853e-01,7.052907280273621460e-01,7.052906211607861131e-01,7.052905146687034055e-01,7.052904085498030717e-01,7.052903028027790455e-01,7.052901974263299234e-01,7.052900924191608523e-01,7.052899877799773121e-01,7.052898835074929984e-01,7.052897796004233832e-01,7.052896760574910440e-01,7.052895728774221107e-01,7.052894700589451560e-01,7.052893676007959689e-01,7.052892655017135581e-01,7.052891637604375985e-01,7.052890623757204214e-01,7.052889613463148022e-01,7.052888606709731834e-01,7.052887603484607748e-01,7.052886603775384566e-01,7.052885607569795434e-01,7.052884614855555734e-01,7.052883625620458563e-01,7.052882639852317004e-01,7.052881657538992988e-01,7.052880678668406178e-01,7.052879703228488451e-01,7.052878731207278262e-01,7.052877762592761890e-01,7.052876797373036633e-01,7.052875835536197568e-01,7.052874877070429704e-01,7.052873921963953574e-01,7.052872970204950853e-01,7.052872021781767531e-01,7.052871076682685203e-01,7.052870134896056520e-01,7.052869196410329611e-01,7.052868261213938172e-01,7.052867329295351428e-01,7.052866400643110767e-01,7.052865475245769789e-01,7.052864553091939825e-01,7.052863634170275509e-01,7.052862718469448122e-01,7.052861805978212217e-01,7.052860896685274605e-01,7.052859990579504190e-01,7.052859087649689940e-01,7.052858187884750718e-01,7.052857291273599838e-01,7.052856397805173927e-01,7.052855507468491769e-01,7.052854620252608786e-01,7.052853736146544872e-01,7.052852855139464250e-01,7.052851977220485624e-01,7.052851102378815407e-01,7.052850230603670001e-01,7.052849361884306889e-01,7.052848496210065710e-01,7.052847633570237251e-01,7.052846773954228876e-01,7.052845917351422411e-01,7.052845063751300714e-01,7.052844213143346641e-01,7.052843365517064145e-01,7.052842520862032671e-01,7.052841679167857203e-01,7.052840840424121627e-01,7.052840004620527514e-01,7.052839171746797531e-01,7.052838341792659893e-01,7.052837514747885006e-01,7.052836690602297676e-01,7.052835869345730480e-01,7.052835050968069286e-01,7.052834235459244372e-01,7.052833422809198227e-01,7.052832613007924412e-01,7.052831806045467555e-01,7.052831001911862296e-01,7.052830200597228760e-01,7.052829402091644884e-01,7.052828606385329602e-01,7.052827813468460771e-01,7.052827023331246226e-01,7.052826235963985946e-01,7.052825451356973252e-01,7.052824669500544763e-01,7.052823890385024885e-01,7.052823114000876803e-01,7.052822340338503748e-01,7.052821569388383338e-01,7.052820801141017615e-01,7.052820035586917502e-01,7.052819272716670529e-01,7.052818512520890870e-01,7.052817754990184929e-01,7.052817000115253476e-01,7.052816247886749546e-01,7.052815498295451624e-01,7.052814751332115994e-01,7.052814006987486728e-01,7.052813265252442232e-01,7.052812526117845371e-01,7.052811789574572332e-01,7.052811055613524838e-01,7.052810324225710081e-01,7.052809595402088627e-01,7.052808869133683212e-01,7.052808145411532115e-01,7.052807424226719135e-01,7.052806705570373591e-01,7.052805989433649225e-01,7.052805275807667584e-01,7.052804564683693433e-01,7.052803856052922704e-01,7.052803149906651248e-01,7.052802446236142719e-01,7.052801745032791780e-01,7.052801046287879849e-01,7.052800349992851547e-01,7.052799656139064899e-01,7.052798964718026697e-01,7.052798275721212651e-01,7.052797589140082923e-01,7.052796904966232017e-01,7.052796223191176717e-01,7.052795543806557044e-01,7.052794866804001916e-01,7.052794192175114718e-01,7.052793519911622067e-01,7.052792850005239478e-01,7.052792182447698011e-01,7.052791517230742047e-01,7.052790854346239202e-01,7.052790193785963835e-01,7.052789535541793553e-01,7.052788879605632610e-01,7.052788225969349734e-01,7.052787574624923561e-01,7.052786925564320519e-01,7.052786278779521467e-01,7.052785634262584979e-01,7.052784992005558529e-01,7.052784352000500689e-01,7.052783714239518886e-01,7.052783078714792708e-01,7.052782445418479540e-01,7.052781814342726774e-01,7.052781185479789494e-01,7.052780558821921675e-01,7.052779934361401715e-01,7.052779312090496910e-01,7.052778692001555605e-01,7.052778074086953897e-01,7.052777458339065664e-01,7.052776844750250351e-01,7.052776233313003962e-01,7.052775624019765877e-01,7.052775016863004343e-01,7.052774411835279755e-01,7.052773808929106991e-01,7.052773208137048666e-01,7.052772609451697372e-01,7.052772012865680118e-01,7.052771418371654999e-01,7.052770825962275669e-01,7.052770235630217988e-01,7.052769647368232198e-01,7.052769061169051890e-01,7.052768477025450622e-01,7.052767894930264125e-01,7.052767314876271509e-01,7.052766736856297403e-01,7.052766160863283007e-01,7.052765586890067384e-01,7.052765014929606169e-01,7.052764444974850555e-01,7.052763877018720651e-01,7.052763311054289774e-01,7.052762747074513561e-01,7.052762185072489753e-01,7.052761625041271687e-01,7.052761066973914916e-01,7.052760510863582688e-01,7.052759956703414934e-01,7.052759404486553807e-01,7.052758854206192529e-01,7.052758305855582055e-01,7.052757759427916717e-01,7.052757214916489659e-01,7.052756672314549613e-01,7.052756131615434132e-01,
        7.052755592812491869e-01,7.052755055899032621e-01,7.052754520868440569e-01,7.052753987714188710e-01,7.052753456429604606e-01,7.052752927008179018e-01,7.052752399443407150e-01,7.052751873728729803e-01,7.052751349857729890e-01,7.052750827823853763e-01,7.052750307620763159e-01,7.052749789241991030e-01,7.052749272681132497e-01,7.052748757931828205e-01,7.052748244987758763e-01,7.052747733842563704e-01,7.052747224489972488e-01,7.052746716923656845e-01,7.052746211137409516e-01,7.052745707124925545e-01,7.052745204880045415e-01,7.052744704396531894e-01,7.052744205668248778e-01,7.052743708689050983e-01,7.052743213452780102e-01,7.052742719953318806e-01,7.052742228184603057e-01,7.052741738140591021e-01,7.052741249815180913e-01,7.052740763202394181e-01,7.052740278296204535e-01,7.052739795090612329e-01,7.052739313579696745e-01,7.052738833757518089e-01,7.052738355618130006e-01,7.052737879155661638e-01,7.052737404364216589e-01,7.052736931237921780e-01,7.052736459770988509e-01,7.052735989957551466e-01,7.052735521791858586e-01,7.052735055268137820e-01,7.052734590380589363e-01,7.052734127123507779e-01,7.052733665491166537e-01,7.052733205477864642e-01,7.052732747077923303e-01,7.052732290285755878e-01,7.052731835095648050e-01,7.052731381501992081e-01,7.052730929499255730e-01,7.052730479081753545e-01,7.052730030244045434e-01,7.052729582980520329e-01,7.052729137285668193e-01,7.052728693154018957e-01,7.052728250580085900e-01,7.052727809558388961e-01,7.052727370083542446e-01,7.052726932150046313e-01,7.052726495752501545e-01,7.052726060885599058e-01,7.052725627543947606e-01,7.052725195722151508e-01,7.052724765414907226e-01,7.052724336616918999e-01,7.052723909322931028e-01,7.052723483527564285e-01,7.052723059225661784e-01,7.052722636411981050e-01,7.052722215081268509e-01,7.052721795228377166e-01,7.052721376848042345e-01,7.052720959935165901e-01,7.052720544484595289e-01,7.052720130491167971e-01,7.052719717949836875e-01,7.052719306855487202e-01,7.052718897202987502e-01,7.052718488987395062e-01,7.052718082203562888e-01,7.052717676846553818e-01,7.052717272911340762e-01,7.052716870392927717e-01,7.052716469286358647e-01,7.052716069586663084e-01,7.052715671288923849e-01,7.052715274388228206e-01,7.052714878879702276e-01,7.052714484758425550e-01,7.052714092019511938e-01,7.052713700658216345e-01,7.052713310669619373e-01,7.052712922048939292e-01,7.052712534791386600e-01,7.052712148892150701e-01,7.052711764346522028e-01,7.052711381149704417e-01,7.052710999297010508e-01,7.052710618783687435e-01,7.052710239605101128e-01,7.052709861756472076e-01,7.052709485233233933e-01,7.052709110030669359e-01,7.052708736144212009e-01,7.052708363569221151e-01,7.052707992301090467e-01,7.052707622335198101e-01,7.052707253667093168e-01,7.052706886292117172e-01,7.052706520205817009e-01,7.052706155403597466e-01,7.052705791881002106e-01,7.052705429633537859e-01,7.052705068656742737e-01,7.052704708946119228e-01,7.052704350497280839e-01,7.052703993305807773e-01,7.052703637367210288e-01,7.052703282677191821e-01,7.052702929231327023e-01,7.052702577025271591e-01,7.052702226054700096e-01,7.052701876315162766e-01,7.052701527802459625e-01,7.052701180512274126e-01,7.052700834440319699e-01,7.052700489582310883e-01,7.052700145934021059e-01,7.052699803491108144e-01,7.052699462249465423e-01,7.052699122204790783e-01,7.052698783352956413e-01,7.052698445689756790e-01,7.052698109211031907e-01,7.052697773912616208e-01,7.052697439790344136e-01,7.052697106840135621e-01,7.052696775057855083e-01,7.052696444439449097e-01,7.052696114980796516e-01,7.052695786677870560e-01,7.052695459526506783e-01,7.052695133522789428e-01,7.052694808662645087e-01,7.052694484942084729e-01,7.052694162357118213e-01,7.052693840903726530e-01,7.052693520577988373e-01,7.052693201375920262e-01,7.052692883293567583e-01,7.052692566327030121e-01,7.052692250472402113e-01,7.052691935725748928e-01,7.052691622083203660e-01,7.052691309540916054e-01,7.052690998095024755e-01,7.052690687741631770e-01,7.052690378476940136e-01,7.052690070297162883e-01,7.052689763198473072e-01,7.052689457177047094e-01,7.052689152229209002e-01,7.052688848351041928e-01,7.052688545538892129e-01,7.052688243789051459e-01,7.052687943097724066e-01,7.052687643461222899e-01,7.052687344875820941e-01,7.052687047337929949e-01,7.052686750843716323e-01,7.052686455389720610e-01,7.052686160972139184e-01,7.052685867587317192e-01,7.052685575231759652e-01,7.052685283901817259e-01,7.052684993593826279e-01,7.052684704304288399e-01,7.052684416029592063e-01,7.052684128766169014e-01,7.052683842510493184e-01,7.052683557259011859e-01,7.052683273008238940e-01,7.052682989754603948e-01,7.052682707494669634e-01,7.052682426224935464e-01,7.052682145941944203e-01,7.052681866642184216e-01,7.052681588322211592e-01,7.052681310978639040e-01,7.052681034608037081e-01,7.052680759206960692e-01,7.052680484772063663e-01,7.052680211299887647e-01,7.052679938787099756e-01,7.052679667230303817e-01,7.052679396626204689e-01,7.052679126971362900e-01,7.052678858262612094e-01,7.052678590496423983e-01,7.052678323669679950e-01,7.052678057778972720e-01,7.052677792821051561e-01,7.052677528792627992e-01,7.052677265690540098e-01,7.052677003511417242e-01,7.052676742252055320e-01,7.052676481909290196e-01,7.052676222479803414e-01,7.052675963960500782e-01,7.052675706348057183e-01,7.052675449639437266e-01,7.052675193831406952e-01,7.052674938920775460e-01,7.052674684904487457e-01,7.052674431779271114e-01,7.052674179542132160e-01,7.052673928189898689e-01,7.052673677719488721e-01,7.052673428127722577e-01,7.052673179411617088e-01,7.052672931568088055e-01,7.052672684594061270e-01,7.052672438486395912e-01,7.052672193242284226e-01,7.052671948858466600e-01,7.052671705331978735e-01,7.052671462659945156e-01,7.052671220839225041e-01,7.052670979866894063e-01,7.052670739740017902e-01,7.052670500455467950e-01,7.052670262010436453e-01,7.052670024401977988e-01,7.052669787627086073e-01,7.052669551682888560e-01,7.052669316566444468e-01,7.052669082274881651e-01,7.052668848805290214e-01,7.052668616154811332e-01,7.052668384320505135e-01,7.052668153299554987e-01,7.052667923089148694e-01,7.052667693686339723e-01,7.052667465088390264e-01,7.052667237292428171e-01,7.052667010295663452e-01,7.052666784095281693e-01,7.052666558688552856e-01,7.052666334072577037e-01,7.052666110244600883e-01,7.052665887201980954e-01,7.052665664941814017e-01,7.052665443461425543e-01,7.052665222758078833e-01,7.052665002829018315e-01,7.052664783671646065e-01,7.052664565283081055e-01,7.052664347660816402e-01,7.052664130801902242e-01,7.052663914703960479e-01,7.052663699364080108e-01,7.052663484779755354e-01,7.052663270948272833e-01,7.052663057866981333e-01,7.052662845533336222e-01,7.052662633944576376e-01,7.052662423098159383e-01,7.052662212991508417e-01,7.052662003621991138e-01,7.052661794987057364e-01,7.052661587084121386e-01,7.052661379910604156e-01,7.052661173463810051e-01,7.052660967741536391e-01,7.052660762740883271e-01,7.052660558459546980e-01,7.052660354894865202e-01,7.052660152044409880e-01,7.052659949905654146e-01,7.052659748476083346e-01,7.052659547753310498e-01,7.052659347734657747e-01,7.052659148417801394e-01,7.052658949800295618e-01,7.052658751879583576e-01,7.052658554653276068e-01,7.052658358118917281e-01,7.052658162274115794e-01,7.052657967116457982e-01,7.052657772643555756e-01,7.052657578852946640e-01,7.052657385742188145e-01,7.052657193309045391e-01,7.052657001551039251e-01,7.052656810465757209e-01,7.052656620051014347e-01,7.052656430304250490e-01,7.052656241223284050e-01,7.052656052805624798e-01,7.052655865049077821e-01,7.052655677951393809e-01,7.052655491510028130e-01,7.052655305722803636e-01,7.052655120587455473e-01,7.052654936101712124e-01,7.052654752263208815e-01,7.052654569069698454e-01,7.052654386518880658e-01,7.052654204608682642e-01,7.052654023336684119e-01,7.052653842700693509e-01,7.052653662698371573e-01,7.052653483327783190e-01,7.052653304586405936e-01,7.052653126472244738e-01,7.052652948982917058e-01,7.052652772116411173e-01,7.052652595870407826e-01,7.052652420242778719e-01,7.052652245231425532e-01,7.052652070834075637e-01,7.052651897048562990e-01,7.052651723872872536e-01,7.052651551304802702e-01,7.052651379342169680e-01,7.052651207982940651e-01,7.052651037224844099e-01,7.052650867065968221e-01,7.052650697504060373e-01,7.052650528537087737e-01,7.052650360162957544e-01,7.052650192379584793e-01,7.052650025184971083e-01,7.052649858576933717e-01,7.052649692553468741e-01,7.052649527112555550e-01,7.052649362252075838e-01,7.052649197970011219e-01,7.052649034264475425e-01,7.052648871133151420e-01,7.052648708574353886e-01,7.052648546585846834e-01,7.052648385165808387e-01,7.052648224312022540e-01,7.052648064022593033e-01,7.052647904295676895e-01,7.052647745129033696e-01,7.052647586521018086e-01,7.052647428469441815e-01,7.052647270972413063e-01,7.052647114028058883e-01,7.052646957634290947e-01,7.052646801789334008e-01,7.052646646491121940e-01,7.052646491737893930e-01,7.052646337527593845e-01,7.052646183858372053e-01,7.052646030728315640e-01,7.052645878135591628e-01,7.052645726078233812e-01,7.052645574554358143e-01,7.052645423562253768e-01,7.052645273099801271e-01,7.052645123165433017e-01,7.052644973757035141e-01,7.052644824872972285e-01,7.052644676511257149e-01,7.052644528670011237e-01,7.052644381347549229e-01,7.052644234542088109e-01,7.052644088251606158e-01,7.052643942474556837e-01,7.052643797208941745e-01,7.052643652452945666e-01,7.052643508205077572e-01,7.052643364463153652e-01,7.052643221225542991e-01,7.052643078490619111e-01,7.052642936256592332e-01,7.052642794521565284e-01,7.052642653283841545e-01,7.052642512541544839e-01,7.052642372293282946e-01,7.052642232537128519e-01,7.052642093271209722e-01,7.052641954494056620e-01,7.052641816203759628e-01,7.052641678398707814e-01,7.052641541077290244e-01,7.052641404237565137e-01,7.052641267878052567e-01,7.052641131996991719e-01,7.052640996592729472e-01,7.052640861663620475e-01,7.052640727207800664e-01,7.052640593223814536e-01,7.052640459709997867e-01,7.052640326664612047e-01,
        7.052640194086038372e-01,7.052640061972764718e-01,7.052639930322947004e-01,7.052639799135117515e-01,7.052639668407614248e-01,7.052639538138767428e-01,7.052639408327090464e-01,7.052639278970812553e-01,7.052639150068445995e-01,7.052639021618385406e-01,7.052638893618983218e-01,7.052638766068802800e-01,7.052638638966134410e-01,7.052638512309428176e-01,7.052638386097099810e-01,7.052638260327733777e-01,7.052638134999662523e-01,7.052638010111276223e-01,7.052637885661222628e-01,7.052637761647732040e-01,7.052637638069396697e-01,7.052637514924733342e-01,7.052637392212145473e-01,7.052637269930018826e-01,7.052637148077188778e-01,7.052637026651746854e-01,7.052636905652464039e-01,7.052636785077709414e-01,7.052636664926070775e-01,7.052636545196033779e-01,7.052636425886128491e-01,7.052636306994748416e-01,7.052636188520623461e-01,7.052636070462283691e-01,7.052635952818094855e-01,7.052635835586692492e-01,7.052635718766646633e-01,7.052635602356563949e-01,7.052635486354881245e-01,7.052635370760452771e-01,7.052635255571388928e-01,7.052635140786549517e-01,7.052635026404592278e-01,7.052634912423849656e-01,7.052634798843179231e-01,7.052634685660912339e-01,7.052634572875946528e-01,7.052634460486731927e-01,7.052634348491763072e-01,7.052634236889899766e-01,7.052634125679468902e-01,7.052634014859374689e-01,7.052633904428121658e-01,7.052633794384332022e-01,7.052633684726693497e-01,7.052633575453839399e-01,7.052633466564429687e-01,7.052633358057175395e-01,7.052633249930557735e-01,7.052633142183397652e-01,7.052633034814377311e-01,7.052632927821882447e-01,7.052632821205111480e-01,7.052632714962249194e-01,7.052632609092263083e-01,7.052632503593742053e-01,7.052632398465386032e-01,7.052632293706003752e-01,7.052632189314156363e-01,7.052632085288678132e-01,7.052631981628124658e-01,7.052631878331425686e-01,7.052631775397023572e-01,7.052631672823928000e-01,7.052631570610768952e-01,7.052631468756268562e-01,7.052631367259043493e-01,7.052631266118121189e-01,7.052631165331987306e-01,7.052631064899496094e-01,7.052630964819537329e-01,7.052630865090747658e-01,7.052630765711851435e-01,7.052630666681696248e-01,7.052630567999115252e-01,7.052630469662768409e-01,7.052630371671353426e-01,7.052630274023897750e-01,7.052630176719045796e-01,7.052630079755636272e-01,7.052629983132595592e-01,7.052629886848410523e-01,7.052629790902096296e-01,7.052629695292577106e-01,7.052629600018369693e-01,7.052629505078670258e-01,7.052629410472176508e-01,7.052629316197495113e-01,7.052629222253616881e-01,7.052629128639531508e-01,7.052629035353739084e-01,7.052628942395313683e-01,7.052628849763215024e-01,7.052628757456078645e-01,7.052628665473027469e-01,7.052628573812373958e-01,7.052628482473640714e-01,7.052628391455375567e-01,7.052628300756327295e-01,7.052628210375790907e-01,7.052628120312336435e-01,7.052628030564912498e-01,7.052627941132312284e-01,7.052627852013508836e-01,7.052627763207542921e-01,7.052627674713226602e-01,7.052627586529414128e-01,7.052627498655040794e-01,7.052627411088915332e-01,7.052627323830144013e-01,7.052627236877671013e-01,7.052627150230058595e-01,7.052627063886734993e-01,7.052626977846055967e-01,7.052626892107698442e-01,7.052626806670092563e-01,7.052626721532136989e-01,7.052626636693111184e-01,7.052626552151852746e-01,7.052626467906898400e-01,7.052626383957868450e-01,7.052626300303299622e-01,7.052626216942307069e-01,7.052626133873783898e-01,7.052626051096838600e-01,7.052625968610404250e-01,7.052625886413410594e-01,7.052625804504782936e-01,7.052625722883416604e-01,7.052625641548644353e-01,7.052625560499373725e-01,7.052625479734268010e-01,7.052625399252777649e-01,7.052625319053636987e-01,7.052625239135918989e-01,7.052625159498631113e-01,7.052625080140767500e-01,7.052625001061508803e-01,7.052624922259627116e-01,7.052624843734266458e-01,7.052624765484385438e-01,7.052624687509110313e-01,7.052624609807541800e-01,7.052624532378737321e-01,7.052624455221488953e-01,7.052624378334962918e-01,7.052624301718437572e-01,7.052624225370627276e-01,7.052624149290588340e-01,7.052624073477734568e-01,7.052623997930878019e-01,7.052623922648894039e-01,7.052623847631417364e-01,7.052623772876918107e-01,7.052623698384691275e-01,7.052623624154130688e-01,7.052623550184005108e-01,7.052623476473341979e-01,7.052623403021390791e-01,7.052623329827167886e-01,7.052623256889879455e-01,7.052623184208514084e-01,7.052623111782042598e-01,7.052623039609923206e-01,7.052622967690946876e-01,7.052622896024239862e-01,7.052622824609229291e-01,7.052622753444723891e-01,7.052622682529913201e-01,7.052622611863874624e-01,7.052622541445972004e-01,7.052622471275150629e-01,7.052622401350563397e-01,7.052622331671274392e-01,7.052622262236594164e-01,7.052622193045521293e-01,7.052622124097392975e-01,7.052622055391029043e-01,7.052621986925740050e-01,7.052621918700580084e-01,7.052621850715193874e-01,7.052621782968221398e-01,7.052621715459068685e-01,7.052621648186774284e-01,7.052621581150305685e-01,7.052621514349065590e-01,7.052621447782606579e-01,7.052621381449550864e-01,7.052621315349363318e-01,7.052621249480811594e-01,7.052621183843593711e-01,7.052621118436882552e-01,7.052621053259652273e-01,7.052620988310832617e-01,7.052620923590320334e-01,7.052620859096684347e-01,7.052620794829242978e-01,7.052620730787743097e-01,7.052620666970622620e-01,7.052620603377631747e-01,7.052620540007912275e-01,7.052620476860237408e-01,7.052620413934231891e-01,7.052620351229088591e-01,7.052620288743937094e-01,7.052620226478126808e-01,7.052620164430813965e-01,7.052620102601212526e-01,7.052620040988624162e-01,7.052619979592207322e-01,7.052619918411305866e-01,7.052619857445068252e-01,7.052619796692686238e-01,7.052619736153720176e-01,7.052619675826907741e-01,7.052619615711732681e-01,7.052619555807898566e-01,7.052619496114077569e-01,7.052619436629814498e-01,7.052619377354037988e-01,7.052619318286238448e-01,7.052619259426257114e-01,7.052619200772313190e-01,7.052619142324371149e-01,7.052619084081421796e-01,7.052619026043068784e-01,7.052618968208238526e-01,7.052618910576398115e-01,7.052618853146721545e-01,7.052618795918790262e-01,7.052618738891557326e-01,7.052618682064436539e-01,7.052618625436686273e-01,7.052618569007662597e-01,7.052618512776707149e-01,7.052618456742892894e-01,7.052618400905942275e-01,7.052618345265098121e-01,7.052618289819109210e-01,7.052618234567857858e-01,7.052618179510586893e-01,7.052618124646390374e-01,7.052618069974709858e-01,7.052618015494739323e-01,7.052617961206447683e-01,7.052617907108222894e-01,7.052617853200073839e-01,7.052617799480973559e-01,7.052617745950405803e-01,7.052617692607535682e-01,7.052617639452177789e-01,7.052617586483163059e-01,7.052617533699968577e-01,7.052617481102139152e-01,7.052617428688739976e-01,7.052617376459672238e-01,7.052617324413429367e-01,7.052617272549955851e-01,7.052617220868586667e-01,7.052617169368332606e-01,7.052617118049196998e-01,7.052617066909866450e-01,7.052617015949826929e-01,7.052616965168965191e-01,7.052616914566056661e-01,7.052616864140939246e-01,7.052616813892665926e-01,7.052616763820662715e-01,7.052616713924662051e-01,7.052616664203398278e-01,7.052616614656701532e-01,7.052616565283935657e-01,7.052616516084443399e-01,7.052616467057735150e-01,7.052616418202790616e-01,7.052616369519346673e-01,7.052616321006888178e-01,7.052616272664606889e-01,7.052616224491952135e-01,7.052616176488329947e-01,7.052616128653279581e-01,7.052616080985865121e-01,7.052616033485826774e-01,7.052615986152561689e-01,7.052615938985227206e-01,7.052615891983674556e-01,7.052615845147054419e-01,7.052615798474702880e-01,7.052615751965859436e-01,7.052615705620782771e-01,7.052615659437724283e-01,7.052615613416867157e-01,7.052615567557707354e-01,7.052615521859545433e-01,7.052615476321502097e-01,7.052615430943650621e-01,7.052615385724727570e-01,7.052615340664469823e-01,7.052615295762613146e-01,7.052615251017994025e-01,7.052615206430555839e-01,7.052615161999665760e-01,7.052615117724434501e-01,7.052615073604880935e-01,7.052615029639728306e-01,7.052614985829230854e-01,7.052614942172306112e-01,7.052614898668935206e-01,7.052614855317902443e-01,7.052614812119182286e-01,7.052614769072066414e-01,7.052614726175715498e-01,7.052614683430237230e-01,7.052614640834280468e-01,7.052614598388056155e-01,7.052614556090911480e-01,7.052614513942155883e-01,7.052614471941166530e-01,7.052614430087751352e-01,7.052614388380986643e-01,7.052614346821038938e-01,7.052614305406545991e-01,7.052614264137514466e-01,7.052614223013288219e-01,7.052614182033012380e-01,7.052614141196964503e-01,7.052614100504246419e-01,7.052614059954220860e-01,7.052614019546311619e-01,7.052613979280537571e-01,7.052613939156288092e-01,7.052613899171955580e-01,7.052613859328574764e-01,7.052613819625369596e-01,7.052613780060906779e-01,7.052613740635315098e-01,7.052613701348153796e-01,7.052613662198961020e-01,7.052613623186857472e-01,7.052613584311805406e-01,7.052613545573244158e-01,7.052613506970538682e-01,7.052613468503511340e-01,7.052613430171310593e-01,7.052613391973923118e-01,7.052613353910102134e-01,7.052613315979956443e-01,7.052613278183225143e-01,7.052613240518874616e-01,7.052613202986786067e-01,7.052613165586268940e-01,7.052613128317357649e-01,7.052613091178787652e-01,7.052613054170998597e-01,7.052613017292859166e-01,7.052612980544058496e-01,7.052612943924363442e-01,7.052612907432944667e-01,7.052612871069633416e-01,7.052612834834032229e-01,7.052612798725501619e-01,7.052612762743936115e-01,7.052612726888601857e-01,7.052612691159098057e-01,7.052612655554698629e-01,7.052612620075312533e-01,7.052612584720825417e-01,7.052612549489731819e-01,7.052612514382894382e-01,7.052612479398782108e-01,7.052612444537744718e-01,7.052612409798717508e-01,7.052612375182122362e-01,7.052612340686345016e-01,7.052612306312473489e-01,7.052612272058500498e-01,7.052612237925022232e-01,7.052612203911254873e-01,7.052612170017366067e-01,7.052612136241641627e-01,7.052612102584785436e-01,7.052612069046012566e-01,7.052612035624914455e-01,7.052612002321207996e-01,7.052611969133921743e-01,7.052611936063627462e-01,7.052611903109675673e-01,7.052611870270700800e-01,7.052611837546921558e-01,
        7.052611804938289097e-01,7.052611772444310478e-01,7.052611740064010926e-01,7.052611707797258322e-01,7.052611675644130385e-01,7.052611643603541314e-01,7.052611611675344561e-01,7.052611579859724422e-01,7.052611548155253152e-01,7.052611516561482219e-01,7.052611485079992582e-01,7.052611453707667843e-01,7.052611422446415368e-01,7.052611391294418830e-01,7.052611360251417327e-01,7.052611329317841626e-01,7.052611298492590386e-01,7.052611267775886761e-01,7.052611237166711566e-01,7.052611206664549659e-01,7.052611176269968363e-01,7.052611145982329299e-01,7.052611115800486719e-01,7.052611085724976858e-01,7.052611055754214320e-01,7.052611025888764207e-01,7.052610996128532150e-01,7.052610966472649956e-01,7.052610936920958862e-01,7.052610907472893764e-01,7.052610878127921756e-01,7.052610848886644579e-01,7.052610819746945037e-01,7.052610790710226452e-01,7.052610761775175430e-01,7.052610732941677618e-01,7.052610704209373305e-01,7.052610675578212529e-01,7.052610647047009573e-01,7.052610618616068638e-01,7.052610590285316450e-01,7.052610562053519550e-01,7.052610533920760094e-01,7.052610505887033643e-01,7.052610477951697376e-01,7.052610450114303875e-01,7.052610422374447907e-01,7.052610394732118371e-01,7.052610367186512574e-01,7.052610339737848122e-01,7.052610312385254598e-01,7.052610285128962930e-01,7.052610257968238150e-01,7.052610230902669475e-01,7.052610203932433430e-01,7.052610177056605201e-01,7.052610150275334666e-01,7.052610123587155222e-01,7.052610096994009758e-01,7.052610070493293692e-01,7.052610044085572127e-01,7.052610017770805095e-01,7.052609991548364210e-01,7.052609965417725446e-01,7.052609939378962078e-01,7.052609913431584499e-01,7.052609887574832204e-01,7.052609861809554515e-01,7.052609836133946208e-01,7.052609810549069769e-01,7.052609785054363423e-01,7.052609759647683330e-01,7.052609734331310998e-01,7.052609709103679902e-01,7.052609683964057297e-01,7.052609658912937229e-01,7.052609633950090995e-01,7.052609609074115271e-01,7.052609584286351208e-01,7.052609559585125698e-01,7.052609534970367688e-01,7.052609510442122698e-01,7.052609486000250838e-01,7.052609461643568611e-01,7.052609437372469037e-01,7.052609413187115317e-01,7.052609389085986447e-01,7.052609365069850700e-01,7.052609341137626719e-01,7.052609317290042812e-01,7.052609293525530232e-01,7.052609269844930528e-01,7.052609246246891450e-01,7.052609222731623939e-01,7.052609199299385567e-01,7.052609175949483555e-01,7.052609152680777704e-01,7.052609129494333828e-01,7.052609106389406968e-01,7.052609083365259934e-01,7.052609060421144438e-01,7.052609037558893457e-01,7.052609014776131113e-01,7.052608992073489125e-01,7.052608969450409049e-01,7.052608946906803178e-01,7.052608924442362870e-01,7.052608902056539675e-01,7.052608879749452386e-01,7.052608857520673569e-01,7.052608835370013374e-01,7.052608813296536994e-01,7.052608791301702151e-01,7.052608769382686660e-01,7.052608747540893841e-01,7.052608725776192689e-01,7.052608704088217939e-01,7.052608682475740576e-01,7.052608660939125862e-01,7.052608639478140651e-01,7.052608618092470749e-01,7.052608596782444783e-01,7.052608575545825653e-01,7.052608554384767192e-01,7.052608533297817228e-01,7.052608512284664899e-01,7.052608491345515596e-01,7.052608470480017377e-01,7.052608449687546299e-01,7.052608428967983567e-01,7.052608408320865108e-01,7.052608387746746033e-01,7.052608367244415088e-01,7.052608346814168705e-01,7.052608326455525045e-01,7.052608306168793861e-01,7.052608285952521872e-01,7.052608265807625010e-01,7.052608245733141823e-01,7.052608225729538605e-01,7.052608205795655172e-01,7.052608185932395246e-01,7.052608166138737422e-01,7.052608146413239520e-01,7.052608126758531659e-01,7.052608107171415286e-01,7.052608087654064217e-01,7.052608068205142855e-01,7.052608048824788867e-01,7.052608029511587828e-01,7.052608010266353533e-01,7.052607991089002715e-01,7.052607971978682722e-01,7.052607952935233682e-01,7.052607933959592623e-01,7.052607915049833309e-01,7.052607896206768423e-01,7.052607877429030170e-01,7.052607858717353517e-01,7.052607840072039336e-01,7.052607821491607698e-01,7.052607802976623708e-01,7.052607784526560009e-01,7.052607766141794077e-01,7.052607747820802686e-01,7.052607729564840389e-01,7.052607711372173016e-01,7.052607693243291287e-01,7.052607675178413915e-01,7.052607657177688560e-01,7.052607639238765991e-01,7.052607621363037316e-01,7.052607603550915538e-01,7.052607585800252377e-01,7.052607568111914915e-01,7.052607550486442722e-01,7.052607532922990918e-01,7.052607515420151740e-01,7.052607497979427320e-01,7.052607480600211476e-01,7.052607463280935463e-01,7.052607446022935989e-01,7.052607428825969915e-01,7.052607411688358585e-01,7.052607394611939418e-01,7.052607377594506399e-01,7.052607360637698219e-01,7.052607343739399903e-01,7.052607326901401130e-01,7.052607310122516182e-01,7.052607293402072264e-01,7.052607276739427666e-01,7.052607260136584122e-01,7.052607243591613173e-01,7.052607227104736864e-01,7.052607210675343463e-01,7.052607194304293392e-01,7.052607177990765086e-01,7.052607161733477348e-01,7.052607145533472677e-01,7.052607129390415786e-01,7.052607113304815156e-01,7.052607097275130910e-01,7.052607081301586200e-01,7.052607065384188800e-01,7.052607049522751081e-01,7.052607033716122853e-01,7.052607017966241454e-01,7.052607002271183978e-01,7.052606986630896024e-01,7.052606971046098128e-01,7.052606955516188547e-01,7.052606940039259920e-01,7.052606924617781381e-01,7.052606909249924394e-01,7.052606893936335108e-01,7.052606878676932478e-01,7.052606863470893828e-01,7.052606848317144461e-01,7.052606833217663906e-01,7.052606818170918945e-01,7.052606803177010608e-01,7.052606788235545876e-01,7.052606773346362656e-01,7.052606758509999407e-01,7.052606743725338134e-01,7.052606728992875107e-01,7.052606714312111835e-01,7.052606699681922553e-01,7.052606685103717243e-01,7.052606670576747616e-01,7.052606656101021443e-01,7.052606641675472909e-01,7.052606627301163389e-01,7.052606612977209144e-01,7.052606598703298202e-01,7.052606584479425011e-01,7.052606570304500444e-01,7.052606556181527653e-01,7.052606542107077159e-01,7.052606528081496462e-01,7.052606514105475011e-01,7.052606500178479898e-01,7.052606486300627697e-01,7.052606472471266708e-01,7.052606458690096058e-01,7.052606444957933984e-01,7.052606431273045207e-01,7.052606417637323766e-01,7.052606404049504008e-01,7.052606390508443512e-01,7.052606377015647743e-01,7.052606363569176029e-01,7.052606350170572691e-01,7.052606336818850741e-01,7.052606323514373221e-01,7.052606310256462896e-01,7.052606297045399542e-01,7.052606283879288007e-01,7.052606270761203611e-01,7.052606257687937807e-01,7.052606244660938328e-01,7.052606231678849591e-01,7.052606218742968336e-01,7.052606205853526600e-01,7.052606193008953417e-01,7.052606180208612630e-01,7.052606167452921682e-01,7.052606154742333544e-01,7.052606142077195717e-01,7.052606129455537554e-01,7.052606116879012177e-01,7.052606104346814675e-01,7.052606091856578052e-01,7.052606079411802842e-01,7.052606067010342983e-01,7.052606054653317580e-01,7.052606042337099534e-01,7.052606030066651543e-01,7.052606017837982355e-01,7.052606005653542232e-01,7.052605993510318028e-01,7.052605981410663416e-01,7.052605969353201720e-01,7.052605957338173859e-01,7.052605945364567308e-01,7.052605933434633600e-01,7.052605921544954359e-01,7.052605909697589048e-01,7.052605897891665032e-01,7.052605886127222279e-01,7.052605874404955788e-01,7.052605862722031160e-01,7.052605851080671062e-01,7.052605839481469463e-01,7.052605827922191484e-01,7.052605816402779393e-01,7.052605804924674260e-01,7.052605793486826924e-01,7.052605782088187114e-01,7.052605770730832058e-01,7.052605759413377307e-01,7.052605748133538022e-01,7.052605736895953825e-01,7.052605725696418082e-01,7.052605714535520320e-01,7.052605703414408511e-01,7.052605692332902798e-01,7.052605681289686457e-01,7.052605670285796435e-01,7.052605659320476672e-01,7.052605648393188709e-01,7.052605637504819613e-01,7.052605626654133708e-01,7.052605615840326081e-01,7.052605605066297745e-01,7.052605594329417471e-01,7.052605583631613717e-01,7.052605572968628778e-01,7.052605562345347634e-01,7.052605551759367764e-01,7.052605541208194495e-01,7.052605530696710590e-01,7.052605520219833446e-01,7.052605509780178750e-01,7.052605499379092091e-01,7.052605489013692441e-01,7.052605478683188212e-01,7.052605468391232035e-01,7.052605458133499594e-01,7.052605447913168346e-01,7.052605437726955362e-01,7.052605427577596231e-01,7.052605417463045923e-01,7.052605407383969460e-01,7.052605397342240900e-01,7.052605387335163512e-01,7.052605377362243244e-01,7.052605367424940042e-01,7.052605357520712603e-01,7.052605347653685408e-01,7.052605337819464193e-01,7.052605328021880338e-01,7.052605318257416656e-01,7.052605308526546102e-01,7.052605298831732261e-01,7.052605289168701885e-01,7.052605279541173111e-01,7.052605269945890765e-01,7.052605260385890196e-01,7.052605250858090535e-01,7.052605241365039745e-01,7.052605231904184313e-01,7.052605222477213998e-01,7.052605213083463775e-01,7.052605203721369342e-01,7.052605194395165089e-01,7.052605185098714813e-01,7.052605175835663376e-01,7.052605166605551146e-01,7.052605157406790504e-01,7.052605148241719579e-01,7.052605139108689691e-01,7.052605130006194267e-01,7.052605120935823146e-01,7.052605111898250234e-01,7.052605102893159117e-01,7.052605093918438151e-01,7.052605084974991057e-01,7.052605076064741851e-01,7.052605067184034571e-01,7.052605058334806554e-01,7.052605049517324254e-01,7.052605040729360564e-01,7.052605031973008254e-01,7.052605023247727756e-01,7.052605014554094165e-01,7.052605005886878331e-01,7.052604997254570129e-01,7.052604988651226536e-01,7.052604980077841201e-01,7.052604971534117695e-01,7.052604963021599227e-01,7.052604954538773674e-01,7.052604946083739224e-01,7.052604937659078255e-01,7.052604929264617573e-01,7.052604920899191443e-01,7.052604912564255368e-01,7.052604904256742913e-01,7.052604895979412980e-01,7.052604887731701577e-01,7.052604879513428848e-01,7.052604871322681879e-01,7.052604863159598336e-01,7.052604855026016750e-01,7.052604846920834669e-01,7.052604838847401636e-01,7.052604830797849500e-01,
        7.052604822778619775e-01,7.052604814789055210e-01,7.052604806825123474e-01,7.052604798889495763e-01,7.052604790983304506e-01,7.052604783102263131e-01,7.052604775250813640e-01,7.052604767422989696e-01,7.052604759627927322e-01,7.052604751857992627e-01,7.052604744116012236e-01,7.052604736400848173e-01,7.052604728713818272e-01,7.052602544268956608e-01
    ])

    TANGENTS = np.array([
        -7.528967034010277141e-04,-7.501394449603263244e-04,-7.447232702207928767e-04,-7.394125893993575005e-04,-7.342079598879891478e-04,-7.291054921347850382e-04,-7.241015175537546256e-04,-7.191925709992830207e-04,-7.143753750430414939e-04,-7.096468258264154727e-04,-7.050039802964924718e-04,-7.004440446594095704e-04,-6.959643639062873532e-04,-6.915624122915131622e-04,-6.872357846571253148e-04,-6.829821885065978648e-04,-6.787994367538630058e-04,-6.746854410730751539e-04,-6.706382057882764869e-04,-6.666558222504503917e-04,-6.627364636521138053e-04,-6.588783802400355327e-04,-6.550798948860014903e-04,-6.513393989835414288e-04,-6.476553486439717844e-04,-6.440262611598912557e-04,-6.404507117155899820e-04,-6.369273303253875085e-04,-6.334547989781391664e-04,-6.300318489700362790e-04,-6.266572584119334266e-04,-6.233298498992123626e-04,-6.200484883258639002e-04,-6.168120788382358555e-04,-6.136195649134590369e-04,-6.104699265527813168e-04,-6.073621785859240063e-04,-6.042953690728047889e-04,-6.012685777992965441e-04,-5.982809148588064119e-04,-5.953315193144570472e-04,-5.924195579368740638e-04,-5.895442240095860598e-04,-5.867047362003718902e-04,-5.839003374941142965e-04,-5.811302941819418422e-04,-5.783938949021072418e-04,-5.756904497320469716e-04,-5.730192893281804700e-04,-5.703797641047891887e-04,-5.677712434594139879e-04,-5.651931150334466025e-04,-5.626447840092474451e-04,-5.601256724426795230e-04,-5.576352186255073562e-04,-5.551728764805474725e-04,-5.527381149829091456e-04,-5.503304176097678635e-04,-5.479492818150077937e-04,-5.455942185260687083e-04,-5.432647516669941723e-04,-5.409604176993543234e-04,-5.386807651859282231e-04,-5.364253543707064864e-04,-5.341937567804322384e-04,-5.319855548413521262e-04,-5.298003415106222747e-04,-5.276377199284754127e-04,-5.254973030787146726e-04,-5.233787134673040242e-04,-5.212815828125050288e-04,-5.192055517490024030e-04,-5.171502695423546569e-04,-5.151153938148800293e-04,-5.131005902840879429e-04,-5.111055325104363334e-04,-5.091299016537487176e-04,-5.071733862409555371e-04,-5.052356819419401290e-04,-5.033164913528231921e-04,-5.014155237901274376e-04,-4.995324950897161997e-04,-4.976671274142807633e-04,-4.958191490695984527e-04,-4.939882943242324131e-04,-4.921743032382241978e-04,-4.903769214976705371e-04,-4.885959002539630447e-04,-4.868309959701333511e-04,-4.850819702716391291e-04,-4.833485898034783901e-04,-4.816306260900793390e-04,-4.799278554025177002e-04,-4.782400586291757349e-04,-4.765670211505090847e-04,-4.749085327181434835e-04,-4.732643873387454292e-04,-4.716343831605573911e-04,-4.700183223643739083e-04,-4.684160110591806259e-04,-4.668272591782374192e-04,-4.652518803819338800e-04,-4.636896919608668455e-04,-4.621405147438029104e-04,-4.606041730087495623e-04,-4.590804943952475625e-04,-4.575693098203270637e-04,-4.560704533966841723e-04,-4.545837623555204488e-04,-4.531090769681611619e-04,-4.516462404717813683e-04,-4.501950989989067509e-04,-4.487555015063593444e-04,-4.473272997079780211e-04,-4.459103480091153315e-04,-4.445045034424666142e-04,-4.431096256065636396e-04,-4.417255766052674559e-04,-4.403522209901478135e-04,-4.389894257034177016e-04,-4.376370600218670859e-04,-4.362949955057926488e-04,-4.349631059451519732e-04,-4.336412673093814618e-04,-4.323293576994347021e-04,-4.310272572980444750e-04,-4.297348483256469009e-04,-4.284520149941961620e-04,-4.271786434627555806e-04,-4.259146217957532343e-04,-4.246598399210155250e-04,-4.234141895903542618e-04,-4.221775643385994314e-04,-4.209498594458516152e-04,-4.197309719018438301e-04,-4.185208003679719013e-04,-4.173192451419893700e-04,-4.161262081245897804e-04,-4.149415927855448771e-04,-4.137653041307309820e-04,-4.125972486707096820e-04,-4.114373343897526070e-04,-4.102854707154213187e-04,-4.091415684895904903e-04,-4.080055399392490401e-04,-4.068772986486335341e-04,-4.057567595324718113e-04,-4.046438388094486527e-04,-4.035384539746722510e-04,-4.024405237763595267e-04,-4.013499681908561101e-04,-4.002667083969901896e-04,-3.991906667544231624e-04,-3.981217667803349514e-04,-3.970599331261093212e-04,-3.960050915561286189e-04,-3.949571689267905583e-04,-3.939160931651919384e-04,-3.928817932488115616e-04,-3.918541991859703089e-04,-3.908332419966242810e-04,-3.898188536923807845e-04,-3.888109672588457855e-04,-3.878095166384154524e-04,-3.868144367101811198e-04,-3.858256632748302550e-04,-3.848431330371049341e-04,-3.838667835890374747e-04,-3.828965533942962907e-04,-3.819323817718656144e-04,-3.809742088817236194e-04,-3.800219757077449856e-04,-3.790756240453774240e-04,-3.781350964859875319e-04,-3.772003364023168714e-04,-3.762712879362695162e-04,-3.753478959833689288e-04,-3.744301061812116416e-04,-3.735178648962556025e-04,-3.726111192102754543e-04,-3.717098169093713267e-04,-3.708139064707571819e-04,-3.699233370513255181e-04,-3.690380584765451388e-04,-3.681580212279156328e-04,-3.672831764330863891e-04,-3.664134758543102777e-04,-3.655488718774524415e-04,-3.646893175035526014e-04,-3.638347663355023798e-04,-3.629851725700516951e-04,-3.621404909884828882e-04,-3.613006769446203137e-04,-3.604656863580579795e-04,-3.596354757038344729e-04,-3.588100020025519754e-04,-3.579892228131598131e-04,-3.571730962225183603e-04,-3.563615808385156569e-04,-3.555546357814076686e-04,-3.547522206744924134e-04,-3.539542956367824900e-04,-3.531608212757886278e-04,-3.523717586794150591e-04,-3.515870694072997793e-04,-3.508067154857075209e-04,-3.500306593983149028e-04,-3.492588640788829579e-04,-3.484912929059280629e-04,-3.477279096955054882e-04,-3.469686786934378375e-04,-3.462135645688757535e-04,-3.454625324085247584e-04,-3.447155477105390275e-04,-3.439725763771939171e-04,-3.432335847090017822e-04,-3.424985393998269956e-04,-3.417674075302246095e-04,-3.410401565608900398e-04,-3.403167543276630624e-04,-3.395971690367538542e-04,-3.388813692584147219e-04,-3.381693239213889868e-04,-3.374610023079149812e-04,-3.367563740483969781e-04,-3.360554091167422541e-04,-3.353580778254761086e-04,-3.346643508198576811e-04,-3.339741990738831490e-04,-3.332875938852897235e-04,-3.326045068711147579e-04,-3.319249099626997435e-04,-3.312487754014714625e-04,-3.305760757348341627e-04,-3.299067838111735540e-04,-3.292408727767481835e-04,-3.285783160698052541e-04,-3.279190874173609771e-04,-3.272631608316478591e-04,-3.266105106056738094e-04,-3.259611113093363599e-04,-3.253149377854258617e-04,-3.246719651459617495e-04,-3.240321687683067609e-04,-3.233955242922803563e-04,-3.227620076159398721e-04,-3.221315948919167838e-04,-3.215042625240860374e-04,-3.208799871646794699e-04,-3.202587457102890056e-04,-3.196405152989800769e-04,-3.190252733061837986e-04,-3.184129973433647010e-04,-3.178036652532467699e-04,-3.171972551062607337e-04,-3.165937452000999741e-04,-3.159931140539473660e-04,-3.153953404070319877e-04,-3.148004032156315191e-04,-3.142082816497415720e-04,-3.136189550905221779e-04,-3.130324031276332519e-04,-3.124486055565700582e-04,-3.118675423756656073e-04,-3.112891937842032775e-04,-3.107135401784200113e-04,-3.101405621506181376e-04,-3.095702404859457246e-04,-3.090025561596210224e-04,-3.084374903349340613e-04,-3.078750243602490499e-04,-3.073151397680051744e-04,-3.067578182712749069e-04,-3.062030417612104927e-04,-3.056507923056006604e-04,-3.051010521473163095e-04,-3.045538036999806408e-04,-3.040090295475250670e-04,-3.034667124418577444e-04,-3.029268353006431269e-04,-3.023893812048594754e-04,-3.018543333969114784e-04,-3.013216752798530962e-04,-3.007913904133907579e-04,-3.002634625135502944e-04,-2.997378754501234255e-04,-2.992146132451134477e-04,-2.986936600705147882e-04,-2.981750002462035809e-04,-2.976586182394935776e-04,-2.971444986624716122e-04,-2.966326262699992000e-04,-2.961229859582692470e-04,-2.956155627630296934e-04,-2.951103418584732907e-04,-2.946073085555722670e-04,-2.941064482996358365e-04,-2.936077466687558868e-04,-2.931111893730298235e-04,-2.926167622532283019e-04,-2.921244512783527369e-04,-2.916342425443030351e-04,-2.911461222728783937e-04,-2.906600768097789000e-04,-2.901760926236063298e-04,-2.896941563046429025e-04,-2.892142545622977678e-04,-2.887363742249959842e-04,-2.882605022390682947e-04,-2.877866256655314814e-04,-2.873147316799773421e-04,-2.868448075725726909e-04,-2.863768407443956221e-04,-2.859108187069914209e-04,-2.854467290821505188e-04,-2.849845595994660030e-04,-2.845242980956674828e-04,-2.840659325131777990e-04,-2.836094508994468910e-04,-2.831548414048423723e-04,-2.827020922822054416e-04,-2.822511918860737268e-04,-2.818021286703498163e-04,-2.813548911886343262e-04,-2.809094680918944320e-04,-2.804658481280197790e-04,-2.800240201408232821e-04,-2.795839730689309022e-04,-2.791456959443383568e-04,-2.787091778914119189e-04,-2.782744081275545511e-04,-2.778413759593201249e-04,-2.774100707837456881e-04,-2.769804820866861306e-04,-2.765525994410378274e-04,-2.761264125077378395e-04,-2.757019110325442668e-04,-2.752790848468134044e-04,-2.748579238662784974e-04,-2.744384180889403169e-04,-2.740205575960663609e-04,-2.736043325498593859e-04,-2.731897331933463846e-04,-2.727767498498234744e-04,-2.723653729203023843e-04,-2.719555928849537452e-04,-2.715474003009976656e-04,-2.711407858019265760e-04,-2.707357400971721617e-04,-2.703322539711061623e-04,-2.699303182823742375e-04,-2.695299239622306331e-04,-2.691310620154263589e-04,-2.687337235184328321e-04,-2.683378996186647214e-04,-2.679435815340358573e-04,-2.675507605520710541e-04,-2.671594280295730428e-04,-2.667695753915122481e-04,-2.663811941305826991e-04,-2.659942758058697621e-04,-2.656088120434052513e-04,-2.652247945348351621e-04,-2.648422150360874028e-04,-2.644610653675938394e-04,-2.640813374137351843e-04,-2.637030231222858845e-04,-2.633261145023046979e-04,-2.629506036252449164e-04,-2.625764826237331206e-04,-2.622037436910140684e-04,-2.618323790801735385e-04,-2.614623811034721967e-04,-2.610937421323455965e-04,-2.607264545960719104e-04,-2.603605109819939756e-04,-2.599959038341870254e-04,-2.596326257534586901e-04,-2.592706693964608178e-04,-2.589100274751343633e-04,-2.585506927567093882e-04,-2.581926580629279044e-04,-2.578359162687116068e-04,-2.574804603028280070e-04,
        -2.571262831467802101e-04,-2.567733778342518036e-04,-2.564217374506627678e-04,-2.560713551335025429e-04,-2.557222240704426497e-04,-2.553743374990036230e-04,-2.550276887079983013e-04,-2.546822710345342244e-04,-2.543380778652348795e-04,-2.539951026349074326e-04,-2.536533388262096622e-04,-2.533127799699830263e-04,-2.529734196441424388e-04,-2.526352514728991139e-04,-2.522982691265385213e-04,-2.519624663220865202e-04,-2.516278368215330019e-04,-2.512943744311657568e-04,-2.509620730031247859e-04,-2.506309264330708331e-04,-2.503009286599633398e-04,-2.499720736672816912e-04,-2.496443554803606801e-04,-2.493177681678337976e-04,-2.489923058403009648e-04,-2.486679626497734219e-04,-2.483447327905619062e-04,-2.480226104972782508e-04,-2.477015900455015185e-04,-2.473816657515559569e-04,-2.470628319706236198e-04,-2.467450830980766341e-04,-2.464284135691441335e-04,-2.461128178570248792e-04,-2.457982904739974828e-04,-2.454848259704212055e-04,-2.451724189340698246e-04,-2.448610639910198117e-04,-2.445507558038739759e-04,-2.442414890725386201e-04,-2.439332585335574066e-04,-2.436260589586680680e-04,-2.433198851566897858e-04,-2.430147319719688781e-04,-2.427105942833795993e-04,-2.424074670051012959e-04,-2.421053450857302280e-04,-2.418042235089457037e-04,-2.415040972921778106e-04,-2.412049614859412827e-04,-2.409068111749457231e-04,-2.406096414769853808e-04,-2.403134475426060845e-04,-2.400182245548831972e-04,-2.397239677296436611e-04,-2.394306723142447524e-04,-2.391383335880181704e-04,-2.388469468619369707e-04,-2.385565074781714756e-04,-2.382670108098672301e-04,-2.379784522604788677e-04,-2.376908272646582887e-04,-2.374041312869223930e-04,-2.371183598213200128e-04,-2.368335083919870243e-04,-2.365495725524802140e-04,-2.362665478855552337e-04,-2.359844300030555786e-04,-2.357032145450244087e-04,-2.354228971802596604e-04,-2.351434736059809794e-04,-2.348649395472746093e-04,-2.345872907568713472e-04,-2.343105230149244989e-04,-2.340346321292319232e-04,-2.337596139343478541e-04,-2.334854642919159673e-04,-2.332121790901142688e-04,-2.329397542436550950e-04,-2.326681856930079562e-04,-2.323974694048436263e-04,-2.321276013723672094e-04,-2.318585776130976939e-04,-2.315903941703112423e-04,-2.313230471133742583e-04,-2.310565325350788513e-04,-2.307908465538632825e-04,-2.305259853127017422e-04,-2.302619449784382155e-04,-2.299987217426746611e-04,-2.297363118203277210e-04,-2.294747114505168994e-04,-2.292139168958984285e-04,-2.289539244422211794e-04,-2.286947303986597291e-04,-2.284363310975923156e-04,-2.281787228933795930e-04,-2.279219021639189435e-04,-2.276658653094232321e-04,-2.274106087520877395e-04,-2.271561289364232294e-04,-2.269024223284787922e-04,-2.266494854167300232e-04,-2.263973147107467554e-04,-2.261459067418591928e-04,-2.258952580623807549e-04,-2.256453652453860315e-04,-2.253962248858210060e-04,-2.251478335988377211e-04,-2.249001880196832559e-04,-2.246532848049209719e-04,-2.244071206308762001e-04,-2.241616921943023755e-04,-2.239169962117149026e-04,-2.236730294192801338e-04,-2.234297885732594580e-04,-2.231872704490101000e-04,-2.229454718414292103e-04,-2.227043895648428418e-04,-2.224640204522287945e-04,-2.222243613552166153e-04,-2.219854091455308875e-04,-2.217471607121046517e-04,-2.215096129628557620e-04,-2.212727628243538192e-04,-2.210366072408209703e-04,-2.208011431752421316e-04,-2.205663676075886315e-04,-2.203322775362615005e-04,-2.200988699778694269e-04,-2.198661419654523996e-04,-2.196340905497029539e-04,-2.194027127990771930e-04,-2.191720057987955883e-04,-2.189419666512870677e-04,-2.187125924758559492e-04,-2.184838804081268293e-04,-2.182558276009327614e-04,-2.180284312232050326e-04,-2.178016884606392978e-04,-2.175755965151404681e-04,-2.173501526042675991e-04,-2.171253539622330919e-04,-2.169011978386814476e-04,-2.166776814993554012e-04,-2.164548022256518323e-04,-2.162325573143997204e-04,-2.160109440778601453e-04,-2.157899598439483313e-04,-2.155696019553454690e-04,-2.153498677698317820e-04,-2.151307546610636834e-04,-2.149122600165753738e-04,-2.146943812390000872e-04,-2.144771157456260013e-04,-2.142604609686182826e-04,-2.140444143545749967e-04,-2.138289733637499523e-04,-2.136141354714959917e-04,-2.133998981669327222e-04,-2.131862589533906061e-04,-2.129732153482999379e-04,-2.127607648821916442e-04,-2.125489051001405727e-04,-2.123376335608773147e-04,-2.121269478359000260e-04,-2.119168455111397620e-04,-2.117073241856282095e-04,-2.114983814710535981e-04,-2.112900149929819449e-04,-2.110822223897468319e-04,-2.108750013130045176e-04,-2.106683494270678025e-04,-2.104622644087950079e-04,-2.102567439481450862e-04,-2.100517857477335326e-04,-2.098473875223882956e-04,-2.096435469999269330e-04,-2.094402619201574112e-04,-2.092375300351001499e-04,-2.090353491093210891e-04,-2.088337169190435105e-04,-2.086326312528141713e-04,-2.084320899112812597e-04,-2.082320907068613280e-04,-2.080326314632952034e-04,-2.078337100166471885e-04,-2.076353242143058608e-04,-2.074374719154281621e-04,-2.072401509900512195e-04,-2.070433593203135914e-04,-2.068470947994560660e-04,-2.066513553312665508e-04,-2.064561388317454060e-04,-2.062614432275511334e-04,-2.060672664562224199e-04,-2.058736064660671161e-04,-2.056804612163842805e-04,-2.054878286775752017e-04,-2.052957068305882871e-04,-2.051040936668080406e-04,-2.049129871879440401e-04,-2.047223854069191162e-04,-2.045322863468701513e-04,-2.043426880407039903e-04,-2.041535885322076638e-04,-2.039649858750491873e-04,-2.037768781333326729e-04,-2.035892633812652619e-04,-2.034021397026020139e-04,-2.032155051910899957e-04,-2.030293579510233926e-04,-2.028436960960222635e-04,-2.026585177494766299e-04,-2.024738210444354536e-04,-2.022896041233845921e-04,-2.021058651389129324e-04,-2.019226022527131903e-04,-2.017398136359149774e-04,-2.015574974687517340e-04,-2.013756519417819746e-04,-2.011942752538908863e-04,-2.010133656126233959e-04,-2.008329212361825711e-04,-2.006529403510981524e-04,-2.004734211923375753e-04,-2.002943620044161932e-04,-2.001157610407311438e-04,-1.999376165637833935e-04,-1.997599268439564923e-04,-1.995826901607378190e-04,-1.994059048030516479e-04,-1.992295690674827924e-04,-1.990536812597198946e-04,-1.988782396930011132e-04,-1.987032426901125248e-04,-1.985286885820558567e-04,-1.983545757074933746e-04,-1.981809024139691289e-04,-1.980076670571317976e-04,-1.978348680008457094e-04,-1.976625036165247096e-04,-1.974905722844644274e-04,-1.973190723927320533e-04,-1.971480023369442947e-04,-1.969773605209335088e-04,-1.968071453567477036e-04,-1.966373552636513367e-04,-1.964679886687914490e-04,-1.962990440075307319e-04,-1.961305197225593488e-04,-1.959624142637839128e-04,-1.957947260891046426e-04,-1.956274536645263851e-04,-1.954605954621602137e-04,-1.952941499622218302e-04,-1.951281156534756533e-04,-1.949624910301261949e-04,-1.947972745941495276e-04,-1.946324648557373749e-04,-1.944680603315207534e-04,-1.943040595454581521e-04,-1.941404610282804200e-04,-1.939772633182679229e-04,-1.938144649605844094e-04,-1.936520645073880331e-04,-1.934900605176093080e-04,-1.933284515568400863e-04,-1.931672361986658260e-04,-1.930064130221120777e-04,-1.928459806135318644e-04,-1.926859375662726137e-04,-1.925262824801210471e-04,-1.923670139613031793e-04,-1.922081306229284081e-04,-1.920496310847674692e-04,-1.918915139725863028e-04,-1.917337779192562763e-04,-1.915764215637549839e-04,-1.914194435517213577e-04,-1.912628425346785122e-04,-1.911066171709219219e-04,-1.909507661250753330e-04,-1.907952880675356511e-04,-1.906401816753611200e-04,-1.904854456317162104e-04,-1.903310786256495746e-04,-1.901770793525381364e-04,-1.900234465141981133e-04,-1.898701788175527483e-04,-1.897172749760756005e-04,-1.895647337094574780e-04,-1.894125537426072370e-04,-1.892607338072060941e-04,-1.891092726399312696e-04,-1.889581689836772327e-04,-1.888074215875557016e-04,-1.886570292054523534e-04,-1.885069905975811366e-04,-1.883573045298181370e-04,-1.882079697737015778e-04,-1.880589851059877304e-04,-1.879103493096501154e-04,-1.877620611729913236e-04,-1.876141194893099495e-04,-1.874665230578997921e-04,-1.873192706834947430e-04,-1.871723611762687867e-04,-1.870257933515029336e-04,-1.868795660301403316e-04,-1.867336780381201322e-04,-1.865881282070436242e-04,-1.864429153732860556e-04,-1.862980383791068562e-04,-1.861534960718724818e-04,-1.860092873032792582e-04,-1.858654109309076929e-04,-1.857218658173342973e-04,-1.855786508304646532e-04,-1.854357648428672789e-04,-1.852932067318846521e-04,-1.851509753805213876e-04,-1.850090696766670817e-04,-1.848674885125412004e-04,-1.847262307856922803e-04,-1.845852953988869061e-04,-1.844446812589994877e-04,-1.843043872779004388e-04,-1.841644123730112881e-04,-1.840247554657503670e-04,-1.838854154823099663e-04,-1.837463913542114469e-04,-1.836076820169729729e-04,-1.834692864109976895e-04,-1.833312034815737235e-04,-1.831934321782080488e-04,-1.830559714550705763e-04,-1.829188202713272204e-04,-1.827819775903627431e-04,-1.826454423797807536e-04,-1.825092136121808650e-04,-1.823732902644925602e-04,-1.822376713174200802e-04,-1.821023557568857143e-04,-1.819673425732526439e-04,-1.818326307606588088e-04,-1.816982193179050853e-04,-1.815641072479001750e-04,-1.814302935579936715e-04,-1.812967772597540161e-04,-1.811635573693015644e-04,-1.810306329063093855e-04,-1.808980028950024632e-04,-1.807656663640466732e-04,-1.806336223458826495e-04,-1.805018698769478291e-04,-1.803704079981205410e-04,-1.802392357544979618e-04,-1.801083521945079369e-04,-1.799777563712412487e-04,-1.798474473415634378e-04,-1.797174241666699146e-04,-1.795876859113088031e-04,-1.794582316440029857e-04,-1.793290604377162367e-04,-1.792001713691870890e-04,-1.790715635185957666e-04,-1.789432359703413411e-04,-1.788151878130417316e-04,-1.786874181385345040e-04,-1.785599260425430046e-04,-1.784327106243432937e-04,-1.783057709877633457e-04,-1.781791062397397596e-04,-1.780527154910949150e-04,-1.779265978562039052e-04,-1.778007524528835148e-04,-1.776751784036134652e-04,-1.775498748336490351e-04,-1.774248408716871950e-04,-1.773000756505327402e-04,-1.771755783064321577e-04,-1.770513479791846478e-04,-1.769273838122531473e-04,
        -1.768036849522092169e-04,-1.766802505491771313e-04,-1.765570797575000128e-04,-1.764341717344075633e-04,-1.763115256399050423e-04,-1.761891406387716685e-04,-1.760670158986732403e-04,-1.759451505898290691e-04,-1.758235438870103806e-04,-1.757021949680970252e-04,-1.755811030134113437e-04,-1.754602672077165693e-04,-1.753396867387735369e-04,-1.752193607970076172e-04,-1.750992885765079166e-04,-1.749794692750272773e-04,-1.748599020933161441e-04,-1.747405862347894967e-04,-1.746215209066370733e-04,-1.745027053192682587e-04,-1.743841386857569731e-04,-1.742658202226188280e-04,-1.741477491497001040e-04,-1.740299246898446839e-04,-1.739123460687830303e-04,-1.737950125153542302e-04,-1.736779232615059954e-04,-1.735610775425167063e-04,-1.734444745965513235e-04,-1.733281136644393428e-04,-1.732119939904519512e-04,-1.730961148215248713e-04,-1.729804754080355167e-04,-1.728650750029148142e-04,-1.727499128617582258e-04,-1.726349882437139271e-04,-1.725203004107056515e-04,-1.724058486273216673e-04,-1.722916321609258006e-04,-1.721776502824345911e-04,-1.720639022648740024e-04,-1.719503873840455555e-04,-1.718371049194145073e-04,-1.717240541523334940e-04,-1.716112343675968432e-04,-1.714986448525523954e-04,-1.713862848969904817e-04,-1.712741537941431247e-04,-1.711622508394627928e-04,-1.710505753308444454e-04,-1.709391265698467777e-04,-1.708279038599158639e-04,-1.707169065071623137e-04,-1.706061338213604728e-04,-1.704955851135059319e-04,-1.703852596981469958e-04,-1.702751568926075265e-04,-1.701652760159877431e-04,-1.700556163906075113e-04,-1.699461773414512322e-04,-1.698369581956127305e-04,-1.697279582830724109e-04,-1.696191769362531687e-04,-1.695106134901314121e-04,-1.694022672824591069e-04,-1.692941376530976427e-04,-1.691862239444619220e-04,-1.690785255017424049e-04,-1.689710416722389752e-04,-1.688637718061380966e-04,-1.687567152558466788e-04,-1.686498713755479884e-04,-1.685432395232000502e-04,-1.684368190584262237e-04,-1.683306093429592920e-04,-1.682246097415296404e-04,-1.681188196209770780e-04,-1.680132383502508375e-04,-1.679078653009646871e-04,-1.678026998471748854e-04,-1.676977413651581372e-04,-1.675929892330785265e-04,-1.674884428322087615e-04,-1.673841015454868852e-04,-1.672799647582934313e-04,-1.671760318586734684e-04,-1.670723022358933108e-04,-1.669687752827719862e-04,-1.668654503939048794e-04,-1.667623269654416873e-04,-1.666594043965297089e-04,-1.665566820884256671e-04,-1.664541594441626415e-04,-1.663518358694382471e-04,-1.662497107719485001e-04,-1.661477835612767961e-04,-1.660460536497820883e-04,-1.659445204514886640e-04,-1.658431833823081902e-04,-1.657420418610389135e-04,-1.656410953081444148e-04,-1.655403431461976993e-04,-1.654397847998811955e-04,-1.653394196958757334e-04,-1.652392472634156562e-04,-1.651392669328455298e-04,-1.650394781375075226e-04,-1.649398803126311819e-04,-1.648404728946673004e-04,-1.647412553232863175e-04,-1.646422270393799181e-04,-1.645433874858381884e-04,-1.644447361078826830e-04,-1.643462723529554026e-04,-1.642479956696085708e-04,-1.641499055087258796e-04,-1.640520013239665786e-04,-1.639542825698780959e-04,-1.638567487031172831e-04,-1.637593991827834827e-04,-1.636622334695303493e-04,-1.635652510257878944e-04,-1.634684513164286201e-04,-1.633718338077683185e-04,-1.632753979677881162e-04,-1.631791432669116304e-04,-1.630830691774498575e-04,-1.629871751728240170e-04,-1.628914607287867966e-04,-1.627959253234223524e-04,-1.627005684359250637e-04,-1.626053895473766886e-04,-1.625103881410794315e-04,-1.624155637015567422e-04,-1.623209157158855831e-04,-1.622264436723641623e-04,-1.621321470608449999e-04,-1.620380253739561738e-04,-1.619440781052139400e-04,-1.618503047503550007e-04,-1.617567048062262813e-04,-1.616632777721171976e-04,-1.615700231487604555e-04,-1.614769404383320506e-04,-1.613840291455614917e-04,-1.612912887759554437e-04,-1.611987188372410174e-04,-1.611063188388106582e-04,-1.610140882915001015e-04,-1.609220267079214395e-04,-1.608301336025741435e-04,-1.607384084914009748e-04,-1.606468508916769622e-04,-1.605554603231196253e-04,-1.604642363066677291e-04,-1.603731783643702613e-04,-1.602822860210517675e-04,-1.601915588023139492e-04,-1.601009962352017979e-04,-1.600105978490917735e-04,-1.599203631743595366e-04,-1.598302917432681269e-04,-1.597403830895238741e-04,-1.596506367487204869e-04,-1.595610522575618972e-04,-1.594716291543063491e-04,-1.593823669793215103e-04,-1.592932652741962940e-04,-1.592043235818518809e-04,-1.591155414466527418e-04,-1.590269184150727710e-04,-1.589384540346960861e-04,-1.588501478547721391e-04,-1.587619994261046941e-04,-1.586740083002746715e-04,-1.585861740315275270e-04,-1.584984961749968946e-04,-1.584109742869266313e-04,-1.583236079256700179e-04,-1.582363966508015807e-04,-1.581493400230060686e-04,-1.580624376051886770e-04,-1.579756889611427795e-04,-1.578890936559940172e-04,-1.578026512566443884e-04,-1.577163613312171364e-04,-1.576302234496118615e-04,-1.575442371825053200e-04,-1.574584021024616476e-04,-1.573727177833772473e-04,-1.572871838003697675e-04,-1.572017997300001468e-04,-1.571165651504946581e-04,-1.570314796411897973e-04,-1.569465427824212611e-04,-1.568617541565231477e-04,-1.567771133471618228e-04,-1.566926199390028529e-04,-1.566082735181550945e-04,-1.565240736721706938e-04,-1.564400199896009980e-04,-1.563561120608847332e-04,-1.562723494774598265e-04,-1.561887318319854501e-04,-1.561052587188971330e-04,-1.560219297330744936e-04,-1.559387444713955517e-04,-1.558557025320705947e-04,-1.557728035140870659e-04,-1.556900470183197882e-04,-1.556074326463097179e-04,-1.555249600011521238e-04,-1.554426286874965868e-04,-1.553604383107698439e-04,-1.552783884778419221e-04,-1.551964787968040937e-04,-1.551147088771909210e-04,-1.550330783295361670e-04,-1.549515867657058621e-04,-1.548702337985652377e-04,-1.547890190428669044e-04,-1.547079421136965394e-04,-1.546270026276941323e-04,-1.545462002032760296e-04,-1.544655344593026669e-04,-1.543850050160777698e-04,-1.543046114946822200e-04,-1.542243535185283676e-04,-1.541442307114726518e-04,-1.540642426981486679e-04,-1.539843891048553459e-04,-1.539046695590018388e-04,-1.538250836893295670e-04,-1.537456311255791519e-04,-1.536663114983793932e-04,-1.535871244395803359e-04,-1.535080695825863373e-04,-1.534291465619119776e-04,-1.533503550125159265e-04,-1.532716945710221879e-04,-1.531931648753870334e-04,-1.531147655641218464e-04,-1.530364962772923221e-04,-1.529583566557413121e-04,-1.528803463415329134e-04,-1.528024649778414457e-04,-1.527247122090624742e-04,-1.526470876809238320e-04,-1.525695910394864185e-04,-1.524922219320323791e-04,-1.524149800076202155e-04,-1.523378649159745635e-04,-1.522608763075972149e-04,-1.521840138344332516e-04,-1.521072771494269560e-04,-1.520306659065218113e-04,-1.519541797605494793e-04,-1.518778183673408222e-04,-1.518015813845030593e-04,-1.517254684697544320e-04,-1.516494792823674942e-04,-1.515736134826140002e-04,-1.514978707313208162e-04,-1.514222506910911648e-04,-1.513467530248613357e-04,-1.512713773968998865e-04,-1.511961234725855974e-04,-1.511209909181854272e-04,-1.510459794007434908e-04,-1.509710885885251486e-04,-1.508963181506839391e-04,-1.508216677577056686e-04,-1.507471370809643219e-04,-1.506727257917228613e-04,-1.505984335636867399e-04,-1.505242600713385670e-04,-1.504502049891609516e-04,-1.503762679934128599e-04,-1.503024487611304139e-04,-1.502287469701268918e-04,-1.501551622994368174e-04,-1.500816944289828925e-04,-1.500083430392429307e-04,-1.499351078122490577e-04,-1.498619884304774885e-04,-1.497889845776256834e-04,-1.497160959385013257e-04,-1.496433221979120987e-04,-1.495706630424420425e-04,-1.494981181597854203e-04,-1.494256872378585399e-04,-1.493533699655769098e-04,-1.492811660329662615e-04,-1.492090751313845942e-04,-1.491370969524119516e-04,-1.490652311885165560e-04,-1.489934775333878747e-04,-1.489218356818255984e-04,-1.488503053290735068e-04,-1.487788861708194688e-04,-1.487075779047497548e-04,-1.486363802291057468e-04,-1.485652928423508712e-04,-1.484943154439477553e-04,-1.484234477350243608e-04,-1.483526894169306942e-04,-1.482820401917939179e-04,-1.482114997628514175e-04,-1.481410678338956899e-04,-1.480707441102735444e-04,-1.480005282976648573e-04,-1.479304201020825715e-04,-1.478604192312049648e-04,-1.477905253934874708e-04,-1.477207382976075678e-04,-1.476510576535750019e-04,-1.475814831721766751e-04,-1.475120145646435788e-04,-1.474426515436499940e-04,-1.473733938220922468e-04,-1.473042411140879082e-04,-1.472351931341986386e-04,-1.471662495980963214e-04,-1.470974102225630631e-04,-1.470286747239368808e-04,-1.469600428205541931e-04,-1.468915142315285749e-04,-1.468230886758625786e-04,-1.467547658741130689e-04,-1.466865455475030444e-04,-1.466184274176995928e-04,-1.465504112072579801e-04,-1.464824966401767625e-04,-1.464146834401214292e-04,-1.463469713320897370e-04,-1.462793600423006879e-04,-1.462118492965291949e-04,-1.461444388227706170e-04,-1.460771283487982686e-04,-1.460099176028295531e-04,-1.459428063150802757e-04,-1.458757942155441967e-04,-1.458088810352142772e-04,-1.457420665059716569e-04,-1.456753503603636091e-04,-1.456087323312704740e-04,-1.455422121527938373e-04,-1.454757895599234629e-04,-1.454094642878711596e-04,-1.453432360724038475e-04,-1.452771046508427588e-04,-1.452110697610642376e-04,-1.451451311406115607e-04,-1.450792885290264067e-04,-1.450135416657394316e-04,-1.449478902912915146e-04,-1.448823341472227355e-04,-1.448168729746290850e-04,-1.447515065164939330e-04,-1.446862345162447383e-04,-1.446210567173089601e-04,-1.445559728646683695e-04,-1.444909827035267824e-04,-1.444260859801982377e-04,-1.443612824409967743e-04,-1.442965718333466540e-04,-1.442319539055603173e-04,-1.441674284060612266e-04,-1.441029950843830676e-04,-1.440386536907256598e-04,-1.439744039759549565e-04,-1.439102456910479333e-04,-1.438461785883138333e-04,-1.437822024206170113e-04,-1.437183169412659112e-04,-1.436545219044571553e-04,-1.435908170648314552e-04,-1.435272021776956564e-04,-1.434636769990227378e-04,-1.434002412857848796e-04,-1.433368947951763062e-04,-1.432736372848353312e-04,-1.432104685138435585e-04,-1.431473882413936138e-04,
        -1.430843962270111902e-04,-1.430214922313322035e-04,-1.429586760158807479e-04,-1.428959473424029625e-04,-1.428333059727560084e-04,-1.427707516701293144e-04,-1.427082841986004880e-04,-1.426459033223581585e-04,-1.425836088060350448e-04,-1.425214004152630665e-04,-1.424592779161182321e-04,-1.423972410753426843e-04,-1.423352896605667439e-04,-1.422734234394207320e-04,-1.422116421804231479e-04,-1.421499456529806693e-04,-1.420883336268330410e-04,-1.420268058721640969e-04,-1.419653621602678939e-04,-1.419040022623274666e-04,-1.418427259507470950e-04,-1.417815329982641259e-04,-1.417204231778379508e-04,-1.416593962639822735e-04,-1.415984520308777306e-04,-1.415375902533710928e-04,-1.414768107075303760e-04,-1.414161131696456408e-04,-1.413554974162289923e-04,-1.412949632246807141e-04,-1.412345103731782459e-04,-1.411741386401210718e-04,-1.411138478045748101e-04,-1.410536376466042796e-04,-1.409935079458302098e-04,-1.409334584833166204e-04,-1.408734890406826423e-04,-1.408135993993253621e-04,-1.407537893423072006e-04,-1.406940586523575121e-04,-1.406344071127607620e-04,-1.405748345081336836e-04,-1.405153406229819879e-04,-1.404559252425885418e-04,-1.403965881525692794e-04,-1.403373291393172906e-04,-1.402781479896697547e-04,-1.402190444911299849e-04,-1.401600184316453834e-04,-1.401010695993853972e-04,-1.400421977838517407e-04,-1.399834027739910169e-04,-1.399246843600820966e-04,-1.398660423334030511e-04,-1.398074764842327511e-04,-1.397489866042933571e-04,-1.396905724859731635e-04,-1.396322339221045539e-04,-1.395739707058529788e-04,-1.395157826304949111e-04,-1.394576694908611358e-04,-1.393996310814493711e-04,-1.393416671974234688e-04,-1.392837776349464818e-04,-1.392259621899594180e-04,-1.391682206591804416e-04,-1.391105528403269176e-04,-1.390529585311162108e-04,-1.389954375297097755e-04,-1.389379896349351995e-04,-1.388806146460641600e-04,-1.388233123631454902e-04,-1.387660825864500680e-04,-1.387089251164708159e-04,-1.386518397549219017e-04,-1.385948263034064709e-04,-1.385378845638607359e-04,-1.384810143398862436e-04,-1.384242154340853403e-04,-1.383674876501705953e-04,-1.383108307928537783e-04,-1.382542446664025704e-04,-1.381977290761948751e-04,-1.381412838279416633e-04,-1.380849087277979947e-04,-1.380286035821409740e-04,-1.379723681981248617e-04,-1.379162023835700523e-04,-1.378601059462969403e-04,-1.378040786947920537e-04,-1.377481204380970325e-04,-1.376922309859196503e-04,-1.376364101479676805e-04,-1.375806577342819637e-04,-1.375249735560135633e-04,-1.374693574245355876e-04,-1.374138091514431892e-04,-1.373583285489976547e-04,-1.373029154297933374e-04,-1.372475696069797024e-04,-1.371922908942613262e-04,-1.371370791056758520e-04,-1.370819340555939903e-04,-1.370268555588305404e-04,-1.369718434306443910e-04,-1.369168974874046540e-04,-1.368620175452583965e-04,-1.368072034205747300e-04,-1.367524549306109449e-04,-1.366977718928463759e-04,-1.366431541256485360e-04,-1.365886014472739163e-04,-1.365341136767561636e-04,-1.364796906333509696e-04,-1.364253321368691374e-04,-1.363710380074545370e-04,-1.363168080656951275e-04,-1.362626421329560245e-04,-1.362085400306023431e-04,-1.361545015803322656e-04,-1.361005266048431750e-04,-1.360466149267214320e-04,-1.359927663691085087e-04,-1.359389807558120111e-04,-1.358852579108615899e-04,-1.358315976585089402e-04,-1.357779998238939356e-04,-1.357244642326005390e-04,-1.356709907094355572e-04,-1.356175790812041981e-04,-1.355642291744896255e-04,-1.355109408159860251e-04,-1.354577138332757613e-04,-1.354045480540522206e-04,-1.353514433063418565e-04,-1.352983994190593009e-04,-1.352454162210081634e-04,-1.351924935415471651e-04,-1.351396312107011610e-04,-1.350868290587170506e-04,-1.350340869161748003e-04,-1.349814046135433543e-04,-1.349287819829569912e-04,-1.348762188561059006e-04,-1.348237150647912941e-04,-1.347712704421466512e-04,-1.347188848209723844e-04,-1.346665580344019730e-04,-1.346142899165680973e-04,-1.345620803017144596e-04,-1.345099290239737400e-04,-1.344578359188108863e-04,-1.344058008213577793e-04,-1.343538235670793668e-04,-1.343019039927728642e-04,-1.342500419345693530e-04,-1.341982372289329817e-04,-1.341464897135491441e-04,-1.340947992262142563e-04,-1.340431656049467790e-04,-1.339915886880982399e-04,-1.339400683140201664e-04,-1.338886043223963540e-04,-1.338371965529105978e-04,-1.337858448448026039e-04,-1.337345490388663904e-04,-1.336833089760069981e-04,-1.336321244966853783e-04,-1.335809954424727053e-04,-1.335299216553842427e-04,-1.334789029774352542e-04,-1.334279392510850926e-04,-1.333770303193482221e-04,-1.333261760253501294e-04,-1.332753762126603903e-04,-1.332246307255147144e-04,-1.331739394082598338e-04,-1.331233021050204357e-04,-1.330727186614755198e-04,-1.330221889230820409e-04,-1.329717127350749095e-04,-1.329212899440213036e-04,-1.328709203963773788e-04,-1.328206039390433801e-04,-1.327703404190305747e-04,-1.327201296840163636e-04,-1.326699715819001923e-04,-1.326198659609145736e-04,-1.325698126698471313e-04,-1.325198115575965119e-04,-1.324698624733944285e-04,-1.324199652669166838e-04,-1.323701197881721470e-04,-1.323203258877247990e-04,-1.322705834162496430e-04,-1.322208922244216822e-04,-1.321712521640261429e-04,-1.321216630869592734e-04,-1.320721248448952778e-04,-1.320226372903965384e-04,-1.319732002762474821e-04,-1.319238136555656027e-04,-1.318744772819124833e-04,-1.318251910089607293e-04,-1.317759546907160129e-04,-1.317267681818501401e-04,-1.316776313372569618e-04,-1.316285440116082839e-04,-1.315795060606861355e-04,-1.315305173403835681e-04,-1.314815777068156777e-04,-1.314326870160975602e-04,-1.313838451251214678e-04,-1.313350518915568088e-04,-1.312863071720737906e-04,-1.312376108247859108e-04,-1.311889627080287113e-04,-1.311403626800267119e-04,-1.310918105994485217e-04,-1.310433063254068387e-04,-1.309948497174584503e-04,-1.309464406350491217e-04,-1.308980789386238186e-04,-1.308497644885164846e-04,-1.308014971449500408e-04,-1.307532767691466091e-04,-1.307051032227724008e-04,-1.306569763672715823e-04,-1.306088960645324093e-04,-1.305608621767762045e-04,-1.305128745668904244e-04,-1.304649330976515031e-04,-1.304170376320579194e-04,-1.303691880341073528e-04,-1.303213841674644158e-04,-1.302736258959047433e-04,-1.302259130845362378e-04,-1.301782455980227127e-04,-1.301306233011390034e-04,-1.300830460595481242e-04,-1.300355137389130888e-04,-1.299880262054520230e-04,-1.299405833252720299e-04,-1.298931849650353243e-04,-1.298458309918482101e-04,-1.297985212728169913e-04,-1.297512556756030833e-04,-1.297040340680899462e-04,-1.296568563184941070e-04,-1.296097222953651595e-04,-1.295626318668086085e-04,-1.295155849025952932e-04,-1.294685812721629858e-04,-1.294216208448384364e-04,-1.293747034909475957e-04,-1.293278290802613029e-04,-1.292809974835495979e-04,-1.292342085722486544e-04,-1.291874622167954456e-04,-1.291407582887371674e-04,-1.290940966603981721e-04,-1.290474772032146333e-04,-1.290008997898439702e-04,-1.289543642930546241e-04,-1.289078705853929918e-04,-1.288614185402936485e-04,-1.288150080315242363e-04,-1.287686389326303527e-04,-1.287223111174906620e-04,-1.286760244607609849e-04,-1.286297788373191864e-04,-1.285835741220431316e-04,-1.285374101900327304e-04,-1.284912869166099370e-04,-1.284452041780959064e-04,-1.283991618505897492e-04,-1.283531598100795534e-04,-1.283071979334415857e-04,-1.282612760977741573e-04,-1.282153941803976238e-04,-1.281695520587433634e-04,-1.281237496103537765e-04,-1.280779867136594419e-04,-1.280322632472019606e-04,-1.279865790893008892e-04,-1.279409341188308957e-04,-1.278953282154438043e-04,-1.278497612585693943e-04,-1.278042331278594901e-04,-1.277587437030769379e-04,-1.277132928648727628e-04,-1.276678804942310563e-04,-1.276225064713587543e-04,-1.275771706777950598e-04,-1.275318729953012209e-04,-1.274866133048613293e-04,-1.274413914887917443e-04,-1.273962074298529146e-04,-1.273510610099171103e-04,-1.273059521119668247e-04,-1.272608806193176179e-04,-1.272158464150630053e-04,-1.271708493831846809e-04,-1.271258894073312717e-04,-1.270809663713734494e-04,-1.270360801604031309e-04,-1.269912306589571216e-04,-1.269464177516832493e-04,-1.269016413240064978e-04,-1.268569012614628733e-04,-1.268121974500324711e-04,-1.267675297754733421e-04,-1.267228981240986485e-04,-1.266783023827766641e-04,-1.266337424381536181e-04,-1.265892181773198288e-04,-1.265447294876986817e-04,-1.265002762571576511e-04,-1.264558583730091001e-04,-1.264114757238976594e-04,-1.263671281984679595e-04,-1.263228156846984973e-04,-1.262785380720110595e-04,-1.262342952497164106e-04,-1.261900871071253150e-04,-1.261459135339926263e-04,-1.261017744199621760e-04,-1.260576696556769960e-04,-1.260135991318911408e-04,-1.259695627388035533e-04,-1.259255603678344215e-04,-1.258815919101818892e-04,-1.258376572572661445e-04,-1.257937563009514648e-04,-1.257498889334351944e-04,-1.257060550468036553e-04,-1.256622545334762364e-04,-1.256184872869825497e-04,-1.255747531997419841e-04,-1.255310521651731293e-04,-1.254873840770276416e-04,-1.254437488289461555e-04,-1.254001463154574836e-04,-1.253565764303132823e-04,-1.253130390683754314e-04,-1.252695341248388772e-04,-1.252260614941214101e-04,-1.251826210717510435e-04,-1.251392127539219246e-04,-1.250958364359400221e-04,-1.250524920136664164e-04,-1.250091793838503662e-04,-1.249658984431301079e-04,-1.249226490881438778e-04,-1.248794312160850239e-04,-1.248362447242579165e-04,-1.247930895100779480e-04,-1.247499654718486894e-04,-1.247068725070965556e-04,-1.246638105143471620e-04,-1.246207793924591911e-04,-1.245777790395141693e-04,-1.245348093552589575e-04,-1.244918702388853049e-04,-1.244489615898070056e-04,-1.244060833077709205e-04,-1.243632352928569773e-04,-1.243204174454781707e-04,-1.242776296658254509e-04,-1.242348718547559017e-04,-1.241921439134596739e-04,-1.241494457432379406e-04,-1.241067772453918749e-04,-1.240641383214446947e-04,-1.240215288736967736e-04,-1.239789488044484855e-04,-1.239363980158891820e-04,-1.238938764105412815e-04,-1.238513838914823140e-04,-1.238089203621228762e-04,-1.237664857257625428e-04,-1.237240798858119106e-04,-1.236817027461256657e-04,-1.236393542112246280e-04,
        -1.235970341849634835e-04,-1.235547425723071413e-04,-1.235124792779984659e-04,-1.234702442070023665e-04,-1.234280372645057966e-04,-1.233858583562508215e-04,-1.233437073879795065e-04,-1.233015842655449390e-04,-1.232594888953553180e-04,-1.232174211835967981e-04,-1.231753810372326896e-04,-1.231333683633373255e-04,-1.230913830687629940e-04,-1.230494250609170948e-04,-1.230074942477621391e-04,-1.229655905368165492e-04,-1.229237138360428361e-04,-1.228818640541806673e-04,-1.228400410998586878e-04,-1.227982448813724758e-04,-1.227564753080168103e-04,-1.227147322891974923e-04,-1.226730157342093008e-04,-1.226313255524580370e-04,-1.225896616544597251e-04,-1.225480239502863000e-04,-1.225064123498986746e-04,-1.224648267643679844e-04,-1.224232671042102538e-04,-1.223817332808296854e-04,-1.223402252054084371e-04,-1.222987427894617340e-04,-1.222572859448378679e-04,-1.222158545833851306e-04,-1.221744486172848809e-04,-1.221330679589405221e-04,-1.220917125215326138e-04,-1.220503822176866038e-04,-1.220090769602499847e-04,-1.219677966625143384e-04,-1.219265412385484026e-04,-1.218853106019768262e-04,-1.218441046665352800e-04,-1.218029233467365913e-04,-1.217617665569825647e-04,-1.217206342121190943e-04,-1.216795262269920741e-04,-1.216384425164473981e-04,-1.215973829959970942e-04,-1.215563475814862571e-04,-1.215153361884269145e-04,-1.214743487329972282e-04,-1.214333851315974044e-04,-1.213924453004056048e-04,-1.213515291560440801e-04,-1.213106366156901927e-04,-1.212697675965213051e-04,-1.212289220157147795e-04,-1.211880997908920676e-04,-1.211473008398966655e-04,-1.211065250807941140e-04,-1.210657724315389316e-04,-1.210250428108627929e-04,-1.209843361374973725e-04,-1.209436523300633226e-04,-1.209029913079584517e-04,-1.208623529902475013e-04,-1.208217372967723691e-04,-1.207811441471529079e-04,-1.207405734612310155e-04,-1.207000251591816564e-04,-1.206594991617349066e-04,-1.206189953895098199e-04,-1.205785137629034054e-04,-1.205380542035339175e-04,-1.204976166327975662e-04,-1.204572009715354497e-04,-1.204168071415878671e-04,-1.203764350652392068e-04,-1.203360846646628346e-04,-1.202957558618100720e-04,-1.202554485796314410e-04,-1.202151627408554191e-04,-1.201748982684325284e-04,-1.201346550856463580e-04,-1.200944331156694744e-04,-1.200542322824516006e-04,-1.200140525096093924e-04,-1.199738937216476842e-04,-1.199337558424051764e-04,-1.198936387967197703e-04,-1.198535425092073226e-04,-1.198134669045947120e-04,-1.197734119081639292e-04,-1.197333774449749200e-04,-1.196933634415309200e-04,-1.196533698230028975e-04,-1.196133965150059097e-04,-1.195734434440431926e-04,-1.195335105365069595e-04,-1.194935977196776022e-04,-1.194537049195032452e-04,-1.194138320630422356e-04,-1.193739790782410992e-04,-1.193341458920471609e-04,-1.192943324321849019e-04,-1.192545386268228924e-04,-1.192147644037966359e-04,-1.191750096910526580e-04,-1.191352744178697520e-04,-1.190955585128605776e-04,-1.190558619041937050e-04,-1.190161845217030390e-04,-1.189765262948894176e-04,-1.189368871526985671e-04,-1.188972670251864372e-04,-1.188576658424089771e-04,-1.188180835343111141e-04,-1.187785200317259537e-04,-1.187389752647094454e-04,-1.186994491640946947e-04,-1.186599416614919633e-04,-1.186204526875123122e-04,-1.185809821737660030e-04,-1.185415300521963644e-04,-1.185020962541916134e-04,-1.184626807119171232e-04,-1.184232833577603117e-04,-1.183839041238865519e-04,-1.183445429431273510e-04,-1.183051997484252382e-04,-1.182658744729447875e-04,-1.182265670498505727e-04,-1.181872774121961456e-04,-1.181480054940342583e-04,-1.181087512296397080e-04,-1.180695145523991130e-04,-1.180302953969203372e-04,-1.179910936980332892e-04,-1.179519093897907211e-04,-1.179127424074666308e-04,-1.178735926862239936e-04,-1.178344601612257847e-04,-1.177953447681900911e-04,-1.177562464423909105e-04,-1.177171651201014413e-04,-1.176781007375948818e-04,-1.176390532308113634e-04,-1.176000225364681739e-04,-1.175610085911715785e-04,-1.175220113321939763e-04,-1.174830306964746995e-04,-1.174440666211751250e-04,-1.174051190441227632e-04,-1.173661879027010357e-04,-1.173272731349594977e-04,-1.172883746793917936e-04,-1.172494924739364563e-04,-1.172106264573091750e-04,-1.171717765683366608e-04,-1.171329427457346029e-04,-1.170941249286627794e-04,-1.170553230565030134e-04,-1.170165370690812168e-04,-1.169777669058902347e-04,-1.169390125067559794e-04,-1.169002738118374296e-04,-1.168615507616266314e-04,-1.168228432966156305e-04,-1.167841513575185175e-04,-1.167454748853824498e-04,-1.167068138211435624e-04,-1.166681681062931020e-04,-1.166295376822112928e-04,-1.165909224906114261e-04,-1.165523224737619046e-04,-1.165137375734870417e-04,-1.164751677320552403e-04,-1.164366128919569476e-04,-1.163980729960156779e-04,-1.163595479873880123e-04,-1.163210378090084873e-04,-1.162825424039226618e-04,-1.162440617156201839e-04,-1.162055956882568353e-04,-1.161671442655443087e-04,-1.161287073911942969e-04,-1.160902850101397377e-04,-1.160518770663143684e-04,-1.160134835045401047e-04,-1.159751042699719292e-04,-1.159367393073207353e-04,-1.158983885619635501e-04,-1.158600519792774008e-04,-1.158217295050834039e-04,-1.157834210850916534e-04,-1.157451266654563327e-04,-1.157068461925536695e-04,-1.156685796124268251e-04,-1.156303268716740718e-04,-1.155920879175598159e-04,-1.155538626969043747e-04,-1.155156511566390876e-04,-1.154774532445834723e-04,-1.154392689082239798e-04,-1.154010980950470611e-04,-1.153629407533163231e-04,-1.153247968310733285e-04,-1.152866662766927064e-04,-1.152485490387711309e-04,-1.152104450662383428e-04,-1.151723543075799938e-04,-1.151342767123919586e-04,-1.150962122298260226e-04,-1.150581608091449937e-04,-1.150201224004998579e-04,-1.149820969534864901e-04,-1.149440844182558763e-04,-1.149060847449590028e-04,-1.148680978845240119e-04,-1.148301237873239344e-04,-1.147921624038428234e-04,-1.147542136858969997e-04,-1.147162775840815385e-04,-1.146783540499907161e-04,-1.146404430354408532e-04,-1.146025444919152037e-04,-1.145646583716741773e-04,-1.145267846268671619e-04,-1.144889232098655896e-04,-1.144510740730408926e-04,-1.144132371692085925e-04,-1.143754124515172776e-04,-1.143375998731155363e-04,-1.142997993870409346e-04,-1.142620109468861500e-04,-1.142242345062438602e-04,-1.141864700193728765e-04,-1.141487174401989435e-04,-1.141109767226478056e-04,-1.140732478217554302e-04,-1.140355306915585842e-04,-1.139978252870932351e-04,-1.139601315637284173e-04,-1.139224494762780537e-04,-1.138847789802222010e-04,-1.138471200309298936e-04,-1.138094725845473221e-04,-1.137718365968876100e-04,-1.137342120238749033e-04,-1.136965988220994817e-04,-1.136589969478185580e-04,-1.136214063579554789e-04,-1.135838270094335911e-04,-1.135462588590652189e-04,-1.135087018639957535e-04,-1.134711559822587645e-04,-1.134336211708886211e-04,-1.133960973876968481e-04,-1.133585845912721268e-04,-1.133210827392039377e-04,-1.132835917899699396e-04,-1.132461117021588137e-04,-1.132086424345812858e-04,-1.131711839462701263e-04,-1.131337361960360610e-04,-1.130962991431339049e-04,-1.130588727472625621e-04,-1.130214569678988923e-04,-1.129840517650748666e-04,-1.129466570987114338e-04,-1.129092729288405650e-04,-1.128718992160493428e-04,-1.128345359207028054e-04,-1.127971830038321244e-04,-1.127598404262464271e-04,-1.127225081489768854e-04,-1.126851861333877380e-04,-1.126478743408432237e-04,-1.126105727332626927e-04,-1.125732812721214060e-04,-1.125359999196717808e-04,-1.124987286382772567e-04,-1.124614673900792283e-04,-1.124242161376631799e-04,-1.123869748438366400e-04,-1.123497434714071375e-04,-1.123125219835152677e-04,-1.122753103438567379e-04,-1.122381085153500990e-04,-1.122009164616910581e-04,-1.121637341472414562e-04,-1.121265615355859779e-04,-1.120893985907533974e-04,-1.120522452774386224e-04,-1.120151015602255384e-04,-1.119779674034759864e-04,-1.119408427725510080e-04,-1.119037276323675556e-04,-1.118666219478425816e-04,-1.118295256850032615e-04,-1.117924388089885923e-04,-1.117553612859367718e-04,-1.117182930817639530e-04,-1.116812341623862892e-04,-1.116441844944970896e-04,-1.116071440441235296e-04,-1.115701127784030078e-04,-1.115330906642508779e-04,-1.114960776682494270e-04,-1.114590737580911650e-04,-1.114220789010245127e-04,-1.113850930642978909e-04,-1.113481162160478988e-04,-1.113111483243001132e-04,-1.112741893566360218e-04,-1.112372392817473354e-04,-1.112002980681037201e-04,-1.111633656842858642e-04,-1.111264420990965007e-04,-1.110895272810052958e-04,-1.110526212000362278e-04,-1.110157238252140743e-04,-1.109788351257856576e-04,-1.109419550716639336e-04,-1.109050836324287914e-04,-1.108682207786593210e-04,-1.108313664802684784e-04,-1.107945207072802418e-04,-1.107576834308288127e-04,-1.107208546211602140e-04,-1.106840342496306917e-04,-1.106472222872634248e-04,-1.106104187046375031e-04,-1.105736234737753065e-04,-1.105368365663661478e-04,-1.105000579539883176e-04,-1.104632876085531734e-04,-1.104265255019720726e-04,-1.103897716068225066e-04,-1.103530258956819665e-04,-1.103162883410169215e-04,-1.102795589152938405e-04,-1.102428375916453263e-04,-1.102061243437590932e-04,-1.101694191444346771e-04,-1.101327219671377478e-04,-1.100960327856670418e-04,-1.100593515738212957e-04,-1.100226783056212909e-04,-1.099860129551988308e-04,-1.099493554967967412e-04,-1.099127059049909150e-04,-1.098760641546903116e-04,-1.098394302203598016e-04,-1.098028040772414116e-04,-1.097661857005771679e-04,-1.097295750653870527e-04,-1.096929721473571817e-04,-1.096563769222846929e-04,-1.096197893661887690e-04,-1.095832094546445035e-04,-1.095466371640041459e-04,-1.095100724709530127e-04,-1.094735153515102866e-04,-1.094369657826943509e-04,-1.094004237414125669e-04,-1.093638892043502508e-04,-1.093273621490808978e-04,-1.092908425527339134e-04,-1.092543303927717702e-04,-1.092178256474340969e-04,-1.091813282940723440e-04,-1.091448383105930731e-04,-1.091083556754579575e-04,-1.090718803672396930e-04,-1.090354123640668860e-04,-1.089989516446232543e-04,-1.089624981880366050e-04,-1.089260519732127008e-04,-1.088896129795013934e-04,-1.088531811859194676e-04,-1.088167565720388197e-04,-1.087803391177644130e-04,
        -1.087439288028901885e-04,-1.087075256072100871e-04,-1.086711295109621389e-04,-1.086347404943843742e-04,-1.085983585383809569e-04,-1.085619836235229840e-04,-1.085256157300484858e-04,-1.084892548393057154e-04,-1.084529009326429261e-04,-1.084165539911863263e-04,-1.083802139963951916e-04,-1.083438809300618644e-04,-1.083075547736456201e-04,-1.082712355090498235e-04,-1.082349231188439731e-04,-1.081986175850424559e-04,-1.081623188901037480e-04,-1.081260270165973481e-04,-1.080897419473147991e-04,-1.080534636650476443e-04,-1.080171921528094714e-04,-1.079809273942800019e-04,-1.079446693724728235e-04,-1.079084180709566354e-04,-1.078721734737442262e-04,-1.078359355641822503e-04,-1.077997043263945187e-04,-1.077634797450599535e-04,-1.077272618041913432e-04,-1.076910504881345432e-04,-1.076548457819015425e-04,-1.076186476700602412e-04,-1.075824561375116062e-04,-1.075462711694896711e-04,-1.075100927514505145e-04,-1.074739208687391923e-04,-1.074377555068117829e-04,-1.074015966513464093e-04,-1.073654442884652838e-04,-1.073292984045126630e-04,-1.072931589850556477e-04,-1.072570260167715617e-04,-1.072208994863377285e-04,-1.071847793803204496e-04,-1.071486656856190933e-04,-1.071125583889109834e-04,-1.070764574778726441e-04,-1.070403629394034439e-04,-1.070042747608468403e-04,-1.069681929305454915e-04,-1.069321174355097881e-04,-1.068960482637493214e-04,-1.068599854038287944e-04,-1.068239288434247314e-04,-1.067878785711018352e-04,-1.067518345754248088e-04,-1.067157968449583549e-04,-1.066797653689333103e-04,-1.066437401359143777e-04,-1.066077211347993270e-04,-1.065717083554851285e-04,-1.065357017870915968e-04,-1.064997014191826352e-04,-1.064637072416552144e-04,-1.064277192439622155e-04,-1.063917374165557206e-04,-1.063557617497767893e-04,-1.063197922334113699e-04,-1.062838288583556334e-04,-1.062478716150616620e-04,-1.062119204942035822e-04,-1.061759754872326766e-04,-1.061400365843789828e-04,-1.061041037774268503e-04,-1.060681770578275618e-04,-1.060322564166993331e-04,-1.059963418460485585e-04,-1.059604333376595875e-04,-1.059245308830947252e-04,-1.058886344745824104e-04,-1.058527441050172158e-04,-1.058168597661834909e-04,-1.057809814506427415e-04,-1.057451091512895403e-04,-1.057092428605743706e-04,-1.056733825718358943e-04,-1.056375282783017511e-04,-1.056016799728665134e-04,-1.055658376492019102e-04,-1.055300013007576254e-04,-1.054941709212053880e-04,-1.054583465045499935e-04,-1.054225280444631707e-04,-1.053867155352827822e-04,-1.053509089712356683e-04,-1.053151083469927585e-04,-1.052793136566698706e-04,-1.052435248950489566e-04,-1.052077420573560573e-04,-1.051719651380400578e-04,-1.051361941326600657e-04,-1.051004290361090554e-04,-1.050646698440571569e-04,-1.050289165520634782e-04,-1.049931691554650826e-04,-1.049574276503761894e-04,-1.049216920329110181e-04,-1.048859622987396989e-04,-1.048502384444205404e-04,-1.048145204662898067e-04,-1.047788083605727394e-04,-1.047431021242717364e-04,-1.047074017541671509e-04,-1.046717072471503585e-04,-1.046360186000017123e-04,-1.046003358102787217e-04,-1.045646588753168515e-04,-1.045289877923405442e-04,-1.044933225590183312e-04,-1.044576631733518113e-04,-1.044220096331205383e-04,-1.043863619362150885e-04,-1.043507200807480828e-04,-1.043150840651652089e-04,-1.042794538880231769e-04,-1.042438295477676746e-04,-1.042082110430664343e-04,-1.041725983725871885e-04,-1.041369915355527809e-04,-1.041013905310750332e-04,-1.040657953581547446e-04,-1.040302060164588482e-04,-1.039946225050991657e-04,-1.039590448240756970e-04,-1.039234729730553752e-04,-1.038879069515941111e-04,-1.038523467603580386e-04,-1.038167923991251129e-04,-1.037812438680063565e-04,-1.037457011676679031e-04,-1.037101642987758865e-04,-1.036746332618854183e-04,-1.036391080576626322e-04,-1.036035886873287737e-04,-1.035680751515499765e-04,-1.035325674517695305e-04,-1.034970655895417480e-04,-1.034615695660878743e-04,-1.034260793831842662e-04,-1.033905950420521691e-04,-1.033551165449120290e-04,-1.033196438937622474e-04,-1.032841770906012258e-04,-1.032487161376494100e-04,-1.032132610372382686e-04,-1.031778117919213145e-04,-1.031423684042520605e-04,-1.031069308772281090e-04,-1.030714992132919505e-04,-1.030360734153301649e-04,-1.030006534867844437e-04,-1.029652394308744334e-04,-1.029298312509308033e-04,-1.028944289503952447e-04,-1.028590325327094490e-04,-1.028236420019812414e-04,-1.027882573616523132e-04,-1.027528786158304897e-04,-1.027175057686235959e-04,-1.026821388241394573e-04,-1.026467777872630549e-04,-1.026114226618801695e-04,-1.025760734525427154e-04,-1.025407301642466962e-04,-1.025053928016550486e-04,-1.024700613697637763e-04,-1.024347358736799052e-04,-1.023994163186214834e-04,-1.023641027096955369e-04,-1.023287950525642032e-04,-1.022934933526675749e-04,-1.022581976154457450e-04,-1.022229078471159625e-04,-1.021876240533403646e-04,-1.021523462401141558e-04,-1.021170744135435626e-04,-1.020818085800678787e-04,-1.020465487459043530e-04,-1.020112949176033013e-04,-1.019760471018260617e-04,-1.019408053049009055e-04,-1.019055695341553047e-04,-1.018703397964726420e-04,-1.018351160987363002e-04,-1.017998984481627289e-04,-1.017646868521904224e-04,-1.017294813180358304e-04,-1.016942818531374471e-04,-1.016590884654888782e-04,-1.016239011626396405e-04,-1.015887199525833395e-04,-1.015535448430915366e-04,-1.015183758423798821e-04,-1.014832129587750487e-04,-1.014480562001596198e-04,-1.014129055754153796e-04,-1.013777610929800232e-04,-1.013426227614022679e-04,-1.013074905894528754e-04,-1.012723645861246524e-04,-1.012372447602993830e-04,-1.012021311209698737e-04,-1.011670236775730203e-04,-1.011319224389906068e-04,-1.010968274153256630e-04,-1.010617386160150843e-04,-1.010266560498296329e-04,-1.009915797272054050e-04,-1.009565096581344079e-04,-1.009214458522755820e-04,-1.008863883197319566e-04,-1.008513370706065615e-04,-1.008162921153354930e-04,-1.007812534644658697e-04,-1.007462211279896991e-04,-1.007111951168981889e-04,-1.006761754418494803e-04,-1.006411621133906920e-04,-1.006061551429571210e-04,-1.005711545410958863e-04,-1.005361603191312625e-04,-1.005011724883875246e-04,-1.004661910598558805e-04,-1.004312160451936720e-04,-1.003962474561692630e-04,-1.003612853039959063e-04,-1.003263296005529881e-04,-1.002913803580529617e-04,-1.002564375877090797e-04,-1.002215013019558398e-04,-1.001865715132277401e-04,-1.001516482334041669e-04,-1.001167314750306403e-04,-1.000818212505416582e-04,-1.000469175722606963e-04,-1.000120204529553192e-04,-9.997712990539309175e-05,-9.994224594245260107e-05,-9.990736857734550114e-05,-9.987249782261731212e-05,-9.983763369170173263e-05,-9.980277619781041665e-05,-9.976792535382195126e-05,-9.973308117405821349e-05,-9.969824367139779042e-05,-9.966341285960744756e-05,-9.962858875278701731e-05,-9.959377136392610907e-05,-9.955896070767966677e-05,-9.952415679792547820e-05,-9.948935964843030888e-05,-9.945456927373808043e-05,-9.941978568794862525e-05,-9.938500890560586498e-05,-9.935023894103167663e-05,-9.931547580865895952e-05,-9.928071952369776909e-05,-9.924597010058100466e-05,-9.921122755418565475e-05,-9.917649189927768560e-05,-9.914176315084510804e-05,-9.910704132443104442e-05,-9.907232643491248325e-05,-9.903761849783254689e-05,-9.900291752851231308e-05,-9.896822354227285956e-05,-9.893353655499037558e-05,-9.889885658209696118e-05,-9.886418363935778331e-05,-9.882951774264903122e-05,-9.879485890795791647e-05,-9.876020715127165062e-05,-9.872556248879948981e-05,-9.869092493675069022e-05,-9.865629451100144109e-05,-9.862167122820508780e-05,-9.858705510512599801e-05,-9.855244615808445019e-05,-9.851784440328970049e-05,-9.848324985806122811e-05,-9.844866253927442301e-05,-9.841408246302751905e-05,-9.837950964663999542e-05,-9.834494410765337591e-05,-9.831038586283202818e-05,-9.827583492938440912e-05,-9.824129132474102022e-05,-9.820675506622134066e-05,-9.817222617103382731e-05,-9.813770465705307089e-05,-9.810319054182059517e-05,-9.806868384321099086e-05,-9.803418457887680404e-05,-9.799969276680364771e-05,-9.796520842497713488e-05,-9.793073157093878933e-05,-9.789626222345138018e-05,-9.786180040027847582e-05,-9.782734612007182307e-05,-9.779289940115010182e-05,-9.775846026183199200e-05,-9.772402872043617350e-05,-9.768960479572541544e-05,-9.765518850668453155e-05,-9.762077987163220172e-05,-9.758637890955323968e-05,-9.755198563943245915e-05,-9.751760008014365155e-05,-9.748322225089367521e-05,-9.744885217066734384e-05,-9.741448985867151578e-05,-9.738013533444611625e-05,-9.734578861697595897e-05,-9.731144972613403610e-05,-9.727711868112720595e-05,-9.724279550138437145e-05,-9.720848020711159165e-05,-9.717417281762674719e-05,-9.713987335280283020e-05,-9.710558183273487742e-05,-9.707129827707383640e-05,-9.703702270624781079e-05,-9.700275513979672581e-05,-9.696849559814868513e-05,-9.693424410195383700e-05,-9.690000067075210666e-05,-9.686576532563773156e-05,-9.683153808692779307e-05,-9.679731897471732793e-05,-9.676310800998955131e-05,-9.672890521317256685e-05,-9.669471060513856742e-05,-9.666052420675974588e-05,-9.662634603890829510e-05,-9.659217612245640794e-05,-9.655801447827627726e-05,-9.652386112779520744e-05,-9.648971609166334673e-05,-9.645557939141902182e-05,-9.642145104860055937e-05,-9.638733108385810766e-05,-9.635321951906306026e-05,-9.631911637575374385e-05,-9.628502167524644051e-05,-9.625093543941254381e-05,-9.621685768923526894e-05,-9.618278844736316557e-05,-9.614872773533456041e-05,-9.611467557446573551e-05,-9.608063198740524058e-05,-9.604659699558038000e-05,-9.601257062130663655e-05,-9.597855288689949305e-05,-9.594454381445238766e-05,-9.591054342583671399e-05,-9.587655174336795483e-05,-9.584256878980568217e-05,-9.580859458757640112e-05,-9.577462915910661678e-05,-9.574067252671181194e-05,-9.570672471304053630e-05,-9.567278574085236187e-05,-9.563885563290686065e-05,-9.560493441207462695e-05,-9.557102210089318817e-05,-9.553711872267722782e-05,-9.550322430007529562e-05,-9.546933885595798586e-05,-9.543546241408407127e-05,-9.540159499699107926e-05,-9.536773662799369333e-05,-9.533388733018455241e-05,-9.530004712721140692e-05,-9.526621604261098497e-05,
        -9.523239409936490318e-05,-9.519858132089886738e-05,-9.516477773086062797e-05,-9.513098335300895769e-05,-9.509719821076956237e-05,-9.506342232823428162e-05,-9.502965572893984358e-05,-9.499589843620093177e-05,-9.496215047444245272e-05,-9.492841186742317916e-05,-9.489468263901290612e-05,-9.486096281330347324e-05,-9.482725241460876475e-05,-9.479355146679857569e-05,-9.475985999374270108e-05,-9.472617802008809207e-05,-9.469250557003761060e-05,-9.465884266812718550e-05,-9.462518933833763413e-05,-9.459154560520488531e-05,-9.455791149304282328e-05,-9.452428702672044381e-05,-9.449067223088469802e-05,-9.445706712984947018e-05,-9.442347174837273371e-05,-9.438988611110143978e-05,-9.435631024290458413e-05,-9.432274416865116251e-05,-9.428918791321017068e-05,-9.425564150145060438e-05,-9.422210495813043707e-05,-9.418857830867377601e-05,-9.415506157806063925e-05,-9.412155479082695564e-05,-9.408805797250785474e-05,-9.405457114874948843e-05,-9.402109434430983015e-05,-9.398762758416889795e-05,-9.395417089441693292e-05,-9.392072429992293081e-05,-9.388728782577793197e-05,-9.385386149840524439e-05,-9.382044534267386382e-05,-9.378703938389687522e-05,-9.375364364827554198e-05,-9.372025816123397135e-05,-9.368688294830729291e-05,-9.365351803503063621e-05,-9.362016344771628695e-05,-9.358681921178835239e-05,-9.355348535322605130e-05,-9.352016189800860246e-05,-9.348684887156011314e-05,-9.345354630008184671e-05,-9.342025420999711116e-05,-9.338697262728512527e-05,-9.335370157736999630e-05,-9.332044108689707684e-05,-9.328719118217865258e-05,-9.325395188897189769e-05,-9.322072323370012015e-05,-9.318750524289765025e-05,-9.315429794243268447e-05,-9.312110135895057539e-05,-9.308791551887463100e-05,-9.305474044829509239e-05,-9.302157617407935675e-05,-9.298842272253970975e-05,-9.295528012032150400e-05,-9.292214839418111438e-05,-9.288902756998673738e-05,-9.285591767504985938e-05,-9.282281873623787760e-05,-9.278973077964103311e-05,-9.275665383223774541e-05,-9.272358792078438938e-05,-9.269053307248142914e-05,-9.265748931386319498e-05,-9.262445667179708408e-05,-9.259143517326151596e-05,-9.255842484512388779e-05,-9.252542571469568600e-05,-9.249243780873328546e-05,-9.245946115421510569e-05,-9.242649577856365539e-05,-9.239354170864633176e-05,-9.236059897133053198e-05,-9.232766759437183168e-05,-9.229474760474865036e-05,-9.226183902966145212e-05,-9.222894189653274566e-05,-9.219605623234095049e-05,-9.216318206473061991e-05,-9.213031942101324034e-05,-9.209746832872234279e-05,-9.206462881494736905e-05,-9.203180090699980553e-05,-9.199898463307931706e-05,-9.196618002016432314e-05,-9.193338709578835477e-05,-9.190060588781800988e-05,-9.186783642367579716e-05,-9.183507873100626995e-05,-9.180233283712091463e-05,-9.176959877021939604e-05,-9.173687655783524519e-05,-9.170416622772403770e-05,-9.167146780764134917e-05,-9.163878132512071062e-05,-9.160610680825076457e-05,-9.157344428467606434e-05,-9.154079378248525245e-05,-9.150815532943390451e-05,-9.147552895349964075e-05,-9.144291468243803678e-05,-9.141031254422671282e-05,-9.137772256728737830e-05,-9.134514477904254193e-05,-9.131257920780289083e-05,-9.128002588154604524e-05,-9.124748482836064767e-05,-9.121495607655738524e-05,-9.118243965400285589e-05,-9.114993558900774673e-05,-9.111744390966070029e-05,-9.108496464382831448e-05,-9.105249782026536565e-05,-9.102004346706049631e-05,-9.098760161219132669e-05,-9.095517228430161083e-05,-9.092275551147999124e-05,-9.089035132203715506e-05,-9.085795974439481171e-05,-9.082558080675262602e-05,-9.079321453775435202e-05,-9.076086096571067685e-05,-9.072852011871024303e-05,-9.069619202561884919e-05,-9.066387671485820476e-05,-9.063157421473899689e-05,-9.059928455346089038e-05,-9.056700775988968388e-05,-9.053474386278015373e-05,-9.050249289044298706e-05,-9.047025487118887099e-05,-9.043802983410564877e-05,-9.040581780750400753e-05,-9.037361881969463440e-05,-9.034143289987639491e-05,-9.030926007635997621e-05,-9.027710037790015463e-05,-9.024495383336272880e-05,-9.021282047116940817e-05,-9.018070031996394675e-05,-9.014859340883418781e-05,-9.011649976631286307e-05,-9.008441942115474887e-05,-9.005235240222564386e-05,-9.002029873805827975e-05,-8.998825845774049981e-05,-8.995623159013810266e-05,-8.992421816389484235e-05,-8.989221820787651751e-05,-8.986023175094892679e-05,-8.982825882186684652e-05,-8.979629944982914225e-05,-8.976435366359059032e-05,-8.973242149190596706e-05,-8.970050296397413803e-05,-8.966859810854987956e-05,-8.963670695461001259e-05,-8.960482953102033576e-05,-8.957296586697971463e-05,-8.954111599146497014e-05,-8.950927993323087861e-05,-8.947745772136528331e-05,-8.944564938484500516e-05,-8.941385495297993202e-05,-8.938207445463586254e-05,-8.935030791890063995e-05,-8.931855537452904059e-05,-8.928681685105299692e-05,-8.925509237733830759e-05,-8.922338198236179352e-05,-8.919168569554436488e-05,-8.916000354552977569e-05,-8.912833556184995842e-05,-8.909668177370377862e-05,-8.906504220984601261e-05,-8.903341689936450365e-05,-8.900180587179118419e-05,-8.897020915632491977e-05,-8.893862678183150905e-05,-8.890705877750981756e-05,-8.887550517266973316e-05,-8.884396599639909908e-05,-8.881244127811882549e-05,-8.878093104691675563e-05,-8.874943533165868814e-05,-8.871795416209860008e-05,-8.868648756732433469e-05,-8.865503557642373522e-05,-8.862359821870668952e-05,-8.859217552326104084e-05,-8.856076751972974392e-05,-8.852937423720064203e-05,-8.849799570465055609e-05,-8.846663195161141857e-05,-8.843528300717107271e-05,-8.840394890086145097e-05,-8.837262966210346349e-05,-8.834132531987393122e-05,-8.831003590337171971e-05,-8.827876144201773911e-05,-8.824750196534392188e-05,-8.821625750210504435e-05,-8.818502808227712819e-05,-8.815381373505903895e-05,-8.812261448931657526e-05,-8.809143037447064728e-05,-8.806026142027523207e-05,-8.802910765581817287e-05,-8.799796911018731294e-05,-8.796684581313662932e-05,-8.793573779364294296e-05,-8.790464508123818632e-05,-8.787356770512122495e-05,-8.784250569482399129e-05,-8.781145907987841781e-05,-8.778042788903928084e-05,-8.774941215194953514e-05,-8.771841189803009087e-05,-8.768742715670185817e-05,-8.765645795727472489e-05,-8.762550432883653428e-05,-8.759456630080819650e-05,-8.756364390272164400e-05,-8.753273716421983153e-05,-8.750184611416855773e-05,-8.747097078187771046e-05,-8.744011119710126678e-05,-8.740926738914911454e-05,-8.737843938710909697e-05,-8.734762722029110193e-05,-8.731683091833808419e-05,-8.728605051067095388e-05,-8.725528602637755426e-05,-8.722453749465675088e-05,-8.719380494548456539e-05,-8.716308840794884105e-05,-8.713238791069333189e-05,-8.710170348402712648e-05,-8.707103515737113497e-05,-8.704038295936911140e-05,-8.700974691955298823e-05,-8.697912706745469791e-05,-8.694852343227310598e-05,-8.691793604342912261e-05,-8.688736493045468023e-05,-8.685681012254864442e-05,-8.682627164890988070e-05,-8.679574953884827693e-05,-8.676524382189576556e-05,-8.673475452725121215e-05,-8.670428168433552685e-05,-8.667382532245859750e-05,-8.664338547070826735e-05,-8.661296215883851346e-05,-8.658255541615922368e-05,-8.655216527142517435e-05,-8.652179175416829793e-05,-8.649143489403154916e-05,-8.646109472010277130e-05,-8.643077126146980760e-05,-8.640046454766459050e-05,-8.637017460799700785e-05,-8.633990147166592521e-05,-8.630964516809225273e-05,-8.627940572625281135e-05,-8.624918317567953352e-05,-8.621897754579332940e-05,-8.618878886523795302e-05,-8.615861716354533684e-05,-8.612846247046945791e-05,-8.609832481498713719e-05,-8.606820422596417330e-05,-8.603810073293249872e-05,-8.600801436509097897e-05,-8.597794515163847962e-05,-8.594789312199591080e-05,-8.591785830502907118e-05,-8.588784073026989319e-05,-8.585784042680622008e-05,-8.582785742383691741e-05,-8.579789175056085071e-05,-8.576794343606586324e-05,-8.573801250977286514e-05,-8.570809900065867737e-05,-8.567820293770012086e-05,-8.564832435054015036e-05,-8.561846326826660913e-05,-8.558861971974529581e-05,-8.555879373428609824e-05,-8.552898534075481507e-05,-8.549919456868337875e-05,-8.546942144715963252e-05,-8.543966600493835273e-05,-8.540992827166249413e-05,-8.538020827608683305e-05,-8.535050604707716815e-05,-8.532082161427645417e-05,-8.529115500655048976e-05,-8.526150625276507355e-05,-8.523187538211907111e-05,-8.520226242392237026e-05,-8.517266740704076966e-05,-8.514309036045109025e-05,-8.511353131290810836e-05,-8.508399029394375646e-05,-8.505446733264587778e-05,-8.502496245776924866e-05,-8.499547569817966775e-05,-8.496600708307600058e-05,-8.493655664165711272e-05,-8.490712440234471359e-05,-8.487771039455971334e-05,-8.484831464705688830e-05,-8.481893718914612634e-05,-8.478957804958220379e-05,-8.476023725689785238e-05,-8.473091484040295995e-05,-8.470161082918536977e-05,-8.467232525188883585e-05,-8.464305813749017915e-05,-8.461380951518826521e-05,-8.458457941340480346e-05,-8.455536786100559254e-05,-8.452617488707847571e-05,-8.449700052093334079e-05,-8.446784479076985264e-05,-8.443870772534278757e-05,-8.440958935407305574e-05,-8.438048970549338890e-05,-8.435140880858060797e-05,-8.432234669175642239e-05,-8.429330338399765310e-05,-8.426427891439214335e-05,-8.423527331158364717e-05,-8.420628660410489630e-05,-8.417731882071066707e-05,-8.414836999071084733e-05,-8.411944014230510192e-05,-8.409052930424820715e-05,-8.406163750562800629e-05,-8.403276477475518647e-05,-8.400391114082861321e-05,-8.397507663215897367e-05,-8.394626127739002186e-05,-8.391746510549857874e-05,-8.388868814501737603e-05,-8.385993042447914547e-05,-8.383119197263866340e-05,-8.380247281825070615e-05,-8.377377298962596086e-05,-8.374509251607431537e-05,-8.371643142568441220e-05,-8.368778974676693849e-05,-8.365916750829871518e-05,-8.363056473903451860e-05,-8.360198146739605818e-05,-8.357341772169402105e-05,-8.354487353101625047e-05,-8.351634892345138894e-05,-8.348784392742114591e-05,-8.345935857201336461e-05,-8.343089288531668757e-05,-8.340244689597486882e-05,-8.337402063229859550e-05,-8.334561412293162164e-05,-8.331722739673974587e-05,-8.328886048170058842e-05,-8.326051340623585872e-05,-8.323218619865624390e-05,-8.320387888793856490e-05,
        -8.317559150228248654e-05,-8.314732406999869596e-05,-8.311907661950890258e-05,-8.309084917934583814e-05,-8.306264177815325667e-05,-8.303445444346468918e-05,-8.300628720392388971e-05,-8.297814008872972380e-05,-8.295001312541572247e-05,-8.292190634229257284e-05,-8.289381976822607356e-05,-8.286575343086077794e-05,-8.283770735906248461e-05,-8.280968158114188071e-05,-8.278167612485454185e-05,-8.275369101884422207e-05,-8.272572629142160849e-05,-8.269778197067534364e-05,-8.266985808491611465e-05,-8.264195466245460864e-05,-8.261407173137946813e-05,-8.258620932011240257e-05,-8.255836745663103216e-05,-8.253054616935706633e-05,-8.250274548637914762e-05,-8.247496543567489624e-05,-8.244720604577704393e-05,-8.241946734477423320e-05,-8.239174936064408428e-05,-8.236405212136421738e-05,-8.233637565546736425e-05,-8.230871999104216741e-05,-8.228108515606624707e-05,-8.225347117851722345e-05,-8.222587808659476138e-05,-8.219830590860954800e-05,-8.217075467242818121e-05,-8.214322440591725893e-05,-8.211571513749849061e-05,-8.208822689503847414e-05,-8.206075970651482976e-05,-8.203331360023824459e-05,-8.200588860407531655e-05,-8.197848474578162126e-05,-8.195110205355682353e-05,-8.192374055582263281e-05,-8.189640027989053550e-05,-8.186908125373815182e-05,-8.184178350578719119e-05,-8.181450706401527384e-05,-8.178725195606695308e-05,-8.176001820980882684e-05,-8.173280585366260453e-05,-8.170561491482875027e-05,-8.167844542150692888e-05,-8.165129740222987209e-05,-8.162417088430906631e-05,-8.159706589550008715e-05,-8.156998246378055484e-05,-8.154292061712808959e-05,-8.151588038352031163e-05,-8.148886179071279656e-05,-8.146186486657214232e-05,-8.143488963885392451e-05,-8.140793613531371875e-05,-8.138100438404016757e-05,-8.135409441256680196e-05,-8.132720624876021986e-05,-8.130033992059804149e-05,-8.127349545594686475e-05,-8.124667288222919836e-05,-8.121987222742266255e-05,-8.119309351939385522e-05,-8.116633678545426278e-05,-8.113960205380355006e-05,-8.111288935208627038e-05,-8.108619870816902164e-05,-8.105953014936329026e-05,-8.103288370386874107e-05,-8.100625939932992736e-05,-8.097965726283629095e-05,-8.095307732280954127e-05,-8.092651960689423163e-05,-8.089998414262389304e-05,-8.087347095731001190e-05,-8.084698007915225304e-05,-8.082051153579516978e-05,-8.079406535455024851e-05,-8.076764156328408717e-05,-8.074124018975226136e-05,-8.071486126148830209e-05,-8.068850480580369577e-05,-8.066217085100912954e-05,-8.063585942441608978e-05,-8.060957055333606291e-05,-8.058330426563564686e-05,-8.055706058907041722e-05,-8.053083955139594963e-05,-8.050464117970168587e-05,-8.047846550163217927e-05,-8.045231254505402774e-05,-8.042618233750076229e-05,-8.040007490639489163e-05,-8.037399027926994677e-05,-8.034792848421457023e-05,-8.032188954842922612e-05,-8.029587349900335624e-05,-8.026988036413662542e-05,-8.024391017147358696e-05,-8.021796294799266036e-05,-8.019203872133839894e-05,-8.016613751948842292e-05,-8.014025936975421871e-05,-8.011440429944727271e-05,-8.008857233632316053e-05,-8.006276350791541319e-05,-8.003697784131347248e-05,-8.001121536438393633e-05,-7.998547610510442496e-05,-7.995976009034233556e-05,-7.993406734752017684e-05,-7.990839790461556902e-05,-7.988275178893999851e-05,-7.985712902780495170e-05,-7.983152964874395963e-05,-7.980595367962362019e-05,-7.978040114742235289e-05,-7.975487207989573335e-05,-7.972936650468831488e-05,-7.970388444877851697e-05,-7.967842594014395985e-05,-7.965299100609612992e-05,-7.962757967372446899e-05,-7.960219197100659727e-05,-7.957682792569809038e-05,-7.955148756455532322e-05,-7.952617091511182679e-05,-7.950087800512317671e-05,-7.947560886201188168e-05,-7.945036351342249503e-05,-7.942514198655548086e-05,-7.939994430883334786e-05,-7.937477050767860476e-05,-7.934962061084682716e-05,-7.932449464587154608e-05,-7.929939263962015872e-05,-7.927431462018130759e-05,-7.924926061486647910e-05,-7.922423065087613736e-05,-7.919922475585483568e-05,-7.917424295722508276e-05,-7.914928528263143193e-05,-7.912435175949639188e-05,-7.909944241513144902e-05,-7.907455727684808977e-05,-7.904969637262393434e-05,-7.902485972977046913e-05,-7.900004737537713595e-05,-7.897525933731053271e-05,-7.895049564299316813e-05,-7.892575631984755091e-05,-7.890104139540721206e-05,-7.887635089709466030e-05,-7.885168485244342662e-05,-7.882704328898704205e-05,-7.880242623403699298e-05,-7.877783371512681043e-05,-7.875326576023411462e-05,-7.872872239644834735e-05,-7.870420365119201733e-05,-7.867970955210967787e-05,-7.865524012651281538e-05,-7.863079540249007238e-05,-7.860637540735293527e-05,-7.858198016796880125e-05,-7.855760971275937976e-05,-7.853326406903615720e-05,-7.850894326399959766e-05,-7.848464732540527677e-05,-7.846037628067570324e-05,-7.843613015801054189e-05,-7.841190898438821222e-05,-7.838771278723122293e-05,-7.836354159440617195e-05,-7.833939543355761259e-05,-7.831527433244112046e-05,-7.829117831814613737e-05,-7.826710741865028353e-05,-7.824306166126504536e-05,-7.821904107407906537e-05,-7.819504568451485227e-05,-7.817107551977287017e-05,-7.814713060838585079e-05,-7.812321097722119134e-05,-7.809931665414548974e-05,-7.807544766691432159e-05,-7.805160404339428482e-05,-7.802778581122993273e-05,-7.800399299762172944e-05,-7.798022563054729517e-05,-7.795648373776220552e-05,-7.793276734746612533e-05,-7.790907648697054100e-05,-7.788541118358693893e-05,-7.786177146562600626e-05,-7.783815736062127399e-05,-7.781456889655036235e-05,-7.779100610116884695e-05,-7.776746900223230341e-05,-7.774395762738528504e-05,-7.772047200416132284e-05,-7.769701216142621547e-05,-7.767357812649144932e-05,-7.765016992655748851e-05,-7.762678759015706476e-05,-7.760343114526779829e-05,-7.758010061964526471e-05,-7.755679604115606196e-05,-7.753351743744474334e-05,-7.751026483671097367e-05,-7.748703826715441778e-05,-7.746383775653065129e-05,-7.744066333259524981e-05,-7.741751502321481127e-05,-7.739439285703308968e-05,-7.737129686180566068e-05,-7.734822706517707758e-05,-7.732518349568007210e-05,-7.730216618118124217e-05,-7.727917514965820800e-05,-7.725621042942165673e-05,-7.723327204844920857e-05,-7.721036003482950605e-05,-7.718747441698425860e-05,-7.716461522289108643e-05,-7.714178248041658748e-05,-7.711897621820451576e-05,-7.709619646434351381e-05,-7.707344324703324645e-05,-7.705071659458440081e-05,-7.702801653519664171e-05,-7.700534309706963398e-05,-7.698269630862508706e-05,-7.696007619817368806e-05,-7.693748279413714641e-05,-7.691491612493717156e-05,-7.689237621855138372e-05,-7.686986310373455922e-05,-7.684737680890840750e-05,-7.682491736238361568e-05,-7.680248479258189320e-05,-7.678007912825801640e-05,-7.675770039772267239e-05,-7.673534862950859292e-05,-7.671302385214850972e-05,-7.669072609428617682e-05,-7.666845538478739286e-05,-7.664621175151875576e-05,-7.662399522379015337e-05,-7.660180583013431743e-05,-7.657964359897295736e-05,-7.655750855939391641e-05,-7.653540073981890401e-05,-7.651332016933576341e-05,-7.649126687658824864e-05,-7.646924088999806912e-05,-7.644724223887511272e-05,-7.642527095197415576e-05,-7.640332705816099690e-05,-7.638141058641245706e-05,-7.635952156548331260e-05,-7.633766002457242905e-05,-7.631582599243458276e-05,-7.629401949837966157e-05,-7.627224057127346413e-05,-7.625048923998178907e-05,-7.622876553392554655e-05,-7.620706948230360211e-05,-7.618540111409277671e-05,-7.616376045826989127e-05,-7.614214754436687826e-05,-7.612056240191567014e-05,-7.609900505956002092e-05,-7.607747554683186308e-05,-7.605597389315210677e-05,-7.603450012783063983e-05,-7.601305428051041702e-05,-7.599163638027928158e-05,-7.597024645678018828e-05,-7.594888453932302497e-05,-7.592755065777279100e-05,-7.590624484155039653e-05,-7.588496711985470711e-05,-7.586371752299481130e-05,-7.584249608016957467e-05,-7.582130282124399656e-05,-7.580013777597205404e-05,-7.577900097377465727e-05,-7.575789244496089481e-05,-7.573681221906269911e-05,-7.571576032583404725e-05,-7.569473679569505009e-05,-7.567374165817764009e-05,-7.565277494314681661e-05,-7.563183668091166822e-05,-7.561092690155923890e-05,-7.559004563495452800e-05,-7.556919291107355718e-05,-7.554836876055848194e-05,-7.552757321338532392e-05,-7.550680629964112711e-05,-7.548606804985702468e-05,-7.546535849434210519e-05,-7.544467766340545722e-05,-7.542402558713412475e-05,-7.540340229628128554e-05,-7.538280782148909509e-05,-7.536224219306664196e-05,-7.534170544154505933e-05,-7.532119759745548038e-05,-7.530071869199517209e-05,-7.528026875525117845e-05,-7.525984781808769952e-05,-7.523945591125791310e-05,-7.521909306573704157e-05,-7.519875931272235192e-05,-7.517845468252293273e-05,-7.515817920622502868e-05,-7.513793291491488446e-05,-7.511771583990078938e-05,-7.509752801182489890e-05,-7.507736946199550232e-05,-7.505724022194293354e-05,-7.503714032264241496e-05,-7.501706979529121355e-05,-7.499702867086455171e-05,-7.497701698144787485e-05,-7.495703475834947227e-05,-7.493708203265558865e-05,-7.491715883634064710e-05,-7.489726520082395922e-05,-7.487740115752483661e-05,-7.485756673819565776e-05,-7.483776197469982350e-05,-7.481798689867869001e-05,-7.479824154221770272e-05,-7.477852593695821781e-05,-7.475884011509670302e-05,-7.473918410827451453e-05,-7.471955794863260891e-05,-7.469996166875603194e-05,-7.468039530017511751e-05,-7.466085887503082219e-05,-7.464135242618574750e-05,-7.462187598583636117e-05,-7.460242958584606399e-05,-7.458301325907745749e-05,-7.456362703800456515e-05,-7.454427095493487698e-05,-7.452494504278650567e-05,-7.450564933414449698e-05,-7.448638386137185208e-05,-7.446714865777526171e-05,-7.444794375599528280e-05,-7.442876918856144997e-05,-7.440962498900249855e-05,-7.439051119034756354e-05,-7.437142782512617956e-05,-7.435237492658952618e-05,-7.433335252859940567e-05,-7.431436066390739725e-05,-7.429539936576468051e-05,-7.427646866781101309e-05,-7.425756860329757458e-05,-7.423869920591963378e-05,-7.421986050920592604e-05,-7.420105254690723129e-05,-7.418227535260779604e-05,-7.416352896022493368e-05,-7.414481340373146878e-05,-7.412612871660062552e-05,-7.410747493280522846e-05,-7.408885208709525827e-05,-7.407026021299945029e-05,
        -7.405169934499022943e-05,-7.403316951726246486e-05,-7.401467076384449228e-05,-7.399620311981935927e-05,-7.397776661904886808e-05,-7.395936129622748822e-05,-7.394098718627173383e-05,-7.392264432348749637e-05,-7.390433274268026764e-05,-7.388605247926616215e-05,-7.386780356749556020e-05,-7.384958604239599822e-05,-7.383139993943910184e-05,-7.381324529320831829e-05,-7.379512213950834010e-05,-7.377703051347772600e-05,-7.375897045031054589e-05,-7.374094198553393653e-05,-7.372294515456401243e-05,-7.370497999303893266e-05,-7.368704653687441208e-05,-7.366914482176412093e-05,-7.365127488329070715e-05,-7.363343675742539673e-05,-7.361563048041697144e-05,-7.359785608840319071e-05,-7.358011361729976940e-05,-7.356240310341100042e-05,-7.354472458309668781e-05,-7.352707809282765794e-05,-7.350946366907473717e-05,-7.349188134847528531e-05,-7.347433116761115102e-05,-7.345681316306418296e-05,-7.343932737213787476e-05,-7.342187383180265314e-05,-7.340445257852934446e-05,-7.338706364989899811e-05,-7.336970708299306310e-05,-7.335238291478196615e-05,-7.333509118306880126e-05,-7.331783192543461780e-05,-7.330060517912739826e-05,-7.328341098172819201e-05,-7.326624937120662651e-05,-7.324912038531028458e-05,-7.323202406206430481e-05,-7.321496043954933697e-05,-7.319792955573500848e-05,-7.318093144892401369e-05,-7.316396615730802466e-05,-7.314703371963382494e-05,-7.313013417414859774e-05,-7.311326755948810430e-05,-7.309643391451015049e-05,-7.307963327785049756e-05,-7.306286568831144024e-05,-7.304613118525038473e-05,-7.302942980769167036e-05,-7.301276159488168105e-05,-7.299612658573373380e-05,-7.297952481988279061e-05,-7.296295633740790265e-05,-7.294642117716687579e-05,-7.292991937918325007e-05,-7.291345098331403207e-05,-7.289701602947173953e-05,-7.288061455801297939e-05,-7.286424660885026938e-05,-7.284791222184061610e-05,-7.283161143784022684e-05,-7.281534429753877546e-05,-7.279911084112633546e-05,-7.278291110940360298e-05,-7.276674514328229648e-05,-7.275061298345208982e-05,-7.273451467143532412e-05,-7.271845024825474013e-05,-7.270241975465552287e-05,-7.268642323249308035e-05,-7.267046072351179831e-05,-7.265453226862339520e-05,-7.263863791007185711e-05,-7.262277768999014782e-05,-7.260695164967856385e-05,-7.259115983171415820e-05,-7.257540227822989465e-05,-7.255967903130322583e-05,-7.254399013340018243e-05,-7.252833562759741781e-05,-7.251271555641647382e-05,-7.249712996226786998e-05,-7.248157888861683773e-05,-7.246606237826247465e-05,-7.245058047439245641e-05,-7.243513322058303672e-05,-7.241972066024393584e-05,-7.240434283672936289e-05,-7.238899979394863848e-05,-7.237369157575557210e-05,-7.235841822617050667e-05,-7.234317978899174051e-05,-7.232797630885023921e-05,-7.231280783021043490e-05,-7.229767439698164821e-05,-7.228257605434995625e-05,-7.226751284694632460e-05,-7.225248481979029691e-05,-7.223749201773488338e-05,-7.222253448596616110e-05,-7.220761226994776294e-05,-7.219272541519883291e-05,-7.217787396757158191e-05,-7.216305797203004246e-05,-7.214827747503704813e-05,-7.213353252294441020e-05,-7.211882316127127268e-05,-7.210414943675802490e-05,-7.208951139564545585e-05,-7.207490908445191025e-05,-7.206034255047288895e-05,-7.204581184006020322e-05,-7.203131700084242084e-05,-7.201685807961544228e-05,-7.200243512361925724e-05,-7.198804818103754499e-05,-7.197369729894376178e-05,-7.195938252507749766e-05,-7.194510390773345421e-05,-7.193086149537286644e-05,-7.191665533562430213e-05,-7.190248547717104088e-05,-7.188835196869636235e-05,-7.187425485888354615e-05,-7.186019419680444997e-05,-7.184617003158644266e-05,-7.183218241207933730e-05,-7.181823138796561423e-05,-7.180431700903877612e-05,-7.179043932431516950e-05,-7.177659838458749775e-05,-7.176279423959375237e-05,-7.174902693929396946e-05,-7.173529653442534126e-05,-7.172160307516994848e-05,-7.170794661276458370e-05,-7.169432719789092801e-05,-7.168074488156372936e-05,-7.166719971535284728e-05,-7.165369175038405203e-05,-7.164022103844924771e-05,-7.162678763100727153e-05,-7.161339158018309448e-05,-7.160003293860128792e-05,-7.158671175799824482e-05,-7.157342809122058114e-05,-7.156018199083735709e-05,-7.154697350930661059e-05,-7.153380270041864719e-05,-7.152066961713110516e-05,-7.150757431279020082e-05,-7.149451684096419513e-05,-7.148149725516583786e-05,-7.146851561012912413e-05,-7.145557195986640409e-05,-7.144266635811247212e-05,-7.142979885982336796e-05,-7.141696952006615362e-05,-7.140417839340829076e-05,-7.139142553497235255e-05,-7.137871100021397908e-05,-7.136603484492187732e-05,-7.135339712444066507e-05,-7.134079789455904930e-05,-7.132823721217596002e-05,-7.131571513324663769e-05,-7.130323171394836734e-05,-7.129078701145763475e-05,-7.127838108261785877e-05,-7.126601398466103632e-05,-7.125368577509672008e-05,-7.124139651115690697e-05,-7.122914625079523887e-05,-7.121693505218740228e-05,-7.120476297323152792e-05,-7.119263007260290266e-05,-7.118053640908783564e-05,-7.116848204119508026e-05,-7.115646702826605718e-05,-7.114449142958667593e-05,-7.113255530460937948e-05,-7.112065871317518884e-05,-7.110880171512512504e-05,-7.109698437079980948e-05,-7.108520674048435239e-05,-7.107346888501897553e-05,-7.106177086541043408e-05,-7.105011274227690521e-05,-7.103849457723576677e-05,-7.102691643201541893e-05,-7.101537836834426187e-05,-7.100388044828376266e-05,-7.099242273378436607e-05,-7.098100528774020646e-05,-7.096962817304541815e-05,-7.095829145237209090e-05,-7.094699518900293711e-05,-7.093573944660924724e-05,-7.092452428864026714e-05,-7.091334977932239880e-05,-7.090221598304857764e-05,-7.089112296382316103e-05,-7.088007078664970706e-05,-7.086905951664279613e-05,-7.085808921897251977e-05,-7.084715995880896955e-05,-7.083627180209939311e-05,-7.082542481517961619e-05,-7.081461906388586414e-05,-7.080385461472049613e-05,-7.079313153496302746e-05,-7.078244989128235076e-05,-7.077180975090247017e-05,-7.076121118199107940e-05,-7.075065425166116029e-05,-7.074013902830245115e-05,-7.072966558069326837e-05,-7.071923397700130565e-05,-7.070884428655999088e-05,-7.069849657842519619e-05,-7.068819092193034948e-05,-7.067792738718603474e-05,-7.066770604408079137e-05,-7.065752696300275915e-05,-7.064739021467314473e-05,-7.063729586970213248e-05,-7.062724399981012979e-05,-7.061723467621794370e-05,-7.060726797031291468e-05,-7.059734395487016201e-05,-7.058746270216520458e-05,-7.057762428452907244e-05,-7.056782877506995177e-05,-7.055807624706256220e-05,-7.054836677428122371e-05,-7.053870043038923399e-05,-7.052907728966051337e-05,-7.051949742675756028e-05,-7.050996091606531735e-05,-7.050046783302343911e-05,-7.049101825312709124e-05,-7.048161225203797287e-05,-7.047224990558431656e-05,-7.046293129020497759e-05,-7.045365648306045614e-05,-7.044442556075614092e-05,-7.043523860078559906e-05,-7.042609568069790882e-05,-7.041699687848623768e-05,-7.040794227269886463e-05,-7.039893194171753521e-05,-7.038996596492319568e-05,-7.038104442153025886e-05,-7.037216739086415984e-05,-7.036333495302748986e-05,-7.035454718873346280e-05,-7.034580417863978141e-05,-7.033710600357068188e-05,-7.032845274473897845e-05,-7.031984448430117496e-05,-7.031128130413621946e-05,-7.030276328684470499e-05,-7.029429051513824689e-05,-7.028586307189499394e-05,-7.027748104126985140e-05,-7.026914450652954613e-05,-7.026085355216205031e-05,-7.025260826309942530e-05,-7.024440872388515444e-05,-7.023625501995089948e-05,-7.022814723733894482e-05,-7.022008546186953026e-05,-7.021206978002902943e-05,-7.020410027891443860e-05,-7.019617704562275406e-05,-7.018830016758403900e-05,-7.018046973300551272e-05,-7.017268583076052835e-05,-7.016494854911181633e-05,-7.015725797721028556e-05,-7.014961420487297872e-05,-7.014201732197244965e-05,-7.013446741932494177e-05,-7.012696458702505353e-05,-7.011950891649965101e-05,-7.011210050011928985e-05,-7.010473942903328037e-05,-7.009742579583422284e-05,-7.009015969361431786e-05,-7.008294121568781065e-05,-7.007577045553547990e-05,-7.006864750724872692e-05,-7.006157246569610919e-05,-7.005454542563516185e-05,-7.004756648243404271e-05,-7.004063573207153226e-05,-7.003375327080396673e-05,-7.002691919522074926e-05,-7.002013360268843911e-05,-7.001339659068461785e-05,-7.000670825751953430e-05,-7.000006870122588154e-05,-6.999347802111310912e-05,-6.998693631682373351e-05,-6.998044368766720424e-05,-6.997400023450728312e-05,-6.996760605770813157e-05,-6.996126125863311174e-05,-6.995496593947825303e-05,-6.994872020193998452e-05,-6.994252414876944712e-05,-6.993637788349493789e-05,-6.993028150981128732e-05,-6.992423513135781477e-05,-6.991823885271752914e-05,-6.991229277969468470e-05,-6.990639701753842417e-05,-6.990055167210851295e-05,-6.989475685048596176e-05,-6.988901265936320328e-05,-6.988331920654289320e-05,-6.987767660060484332e-05,-6.987208494929619818e-05,-6.986654436230699261e-05,-6.986105494960481721e-05,-6.985561682093521796e-05,-6.985023008693191926e-05,-6.984489485928335739e-05,-6.983961124956694633e-05,-6.983437937002623386e-05,-6.982919933384845734e-05,-6.982407125427636529e-05,-6.981899524510781774e-05,-6.981397142097334196e-05,-6.980899989683653217e-05,-6.980408078860467214e-05,-6.979921421246260138e-05,-6.979440028470618174e-05,-6.978963912296354266e-05,-6.978493084502934707e-05,-6.978027556958643629e-05,-6.977567341542867396e-05,-6.977112450218259099e-05,-6.976662895030738554e-05,-6.976218688020674463e-05,-6.975779841339457832e-05,-6.975346367227297506e-05,-6.974918277891095642e-05,-6.974495585648776697e-05,-6.974078302918185202e-05,-6.973666442122716802e-05,-6.973260015752380525e-05,-6.972859036408207700e-05,-6.972463516680127427e-05,-6.972073469252437761e-05,-6.971688906931561291e-05,-6.971309842501716147e-05,-6.970936288830387184e-05,-6.970568258879428214e-05,-6.970205765655101970e-05,-6.969848822241386799e-05,-6.969497441777772195e-05,-6.969151637453707693e-05,-6.968811422586318471e-05,-6.968476810498280827e-05,-6.968147814584435551e-05,-6.967824448328441278e-05,-6.967506725280570024e-05,-6.967194659041053839e-05,-6.966888263315595964e-05,-6.966587551865410788e-05,-6.966292538512774968e-05,-6.966003237113271851e-05,-6.965719661644609317e-05,
        -6.965441826189966434e-05,-6.965169744832522269e-05,-6.964903431760927077e-05,-6.964642901186035573e-05,-6.964388167474133695e-05,-6.964139245058120764e-05,-6.963896148381998330e-05,-6.963658891984136901e-05,-6.963427490502827055e-05,-6.963201958670728331e-05,-6.962982311259358070e-05,-6.962768563112398112e-05,-6.962560729151245908e-05,-6.962358824408321212e-05,-6.962162864004861618e-05,-6.961972863062104722e-05,-6.961788836862270458e-05,-6.961610800726436565e-05,-6.961438770081151972e-05,-6.961272760419578987e-05,-6.961112787284839953e-05,-6.960958866375488441e-05,-6.960811013395629132e-05,-6.960669244215900164e-05,-6.960533574723593020e-05,-6.960404020900368138e-05,-6.960280598866663837e-05,-6.960163324726265088e-05,-6.960052214760592548e-05,-6.959947285334333600e-05,-6.959848552845482317e-05,-6.959756033819708421e-05,-6.959669744860397245e-05,-6.959589702643098619e-05,-6.959515923976589136e-05,-6.959448425747361000e-05,-6.959387224891866452e-05,-6.959332338463131151e-05,-6.959283783636305287e-05,-6.959241577647601318e-05,-6.959205737838702888e-05,-6.959176281656764829e-05,-6.959153226593350894e-05,-6.959136590289904944e-05,-6.959126390476688684e-05,-6.959122644967230542e-05,-6.959125371674979021e-05,-6.959134588624404927e-05,-6.959150313923245790e-05,-6.959172565784710329e-05,-6.959201362544131797e-05,-6.959236722597905711e-05,-6.959278664459001007e-05,-6.959327206762511153e-05,-6.959382368249000805e-05,-6.959444167725648001e-05,-6.959512624155061999e-05,-6.959587756572016559e-05,-6.959669584089001049e-05,-6.959758126012793866e-05,-6.959853401711235676e-05,-6.959955430602127180e-05,-6.960064232308660337e-05,-6.960179826559498295e-05,-6.960302233111059778e-05,-6.960431471864092501e-05,-6.960567562885877635e-05,-6.960710526360269768e-05,-6.960860382487776832e-05,-6.961017151663195790e-05,-6.961180854392345907e-05,-6.961351511269864289e-05,-6.961529143034717038e-05,-6.961713770525790324e-05,-6.961905414715197082e-05,-6.962104096680521437e-05,-6.962309837649227617e-05,-6.962522658954251042e-05,-6.962742582072856123e-05,-6.962969628537818423e-05,-6.963203820059549187e-05,-6.963445178526095347e-05,-6.963693725842157178e-05,-6.963949484090070641e-05,-6.964212475535358493e-05,-6.964482722521259106e-05,-6.964760247490930922e-05,-6.965045073054065838e-05,-6.965337221986889205e-05,-6.965636717176648673e-05,-6.965943581588307509e-05,-6.966257838386669121e-05,-6.966579510891968141e-05,-6.966908622496603698e-05,-6.967245196781712835e-05,-6.967589257428352667e-05,-6.967940828306318224e-05,-6.968299933407529068e-05,-6.968666596834927063e-05,-6.969040842885743103e-05,-6.969422695973781501e-05,-6.969812180707135596e-05,-6.970209321738307651e-05,-6.970614143969600107e-05,-6.971026672453195516e-05,-6.971446932269032004e-05,-6.971874948813461259e-05,-6.972310747543897236e-05,-6.972754354073185112e-05,-6.973205794225112442e-05,-6.973665093901182388e-05,-6.974132279230493836e-05,-6.974607376475372433e-05,-6.975090412036921705e-05,-6.975581412527187553e-05,-6.976080404702544868e-05,-6.976587415463697539e-05,-6.977102471894536251e-05,-6.977625601251036258e-05,-6.978156830950155154e-05,-6.978696188569832870e-05,-6.979243701887849483e-05,-6.979799398848518521e-05,-6.980363307546033624e-05,-6.980935456274428574e-05,-6.981515873472066147e-05,-6.982104587788251493e-05,-6.982701628110987713e-05,-6.983307023372686828e-05,-6.983920802738907696e-05,-6.984542995647213814e-05,-6.985173631629537638e-05,-6.985812740428753997e-05,-6.986460351993128981e-05,-6.987116496437462132e-05,-6.987781204065290908e-05,-6.988454505407748485e-05,-6.989136431184705955e-05,-6.989827012277016749e-05,-6.990526279804232246e-05,-6.991234265052437280e-05,-6.991950999524210175e-05,-6.992676514977480551e-05,-6.993410843275649214e-05,-6.994154016554121611e-05,-6.994906067159245566e-05,-6.995667027615004585e-05,-6.996436930700733470e-05,-6.997215809345647131e-05,-6.998003696773169580e-05,-6.998800626384360513e-05,-6.999606631752364194e-05,-7.000421746761187336e-05,-7.001246005483574564e-05,-7.002079442197661763e-05,-7.002922091414731653e-05,-7.003773987901418252e-05,-7.004635166640849064e-05,-7.005505662827093971e-05,-7.006385511904023033e-05,-7.007274749548653148e-05,-7.008173411737761427e-05,-7.009081534609107322e-05,-7.009999154539148236e-05,-7.010926308204101787e-05,-7.011863032524434658e-05,-7.012809364631555908e-05,-7.013765341945532583e-05,-7.014731002091822987e-05,-7.015706382984543410e-05,-7.016691522843121476e-05,-7.017686460047967145e-05,-7.018691233340312863e-05,-7.019705881633475641e-05,-7.020730444196043862e-05,-7.021764960524201626e-05,-7.022809470330626525e-05,-7.023864013705471976e-05,-7.024928631005344926e-05,-7.026003362814448039e-05,-7.027088249977886392e-05,-7.028183333712689773e-05,-7.029288655474585923e-05,-7.030404256974653876e-05,-7.031530180245937345e-05,-7.032666467693404755e-05,-7.033813161944069137e-05,-7.034970305858090356e-05,-7.036137942723064143e-05,-7.037316116093039753e-05,-7.038504869810724429e-05,-7.039704248046341206e-05,-7.040914295264322220e-05,-7.042135056284370975e-05,-7.043366576209297847e-05,-7.044608900480531233e-05,-7.045862074883668669e-05,-7.047126145498516792e-05,-7.048401158776806952e-05,-7.049687161475581831e-05,-7.050984200684951020e-05,-7.052292323866948820e-05,-7.053611578811125327e-05,-7.054942013684506463e-05,-7.056283676926122794e-05,-7.057636617424645209e-05,-7.059000884401811504e-05,-7.060376527362466348e-05,-7.061763596283299194e-05,-7.063162141490719748e-05,-7.064572213616449048e-05,-7.065993863719643997e-05,-7.067427143242488441e-05,-7.068872103999090939e-05,-7.070328798186586994e-05,-7.071797278362934591e-05,-7.073277597552385387e-05,-7.074769809167769097e-05,-7.076273966916124536e-05,-7.077790125026295343e-05,-7.079318338137907674e-05,-7.080858661184796787e-05,-7.082411149650358340e-05,-7.083975859395463814e-05,-7.085552846691767215e-05,-7.087142168255011754e-05,-7.088743881195069818e-05,-7.090358043149169731e-05,-7.091984712148668990e-05,-7.093623946607952035e-05,-7.095275805485412590e-05,-7.096940348216840277e-05,-7.098617634615500549e-05,-7.100307724966503642e-05,-7.102010680087866845e-05,-7.103726561225043312e-05,-7.105455430145291018e-05,-7.107197349048854917e-05,-7.108952380635580326e-05,-7.110720588165975187e-05,-7.112502035316881077e-05,-7.114296786303597742e-05,-7.116104905890985322e-05,-7.117926459304646514e-05,-7.119761512303091067e-05,-7.121610131166633550e-05,-7.123472382719597817e-05,-7.125348334358072577e-05,-7.127238053988849131e-05,-7.129141610046074717e-05,-7.131059071513456971e-05,-7.132990507985326190e-05,-7.134935989638879761e-05,-7.136895587134262087e-05,-7.138869371769995809e-05,-7.140857415444124001e-05,-7.142859790565392331e-05,-7.144876570247538083e-05,-7.146907828148307829e-05,-7.148953638519417453e-05,-7.151014076278716658e-05,-7.153089216921371118e-05,-7.155179136608680324e-05,-7.157283912151424232e-05,-7.159403620937698776e-05,-7.161538341010631470e-05,-7.163688151184954833e-05,-7.165853130852717356e-05,-7.168033360033243540e-05,-7.170228919523014000e-05,-7.172439890767989823e-05,-7.174666355880265911e-05,-7.176908397715786592e-05,-7.179166099863243389e-05,-7.181439546549706066e-05,-7.183728822768298272e-05,-7.186034014272646431e-05,-7.188355207521368584e-05,-7.190692489761341122e-05,-7.193045948983289861e-05,-7.195415673943994506e-05,-7.197801754155186416e-05,-7.200204279950161990e-05,-7.202623342439373744e-05,-7.205059033571492577e-05,-7.207511446077896622e-05,-7.209980673467120127e-05,-7.212466810196938027e-05,-7.214969951468974685e-05,-7.217490193356379535e-05,-7.220027632848236010e-05,-7.222582367732988118e-05,-7.225154496720564978e-05,-7.227744119397971900e-05,-7.230351336279250418e-05,-7.232976248783273832e-05,-7.235618959250400550e-05,-7.238279570953576325e-05,-7.240958188109436477e-05,-7.243654915900510360e-05,-7.246369860486323589e-05,-7.249103128981193578e-05,-7.251854829520842927e-05,-7.254625071251297186e-05,-7.257413964323333744e-05,-7.260221619920237401e-05,-7.263048150235595912e-05,-7.265893668573220054e-05,-7.268758289269428019e-05,-7.271642127781863252e-05,-7.274545300633983302e-05,-7.277467925415059824e-05,-7.280410120924507567e-05,-7.283372007066413190e-05,-7.286353704838433032e-05,-7.289355336481673220e-05,-7.292377025380769595e-05,-7.295418896108296636e-05,-7.298481074513585298e-05,-7.301563687572842909e-05,-7.304666863522379927e-05,-7.307790731908569981e-05,-7.310935423554543178e-05,-7.314101070515777181e-05,-7.317287806141159479e-05,-7.320495765189560800e-05,-7.323725083685506121e-05,-7.326975899102361467e-05,-7.330248350134738189e-05,-7.333542576959395376e-05,-7.336858721262995431e-05,-7.340196925964548313e-05,-7.343557335570682909e-05,-7.346940096025766920e-05,-7.350345354739662440e-05,-7.353773260643237109e-05,-7.357223964205017452e-05,-7.360697617431188888e-05,-7.364194373882249067e-05,-7.367714388750723487e-05,-7.371257818822307684e-05,-7.374824822514725042e-05,-7.378415559861073447e-05,-7.382030192631949816e-05,-7.385668884335450102e-05,-7.389331800078391410e-05,-7.393019106810561070e-05,-7.396730973263654363e-05,-7.400467569929070066e-05,-7.404229069113421602e-05,-7.408015645005150418e-05,-7.411827473674525990e-05,-7.415664733062543590e-05,-7.419527603058639897e-05,-7.423416265511795231e-05,-7.427330904230533548e-05,-7.431271705060638055e-05,-7.435238855912906786e-05,-7.439232546696539217e-05,-7.443252969463465263e-05,-7.447300318463856428e-05,-7.451374790012899041e-05,-7.455476582607367675e-05,-7.459605897069954139e-05,-7.463762936454898522e-05,-7.467947906059091423e-05,-7.472161013499789561e-05,-7.476402468853393657e-05,-7.480672484505568320e-05,-7.484971275290019932e-05,-7.489299058538456677e-05,-7.493656054052832971e-05,-7.498042484260780682e-05,-7.502458574054626794e-05,-7.506904551018989125e-05,-7.511380645419674096e-05,-7.515887090159267814e-05,-7.520424120971425097e-05,-7.524991976298744945e-05,-7.529590897492610679e-05,-7.534221128752127683e-05,-7.538882917140776740e-05,-7.543576512758498609e-05,-7.548302168691733982e-05,
        -7.553060141124445792e-05,-7.557850689315914749e-05,-7.562674075700659415e-05,-7.567530565921742891e-05,-7.572420428869630626e-05,-7.577343936771008259e-05,-7.582301365233190538e-05,-7.587292993294081356e-05,-7.592319103433275984e-05,-7.597379981683083372e-05,-7.602475917711792874e-05,-7.607607204790367561e-05,-7.612774139920119865e-05,-7.617977023938182768e-05,-7.623216161450896422e-05,-7.628491861011443831e-05,-7.633804435142055311e-05,-7.639154200378417414e-05,-7.644541477397348572e-05,-7.649966591083412482e-05,-7.655429870551122562e-05,-7.660931649172697533e-05,-7.666472264794554903e-05,-7.672052059737310969e-05,-7.677671380879047547e-05,-7.683330579721925346e-05,-7.689030012464348474e-05,-7.694770040139742306e-05,-7.700551028649860186e-05,-7.706373348864703487e-05,-7.712237376761299501e-05,-7.718143493418150314e-05,-7.724092085165112920e-05,-7.730083543738830443e-05,-7.736118266243874331e-05,-7.742196655335931155e-05,-7.748319119343927142e-05,-7.754486072347743786e-05,-7.760697934233729001e-05,-7.766955130900088378e-05,-7.773258094273538532e-05,-7.779607262492493902e-05,-7.786003080084702432e-05,-7.792445997867325502e-05,-7.798936473224493682e-05,-7.805474970318249106e-05,-7.812061960055238785e-05,-7.818697920242145827e-05,-7.825383335813285157e-05,-7.832118698863910211e-05,-7.838904508900013113e-05,-7.845741272860529136e-05,-7.852629505356034656e-05,-7.859569728829729485e-05,-7.866562473601845795e-05,-7.873608278180510567e-05,-7.880707689278398931e-05,-7.887861262140249963e-05,-7.895069560576173373e-05,-7.902333157178142997e-05,-7.909652633558694745e-05,-7.917028580434193330e-05,-7.924461597957899173e-05,-7.931952295797684016e-05,-7.939501293330319953e-05,-7.947109219913484068e-05,-7.954776715152211963e-05,-7.962504428932204448e-05,-7.970293021763996677e-05,-7.978143165104922829e-05,-7.986055541364667221e-05,-7.994030844327149055e-05,-8.002069779339260336e-05,-8.010173063571768282e-05,-8.018341426274666617e-05,-8.026575609043629100e-05,-8.034876366130871972e-05,-8.043244464694954132e-05,-8.051680685122741821e-05,-8.060185821268106565e-05,-8.068760680829401011e-05,-8.077406085682525827e-05,-8.086122872152934349e-05,-8.094911891315392793e-05,-8.103774009493580621e-05,-8.112710108454379565e-05,-8.121721085879718416e-05,-8.130807855760702196e-05,-8.139971348641861226e-05,-8.149212512159609290e-05,-8.158532311453026153e-05,-8.167931729524680051e-05,-8.177411767718023583e-05,-8.186973446211442962e-05,-8.196617804390182727e-05,-8.206345901379252794e-05,-8.216158816592988856e-05,-8.226057650162488244e-05,-8.236043523529579247e-05,-8.246117579935319242e-05,-8.256280985008412898e-05,-8.266534927420243761e-05,-8.276880619367821268e-05,-8.287319297267670137e-05,-8.297852222355350804e-05,-8.308480681390451039e-05,-8.319205987378230915e-05,-8.330029480141387666e-05,-8.340952527158274066e-05,-8.351976524312298977e-05,-8.363102896707941269e-05,-8.374333099342434750e-05,-8.385668618082764425e-05,-8.397110970531640461e-05,-8.408661706799103186e-05,-8.420322410612746111e-05,-8.432094700100423168e-05,-8.443980228950431766e-05,-8.455980687349651248e-05,-8.468097803060459228e-05,-8.480333342553159071e-05,-8.492689112216122993e-05,-8.505166959488219547e-05,-8.517768774052303371e-05,-8.530496489256300663e-05,-8.543352083351107851e-05,-8.556337580928330411e-05,-8.569455054335817223e-05,-8.582706625231972808e-05,-8.596094466067905060e-05,-8.609620801730555328e-05,-8.623287911335708600e-05,-8.637098129865572460e-05,-8.651053850000645085e-05,-8.665157524084809992e-05,-8.679411666068226339e-05,-8.693818853583445971e-05,-8.708381730088143868e-05,-8.723103007135524223e-05,-8.737985466672482104e-05,-8.753031963487645228e-05,-8.768245427792642488e-05,-8.783628867842230292e-05,-8.799185372709850128e-05,-8.814918115224168460e-05,-8.830830354961127782e-05,-8.846925441413633351e-05,-8.863206817361080070e-05,-8.879678022211123789e-05,-8.896342695774439591e-05,-8.913204582011724497e-05,-8.930267532947233633e-05,-8.947535512926485524e-05,-8.965012602979438583e-05,-8.982703005366854399e-05,-9.000611048431972350e-05,-9.018741191685331060e-05,-9.037098031045021074e-05,-9.055686304476617821e-05,-9.074510897816301380e-05,-9.093576851104678838e-05,-9.112889364831788797e-05,-9.132453806975915356e-05,-9.152275720247793345e-05,-9.172360829501347013e-05,-9.192715049971544872e-05,-9.213344495623276842e-05,-9.234255488144160751e-05,-9.255454566370335812e-05,-9.276948496378389919e-05,-9.298744281949211654e-05,-9.320849175808998410e-05,-9.343270691519744986e-05,-9.366016616074723800e-05,-9.389095023176752264e-05,-9.412514287471251961e-05,-9.436283099589770629e-05,-9.460410482081682559e-05,-9.484905806617094370e-05,-9.509778811939151311e-05,-9.535039623215224580e-05,-9.560698772759224084e-05,-9.586767221936298711e-05,-9.613256384660706644e-05,-9.640178152531264644e-05,-9.667544921726500817e-05,-9.695369621842697683e-05,-9.723665746735887794e-05,-9.752447387695317715e-05,-9.781729269003891503e-05,-9.811526786301927316e-05,-9.841856047865249479e-05,-9.872733918947496079e-05,-9.904178069730651401e-05,-9.936207027294585714e-05,-9.968840231522335671e-05,-1.000209809576269038e-04,-1.003600207281629686e-04,-1.007057472617312044e-04,-1.010583980783907876e-04,-1.014182234292948515e-04,-1.017854872180645742e-04,-1.021604680092602635e-04,-1.025434601291119741e-04,-1.029347748738862478e-04,-1.033347418340491153e-04,-1.037437103497129698e-04,-1.041620511166296836e-04,-1.045901579546648641e-04,-1.050284497651765570e-04,-1.054773726993918359e-04,-1.059374025646486750e-04,-1.064090475048073969e-04,-1.068928509872502097e-04,-1.073893951423765536e-04,-1.078993045091625191e-04,-1.084232502427950884e-04,-1.089619548583220521e-04,-1.095161975916969510e-04,-1.100868204778060466e-04,-1.106747352660475414e-04,-1.112809313116858156e-04,-1.119064846166750726e-04,-1.125525682256767190e-04,-1.132204642251832816e-04,-1.139115776571109073e-04,-1.146274527241697427e-04,-1.153697917521401273e-04,-1.161404774928098682e-04,-1.169415995025402388e-04,-1.177754855217760799e-04,-1.186447390379985478e-04,-1.195522845546803659e-04,-1.205014225438838515e-04,-1.214958966846424460e-04,-1.225399768333135242e-04,-1.236385623644697951e-04,-1.247973121972223431e-04,-1.260228102345495316e-04,-1.273227784669983542e-04,-1.287063552429801305e-04,-1.301844642134231123e-04,-1.317703119416702151e-04,-1.334800721635165388e-04,-1.353338477331766931e-04,-1.373570578522831376e-04,-1.395824989293692475e-04,-1.420535157107827295e-04,-1.448290915649352861e-04,-1.479924550093492108e-04,-1.516666174656844390e-04,-1.560449380977879663e-04,-1.614588029120733736e-04,-1.685565410893818949e-04,-1.792970627148341681e-04,-1.991855474436817744e-04,-2.704640693184753755e-04,-1.991950106244279252e-04,-1.792815538694218910e-04,-1.684550396846562492e-04,-1.613287110624472831e-04,-1.558874462317394460e-04,-1.514826583346451372e-04,-1.477827946992915642e-04,-1.445943791255066557e-04,-1.417943133655841237e-04,-1.392993018836108376e-04,-1.370503080002083429e-04,-1.350039435438210589e-04,-1.331273760530327088e-04,-1.313951559156478766e-04,-1.297871542825235203e-04,-1.282871749456182542e-04,-1.268819917208396930e-04,-1.255606636823558553e-04,-1.243140371576245684e-04,-1.231343764684922348e-04,-1.220150854160384490e-04,-1.209504939763128561e-04,-1.199356926946393287e-04,-1.189664025277870074e-04,-1.180388713873381512e-04,-1.171497910759655703e-04,-1.162962299719461079e-04,-1.154755780055638503e-04,-1.146855013333114570e-04,-1.139239047240336866e-04,-1.131889001378394433e-04,-1.124787803098992178e-04,-1.117919964114810760e-04,-1.111271390617507571e-04,-1.104829220968661652e-04,-1.098581686321264961e-04,-1.092517990435304398e-04,-1.086628205546613657e-04,-1.080903181786552203e-04,-1.075334468138011701e-04,-1.069914243138625487e-04,-1.064635253985035668e-04,-1.059490762839732092e-04,-1.054474499257440634e-04,-1.049580618062151416e-04,-1.044803661799931227e-04,-1.040138527254597101e-04,-1.035580435541083588e-04,-1.031124905210289988e-04,-1.026767728163346938e-04,-1.022504947898461580e-04,-1.018332839884950047e-04,-1.014247893853514881e-04,-1.010246797673586272e-04,-1.006326422791081754e-04,-1.002483810927934371e-04,-9.987161619712248140e-05,-9.950208229442258911e-05,-9.913952778894952189e-05,-9.878371386395912168e-05,-9.843441363099891817e-05,-9.809141135141974388e-05,-9.775450172172517327e-05,-9.742348921076837698e-05,-9.709818745301523890e-05,-9.677841868538372694e-05,-9.646401322754849517e-05,-9.615480900165840339e-05,-9.585065108586032778e-05,-9.555139130007495041e-05,-9.525688782247021535e-05,-9.496700483230258172e-05,-9.468161217623949355e-05,-9.440058505955084556e-05,-9.412380375672935173e-05,-9.385115334098470541e-05,-9.358252343294459763e-05,-9.331780796384414600e-05,-9.305690495670093654e-05,-9.279971631753758388e-05,-9.254614764131474658e-05,-9.229610803157539678e-05,-9.204950992808269561e-05,-9.180626894728094456e-05,-9.156630372991747535e-05,-9.132953579965574775e-05,-9.109588942857183014e-05,-9.086529151131061965e-05,-9.063767144568135592e-05,-9.041296101991447287e-05,-9.019109430646876646e-05,-8.997200756227785590e-05,-8.975563913360407042e-05,-8.954192936450056095e-05,-8.933082051365559551e-05,-8.912225667390138995e-05,-8.891618369416542933e-05,-8.871254910763903823e-05,-8.851130206405377621e-05,-8.831239326229090025e-05,-8.811577489020727683e-05,-8.792140056523845004e-05,-8.772922527683357785e-05,-8.753920533405290527e-05,-8.735129831588528404e-05,-8.716546302145466996e-05,-8.698165942366831160e-05,-8.679984862591805239e-05,-8.661999281983634447e-05,-8.644205524604986479e-05,-8.626600015482210893e-05,-8.609179277041523193e-05,-8.591939925584046733e-05,-8.574878667910734720e-05,-8.557992298258154662e-05,-8.541277695173210560e-05,-8.524731818521091853e-05,-8.508351706937311576e-05,-8.492134474907819808e-05,-8.476077310271001863e-05,-8.460177471858454368e-05,-8.444432287046943486e-05,-8.428839149565714450e-05,-8.413395517231636589e-05,-8.398098910000761919e-05,-8.382946907947719239e-05,-8.367937149267312691e-05,
        -8.353067328475960451e-05,-8.338335194724155741e-05,-8.323738549992354407e-05,-8.309275247420089272e-05,-8.294943189968151387e-05,-8.280740328731051036e-05,-8.266664661354949928e-05,-8.252714230899682590e-05,-8.238887124412119789e-05,-8.225181471516185283e-05,-8.211595443291530572e-05,-8.198127250969022839e-05,-8.184775144853828621e-05,-8.171537413176332976e-05,-8.158412380998569802e-05,-8.145398409115101046e-05,-8.132493893131531593e-05,-8.119697262598535303e-05,-8.107006979768405230e-05,-8.094421538956675377e-05,-8.081939465537368861e-05,-8.069559315165841795e-05,-8.057279672984973828e-05,-8.045099152664825226e-05,-8.033016395853076475e-05,-8.021030071320156551e-05,-8.009138874259802421e-05,-7.997341525622925218e-05,-7.985636771429271974e-05,-7.974023382156802953e-05,-7.962500152020046684e-05,-7.951065898437192914e-05,-7.939719461563798930e-05,-7.928459703432366723e-05,-7.917285507602622729e-05,-7.906195778661917473e-05,-7.895189441592398438e-05,-7.884265441326920865e-05,-7.873422742160629539e-05,-7.862660327484505274e-05,-7.851977199174742239e-05,-7.841372377154209872e-05,-7.830844899014977045e-05,-7.820393819568671745e-05,-7.810018210491209700e-05,-7.799717159800989563e-05,-7.789489771625746073e-05,-7.779335165758460846e-05,-7.769252477279886548e-05,-7.759240856269888909e-05,-7.749299467418868659e-05,-7.739427489711347974e-05,-7.729624116059596872e-05,-7.719888553081588611e-05,-7.710220020745728320e-05,-7.700617752032234975e-05,-7.691080992766607949e-05,-7.681609001197742259e-05,-7.672201047820292885e-05,-7.662856415063812321e-05,-7.653574397042950395e-05,-7.644354299407574160e-05,-7.635195438893127573e-05,-7.626097143237364762e-05,-7.617058750908345388e-05,-7.608079610932350079e-05,-7.599159082533057941e-05,-7.590296535014973145e-05,-7.581491347663504854e-05,-7.572742909289775781e-05,-7.564050618169559925e-05,-7.555413882004424764e-05,-7.546832117416579777e-05,-7.538304749948876449e-05,-7.529831213948234847e-05,-7.521410952204821143e-05,-7.513043415974252071e-05,-7.504728064638976903e-05,-7.496464365730481916e-05,-7.488251794579570131e-05,-7.480089834266401283e-05,-7.471977975509469516e-05,-7.463915716449109894e-05,-7.455902562514271636e-05,-7.447938026250433552e-05,-7.440021627280746230e-05,-7.432152892089538554e-05,-7.424331353927948740e-05,-7.416556552691799808e-05,-7.408828034760617243e-05,-7.401145352903260033e-05,-7.393508066189102834e-05,-7.385915739804849167e-05,-7.378367944910202425e-05,-7.370864258715581485e-05,-7.363404264271178334e-05,-7.355987550206055658e-05,-7.348613710800311338e-05,-7.341282345907362838e-05,-7.333993060804067099e-05,-7.326745465968675930e-05,-7.319539177186307199e-05,-7.312373815376860264e-05,-7.305249006478442553e-05,-7.298164381391858413e-05,-7.291119575836280120e-05,-7.284114230382554567e-05,-7.277147990242260889e-05,-7.270220505301017155e-05,-7.263331429907537995e-05,-7.256480422940247976e-05,-7.249667147668503731e-05,-7.242891271541651577e-05,-7.236152466433276587e-05,-7.229450408263726757e-05,-7.222784777066726392e-05,-7.216155256956069408e-05,-7.209561535931330312e-05,-7.203003305961130920e-05,-7.196480262788851334e-05,-7.189992105971487746e-05,-7.183538538774181248e-05,-7.177119268042542188e-05,-7.170734004285916896e-05,-7.164382461494200882e-05,-7.158064357215554452e-05,-7.151779412328806984e-05,-7.145527351071212507e-05,-7.139307901071756390e-05,-7.133120793162417428e-05,-7.126965761378167841e-05,-7.120842542907013240e-05,-7.114750878084441510e-05,-7.108690510299053855e-05,-7.102661185920400300e-05,-7.096662654298979689e-05,-7.090694667660768502e-05,-7.084756981168283119e-05,-7.078849352848415322e-05,-7.072971543392592153e-05,-7.067123316334411598e-05,-7.061304437883109131e-05,-7.055514676895802140e-05,-7.049753804844183236e-05,-7.044021595881133635e-05,-7.038317826574269631e-05,-7.032642276033618245e-05,-7.026994725900514993e-05,-7.021374960186621550e-05,-7.015782765334988014e-05,-7.010217930147888410e-05,-7.004680245747962886e-05,-6.999169505511604328e-05,-6.993685505163327321e-05,-6.988228042614785807e-05,-6.982796917925915281e-05,-6.977391933404852864e-05,-6.972012893463608307e-05,-6.966659604557001728e-05,-6.961331875326992602e-05,-6.956029516380635158e-05,-6.950752340356691761e-05,-6.945500161881223988e-05,-6.940272797556490403e-05,-6.935070065883230939e-05,-6.929891787327280284e-05,-6.924737784214096692e-05,-6.919607880606637451e-05,-6.914501902566261293e-05,-6.909419677902928214e-05,-6.904361036141892782e-05,-6.899325808645828673e-05,-6.894313828442744096e-05,-6.889324930287044069e-05,-6.884358950648428177e-05,-6.879415727606419395e-05,-6.874495100928079694e-05,-6.869596911979192200e-05,-6.864721003690954504e-05,-6.859867220576632008e-05,-6.855035408781517958e-05,-6.850225415899746650e-05,-6.845437091035355692e-05,-6.840670284818939351e-05,-6.835924849379892976e-05,-6.831200638224288468e-05,-6.826497506340345467e-05,-6.821815310170675772e-05,-6.817153907445749894e-05,-6.812513157428146116e-05,-6.807892920634994738e-05,-6.803293058910142577e-05,-6.798713435501868574e-05,-6.794153914946310380e-05,-6.789614363067464353e-05,-6.785094646932776641e-05,-6.780594634936409904e-05,-6.776114196682669899e-05,-6.771653203019312173e-05,-6.767211525987582021e-05,-6.762789038822214494e-05,-6.758385616029150000e-05,-6.754001133169040827e-05,-6.749635467001580125e-05,-6.745288495479950797e-05,-6.740960097617598734e-05,-6.736650153571499544e-05,-6.732358544586647398e-05,-6.728085153023810605e-05,-6.723829862304020466e-05,-6.719592556880815692e-05,-6.715373122279100215e-05,-6.711171445045183148e-05,-6.706987412813392169e-05,-6.702820914156193410e-05,-6.698671838645253729e-05,-6.694540076890298508e-05,-6.690425520428089357e-05,-6.686328061822344182e-05,-6.682247594519408196e-05,-6.678184012920418411e-05,-6.674137212409059217e-05,-6.670107089207233386e-05,-6.666093540491635494e-05,-6.662096464415956376e-05,-6.658115759866634065e-05,-6.654151326684898393e-05,-6.650203065627913190e-05,-6.646270878191140596e-05,-6.642354666874794589e-05,-6.638454334867427420e-05,-6.634569786223565302e-05,-6.630700925946975133e-05,-6.626847659679802049e-05,-6.623009893924614033e-05,-6.619187535999992988e-05,-6.615380493979472476e-05,-6.611588676747048865e-05,-6.607811993908363490e-05,-6.604050355846213805e-05,-6.600303673709451147e-05,-6.596571859385225167e-05,-6.592854825421268217e-05,-6.589152485136917647e-05,-6.585464752678626965e-05,-6.581791542709103382e-05,-6.578132770690414688e-05,-6.574488352784069178e-05,-6.570858205806606733e-05,-6.567242247296212199e-05,-6.563640395390590854e-05,-6.560052568965746289e-05,-6.556478687519406989e-05,-6.552918671165475217e-05,-6.549372440706191512e-05,-6.545839917587725765e-05,-6.542321023839114957e-05,-6.538815682144427655e-05,-6.535323815815008430e-05,-6.531845348695108910e-05,-6.528380205322870111e-05,-6.524928310830402367e-05,-6.521489590827211913e-05,-6.518063971605592144e-05,-6.514651380068459119e-05,-6.511251743573920336e-05,-6.507864990124012650e-05,-6.504491048281435539e-05,-6.501129847091835501e-05,-6.497781316244788385e-05,-6.494445385929470405e-05,-6.491121986884618167e-05,-6.487811050320813067e-05,-6.484512508020401356e-05,-6.481226292326391913e-05,-6.477952336031433944e-05,-6.474690572488839280e-05,-6.471440935496008962e-05,-6.468203359383251083e-05,-6.464977779013780790e-05,-6.461764129700453552e-05,-6.458562347200214049e-05,-6.455372367841771819e-05,-6.452194128375721149e-05,-6.449027566007847767e-05,-6.445872618449088876e-05,-6.442729223821164197e-05,-6.439597320784251622e-05,-6.436476848376004867e-05,-6.433367746100371320e-05,-6.430269953949796502e-05,-6.427183412260895068e-05,-6.424108061880984266e-05,-6.421043844095919439e-05,-6.417990700569031759e-05,-6.414948573374434915e-05,-6.411917405096945188e-05,-6.408897138665547999e-05,-6.405887717375602364e-05,-6.402889084999863201e-05,-6.399901185721867947e-05,-6.396923964108180982e-05,-6.393957365063984710e-05,-6.391001333949652974e-05,-6.388055816486382099e-05,-6.385120758778395356e-05,-6.382196107324045187e-05,-6.379281808982506519e-05,-6.376377811007083451e-05,-6.373484061000800338e-05,-6.370600506910850669e-05,-6.367727097100761569e-05,-6.364863780261575954e-05,-6.362010505395199189e-05,-6.359167221969830308e-05,-6.356333879731224101e-05,-6.353510428697140000e-05,-6.350696819373835567e-05,-6.347893002561777465e-05,-6.345098929294579193e-05,-6.342314551055494576e-05,-6.339539819633088769e-05,-6.336774687121238259e-05,-6.334019105957988671e-05,-6.331273028864492503e-05,-6.328536408939378077e-05,-6.325809199519971671e-05,-6.323091354326626501e-05,-6.320382827373904888e-05,-6.317683572953924909e-05,-6.314993545697422661e-05,-6.312312700507138885e-05,-6.309640992602227882e-05,-6.306978377468297481e-05,-6.304324810929573530e-05,-6.301680249093388753e-05,-6.299044648316876049e-05,-6.296417965284684115e-05,-6.293800156942364055e-05,-6.291191180490818269e-05,-6.288590993486220526e-05,-6.285999553667931394e-05,-6.283416819119480579e-05,-6.280842748201953540e-05,-6.278277299465173655e-05,-6.275720431769826746e-05,-6.273172104220847700e-05,-6.270632276283993889e-05,-6.268100907552698331e-05,-6.265577957914603147e-05,-6.263063387584866248e-05,-6.260557156911872312e-05,-6.258059226610379611e-05,-6.255569557589435448e-05,-6.253088110930171695e-05,-6.250614848085644937e-05,-6.248149730725405249e-05,-6.245692720707740619e-05,-6.243243780112983643e-05,-6.240802871315676015e-05,-6.238369956956812956e-05,-6.235944999810616451e-05,-6.233527962917762011e-05,-6.231118809579827555e-05,-6.228717503298231151e-05,-6.226324007768679891e-05,-6.223938286975538858e-05,-6.221560305091911047e-05,-6.219190026490739598e-05,-6.216827415794767830e-05,-6.214472437798823634e-05,-6.212125057558637309e-05,-6.209785240318677069e-05,-6.207452951517700157e-05,-6.205128156838712883e-05,-6.202810822136806124e-05,-6.200500913478013132e-05,-6.198198397161513995e-05,-6.195903239664124484e-05,-6.193615407629193825e-05,-6.191334867949871423e-05,-6.189061587730249059e-05,-6.186795534185440815e-05,-6.184536674774809839e-05,
        -6.182284977196417231e-05,-6.180040409298204196e-05,-6.177802939016929784e-05,-6.175572534616868836e-05,-6.173349164539931877e-05,-6.171132797333500619e-05,-6.168923401761450265e-05,-6.166720946770842815e-05,-6.164525401503029300e-05,-6.162336735276996436e-05,-6.160154917561611043e-05,-6.157979917992273400e-05,-6.155811706404223926e-05,-6.153650252832543188e-05,-6.151495527467742974e-05,-6.149347500583601800e-05,-6.147206142737005052e-05,-6.145071424618064881e-05,-6.142943317066773545e-05,-6.140821791100758986e-05,-6.138706817865324794e-05,-6.136598368727819164e-05,-6.134496415177714823e-05,-6.132400928871017953e-05,-6.130311881624717074e-05,-6.128229245377925238e-05,-6.126152992314004564e-05,-6.124083094705135011e-05,-6.122019524940069957e-05,-6.119962255668465190e-05,-6.117911259584385419e-05,-6.115866509609491075e-05,-6.113827978743158198e-05,-6.111795640190154089e-05,-6.109769467343983962e-05,-6.107749433575948572e-05,-6.105735512557108891e-05,-6.103727678086201536e-05,-6.101725904011923163e-05,-6.099730164416117262e-05,-6.097740433474996280e-05,-6.095756685525755003e-05,-6.093778895033263865e-05,-6.091807036578966716e-05,-6.089841084921943093e-05,-6.087881014943397062e-05,-6.085926801607799419e-05,-6.083978420085012218e-05,-6.082035845672573160e-05,-6.080099053712428869e-05,-6.078168019757468343e-05,-6.076242719482705112e-05,-6.074323128696379470e-05,-6.072409223267793976e-05,-6.070500979243886874e-05,-6.068598372799272056e-05,-6.066701380252892406e-05,-6.064809977990304191e-05,-6.062924142524739324e-05,-6.061043850563718749e-05,-6.059169078853621215e-05,-6.057299804318461156e-05,-6.055436003959968616e-05,-6.053577654868691482e-05,-6.051724734351671131e-05,-6.049877219765908976e-05,-6.048035088618286537e-05,-6.046198318482298717e-05,-6.044366887031360491e-05,-6.042540772127624749e-05,-6.040719951727613335e-05,-6.038904403882217053e-05,-6.037094106692286744e-05,-6.035289038464064504e-05,-6.033489177587059160e-05,-6.031694502500739574e-05,-6.029904991833312522e-05,-6.028120624318455967e-05,-6.026341378706501217e-05,-6.024567233897659690e-05,-6.022798168953125142e-05,-6.021034162967398018e-05,-6.019275195179307758e-05,-6.017521244883194953e-05,-6.015772291512178072e-05,-6.014028314615949000e-05,-6.012289293810813007e-05,-6.010555208812995431e-05,-6.008826039433090571e-05,-6.007101765603817256e-05,-6.005382367363365503e-05,-6.003667824833192057e-05,-6.001958118156958122e-05,-6.000253227700369507e-05,-5.998553133895745404e-05,-5.996857817142098313e-05,-5.995167258088240914e-05,-5.993481437444048154e-05,-5.991800335930497212e-05,-5.990123934407343143e-05,-5.988452213850914418e-05,-5.986785155320806240e-05,-5.985122739915471612e-05,-5.983464948927652571e-05,-5.981811763594580000e-05,-5.980163165325569352e-05,-5.978519135635407267e-05,-5.976879656066635960e-05,-5.975244708344984446e-05,-5.973614274157323933e-05,-5.971988335295996819e-05,-5.970366873736532298e-05,-5.968749871493317372e-05,-5.967137310591841270e-05,-5.965529173196371104e-05,-5.963925441582196285e-05,-5.962326098080117376e-05,-5.960731125059792745e-05,-5.959140505024107526e-05,-5.957554220542560230e-05,-5.955972254245711639e-05,-5.954394588886247064e-05,-5.952821207250158508e-05,-5.951252092190051357e-05,-5.949687226730615563e-05,-5.948126593846581045e-05,-5.946570176640353367e-05,-5.945017958347564857e-05,-5.943469922187194499e-05,-5.941926051511448037e-05,-5.940386329722491254e-05,-5.938850740311307774e-05,-5.937319266813290142e-05,-5.935791892891506549e-05,-5.934268602242331880e-05,-5.932749378562141018e-05,-5.931234205774904567e-05,-5.929723067804593128e-05,-5.928215948558523962e-05,-5.926712832116098895e-05,-5.925213702673293170e-05,-5.923718544353917537e-05,-5.922227341403907275e-05,-5.920740078185771083e-05,-5.919256739089773234e-05,-5.917777308583893614e-05,-5.916301771169418799e-05,-5.914830111491964360e-05,-5.913362314180492518e-05,-5.911898363952783342e-05,-5.910438245620985853e-05,-5.908981944041657997e-05,-5.907529444115766637e-05,-5.906080730877505403e-05,-5.904635789327761231e-05,-5.903194604556238900e-05,-5.901757161785869954e-05,-5.900323446245137049e-05,-5.898893443240238454e-05,-5.897467138082923555e-05,-5.896044516240372957e-05,-5.894625563168665039e-05,-5.893210264423798250e-05,-5.891798605589526616e-05,-5.890390572332870889e-05,-5.888986150381914086e-05,-5.887585325486943688e-05,-5.886188083525922821e-05,-5.884794410365712380e-05,-5.883404291956439991e-05,-5.882017714303744427e-05,-5.880634663490980074e-05,-5.879255125629256895e-05,-5.877879086890747118e-05,-5.876506533519787467e-05,-5.875137451805123590e-05,-5.873771828068807821e-05,-5.872409648738363686e-05,-5.871050900269070283e-05,-5.869695569143962288e-05,-5.868343641945994449e-05,-5.866995105280325973e-05,-5.865649945813178334e-05,-5.864308150260733044e-05,-5.862969705411336108e-05,-5.861634598069986879e-05,-5.860302815136053667e-05,-5.858974343553313702e-05,-5.857649170232237523e-05,-5.856327282260931355e-05,-5.855008666710848075e-05,-5.853693310708951714e-05,-5.852381201437717451e-05,-5.851072326101824927e-05,-5.849766672028078318e-05,-5.848464226537730681e-05,-5.847164976974239536e-05,-5.845868910758778014e-05,-5.844576015412439318e-05,-5.843286278434112191e-05,-5.841999687400400987e-05,-5.840716229921216751e-05,-5.839435893628674989e-05,-5.838158666277015740e-05,-5.836884535642683502e-05,-5.835613489468816084e-05,-5.834345515615124711e-05,-5.833080602030138451e-05,-5.831818736606875220e-05,-5.830559907293864086e-05,-5.829304102200616455e-05,-5.828051309375581468e-05,-5.826801516889412724e-05,-5.825554712979297278e-05,-5.824310885804706572e-05,-5.823070023602827661e-05,-5.821832114694114324e-05,-5.820597147410122574e-05,-5.819365110110163997e-05,-5.818135991231265791e-05,-5.816909779249312962e-05,-5.815686462634639398e-05,-5.814466029946396830e-05,-5.813248469777043681e-05,-5.812033770768998409e-05,-5.810821921559128356e-05,-5.809612910850914247e-05,-5.808406727442205764e-05,-5.807203360097545897e-05,-5.806002797653642133e-05,-5.804805028974957537e-05,-5.803610042964812976e-05,-5.802417828570938241e-05,-5.801228374791023157e-05,-5.800041670667166471e-05,-5.798857705247018046e-05,-5.797676467622636665e-05,-5.796497946941592261e-05,-5.795322132401414805e-05,-5.794149013194083153e-05,-5.792978578605945117e-05,-5.791810817917797394e-05,-5.790645720443743372e-05,-5.789483275575602050e-05,-5.788323472682987969e-05,-5.787166301213231279e-05,-5.786011750658071051e-05,-5.784859810531450819e-05,-5.783710470363967460e-05,-5.782563719758382348e-05,-5.781419548300803513e-05,-5.780277945649503479e-05,-5.779138901523817040e-05,-5.778002405609772296e-05,-5.776868447721072997e-05,-5.775737017593707279e-05,-5.774608105046930007e-05,-5.773481699983262772e-05,-5.772357792299676049e-05,-5.771236371887589200e-05,-5.770117428699483852e-05,-5.769000952754455014e-05,-5.767886934099353269e-05,-5.766775362758824741e-05,-5.765666228835231166e-05,-5.764559522464240970e-05,-5.763455233770420350e-05,-5.762353352983806687e-05,-5.761253870328886251e-05,-5.760156775996838618e-05,-5.759062060328723476e-05,-5.757969713626742703e-05,-5.756879726220853755e-05,-5.755792088535383044e-05,-5.754706790928043603e-05,-5.753623823850917418e-05,-5.752543177772739824e-05,-5.751464843223308421e-05,-5.750388810693563002e-05,-5.749315070774363434e-05,-5.748243614039916238e-05,-5.747174431092183511e-05,-5.746107512627496305e-05,-5.745042849286674524e-05,-5.743980431782702567e-05,-5.742920250872973753e-05,-5.741862297303779172e-05,-5.740806561860267720e-05,-5.739753035405303905e-05,-5.738701708762894427e-05,-5.737652572795903794e-05,-5.736605618439361010e-05,-5.735560836611641733e-05,-5.734518218297735004e-05,-5.733477754488180977e-05,-5.732439436167968694e-05,-5.731403254394251690e-05,-5.730369200257490192e-05,-5.729337264870348889e-05,-5.728307439323288008e-05,-5.727279714767830043e-05,-5.726254082416559754e-05,-5.725230533443204095e-05,-5.724209059104756747e-05,-5.723189650663762507e-05,-5.722172299377215055e-05,-5.721156996596477029e-05,-5.720143733611848802e-05,-5.719132501785795242e-05,-5.718123292552945713e-05,-5.717116097281316200e-05,-5.716110907438842759e-05,-5.715107714482359214e-05,-5.714106509874250506e-05,-5.713107285143514957e-05,-5.712110031835804236e-05,-5.711114741491218894e-05,-5.710121405710921749e-05,-5.709130016123831197e-05,-5.708140564331110056e-05,-5.707153041978330066e-05,-5.706167440783227462e-05,-5.705183752457987367e-05,-5.704201968698141556e-05,-5.703222081265835186e-05,-5.702244081945417875e-05,-5.701267962548994817e-05,-5.700293714849813398e-05,-5.699321330715489964e-05,-5.698350802063600895e-05,-5.697382120728455845e-05,-5.696415278644284541e-05,-5.695450267739765593e-05,-5.694487079943577612e-05,-5.693525707306523742e-05,-5.692566141785038170e-05,-5.691608375385515117e-05,-5.690652400186513304e-05,-5.689698208255489220e-05,-5.688745791676552699e-05,-5.687795142539364690e-05,-5.686846253016852870e-05,-5.685899115231984879e-05,-5.684953721363239509e-05,-5.684010063622402242e-05,-5.683068134221258561e-05,-5.682127925382696176e-05,-5.681189429374011723e-05,-5.680252638484706296e-05,-5.679317544998729872e-05,-5.678384141255543582e-05,-5.677452419572404096e-05,-5.676522372327630350e-05,-5.675593991893990164e-05,-5.674667270666455821e-05,-5.673742201106612981e-05,-5.672818775620536158e-05,-5.671896986642055438e-05,-5.670976826699369866e-05,-5.670058288281820680e-05,-5.669141363889851348e-05,-5.668226046057212031e-05,-5.667312327367612923e-05,-5.666400200404764220e-05,-5.665489657735722773e-05,-5.664580691983056582e-05,-5.663673295785986994e-05,-5.662767461783735357e-05,-5.661863182659931937e-05,-5.660960451120411463e-05,-5.660059259826599742e-05,-5.659159601517638194e-05,-5.658261468977077158e-05,-5.657364854949609168e-05,-5.656469752207682333e-05,-5.655576153534846995e-05,-5.654684051742409068e-05,-5.653793439697185619e-05,-5.652904310249340369e-05,-5.652016656254588156e-05,-5.651130470590848276e-05,-5.650245746174897832e-05,-5.649362475929065042e-05,-5.648480652781229239e-05,
        -5.647600269703678677e-05,-5.646721319663150496e-05,-5.645843795631932949e-05,-5.644967690648927672e-05,-5.644092997725280725e-05,-5.643219709883240398e-05,-5.642347820189463903e-05,-5.641477321743915141e-05,-5.640608207613251324e-05,-5.639740470897436353e-05,-5.638874104729740822e-05,-5.638009102243435322e-05,-5.637145456621750483e-05,-5.636283160997956898e-05,-5.635422208560836310e-05,-5.634562592532477154e-05,-5.633704306151621211e-05,-5.632847342629254683e-05,-5.631991695187466007e-05,-5.631137357131610344e-05,-5.630284321739287279e-05,-5.629432582304749744e-05,-5.628582132150006245e-05,-5.627732964558207485e-05,-5.626885072923526465e-05,-5.626038450590176154e-05,-5.625193090919022865e-05,-5.624348987315341830e-05,-5.623506133173306054e-05,-5.622664521909293001e-05,-5.621824146934129018e-05,-5.620985001758560529e-05,-5.620147079810067225e-05,-5.619310374532782149e-05,-5.618474879470758410e-05,-5.617640588090333509e-05,-5.616807493930009443e-05,-5.615975590567146014e-05,-5.615144871479182953e-05,-5.614315330221275602e-05,-5.613486960454050489e-05,-5.612659755710458498e-05,-5.611833709590063890e-05,-5.611008815759044310e-05,-5.610185067778106216e-05,-5.609362459341182827e-05,-5.608540984142207364e-05,-5.607720635814050780e-05,-5.606901408006237375e-05,-5.606083294468211520e-05,-5.605266288927213125e-05,-5.604450385066073181e-05,-5.603635576656440520e-05,-5.602821857464412858e-05,-5.602009221222781221e-05,-5.601197661719847787e-05,-5.600387172777221423e-05,-5.599577748177653191e-05,-5.598769381720547500e-05,-5.597962067283024368e-05,-5.597155798703346008e-05,-5.596350569808672404e-05,-5.595546374492776920e-05,-5.594743206632779575e-05,-5.593941060111351504e-05,-5.593139928883328338e-05,-5.592339806836932325e-05,-5.591540687877039062e-05,-5.590742565991790869e-05,-5.589945435130472262e-05,-5.589149289259021103e-05,-5.588354122343375252e-05,-5.587559928410534837e-05,-5.586766701481948871e-05,-5.585974435534657445e-05,-5.585183124579007341e-05,-5.584392762697509838e-05,-5.583603343950471753e-05,-5.582814862375995446e-05,-5.582027312084347770e-05,-5.581240687141386658e-05,-5.580454981651827850e-05,-5.579670189731489316e-05,-5.578886305512842370e-05,-5.578103323100602751e-05,-5.577321236660548465e-05,-5.576540040386213093e-05,-5.575759728393414605e-05,-5.574980294886788812e-05,-5.574201734065420411e-05,-5.573424040139496327e-05,-5.572647207296999028e-05,-5.571871229764768785e-05,-5.571096101808503676e-05,-5.570321817660595087e-05,-5.569548371558985522e-05,-5.568775757808230864e-05,-5.568003970679580306e-05,-5.567233004444283040e-05,-5.566462853406894951e-05,-5.565693511910829727e-05,-5.564924974282847714e-05,-5.564157234799749219e-05,-5.563390287832703507e-05,-5.562624127769533189e-05,-5.561858748931447494e-05,-5.561094145711820147e-05,-5.560330312520678220e-05,-5.559567243701435402e-05,-5.558804933708527685e-05,-5.558043376951982140e-05,-5.557282567847376953e-05,-5.556522500821392541e-05,-5.555763170311811550e-05,-5.555004570823030008e-05,-5.554246696798381677e-05,-5.553489542714507010e-05,-5.552733103075802035e-05,-5.551977372320049398e-05,-5.551222344990502933e-05,-5.550468015652620934e-05,-5.549714378783043855e-05,-5.548961428925025530e-05,-5.548209160610717561e-05,-5.547457568427782704e-05,-5.546706646913923677e-05,-5.545956390629047661e-05,-5.545206794232981906e-05,-5.544457852252326902e-05,-5.543709559302500978e-05,-5.542961909998922465e-05,-5.542214898957009694e-05,-5.541468520858794378e-05,-5.540722770291939270e-05,-5.539977641888516047e-05,-5.539233130341658651e-05,-5.538489230333398794e-05,-5.537745936529114843e-05,-5.537003243605287395e-05,-5.536261146255050392e-05,-5.535519639193742236e-05,-5.534778717136701331e-05,-5.534038374804817195e-05,-5.533298606924530461e-05,-5.532559408250037336e-05,-5.531820773496676225e-05,-5.531082697435296680e-05,-5.530345174897810523e-05,-5.529608200566249465e-05,-5.528871769255871982e-05,-5.528135875793038778e-05,-5.527400514948599408e-05,-5.526665681548914577e-05,-5.525931370381487184e-05,-5.525197576311535741e-05,-5.524464294154318722e-05,-5.523731518747299063e-05,-5.522999244977899735e-05,-5.522267467666930330e-05,-5.521536181701813817e-05,-5.520805381969973169e-05,-5.520075063342178012e-05,-5.519345220722504664e-05,-5.518615849020580555e-05,-5.517886943112726428e-05,-5.517158497958529750e-05,-5.516430508484271300e-05,-5.515702969610680739e-05,-5.514975876269589961e-05,-5.514249223403933087e-05,-5.513523006017706507e-05,-5.512797219064946574e-05,-5.512071857510791872e-05,-5.511346916325932099e-05,-5.510622390508812529e-05,-5.509898275091185127e-05,-5.509174565082597397e-05,-5.508451255420432346e-05,-5.507728341186401977e-05,-5.507005817434462713e-05,-5.506283679146406485e-05,-5.505561921420598637e-05,-5.504840539266586674e-05,-5.504119527777184828e-05,-5.503398882017451754e-05,-5.502678597041343878e-05,-5.501958667963879890e-05,-5.501239089839016216e-05,-5.500519857809527124e-05,-5.499800966962675730e-05,-5.499082412419031840e-05,-5.498364189299165261e-05,-5.497646292712543570e-05,-5.496928717829696609e-05,-5.496211459776745301e-05,-5.495494513679810566e-05,-5.494777874759382286e-05,-5.494061538147132495e-05,-5.493345498985835462e-05,-5.492629752523736641e-05,-5.491914293914712530e-05,-5.491199118345946317e-05,-5.490484221004621190e-05,-5.489769597138982604e-05,-5.489055241963969323e-05,-5.488341150661213419e-05,-5.487627318512267038e-05,-5.486913740720966715e-05,-5.486200412546660132e-05,-5.485487329259797207e-05,-5.484774486103072277e-05,-5.484061878308077453e-05,-5.483349501173018226e-05,-5.482637350029406775e-05,-5.481925420097732982e-05,-5.481213706692855681e-05,-5.480502205118531478e-05,-5.479790910672965865e-05,-5.479079818693222137e-05,-5.478368924483056901e-05,-5.477658223379533453e-05,-5.476947710708612860e-05,-5.476237381801807302e-05,-5.475527232023935653e-05,-5.474817256750919015e-05,-5.474107451303167338e-05,-5.473397811051050610e-05,-5.472688331414898855e-05,-5.471979007742877599e-05,-5.471269835427561290e-05,-5.470560809867075491e-05,-5.469851926442892420e-05,-5.469143180597546561e-05,-5.468434567734714591e-05,-5.467726083269175419e-05,-5.467017722610156838e-05,-5.466309481205744447e-05,-5.465601354548432766e-05,-5.464893338008591783e-05,-5.464185427062062672e-05,-5.463477617201339953e-05,-5.462769903868958110e-05,-5.462062282524104972e-05,-5.461354748653723945e-05,-5.460647297772514008e-05,-5.459939925339662992e-05,-5.459232626847665415e-05,-5.458525397833424719e-05,-5.457818233789435425e-05,-5.457111130230396512e-05,-5.456404082693211421e-05,-5.455697086692579134e-05,-5.454990137759851976e-05,-5.454283231443035618e-05,-5.453576363323442422e-05,-5.452869528904669139e-05,-5.452162723773579245e-05,-5.451455943483729527e-05,-5.450749183599779002e-05,-5.450042439753000068e-05,-5.449335707463642819e-05,-5.448628982368530771e-05,-5.447922260048976284e-05,-5.447215536086291721e-05,-5.446508806128402824e-05,-5.445802065762173072e-05,-5.445095310613323747e-05,-5.444388536318678362e-05,-5.443681738509509316e-05,-5.442974912822640121e-05,-5.442268054917098752e-05,-5.441561160418606491e-05,-5.440854224980640197e-05,-5.440147244295534534e-05,-5.439440214022317477e-05,-5.438733129831119228e-05,-5.438025987392069993e-05,-5.437318782414157781e-05,-5.436611510578615025e-05,-5.435904167572225276e-05,-5.435196749120629889e-05,-5.434489250916163527e-05,-5.433781668673365317e-05,-5.433073998156734419e-05,-5.432366235053054382e-05,-5.431658375071313216e-05,-5.430950413998214543e-05,-5.430242347553848603e-05,-5.429534171508265672e-05,-5.428825881603760450e-05,-5.428117473604832099e-05,-5.427408943281530895e-05,-5.426700286376151539e-05,-5.425991498719806572e-05,-5.425282576093648501e-05,-5.424573514239972027e-05,-5.423864308967685233e-05,-5.423154956119002890e-05,-5.422445451475077505e-05,-5.421735790833714930e-05,-5.421025970037129937e-05,-5.420315984899781725e-05,-5.419605831269436180e-05,-5.418895504955001385e-05,-5.418185001798692113e-05,-5.417474317659376482e-05,-5.416763448395922609e-05,-5.416052389850545268e-05,-5.415341137882112577e-05,-5.414629688382799344e-05,-5.413918037211473688e-05,-5.413206180254759303e-05,-5.412494113404830998e-05,-5.411781832526108005e-05,-5.411069333532969594e-05,-5.410356612328692805e-05,-5.409643664822105791e-05,-5.408930486927587822e-05,-5.408217074531762592e-05,-5.407503423604520520e-05,-5.406789530065791993e-05,-5.406075389829956279e-05,-5.405360998850250454e-05,-5.404646353057707131e-05,-5.403931448422216732e-05,-5.403216280908118563e-05,-5.402500846451996352e-05,-5.401785141029291637e-05,-5.401069160637650413e-05,-5.400352901235860870e-05,-5.399636358777160083e-05,-5.398919529292500741e-05,-5.398202408773977723e-05,-5.397484993213685911e-05,-5.396767278603720186e-05,-5.396049260958379890e-05,-5.395330936347475514e-05,-5.394612300707590791e-05,-5.393893350097433981e-05,-5.393174080597917808e-05,-5.392454488178932692e-05,-5.391734568899186897e-05,-5.391014318822939799e-05,-5.390293734003348547e-05,-5.389572810515774748e-05,-5.388851544385619974e-05,-5.388129931688245833e-05,-5.387407968526769508e-05,-5.386685650948797033e-05,-5.385962975079650050e-05,-5.385239936972485708e-05,-5.384516532735972305e-05,-5.383792758489880370e-05,-5.383068610292918166e-05,-5.382344084309265142e-05,-5.381619176636487367e-05,-5.380893883383253140e-05,-5.380168200697088565e-05,-5.379442124725519747e-05,-5.378715651588317215e-05,-5.377988777383047037e-05,-5.377261498312746468e-05,-5.376533810541594960e-05,-5.375805710194914155e-05,-5.375077193464639080e-05,-5.374348256537153645e-05,-5.373618895498921688e-05,-5.372889106597389386e-05,-5.372158886068900685e-05,-5.371428229994368309e-05,-5.370697134621238433e-05,-5.369965596185855006e-05,-5.369233610841295246e-05,-5.368501174807249754e-05,-5.367768284325613593e-05,-5.367034935638281823e-05,-5.366301124920536125e-05,-5.365566848419822676e-05,-5.364832102389138768e-05,-5.364096883042623887e-05,-5.363361186688786475e-05,-5.362625009525112674e-05,-5.361888347826804235e-05,-5.361151197880165142e-05,
        -5.360413555915988226e-05,-5.359675418242781930e-05,-5.358936781135748006e-05,-5.358197640842332632e-05,-5.357457993709902055e-05,-5.356717836035862490e-05,-5.355977164039904537e-05,-5.355235974074945560e-05,-5.354494262499454038e-05,-5.353752025594182840e-05,-5.353009259656538177e-05,-5.352265961039437414e-05,-5.351522126079144570e-05,-5.350777751106372548e-05,-5.350032832462936483e-05,-5.349287366485100392e-05,-5.348541349547986101e-05,-5.347794778010062089e-05,-5.347047648218694604e-05,-5.346299956549005472e-05,-5.345551699376116517e-05,-5.344802873102905139e-05,-5.344053474082288702e-05,-5.343303498700491261e-05,-5.342552943377043562e-05,-5.341801804520374120e-05,-5.341050078505604759e-05,-5.340297761763368456e-05,-5.339544850690991495e-05,-5.338791341724657968e-05,-5.338037231322756426e-05,-5.337282515843755348e-05,-5.336527191796003322e-05,-5.335771255604582208e-05,-5.335014703672369407e-05,-5.334257532491060161e-05,-5.333499738541247481e-05,-5.332741318264666575e-05,-5.331982268091950417e-05,-5.331222584559203170e-05,-5.330462264102608927e-05,-5.329701303219414044e-05,-5.328939698445722684e-05,-5.328177446201065592e-05,-5.327414543027098048e-05,-5.326650985432168639e-05,-5.325886769896870376e-05,-5.325121892973960769e-05,-5.324356351188441749e-05,-5.323590141059764136e-05,-5.322823259079623170e-05,-5.322055701811878592e-05,-5.321287465825941254e-05,-5.320518547657915320e-05,-5.319748943866109414e-05,-5.318978650964423238e-05,-5.318207665566676567e-05,-5.317435984242280256e-05,-5.316663603538440697e-05,-5.315890520068977665e-05,-5.315116730381097554e-05,-5.314342231071966793e-05,-5.313567018760956273e-05,-5.312791090072987998e-05,-5.312014441566370593e-05,-5.311237069860474946e-05,-5.310458971591325295e-05,-5.309680143367190297e-05,-5.308900581835196419e-05,-5.308120283620265667e-05,-5.307339245363973390e-05,-5.306557463696792709e-05,-5.305774935304707896e-05,-5.304991656807089839e-05,-5.304207624862167236e-05,-5.303422836189231049e-05,-5.302637287413203282e-05,-5.301850975192312632e-05,-5.301063896262503405e-05,-5.300276047293106529e-05,-5.299487424986759621e-05,-5.298698026023895835e-05,-5.297907847101601675e-05,-5.297116884950270332e-05,-5.296325136272539424e-05,-5.295532597826557719e-05,-5.294739266292758373e-05,-5.293945138412636808e-05,-5.293150210949892909e-05,-5.292354480612715406e-05,-5.291557944181457529e-05,-5.290760598380961355e-05,-5.289962439974926767e-05,-5.289163465771462569e-05,-5.288363672506513069e-05,-5.287563056943778150e-05,-5.286761615891366617e-05,-5.285959346112978352e-05,-5.285156244394517699e-05,-5.284352307577400154e-05,-5.283547532458632290e-05,-5.282741915813016220e-05,-5.281935454470865210e-05,-5.281128145273594754e-05,-5.280319985001558081e-05,-5.279510970535028491e-05,-5.278701098687665905e-05,-5.277890366284232471e-05,-5.277078770227205951e-05,-5.276266307330246264e-05,-5.275452974445871135e-05,-5.274638768471007211e-05,-5.273823686274825562e-05,-5.273007724698741683e-05,-5.272190880650784450e-05,-5.271373151033431625e-05,-5.270554532704752049e-05,-5.269735022578325712e-05,-5.268914617584385951e-05,-5.268093314603206068e-05,-5.267271110570570514e-05,-5.266448002394508165e-05,-5.265623986988599015e-05,-5.264799061338587549e-05,-5.263973222352502646e-05,-5.263146466955026526e-05,-5.262318792126352562e-05,-5.261490194818918553e-05,-5.260660671985162296e-05,-5.259830220605277162e-05,-5.258998837642803181e-05,-5.258166520089035956e-05,-5.257333264929719974e-05,-5.256499069122844148e-05,-5.255663929731868578e-05,-5.254827843698128831e-05,-5.253990808040676086e-05,-5.253152819822970443e-05,-5.252313875997449699e-05,-5.251473973633125070e-05,-5.250633109771252194e-05,-5.249791281419780020e-05,-5.248948485631066418e-05,-5.248104719451918143e-05,-5.247259979951346409e-05,-5.246414264170606856e-05,-5.245567569217568504e-05,-5.244719892161242569e-05,-5.243871230020680230e-05,-5.243021579925954967e-05,-5.242170938979384687e-05,-5.241319304255531719e-05,-5.240466672862265085e-05,-5.239613041918556036e-05,-5.238758408532273592e-05,-5.237902769833491234e-05,-5.237046122941180215e-05,-5.236188464974311785e-05,-5.235329793107368346e-05,-5.234470104475974495e-05,-5.233609396182448137e-05,-5.232747665440129481e-05,-5.231884909440154274e-05,-5.231021125268187077e-05,-5.230156310126465868e-05,-5.229290461256086431e-05,-5.228423575781571131e-05,-5.227555650916260177e-05,-5.226686683879044892e-05,-5.225816671833305449e-05,-5.224945612031239861e-05,-5.224073501686188337e-05,-5.223200337978184393e-05,-5.222326118192732736e-05,-5.221450839570929148e-05,-5.220574499320562722e-05,-5.219697094682729244e-05,-5.218818622959586762e-05,-5.217939081392231060e-05,-5.217058467221757923e-05,-5.216176777766978745e-05,-5.215294010291193771e-05,-5.214410162057703246e-05,-5.213525230396420795e-05,-5.212639212592851123e-05,-5.211752105943601165e-05,-5.210863907767482317e-05,-5.209974615394408204e-05,-5.209084226143190222e-05,-5.208192737315986420e-05,-5.207300146287119347e-05,-5.206406450392053742e-05,-5.205511646971805462e-05,-5.204615733400697053e-05,-5.203718707025295487e-05,-5.202820565225474425e-05,-5.201921305353351954e-05,-5.201020924799903966e-05,-5.200119420994964159e-05,-5.199216791285099504e-05,-5.198313033072388123e-05,-5.197408143808868175e-05,-5.196502120857759977e-05,-5.195594961659999456e-05,-5.194686663673175886e-05,-5.193777224277162929e-05,-5.192866640946203205e-05,-5.191954911115681526e-05,-5.191042032248738280e-05,-5.190128001802962743e-05,-5.189212817224841956e-05,-5.188296476021925230e-05,-5.187378975662904068e-05,-5.186460313599816629e-05,-5.185540487356865569e-05,-5.184619494452702426e-05,-5.183697332339365360e-05,-5.182773998574363716e-05,-5.181849490687451265e-05,-5.180923806141768395e-05,-5.179996942522580028e-05,-5.179068897354088818e-05,-5.178139668177150767e-05,-5.177209252565928566e-05,-5.176277648066829329e-05,-5.175344852237362403e-05,-5.174410862629486019e-05,-5.173475676878425134e-05,-5.172539292541689093e-05,-5.171601707210093934e-05,-5.170662918474455694e-05,-5.169722923953345983e-05,-5.168781721276438645e-05,-5.167839308023447487e-05,-5.166895681862904155e-05,-5.165950840413380263e-05,-5.165004781310100768e-05,-5.164057502204943972e-05,-5.163109000744237065e-05,-5.162159274585409463e-05,-5.161208321430299506e-05,-5.160256138936336612e-05,-5.159302724749847968e-05,-5.158348076594876375e-05,-5.157392192201015746e-05,-5.156435069203491040e-05,-5.155476705331896170e-05,-5.154517098343580628e-05,-5.153556245907076061e-05,-5.152594145779731960e-05,-5.151630795696693355e-05,-5.150666193404207505e-05,-5.149700336654072785e-05,-5.148733223203638687e-05,-5.147764850826908045e-05,-5.146795217275679235e-05,-5.145824320340608438e-05,-5.144852157823454064e-05,-5.143878727487116720e-05,-5.142904027155559277e-05,-5.141928054641642376e-05,-5.140950807741573314e-05,-5.139972284307070538e-05,-5.138992482139892459e-05,-5.138011399064001949e-05,-5.137029032958873032e-05,-5.136045381676224153e-05,-5.135060443040018185e-05,-5.134074214935280267e-05,-5.133086695263688881e-05,-5.132097881849206900e-05,-5.131107772604615036e-05,-5.130116365431591774e-05,-5.129123658220713367e-05,-5.128129648906964988e-05,-5.127134335364269546e-05,-5.126137715544265561e-05,-5.125139787387489321e-05,-5.124140548817823770e-05,-5.123139997748049623e-05,-5.122138132179765435e-05,-5.121134950086814186e-05,-5.120130449370874359e-05,-5.119124628044646741e-05,-5.118117484087525426e-05,-5.117109015490006740e-05,-5.116099220264791470e-05,-5.115088096402375939e-05,-5.114075641898807589e-05,-5.113061854772338322e-05,-5.112046733091180073e-05,-5.111030274873584744e-05,-5.110012478137804237e-05,-5.108993340940948258e-05,-5.107972861345677629e-05,-5.106951037420204287e-05,-5.105927867254944630e-05,-5.104903348884803904e-05,-5.103877480400198507e-05,-5.102850259930402643e-05,-5.101821685571383824e-05,-5.100791755452416254e-05,-5.099760467625058524e-05,-5.098727820246340414e-05,-5.097693811451087242e-05,-5.096658439390777673e-05,-5.095621702216890370e-05,-5.094583598064250651e-05,-5.093544125106541642e-05,-5.092503281495242007e-05,-5.091461065426239330e-05,-5.090417475084318966e-05,-5.089372508687572960e-05,-5.088326164398582208e-05,-5.087278440435438753e-05,-5.086229335021785758e-05,-5.085178846386817497e-05,-5.084126972765279362e-05,-5.083073712364161167e-05,-5.082019063462617225e-05,-5.080963024306495157e-05,-5.079905593180500389e-05,-5.078846768352685004e-05,-5.077786548068896622e-05,-5.076724930624942900e-05,-5.075661914361040417e-05,-5.074597497511934563e-05,-5.073531678401188572e-05,-5.072464455418979057e-05,-5.071395826822255870e-05,-5.070325790956786705e-05,-5.069254346196094829e-05,-5.068181490824885671e-05,-5.067107223249989190e-05,-5.066031541867133114e-05,-5.064954444983227333e-05,-5.063875930999550690e-05,-5.062795998356239835e-05,-5.061714645387960232e-05,-5.060631870501541840e-05,-5.059547672148223540e-05,-5.058462048745937523e-05,-5.057374998718167092e-05,-5.056286520471742207e-05,-5.055196612491208441e-05,-5.054105273238906904e-05,-5.053012501138320900e-05,-5.051918294673996002e-05,-5.050822652324926665e-05,-5.049725572592311806e-05,-5.048627053960696998e-05,-5.047527094914627810e-05,-5.046425693999712081e-05,-5.045322849694944267e-05,-5.044218560573687782e-05,-5.043112825137141542e-05,-5.042005641925362269e-05,-5.040897009511713378e-05,-5.039786926447353821e-05,-5.038675391316749241e-05,-5.037562402671058592e-05,-5.036447959089196402e-05,-5.035332059166730545e-05,-5.034214701538086700e-05,-5.033095884787730512e-05,-5.031975607527883199e-05,-5.030853868404072671e-05,-5.029730666028520147e-05,-5.028605999046753539e-05,-5.027479866109851869e-05,-5.026352265907751971e-05,-5.025223197052675062e-05,-5.024092658262313549e-05,-5.022960648226604263e-05,-5.021827165579972885e-05,-5.020692209106725201e-05,-5.019555777446838007e-05,-5.018417869323554825e-05,-5.017278483510079212e-05,-5.016137618696348000e-05,-5.014995273650013630e-05,-5.013851447116524085e-05,-5.012706137846878462e-05,-5.011559344597626975e-05,
        -5.010411066147524295e-05,-5.009261301341938477e-05,-5.008110048926317504e-05,-5.006957307651660471e-05,-5.005803076396642126e-05,-5.004647353984426061e-05,-5.003490139204869180e-05,-5.002331430892237307e-05,-5.001171227947409648e-05,-5.000009529160243105e-05,-4.998846333398310193e-05,-4.997681639579143464e-05,-4.996515446520355397e-05,-4.995347753156131887e-05,-4.994178558354045450e-05,-4.993007861031628636e-05,-4.991835660117516227e-05,-4.990661954484831853e-05,-4.989486743106619215e-05,-4.988310024905961981e-05,-4.987131798805943816e-05,-4.985952063801812884e-05,-4.984770818872164000e-05,-4.983588062956734177e-05,-4.982403795000811542e-05,-4.981218014044053177e-05,-4.980030719103911707e-05,-4.978841909147879718e-05,-4.977651583204512065e-05,-4.976459740291261369e-05,-4.975266379458886945e-05,-4.974071499752596992e-05,-4.972875100184293018e-05,-4.971677179865796603e-05,-4.970477737853418176e-05,-4.969276773186814822e-05,-4.968074284955603659e-05,-4.966870272310464074e-05,-4.965664734329910956e-05,-4.964457670109112541e-05,-4.963249078770992639e-05,-4.962038959466230637e-05,-4.960827311323301458e-05,-4.959614133454026685e-05,-4.958399425075699085e-05,-4.957183185338998044e-05,-4.955965413372398487e-05,-4.954746108398744298e-05,-4.953525269629777128e-05,-4.952302896199523019e-05,-4.951078987369683659e-05,-4.949853542363102932e-05,-4.948626560369318028e-05,-4.947398040655581752e-05,-4.946167982455840217e-05,-4.944936385020692882e-05,-4.943703247611841434e-05,-4.942468569502089792e-05,-4.941232349991997452e-05,-4.939994588337714987e-05,-4.938755283850904121e-05,-4.937514435848777694e-05,-4.936272043631895201e-05,-4.935028106539673942e-05,-4.933782623911531218e-05,-4.932535595064679868e-05,-4.931287019366292768e-05,-4.930036896189093909e-05,-4.928785224889153938e-05,-4.927532004822543499e-05,-4.926277235400844390e-05,-4.925020916041189523e-05,-4.923763046121854003e-05,-4.922503625087726320e-05,-4.921242652317081578e-05,-4.919980127265910497e-05,-4.918716049362448217e-05,-4.917450418079338803e-05,-4.916183232900328548e-05,-4.914914493236999249e-05,-4.913644198600852775e-05,-4.912372348453430959e-05,-4.911098942306235671e-05,-4.909823979676319894e-05,-4.908547460069634383e-05,-4.907269383008783237e-05,-4.905989748016370555e-05,-4.904708554642756013e-05,-4.903425802443850401e-05,-4.902141490975564508e-05,-4.900855619793809126e-05,-4.899568188482250619e-05,-4.898279196630106469e-05,-4.896988643848798617e-05,-4.895696529721993429e-05,-4.894402853872215076e-05,-4.893107615933089960e-05,-4.891810815499386678e-05,-4.890512452226936091e-05,-4.889212525810426868e-05,-4.887911035872383181e-05,-4.886607982063084776e-05,-4.885303364088322553e-05,-4.883997181642785179e-05,-4.882689434415610208e-05,-4.881380122090384077e-05,-4.880069244395102146e-05,-4.878756801057759773e-05,-4.877442791800801203e-05,-4.876127216374426254e-05,-4.874810074551039207e-05,-4.873491366058635421e-05,-4.872171090669619176e-05,-4.870849248167496981e-05,-4.869525838335775347e-05,-4.868200860980165245e-05,-4.866874315928582106e-05,-4.865546202942327980e-05,-4.864216521871522758e-05,-4.862885272588490793e-05,-4.861552454871187479e-05,-4.860218068603039399e-05,-4.858882113639717559e-05,-4.857544589870199658e-05,-4.856205497166810048e-05,-4.854864835396321965e-05,-4.853522604458815337e-05,-4.852178804282125668e-05,-4.850833434777435116e-05,-4.849486495867028069e-05,-4.848137987489842260e-05,-4.846787909590366539e-05,-4.845436262085334178e-05,-4.844083044991398523e-05,-4.842728258264150654e-05,-4.841371901853630533e-05,-4.840013975804247082e-05,-4.838654480060489149e-05,-4.837293414661214541e-05,-4.835930779623076603e-05,-4.834566574973830910e-05,-4.833200800724579693e-05,-4.831833456930834103e-05,-4.830464543681411982e-05,-4.829094061015171135e-05,-4.827722008998724945e-05,-4.826348387709789023e-05,-4.824973197259385671e-05,-4.823596437736332732e-05,-4.822218109240550277e-05,-4.820838211921918415e-05,-4.819456745885908333e-05,-4.818073711276849025e-05,-4.816689108239069483e-05,-4.815302936922449817e-05,-4.813915197515727940e-05,-4.812525890146579499e-05,-4.811135015053702446e-05,-4.809742572398079119e-05,-4.808348562373998547e-05,-4.806952985220158681e-05,-4.805555841141950779e-05,-4.804157130383623908e-05,-4.802756853156120442e-05,-4.801355009748098368e-05,-4.799951600376051175e-05,-4.798546625289779044e-05,-4.797140084839002228e-05,-4.795731979279072021e-05,-4.794322308876441951e-05,-4.792911073958627810e-05,-4.791498274803185353e-05,-4.790083911782039294e-05,-4.788667985217154310e-05,-4.787250495419392848e-05,-4.785831442766230737e-05,-4.784410827596285998e-05,-4.782988650303687805e-05,-4.781564911232605297e-05,-4.780139610771616532e-05,-4.778712749364810719e-05,-4.777284327373010342e-05,-4.775854345218100150e-05,-4.774422803316413777e-05,-4.772989702123142663e-05,-4.771555042054620444e-05,-4.770118823560487442e-05,-4.768681047145895135e-05,-4.767241713232728273e-05,-4.765800822303933870e-05,-4.764358374870214519e-05,-4.762914371408966119e-05,-4.761468812408686802e-05,-4.760021698435590309e-05,-4.758573029994828119e-05,-4.757122807586000590e-05,-4.755671031769770352e-05,-4.754217703123453376e-05,-4.752762822168854484e-05,-4.751306389505494110e-05,-4.749848405699585996e-05,-4.748388871333997230e-05,-4.746927787030452706e-05,-4.745465153349615051e-05,-4.744000970946515849e-05,-4.742535240470635571e-05,-4.741067962460432383e-05,-4.739599137615346791e-05,-4.738128766645921530e-05,-4.736656850112819228e-05,-4.735183388726582621e-05,-4.733708383192203328e-05,-4.732231834164712936e-05,-4.730753742349103064e-05,-4.729274108444814217e-05,-4.727792933167940248e-05,-4.726310217273432812e-05,-4.724825961471834646e-05,-4.723340166501444060e-05,-4.721852833111661596e-05,-4.720363962079643372e-05,-4.718873554171443274e-05,-4.717381610136461845e-05,-4.715888130774059661e-05,-4.714393116933557337e-05,-4.712896569364355415e-05,-4.711398488876916701e-05,-4.709898876337215157e-05,-4.708397732544611358e-05,-4.706895058370630380e-05,-4.705390854620183916e-05,-4.703885122187001500e-05,-4.702377861942608206e-05,-4.700869074725222418e-05,-4.699358761484084823e-05,-4.697846923057413804e-05,-4.696333560400001161e-05,-4.694818674377820855e-05,-4.693302265917909111e-05,-4.691784336008364420e-05,-4.690264885520711857e-05,-4.688743915431947684e-05,-4.687221426702414817e-05,-4.685697420292456172e-05,-4.684171897184619127e-05,-4.682644858372553287e-05,-4.681116304816601570e-05,-4.679586237543720273e-05,-4.678054657553110118e-05,-4.676521565905034095e-05,-4.674986963576488463e-05,-4.673450851599980638e-05,-4.671913231080182527e-05,-4.670374103016294853e-05,-4.668833468496336181e-05,-4.667291328613876189e-05,-4.665747684406973406e-05,-4.664202536980299740e-05,-4.662655887460731563e-05,-4.661107736925185208e-05,-4.659558086489434814e-05,-4.658006937302561212e-05,-4.656454290480338543e-05,-4.654900147160745405e-05,-4.653344508537271551e-05,-4.651787375709037775e-05,-4.650228749863982713e-05,-4.648668632217800578e-05,-4.647107023919572200e-05,-4.645543926162787329e-05,-4.643979340163140179e-05,-4.642413267141876076e-05,-4.640845708314689233e-05,-4.639276664875069400e-05,-4.637706138116426402e-05,-4.636134129271107795e-05,-4.634560639554807793e-05,-4.632985670283140678e-05,-4.631409222688454008e-05,-4.629831298091913183e-05,-4.628251897753621336e-05,-4.626671022950334944e-05,-4.625088675036526098e-05,-4.623504855316706852e-05,-4.621919565112042605e-05,-4.620332805704840951e-05,-4.618744578499534015e-05,-4.617154884861696118e-05,-4.615563726090288199e-05,-4.613971103556435693e-05,-4.612377018636815151e-05,-4.610781472763614275e-05,-4.609184467291305154e-05,-4.607586003579910994e-05,-4.605986083117130647e-05,-4.604384707290742895e-05,-4.602781877494077634e-05,-4.601177595181527025e-05,-4.599571861785278770e-05,-4.597964678793031723e-05,-4.596356047620320240e-05,-4.594745969738189828e-05,-4.593134446634339341e-05,-4.591521479790916516e-05,-4.589907070723375782e-05,-4.588291220875007070e-05,-4.586673931766815926e-05,-4.585055204980870158e-05,-4.583435041982664160e-05,-4.581813444287652359e-05,-4.580190413466800337e-05,-4.578565951074420326e-05,-4.576940058664824562e-05,-4.575312737781223049e-05,-4.573683990027888058e-05,-4.572053816948029592e-05,-4.570422220173675498e-05,-4.568789201297995817e-05,-4.567154761880853897e-05,-4.565518903570930931e-05,-4.563881627989152534e-05,-4.562242936767546553e-05,-4.560602831527038603e-05,-4.558961313938514337e-05,-4.557318385634001601e-05,-4.555674048278834931e-05,-4.554028303555002211e-05,-4.552381153127837976e-05,-4.550732598690432340e-05,-4.549082641930324300e-05,-4.547431284562808429e-05,-4.545778528292077070e-05,-4.544124374816771450e-05,-4.542468825852186143e-05,-4.540811883191331333e-05,-4.539153548532848248e-05,-4.537493823625338152e-05,-4.535832710239606769e-05,-4.534170210129806478e-05,-4.532506325066743003e-05,-4.530841056832324298e-05,-4.529174407236213895e-05,-4.527506378015910826e-05,-4.525836971003283082e-05,-4.524166188063505345e-05,-4.522494030945178878e-05,-4.520820501491273902e-05,-4.519145601578067328e-05,-4.517469332987467112e-05,-4.515791697584647935e-05,-4.514112697218131132e-05,-4.512432333780846960e-05,-4.510750609154623447e-05,-4.509067525193533044e-05,-4.507383083768301546e-05,-4.505697286771859211e-05,-4.504010136158198563e-05,-4.502321633803596512e-05,-4.500631781612085547e-05,-4.498940581526555960e-05,-4.497248035467693583e-05,-4.495554145383939826e-05,-4.493858913218184981e-05,-4.492162340918870456e-05,-4.490464430456642120e-05,-4.488765183779941381e-05,-4.487064602903823030e-05,-4.485362689793381819e-05,-4.483659446402610271e-05,-4.481954874774318753e-05,-4.480248976906908709e-05,-4.478541754798781582e-05,-4.476833210470543278e-05,-4.475123345976106393e-05,-4.473412163319423485e-05,-4.471699664532202689e-05,-4.469985851679458833e-05,-4.468270726837308970e-05,-4.466554292037461238e-05,-4.464836549356032691e-05,-4.463117500891344847e-05,-4.461397148708412530e-05,-4.459675494866699452e-05,-4.457952541520038281e-05,
        -4.456228290761199418e-05,-4.454502744677402148e-05,-4.452775905405825796e-05,-4.451047775044791877e-05,-4.449318355764786403e-05,-4.447587649702988699e-05,-4.445855658985475856e-05,-4.444122385749427195e-05,-4.442387832204186537e-05,-4.440652000492484319e-05,-4.438914892745948748e-05,-4.437176511223883679e-05,-4.435436858085672895e-05,-4.433695935462944604e-05,-4.431953745637207120e-05,-4.430210290801150919e-05,-4.428465573130813127e-05,-4.426719594896599830e-05,-4.424972358291201502e-05,-4.423223865540615307e-05,-4.421474118920798446e-05,-4.419723120668850314e-05,-4.417970873005216959e-05,-4.416217378228060042e-05,-4.414462638602234534e-05,-4.412706656370390945e-05,-4.410949433869548741e-05,-4.409190973347909548e-05,-4.407431277075879450e-05,-4.405670347396029030e-05,-4.403908186595417718e-05,-4.402144796972207175e-05,-4.400380180885621328e-05,-4.398614340628270725e-05,-4.396847278526072600e-05,-4.395078996943801997e-05,-4.393309498207376151e-05,-4.391538784664916761e-05,-4.389766858697852214e-05,-4.387993722654304207e-05,-4.386219378882394437e-05,-4.384443829763551292e-05,-4.382667077706958736e-05,-4.380889125105147386e-05,-4.379109974289585594e-05,-4.377329627713866245e-05,-4.375548087770519956e-05,-4.373765356852077346e-05,-4.371981437417682415e-05,-4.370196331870968010e-05,-4.368410042637771440e-05,-4.366622572160583360e-05,-4.364833922854138848e-05,-4.363044097183133019e-05,-4.361253097640016563e-05,-4.359460926622871213e-05,-4.357667586618596545e-05,-4.355873080125194363e-05,-4.354077409590706438e-05,-4.352280577507583459e-05,-4.350482586362725002e-05,-4.348683438626377296e-05,-4.346883136840951067e-05,-4.345081683509999237e-05,-4.343279081098216921e-05,-4.341475332164668188e-05,-4.339670439224008192e-05,-4.337864404796443196e-05,-4.336057231446588389e-05,-4.334248921639138885e-05,-4.332439477983118792e-05,-4.330628903054245527e-05,-4.328817199322765319e-05,-4.327004369392151162e-05,-4.325190415849222703e-05,-4.323375341264146243e-05,-4.321559148184883625e-05,-4.319741839209356726e-05,-4.317923416935487424e-05,-4.316103883927890905e-05,-4.314283242828897968e-05,-4.312461496236430492e-05,-4.310638646709552546e-05,-4.308814696918350506e-05,-4.306989649471848480e-05,-4.305163506967968345e-05,-4.303336272060143131e-05,-4.301507947390703634e-05,-4.299678535585327310e-05,-4.297848039314100532e-05,-4.296016461202700754e-05,-4.294183803882356543e-05,-4.292350070089767655e-05,-4.290515262445060429e-05,-4.288679383601667894e-05,-4.286842436262983114e-05,-4.285004423132399154e-05,-4.283165346852246813e-05,-4.281325210120368041e-05,-4.279484015651258133e-05,-4.277641766120554578e-05,-4.275798464276059363e-05,-4.273954112798961091e-05,-4.272108714403755059e-05,-4.270262271827141021e-05,-4.268414787772512042e-05,-4.266566264971016764e-05,-4.264716706176008287e-05,-4.262866114107533022e-05,-4.261014491518944070e-05,-4.259161841185798991e-05,-4.257308165805939737e-05,-4.255453468154923868e-05,-4.253597751013860062e-05,-4.251741017124999189e-05,-4.249883269275001041e-05,-4.248024510250525410e-05,-4.246164742804925396e-05,-4.244303969741514138e-05,-4.242442193841400311e-05,-4.240579417880141477e-05,-4.238715644688806350e-05,-4.236850877054054720e-05,-4.234985117768097496e-05,-4.233118369662003388e-05,-4.231250635533534421e-05,-4.229381918219310421e-05,-4.227512220539297871e-05,-4.225641545285707679e-05,-4.223769895334017477e-05,-4.221897273509744863e-05,-4.220023682627305206e-05,-4.218149125556625023e-05,-4.216273605162079718e-05,-4.214397124246982429e-05,-4.212519685675708558e-05,-4.210641292345940201e-05,-4.208761947105399415e-05,-4.206881652796257143e-05,-4.205000412299542134e-05,-4.203118228512936483e-05,-4.201235104311917823e-05,-4.199351042538657097e-05,-4.197466046129694206e-05,-4.195580117977160128e-05,-4.193693260917674692e-05,-4.191805477887777798e-05,-4.189916771796253769e-05,-4.188027145562989162e-05,-4.186136602041257149e-05,-4.184245144167597630e-05,-4.182352774922959426e-05,-4.180459497110655676e-05,-4.178565313711635199e-05,-4.176670227717949047e-05,-4.174774241955114817e-05,-4.172877359398530217e-05,-4.170979583001388491e-05,-4.169080915694678424e-05,-4.167181360387184341e-05,-4.165280920082059524e-05,-4.163379597710292757e-05,-4.161477396208423940e-05,-4.159574318551850780e-05,-4.157670367688215407e-05,-4.155765546592915527e-05,-4.153859858230246616e-05,-4.151953305586708609e-05,-4.150045891621045868e-05,-4.148137619303104984e-05,-4.146228491602732547e-05,-4.144318511534184069e-05,-4.142407682067306141e-05,-4.140496006194149814e-05,-4.138583486928970601e-05,-4.136670127236063976e-05,-4.134755930090827647e-05,-4.132840898540823815e-05,-4.130925035594756878e-05,-4.129008344244677886e-05,-4.127090827454882316e-05,-4.125172488284034600e-05,-4.123253329785248056e-05,-4.121333354933920390e-05,-4.119412566733204883e-05,-4.117490968202908164e-05,-4.115568562435001354e-05,-4.113645352410433276e-05,-4.111721341126806095e-05,-4.109796531709397627e-05,-4.107870927150258922e-05,-4.105944530452543262e-05,-4.104017344697119540e-05,-4.102089372931549960e-05,-4.100160618192294493e-05,-4.098231083538017572e-05,-4.096300772010730284e-05,-4.094369686680199294e-05,-4.092437830588435688e-05,-4.090505206816308359e-05,-4.088571818400277280e-05,-4.086637668387904654e-05,-4.084702759904468294e-05,-4.082767096003081519e-05,-4.080830679709102071e-05,-4.078893514125603303e-05,-4.076955602355658570e-05,-4.075016947446830073e-05,-4.073077552468884477e-05,-4.071137420530446249e-05,-4.069196554734588744e-05,-4.067254958106669704e-05,-4.065312633749762483e-05,-4.063369584794696010e-05,-4.061425814344543639e-05,-4.059481325435765342e-05,-4.057536121171434473e-05,-4.055590204687931077e-05,-4.053643579082777393e-05,-4.051696247431291198e-05,-4.049748212836545846e-05,-4.047799478429370268e-05,-4.045850047323940046e-05,-4.043899922584470730e-05,-4.041949107363995708e-05,-4.039997604776690565e-05,-4.038045417892321964e-05,-4.036092549886127756e-05,-4.034139003838976834e-05,-4.032184782887249241e-05,-4.030229890156222794e-05,-4.028274328721215269e-05,-4.026318101768566748e-05,-4.024361212451310621e-05,-4.022403663783702399e-05,-4.020445458952082163e-05,-4.018486601120585533e-05,-4.016527093397836978e-05,-4.014566938925767658e-05,-4.012606140813002042e-05,-4.010644702212573520e-05,-4.008682626255311021e-05,-4.006719916088696820e-05,-4.004756574860213192e-05,-4.002792605695137951e-05,-4.000828011729851141e-05,-3.998862796134039499e-05,-3.996896962049634183e-05,-3.994930512624117469e-05,-3.992963450977216056e-05,-3.990995780273065563e-05,-3.989027503659148266e-05,-3.987058624305150900e-05,-3.985089145347453510e-05,-3.983119069916885024e-05,-3.981148401238643331e-05,-3.979177142399148437e-05,-3.977205296551433733e-05,-3.975232866904043760e-05,-3.973259856576705218e-05,-3.971286268755758186e-05,-3.969312106566480480e-05,-3.967337373195212180e-05,-3.965362071789435561e-05,-3.963386205513286242e-05,-3.961409777536450960e-05,-3.959432791011963104e-05,-3.957455249109509410e-05,-3.955477154987674382e-05,-3.953498511838349216e-05,-3.951519322809016188e-05,-3.949539591058259802e-05,-3.947559319728011218e-05,-3.945578512049019437e-05,-3.943597171157664505e-05,-3.941615300212530926e-05,-3.939632902405509896e-05,-3.937649980878532574e-05,-3.935666538823490157e-05,-3.933682579421171610e-05,-3.931698105819059208e-05,-3.929713121197941916e-05,-3.927727628716404240e-05,-3.925741631544132915e-05,-3.923755132889672481e-05,-3.921768135922709675e-05,-3.919780643785175656e-05,-3.917792659624552698e-05,-3.915804186671589804e-05,-3.913815228107075939e-05,-3.911825787067391147e-05,-3.909835866705568819e-05,-3.907845470235704610e-05,-3.905854600844138602e-05,-3.903863261667250839e-05,-3.901871455891381402e-05,-3.899879186719523716e-05,-3.897886457288057827e-05,-3.895893270750017123e-05,-3.893899630325048378e-05,-3.891905539199491670e-05,-3.889911000520829276e-05,-3.887916017458747930e-05,-3.885920593182934368e-05,-3.883924730868626440e-05,-3.881928433718817573e-05,-3.879931704864336695e-05,-3.877934547502626117e-05,-3.875936964803372575e-05,-3.873938959936262805e-05,-3.871940536076534656e-05,-3.869941696377221518e-05,-3.867942444046867934e-05,-3.865942782227405061e-05,-3.863942714088519637e-05,-3.861942242805449510e-05,-3.859941371553432532e-05,-3.857940103513257668e-05,-3.855938441810202733e-05,-3.853936389652812267e-05,-3.851933950205221890e-05,-3.849931126614913879e-05,-3.847927922073779428e-05,-3.845924339712647466e-05,-3.843920382712306960e-05,-3.841916054247995760e-05,-3.839911357483849486e-05,-3.837906295589554873e-05,-3.835900871695940850e-05,-3.833895088978245269e-05,-3.831888950622808210e-05,-3.829882459749356372e-05,-3.827875619533127605e-05,-3.825868433154910875e-05,-3.823860903756637342e-05,-3.821853034469135935e-05,-3.819844828484297850e-05,-3.817836288955156476e-05,-3.815827419006989629e-05,-3.813818221820586274e-05,-3.811808700532326455e-05,-3.809798858317448023e-05,-3.807788698301228791e-05,-3.805778223636702151e-05,-3.803767437482452607e-05,-3.801756342969309088e-05,-3.799744943244753870e-05,-3.797733241472922572e-05,-3.795721240801297469e-05,-3.793708944321849685e-05,-3.791696355226470416e-05,-3.789683476673744167e-05,-3.787670311727886485e-05,-3.785656863553032991e-05,-3.783643135346625996e-05,-3.781629130195085509e-05,-3.779614851195933767e-05,-3.777600301529959737e-05,-3.775585484316890117e-05,-3.773570402704207183e-05,-3.771555059800535403e-05,-3.769539458731152592e-05,-3.767523602626887680e-05,-3.765507494601916250e-05,-3.763491137787067231e-05,-3.761474535302067324e-05,-3.759457690294398802e-05,-3.757440605839379444e-05,-3.755423285056735949e-05,-3.753405731099501708e-05,-3.751387947037443382e-05,-3.749369936001389902e-05,-3.747351701122170198e-05,-3.745333245452897586e-05,-3.743314572135503226e-05,-3.741295684300816049e-05,-3.739276585007500486e-05,-3.737257277380834353e-05,-3.735237764512788772e-05,-3.733218049484232637e-05,-3.731198135425994877e-05,-3.729178025430046617e-05,-3.727157722571705634e-05,-3.725137229920738591e-05,
        -3.723116550591321072e-05,-3.721095687686526432e-05,-3.719074644259467988e-05,-3.717053423407667978e-05,-3.715032028228648642e-05,-3.713010461775523297e-05,-3.710988727118058605e-05,-3.708966827370430153e-05,-3.706944765602404601e-05,-3.704922544828237463e-05,-3.702900168156553207e-05,-3.700877638662669611e-05,-3.698854959399699993e-05,-3.696832133442962132e-05,-3.694809163828915999e-05,-3.692786053632879373e-05,-3.690762805919067802e-05,-3.688739423718390142e-05,-3.686715910095061943e-05,-3.684692268107747637e-05,-3.682668500776253850e-05,-3.680644611181449477e-05,-3.678620602348692259e-05,-3.676596477308891053e-05,-3.674572239104056948e-05,-3.672547890781752145e-05,-3.670523435383987731e-05,-3.668498875925019220e-05,-3.666474215419102123e-05,-3.664449456902696411e-05,-3.662424603451119864e-05,-3.660399657995361267e-05,-3.658374623621840627e-05,-3.656349503322608996e-05,-3.654324300095268541e-05,-3.652299017020688154e-05,-3.650273657029856622e-05,-3.648248223148131686e-05,-3.646222718395319973e-05,-3.644197145763472534e-05,-3.642171508261293766e-05,-3.640145808903039182e-05,-3.638120050625248680e-05,-3.636094236458831119e-05,-3.634068369406939780e-05,-3.632042452389461218e-05,-3.630016488453957635e-05,-3.627990480553622277e-05,-3.625964431669403965e-05,-3.623938344815558210e-05,-3.621912222878664878e-05,-3.619886068895183939e-05,-3.617859885829410871e-05,-3.615833676590129997e-05,-3.613807444202699060e-05,-3.611781191592555729e-05,-3.609754921701791019e-05,-3.607728637516904868e-05,-3.605702341979988290e-05,-3.603676038005376725e-05,-3.601649728562916764e-05,-3.599623416600250536e-05,-3.597597105037264598e-05,-3.595570796827152193e-05,-3.593544494856493188e-05,-3.591518202122889747e-05,-3.589491921518472850e-05,-3.587465655918720131e-05,-3.585439408315682641e-05,-3.583413181601491360e-05,-3.581386978696032841e-05,-3.579360802469233604e-05,-3.577334655863184665e-05,-3.575308541797772577e-05,-3.573282463142923859e-05,-3.571256422818525067e-05,-3.569230423677849373e-05,-3.567204468640783332e-05,-3.565178560610560154e-05,-3.563152702451555243e-05,-3.561126897061450691e-05,-3.559101147310173019e-05,-3.557075456073199859e-05,-3.555049826214906616e-05,-3.553024260605219808e-05,-3.550998762114065954e-05,-3.548973333600269342e-05,-3.546947977939307606e-05,-3.544922697967800573e-05,-3.542897496527919188e-05,-3.540872376495141083e-05,-3.538847340717188317e-05,-3.536822392019578487e-05,-3.534797533233380307e-05,-3.532772767189662488e-05,-3.530748096763902666e-05,-3.528723524748311746e-05,-3.526699053940651751e-05,-3.524674687233053660e-05,-3.522650427412177265e-05,-3.520626277264682358e-05,-3.518602239627188766e-05,-3.516578317291907396e-05,-3.514554513095458077e-05,-3.512530829813398370e-05,-3.510507270243490296e-05,-3.508483837166842534e-05,-3.506460533375665989e-05,-3.504437361695478259e-05,-3.502414324862979100e-05,-3.500391425692583880e-05,-3.498368666937645699e-05,-3.496346051340415428e-05,-3.494323581720859551e-05,-3.492301260826780052e-05,-3.490279091405978917e-05,-3.488257076222911479e-05,-3.486235218042033068e-05,-3.484213519605594556e-05,-3.482191983622540121e-05,-3.480170612879529557e-05,-3.478149410107711503e-05,-3.476128378043785716e-05,-3.474107519391145260e-05,-3.472086836908694352e-05,-3.470066333299826056e-05,-3.468046011267933437e-05,-3.466025873571920712e-05,-3.464005922892976486e-05,-3.461986161956698282e-05,-3.459966593466479168e-05,-3.457947220075752170e-05,-3.455928044493461471e-05,-3.453909069484062400e-05,-3.451890297689885756e-05,-3.449871731747711223e-05,-3.447853374394238557e-05,-3.445835228305105247e-05,-3.443817296122642091e-05,-3.441799580533588809e-05,-3.439782084202480661e-05,-3.437764809788301790e-05,-3.435747759927831879e-05,-3.433730937313361764e-05,-3.431714344570568898e-05,-3.429697984325130733e-05,-3.427681859252684760e-05,-3.425665971967806200e-05,-3.423650325107274739e-05,-3.421634921291216713e-05,-3.419619763145309577e-05,-3.417604853317435243e-05,-3.415590194383311129e-05,-3.413575788979716918e-05,-3.411561639732330065e-05,-3.409547749178010179e-05,-3.407534119959088059e-05,-3.405520754690138929e-05,-3.403507655941329091e-05,-3.401494826277273731e-05,-3.399482268323650302e-05,-3.397469984650625108e-05,-3.395457977783955528e-05,-3.393446250332665670e-05,-3.391434804894677413e-05,-3.389423643967992561e-05,-3.387412770133879647e-05,-3.385402185973607203e-05,-3.383391893974074804e-05,-3.381381896749857674e-05,-3.379372196810059847e-05,-3.377362796663785360e-05,-3.375353698886751630e-05,-3.373344905982511577e-05,-3.371336420504578157e-05,-3.369328244934299832e-05,-3.367320381791882866e-05,-3.365312833597533526e-05,-3.363305602838151387e-05,-3.361298692072800520e-05,-3.359292103743971580e-05,-3.357285840333013027e-05,-3.355279904393437818e-05,-3.353274298339981030e-05,-3.351269024692848930e-05,-3.349264085944492209e-05,-3.347259484492992598e-05,-3.345255222886311941e-05,-3.343251303544736430e-05,-3.341247728927410066e-05,-3.339244501499027962e-05,-3.337241623690978543e-05,-3.335239097962405808e-05,-3.333236926761351526e-05,-3.331235112513653007e-05,-3.329233657639596444e-05,-3.327232564581672491e-05,-3.325231835749065112e-05,-3.323231473573162731e-05,-3.321231480463149310e-05,-3.319231858822657699e-05,-3.317232611060871861e-05,-3.315233739575873528e-05,-3.313235246765744435e-05,-3.311237135000810738e-05,-3.309239406690256402e-05,-3.307242064221060929e-05,-3.305245109957999361e-05,-3.303248546260295626e-05,-3.301252375498275882e-05,-3.299256600081124091e-05,-3.297261222351410836e-05,-3.295266244601746664e-05,-3.293271669213559960e-05,-3.291277498584932459e-05,-3.289283735008474707e-05,-3.287290380815655055e-05,-3.285297438354595201e-05,-3.283304909945661265e-05,-3.281312797920321600e-05,-3.279321104571186751e-05,-3.277329832229725071e-05,-3.275338983232956025e-05,-3.273348559817979009e-05,-3.271358564327364604e-05,-3.269368999070376702e-05,-3.267379866295216928e-05,-3.265391168305598057e-05,-3.263402907377477291e-05,-3.261415085797914060e-05,-3.259427705848416679e-05,-3.257440769749431198e-05,-3.255454279810221507e-05,-3.253468238267887003e-05,-3.251482647348424848e-05,-3.249497509361098935e-05,-3.247512826498599736e-05,-3.245528601009128877e-05,-3.243544835140887983e-05,-3.241561531114323103e-05,-3.239578691149880285e-05,-3.237596317462454465e-05,-3.235614412283593921e-05,-3.233632977811540243e-05,-3.231652016250086135e-05,-3.229671529791922069e-05,-3.227691520679698556e-05,-3.225711991050594918e-05,-3.223732943080648283e-05,-3.221754378984753586e-05,-3.219776300950050185e-05,-3.217798711141472978e-05,-3.215821611685099057e-05,-3.213845004784721127e-05,-3.211868892588620739e-05,-3.209893277261732791e-05,-3.207918160941236607e-05,-3.205943545736555933e-05,-3.203969433812625667e-05,-3.201995827345482937e-05,-3.200022728422347029e-05,-3.198050139147090576e-05,-3.196078061656892899e-05,-3.194106498088933321e-05,-3.192135450524880014e-05,-3.190164921096361184e-05,-3.188194911879493887e-05,-3.186225424950395180e-05,-3.184256462462897730e-05,-3.182288026443158557e-05,-3.180320118989499178e-05,-3.178352742200241110e-05,-3.176385898107092487e-05,-3.174419588808374826e-05,-3.172453816358000722e-05,-3.170488582793229426e-05,-3.168523890195729109e-05,-3.166559740586105676e-05,-3.164596136007169491e-05,-3.162633078535037612e-05,-3.160670570162560367e-05,-3.158708612915894776e-05,-3.156747208832300089e-05,-3.154786359915728866e-05,-3.152826068203440357e-05,-3.150866335682733776e-05,-3.148907164363112798e-05,-3.146948556248529982e-05,-3.144990513309631197e-05,-3.143033037567022348e-05,-3.141076130991349302e-05,-3.139119795542155700e-05,-3.137164033212291869e-05,-3.135208845972403680e-05,-3.133254235770932539e-05,-3.131300204595177661e-05,-3.129346754382478224e-05,-3.127393887047968946e-05,-3.125441604590051270e-05,-3.123489908957166605e-05,-3.121538802014489633e-05,-3.119588285732666222e-05,-3.117638362054586665e-05,-3.115689032884283449e-05,-3.113740300125789062e-05,-3.111792165688687106e-05,-3.109844631504765644e-05,-3.107897699439199357e-05,-3.105951371401571848e-05,-3.104005649301466718e-05,-3.102060535015160880e-05,-3.100116030391175670e-05,-3.098172137355748035e-05,-3.096228857757399311e-05,-3.094286193450201949e-05,-3.092344146293779517e-05,-3.090402718164408924e-05,-3.088461910910611508e-05,-3.086521726369806373e-05,-3.084582166396065972e-05,-3.082643232815707179e-05,-3.080704927460597986e-05,-3.078767252151504152e-05,-3.076830208748049245e-05,-3.074893798998834527e-05,-3.072958024741279104e-05,-3.071022887862762119e-05,-3.069088390056373683e-05,-3.067154533142879558e-05,-3.065221318976352194e-05,-3.063288749283188395e-05,-3.061356825856398345e-05,-3.059425550500094459e-05,-3.057494924968429117e-05,-3.055564951021105813e-05,-3.053635630423379155e-05,-3.051706964957157098e-05,-3.049778956348836445e-05,-3.047851606341467345e-05,-3.045924916739162214e-05,-3.043998889218357817e-05,-3.042073525516553190e-05,-3.040148827376798479e-05,-3.038224796531041605e-05,-3.036301434689026024e-05,-3.034378743554944080e-05,-3.032456724871845921e-05,-3.030535380299514969e-05,-3.028614711553245797e-05,-3.026694720348332979e-05,-3.024775408361213280e-05,-3.022856777268323469e-05,-3.020938828762753658e-05,-3.019021564482082809e-05,-3.017104986136054379e-05,-3.015189095417758480e-05,-3.013273893920365154e-05,-3.011359383320311167e-05,-3.009445565299584402e-05,-3.007532441468008244e-05,-3.005620013463161655e-05,-3.003708282955930287e-05,-3.001797251578341985e-05,-2.999886920918015676e-05,-2.997977292590325860e-05,-2.996068368255055958e-05,-2.994160149505376012e-05,-2.992252637945558291e-05,-2.990345835190977297e-05,-2.988439742812598610e-05,-2.986534362409143384e-05,-2.984629695590435006e-05,-2.982725743916336825e-05,-2.980822508968916651e-05,-2.978919992335793410e-05,-2.977018195543523760e-05,-2.975117120179726626e-05,-2.973216767854225395e-05,-2.971317140038065574e-05,-2.969418238313314973e-05,-2.967520064239836941e-05,-2.965622619310881447e-05,-2.963725905125169646e-05,-2.961829923142644816e-05,
        -2.959934674917619191e-05,-2.958040162004405005e-05,-2.956146385846292191e-05,-2.954253348030899673e-05,-2.952361050018170729e-05,-2.950469493273599753e-05,-2.948578679362601207e-05,-2.946688609745118370e-05,-2.944799285908850095e-05,-2.942910709324841889e-05,-2.941022881469690375e-05,-2.939135803836645522e-05,-2.937249477868997261e-05,-2.935363905021137754e-05,-2.933479086775214739e-05,-2.931595024596722610e-05,-2.929711719884542376e-05,-2.927829174104168430e-05,-2.925947388687788475e-05,-2.924066365067590212e-05,-2.922186104697965803e-05,-2.920306608983347374e-05,-2.918427879322615937e-05,-2.916549917159061422e-05,-2.914672723891564843e-05,-2.912796300896802748e-05,-2.910920649634718416e-05,-2.909045771459783936e-05,-2.907171667737573628e-05,-2.905298339894724080e-05,-2.903425789291258496e-05,-2.901554017359364579e-05,-2.899683025386901036e-05,-2.897812814733891074e-05,-2.895943386849175738e-05,-2.894074743031715968e-05,-2.892206884641534970e-05,-2.890339813022002602e-05,-2.888473529516488725e-05,-2.886608035485016543e-05,-2.884743332232098112e-05,-2.882879421078898829e-05,-2.881016303385441901e-05,-2.879153980467341611e-05,-2.877292453606905553e-05,-2.875431724103094666e-05,-2.873571793321483270e-05,-2.871712662533276728e-05,-2.869854332998578172e-05,-2.867996806049655234e-05,-2.866140082952162160e-05,-2.864284164999508775e-05,-2.862429053485104902e-05,-2.860574749646849213e-05,-2.858721254755947072e-05,-2.856868570094706072e-05,-2.855016696901024886e-05,-2.853165636440557762e-05,-2.851315389973407832e-05,-2.849465958720820424e-05,-2.847617343920694211e-05,-2.845769546822030094e-05,-2.843922568673828977e-05,-2.842076410664029495e-05,-2.840231074019428092e-05,-2.838386559977923440e-05,-2.836542869749658635e-05,-2.834700004505918969e-05,-2.832857965473500883e-05,-2.831016753873649705e-05,-2.829176370855446265e-05,-2.827336817645687006e-05,-2.825498095421208333e-05,-2.823660205336642193e-05,-2.821823148579927221e-05,-2.819986926333450938e-05,-2.818151539746294176e-05,-2.816316989989742225e-05,-2.814483278207324801e-05,-2.812650405581429425e-05,-2.810818373255585811e-05,-2.808987182328914756e-05,-2.807156833972701548e-05,-2.805327329297169214e-05,-2.803498669490256390e-05,-2.801670855634430524e-05,-2.799843888845465756e-05,-2.798017770250238456e-05,-2.796192500931216074e-05,-2.794368082059683900e-05,-2.792544514695904923e-05,-2.790721799944551051e-05,-2.788899938860334160e-05,-2.787078932581232849e-05,-2.785258782217470142e-05,-2.783439488784900107e-05,-2.781621053382643538e-05,-2.779803477082065655e-05,-2.777986760982287251e-05,-2.776170906088060164e-05,-2.774355913459647383e-05,-2.772541784212823046e-05,-2.770728519374543453e-05,-2.768916119944009324e-05,-2.767104586981483649e-05,-2.765293921558331647e-05,-2.763484124668202924e-05,-2.761675197354707123e-05,-2.759867140622596082e-05,-2.758059955504377214e-05,-2.756253643010353471e-05,-2.754448204161930036e-05,-2.752643639952756516e-05,-2.750839951359829172e-05,-2.749037139437859878e-05,-2.747235205119435975e-05,-2.745434149409309299e-05,-2.743633973312231689e-05,-2.741834677799648290e-05,-2.740036263820799789e-05,-2.738238732341580217e-05,-2.736442084377843642e-05,-2.734646320878830750e-05,-2.732851442760475535e-05,-2.731057450988672031e-05,-2.729264346540416497e-05,-2.727472130331642930e-05,-2.725680803317143130e-05,-2.723890366418402209e-05,-2.722100820584660852e-05,-2.720312166731853054e-05,-2.718524405787015041e-05,-2.716737538671631924e-05,-2.714951566257228777e-05,-2.713166489531904091e-05,-2.711382309356080711e-05,-2.709599026601283711e-05,-2.707816642227856008e-05,-2.706035157101771560e-05,-2.704254572105657672e-05,-2.702474888111039419e-05,-2.700696106000544106e-05,-2.698918226701207956e-05,-2.697141251040147125e-05,-2.695365179872233341e-05,-2.693590014107849484e-05,-2.691815754568560592e-05,-2.690042402087033935e-05,-2.688269957579203506e-05,-2.686498421861083230e-05,-2.684727795770891490e-05,-2.682958080146846669e-05,-2.681189275816064921e-05,-2.679421383627866859e-05,-2.677654404387164178e-05,-2.675888338948828604e-05,-2.674123188101118487e-05,-2.672358952665598864e-05,-2.670595633441630312e-05,-2.668833231250777871e-05,-2.667071746903504348e-05,-2.665311181165863630e-05,-2.663551534837216295e-05,-2.661792808722474035e-05,-2.660035003609895199e-05,-2.658278120265533673e-05,-2.656522159466545574e-05,-2.654767121990087020e-05,-2.653013008607763012e-05,-2.651259820068974093e-05,-2.649507557173080841e-05,-2.647756220630625990e-05,-2.646005811213214542e-05,-2.644256329692451502e-05,-2.642507776773328487e-05,-2.640760153210797156e-05,-2.639013459743155821e-05,-2.637267697119805021e-05,-2.635522866023531918e-05,-2.633778967231492629e-05,-2.632036001448678775e-05,-2.630293969346775285e-05,-2.628552871702938276e-05,-2.626812709188852679e-05,-2.625073482526163460e-05,-2.623335192419862238e-05,-2.621597839541633945e-05,-2.619861424596470201e-05,-2.618125948267158165e-05,-2.616391411275342804e-05,-2.614657814248300127e-05,-2.612925157896572870e-05,-2.611193442897397077e-05,-2.609462669889150988e-05,-2.607732839582377338e-05,-2.606003952582147676e-05,-2.604276009582351392e-05,-2.602549011271326762e-05,-2.600822958220838643e-05,-2.599097851108123081e-05,-2.597373690610416119e-05,-2.595650477349442653e-05,-2.593928211946927576e-05,-2.592206895013493551e-05,-2.590486527215274393e-05,-2.588767109162892766e-05,-2.587048641455869102e-05,-2.585331124749234988e-05,-2.583614559642510855e-05,-2.581898946729666022e-05,-2.580184286599118693e-05,-2.578470579878144875e-05,-2.576757827188469463e-05,-2.575046029068550624e-05,-2.573335186134562136e-05,-2.571625298986024433e-05,-2.569916368172497911e-05,-2.568208394287951890e-05,-2.566501377915253457e-05,-2.564795319626167469e-05,-2.563090219981356555e-05,-2.561386079530381110e-05,-2.559682898856108224e-05,-2.557980678502547178e-05,-2.556279419019258370e-05,-2.554579120955802196e-05,-2.552879784850636824e-05,-2.551181411275527111e-05,-2.549484000752277879e-05,-2.547787553791591719e-05,-2.546092070959682374e-05,-2.544397552739496859e-05,-2.542703999697248918e-05,-2.541011412338090025e-05,-2.539319791183825004e-05,-2.537629136745156444e-05,-2.535939449488378017e-05,-2.534250729990805695e-05,-2.532562978702079803e-05,-2.530876196144005164e-05,-2.529190382805079906e-05,-2.527505539162699932e-05,-2.525821665716465603e-05,-2.524138762977079509e-05,-2.522456831377528630e-05,-2.520775871389657752e-05,-2.519095883546373926e-05,-2.517416868275113018e-05,-2.515738826058822042e-05,-2.514061757347141324e-05,-2.512385662600813419e-05,-2.510710542258376421e-05,-2.509036396797226232e-05,-2.507363226694758751e-05,-2.505691032328449808e-05,-2.504019814186797532e-05,-2.502349572702788905e-05,-2.500680308281655329e-05,-2.499012021378588244e-05,-2.497344712432125746e-05,-2.495678381853050354e-05,-2.494013030057695701e-05,-2.492348657479048768e-05,-2.490685264511238728e-05,-2.489022851598354791e-05,-2.487361419112321670e-05,-2.485700967441717424e-05,-2.484041497058386838e-05,-2.482383008306499050e-05,-2.480725501585734349e-05,-2.479068977290221909e-05,-2.477413435830744248e-05,-2.475758877573674965e-05,-2.474105302846529852e-05,-2.472452712098949235e-05,-2.470801105697306710e-05,-2.469150484002424761e-05,-2.467500847364023642e-05,-2.465852196159579179e-05,-2.464204530749913857e-05,-2.462557851490299043e-05,-2.460912158752659451e-05,-2.459267452864510872e-05,-2.457623734153369099e-05,-2.455981003013363306e-05,-2.454339259777560400e-05,-2.452698504762373943e-05,-2.451058738300870843e-05,-2.449419960731669121e-05,-2.447782172387835686e-05,-2.446145373580232985e-05,-2.444509564686336844e-05,-2.442874745961498562e-05,-2.441240917722131698e-05,-2.439608080306854276e-05,-2.437976234004324283e-05,-2.436345379153159740e-05,-2.434715516025365289e-05,-2.433086644948456723e-05,-2.431458766177785336e-05,-2.429831880040866920e-05,-2.428205986848563924e-05,-2.426581086800716491e-05,-2.424957180263698220e-05,-2.423334267509513751e-05,-2.421712348754656574e-05,-2.420091424354398058e-05,-2.418471494530782806e-05,-2.416852559566917691e-05,-2.415234619740358468e-05,-2.413617675262047513e-05,-2.412001726442847271e-05,-2.410386773538109040e-05,-2.408772816797632998e-05,-2.407159856432361522e-05,-2.405547892714299252e-05,-2.403936925915450828e-05,-2.402326956252309742e-05,-2.400717983930267252e-05,-2.399110009237981345e-05,-2.397503032386394395e-05,-2.395897053586448777e-05,-2.394292073104598018e-05,-2.392688091129580030e-05,-2.391085107911194996e-05,-2.389483123627078598e-05,-2.387882138499275442e-05,-2.386282152772034593e-05,-2.384683166622991735e-05,-2.383085180274191472e-05,-2.381488193936576181e-05,-2.379892207771128199e-05,-2.378297221994341015e-05,-2.376703236817157006e-05,-2.375110252406109623e-05,-2.373518268949936783e-05,-2.371927286637376397e-05,-2.370337305690473073e-05,-2.368748326236902457e-05,-2.367160348454300234e-05,-2.365573372536955432e-05,-2.363987398640299276e-05,-2.362402426969723024e-05,-2.360818457636248979e-05,-2.359235490806410596e-05,-2.357653526685599132e-05,-2.356072565407041353e-05,-2.354492607115066249e-05,-2.352913651948451701e-05,-2.351335700101486736e-05,-2.349758751685193658e-05,-2.348182806821697000e-05,-2.346607865683081329e-05,-2.345033928402573409e-05,-2.343460995135604463e-05,-2.341889065965441219e-05,-2.340318141041963784e-05,-2.338748220487296692e-05,-2.337179304429115589e-05,-2.335611393011749470e-05,-2.334044486307362831e-05,-2.332478584454733550e-05,-2.330913687581537275e-05,-2.329349795754387387e-05,-2.327786909100959534e-05,-2.326225027754480479e-05,-2.324664151787114719e-05,-2.323104281276577865e-05,-2.321545416356096680e-05,-2.319987557114489007e-05,-2.318430703618368227e-05,-2.316874855989858872e-05,-2.315320014306676555e-05,-2.313766178618781311e-05,-2.312213349020542097e-05,-2.310661525617430101e-05,-2.309110708453854244e-05,-2.307560897624183482e-05,-2.306012093178377853e-05,-2.304464295205255198e-05,-2.302917503743673322e-05,-2.301371718854694493e-05,-2.299826940638238781e-05,-2.298283169110959534e-05,
        -2.296740404328367902e-05,-2.295198646362628381e-05,-2.293657895274803238e-05,-2.292118151070443588e-05,-2.290579413832816158e-05,-2.289041683561920948e-05,-2.287504960313269109e-05,-2.285969244159025138e-05,-2.284434535071433459e-05,-2.282900833133760798e-05,-2.281368138373762733e-05,-2.279836450774785916e-05,-2.278305770386790385e-05,-2.276776097237531715e-05,-2.275247431343663251e-05,-2.273719772716287224e-05,-2.272193121355403633e-05,-2.270667477305421400e-05,-2.269142840560789409e-05,-2.267619211121507661e-05,-2.266096588987576155e-05,-2.264574974170097121e-05,-2.263054366696826136e-05,-2.261534766523354278e-05,-2.260016173649681548e-05,-2.258498588081359060e-05,-2.256982009790631238e-05,-2.255466438805253659e-05,-2.253951875080817402e-05,-2.252438318589566890e-05,-2.250925769331502124e-05,-2.249414227284418644e-05,-2.247903692431663103e-05,-2.246394164717724351e-05,-2.244885644131500158e-05,-2.243378130639683832e-05,-2.241871624208968683e-05,-2.240366124828252481e-05,-2.238861632430921844e-05,-2.237358146955914506e-05,-2.235855668414332698e-05,-2.234354196745114152e-05,-2.232853731864992142e-05,-2.231354273751762207e-05,-2.229855822355464312e-05,-2.228358377631689535e-05,-2.226861939502722265e-05,-2.225366507913051350e-05,-2.223872082829370100e-05,-2.222378664162860673e-05,-2.220886251830256342e-05,-2.219394845798250415e-05,-2.217904446005780628e-05,-2.216415052330722446e-05,-2.214926664745320295e-05,-2.213439283177409678e-05,-2.211952907504866062e-05,-2.210467537655524950e-05,-2.208983173590528537e-05,-2.207499815221058981e-05,-2.206017462386133943e-05,-2.204536115063548962e-05,-2.203055773175588428e-05,-2.201576436577923346e-05,-2.200098105203940335e-05,-2.198620778914861518e-05,-2.197144457655175742e-05,-2.195669141347167397e-05,-2.194194829835405258e-05,-2.192721523014418139e-05,-2.191249220806490428e-05,-2.189777923106150936e-05,-2.188307629735763982e-05,-2.186838340634267297e-05,-2.185370055707291925e-05,-2.183902774782753298e-05,-2.182436497771833572e-05,-2.180971224552408216e-05,-2.179506954974597122e-05,-2.178043688932929101e-05,-2.176581426294177390e-05,-2.175120166925115228e-05,-2.173659910714720311e-05,-2.172200657485356956e-05,-2.170742407092696169e-05,-2.169285159470124569e-05,-2.167828914462210932e-05,-2.166373671846910653e-05,-2.164919431535405892e-05,-2.163466193388918768e-05,-2.162013957229813599e-05,-2.160562722947068082e-05,-2.159112490351944302e-05,-2.157663259289011037e-05,-2.156215029641694869e-05,-2.154767801226808999e-05,-2.153321573855615512e-05,-2.151876347400438760e-05,-2.150432121672540831e-05,-2.148988896533143844e-05,-2.147546671821265463e-05,-2.146105447331514426e-05,-2.144665222897357282e-05,-2.143225998363362805e-05,-2.141787773546344198e-05,-2.140350548263114661e-05,-2.138914322347140740e-05,-2.137479095598582290e-05,-2.136044867856456975e-05,-2.134611638915373533e-05,-2.133179408597696280e-05,-2.131748176720238419e-05,-2.130317943060955344e-05,-2.128888707458864715e-05,-2.127460469747433081e-05,-2.126033229693513604e-05,-2.124606987041754991e-05,-2.123181741686686053e-05,-2.121757493400711070e-05,-2.120334241950683207e-05,-2.118911987175620126e-05,-2.117490728842374992e-05,-2.116070466740005429e-05,-2.114651200652017948e-05,-2.113232930384123520e-05,-2.111815655736482000e-05,-2.110399376453742093e-05,-2.108984092336063654e-05,-2.107569803155850963e-05,-2.106156508696610530e-05,-2.104744208791808902e-05,-2.103332903125032516e-05,-2.101922591518645689e-05,-2.100513273728399355e-05,-2.099104949532248909e-05,-2.097697618719251977e-05,-2.096291281017403918e-05,-2.094885936221313472e-05,-2.093481584081180458e-05,-2.092078224358306926e-05,-2.090675856819546041e-05,-2.089274481242853199e-05,-2.087874097339570412e-05,-2.086474704882101960e-05,-2.085076303659505470e-05,-2.083678893366469609e-05,-2.082282473797603117e-05,-2.080887044703105815e-05,-2.079492605783217485e-05,-2.078099156815893522e-05,-2.076706697545782632e-05,-2.075315227728635747e-05,-2.073924747081345998e-05,-2.072535255326357628e-05,-2.071146752258279378e-05,-2.069759237549595454e-05,-2.068372710978261253e-05,-2.066987172288925478e-05,-2.065602621120765647e-05,-2.064219057307248306e-05,-2.062836480554164353e-05,-2.061454890550651342e-05,-2.060074287046909092e-05,-2.058694669782035191e-05,-2.057316038445167194e-05,-2.055938392736544884e-05,-2.054561732439674770e-05,-2.053186057249245522e-05,-2.051811366859945807e-05,-2.050437660994219868e-05,-2.049064939357858606e-05,-2.047693201689959608e-05,-2.046322447685211543e-05,-2.044952677027200849e-05,-2.043583889466127346e-05,-2.042216084685577471e-05,-2.040849262363586547e-05,-2.039483422250354394e-05,-2.038118564029467450e-05,-2.036754687373409922e-05,-2.035391792010177170e-05,-2.034029877628906746e-05,-2.032668943946491780e-05,-2.031308990624314248e-05,-2.029950017340409474e-05,-2.028592023817221701e-05,-2.027235009732786253e-05,-2.025878974792894027e-05,-2.024523918658927002e-05,-2.023169841025573845e-05,-2.021816741593074340e-05,-2.020464620022810465e-05,-2.019113475981715311e-05,-2.017763309181130893e-05,-2.016414119293541418e-05,-2.015065905963675519e-05,-2.013718668902875208e-05,-2.012372407778073580e-05,-2.011027122261754840e-05,-2.009682812020852083e-05,-2.008339476711196170e-05,-2.006997116021924654e-05,-2.005655729636623974e-05,-2.004315317172267186e-05,-2.002975878317991842e-05,-2.001637412768486612e-05,-2.000299920124071207e-05,-1.998963400090536524e-05,-1.997627852318162311e-05,-1.996293276440574971e-05,-1.994959672141360940e-05,-1.993627039098555542e-05,-1.992295376923580719e-05,-1.990964685261165101e-05,-1.989634963800446243e-05,-1.988306212175050547e-05,-1.986978430029706644e-05,-1.985651617036898742e-05,-1.984325772808048782e-05,-1.983000897004538743e-05,-1.981676989276648371e-05,-1.980354049235799607e-05,-1.979032076582232236e-05,-1.977711070927368198e-05,-1.976391031882629434e-05,-1.975071959098295693e-05,-1.973753852230197836e-05,-1.972436710928615611e-05,-1.971120534782766498e-05,-1.969805323431828015e-05,-1.968491076570488829e-05,-1.967177793754659731e-05,-1.965865474612416008e-05,-1.964554118832895213e-05,-1.963243726027519287e-05,-1.961934295785505711e-05,-1.960625827718276426e-05,-1.959318321514968986e-05,-1.958011776803658677e-05,-1.956706193129154059e-05,-1.955401570119530419e-05,-1.954097907475027540e-05,-1.952795204746005098e-05,-1.951493461538333918e-05,-1.950192677535600438e-05,-1.948892852304817680e-05,-1.947593985457407584e-05,-1.946296076615894322e-05,-1.944999125375046489e-05,-1.943703131385143834e-05,-1.942408094235403837e-05,-1.941114013498390634e-05,-1.939820888813281741e-05,-1.938528719802601330e-05,-1.937237506050015767e-05,-1.935947247155844764e-05,-1.934657942709305800e-05,-1.933369592327371933e-05,-1.932082195632567334e-05,-1.930795752208558369e-05,-1.929510261644562519e-05,-1.928225723518695034e-05,-1.926942137453480086e-05,-1.925659503076992962e-05,-1.924377819922939992e-05,-1.923097087574987540e-05,-1.921817305688966471e-05,-1.920538473831889803e-05,-1.919260591576321673e-05,-1.917983658522581791e-05,-1.916707674259887639e-05,-1.915432638344150007e-05,-1.914158550386790836e-05,-1.912885410004783182e-05,-1.911613216748486721e-05,-1.910341970196016703e-05,-1.909071669925488379e-05,-1.907802315531670345e-05,-1.906533906609331197e-05,-1.905266442714381725e-05,-1.903999923413834949e-05,-1.902734348308010581e-05,-1.901469716963921641e-05,-1.900206028943030034e-05,-1.898943283873411048e-05,-1.897681481255464320e-05,-1.896420620706162907e-05,-1.895160701792519831e-05,-1.893901724048241419e-05,-1.892643687101402961e-05,-1.891386590485710784e-05,-1.890130433768177909e-05,-1.888875216532470702e-05,-1.887620938306744378e-05,-1.886367598691318648e-05,-1.885115197275410992e-05,-1.883863733548318820e-05,-1.882613207110361842e-05,-1.881363617539655309e-05,-1.880114964369905550e-05,-1.878867247195881163e-05,-1.877620465523532900e-05,-1.876374618919873782e-05,-1.875129706996325751e-05,-1.873885729269941791e-05,-1.872642685296632692e-05,-1.871400574610104783e-05,-1.870159396794024431e-05,-1.868919151398751310e-05,-1.867679837969093981e-05,-1.866441456072065463e-05,-1.865204005196963166e-05,-1.863967484955209031e-05,-1.862731894913816078e-05,-1.861497234556530600e-05,-1.860263503455916734e-05,-1.859030701156783039e-05,-1.857798827203938075e-05,-1.856567881142190402e-05,-1.855337862538553040e-05,-1.854108770904527859e-05,-1.852880605762718957e-05,-1.851653366713446047e-05,-1.850427053262659882e-05,-1.849201664944066792e-05,-1.847977201324679797e-05,-1.846753661904898536e-05,-1.845531046218429339e-05,-1.844309353848938571e-05,-1.843088584319030332e-05,-1.841868737134655376e-05,-1.840649811851724493e-05,-1.839431807987290668e-05,-1.838214725075060230e-05,-1.836998562654290623e-05,-1.835783320291994869e-05,-1.834568997455265915e-05,-1.833355593700014552e-05,-1.832143108582151569e-05,-1.830931541568769916e-05,-1.829720892254638187e-05,-1.828511160123502677e-05,-1.827302344703518600e-05,-1.826094445533943400e-05,-1.824887462126278947e-05,-1.823681394014231572e-05,-1.822476240725956487e-05,-1.821272001784057792e-05,-1.820068676688935128e-05,-1.818866264968743707e-05,-1.817664766157189860e-05,-1.816464179765775455e-05,-1.815264505328206823e-05,-1.814065742350434718e-05,-1.812867890321756548e-05,-1.811670948786980873e-05,-1.810474917274262907e-05,-1.809279795284002290e-05,-1.808085582338803121e-05,-1.806892277933513924e-05,-1.805699881601841028e-05,-1.804508392860837418e-05,-1.803317811194249387e-05,-1.802128136135783265e-05,-1.800939367185838691e-05,-1.799751503867019764e-05,-1.798564545696379469e-05,-1.797378492174317444e-05,-1.796193342767926637e-05,-1.795009097033117840e-05,-1.793825754475841805e-05,-1.792643314602049287e-05,-1.791461776917691040e-05,-1.790281140878757782e-05,-1.789101406035609187e-05,-1.787922571905298241e-05,-1.786744637977122352e-05,-1.785567603734827813e-05,-1.784391468712120954e-05,-1.783216232387196953e-05,-1.782041894266006565e-05,-1.780868453854500544e-05,-1.779695910636425182e-05,-1.778524264139935696e-05,
        -1.777353513859880607e-05,-1.776183659263352865e-05,-1.775014699867405454e-05,-1.773846635183540243e-05,-1.772679464706605756e-05,-1.771513187909246057e-05,-1.770347804286309668e-05,-1.769183313371502919e-05,-1.768019714604163184e-05,-1.766857007517996792e-05,-1.765695191618954496e-05,-1.764534266368578130e-05,-1.763374231267267334e-05,-1.762215085787666169e-05,-1.761056829446827621e-05,-1.759899461750702443e-05,-1.758742982171934699e-05,-1.757587390183168452e-05,-1.756432685273701111e-05,-1.755278866971687890e-05,-1.754125934716466162e-05,-1.752973888041742256e-05,-1.751822726414609122e-05,-1.750672449291057475e-05,-1.749523056226998108e-05,-1.748374546634012816e-05,-1.747226920029154584e-05,-1.746080175940578627e-05,-1.744934313757662281e-05,-1.743789333036316336e-05,-1.742645233271389316e-05,-1.741502013880014132e-05,-1.740359674379243771e-05,-1.739218214275028984e-05,-1.738077633012258261e-05,-1.736937930096882354e-05,-1.735799105012647559e-05,-1.734661157182237901e-05,-1.733524086139359710e-05,-1.732387891373310396e-05,-1.731252572323427330e-05,-1.730118128506763497e-05,-1.728984559379309616e-05,-1.727851864391505288e-05,-1.726720043082607958e-05,-1.725589094875301654e-05,-1.724459019286639361e-05,-1.723329815761509565e-05,-1.722201483772556330e-05,-1.721074022847934870e-05,-1.719947432382573638e-05,-1.718821711882423386e-05,-1.717696860870088216e-05,-1.716572878779354383e-05,-1.715449765077314837e-05,-1.714327519219960294e-05,-1.713206140707690395e-05,-1.712085629018700317e-05,-1.710965983597878548e-05,-1.709847203945624727e-05,-1.708729289501276227e-05,-1.707612239754130457e-05,-1.706496054187933709e-05,-1.705380732247574471e-05,-1.704266273389043462e-05,-1.703152677112740321e-05,-1.702039942891309110e-05,-1.700928070180740548e-05,-1.699817058425923122e-05,-1.698706907110603126e-05,-1.697597615724077968e-05,-1.696489183694582792e-05,-1.695381610516966120e-05,-1.694274895641667555e-05,-1.693169038563535622e-05,-1.692064038710805463e-05,-1.690959895539467794e-05,-1.689856608566575602e-05,-1.688754177237017373e-05,-1.687652600967926020e-05,-1.686551879254150066e-05,-1.685452011607191380e-05,-1.684352997427529530e-05,-1.683254836187808579e-05,-1.682157527366223704e-05,-1.681061070413214509e-05,-1.679965464801425057e-05,-1.678870709986846066e-05,-1.677776805425468254e-05,-1.676683750589935684e-05,-1.675591544919585729e-05,-1.674500187892613567e-05,-1.673409678965009917e-05,-1.672320017598316610e-05,-1.671231203242973251e-05,-1.670143235377175017e-05,-1.669056113440259281e-05,-1.667969836888216761e-05,-1.666884405199242636e-05,-1.665799817807123162e-05,-1.664716074167849058e-05,-1.663633173776268848e-05,-1.662551116099475479e-05,-1.661469900504641828e-05,-1.660389526503269764e-05,-1.659309993579105580e-05,-1.658231301171486649e-05,-1.657153448703096998e-05,-1.656076435646580691e-05,-1.655000261480132906e-05,-1.653924925620886555e-05,-1.652850427574792391e-05,-1.651776766753432213e-05,-1.650703942612796737e-05,-1.649631954625530028e-05,-1.648560802230969458e-05,-1.647490484901759089e-05,-1.646421002060582950e-05,-1.645352353174533988e-05,-1.644284537721807382e-05,-1.643217555119536044e-05,-1.642151404845915152e-05,-1.641086086340282080e-05,-1.640021599025320853e-05,-1.638957942401431112e-05,-1.637895115902399112e-05,-1.636833118973113343e-05,-1.635771951097320098e-05,-1.634711611681050059e-05,-1.633652100169191712e-05,-1.632593416062144698e-05,-1.631535558765939697e-05,-1.630478527742118544e-05,-1.629422322468876416e-05,-1.628366942368897341e-05,-1.627312386875967576e-05,-1.626258655473833414e-05,-1.625205747618485574e-05,-1.624153662699301393e-05,-1.623102400222231623e-05,-1.622051959615511407e-05,-1.621002340312927004e-05,-1.619953541787122475e-05,-1.618905563482986310e-05,-1.617858404850958109e-05,-1.616812065319273017e-05,-1.615766544360575097e-05,-1.614721841375343914e-05,-1.613677955852876877e-05,-1.612634887254715821e-05,-1.611592634981340311e-05,-1.610551198510945525e-05,-1.609510577271766607e-05,-1.608470770730896504e-05,-1.607431778333223704e-05,-1.606393599501432234e-05,-1.605356233685961698e-05,-1.604319680337251697e-05,-1.603283938939048525e-05,-1.602249008886280635e-05,-1.601214889623836513e-05,-1.600181580629911338e-05,-1.599149081327189137e-05,-1.598117391171660628e-05,-1.597086509597112070e-05,-1.596056436053983063e-05,-1.595027169998264327e-05,-1.593998710869293234e-05,-1.592971058100856041e-05,-1.591944211137841236e-05,-1.590918169425137307e-05,-1.589892932407632742e-05,-1.588868499546869373e-05,-1.587844870259980112e-05,-1.586822044014057909e-05,-1.585800020253991249e-05,-1.584778798385810816e-05,-1.583758377865507327e-05,-1.582738758182378191e-05,-1.581719938720249630e-05,-1.580701918951765705e-05,-1.579684698344019367e-05,-1.578668276297490181e-05,-1.577652652251515519e-05,-1.576637825662086101e-05,-1.575623795996294874e-05,-1.574610562671274749e-05,-1.573598125143016446e-05,-1.572586482850857337e-05,-1.571575635217481448e-05,-1.570565581693328383e-05,-1.569556321734388860e-05,-1.568547854780000250e-05,-1.567540180269499928e-05,-1.566533297636674149e-05,-1.565527206337513633e-05,-1.564521905783600175e-05,-1.563517395453128955e-05,-1.562513674785437345e-05,-1.561510743203209373e-05,-1.560508600140231295e-05,-1.559507245041391599e-05,-1.558506677407089924e-05,-1.557506896621152492e-05,-1.556507902095161100e-05,-1.555509693346168731e-05,-1.554512269785757184e-05,-1.553515630825508254e-05,-1.552519775932514889e-05,-1.551524704562767809e-05,-1.550530416116746579e-05,-1.549536910050441918e-05,-1.548544185842049004e-05,-1.547552242892047403e-05,-1.546561080673081179e-05,-1.545570698580078783e-05,-1.544581096080133165e-05,-1.543592272640337271e-05,-1.542604227638966208e-05,-1.541616960565317385e-05,-1.540630470864279289e-05,-1.539644757952984833e-05,-1.538659821259669158e-05,-1.537675660256976329e-05,-1.536692274384243717e-05,-1.535709663075257581e-05,-1.534727825730497486e-05,-1.533746761833709726e-05,-1.532766470874191711e-05,-1.531786952196911855e-05,-1.530808205257860877e-05,-1.529830229551887300e-05,-1.528853024501675151e-05,-1.527876589524357342e-05,-1.526900924075924593e-05,-1.525926027579060928e-05,-1.524951899467552607e-05,-1.523978539247350383e-05,-1.523005946307831593e-05,-1.522034120093884724e-05,-1.521063060050398263e-05,-1.520092765594505124e-05,-1.519123236215502715e-05,-1.518154471330523947e-05,-1.517186470390008424e-05,-1.516219232766680136e-05,-1.515252757988694299e-05,-1.514287045489837169e-05,-1.513322094648383853e-05,-1.512357904981387335e-05,-1.511394475861571607e-05,-1.510431806756029616e-05,-1.509469897131854310e-05,-1.508508746406178602e-05,-1.507548354018339865e-05,-1.506588719402124354e-05,-1.505629842030176135e-05,-1.504671721291872544e-05,-1.503714356659857643e-05,-1.502757747606775496e-05,-1.501801893538656785e-05,-1.500846793883736652e-05,-1.499892448086903585e-05,-1.498938855626352762e-05,-1.497986015908114865e-05,-1.497033928382629497e-05,-1.496082592500336261e-05,-1.495132007695021414e-05,-1.494182173411573444e-05,-1.493233089094880839e-05,-1.492284754156525395e-05,-1.491337168074702291e-05,-1.490390330305402244e-05,-1.489444240232451477e-05,-1.488498897328494053e-05,-1.487554301049520689e-05,-1.486610450818215412e-05,-1.485667346085017826e-05,-1.484724986294816418e-05,-1.483783370898050791e-05,-1.482842499311853857e-05,-1.481902370975562988e-05,-1.480962985372924479e-05,-1.480024341915520125e-05,-1.479086440048238416e-05,-1.478149279215967837e-05,-1.477212858874699108e-05,-1.476277178452667371e-05,-1.475342237372556653e-05,-1.474408035112562132e-05,-1.473474571117572296e-05,-1.472541844804720057e-05,-1.471609855618893903e-05,-1.470678603049391242e-05,-1.469748086491140526e-05,-1.468818305377928013e-05,-1.467889259199051111e-05,-1.466960947360540501e-05,-1.466033369329489133e-05,-1.465106524561887724e-05,-1.464180412463766956e-05,-1.463255032485566431e-05,-1.462330384116583559e-05,-1.461406466746195676e-05,-1.460483279813740154e-05,-1.459560822836269978e-05,-1.458639095197611368e-05,-1.457718096353755044e-05,-1.456797825755140607e-05,-1.455878282846656546e-05,-1.454959467078742463e-05,-1.454041377874082386e-05,-1.453124014721973722e-05,-1.452207377011793810e-05,-1.451291464216186711e-05,-1.450376275802245374e-05,-1.449461811187102711e-05,-1.448548069815647210e-05,-1.447635051149420704e-05,-1.446722754627760565e-05,-1.445811179706657512e-05,-1.444900325819897802e-05,-1.443990192401267691e-05,-1.443080778945615705e-05,-1.442172084836768065e-05,-1.441264109552919948e-05,-1.440356852544510957e-05,-1.439450313250878466e-05,-1.438544491155768767e-05,-1.437639385654110313e-05,-1.436734996201893821e-05,-1.435831322255110010e-05,-1.434928363269749596e-05,-1.434026118707354414e-05,-1.433124587990608489e-05,-1.432223770558849196e-05,-1.431323665879169482e-05,-1.430424273402008950e-05,-1.429525592561153857e-05,-1.428627622845901612e-05,-1.427730363673385128e-05,-1.426833814444083970e-05,-1.425937974680602238e-05,-1.425042843838930651e-05,-1.424148421302895429e-05,-1.423254706556242866e-05,-1.422361699066065910e-05,-1.421469398271701934e-05,-1.420577803612488310e-05,-1.419686914538864642e-05,-1.418796730529026107e-05,-1.417907250977901157e-05,-1.417018475363684971e-05,-1.416130403175674957e-05,-1.415243033808799566e-05,-1.414356366746805094e-05,-1.413470401467886717e-05,-1.412585137350319542e-05,-1.411700573844543172e-05,-1.410816710500917281e-05,-1.409933546697716977e-05,-1.409051081885381862e-05,-1.408169315558760459e-05,-1.407288247134985681e-05,-1.406407876075599361e-05,-1.405528201808836641e-05,-1.404649223818443815e-05,-1.403770941571513831e-05,-1.402893354512935176e-05,-1.402016462054289647e-05,-1.401140263701527999e-05,-1.400264758882885374e-05,-1.399389947026596914e-05,-1.398515827660817834e-05,-1.397642400191578815e-05,-1.396769664041563885e-05,-1.395897618711172683e-05,-1.395026263684151502e-05,-1.394155598355428793e-05,-1.393285622186546391e-05,-1.392416334666801703e-05,-1.391547735241083217e-05,-1.390679823365381651e-05,-1.389812598462381033e-05,
        -1.388946060021378770e-05,-1.388080207492814466e-05,-1.387215040338229954e-05,-1.386350558013615952e-05,-1.385486759969412063e-05,-1.384623645667160119e-05,-1.383761214562850839e-05,-1.382899466129128285e-05,-1.382038399777574256e-05,-1.381178014991935044e-05,-1.380318311272610288e-05,-1.379459288031181785e-05,-1.378600944723640254e-05,-1.377743280839283102e-05,-1.376886295795243242e-05,-1.376029989097471429e-05,-1.375174360163100573e-05,-1.374319408475876969e-05,-1.373465133513995795e-05,-1.372611534694589963e-05,-1.371758611501405767e-05,-1.370906363384882809e-05,-1.370054789834318498e-05,-1.369203890283499092e-05,-1.368353664182864193e-05,-1.367504111010608980e-05,-1.366655230256030862e-05,-1.365807021336262750e-05,-1.364959483707295362e-05,-1.364112616863977223e-05,-1.363266420262299050e-05,-1.362420893374904907e-05,-1.361576035657785511e-05,-1.360731846544727119e-05,-1.359888325513924912e-05,-1.359045472043574065e-05,-1.358203285595216414e-05,-1.357361765624842675e-05,-1.356520911588443568e-05,-1.355680722986418729e-05,-1.354841199252554418e-05,-1.354002339837290236e-05,-1.353164144218821363e-05,-1.352326611869791861e-05,-1.351489742285050255e-05,-1.350653534892831686e-05,-1.349817989149126873e-05,-1.348983104537682109e-05,-1.348148880525590343e-05,-1.347315316591046752e-05,-1.346482412178939825e-05,-1.345650166756362509e-05,-1.344818579801509983e-05,-1.343987650787026311e-05,-1.343157379191106671e-05,-1.342327764436435089e-05,-1.341498806017860090e-05,-1.340670503430230198e-05,-1.339842856079576094e-05,-1.339015863482950763e-05,-1.338189525129651614e-05,-1.337363840436811557e-05,-1.336538808899279118e-05,-1.335714429973045014e-05,-1.334890703119651079e-05,-1.334067627867252526e-05,-1.333245203638533383e-05,-1.332423429900586598e-05,-1.331602306137158465e-05,-1.330781831815341931e-05,-1.329962006402229946e-05,-1.329142829409324378e-05,-1.328324300281513715e-05,-1.327506418447033099e-05,-1.326689183439588859e-05,-1.325872594704069485e-05,-1.325056651713119038e-05,-1.324241353961586043e-05,-1.323426700916563448e-05,-1.322612692022939740e-05,-1.321799326770012328e-05,-1.320986604658180852e-05,-1.320174525126782683e-05,-1.319363087676217461e-05,-1.318552291779129249e-05,-1.317742136880406534e-05,-1.316932622469346725e-05,-1.316123748046349462e-05,-1.315315513067405462e-05,-1.314507917005158788e-05,-1.313700959360009080e-05,-1.312894639587947054e-05,-1.312088957150514545e-05,-1.311283911553662307e-05,-1.310479502281136632e-05,-1.309675728777826009e-05,-1.308872590533027847e-05,-1.308070087052692898e-05,-1.307268217792811882e-05,-1.306466982203824401e-05,-1.305666379802783439e-05,-1.304866410073435290e-05,-1.304067072505077363e-05,-1.303268366542598145e-05,-1.302470291647539469e-05,-1.301672847353607665e-05,-1.300876033111242336e-05,-1.300079848415292005e-05,-1.299284292755054082e-05,-1.298489365580968169e-05,-1.297695066387882790e-05,-1.296901394676197583e-05,-1.296108349924107728e-05,-1.295315931582052826e-05,-1.294524139155983633e-05,-1.293732972157402017e-05,-1.292942430003440890e-05,-1.292152512216704352e-05,-1.291363218319796502e-05,-1.290574547729850252e-05,-1.289786499952816357e-05,-1.288999074483543339e-05,-1.288212270805777493e-05,-1.287426088397713997e-05,-1.286640526731996914e-05,-1.285855585336781459e-05,-1.285071263668058350e-05,-1.284287561204022765e-05,-1.284608059914926770e-05,-1.287138624966877032e-05,-1.290779097345984283e-05,-1.294427033754486089e-05,-1.298082431339109277e-05,-1.301745286991229378e-05,-1.305415597635528613e-05,-1.309093359980195714e-05,-1.312778570505823694e-05,-1.316471225681903334e-05,-1.320171321766983041e-05,-1.323878854853077769e-05,-1.327593820932282398e-05,-1.331316215835709471e-05,-1.335046035255693653e-05,-1.338783274690280578e-05,-1.342527929520942465e-05,-1.346279994995924767e-05,-1.350039466163632795e-05,-1.353806337967000673e-05,-1.357580605199082413e-05,-1.361362262425336311e-05,-1.365151304139056165e-05,-1.368947724655900089e-05,-1.372751518080583821e-05,-1.376562678423454145e-05,-1.380381199533875503e-05,-1.384207075111332230e-05,-1.388040298627712943e-05,-1.391880863399475032e-05,-1.395728762682013624e-05,-1.399583989503128123e-05,-1.403446536679675560e-05,-1.407316396956348470e-05,-1.411193562866897011e-05,-1.415078026750782314e-05,-1.418969780841994321e-05,-1.422868817169131717e-05,-1.426775127583157499e-05,-1.430688703796256789e-05,-1.434609537337427909e-05,-1.438537619574686843e-05,-1.442472941665107200e-05,-1.446415494632535825e-05,-1.450365269356490572e-05,-1.454322256455586881e-05,-1.458286446398560088e-05,-1.462257829559776567e-05,-1.466236396063802516e-05,-1.470222135846466216e-05,-1.474215038704818070e-05,-1.478215094224966109e-05,-1.482222291848689366e-05,-1.486236620812375619e-05,-1.490258070185879191e-05,-1.494286628822560914e-05,-1.498322285425901512e-05,-1.502365028543950487e-05,-1.506414846430548238e-05,-1.510471727228512862e-05,-1.514535658952986807e-05,-1.518606629347107884e-05,-1.522684625959724869e-05,-1.526769636195357549e-05,-1.530861647253134450e-05,-1.534960646115690608e-05,-1.539066619615780951e-05,-1.543179554391871378e-05,-1.547299436832627606e-05,-1.551426253193488591e-05,-1.555559989518950914e-05,-1.559700631648119895e-05,-1.563848165214709596e-05,-1.568002575658145048e-05,-1.572163848290175636e-05,-1.576331968122790528e-05,-1.580506919979240976e-05,-1.584688688555102587e-05,-1.588877258273946325e-05,-1.593072613387258585e-05,-1.597274737991094540e-05,-1.601483615870646915e-05,-1.605699230633472752e-05,-1.609921565776106789e-05,-1.614150604528630240e-05,-1.618386329871324136e-05,-1.622628724612384943e-05,-1.626877771349066748e-05,-1.631133452523192418e-05,-1.635395750304580176e-05,-1.639664646618799182e-05,-1.643940123241538487e-05,-1.648222161743095882e-05,-1.652510743455071207e-05,-1.656805849487019699e-05,-1.661107460770860911e-05,-1.665415557966509752e-05,-1.669730121556245450e-05,-1.674051131789200397e-05,-1.678378568725769071e-05,-1.682712412187647999e-05,-1.687052641768937988e-05,-1.691399236847246357e-05,-1.695752176561482472e-05,-1.700111439900675592e-05,-1.704477005587401450e-05,-1.708848852083333369e-05,-1.713226957666957873e-05,-1.717611300383614648e-05,-1.722001858056598778e-05,-1.726398608309365201e-05,-1.730801528493364216e-05,-1.735210595749103746e-05,-1.739625786989495992e-05,-1.744047078922061900e-05,-1.748474447993420000e-05,-1.752907870433695336e-05,-1.757347322256519462e-05,-1.761792779214621518e-05,-1.766244216810930467e-05,-1.770701610387392932e-05,-1.775164935041706471e-05,-1.779634165560706194e-05,-1.784109276570244873e-05,-1.788590242440823985e-05,-1.793077037265389251e-05,-1.797569634970352936e-05,-1.802068009243429358e-05,-1.806572133461470386e-05,-1.811081980790385515e-05,-1.815597524190692980e-05,-1.820118736423070871e-05,-1.824645589887374797e-05,-1.829178056805824681e-05,-1.833716109200800304e-05,-1.838259718744961191e-05,-1.842808857016597912e-05,-1.847363495238729669e-05,-1.851923604390126599e-05,-1.856489155288576498e-05,-1.861060118402146912e-05,-1.865636464049025278e-05,-1.870218162275394391e-05,-1.874805182811023485e-05,-1.879397495246903915e-05,-1.883995068829857900e-05,-1.888597872601316396e-05,-1.893205875413972450e-05,-1.897819045759696621e-05,-1.902437351958274903e-05,-1.907060762035284185e-05,-1.911689243805358984e-05,-1.916322764816680291e-05,-1.920961292350975569e-05,-1.925604793451274332e-05,-1.930253234888601455e-05,-1.934906583250795009e-05,-1.939564804787075047e-05,-1.944227865502412556e-05,-1.948895731229693951e-05,-1.953568367452085397e-05,-1.958245739447361800e-05,-1.962927812243497883e-05,-1.967614550552054808e-05,-1.972305918906958055e-05,-1.977001881564577346e-05,-1.981702402464868840e-05,-1.986407445370153013e-05,-1.991116973765194587e-05,-1.995830950834998063e-05,-2.000549339542523342e-05,-2.005272102606481255e-05,-2.009999202445822419e-05,-2.014730601213043926e-05,-2.019466260883007180e-05,-2.024206143091955568e-05,-2.028950209209678945e-05,-2.033698420433882603e-05,-2.038450737623653808e-05,-2.043207121366075185e-05,-2.047967532053940332e-05,-2.052731929746975936e-05,-2.057500274321721889e-05,-2.062272525332753403e-05,-2.067048642084845511e-05,-2.071828583649626410e-05,-2.076612308776759619e-05,-2.081399776004966284e-05,-2.086190943584309565e-05,-2.090985769526154670e-05,-2.095784211597617741e-05,-2.100586227227196900e-05,-2.105391773604692318e-05,-2.110200807703410675e-05,-2.115013286213551780e-05,-2.119829165531106341e-05,-2.124648401818918231e-05,-2.129470950973377796e-05,-2.134296768607768513e-05,-2.139125810091124791e-05,-2.143958030509374169e-05,-2.148793384693092889e-05,-2.153631827217505901e-05,-2.158473312380282394e-05,-2.163317794257046955e-05,-2.168165226607010609e-05,-2.173015562900726394e-05,-2.177868756431111663e-05,-2.182724760163567979e-05,-2.187583526813696722e-05,-2.192445008858401323e-05,-2.197309158485927227e-05,-2.202175927595861893e-05,-2.207045267876850403e-05,-2.211917130734430970e-05,-2.216791467274381588e-05,-2.221668228408191226e-05,-2.226547364742037516e-05,-2.231428826576786761e-05,-2.236312564046771811e-05,-2.241198526964360838e-05,-2.246086664858815141e-05,-2.250976927065106992e-05,-2.255869262607346215e-05,-2.260763620248740224e-05,-2.265659948497145137e-05,-2.270558195610616892e-05,-2.275458309586309014e-05,-2.280360238116063698e-05,-2.285263928719638571e-05,-2.290169328583724351e-05,-2.295076384617456000e-05,-2.299985043541230567e-05,-2.304895251814542689e-05,-2.309806955580473442e-05,-2.314720100698997030e-05,-2.319634632869105317e-05,-2.324550497495581070e-05,-2.329467639711202409e-05,-2.334386004393396163e-05,-2.339305536142033404e-05,-2.344226179340491711e-05,-2.349147878133450718e-05,-2.354070576338074261e-05,-2.358994217577237151e-05,-2.363918745179605096e-05,-2.368844102235145854e-05,-2.373770231633987038e-05,-2.378697075938740468e-05,-2.383624577484422247e-05,-2.388552678378452754e-05,-2.393481320417389924e-05,-2.398410445197951546e-05,-2.403339994083708575e-05,-2.408269908138471749e-05,-2.413200128215109430e-05,
        -2.418130594916689802e-05,-2.423061248563174175e-05,-2.427992029269132601e-05,-2.432922876888232722e-05,-2.437853731013239766e-05,-2.442784531031527706e-05,-2.447715216058465870e-05,-2.452645724976276753e-05,-2.457575996439587129e-05,-2.462505968792161326e-05,-2.467435580238985793e-05,-2.472364768696388992e-05,-2.477293471808694747e-05,-2.482221627059244540e-05,-2.487149171648272983e-05,-2.492076042537316738e-05,-2.497002176449214517e-05,-2.501927509929169346e-05,-2.506851979178215117e-05,-2.511775520275261186e-05,-2.516698069038314500e-05,-2.521619561024479594e-05,-2.526539931602123090e-05,-2.531459115850953623e-05,-2.536377048678595258e-05,-2.541293664776178574e-05,-2.546208898590585079e-05,-2.551122684268936069e-05,-2.556034955869534997e-05,-2.560945647162027328e-05,-2.565854691677360577e-05,-2.570762022774397693e-05,-2.575667573556650325e-05,-2.580571276927789981e-05,-2.585473065530585757e-05,-2.590372871891233331e-05,-2.595270628275025970e-05,-2.600166266653047842e-05,-2.605059718907565269e-05,-2.609950916671044396e-05,-2.614839791315048956e-05,-2.619726274100120378e-05,-2.624610295992590991e-05,-2.629491787792259672e-05,-2.634370680132391840e-05,-2.639246903352043816e-05,-2.644120387657045157e-05,-2.648991063081140851e-05,-2.653858859397173475e-05,-2.658723706194798808e-05,-2.663585532913792520e-05,-2.668444268721925638e-05,-2.673299842664844661e-05,-2.678152183593907054e-05,-2.683001220155079025e-05,-2.687846880777833292e-05,-2.692689093725109117e-05,-2.697527787104414543e-05,-2.702362888834519694e-05,-2.707194326667661244e-05,-2.712022028067417878e-05,-2.716845920430754902e-05,-2.721665930977001935e-05,-2.726481986742301800e-05,-2.731294014535201597e-05,-2.736101941003266091e-05,-2.740905692738548893e-05,-2.745705196027792283e-05,-2.750500377035614008e-05,-2.755291161793405053e-05,-2.760077476132716257e-05,-2.764859245768525042e-05,-2.769636396221519803e-05,-2.774408852823651017e-05,-2.779176540823602437e-05,-2.783939385314626591e-05,-2.788697311190135864e-05,-2.793450243210315875e-05,-2.798198105979921024e-05,-2.802940823976030060e-05,-2.807678321548046085e-05,-2.812410522867736518e-05,-2.817137352012499818e-05,-2.821858732848792073e-05,-2.826574589137598181e-05,-2.831284844601045236e-05,-2.835989422711460151e-05,-2.840688246802391959e-05,-2.845381240168531889e-05,-2.850068325965793292e-05,-2.854749427178004950e-05,-2.859424466644666651e-05,-2.864093367155318148e-05,-2.868756051399579121e-05,-2.873412441894984681e-05,-2.878062461031394292e-05,-2.882706031120951806e-05,-2.887343074409187693e-05,-2.891973512963996740e-05,-2.896597268770007005e-05,-2.901214263734130938e-05,-2.905824419635605338e-05,-2.910427658203706969e-05,-2.915023901012281371e-05,-2.919613069535254013e-05,-2.924195085263203708e-05,-2.928769869492420241e-05,-2.933337343463682245e-05,-2.937897428356706087e-05,-2.942450045229083599e-05,-2.946995115105099927e-05,-2.951532558903569026e-05,-2.956062297471140354e-05,-2.960584251604503336e-05,-2.965098342028182898e-05,-2.969604489383437240e-05,-2.974102614239360065e-05,-2.978592637137289501e-05,-2.983074478540848062e-05,-2.987548058852596000e-05,-2.992013298414031297e-05,-2.996470117572203051e-05,-3.000918436552035828e-05,-3.005358175556249734e-05,-3.009789254826422678e-05,-3.014211594398741312e-05,-3.018625114364903439e-05,-3.023029734822157977e-05,-3.027425375762282656e-05,-3.031811957188157436e-05,-3.036189399086008933e-05,-3.040557621358797036e-05,-3.044916543903930517e-05,-3.049266086641022611e-05,-3.053606169439726514e-05,-3.057936712175246541e-05,-3.062257634706133658e-05,-3.066568856857632142e-05,-3.070870298443884039e-05,-3.075161879345644778e-05,-3.079443519365954174e-05,-3.083715138341158735e-05,-3.087976656140911658e-05,-3.092227992573803874e-05,-3.096469067515039697e-05,-3.100699800823170094e-05,-3.104920112412257183e-05,-3.109129922174158622e-05,-3.113329150034038761e-05,-3.117517715972573100e-05,-3.121695539914925988e-05,-3.125862541919488535e-05,-3.130018642039100740e-05,-3.134163760304398139e-05,-3.138297816890345260e-05,-3.142420731933048827e-05,-3.146532425635228947e-05,-3.150632818271770219e-05,-3.154721830139761707e-05,-3.158799381591803623e-05,-3.162865393047109563e-05,-3.166919784991506504e-05,-3.170962477949679226e-05,-3.174993392551783700e-05,-3.179012449444629240e-05,-3.183019569397149695e-05,-3.187014673222687833e-05,-3.190997681801199803e-05,-3.194968516156970750e-05,-3.198927097275428011e-05,-3.202873346369594643e-05,-3.206807184674698163e-05,-3.210728533459272782e-05,-3.214637314230550658e-05,-3.218533448490212834e-05,-3.222416857839860427e-05,-3.226287464042076891e-05,-3.230145188920507948e-05,-3.233989954409821621e-05,-3.237821682605668272e-05,-3.241640295698067220e-05,-3.245445715965855626e-05,-3.249237865837750761e-05,-3.253016667875696655e-05,-3.256782044774864104e-05,-3.260533919330343977e-05,-3.264272214481556134e-05,-3.267996853362209464e-05,-3.271707759178177355e-05,-3.275404855279662186e-05,-3.279088065250013173e-05,-3.282757312722539567e-05,-3.286412521524839647e-05,-3.290053615712107415e-05,-3.293680519367292447e-05,-3.297293156812042270e-05,-3.300891452528986747e-05,-3.304475331167289198e-05,-3.308044717603708662e-05,-3.311599536759413098e-05,-3.315139713838677338e-05,-3.318665174240065241e-05,-3.322175843462060740e-05,-3.325671647264050179e-05,-3.329152511566402239e-05,-3.332618362511530208e-05,-3.336069126408380825e-05,-3.339504729765740976e-05,-3.342925099319993265e-05,-3.346330162024013788e-05,-3.349719845013865438e-05,-3.353094075619900138e-05,-3.356452781461127799e-05,-3.359795890317540668e-05,-3.363123330180073367e-05,-3.366435029339420737e-05,-3.369730916258362186e-05,-3.373010919621721726e-05,-3.376274968386328013e-05,-3.379522991764360995e-05,-3.382754919162289653e-05,-3.385970680219729800e-05,-3.389170204864955238e-05,-3.392353423270488832e-05,-3.395520265864204745e-05,-3.398670663351532895e-05,-3.401804546632192228e-05,-3.404921846911213024e-05,-3.408022495676732433e-05,-3.411106424638932211e-05,-3.414173565857714365e-05,-3.417223851598372164e-05,-3.420257214403754631e-05,-3.423273587127573236e-05,-3.426272902928850783e-05,-3.429255095221961369e-05,-3.432220097687732618e-05,-3.435167844356712408e-05,-3.438098269514799910e-05,-3.441011307786512319e-05,-3.443906894068371471e-05,-3.446784963534454960e-05,-3.449645451736316204e-05,-3.452488294508615496e-05,-3.455313427969119999e-05,-3.458120788596419359e-05,-3.460910313174414554e-05,-3.463681938803420124e-05,-3.466435602911266400e-05,-3.469171243258850623e-05,-3.471888797934585824e-05,-3.474588205365503057e-05,-3.477269404345006976e-05,-3.479932333955160217e-05,-3.482576933616643444e-05,-3.485203143177573182e-05,-3.487810902774723942e-05,-3.490400152866834915e-05,-3.492970834312325579e-05,-3.495522888341540124e-05,-3.498056256528991881e-05,-3.500570880782261085e-05,-3.503066703430812723e-05,-3.505543667103871996e-05,-3.508001714830344397e-05,-3.510440790094326857e-05,-3.512860836629716488e-05,-3.515261798558988460e-05,-3.517643620515320535e-05,-3.520006247431650692e-05,-3.522349624557330472e-05,-3.524673697635760661e-05,-3.526978412804471219e-05,-3.529263716522956784e-05,-3.531529555705903434e-05,-3.533775877667677534e-05,-3.536002630061263474e-05,-3.538209761011490428e-05,-3.540397219037316745e-05,-3.542564953068483291e-05,-3.544712912439962338e-05,-3.546841046880855330e-05,-3.548949306553250693e-05,-3.551037642085530521e-05,-3.553106004500206083e-05,-3.555154345180611131e-05,-3.557182616004128661e-05,-3.559190769292230883e-05,-3.561178757749416945e-05,-3.563146534524275211e-05,-3.565094053248341055e-05,-3.567021267930625683e-05,-3.568928133051985085e-05,-3.570814603542915577e-05,-3.572680634766900454e-05,-3.574526182537063335e-05,-3.576351203110617050e-05,-3.578155653194414754e-05,-3.579939489972705502e-05,-3.581702671096032020e-05,-3.583445154609066208e-05,-3.585166899089387016e-05,-3.586867863519804800e-05,-3.588548007377179161e-05,-3.590207290654623407e-05,-3.591845673700522212e-05,-3.593463117423922881e-05,-3.595059583166859696e-05,-3.596635032748762839e-05,-3.598189428505316201e-05,-3.599722733194088420e-05,-3.601234910088901842e-05,-3.602725922963179173e-05,-3.604195735995574523e-05,-3.605644313936506862e-05,-3.607071621991586596e-05,-3.608477625816064460e-05,-3.609862291625853814e-05,-3.611225586114263919e-05,-3.612567476463102167e-05,-3.613887930281611816e-05,-3.615186915756352093e-05,-3.616464401590135935e-05,-3.617720356913212143e-05,-3.618954751410941029e-05,-3.620167555251629921e-05,-3.621358739108737623e-05,-3.622528274199732223e-05,-3.623676132197273247e-05,-3.624802285279171699e-05,-3.625906706200554552e-05,-3.626989368210598030e-05,-3.628050244997016449e-05,-3.629089310858146789e-05,-3.630106540569721929e-05,-3.631101909423728458e-05,-3.632075393217304438e-05,-3.633026968291597214e-05,-3.633956611531763414e-05,-3.634864300283702221e-05,-3.635750012481731019e-05,-3.636613726537563096e-05,-3.637455421423574364e-05,-3.638275076617292214e-05,-3.639072672106946627e-05,-3.639848188480288016e-05,-3.640601606796911582e-05,-3.641332908643768462e-05,-3.642042076168472420e-05,-3.642729092057095386e-05,-3.643393939517514113e-05,-3.644036602279410175e-05,-3.644657064622025544e-05,-3.645255311368611473e-05,-3.645831327847570691e-05,-3.646385099981275246e-05,-3.646916614208350893e-05,-3.647425857444819286e-05,-3.647912817250631434e-05,-3.648377481690889823e-05,-3.648819839335848414e-05,-3.649239879294219335e-05,-3.649637591301990724e-05,-3.650012965561444389e-05,-3.650365992835524764e-05,-3.650696664447838913e-05,-3.651004972238247603e-05,-3.651290908623927578e-05,-3.651554466549411515e-05,-3.651795639536548066e-05,-3.652014421595684013e-05,-3.652210807325584341e-05,-3.652384791880125547e-05,-3.652536370934988952e-05,-3.652665540693211810e-05,-3.652772297968454041e-05,-3.652856640096180385e-05,-3.652918564928109291e-05,-3.652958070887724062e-05,-3.652975156948068403e-05,-3.652969822665053101e-05,-3.652942068083087079e-05,-3.652891893834997461e-05,-3.652819301058762846e-05,
        -3.652724291475228924e-05,-3.652606867377006239e-05,-3.652467031567407929e-05,-3.652304787393756413e-05,-3.652120138764036739e-05,-3.651913090146896579e-05,-3.651683646538339545e-05,-3.651431813500582990e-05,-3.651157597106546859e-05,-3.650861004017569300e-05,-3.650542041416793282e-05,-3.650200717053575517e-05,-3.649837039215730883e-05,-3.649451016712879081e-05,-3.649042658915302439e-05,-3.648611975742843683e-05,-3.648158977653803703e-05,-3.647683675678248250e-05,-3.647186081329190088e-05,-3.646666206691406842e-05,-3.646124064421440991e-05,-3.645559667669884263e-05,-3.644973030142439896e-05,-3.644364166111024872e-05,-3.643733090369360994e-05,-3.643079818183014851e-05,-3.642404365455931270e-05,-3.641706748613859901e-05,-3.640986984548844063e-05,-3.640245090735794165e-05,-3.639481085176976549e-05,-3.638694986429769074e-05,-3.637886813556701071e-05,-3.637056586142106696e-05,-3.636204324286573808e-05,-3.635330048701312933e-05,-3.634433780569379380e-05,-3.633515541523468784e-05,-3.632575353890166170e-05,-3.631613240384634622e-05,-3.630629224299353197e-05,-3.629623329437503543e-05,-3.628595580118521013e-05,-3.627546001233605821e-05,-3.626474618095842928e-05,-3.625381456590082152e-05,-3.624266543122978135e-05,-3.623129904639643684e-05,-3.621971568529280816e-05,-3.620791562697345256e-05,-3.619589915654364276e-05,-3.618366656332749898e-05,-3.617121814153412274e-05,-3.615855419153435335e-05,-3.614567501791787762e-05,-3.613258092982629677e-05,-3.611927224245192747e-05,-3.610574927576104542e-05,-3.609201235410530728e-05,-3.607806180733197365e-05,-3.606389797017328647e-05,-3.604952118202442435e-05,-3.603493178755412529e-05,-3.602013013614957515e-05,-3.600511658180538532e-05,-3.598989148362319312e-05,-3.597445520597819524e-05,-3.595880811751994699e-05,-3.594295059161645156e-05,-3.592688300674273805e-05,-3.591060574631432800e-05,-3.589411919757701241e-05,-3.587742375360525315e-05,-3.586051981191440419e-05,-3.584340777385008892e-05,-3.582608804675313507e-05,-3.580856104185015099e-05,-3.579082717475312592e-05,-3.577288686612556390e-05,-3.575474054146043912e-05,-3.573638863008099520e-05,-3.571783156625096822e-05,-3.569906978917458673e-05,-3.568010374177532640e-05,-3.566093387180613306e-05,-3.564156063184942269e-05,-3.562198447815134728e-05,-3.560220587178752893e-05,-3.558222527866305995e-05,-3.556204316834676860e-05,-3.554166001473735292e-05,-3.552107629656298116e-05,-3.550029249665964670e-05,-3.547930910186014586e-05,-3.545812660399327854e-05,-3.543674549816300257e-05,-3.541516628408070133e-05,-3.539338946573211686e-05,-3.537141555115530522e-05,-3.534924505288472574e-05,-3.532687848689652910e-05,-3.530431637344122464e-05,-3.528155923687714690e-05,-3.525860760633658941e-05,-3.523546201367189212e-05,-3.521212299523179823e-05,-3.518859109197247648e-05,-3.516486684768116433e-05,-3.514095081064150250e-05,-3.511684353346700149e-05,-3.509254557149121823e-05,-3.506805748415553481e-05,-3.504337983578631466e-05,-3.501851319342996760e-05,-3.499345812818521750e-05,-3.496821521442594616e-05,-3.494278503052283824e-05,-3.491716815945400398e-05,-3.489136518608493276e-05,-3.486537670005507294e-05,-3.483920329427903084e-05,-3.481284556489105952e-05,-3.478630411224425956e-05,-3.475957953991137828e-05,-3.473267245424072058e-05,-3.470558346574392772e-05,-3.467831318865188805e-05,-3.465086223941593602e-05,-3.462323123876176467e-05,-3.459542081085675846e-05,-3.456743158231079249e-05,-3.453926418361952244e-05,-3.451091924866478422e-05,-3.448239741360437094e-05,-3.445369931859287860e-05,-3.442482560678250536e-05,-3.439577692471162962e-05,-3.436655392147214272e-05,-3.433715724887598242e-05,-3.430758756278740051e-05,-3.427784552167967291e-05,-3.424793178669061078e-05,-3.421784702212216089e-05,-3.418759189505182761e-05,-3.415716707561022858e-05,-3.412657323659251674e-05,-3.409581105395798062e-05,-3.406488120588635482e-05,-3.403378437344395380e-05,-3.400252124124980568e-05,-3.397109249558827315e-05,-3.393949882551927644e-05,-3.390774092326687139e-05,-3.387581948355311567e-05,-3.384373520315397954e-05,-3.381148878184303541e-05,-3.377908092194736867e-05,-3.374651232784797728e-05,-3.371378370670141678e-05,-3.368089576821775566e-05,-3.364784922393893041e-05,-3.361464478801590161e-05,-3.358128317781927663e-05,-3.354776511149681895e-05,-3.351409131002736075e-05,-3.348026249727631409e-05,-3.344627939838584751e-05,-3.341214274121817596e-05,-3.337785325557840466e-05,-3.334341167349208490e-05,-3.330881872876112482e-05,-3.327407515762992318e-05,-3.323918169828576907e-05,-3.320413909024821919e-05,-3.316894807581238780e-05,-3.313360939916076831e-05,-3.309812380564158829e-05,-3.306249204299005484e-05,-3.302671486077324303e-05,-3.299079301011254017e-05,-3.295472724429426847e-05,-3.291851831765946201e-05,-3.288216698671408977e-05,-3.284567400985149987e-05,-3.280904014663077461e-05,-3.277226615827633083e-05,-3.273535280767791988e-05,-3.269830085922409424e-05,-3.266111107935731894e-05,-3.262378423501965941e-05,-3.258632109476300442e-05,-3.254872242969275575e-05,-3.251098901113635975e-05,-3.247312161180904155e-05,-3.243512100653545005e-05,-3.239698797058432334e-05,-3.235872328088973404e-05,-3.232032771560700013e-05,-3.228180205411268489e-05,-3.224314707717113038e-05,-3.220436356576872328e-05,-3.216545230300127400e-05,-3.212641407268623794e-05,-3.208724965919618199e-05,-3.204795984923514141e-05,-3.200854542922959567e-05,-3.196900718643869155e-05,-3.192934591028651070e-05,-3.188956239030815709e-05,-3.184965741664935734e-05,-3.180963178051054996e-05,-3.176948627442444106e-05,-3.172922169086822564e-05,-3.168883882365136628e-05,-3.164833846713843712e-05,-3.160772141597156804e-05,-3.156698846629168997e-05,-3.152614041379564469e-05,-3.148517805573458617e-05,-3.144410218935966839e-05,-3.140291361264369030e-05,-3.136161312411456237e-05,-3.132020152246672851e-05,-3.127867960733832220e-05,-3.123704817808992118e-05,-3.119530803519232620e-05,-3.115345997911633802e-05,-3.111150481077684660e-05,-3.106944333125527535e-05,-3.102727634168855886e-05,-3.098500464393527665e-05,-3.094262903979849710e-05,-3.090015033146986667e-05,-3.085756932097449834e-05,-3.081488681067057200e-05,-3.077210360302728986e-05,-3.072922050034732067e-05,-3.068623830521088891e-05,-3.064315782042026370e-05,-3.059997984838913609e-05,-3.055670519119813022e-05,-3.051333465164951519e-05,-3.046986903226800436e-05,-3.042630913485666611e-05,-3.038265576138510227e-05,-3.033890971387842583e-05,-3.029507179430623864e-05,-3.025114280369445297e-05,-3.020712354323551452e-05,-3.016301481401084672e-05,-3.011881741649125033e-05,-3.007453215103650379e-05,-3.003015981700718484e-05,-2.998570121437449387e-05,-2.994115714244349746e-05,-2.989652839935352802e-05,-2.985181578307738448e-05,-2.980702009175439926e-05,-2.976214212224714828e-05,-2.971718267125167401e-05,-2.967214253496441856e-05,-2.962702250841608986e-05,-2.958182338658188471e-05,-2.953654596365984375e-05,-2.949119103307085155e-05,-2.944575938779170343e-05,-2.940025181996652748e-05,-2.935466912024065067e-05,-2.930901207975900036e-05,-2.926328148850076971e-05,-2.921747813494635082e-05,-2.917160280724306887e-05,-2.912565629287211522e-05,-2.907963937814894706e-05,-2.903355284833430972e-05,-2.898739748846690389e-05,-2.894117408147600656e-05,-2.889488341006885008e-05,-2.884852625623102185e-05,-2.880210340044930817e-05,-2.875561562193373888e-05,-2.870906369906167654e-05,-2.866244840959986107e-05,-2.861577052964969781e-05,-2.856903083436890256e-05,-2.852223009752741234e-05,-2.847536909195147459e-05,-2.842844858913506911e-05,-2.838146935985053076e-05,-2.833443217292730409e-05,-2.828733779586256603e-05,-2.824018699576491542e-05,-2.819298053796659431e-05,-2.814571918585695443e-05,-2.809840370227023598e-05,-2.805103484876392272e-05,-2.800361338467505234e-05,-2.795614006873003987e-05,-2.790861565815649925e-05,-2.786104090773955377e-05,-2.781341657226432673e-05,-2.776574340451753997e-05,-2.771802215489893584e-05,-2.767025357380825668e-05,-2.762243840870315381e-05,-2.757457740631963361e-05,-2.752667131161734559e-05,-2.747872086766856015e-05,-2.743072681676839153e-05,-2.738268989849190760e-05,-2.733461085141497549e-05,-2.728649041239261663e-05,-2.723832931644798450e-05,-2.719012829699440914e-05,-2.714188808566886379e-05,-2.709360941266503175e-05,-2.704529300590063912e-05,-2.699693959190563319e-05,-2.694854989510053755e-05,-2.690012463846258584e-05,-2.685166454347021059e-05,-2.680317032865975335e-05,-2.675464271151284379e-05,-2.670608240767924357e-05,-2.665749013069929063e-05,-2.660886659250349950e-05,-2.656021250235784947e-05,-2.651152856841809680e-05,-2.646281549678608513e-05,-2.641407399101014519e-05,-2.636530475369491811e-05,-2.631650848444744284e-05,-2.626768588093186807e-05,-2.621883763986865290e-05,-2.616996445509167657e-05,-2.612106701804783881e-05,-2.607214601912932750e-05,-2.602320214600828407e-05,-2.597423608452498200e-05,-2.592524851841027100e-05,-2.587624012867495438e-05,-2.582721159516410125e-05,-2.577816359533580126e-05,-2.572909680409463107e-05,-2.568001189440227705e-05,-2.563090953727753529e-05,-2.558179040135222237e-05,-2.553265515292668653e-05,-2.548350445635838568e-05,-2.543433897383984288e-05,-2.538515936512109050e-05,-2.533596628739864798e-05,-2.528676039648125595e-05,-2.523754234556863096e-05,-2.518831278508493199e-05,-2.513907236384449462e-05,-2.508982172771956343e-05,-2.504056152086153730e-05,-2.499129238525688024e-05,-2.494201495978343175e-05,-2.489272988143165222e-05,-2.484343778508257827e-05,-2.479413930317475590e-05,-2.474483506520464005e-05,-2.469552569883681770e-05,-2.464621182984849668e-05,-2.459689408007559308e-05,-2.454757307074340034e-05,-2.449824941985756510e-05,-2.444892374248164302e-05,-2.439959665234692210e-05,-2.435026875996504359e-05,-2.430094067390475843e-05,-2.425161299979272655e-05,-2.420228634103516185e-05,-2.415296129909538791e-05,-2.410363847210605925e-05,-2.405431845631245125e-05,-2.400500184529530401e-05,-2.395568922980428894e-05,-2.390638119903476522e-05,-2.385707833874040062e-05,-2.380778123262095036e-05,-2.375849046193367897e-05,
        -2.370920660493824883e-05,-2.365993023789592087e-05,-2.361066193445893191e-05,-2.356140226550396122e-05,-2.351215179929866395e-05,-2.346291110222331611e-05,-2.341368073766059155e-05,-2.336446126588453964e-05,-2.331525324550387523e-05,-2.326605723235175560e-05,-2.321687377915271355e-05,-2.316770343691043621e-05,-2.311854675351998623e-05,-2.306940427421189099e-05,-2.302027654188520955e-05,-2.297116409677446569e-05,-2.292206747661618138e-05,-2.287298721614927643e-05,-2.282392384816978037e-05,-2.277487790253163169e-05,-2.272584990603565558e-05,-2.267684038359529808e-05,-2.262784985740395882e-05,-2.257887884643539067e-05,-2.252992786766494504e-05,-2.248099743518139348e-05,-2.243208806079755036e-05,-2.238320025316209438e-05,-2.233433451842570250e-05,-2.228549136040758327e-05,-2.223667128015138772e-05,-2.218787477598072044e-05,-2.213910234361016194e-05,-2.209035447620077974e-05,-2.204163166386052808e-05,-2.199293439508753778e-05,-2.194426315454967025e-05,-2.189561842469434083e-05,-2.184700068585954114e-05,-2.179841041516361599e-05,-2.174984808689384153e-05,-2.170131417300602550e-05,-2.165280914356859654e-05,-2.160433346443113578e-05,-2.155588759950033406e-05,-2.150747201101754769e-05,-2.145908715700528546e-05,-2.141073349365418821e-05,-2.136241147432382803e-05,-2.131412155004230868e-05,-2.126586416878462060e-05,-2.121763977591673012e-05,-2.116944881447313520e-05,-2.112129172432419821e-05,-2.107316894328636891e-05,-2.102508090634502835e-05,-2.097702804565448886e-05,-2.092901079070452752e-05,-2.088102956832038615e-05,-2.083308480294032705e-05,-2.078517691678216650e-05,-2.073730632834447363e-05,-2.068947345396088267e-05,-2.064167870785560410e-05,-2.059392250092217935e-05,-2.054620524161165918e-05,-2.049852733598811483e-05,-2.045088918722903770e-05,-2.040329119623596199e-05,-2.035573376080179742e-05,-2.030821727616594075e-05,-2.026074213545836500e-05,-2.021330872875592988e-05,-2.016591744358198213e-05,-2.011856866507288899e-05,-2.007126277547843785e-05,-2.002400015427285851e-05,-1.997678117882095705e-05,-1.992960622387851544e-05,-1.988247566142575806e-05,-1.983538986066735177e-05,-1.978834918814342814e-05,-1.974135400856225075e-05,-1.969440468319039184e-05,-1.964750157101846639e-05,-1.960064502931624375e-05,-1.955383541119015689e-05,-1.950707306808130426e-05,-1.946035834915482710e-05,-1.941369160018968643e-05,-1.936707316524399758e-05,-1.932050338537827372e-05,-1.927398259882195930e-05,-1.922751114219467539e-05,-1.918108934884088512e-05,-1.913471754932949409e-05,-1.908839607267509564e-05,-1.904212524445059174e-05,-1.899590538834150522e-05,-1.894973682531331249e-05,-1.890361987338939898e-05,-1.885755484876128207e-05,-1.881154206478941049e-05,-1.876558183233623112e-05,-1.871967445982170020e-05,-1.867382025333430562e-05,-1.862801951624248886e-05,-1.858227254963873420e-05,-1.853657965195099067e-05,-1.849094111922022776e-05,-1.844535724504492435e-05,-1.839982832080311326e-05,-1.835435463520829202e-05,-1.830893647436493410e-05,-1.826357412204604458e-05,-1.821826785997071596e-05,-1.817301796730452779e-05,-1.812782471999341283e-05,-1.808268839265103622e-05,-1.803760925717101671e-05,-1.799258758278243775e-05,-1.794762363649393677e-05,-1.790271768281614939e-05,-1.785786998409477633e-05,-1.781308080039956110e-05,-1.776835038896917851e-05,-1.772367900487736847e-05,-1.767906690119946944e-05,-1.763451432801321772e-05,-1.759002153361999277e-05,-1.754558876426726144e-05,-1.750121626270528807e-05,-1.745690427024104707e-05,-1.741265302596106679e-05,-1.736846276584325111e-05,-1.732433372475528088e-05,-1.728026613401212330e-05,-1.723626022348545561e-05,-1.719231622110406477e-05,-1.714843435102197944e-05,-1.710461483650504988e-05,-1.706085789826561339e-05,-1.701716375424044969e-05,-1.697353262081202629e-05,-1.692996471158725313e-05,-1.688646023839668331e-05,-1.684301941057286811e-05,-1.679964243500586818e-05,-1.675632951714245422e-05,-1.671308085943179478e-05,-1.666989666238016810e-05,-1.662677712477300673e-05,-1.658372244239814108e-05,-1.654073280948908931e-05,-1.649780841822545696e-05,-1.645494945778924745e-05,-1.641215611597468538e-05,-1.636942857835554932e-05,-1.632676702811863834e-05,-1.628417164645235005e-05,-1.624164261265770293e-05,-1.619918010353771365e-05,-1.615678429384148629e-05,-1.611445535643074578e-05,-1.607219346216881561e-05,-1.602999877942101747e-05,-1.598787147477631621e-05,-1.594581171276976406e-05,-1.590381965593801183e-05,-1.586189546470828660e-05,-1.582003929700981359e-05,-1.577825130955057276e-05,-1.573653165681809796e-05,-1.569488049046885436e-05,-1.565329796116010641e-05,-1.561178421738418365e-05,-1.557033940496888036e-05,-1.552896366840972320e-05,-1.548765715020383738e-05,-1.544641999073892435e-05,-1.540525232818223955e-05,-1.536415429909121499e-05,-1.532312603824692587e-05,-1.528216767809897902e-05,-1.524127934893204639e-05,-1.520046118003159918e-05,-1.515971329824061797e-05,-1.511903582823714842e-05,-1.507842889336696857e-05,-1.503789261442234348e-05,-1.499742711091878178e-05,-1.495703250048441291e-05,-1.491670889847140913e-05,-1.487645641867763047e-05,-1.483627517312458011e-05,-1.479616527177984864e-05,-1.475612682272364751e-05,-1.471615993270392053e-05,-1.467626470608163203e-05,-1.463644124560792292e-05,-1.459668965253513306e-05,-1.455701002628373431e-05,-1.451740246394273015e-05,-1.447786706126885647e-05,-1.443840391224249231e-05,-1.439901310906765985e-05,-1.435969474244958022e-05,-1.432044890081751731e-05,-1.428127567121295627e-05,-1.424217513906755883e-05,-1.420314738798111875e-05,-1.416419249938849489e-05,-1.412531055389187884e-05,-1.408650163026159419e-05,-1.404776580488098503e-05,-1.400910315307868359e-05,-1.397051374812940949e-05,-1.393199766236419279e-05,-1.389355496572708404e-05,-1.385518572677435500e-05,-1.381689001273000983e-05,-1.377866788870862891e-05,-1.374051941865905846e-05,-1.370244466447623211e-05,-1.366444368711139390e-05,-1.362651654512880839e-05,-1.358866329631558401e-05,-1.355088399651593889e-05,-1.351317869963120089e-05,-1.347554745889656402e-05,-1.343799032504922053e-05,-1.340050734827125112e-05,-1.336309857652429045e-05,-1.332576405621566096e-05,-1.328850383280899550e-05,-1.325131794965850318e-05,-1.321420644928572585e-05,-1.317716937226931506e-05,-1.314020675746707667e-05,-1.310331864290414927e-05,-1.306650506510687038e-05,-1.302976605843664260e-05,-1.299310165642220127e-05,-1.295651189137103643e-05,-1.291999679353672548e-05,-1.288355639195160052e-05,-1.284719071442674831e-05,-1.281089978749649916e-05,-1.277468363580780419e-05,-1.273854228295290270e-05,-1.270247575158034437e-05,-1.266648406200721055e-05,-1.263056723382893765e-05,-1.259472528530869440e-05,-1.255895823298880387e-05,-1.252326609235687727e-05,-1.248764887745723584e-05,-1.245210660144602244e-05,-1.241663927575853421e-05,-1.238124691022024493e-05,-1.234592951404600569e-05,-1.231068709467431077e-05,-1.227551965832240910e-05,-1.224042721054141580e-05,-1.220540975488404456e-05,-1.217046729373727487e-05,-1.213559982887746358e-05,-1.210080735991603262e-05,-1.206608988557622553e-05,-1.203144740413719660e-05,-1.199687991171316526e-05,-1.196238740347466134e-05,-1.192796987342648052e-05,-1.189362731446319543e-05,-1.185935971814711110e-05,-1.182516707498582065e-05,-1.179104937432118305e-05,-1.175700660432932310e-05,-1.172303875202063139e-05,-1.168914580335078668e-05,-1.165532774310973352e-05,-1.162158455431105963e-05,-1.158791621969079699e-05,-1.155432272120782144e-05,-1.152080403871158509e-05,-1.148736015121887277e-05,-1.145399103685829090e-05,-1.142069667275924516e-05,-1.138747703494091823e-05,-1.135433209809022514e-05,-1.132126183611692483e-05,-1.128826622148748626e-05,-1.125534522650184499e-05,-1.122249882140602395e-05,-1.118972697555786766e-05,-1.115702965809317604e-05,-1.112440683653792561e-05,-1.109185847741889219e-05,-1.105938454631916201e-05,-1.102698500787813174e-05,-1.099465982562497501e-05,-1.096240896236722051e-05,-1.093023237952461812e-05,-1.089813003823936199e-05,-1.086610189798831172e-05,-1.083414791758219309e-05,-1.080226805488804231e-05,-1.077046226699573950e-05,-1.073873050977391941e-05,-1.070707273820303840e-05,-1.067548890681946361e-05,-1.064397896838320534e-05,-1.061254287570978505e-05,-1.058118058022694541e-05,-1.054989203230771722e-05,-1.051867718188104206e-05,-1.048753597759910505e-05,-1.045646836750346864e-05,-1.042547429885853916e-05,-1.039455371765196645e-05,-1.036370656970486692e-05,-1.033293279933955588e-05,-1.030223235032323714e-05,-1.027160516575698068e-05,-1.024105118746510001e-05,-1.021057035704986404e-05,-1.018016261500331865e-05,-1.014982790092933129e-05,-1.011956615359910217e-05,-1.008937731139525340e-05,-1.005926131181222871e-05,-1.002921809106771533e-05,-9.999247585101844749e-06,-9.969349729077592315e-06,-9.939524457491799581e-06,-9.909771703564551615e-06,-9.880091400182866579e-06,-9.850483479567628819e-06,-9.820947873162566566e-06,-9.791484511745274233e-06,-9.762093324983123210e-06,-9.732774241821839922e-06,-9.703527191373684246e-06,-9.674352101751715338e-06,-9.645248899181613211e-06,-9.616217510832747450e-06,-9.587257862486708859e-06,-9.558369878592820612e-06,-9.529553484433073152e-06,-9.500808603790655837e-06,-9.472135159283023853e-06,-9.443533073694165836e-06,-9.415002269086425457e-06,-9.386542666522945666e-06,-9.358154186789313655e-06,-9.329836749671915896e-06,-9.301590274457538499e-06,-9.273414680488478723e-06,-9.245309885497210445e-06,-9.217275806994162934e-06,-9.189312362101187404e-06,-9.161419467274001249e-06,-9.133597038524232659e-06,-9.105844990697775643e-06,-9.078163238640524213e-06,-9.050551696698772020e-06,-9.023010277942056234e-06,-8.995538895606447483e-06,-8.968137462428416029e-06,-8.940805889978697962e-06,-8.913544089883540522e-06,-8.886351972825501377e-06,-8.859229449154071290e-06,-8.832176428552607206e-06,-8.805192820704466072e-06,-8.778278534238292963e-06,-8.751433477283132589e-06,-8.724657557745985059e-06,-8.697950683034250119e-06,-8.671312759833682549e-06,-8.644743694274925616e-06,-8.618243392766178346e-06,-8.591811760105816376e-06,-8.565448701758349159e-06,
        -8.539154121800507369e-06,-8.512927924309021677e-06,-8.486770012916533545e-06,-8.460680290756084077e-06,-8.434658660738669766e-06,-8.408705024331997180e-06,-8.382819284113995906e-06,-8.357001341385839055e-06,-8.331251096616032470e-06,-8.305568450661660052e-06,-8.279953303658160735e-06,-8.254405555518928850e-06,-8.228925105435713760e-06,-8.203511852045153319e-06,-8.178165694372463435e-06,-8.152886530499170448e-06,-8.127674258007200336e-06,-8.102528774478479079e-06,-8.077449976662265385e-06,-8.052437761529862570e-06,-8.027492025497462436e-06,-8.002612664037567214e-06,-7.977799573122279497e-06,-7.953052647891034610e-06,-7.928371783094689818e-06,-7.903756873761658142e-06,-7.879207813865640730e-06,-7.854724496991760674e-06,-7.830306816891674515e-06,-7.805954666983971890e-06,-7.781667940409686679e-06,-7.757446529366163190e-06,-7.733290326272790338e-06,-7.709199223326912431e-06,-7.685173111948717661e-06,-7.661211883724927674e-06,-7.637315429631641450e-06,-7.613483641033536031e-06,-7.589716408018531979e-06,-7.566013620452505251e-06,-7.542375169145021374e-06,-7.518800943795422853e-06,-7.495290833381407225e-06,-7.471844727324761237e-06,-7.448462514603182427e-06,-7.425144083639256820e-06,-7.401889323244148500e-06,-7.378698121562887735e-06,-7.355570366573971341e-06,-7.332505945811806924e-06,-7.309504746588757484e-06,-7.286566656772297534e-06,-7.263691562897633958e-06,-7.240879351833040545e-06,-7.218129910668835691e-06,-7.195443125052047861e-06,-7.172818881240328182e-06,-7.150257065657861233e-06,-7.127757563674119723e-06,-7.105320260991643266e-06,-7.082945043146438024e-06,-7.060631794841842890e-06,-7.038380401391819419e-06,-7.016190747499706504e-06,-6.994062717646798433e-06,-6.971996196702967552e-06,-6.949991068372352032e-06,-6.928047216803179253e-06,-6.906164526476743504e-06,-6.884342880597582592e-06,-6.862582162758812387e-06,-6.840882256720082211e-06,-6.819243045574907569e-06,-6.797664412472315121e-06,-6.776146240283775768e-06,-6.754688412047293866e-06,-6.733290810356784561e-06,-6.711953317806163000e-06,-6.690675816656277419e-06,-6.669458189390020664e-06,-6.648300318656819030e-06,-6.627202085662808884e-06,-6.606163372446793858e-06,-6.585184061214111040e-06,-6.564264033448452551e-06,-6.543403170855555118e-06,-6.522601354919110861e-06,-6.501858466845256146e-06,-6.481174387729105035e-06,-6.460548999109860802e-06,-6.439982182027126356e-06,-6.419473817464993459e-06,-6.399023786352042720e-06,-6.378631968784187478e-06,-6.358298246189608705e-06,-6.338022499274842403e-06,-6.317804608524379972e-06,-6.297644454200668207e-06,-6.277541916621665052e-06,-6.257496876382884210e-06,-6.237509213524727869e-06,-6.217578808753732034e-06,-6.197705541777231986e-06,-6.177889292580118763e-06,-6.158129941646883765e-06,-6.138427368518328819e-06,-6.118781453123833813e-06,-6.099192075892378995e-06,-6.079659116309255040e-06,-6.060182454081797232e-06,-6.040761969083874305e-06,-6.021397540745265786e-06,-6.002089049106373864e-06,-5.982836373763511517e-06,-5.963639394312991726e-06,-5.944497990462149772e-06,-5.925412041252187123e-06,-5.906381426667994816e-06,-5.887406026472419285e-06,-5.868485719484617391e-06,-5.849620385800502476e-06,-5.830809904738831762e-06,-5.812054155673873623e-06,-5.793353018035407587e-06,-5.774706371253213177e-06,-5.756114095090136829e-06,-5.737576068587380007e-06,-5.719092171563300298e-06,-5.700662283281143772e-06,-5.682286283226201107e-06,-5.663964051050296433e-06,-5.645695465905653521e-06,-5.627480407333074197e-06,-5.609318755150916047e-06,-5.591210388899980899e-06,-5.573155187621470219e-06,-5.555153031355786197e-06,-5.537203799477197208e-06,-5.519307371693038533e-06,-5.501463627766156606e-06,-5.483672446959797497e-06,-5.465933709425385700e-06,-5.448247294703723043e-06,-5.430613082557655957e-06,-5.413030952916564331e-06,-5.395500785654316900e-06,-5.378022460589271247e-06,-5.360595857595296110e-06,-5.343220856546260222e-06,-5.325897337815632682e-06,-5.308625181388304526e-06,-5.291404267360189095e-06,-5.274234476049244336e-06,-5.257115687440361285e-06,-5.240047781684964434e-06,-5.223030639545100939e-06,-5.206064141394239897e-06,-5.189148167605850404e-06,-5.172282598886468463e-06,-5.155467315498540870e-06,-5.138702198148603628e-06,-5.121987128264837708e-06,-5.105321986165201054e-06,-5.088706652389696217e-06,-5.072141008644059923e-06,-5.055624935468294723e-06,-5.039158314068536981e-06,-5.022741025817456517e-06,-5.006372951810167393e-06,-4.990053973530361731e-06,-4.973783972406220499e-06,-4.957562829921435821e-06,-4.941390427837255572e-06,-4.925266647970438783e-06,-4.909191371749166422e-06,-4.893164481045708669e-06,-4.877185858120913764e-06,-4.861255385235629944e-06,-4.845372944539683147e-06,-4.829538417738810097e-06,-4.813751687426925940e-06,-4.798012636531012731e-06,-4.782321147256407556e-06,-4.766677102419070167e-06,-4.751080385057004918e-06,-4.735530877431060048e-06,-4.720028462801284519e-06,-4.704573024760794198e-06,-4.689164446014526533e-06,-4.673802609989063939e-06,-4.658487400277522283e-06,-4.643218700195461679e-06,-4.627996393780087203e-06,-4.612820364569003573e-06,-4.597690496432882412e-06,-4.582606673853018009e-06,-4.567568780533548534e-06,-4.552576700789234820e-06,-4.537630319101371157e-06,-4.522729520062274133e-06,-4.507874188430793794e-06,-4.493064208799246728e-06,-4.478299466037505283e-06,-4.463579845292997561e-06,-4.448905231713151665e-06,-4.434275510389884545e-06,-4.419690567303291573e-06,-4.405150287600800851e-06,-4.390654556873929693e-06,-4.376203261435840375e-06,-4.361796286822539059e-06,-4.347433519180654571e-06,-4.333114845045393793e-06,-4.318840150674407852e-06,-4.304609322269836724e-06,-4.290422246921998806e-06,-4.276278811499167887e-06,-4.262178902814106607e-06,-4.248122407735088757e-06,-4.234109213796521942e-06,-4.220139208532813768e-06,-4.206212279089793782e-06,-4.192328313279425345e-06,-4.178487199302249877e-06,-4.164688824914719589e-06,-4.150933077873286692e-06,-4.137219847044626420e-06,-4.123549020851324798e-06,-4.109920487549434398e-06,-4.096334136338697363e-06,-4.082789855808233170e-06,-4.069287534991250510e-06,-4.055827063365047280e-06,-4.042408330018343321e-06,-4.029031224872525740e-06,-4.015695637460403589e-06,-4.002401457536830520e-06,-3.989148575633816307e-06,-3.975936881395192302e-06,-3.962766265186434822e-06,-3.949636617928131699e-06,-3.936547830096781553e-06,-3.923499792390927610e-06,-3.910492396175246910e-06,-3.897525532314816132e-06,-3.884599092229823469e-06,-3.871712967729035171e-06,-3.858867050232639428e-06,-3.846061231771447098e-06,-3.833295404043202126e-06,-3.820569459578315730e-06,-3.807883290740665672e-06,-3.795236789783107412e-06,-3.782629849846674830e-06,-3.770062363683823747e-06,-3.757534223935987683e-06,-3.745045324410334331e-06,-3.732595558469942176e-06,-3.720184819311356250e-06,-3.707813001296855759e-06,-3.695479997789519189e-06,-3.683185702874069989e-06,-3.670930011412387728e-06,-3.658712817766751613e-06,-3.646534016410463153e-06,-3.634393501872335008e-06,-3.622291169624869411e-06,-3.610226914807501686e-06,-3.598200632726200610e-06,-3.586212219186535322e-06,-3.574261569438963448e-06,-3.562348579788654490e-06,-3.550473146318733342e-06,-3.538635165056813747e-06,-3.526834532974199021e-06,-3.515071146320547513e-06,-3.503344901678584478e-06,-3.491655696741258197e-06,-3.480003428590894288e-06,-3.468387994365329519e-06,-3.456809291924045624e-06,-3.445267218682435129e-06,-3.433761673110602430e-06,-3.422292553123540415e-06,-3.410859756691753120e-06,-3.399463182840456454e-06,-3.388102730095265969e-06,-3.376778297425886421e-06,-3.365489784190600631e-06,-3.354237089192579901e-06,-3.343020111956640505e-06,-3.331838752729243680e-06,-3.320692910757649940e-06,-3.309582485844231314e-06,-3.298507379345672064e-06,-3.287467491008833065e-06,-3.276462721191197858e-06,-3.265492971027406099e-06,-3.254558141596586296e-06,-3.243658134310933860e-06,-3.232792850083043845e-06,-3.221962190602667420e-06,-3.211166057948133812e-06,-3.200404353975727645e-06,-3.189676980652755844e-06,-3.178983840501636848e-06,-3.168324836266833699e-06,-3.157699870637298289e-06,-3.147108846301982510e-06,-3.136551666504949765e-06,-3.126028234767819214e-06,-3.115538454612210018e-06,-3.105082229504230185e-06,-3.094659463798166144e-06,-3.084270061459726264e-06,-3.073913926565641219e-06,-3.063590964025308949e-06,-3.053301078470571639e-06,-3.043044174644293776e-06,-3.032820157566895602e-06,-3.022628932869420026e-06,-3.012470405960865349e-06,-3.002344482583296781e-06,-2.992251068589801832e-06,-2.982190070000001469e-06,-2.972161393333117019e-06,-2.962164944886325202e-06,-2.952200631456403102e-06,-2.942268360062172405e-06,-2.932368037722454801e-06,-2.922499571456071976e-06,-2.912662868892468282e-06,-2.902857838049666128e-06,-2.893084386723643320e-06,-2.883342422932422267e-06,-2.873631854527491925e-06,-2.863952590470564274e-06,-2.854304539667840146e-06,-2.844687610636942310e-06,-2.835101712561627352e-06,-2.825546754459118404e-06,-2.816022646290328169e-06,-2.806529297683102442e-06,-2.797066618265287019e-06,-2.787634518275350359e-06,-2.778232907785227468e-06,-2.768861697588498316e-06,-2.759520798478742876e-06,-2.750210120916474210e-06,-2.740929576083850350e-06,-2.731679075496096232e-06,-2.722458530501903340e-06,-2.713267853005074670e-06,-2.704106954576346311e-06,-2.694975747175032410e-06,-2.685874143648625534e-06,-2.676802055956439830e-06,-2.667759397279034772e-06,-2.658746080075324869e-06,-2.649762017470358444e-06,-2.640807123088784181e-06,-2.631881309944628100e-06,-2.622984491995605794e-06,-2.614116583254944004e-06,-2.605277497569336020e-06,-2.596467149396097795e-06,-2.587685453081522979e-06,-2.578932323138438676e-06,-2.570207674856828106e-06,-2.561511423193607584e-06,-2.552843483272226877e-06,-2.544203770493691508e-06,-2.535592200814118513e-06,-2.527008689967580324e-06,-2.518453154187749732e-06,-2.509925509486254924e-06,-2.501425672596369054e-06,-2.492953560084831821e-06,-2.484509088629405227e-06,-2.476092175462962786e-06,-2.467702737707355709e-06,-2.459340693317102478e-06,-2.451005959580587756e-06,-2.442698454285796572e-06,
        -2.434418095664803161e-06,-2.426164801727637155e-06,-2.417938491039439697e-06,-2.409739082720463443e-06,-2.401566495335849538e-06,-2.393420647561761427e-06,-2.385301458740496372e-06,-2.377208848380885087e-06,-2.369142736546869799e-06,-2.361103042636258920e-06,-2.353089686712994677e-06,-2.345102589285108508e-06,-2.337141670527564941e-06,-2.329206850837373111e-06,-2.321298051277675967e-06,-2.313415193244683365e-06,-2.305558197523982500e-06,-2.297726985345249773e-06,-2.289921478493273099e-06,-2.282141598919373848e-06,-2.274387268685895691e-06,-2.266658409633137694e-06,-2.258954944045488133e-06,-2.251276794928980252e-06,-2.243623884790046930e-06,-2.235996136801254863e-06,-2.228393473913126144e-06,-2.220815819575783223e-06,-2.213263097294859705e-06,-2.205735230575989192e-06,-2.198232143701961405e-06,-2.190753760678010309e-06,-2.183300005897947926e-06,-2.175870803700075129e-06,-2.168466078811270847e-06,-2.161085755958414012e-06,-2.153729760701050822e-06,-2.146398018099127114e-06,-2.139090453157077576e-06,-2.131806992211604523e-06,-2.124547560822254155e-06,-2.117312084992661880e-06,-2.110100491059530015e-06,-2.102912705248538572e-06,-2.095748654507012532e-06,-2.088608265282676513e-06,-2.081491464689388948e-06,-2.074398180007541725e-06,-2.067328338350993278e-06,-2.060281867499735853e-06,-2.053258695233761699e-06,-2.046258749222040763e-06,-2.039281958021721408e-06,-2.032328249690351640e-06,-2.025397552507524068e-06,-2.018489795363453965e-06,-2.011604907203867754e-06,-2.004742817196536464e-06,-1.997903454231675369e-06,-1.991086747921144706e-06,-1.984292627821293564e-06,-1.977521023877049089e-06,-1.970771866144360729e-06,-1.964045084568155630e-06,-1.957340609648472451e-06,-1.950658371829838700e-06,-1.943998301889848790e-06,-1.937360330772630590e-06,-1.930744389533334271e-06,-1.924150409726710365e-06,-1.917578322407909042e-06,-1.911028059187191985e-06,-1.904499552562999298e-06,-1.897992733979059210e-06,-1.891507535378700311e-06,-1.885043889926496519e-06,-1.878601729954354482e-06,-1.872180987849692002e-06,-1.865781596499527240e-06,-1.859403489234967566e-06,-1.853046599775698411e-06,-1.846710861064249087e-06,-1.840396206598260420e-06,-1.834102570708040503e-06,-1.827829887335319370e-06,-1.821578090366315905e-06,-1.815347114131338202e-06,-1.809136893571317017e-06,-1.802947363183093898e-06,-1.796778457685554997e-06,-1.790630112574742583e-06,-1.784502262958120866e-06,-1.778394844054176360e-06,-1.772307791692018242e-06,-1.766241041811777990e-06,-1.760194530187053630e-06,-1.754168192702465490e-06,-1.748161966019790015e-06,-1.742175786911825952e-06,-1.736209591318704781e-06,-1.730263316512825611e-06,-1.724336899711076398e-06,-1.718430277464211287e-06,-1.712543387932807804e-06,-1.706676168389265058e-06,-1.700828556105982159e-06,-1.695000489299047786e-06,-1.689191906240061769e-06,-1.683402745034090486e-06,-1.677632943730689163e-06,-1.671882441378613748e-06,-1.666151177026620189e-06,-1.660439088724263712e-06,-1.654746115520300265e-06,-1.649072197629219971e-06,-1.643417274266312234e-06,-1.637781284480332999e-06,-1.632164168152705486e-06,-1.626565865553430967e-06,-1.620986316730466115e-06,-1.615425461842789900e-06,-1.609883241548981658e-06,-1.604359596063531512e-06,-1.598854466378085704e-06,-1.593367793928379683e-06,-1.587899519095437029e-06,-1.582449583426015494e-06,-1.577017928688917436e-06,-1.571604496153344854e-06,-1.566209227310544350e-06,-1.560832064817496700e-06,-1.555472950776071173e-06,-1.550131826788536671e-06,-1.544808635678407427e-06,-1.539503320213686521e-06,-1.534215823217888186e-06,-1.528946087125948594e-06,-1.523694055372004641e-06,-1.518459671279170919e-06,-1.513242878115050871e-06,-1.508043619757870601e-06,-1.502861839808300459e-06,-1.497697481978033096e-06,-1.492550490478361525e-06,-1.487420809576089908e-06,-1.482308383704555865e-06,-1.477213157519141618e-06,-1.472135075619718236e-06,-1.467074082550645642e-06,-1.462030123855484476e-06,-1.457003144800239625e-06,-1.451993089984782159e-06,-1.446999905785339990e-06,-1.442023537467918004e-06,-1.437063930742610296e-06,-1.432121031874622474e-06,-1.427194786740582089e-06,-1.422285142105295108e-06,-1.417392043956411385e-06,-1.412515439058736888e-06,-1.407655274121566435e-06,-1.402811496076239450e-06,-1.397984052298184565e-06,-1.393172889607718901e-06,-1.388377955546804543e-06,-1.383599197712914730e-06,-1.378836563759033851e-06,-1.374090001560190899e-06,-1.369359459102437171e-06,-1.364644884538357417e-06,-1.359946226076047537e-06,-1.355263432256670342e-06,-1.350596451787922092e-06,-1.345945233488521353e-06,-1.341309725788608631e-06,-1.336689878173036305e-06,-1.332085640126656756e-06,-1.327496960357166245e-06,-1.322923788737995210e-06,-1.318366074753996031e-06,-1.313823768223087995e-06,-1.309296819351768448e-06,-1.304785178402045887e-06,-1.300288795191839597e-06,-1.295807619927646925e-06,-1.291341603759654788e-06,-1.286890697005382833e-06,-1.282454850815017977e-06,-1.278034015950169078e-06,-1.273628143394489598e-06,-1.269237184853277967e-06,-1.264861091310187646e-06,-1.260499814637050520e-06,-1.256153306594676167e-06,-1.251821518888363016e-06,-1.247504403445454102e-06,-1.243201912304314760e-06,-1.238913998335977595e-06,-1.234640613523296793e-06,-1.230381710182193444e-06,-1.226137241572278214e-06,-1.221907160620094857e-06,-1.217691419919120221e-06,-1.213489972617942669e-06,-1.209302772253728619e-06,-1.205129771808532979e-06,-1.200970925541167134e-06,-1.196826186766752897e-06,-1.192695509077967841e-06,-1.188578847066690258e-06,-1.184476154325597719e-06,-1.180387385335546213e-06,-1.176312494577391732e-06,-1.172251436087901055e-06,-1.168204165180597442e-06,-1.164170636613892640e-06,-1.160150804979664940e-06,-1.156144625369392998e-06,-1.152152053263133524e-06,-1.148173043752365174e-06,-1.144207552539189265e-06,-1.140255535436729417e-06,-1.136316947758508888e-06,-1.132391745539695904e-06,-1.128479884926480992e-06,-1.124581322065054678e-06,-1.120696013268140945e-06,-1.116823914681930319e-06,-1.112964983340791747e-06,-1.109119175779493816e-06,-1.105286448199738203e-06,-1.101466758579583427e-06,-1.097660063564820376e-06,-1.093866320356351451e-06,-1.090085486099567902e-06,-1.086317518383950187e-06,-1.082562375076534522e-06,-1.078820013822312518e-06,-1.075090392432809239e-06,-1.071373468886083202e-06,-1.067669201382237532e-06,-1.063977548065864198e-06,-1.060298467414622081e-06,-1.056631917850658908e-06,-1.052977858184700466e-06,-1.049336247116450238e-06,-1.045707043512145162e-06,-1.042090206460066781e-06,-1.038485694992985486e-06,-1.034893468476738576e-06,-1.031313486166141047e-06,-1.027745707926630558e-06,-1.024190093124044409e-06,-1.020646601623820260e-06,-1.017115193402418072e-06,-1.013595827936697447e-06,-1.010088466146807917e-06,-1.006593068120231749e-06,-1.003109594055473508e-06,-9.996380046506381234e-07,-9.961782602707636158e-07,-9.927303217804883673e-07,-9.892941501554730621e-07,-9.858697062048449311e-07,-9.824569516259096247e-07,-9.790558472833055248e-07,-9.756663541526933159e-07,-9.722884342644455558e-07,-9.689220489828009875e-07,-9.655671601715987151e-07,-9.622237295281443892e-07,-9.588917191383217187e-07,-9.555710911990367151e-07,-9.522618076296396339e-07,-9.489638309045922426e-07,-9.456771235538674603e-07,-9.424016479964159032e-07,-9.391373668177216416e-07,-9.358842428253133505e-07,-9.326422391042754612e-07,-9.294113187952035560e-07,-9.261914444280705538e-07,-9.229825799206281545e-07,-9.197846886355165452e-07,-9.165977334357755524e-07,-9.134216783501791781e-07,-9.102564870189233659e-07,-9.071021232487375130e-07,-9.039585509018621678e-07,-9.008257342846270888e-07,-8.977036375368285803e-07,-8.945922244096848885e-07,-8.914914598756595865e-07,-8.884013083521047349e-07,-8.853217343118835458e-07,-8.822527024499038362e-07,-8.791941776276068765e-07,-8.761461247619450887e-07,-8.731085091584489533e-07,-8.700812957895820432e-07,-8.670644498498525365e-07,-8.640579370888801236e-07,-8.610617222570837725e-07,-8.580757714926612323e-07,-8.551000510448325542e-07,-8.521345258860613114e-07,-8.491791619325006479e-07,-8.462339259329709762e-07,-8.432987831374916254e-07,-8.403737002948830082e-07,-8.374586440429432344e-07,-8.345535804643589017e-07,-8.316584763079504228e-07,-8.287732979339601513e-07,-8.258980123132531048e-07,-8.230325863611831494e-07,-8.201769873816822098e-07,-8.173311820125483962e-07,-8.144951375022024820e-07,-8.116688213766209969e-07,-8.088522009952470171e-07,-8.060452439950793746e-07,-8.032479175690276918e-07,-8.004601895761354058e-07,-7.976820281530017098e-07,-7.949134012141811922e-07,-7.921542764521838365e-07,-7.894046221701422894e-07,-7.866644067267003493e-07,-7.839335982029460581e-07,-7.812121651795678190e-07,-7.785000762372540350e-07,-7.757973000122042606e-07,-7.731038050851068988e-07,-7.704195603142061088e-07,-7.677445348353018062e-07,-7.650786973401046964e-07,-7.624220174640150560e-07,-7.597744643983439516e-07,-7.571360071123578450e-07,-7.545066149639012565e-07,-7.518862575883744626e-07,-7.492749051207781008e-07,-7.466725265858897842e-07,-7.440790921742213015e-07,-7.414945719652621392e-07,-7.389189355389014224e-07,-7.363521532521843938e-07,-7.337941953511339932e-07,-7.312450319707508584e-07,-7.287046336346136854e-07,-7.261729709218123219e-07,-7.236500140783697077e-07,-7.211357342939983539e-07,-7.186301020922769567e-07,-7.161330878857619098e-07,-7.136446631972326315e-07,-7.111647994384462379e-07,-7.086934670774702738e-07,-7.062306373040172502e-07,-7.037762820294446442e-07,-7.013303723324426642e-07,-6.988928796802795773e-07,-6.964637757622682557e-07,-6.940430326007884787e-07,-6.916306213855527574e-07,-6.892265143054743248e-07,-6.868306837159998679e-07,-6.844431011954199562e-07,-6.820637389326478228e-07,-6.796925695051747596e-07,-6.773295649353805459e-07,-6.749746974121784149e-07,-6.726279404012380780e-07,-6.702892657584058611e-07,-6.679586461721953583e-07,-6.656360545531647688e-07,-6.633214639784057454e-07,-6.610148475250099409e-07,-6.587161773263794373e-07,-6.564254275698289121e-07,-6.541425711659165643e-07,-6.518675809141782906e-07,-6.496004310019287686e-07,
        -6.473410943952373486e-07,-6.450895450593741032e-07,-6.428457562379641388e-07,-6.406097015632106206e-07,-6.383813554999839823e-07,-6.361606914584427841e-07,-6.339476834038570985e-07,-6.317423058566085103e-07,-6.295445326154336385e-07,-6.273543382007140679e-07,-6.251716969662979295e-07,-6.229965825998995399e-07,-6.208289701215008449e-07,-6.186688348841506979e-07,-6.165161503535188103e-07,-6.143708913830536744e-07,-6.122330337143822021e-07,-6.101025515903302221e-07,-6.079794203084354365e-07,-6.058636144445905813e-07,-6.037551092963333588e-07,-6.016538808828464369e-07,-5.995599036134890980e-07,-5.974731532298882541e-07,-5.953936054181596660e-07,-5.933212349207295233e-07,-5.912560185894477627e-07,-5.891979315553186325e-07,-5.871469492824132885e-07,-5.851030481784924575e-07,-5.830662037631384464e-07,-5.810363924441119821e-07,-5.790135905736626398e-07,-5.769977738934173317e-07,-5.749889183670475745e-07,-5.729870008464033049e-07,-5.709919978502675519e-07,-5.690038856198675887e-07,-5.670226406739864444e-07,-5.650482400865186605e-07,-5.630806604317584174e-07,-5.611198781174664418e-07,-5.591658705506041827e-07,-5.572186144719992740e-07,-5.552780866779905011e-07,-5.533442647420727667e-07,-5.514171256826294609e-07,-5.494966468511108815e-07,-5.475828053214115698e-07,-5.456755786115152773e-07,-5.437749444614503602e-07,-5.418808802226671162e-07,-5.399933634686604478e-07,-5.381123719949698625e-07,-5.362378836526460191e-07,-5.343698763482507275e-07,-5.325083280993681001e-07,-5.306532162574484346e-07,-5.288045195062096582e-07,-5.269622160297693370e-07,-5.251262837346892809e-07,-5.232967011381539635e-07,-5.214734466463255558e-07,-5.196564984988327751e-07,-5.178458352128600950e-07,-5.160414355831477451e-07,-5.142432778493244427e-07,-5.124513411391973250e-07,-5.106656042475066215e-07,-5.088860458579702595e-07,-5.071126447098173173e-07,-5.053453802084106883e-07,-5.035842314260463581e-07,-5.018291769354199516e-07,-5.000801967525170255e-07,-4.983372695610555070e-07,-4.966003747663982892e-07,-4.948694919404417192e-07,-4.931446004885486900e-07,-4.914256803156824560e-07,-4.897127106606724567e-07,-4.880056708178592828e-07,-4.863045411362953985e-07,-4.846093017429886629e-07,-4.829199317102350619e-07,-4.812364114425982109e-07,-4.795587210115748178e-07,-4.778868402666169857e-07,-4.762207492237102713e-07,-4.745604287870186511e-07,-4.729058588059942281e-07,-4.712570189080445004e-07,-4.696138906634672594e-07,-4.679764543658038178e-07,-4.663446900310397325e-07,-4.647185787298724335e-07,-4.630981008113543851e-07,-4.614832370686272611e-07,-4.598739686278996430e-07,-4.582702763378243560e-07,-4.566721403809204105e-07,-4.550795421054409928e-07,-4.534924633592396503e-07,-4.519108843803465447e-07,-4.503347868500817697e-07,-4.487641515060758479e-07,-4.471989598076042682e-07,-4.456391934914982755e-07,-4.440848337394776024e-07,-4.425358623993957963e-07,-4.409922600978610774e-07,-4.394540090713050517e-07,-4.379210912786035692e-07,-4.363934880680098161e-07,-4.348711813983996421e-07,-4.333541527290485362e-07,-4.318423845739438605e-07,-4.303358587809391622e-07,-4.288345569758433840e-07,-4.273384616171327366e-07,-4.258475548857276749e-07,-4.243618187405040487e-07,-4.228812356954492202e-07,-4.214057882090394003e-07,-4.199354583511727412e-07,-4.184702288023700589e-07,-4.170100819100852618e-07,-4.155550005213726195e-07,-4.141049671502194940e-07,-4.126599643106132476e-07,-4.112199746275635448e-07,-4.097849813367027139e-07,-4.083549672850850243e-07,-4.069299152642535944e-07,-4.055098082322849962e-07,-4.040946288141888942e-07,-4.026843607451979778e-07,-4.012789872609445752e-07,-3.998784911529718045e-07,-3.984828554903785403e-07,-3.970920637308417156e-07,-3.957060998316386247e-07,-3.943249468063569907e-07,-3.929485879461402931e-07,-3.915770068751989186e-07,-3.902101877173436151e-07,-3.888481133751398033e-07,-3.874907675838201726e-07,-3.861381348002623781e-07,-3.847901983711210505e-07,-3.834469423091846352e-07,-3.821083505717304263e-07,-3.807744067829688106e-07,-3.794450951777328385e-07,-3.781203998798332577e-07,-3.768003051796142699e-07,-3.754847953674200767e-07,-3.741738539009276110e-07,-3.728674661251929479e-07,-3.715656157199376253e-07,-3.702682869199946936e-07,-3.689754648483756228e-07,-3.676871337399134632e-07,-3.664032782735304750e-07,-3.651238825730374060e-07,-3.638489316504234239e-07,-3.625784102401219400e-07,-3.613123028545217608e-07,-3.600505949497012637e-07,-3.587932709270269527e-07,-3.575403155764433905e-07,-3.562917140764731982e-07,-3.550474511060386362e-07,-3.538075123432626867e-07,-3.525718828556456685e-07,-3.513405471555763881e-07,-3.501134914207781890e-07,-3.488907000526175750e-07,-3.476721585071729237e-07,-3.464578526291006710e-07,-3.452477673748788334e-07,-3.440418888112084517e-07,-3.428402017835452398e-07,-3.416426924141013899e-07,-3.404493458258883720e-07,-3.392601476415180173e-07,-3.380750841497359716e-07,-3.368941407511094610e-07,-3.357173035123395266e-07,-3.345445578339933945e-07,-3.333758897827721057e-07,-3.322112856474213061e-07,-3.310507309950416754e-07,-3.298942124474457671e-07,-3.287417157382677146e-07,-3.275932262680747442e-07,-3.264487313248132239e-07,-3.253082172527399507e-07,-3.241716692858886972e-07,-3.230390746011835290e-07,-3.219104191543031845e-07,-3.207856892339933097e-07,-3.196648721837114238e-07,-3.185479535705582066e-07,-3.174349203494131189e-07,-3.163257591420887138e-07,-3.152204568479533009e-07,-3.141189993116633161e-07,-3.130213734880982202e-07,-3.119275674978716495e-07,-3.108375666305285279e-07,-3.097513582850375258e-07,-3.086689293607669526e-07,-3.075902664795293617e-07,-3.065153575954049359e-07,-3.054441891081616234e-07,-3.043767481392123386e-07,-3.033130215879253910e-07,-3.022529971308252073e-07,-3.011966620558581553e-07,-3.001440027072810324e-07,-2.990950073722409286e-07,-2.980496635052176657e-07,-2.970079576170014946e-07,-2.959698774396279930e-07,-2.949354109271773439e-07,-2.939045453120847640e-07,-2.928772680488300750e-07,-2.918535669249600062e-07,-2.908334295614878329e-07,-2.898168436349379817e-07,-2.888037970438794844e-07,-2.877942769097252551e-07,-2.867882716306446866e-07,-2.857857692162291130e-07,-2.847867570099360535e-07,-2.837912237430018081e-07,-2.827991565368392912e-07,-2.818105440671736517e-07,-2.808253741770627698e-07,-2.798436345985422236e-07,-2.788653140073371617e-07,-2.778904001354831621e-07,-2.769188816031942224e-07,-2.759507470306843402e-07,-2.749859838169221860e-07,-2.740245804155883036e-07,-2.730665261685416567e-07,-2.721118085302620671e-07,-2.711604159544300785e-07,-2.702123378384158059e-07,-2.692675625248774907e-07,-2.683260775793172570e-07,-2.673878722880829173e-07,-2.664529357154776790e-07,-2.655212559821151785e-07,-2.645928217637205648e-07,-2.636676220135747428e-07,-2.627456459625143737e-07,-2.618268824527980598e-07,-2.609113196050394379e-07,-2.599989468166086226e-07,-2.590897529297642166e-07,-2.581837267867648222e-07,-2.572808582290697643e-07,-2.563811357103595867e-07,-2.554845481284040432e-07,-2.545910850471067022e-07,-2.537007356417930737e-07,-2.528134888657440627e-07,-2.519293343383743888e-07,-2.510482606798980498e-07,-2.501702582313747314e-07,-2.492953159460853385e-07,-2.484234226662884737e-07,-2.475545686775326715e-07,-2.466887429330988368e-07,-2.458259354409797481e-07,-2.449661349879228567e-07,-2.441093313598763359e-07,-2.432555152309667790e-07,-2.424046749438524273e-07,-2.415568001179480007e-07,-2.407118818159581508e-07,-2.398699091021860852e-07,-2.390308720956468846e-07,-2.381947596385991517e-07,-2.373615620721025721e-07,-2.365312698482391340e-07,-2.357038718092674401e-07,-2.348793593509590494e-07,-2.340577217041506231e-07,-2.332389488213237883e-07,-2.324230312100716844e-07,-2.316099580457198215e-07,-2.307997202244393975e-07,-2.299923084203570056e-07,-2.291877120863539119e-07,-2.283859209528671386e-07,-2.275869265822016985e-07,-2.267907186492834626e-07,-2.259972870510829068e-07,-2.252066233499050441e-07,-2.244187166100530817e-07,-2.236335578942316715e-07,-2.228511381541231628e-07,-2.220714471201645779e-07,-2.212944756885271147e-07,-2.205202144223150640e-07,-2.197486536070769603e-07,-2.189797844165397578e-07,-2.182135973027854448e-07,-2.174500828844294631e-07,-2.166892323351987670e-07,-2.159310353300192276e-07,-2.151754829871066477e-07,-2.144225673683664013e-07,-2.136722787593470230e-07,-2.129246070015078374e-07,-2.121795438236873110e-07,-2.114370803996123982e-07,-2.106972072368762383e-07,-2.099599155647169368e-07,-2.092251967233949017e-07,-2.084930411094809699e-07,-2.077634402852801543e-07,-2.070363857575863165e-07,-2.063118679229702934e-07,-2.055898777886255857e-07,-2.048704075829910209e-07,-2.041534487018381583e-07,-2.034389905980482638e-07,-2.027270259441493749e-07,-2.020175470240914706e-07,-2.013105434572892705e-07,-2.006060067505366362e-07,-1.999039300204508152e-07,-1.992043035525803418e-07,-1.985071180765629606e-07,-1.978123660428821040e-07,-1.971200395134431460e-07,-1.964301296064618896e-07,-1.957426274401541377e-07,-1.950575245768249033e-07,-1.943748135779799213e-07,-1.936944858393907509e-07,-1.930165324792731951e-07,-1.923409462811775938e-07,-1.916677188074089599e-07,-1.909968407320938866e-07,-1.903283044502046550e-07,-1.896621028563139078e-07,-1.889982262914813305e-07,-1.883366679278353217e-07,-1.876774192721697432e-07,-1.870204719423007589e-07,-1.863658189993344649e-07,-1.857134511729086057e-07,-1.850633609135066138e-07,-1.844155405605896192e-07,-1.837699828977079619e-07,-1.831266784324547814e-07,-1.824856195598023589e-07,-1.818468007841467227e-07,-1.812102126130810120e-07,-1.805758466089102399e-07,-1.799436958882516535e-07,-1.793137527350552318e-07,-1.786860093222486512e-07,-1.780604582668487978e-07,-1.774370919638279531e-07,-1.768159021420245836e-07,-1.761968823621451463e-07,-1.755800241309835030e-07,-1.749653198990230862e-07,-1.743527632824815043e-07,-1.737423460101972239e-07,-1.731340601995867701e-07,-1.725278989672673902e-07,-1.719238556519009364e-07,-1.713219212606809094e-07,-1.707220897984029762e-07,-1.701243536600394179e-07,-1.695287053515848186e-07,
        -1.689351376565895180e-07,-1.683436430810480999e-07,-1.677542150191335679e-07,-1.671668459213293545e-07,-1.665815292928307656e-07,-1.659982565294093604e-07,-1.654170211362604448e-07,-1.648378170071573834e-07,-1.642606358709386427e-07,-1.636854717323998898e-07,-1.631123165424241961e-07,-1.625411633621176577e-07,-1.619720065293428490e-07,-1.614048382170274465e-07,-1.608396505980991265e-07,-1.602764384545096732e-07,-1.597151943477648217e-07,-1.591559103397699459e-07,-1.585985812679879814e-07,-1.580431995829023606e-07,-1.574897579570411210e-07,-1.569382504507110809e-07,-1.563886701805294877e-07,-1.558410099855578324e-07,-1.552952642591698407e-07,-1.547514252853154915e-07,-1.542094863471454858e-07,-1.536694415049666418e-07,-1.531312837088627532e-07,-1.525950070191406382e-07,-1.520606044969063930e-07,-1.515280691477549624e-07,-1.509973948099485597e-07,-1.504685757103274568e-07,-1.499416046879531450e-07,-1.494164751369986277e-07,-1.488931810067484207e-07,-1.483717156358643763e-07,-1.478520734177202200e-07,-1.473342465807547796e-07,-1.468182299624309906e-07,-1.463040176785668223e-07,-1.457916020131122536e-07,-1.452809776369967665e-07,-1.447721381664379692e-07,-1.442650769956088652e-07,-1.437597884623720290e-07,-1.432562656833447079e-07,-1.427545040511013497e-07,-1.422544975704376213e-07,-1.417562360828128476e-07,-1.412597167016471644e-07,-1.407649350970707758e-07,-1.402718816101433674e-07,-1.397805517444616896e-07,-1.392909407260667365e-07,-1.388030409499307893e-07,-1.383168465318718177e-07,-1.378323519207746983e-07,-1.373495515100131570e-07,-1.368684383051821385e-07,-1.363890079209006956e-07,-1.359112542509421928e-07,-1.354351696902789115e-07,-1.349607496869964507e-07,-1.344879898557138631e-07,-1.340168828134480350e-07,-1.335474212327270038e-07,-1.330796020604374519e-07,-1.326134194123973487e-07,-1.321488664607350927e-07,-1.316859377653578633e-07,-1.312246277751505374e-07,-1.307649311610425968e-07,-1.303068412616958938e-07,-1.298503539137740859e-07,-1.293954625669613279e-07,-1.289421620587205553e-07,-1.284904476150927621e-07,-1.280403117420725323e-07,-1.275917501653012209e-07,-1.271447574446860074e-07,-1.266993273629779537e-07,-1.262554554792849615e-07,-1.258131369086257223e-07,-1.253723638794390638e-07,-1.249331327835001559e-07,-1.244954380696938756e-07,-1.240592737983270410e-07,-1.236246353619741001e-07,-1.231915173205422320e-07,-1.227599137898494064e-07,-1.223298195518474074e-07,-1.219012296660437755e-07,-1.214741389143902950e-07,-1.210485426894614136e-07,-1.206244347184970422e-07,-1.202018100054935701e-07,-1.197806635544473863e-07,-1.193609897587322166e-07,-1.189427848991009284e-07,-1.185260425917711302e-07,-1.181107578407392111e-07,-1.176969259830684678e-07,-1.172845414121326257e-07,-1.168735999645953427e-07,-1.164640955342299833e-07,-1.160560234580998440e-07,-1.156493798504243387e-07,-1.152441587159991343e-07,-1.148403538930864443e-07,-1.144379623285729508e-07,-1.140369791929884968e-07,-1.136373976029503297e-07,-1.132392149494343414e-07,-1.128424262919480725e-07,-1.124470240254638043e-07,-1.120530053744239751e-07,-1.116603660644699403e-07,-1.112691001559085180e-07,-1.108792028192695511e-07,-1.104906697801943949e-07,-1.101034968198355557e-07,-1.097176769437879784e-07,-1.093332073209829503e-07,-1.089500845652402461e-07,-1.085683009605098448e-07,-1.081878533426561262e-07,-1.078087376593650504e-07,-1.074309483040103430e-07,-1.070544803360995445e-07,-1.066793293702517076e-07,-1.063054915206862461e-07,-1.059329621799776078e-07,-1.055617365741667868e-07,-1.051918090966275088e-07,-1.048231763056683974e-07,-1.044558339269308078e-07,-1.040897769644111293e-07,-1.037250009772172632e-07,-1.033615006362786914e-07,-1.029992729439932475e-07,-1.026383130708907743e-07,-1.022786157989230560e-07,-1.019201791851997996e-07,-1.015629965683828573e-07,-1.012070613981563838e-07,-1.008523736190092279e-07,-1.004989270136924517e-07,-1.001467173633585617e-07,-9.979574061569351784e-08,-9.944599144162680204e-08,-9.909746684355624780e-08,-9.875016193650054674e-08,-9.840407327876832255e-08,-9.805919554128905702e-08,-9.771552522686022257e-08,-9.737305639578863747e-08,-9.703178632802789139e-08,-9.669171208148696905e-08,-9.635282821607304982e-08,-9.601513034640518640e-08,-9.567861219972328968e-08,-9.534327394256081334e-08,-9.500910930215766825e-08,-9.467611328251024361e-08,-9.434428555055163201e-08,-9.401361777960914878e-08,-9.368410591736875404e-08,-9.335574890911857437e-08,-9.302854137027694037e-08,-9.270247908199635845e-08,-9.237755854707430103e-08,-9.205377471399600608e-08,-9.173112458515930712e-08,-9.140960316456059331e-08,-9.108920867584302528e-08,-9.076993540135802618e-08,-9.045177923328040492e-08,-9.013473767360835609e-08,-8.981880450509294178e-08,-8.950397883955574230e-08,-8.919025540343739067e-08,-8.887762825704470515e-08,-8.856609734486653451e-08,-8.825565722681005809e-08,-8.794630296238281630e-08,-8.763803244216106236e-08,-8.733084078116348792e-08,-8.702472542587713633e-08,-8.671968021456422093e-08,-8.641570348189020478e-08,-8.611279184167486278e-08,-8.581094074200379396e-08,-8.551014640811871459e-08,-8.521040378850486263e-08,-8.491171182845036469e-08,-8.461406569848506365e-08,-8.431746034709419746e-08,-8.402189555223316120e-08,-8.372736648443179774e-08,-8.343386620479620319e-08,-8.314139338105874799e-08,-8.284994562623992920e-08,-8.255951811086958969e-08,-8.227010772632326052e-08,-8.198171141948762397e-08,-8.169432430538137169e-08,-8.140794355293579088e-08,-8.112256599801526136e-08,-8.083818653359386985e-08,-8.055480310575902081e-08,-8.027241193975243050e-08,-7.999101087063920090e-08,-7.971059495792687244e-08,-7.943115920561183430e-08,-7.915270416880559878e-08,-7.887522374128153047e-08,-7.859871364868098453e-08,-7.832317205913597036e-08,-7.804859741833425346e-08,-7.777498339800459348e-08,-7.750232605685525300e-08,-7.723062467324126601e-08,-7.695987430667017293e-08,-7.669007329180743682e-08,-7.642121735429441287e-08,-7.615330133159403658e-08,-7.588632428001673702e-08,-7.562028186969271815e-08,-7.535517176915362825e-08,-7.509099009261888114e-08,-7.482773167755141230e-08,-7.456539824479690992e-08,-7.430398368812873855e-08,-7.404348206785371644e-08,-7.378389349499414607e-08,-7.352521602665973433e-08,-7.326744311253463593e-08,-7.301057181052783562e-08,-7.275460134348321617e-08,-7.249952782562019138e-08,-7.224534820382544353e-08,-7.199205792618457167e-08,-7.173965349549504822e-08,-7.148813530033493180e-08,-7.123749778958909928e-08,-7.098773724401041818e-08,-7.073885210928665401e-08,-7.049083899923758167e-08,-7.024369297337074158e-08,-6.999741314350771404e-08,-6.975199701164669364e-08,-6.950744074751824542e-08,-6.926374124249790043e-08,-6.902089261240362816e-08,-6.877889718870378033e-08,-6.853774903170517518e-08,-6.829744531033909993e-08,-6.805798374864835409e-08,-6.781935874000666331e-08,-6.758157211628201821e-08,-6.734461899249311045e-08,-6.710849376201366567e-08,-6.687319620279907895e-08,-6.663872487155941826e-08,-6.640507294042308217e-08,-6.617223891058898744e-08,-6.594022228245677297e-08,-6.570901656122174472e-08,-6.547862219097311254e-08,-6.524903545246374392e-08,-6.502025146071233053e-08,-6.479227010469656989e-08,-6.456508677699090981e-08,-6.433869814692627642e-08,-6.411310421450266972e-08,-6.388830198211792322e-08,-6.366428539905655271e-08,-6.344105196731675278e-08,-6.321860329672190915e-08,-6.299693450229071345e-08,-6.277604025495264750e-08,-6.255591938897353543e-08,-6.233657123821956247e-08,-6.211799230548820105e-08,-6.190017715068663051e-08,-6.168312449705837253e-08,-6.146683517727069557e-08,-6.125130308509696420e-08,-6.103652422373428976e-08,-6.082249998096145305e-08,-6.060922430606296984e-08,-6.039669486757048844e-08,-6.018491155446170637e-08,-5.997387048095603745e-08,-5.976356842740671027e-08,-5.955400211865580218e-08,-5.934517022243568363e-08,-5.913707185056793492e-08,-5.892970217358239893e-08,-5.872305830489921163e-08,-5.851713907878419718e-08,-5.831193938821144229e-08,-5.810745895562519081e-08,-5.790369833613695505e-08,-5.770064942511865524e-08,-5.749831150092532539e-08,-5.729668384191199948e-08,-5.709576084145240316e-08,-5.689554427590337582e-08,-5.669602803903828203e-08,-5.649721035450028239e-08,-5.629909000104404981e-08,-5.610166209368827595e-08,-5.590492874185670757e-08,-5.570888444994537281e-08,-5.551352499910677807e-08,-5.531885144405279675e-08,-5.512486112024816975e-08,-5.493154858760007642e-08,-5.473891379059736551e-08,-5.454695461981629023e-08,-5.435566835521044027e-08,-5.416505277633376636e-08,-5.397510288718265770e-08,-5.378582007553589506e-08,-5.359720256503663904e-08,-5.340924219554565866e-08,-5.322194179813166670e-08,-5.303530048461624347e-08,-5.284930987281555304e-08,-5.266397246073140082e-08,-5.247928386298283954e-08,-5.229524380201411304e-08,-5.211184833653348392e-08,-5.192909313667115612e-08,-5.174698153309620352e-08,-5.156550814122695670e-08,-5.138466868670477083e-08,-5.120446322504079717e-08,-5.102488931374438152e-08,-5.084594534299213819e-08,-5.066762814864844700e-08,-5.048993656497913207e-08,-5.031286820500469048e-08,-5.013641968254489711e-08,-4.996058983186557612e-08,-4.978537404554117529e-08,-4.961077343379471927e-08,-4.943678660884742726e-08,-4.926340885225144461e-08,-4.909063638924848760e-08,-4.891847110721769809e-08,-4.874690984202345589e-08,-4.857594881890747729e-08,-4.840558909258163567e-08,-4.823582538948656406e-08,-4.806665471202009599e-08,-4.789807833693870975e-08,-4.773009282255102903e-08,-4.756269567085524841e-08,-4.739588699287367035e-08,-4.722966140402462543e-08,-4.706401851573005501e-08,-4.689895649612196848e-08,-4.673447068226366241e-08,-4.657056346113463974e-08,-4.640723150206582659e-08,-4.624447114132124170e-08,-4.608228215685628015e-08,-4.592066216169143900e-08,-4.575960543817814141e-08,-4.559911376267322680e-08,-4.543918818988856856e-08,-4.527981961599536476e-08,-4.512101192677420158e-08,-4.496276051479952685e-08,-4.480506166082420805e-08,-4.464791886205077276e-08,-4.449132506856301461e-08,-4.433527911462675775e-08,-4.417978149984236325e-08,-4.402482717269506907e-08,
        -4.387041713238559737e-08,-4.371655204504776293e-08,-4.356322236276355397e-08,-4.341042769695491188e-08,-4.325817237749163269e-08,-4.310645201899276913e-08,-4.295526162545471038e-08,-4.280459880989795352e-08,-4.265446773565884087e-08,-4.250486346224491285e-08,-4.235577916178456803e-08,-4.220721966374796352e-08,-4.205918019417609344e-08,-4.191166130818047009e-08,-4.176466056327043930e-08,-4.161817235281972671e-08,-4.147219989647510374e-08,-4.132673792067720342e-08,-4.118178403844652280e-08,-4.103734246863055546e-08,-4.089340460700086055e-08,-4.074997123071355531e-08,-4.060704206221288359e-08,-4.046461110629451241e-08,-4.032268130504945702e-08,-4.018124910576403863e-08,-4.004031323168177892e-08,-3.989987173991238478e-08,-3.975992174387599221e-08,-3.962046107863770317e-08,-3.948149013277557628e-08,-3.934300951691227510e-08,-3.920501046028590508e-08,-3.906749462823100316e-08,-3.893046385261555997e-08,-3.879391163863488146e-08,-3.865783942957889963e-08,-3.852224322864472583e-08,-3.838712231418739407e-08,-3.825247557598387971e-08,-3.811830084909928473e-08,-3.798459546899835004e-08,-3.785136076794870519e-08,-3.771859335977012506e-08,-3.758629013583814071e-08,-3.745445109615275214e-08,-3.732307241044452439e-08,-3.719215607711490179e-08,-3.706170115247431340e-08,-3.693169986496158685e-08,-3.680215404644471278e-08,-3.667306380794599363e-08,-3.654442598532980924e-08,-3.641624074512961329e-08,-3.628850298031949251e-08,-3.616121296845520305e-08,-3.603437093158134985e-08,-3.590797309493964917e-08,-3.578201557274951483e-08,-3.565650352754801133e-08,-3.553143079759735201e-08,-3.540679416325076545e-08,-3.528259601148775459e-08,-3.515883134630470863e-08,-3.503550544126099453e-08,-3.491261202359652316e-08,-3.479014443197314677e-08,-3.466810699626066139e-08,-3.454649816214683256e-08,-3.442531493202949378e-08,-3.430456013697735784e-08,-3.418422778178609178e-08,-3.406431586805425127e-08,-3.394482545049370970e-08,-3.382575169963430994e-08,-3.370709622529943772e-08,-3.358885702908764870e-08,-3.347103438855469904e-08,-3.335362402934194392e-08,-3.323662445264830012e-08,-3.312003432620613808e-08,-3.300385292837049178e-08,-3.288807892687373169e-08,-3.277271448665075582e-08,-3.265775405658644104e-08,-3.254319308476638639e-08,-3.242903523492657314e-08,-3.231527634373065894e-08,-3.220191790997972703e-08,-3.208895649198240108e-08,-3.197639031338184168e-08,-3.186422103951258578e-08,-3.175244450703829102e-08,-3.164105827346830324e-08,-3.153006405964831060e-08,-3.141945992268802001e-08,-3.130924164373993790e-08,-3.119940894524830810e-08,-3.108996421419263356e-08,-3.098090423092614287e-08,-3.087222344433371291e-08,-3.076392335321642690e-08,-3.065600578944227550e-08,-3.054846320349469124e-08,-3.044130025831037756e-08,-3.033451584366630982e-08,-3.022809952346605655e-08,-3.012205840313697536e-08,-3.001639142796719284e-08,-2.991109515626533266e-08,-2.980616464753893524e-08,-2.970160523085851878e-08,-2.959741046693054045e-08,-2.949357885695391701e-08,-2.939012017089126516e-08,-2.928701897664254261e-08,-2.918427499665199321e-08,-2.908189666861460410e-08,-2.897987644301380783e-08,-2.887821171082549654e-08,-2.877690596925219779e-08,-2.867595494393526678e-08,-2.857535785771858627e-08,-2.847511271220071194e-08,-2.837522039556006348e-08,-2.827567846530598672e-08,-2.817648342423595409e-08,-2.807763627155068775e-08,-2.797913756236170002e-08,-2.788098663053517612e-08,-2.778317725882217815e-08,-2.768570872557774010e-08,-2.758858391738172600e-08,-2.749179867089779350e-08,-2.739535365225975738e-08,-2.729924819533380287e-08,-2.720347919149546101e-08,-2.710804475336558994e-08,-2.701294704587908768e-08,-2.691817962974241141e-08,-2.682374694584765962e-08,-2.672964660721532937e-08,-2.663587256312993645e-08,-2.654243141941847739e-08,-2.644931723638777044e-08,-2.635652557314571709e-08,-2.626406053751750846e-08,-2.617192423892689135e-08,-2.608010635229973673e-08,-2.598861259528462142e-08,-2.589743947067901786e-08,-2.580658731154983343e-08,-2.571605317580605288e-08,-2.562583251153327524e-08,-2.553593309029267289e-08,-2.544634597478889759e-08,-2.535707521733598924e-08,-2.526811776482063010e-08,-2.517946862123920937e-08,-2.509113217197267431e-08,-2.500310880559908355e-08,-2.491539358162597750e-08,-2.482798172609435028e-08,-2.474087779091860284e-08,-2.465408460716744798e-08,-2.456758885216459021e-08,-2.448139646560321125e-08,-2.439551527455563473e-08,-2.430992607216353463e-08,-2.422463768469995671e-08,-2.413965255465555515e-08,-2.405496035695620094e-08,-2.397057130565372063e-08,-2.388647907247687385e-08,-2.380267449808570746e-08,-2.371916751897629183e-08,-2.363595447141264572e-08,-2.355303208023684647e-08,-2.347040034544889409e-08,-2.338805771273655409e-08,-2.330600351596601172e-08,-2.322423925393835020e-08,-2.314276281722982276e-08,-2.306156782205803779e-08,-2.298066076322768936e-08,-2.290003792149164497e-08,-2.281969557760277212e-08,-2.273963634058517869e-08,-2.265985732385900064e-08,-2.258035552982207150e-08,-2.250113523283303607e-08,-2.242219759862607020e-08,-2.234352980412523948e-08,-2.226513934333596012e-08,-2.218702777057046660e-08,-2.210918675915607423e-08,-2.203162036140682289e-08,-2.195432163842880868e-08,-2.187729686298212073e-08,-2.180054148315235807e-08,-2.172405894063089704e-08,-2.164784279612419482e-08,-2.157189110674195831e-08,-2.149621025626657911e-08,-2.142079036371313805e-08,-2.134563870104244643e-08,-2.127074683055951709e-08,-2.119612024786832194e-08,-2.112175884194655850e-08,-2.104765339794312240e-08,-2.097380841226126336e-08,-2.090022255263335182e-08,-2.082689348759103609e-08,-2.075382266042424817e-08,-2.068101556673695995e-08,-2.060845338824890405e-08,-2.053614633901190700e-08,-2.046409947054073086e-08,-2.039230090344901214e-08,-2.032075685498568873e-08,-2.024946055279031043e-08,-2.017841632673267327e-08,-2.010761718240772211e-08,-2.003707127995468795e-08,-1.996677545523795061e-08,-1.989671810642690275e-08,-1.982691089086330294e-08,-1.975735097747843838e-08,-1.968803331475754703e-08,-1.961895801372293136e-08,-1.955012296495084456e-08,-1.948152800190783296e-08,-1.941317556708455072e-08,-1.934506838052740818e-08,-1.927719506245040293e-08,-1.920955744472152560e-08,-1.914216440912497319e-08,-1.907500124520566942e-08,-1.900807533594672805e-08,-1.894138651481469537e-08,-1.887493128460704384e-08,-1.880871031145758820e-08,-1.874271154944651130e-08,-1.867695248458645096e-08,-1.861142950865257717e-08,-1.854612863283477964e-08,-1.848105457558091302e-08,-1.841621233289458814e-08,-1.835160068353047791e-08,-1.828721640784181091e-08,-1.822305079057784383e-08,-1.815910982694290965e-08,-1.809539945663019012e-08,-1.803191562732564535e-08,-1.796864551595334092e-08,-1.790559828185322999e-08,-1.784277692262747905e-08,-1.778017444387103296e-08,-1.771779167825116019e-08,-1.765562546163224056e-08,-1.759368162268515334e-08,-1.753194911469080353e-08,-1.747043515409885117e-08,-1.740914312708952139e-08,-1.734806015507572852e-08,-1.728718801441431197e-08,-1.722653514280025888e-08,-1.716609310253858212e-08,-1.710586627901022894e-08,-1.704584862149971514e-08,-1.698603918631746978e-08,-1.692644402417897709e-08,-1.686705514147845975e-08,-1.680787514724002563e-08,-1.674889932301582007e-08,-1.669013255378715144e-08,-1.663157511710977587e-08,-1.657321890835561362e-08,-1.651506964517324150e-08,-1.645712721654035704e-08,-1.639938002062635292e-08,-1.634184298993091033e-08,-1.628450607693565644e-08,-1.622736522932655134e-08,-1.617042727497519650e-08,-1.611369121468086973e-08,-1.605716076769070355e-08,-1.600081728225788424e-08,-1.594467796683929350e-08,-1.588873294045001217e-08,-1.583297959406593236e-08,-1.577743025116262743e-08,-1.572206914657314769e-08,-1.566690610577126108e-08,-1.561193624377565925e-08,-1.555715845036331757e-08,-1.550257200388927004e-08,-1.544817640475315557e-08,-1.539397609384707266e-08,-1.533996391023251249e-08,-1.528613902124220658e-08,-1.523250409141141404e-08,-1.517906167425309150e-08,-1.512579900220245577e-08,-1.507272784362356788e-08,-1.501984103757791900e-08,-1.496713830650975297e-08,-1.491462303659929489e-08,-1.486228889957530441e-08,-1.481014805237990117e-08,-1.475818539597995027e-08,-1.470639615641644582e-08,-1.465479715356821089e-08,-1.460337656356003322e-08,-1.455213322065773696e-08,-1.450108055855992006e-08,-1.445019320867046986e-08,-1.439948960069159511e-08,-1.434897672902835097e-08,-1.429862672708281934e-08,-1.424844753294962629e-08,-1.419845269134967225e-08,-1.414863753934625379e-08,-1.409900102222749751e-08,-1.404952792993796606e-08,-1.400022081599061607e-08,-1.395110366120277945e-08,-1.390215359498014891e-08,-1.385337250470186632e-08,-1.380476744028413805e-08,-1.375632346922728289e-08,-1.370805713385436775e-08,-1.365996188384954735e-08,-1.361202867089517099e-08,-1.356426615473083075e-08,-1.351666856219679858e-08,-1.346924644041180841e-08,-1.342199029696899970e-08,-1.337489013986115083e-08,-1.332796012443182576e-08,-1.328119797472382402e-08,-1.323459236646229442e-08,-1.318815456841093692e-08,-1.314188236012370226e-08,-1.309576663777178851e-08,-1.304981750438471977e-08,-1.300402618920060149e-08,-1.295839774373419573e-08,-1.291293222349665371e-08,-1.286762135732644197e-08,-1.282247313882933781e-08,-1.277747913031035409e-08,-1.273264244039395976e-08,-1.268796873121758040e-08,-1.264345173002112688e-08,-1.259908471995530022e-08,-1.255487186435644276e-08,-1.251082287767601997e-08,-1.246692199474708218e-08,-1.242317720917540669e-08,-1.237959129651855505e-08,-1.233614993489950962e-08,-1.229286256121397969e-08,-1.224973572577781056e-08,-1.220674783475317327e-08,-1.216391359859514409e-08,-1.212123623695049446e-08,-1.207870187203141654e-08,-1.203631966317786350e-08,-1.199408378171895606e-08,-1.195199711423455824e-08,-1.191006426815022223e-08,-1.186827525145872642e-08,-1.182662323628846934e-08,-1.178512643029705487e-08,-1.174377955992511602e-08,-1.170257241112082625e-08,-1.166150642717411756e-08,-1.162058826942313772e-08,-1.157980555888116214e-08,-1.153917461582665283e-08,-1.149869516270385361e-08,-1.145833983251520749e-08,-1.141813499305754931e-08,
        -1.137807065232365744e-08,-1.133814686582468312e-08,-1.129836163515918201e-08,-1.125870902063397239e-08,-1.121920917279695118e-08,-1.117984160803331406e-08,-1.114061470852689695e-08,-1.110152625383165059e-08,-1.106256736216337799e-08,-1.102375213335449189e-08,-1.098507140806503912e-08,-1.094652463118350738e-08,-1.090811319048867745e-08,-1.086983320019996313e-08,-1.083169359761271267e-08,-1.079368888712295416e-08,-1.075580902121231475e-08,-1.071807576025207709e-08,-1.068046329155691865e-08,-1.064298166264521228e-08,-1.060564241983641409e-08,-1.056842535707147590e-08,-1.053134385253784444e-08,-1.049438691502757592e-08,-1.045756148343457426e-08,-1.042086628100236112e-08,-1.038430158528669267e-08,-1.034786262232856302e-08,-1.031155505426539776e-08,-1.027537488429430823e-08,-1.023931273103073636e-08,-1.020338297186285104e-08,-1.016758627292446704e-08,-1.013190736864899577e-08,-1.009635280935228252e-08,-1.006093125477391936e-08,-1.002562249885485812e-08,-9.990437421780740124e-09,-9.955384294713098825e-09,-9.920452570533200287e-09,-9.885636809148223847e-09,-9.850946391942727587e-09,-9.816388091277161720e-09,-9.781943532960468701e-09,-9.747615381527907630e-09,-9.713413073875187820e-09,-9.679322510169896532e-09,-9.645361287446974075e-09,-9.611524964814321947e-09,-9.577791670878355035e-09,-9.544186330145976171e-09,-9.510695342385133699e-09,-9.477309659278176923e-09,-9.444058757246409641e-09,-9.410927981345906801e-09,-9.377898180229493619e-09,-9.344993889826014311e-09,-9.312210891287975301e-09,-9.279526202998766848e-09,-9.246961751863125301e-09,-9.214509377741819662e-09,-9.182185234379858230e-09,-9.149969337762797750e-09,-9.117849919526577196e-09,-9.085865670943604755e-09,-9.053979233009101790e-09,-9.022199543018416534e-09,-8.990543920450733140e-09,-8.959001207564654123e-09,-8.927570183114852398e-09,-8.896238468114603393e-09,-8.865014111680835640e-09,-8.833906439686955991e-09,-8.802909290395177777e-09,-8.772024939762701479e-09,-8.741244839072237482e-09,-8.710562215963335575e-09,-8.680004270900099073e-09,-8.649546801020591147e-09,-8.619200131398940812e-09,-8.588951827537272266e-09,-8.558798170188453014e-09,-8.528778072669496169e-09,-8.498849068949709817e-09,-8.469014212142411679e-09,-8.439310639207775466e-09,-8.409699214784183141e-09,-8.380179938871634704e-09,-8.350778069043940377e-09,-8.321475786221554927e-09,-8.292273423471385740e-09,-8.263168982391988493e-09,-8.234171400278711417e-09,-8.205281065709613131e-09,-8.176499699530381804e-09,-8.147806929148515565e-09,-8.119202143941350869e-09,-8.090718817133080165e-09,-8.062329803770751369e-09,-8.034031551140685679e-09,-8.005842933034301723e-09,-7.977750293708396612e-09,-7.949757574454707765e-09,-7.921856115533643106e-09,-7.894057019175448886e-09,-7.866362117248115737e-09,-7.838762083878236808e-09,-7.811257807244231799e-09,-7.783842181918743108e-09,-7.756518649593147074e-09,-7.729302642367485987e-09,-7.702188720148939183e-09,-7.675152569053267371e-09,-7.648216782119021673e-09,-7.621392461576448341e-09,-7.594644635400271682e-09,-7.567985516043762573e-09,-7.541442026415268174e-09,-7.514973032751726123e-09,-7.488598796623335829e-09,-7.462336304442374058e-09,-7.436139370931016401e-09,-7.410034308374946477e-09,-7.384039435454070599e-09,-7.358130715839905633e-09,-7.332308704643963893e-09,-7.306575566801143395e-09,-7.280932690090224924e-09,-7.255391842875269504e-09,-7.229929543939306313e-09,-7.204554786088834817e-09,-7.179284167158073160e-09,-7.154085879257365832e-09,-7.128975798575964973e-09,-7.103965804500234071e-09,-7.079042463331575163e-09,-7.054197059819244942e-09,-7.029436199790239925e-09,-7.004770430363294054e-09,-6.980193811845225582e-09,-6.955698461652559672e-09,-6.931287654943218968e-09,-6.906960059449573919e-09,-6.882724612466972758e-09,-6.858582646263045035e-09,-6.834522447984880955e-09,-6.810527308775959909e-09,-6.786617934295691157e-09,-6.762806259441589418e-09,-6.739072355710362672e-09,-6.715438150006747264e-09,-6.691885434673139343e-09,-6.668384955332840036e-09,-6.644975736325164917e-09,-6.621662163031061255e-09,-6.598436963489717755e-09,-6.575288480359375853e-09,-6.552196452069836141e-09,-6.529197071891701398e-09,-6.506302496767091270e-09,-6.483466596929332582e-09,-6.460707413502575491e-09,-6.438053867796611485e-09,-6.415447062479984197e-09,-6.392931017895620016e-09,-6.370505567510065248e-09,-6.348135517253439275e-09,-6.325869716938825604e-09,-6.303684019215438639e-09,-6.281557440868112963e-09,-6.259514739870297717e-09,-6.237548089149669295e-09,-6.215656156438598146e-09,-6.193845714097534483e-09,-6.172109490165667012e-09,-6.150450260200557295e-09,-6.128865914778458546e-09,-6.107356953499731844e-09,-6.085930981392095873e-09,-6.064567403818443836e-09,-6.043288924839629317e-09,-6.022099374725087273e-09,-6.000950458773246510e-09,-5.979882755635657077e-09,-5.958904203406945044e-09,-5.938001979011175990e-09,-5.917178802494760248e-09,-5.896389043691385723e-09,-5.875696262869212205e-09,-5.855094298290453025e-09,-5.834544569705002459e-09,-5.814065118237010665e-09,-5.793659163533249057e-09,-5.773333533465319078e-09,-5.753075849046496160e-09,-5.732880392628203481e-09,-5.712758210929536062e-09,-5.692717686134329824e-09,-5.672743552675996170e-09,-5.652838697134399126e-09,-5.632998956173196348e-09,-5.613238485135951805e-09,-5.593538243697793177e-09,-5.573889738652582082e-09,-5.554341653279948332e-09,-5.534862013156782723e-09,-5.515426115820787345e-09,-5.496073374189336391e-09,-5.476793851766359467e-09,-5.457582552548245758e-09,-5.438413108738160417e-09,-5.419325821431897339e-09,-5.400318692228012196e-09,-5.381362466749806117e-09,-5.362483401771811486e-09,-5.343671893864865297e-09,-5.324911400705900633e-09,-5.306218631151438103e-09,-5.287610460591452011e-09,-5.269051195355700656e-09,-5.250561430081290837e-09,-5.232123345688677318e-09,-5.213765530420744199e-09,-5.195485819342593459e-09,-5.177242135001591805e-09,-5.159081162275924726e-09,-5.140969483452551003e-09,-5.122923307787630165e-09,-5.104968503477635977e-09,-5.087045062968087450e-09,-5.069188513395772588e-09,-5.051401297251345568e-09,-5.033669536746998574e-09,-5.016013826519838403e-09,-4.998400193745311526e-09,-4.980863221870635016e-09,-4.963398025914500522e-09,-4.945973131054159921e-09,-4.928612906685003736e-09,-4.911310025335069440e-09,-4.894082639150809655e-09,-4.876919423857373204e-09,-4.859795121880949864e-09,-4.842736989196794184e-09,-4.825746802161745563e-09,-4.808813847123616370e-09,-4.791933627679156871e-09,-4.775123241262946294e-09,-4.758372640356611782e-09,-4.741665171614783958e-09,-4.725029978391859231e-09,-4.708444356626984018e-09,-4.691908639387065705e-09,-4.675448583846275596e-09,-4.659059749112515192e-09,-4.642696005419111316e-09,-4.626406757690659788e-09,-4.610195780685444333e-09,-4.594000846402934712e-09,-4.577866086208359775e-09,-4.561810262870835686e-09,-4.545794740629105490e-09,-4.529838726341495203e-09,-4.513959039886827895e-09,-4.498105721228995435e-09,-4.482323789911646372e-09,-4.466602865349500462e-09,-4.450916413212269163e-09,-4.435306066863375918e-09,-4.419739241257047979e-09,-4.404235309785065056e-09,-4.388782504083366121e-09,-4.373371109700485704e-09,-4.358014782379626695e-09,-4.342720183458936845e-09,-4.327487646005323541e-09,-4.312291801422674098e-09,-4.297173339384841029e-09,-4.282091459195669358e-09,-4.267071751495876697e-09,-4.252109053748398537e-09,-4.237168449439110418e-09,-4.222295013622812121e-09,-4.207487247498420402e-09,-4.192720226559032426e-09,-4.178008994326631864e-09,-4.163335620699371020e-09,-4.148722032581986241e-09,-4.134176223580254828e-09,-4.119662888601993700e-09,-4.105225104300558314e-09,-4.090816185797763183e-09,-4.076438298028506324e-09,-4.062137182181402295e-09,-4.047896740022594031e-09,-4.033684164461703858e-09,-4.019501176344419946e-09,-4.005429099507296087e-09,-3.991385666424207557e-09,-3.977329299242882144e-09,-3.963383399252506933e-09,-3.949502724864828451e-09,-3.935637649110645953e-09,-3.921817703922414466e-09,-3.908064816204870340e-09,-3.894357780698243232e-09,-3.880680665702129772e-09,-3.867063336215892377e-09,-3.853486640892356263e-09,-3.839961404406011525e-09,-3.826479411106475936e-09,-3.813060422963587826e-09,-3.799698333750711754e-09,-3.786343849565554365e-09,-3.773054091382732622e-09,-3.759821953774888925e-09,-3.746630283796292815e-09,-3.733471420908074379e-09,-3.720374064375420176e-09,-3.707317952628130797e-09,-3.694296424328058492e-09,-3.681347005013435592e-09,-3.668424786162916007e-09,-3.655555691484124736e-09,-3.642736501330290366e-09,-3.629945621863583938e-09,-3.617206589812127504e-09,-3.604504805743147244e-09,-3.591844988104497816e-09,-3.579241902862406732e-09,-3.566695161438815376e-09,-3.554189276222530225e-09,-3.541693660569222857e-09,-3.529279035685561894e-09,-3.516896551758463829e-09,-3.504547485544406982e-09,-3.492257483195260193e-09,-3.480003230027506333e-09,-3.467781117816315373e-09,-3.455618624581546783e-09,-3.443482443632461809e-09,-3.431392225916596317e-09,-3.419372562873945753e-09,-3.407362281215853272e-09,-3.395382808246694140e-09,-3.383480662311200149e-09,-3.371629031523326603e-09,-3.359781453049492939e-09,-3.347993549063232877e-09,-3.336240228524189888e-09,-3.324549635586038221e-09,-3.312879026662329807e-09,-3.301212303519207580e-09,-3.289646333115570087e-09,-3.278137705731154483e-09,-3.266620585140600497e-09,-3.255130553991847364e-09,-3.243713908585021954e-09,-3.232352607795974109e-09,-3.220994027053336595e-09,-3.209685850436017063e-09,-3.198439291196564227e-09,-3.187190844577969528e-09,-3.176004570448753839e-09,-3.164912110165118975e-09,-3.153763028507228228e-09,-3.142690718771490310e-09,-3.131678860679443233e-09,-3.120662450672995192e-09,-3.109749902030500834e-09,-3.098828860181868095e-09,-3.087941347068579034e-09,-3.077097743275913899e-09,-3.066293940978681576e-09,-3.055566855092450851e-09,-3.044837160182112257e-09,-3.034111017985452463e-09,-3.023474082208821301e-09,-3.012867622054216099e-09,-3.002313619937524436e-09,-2.991810632568814299e-09,-2.981284108471982108e-09,-2.970805712543267418e-09,-2.960378664429441642e-09,
        -2.949987809586218646e-09,-2.939651855271563363e-09,-2.929312459265531743e-09,-2.919048835980930789e-09,-2.908820906366571535e-09,-2.898588480348962548e-09,-2.888427441671836959e-09,-2.878290106256287118e-09,-2.868173254455541610e-09,-2.858118186566116492e-09,-2.848107194530768993e-09,-2.838084656175965392e-09,-2.828127454446160982e-09,-2.818227817780183386e-09,-2.808324295333619602e-09,-2.798453135888223642e-09,-2.788636488393336776e-09,-2.778862473462595517e-09,-2.769109275213565979e-09,-2.759396766638388954e-09,-2.749713512439910801e-09,-2.740062898798356628e-09,-2.730444259579911659e-09,-2.720850822424125681e-09,-2.711314672776410362e-09,-2.701790235981604837e-09,-2.692303213702729181e-09,-2.682854938207412943e-09,-2.673472498937456976e-09,-2.664066245383622800e-09,-2.654693076475922453e-09,-2.645421881553033927e-09,-2.636124429855613016e-09,-2.626860395871233322e-09,-2.617653982461831674e-09,-2.608470106579829917e-09,-2.599307047379539881e-09,-2.590206493735536242e-09,-2.581101499199434102e-09,-2.572044299764542075e-09,-2.563024237289823759e-09,-2.553979139285900146e-09,-2.545069321957527109e-09,-2.536152621246401395e-09,-2.527200171353882752e-09,-2.518362685055564043e-09,-2.509537133654760055e-09,-2.500698315088811796e-09,-2.491941097915173486e-09,-2.483186989366004127e-09,-2.474449478651052914e-09,-2.465799175954686007e-09,-2.457150150014797418e-09,-2.448505842522763487e-09,-2.439927704322997215e-09,-2.431389090062907599e-09,-2.422822664716051122e-09,-2.414324129507150474e-09,-2.405863230858784618e-09,-2.397417986355065977e-09,-2.388990394397438877e-09,-2.380604824381293838e-09,-2.372255614169205273e-09,-2.363931939086683087e-09,-2.355665329467626634e-09,-2.347375294142750590e-09,-2.339107019189157199e-09,-2.330930393146246615e-09,-2.322757486350468525e-09,-2.314568536831984602e-09,-2.306453694700394408e-09,-2.298413126489151637e-09,-2.290318323883155927e-09,-2.282250666230112301e-09,-2.274270216595652983e-09,-2.266286103225212401e-09,-2.258334297344788411e-09,-2.250402308945353980e-09,-2.242513175154670080e-09,-2.234645524179512677e-09,-2.226793305304397563e-09,-2.218971950629367029e-09,-2.211206828750533759e-09,-2.203415561119470567e-09,-2.195692516693270591e-09,-2.188040415518344162e-09,-2.180324698564106711e-09,-2.172633795094469633e-09,-2.165022605638000641e-09,-2.157454603857189568e-09,-2.149879996249381975e-09,-2.142332755639131392e-09,-2.134784793383914803e-09,-2.127332254797664746e-09,-2.119883879547757033e-09,-2.112400421250271165e-09,-2.105005114660940535e-09,-2.097615081630976874e-09,-2.090232376072975740e-09,-2.082898464816906881e-09,-2.075601690521011733e-09,-2.068334059579512996e-09,-2.061061876723613295e-09,-2.053822500958091268e-09,-2.046628866381183798e-09,-2.039444502166531947e-09,-2.032268742180320942e-09,-2.025150158679878132e-09,-2.018058442576631251e-09,-2.010969002430584851e-09,-2.003904320257987592e-09,-1.996874332554909870e-09,-1.989857167927766568e-09,-1.982845887482653779e-09,-1.975957786282123152e-09,-1.969010288149775079e-09,-1.962091211726857409e-09,-1.955223372096526191e-09,-1.948313621547015373e-09,-1.941463767529683082e-09,-1.934677751336266738e-09,-1.927921766675666504e-09,-1.921129255677556102e-09,-1.914383540579933651e-09,-1.907688451652234107e-09,-1.900954504918672683e-09,-1.894270018620858309e-09,-1.887641210007728887e-09,-1.881040156970215094e-09,-1.874470745288903117e-09,-1.867849541703492378e-09,-1.861287290960689234e-09,-1.854770781406500646e-09,-1.848252384473170196e-09,-1.841741481545255965e-09,-1.835287866125412393e-09,-1.828847684404166785e-09,-1.822431483500253080e-09,-1.816077455085718384e-09,-1.809687955045546914e-09,-1.803280136325469130e-09,-1.796982396218282929e-09,-1.790732562234609304e-09,-1.784392522630184885e-09,-1.778139246955134922e-09,-1.771950031148605831e-09,-1.765690482713466736e-09,-1.759446199844916237e-09,-1.753314549102213959e-09,-1.747178179911657026e-09,-1.740997013222056466e-09,-1.734924592877717942e-09,-1.728863774363986749e-09,-1.722748277366292768e-09,-1.716728870171380095e-09,-1.710701524881841351e-09,-1.704692609294511385e-09,-1.698712059905460592e-09,-1.692755324800288008e-09,-1.686814854462426183e-09,-1.680844186058294554e-09,-1.674966443321324050e-09,-1.669154425787411355e-09,-1.663274407093240370e-09,-1.657390724663088122e-09,-1.651592362872378317e-09,-1.645813485495750683e-09,-1.640022173621247248e-09,-1.634228918856450719e-09,-1.628540802212086192e-09,-1.622864620465236385e-09,-1.617153078115052267e-09,-1.611432876025276073e-09,-1.605764243794993718e-09,-1.600161447790071634e-09,-1.594547105465693448e-09,-1.588941755947814727e-09,-1.583403297367169671e-09,-1.577825481380301653e-09,-1.572246943748467629e-09,-1.566767271476976475e-09,-1.561254292514746567e-09,-1.555735373859334914e-09,-1.550341910405705903e-09,-1.544920358309553876e-09,-1.539466776279141413e-09,-1.534072313624790240e-09,-1.528649595794462357e-09,-1.523272064041236717e-09,-1.517989400845465298e-09,-1.512661496061440403e-09,-1.507311275794620542e-09,-1.502032664912889004e-09,-1.496768653463931287e-09,-1.491532397590589198e-09,-1.486277323436979714e-09,-1.481051226104312946e-09,-1.475856215016335682e-09,-1.470661314950660881e-09,-1.465547627699237410e-09,-1.460419729593098737e-09,-1.455258746840826234e-09,-1.450134790026424980e-09,-1.445063735339147115e-09,-1.439992902696474175e-09,-1.434913188269604234e-09,-1.429893259352610357e-09,-1.424889206624868621e-09,-1.419939887892240904e-09,-1.414894867934890499e-09,-1.409871164259612897e-09,-1.405001559540153266e-09,-1.400080107405443641e-09,-1.395150828198410409e-09,-1.390242365673088898e-09,-1.385360659522660853e-09,-1.380526581939989228e-09,-1.375666636160843836e-09,-1.370852542592615464e-09,-1.366030510929761022e-09,-1.361177059955309687e-09,-1.356444012667878951e-09,-1.351692813233995594e-09,-1.346963374171394889e-09,-1.342253752589783744e-09,-1.337516264410254507e-09,-1.332809973497717237e-09,-1.328119392240978414e-09,-1.323448683976380380e-09,-1.318858744436823827e-09,-1.314209796543508446e-09,-1.309567509988340817e-09,-1.305013264119025962e-09,-1.300454410824158913e-09,-1.295898499620307120e-09,-1.291266149561209886e-09,-1.286700745950497549e-09,-1.282245698508432952e-09,-1.277801586763160913e-09,-1.273313954275323567e-09,-1.268786298247448485e-09,-1.264339799522673502e-09,-1.259909176987150659e-09,-1.255436143932087134e-09,-1.251121706236091313e-09,-1.246750591654688378e-09,-1.242311753468783309e-09,-1.237924263097767152e-09,-1.233662172417382408e-09,-1.229333301822066460e-09,-1.224955858969423161e-09,-1.220655465594688849e-09,-1.216383216373628784e-09,-1.212143940776400086e-09,-1.207855815366087882e-09,-1.203624477863485254e-09,-1.199431720610988350e-09,-1.195177012913717363e-09,-1.190980647436390427e-09,-1.186852227608170551e-09,-1.182664466359284461e-09,-1.178518338473821814e-09,-1.174335462206244074e-09,-1.170259555927088968e-09,-1.166214236292262285e-09,-1.162028084866761901e-09,-1.157923978922781316e-09,-1.153934503506093279e-09,-1.149875861194971094e-09,-1.145797623447464275e-09,-1.141824357286935765e-09,-1.137799743311518341e-09,-1.133740490377732613e-09,-1.129864701798766191e-09,-1.125871174068038272e-09,-1.121855996988330162e-09,-1.118021675239333490e-09,-1.114100478538659900e-09,-1.110130876114112652e-09,-1.106236102721425141e-09,-1.102355318138847906e-09,-1.098460489235009163e-09,-1.094630386333506067e-09,-1.090782852930516356e-09,-1.086952750029013259e-09,-1.083224676623473215e-09,-1.079391798164408556e-09,-1.075545430495594701e-09,-1.071734201385510232e-09,-1.068039434670708943e-09,-1.064313304155461992e-09,-1.060463050706061949e-09,-1.056857656944742985e-09,-1.053132858697125585e-09,-1.049418052456729811e-09,-1.045826591994369892e-09,-1.042037067744416845e-09,-1.038324315416616628e-09,-1.034795193977089411e-09,-1.031123020300839244e-09,-1.027484430871083987e-09,-1.023903184460550619e-09,-1.020327211609384221e-09,-1.016778605755774834e-09,-1.013195471966099603e-09,-1.009681338537404827e-09,-1.006040251105844163e-09,-1.002440297437345862e-09,-9.989998828174861956e-10,-9.955111734960553349e-10,-9.920765320359237194e-10,-9.885627316741363302e-10,-9.850624205220981366e-10,-9.815902535237341908e-10,-9.781023768695717990e-10,-9.747667117920855162e-10,-9.712842197195925564e-10,-9.678922108236065469e-10,-9.645279575032361663e-10,-9.611405560328023512e-10,-9.577641457703123251e-10,-9.543900114650227806e-10,-9.510425225123242399e-10,-9.476953111153818554e-10,-9.443912873940973896e-10,-9.410360268802264727e-10,-9.377922327580279216e-10,-9.344149343171181954e-10,-9.311327264782676139e-10,-9.279750856627799749e-10,-9.246581833544098572e-10,-9.214436436089101790e-10,-9.181915783251781704e-10,-9.149050406165315508e-10,-9.117762655996841659e-10,-9.086433272464944366e-10,-9.053972016559441727e-10,-9.021332014746974437e-10,-8.989906041811934756e-10,-8.959084030202291160e-10,-8.926848149570787427e-10,-8.895478242898491317e-10,-8.864967648847255077e-10,-8.833355158444078370e-10,-8.802722994971645676e-10,-8.772157999992202804e-10,-8.740816404007034635e-10,-8.709457044453472463e-10,-8.680424712359524619e-10,-8.649762017753914733e-10,-8.617825897339059793e-10,-8.588364464046094326e-10,-8.558554420723396561e-10,-8.528120987172371770e-10,-8.497672010499002226e-10,-8.468930001725993861e-10,-8.439847709595937886e-10,-8.409560825484163615e-10,-8.379943405856238314e-10,-8.350181102123599430e-10,-8.321224265195326097e-10,-8.292421194155963349e-10,-8.262520667656758633e-10,-8.232992065870803344e-10,-8.204509849285557266e-10,-8.176915256008499000e-10,-8.148332564239524345e-10,-8.119381833537886450e-10,-8.089307557135327897e-10,-8.061492584587881538e-10,-8.034390930333756842e-10,-8.004397145100483613e-10,-7.976748150895218714e-10,-7.949779723404049037e-10,-7.920909483871696466e-10,-7.893616316145823930e-10,-7.866245432808227633e-10,-7.839636717577036507e-10,-7.811443158978192969e-10,-7.782498534503190513e-10,-7.755957542876501520e-10,-7.728572226639585097e-10,-7.701096982337674035e-10,-7.675299285025971585e-10,
        0.000000000000000000e+00,
    ])

    X_SCALE = np.array(1024, dtype=np.int64)

    def __init__(self):
        super().__init__()
        self.register_buffer('spline_x_scale',
            torch.tensor(Distribution.X_SCALE.copy()).float()
        )
        self.register_buffer('spline_values',
            torch.tensor(Distribution.VALUES.copy()).float()
        )
        self.register_buffer('spline_tangents',
            torch.tensor(Distribution.TANGENTS.copy()).float()
        )

    def log_base_partition_function(self, alpha: torch.Tensor) -> torch.Tensor:
        r"""Approximate the distribution's log-partition function with a 1D spline.

        Because the partition function (Z(\alpha) in the paper) of the distribution
        is difficult to model analytically, we approximate it with a (transformed)
        cubic hermite spline: Each alpha is pushed through a nonlinearity before
        being used to interpolate into a spline, which allows us to use a relatively
        small spline to accurately model the log partition function over the range
        of all non-negative input values.

        Args:
        alpha: A tensor or scalar of single or double precision floats containing
            the set of alphas for which we would like an approximate log partition
            function. Must be non-negative, as the partition function is undefined
            when alpha < 0.

        Returns:
        An approximation of log(Z(alpha)) accurate to within 1e-6
        """
        x = partition_spline_curve(alpha) # Transform `alpha` to the form expected by the spline.        
        return cubic_spline.interpolate1d( # Interpolate into the spline.
            x * self.spline_x_scale,
            self.spline_values,
            self.spline_tangents
        )

    def forward(self, 
        residual:       torch.Tensor,
        alpha:          torch.Tensor,
        scale:          torch.Tensor
    ) -> torch.Tensor:
        r"""Implements the negative log-likelihood (NLL).

        Specifically, we implement -log(p(x | 0, \alpha, c) of Equation 16 in the
        paper as nllfun(x, alpha, shape).

        Args:
        x: The residual for which the NLL is being computed. x can have any shape,
            and alpha and scale will be broadcasted to match x's shape if necessary.
            Must be a tensor or numpy array of floats.
        alpha: The shape parameter of the NLL (\alpha in the paper), where more
            negative values cause outliers to "cost" more and inliers to "cost"
            less. Alpha can be any non-negative value, but the gradient of the NLL
            with respect to alpha has singularities at 0 and 2 so you may want to
            limit usage to (0, 2) during gradient descent. Must be a tensor or numpy
            array of floats. Varying alpha in that range allows for smooth
            interpolation between a Cauchy distribution (alpha = 0) and a Normal
            distribution (alpha = 2) similar to a Student's T distribution.
        scale: The scale parameter of the loss. When |x| < scale, the NLL is like
            that of a (possibly unnormalized) normal distribution, and when |x| >
            scale the NLL takes on a different shape according to alpha. Must be a
            tensor or numpy array of floats.

        Returns:
        The NLLs for each element of x, in the same shape and precision as x.
        """
        loss = generic.loss(residual, alpha, scale, approximate=False)
        log_partition = torch.log(scale) + self.log_base_partition_function(alpha)
        nll = loss + log_partition
        return nll