"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = exports.MethodLoggingLevel = void 0;
const core_1 = require("@aws-cdk/core");
const access_log_1 = require("./access-log");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
var MethodLoggingLevel;
(function (MethodLoggingLevel) {
    MethodLoggingLevel["OFF"] = "OFF";
    MethodLoggingLevel["ERROR"] = "ERROR";
    MethodLoggingLevel["INFO"] = "INFO";
})(MethodLoggingLevel = exports.MethodLoggingLevel || (exports.MethodLoggingLevel = {}));
class Stage extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.enableCacheCluster = props.cacheClusterEnabled;
        const methodSettings = this.renderMethodSettings(props); // this can mutate `this.cacheClusterEnabled`
        // custom access logging
        let accessLogSetting;
        const accessLogDestination = props.accessLogDestination;
        const accessLogFormat = props.accessLogFormat;
        if (!accessLogDestination && !accessLogFormat) {
            accessLogSetting = undefined;
        }
        else {
            if (accessLogFormat !== undefined && !/.*\$context.requestId.*/.test(accessLogFormat.toString())) {
                throw new Error('Access log must include at least `AccessLogFormat.contextRequestId()`');
            }
            if (accessLogFormat !== undefined && accessLogDestination === undefined) {
                throw new Error('Access log format is specified without a destination');
            }
            accessLogSetting = {
                destinationArn: accessLogDestination === null || accessLogDestination === void 0 ? void 0 : accessLogDestination.bind(this).destinationArn,
                format: (accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString()) ? accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString() : access_log_1.AccessLogFormat.clf().toString(),
            };
        }
        // enable cache cluster if cacheClusterSize is set
        if (props.cacheClusterSize !== undefined) {
            if (this.enableCacheCluster === undefined) {
                this.enableCacheCluster = true;
            }
            else if (this.enableCacheCluster === false) {
                throw new Error(`Cannot set "cacheClusterSize" to ${props.cacheClusterSize} and "cacheClusterEnabled" to "false"`);
            }
        }
        const cacheClusterSize = this.enableCacheCluster ? (props.cacheClusterSize || '0.5') : undefined;
        const resource = new apigateway_generated_1.CfnStage(this, 'Resource', {
            stageName: props.stageName || 'prod',
            accessLogSetting,
            cacheClusterEnabled: this.enableCacheCluster,
            cacheClusterSize,
            clientCertificateId: props.clientCertificateId,
            deploymentId: props.deployment.deploymentId,
            restApiId: props.deployment.api.restApiId,
            description: props.description,
            documentationVersion: props.documentationVersion,
            variables: props.variables,
            tracingEnabled: props.tracingEnabled,
            methodSettings,
        });
        this.stageName = resource.ref;
        this.restApi = props.deployment.api;
    }
    /**
     * Returns the invoke URL for a certain path.
     * @param path The resource path
     */
    urlForPath(path = '/') {
        if (!path.startsWith('/')) {
            throw new Error(`Path must begin with "/": ${path}`);
        }
        return `https://${this.restApi.restApiId}.execute-api.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}/${this.stageName}${path}`;
    }
    renderMethodSettings(props) {
        const settings = new Array();
        const self = this;
        // extract common method options from the stage props
        const commonMethodOptions = {
            metricsEnabled: props.metricsEnabled,
            loggingLevel: props.loggingLevel,
            dataTraceEnabled: props.dataTraceEnabled,
            throttlingBurstLimit: props.throttlingBurstLimit,
            throttlingRateLimit: props.throttlingRateLimit,
            cachingEnabled: props.cachingEnabled,
            cacheTtl: props.cacheTtl,
            cacheDataEncrypted: props.cacheDataEncrypted,
        };
        // if any of them are defined, add an entry for '/*/*'.
        const hasCommonOptions = Object.keys(commonMethodOptions).map(v => commonMethodOptions[v]).filter(x => x).length > 0;
        if (hasCommonOptions) {
            settings.push(renderEntry('/*/*', commonMethodOptions));
        }
        if (props.methodOptions) {
            for (const path of Object.keys(props.methodOptions)) {
                settings.push(renderEntry(path, props.methodOptions[path]));
            }
        }
        return settings.length === 0 ? undefined : settings;
        function renderEntry(path, options) {
            if (options.cachingEnabled) {
                if (self.enableCacheCluster === undefined) {
                    self.enableCacheCluster = true;
                }
                else if (self.enableCacheCluster === false) {
                    throw new Error(`Cannot enable caching for method ${path} since cache cluster is disabled on stage`);
                }
            }
            const { httpMethod, resourcePath } = util_1.parseMethodOptionsPath(path);
            return {
                httpMethod, resourcePath,
                cacheDataEncrypted: options.cacheDataEncrypted,
                cacheTtlInSeconds: options.cacheTtl && options.cacheTtl.toSeconds(),
                cachingEnabled: options.cachingEnabled,
                dataTraceEnabled: options.dataTraceEnabled,
                loggingLevel: options.loggingLevel,
                metricsEnabled: options.metricsEnabled,
                throttlingBurstLimit: options.throttlingBurstLimit,
                throttlingRateLimit: options.throttlingRateLimit,
            };
        }
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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