"use strict";
const assert_1 = require("@aws-cdk/assert");
const acm = require("@aws-cdk/aws-certificatemanager");
const core_1 = require("@aws-cdk/core");
const apigw = require("../lib");
module.exports = {
    'can define either an EDGE or REGIONAL domain name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cert = new acm.Certificate(stack, 'Cert', { domainName: 'example.com' });
        // WHEN
        const regionalDomain = new apigw.DomainName(stack, 'my-domain', {
            domainName: 'example.com',
            certificate: cert,
            endpointType: apigw.EndpointType.REGIONAL,
        });
        const edgeDomain = new apigw.DomainName(stack, 'your-domain', {
            domainName: 'example.com',
            certificate: cert,
            endpointType: apigw.EndpointType.EDGE,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'example.com',
            'EndpointConfiguration': { 'Types': ['REGIONAL'] },
            'RegionalCertificateArn': { 'Ref': 'Cert5C9FAEC1' },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'example.com',
            'EndpointConfiguration': { 'Types': ['EDGE'] },
            'CertificateArn': { 'Ref': 'Cert5C9FAEC1' },
        }));
        test.deepEqual(stack.resolve(regionalDomain.domainNameAliasDomainName), { 'Fn::GetAtt': ['mydomain592C948B', 'RegionalDomainName'] });
        test.deepEqual(stack.resolve(regionalDomain.domainNameAliasHostedZoneId), { 'Fn::GetAtt': ['mydomain592C948B', 'RegionalHostedZoneId'] });
        test.deepEqual(stack.resolve(edgeDomain.domainNameAliasDomainName), { 'Fn::GetAtt': ['yourdomain5FE30C81', 'DistributionDomainName'] });
        test.deepEqual(stack.resolve(edgeDomain.domainNameAliasHostedZoneId), { 'Fn::GetAtt': ['yourdomain5FE30C81', 'DistributionHostedZoneId'] });
        test.done();
    },
    'default endpoint type is REGIONAL'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cert = new acm.Certificate(stack, 'Cert', { domainName: 'example.com' });
        // WHEN
        new apigw.DomainName(stack, 'my-domain', {
            domainName: 'example.com',
            certificate: cert,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'example.com',
            'EndpointConfiguration': { 'Types': ['REGIONAL'] },
            'RegionalCertificateArn': { 'Ref': 'Cert5C9FAEC1' },
        }));
        test.done();
    },
    'accepts different security policies'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cert = new acm.Certificate(stack, 'Cert', { domainName: 'example.com' });
        // WHEN
        new apigw.DomainName(stack, 'my-domain', {
            domainName: 'old.example.com',
            certificate: cert,
            securityPolicy: apigw.SecurityPolicy.TLS_1_0,
        });
        new apigw.DomainName(stack, 'your-domain', {
            domainName: 'new.example.com',
            certificate: cert,
            securityPolicy: apigw.SecurityPolicy.TLS_1_2,
        });
        new apigw.DomainName(stack, 'default-domain', {
            domainName: 'default.example.com',
            certificate: cert,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'old.example.com',
            'EndpointConfiguration': { 'Types': ['REGIONAL'] },
            'RegionalCertificateArn': { 'Ref': 'Cert5C9FAEC1' },
            'SecurityPolicy': 'TLS_1_0',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'new.example.com',
            'EndpointConfiguration': { 'Types': ['REGIONAL'] },
            'RegionalCertificateArn': { 'Ref': 'Cert5C9FAEC1' },
            'SecurityPolicy': 'TLS_1_2',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'default.example.com',
            'EndpointConfiguration': { 'Types': ['REGIONAL'] },
            'RegionalCertificateArn': { 'Ref': 'Cert5C9FAEC1' },
            'SecurityPolicy': assert_1.ABSENT,
        }));
        test.done();
    },
    '"mapping" can be used to automatically map this domain to the deployment stage of an API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // WHEN
        new apigw.DomainName(stack, 'Domain', {
            domainName: 'foo.com',
            certificate: acm.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d'),
            endpointType: apigw.EndpointType.EDGE,
            mapping: api,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'Domain66AC69E0',
            },
            'RestApiId': {
                'Ref': 'apiC8550315',
            },
            'Stage': {
                'Ref': 'apiDeploymentStageprod896C8101',
            },
        }));
        test.done();
    },
    '"addBasePathMapping" can be used to add base path mapping to the domain'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api1 = new apigw.RestApi(stack, 'api1');
        const api2 = new apigw.RestApi(stack, 'api2');
        const domain = new apigw.DomainName(stack, 'my-domain', {
            domainName: 'example.com',
            certificate: acm.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d'),
            endpointType: apigw.EndpointType.REGIONAL,
        });
        api1.root.addMethod('GET');
        api2.root.addMethod('GET');
        // WHEN
        domain.addBasePathMapping(api1, { basePath: 'api1' });
        domain.addBasePathMapping(api2, { basePath: 'api2' });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'mydomain592C948B',
            },
            'BasePath': 'api1',
            'RestApiId': {
                'Ref': 'api1A91238E2',
            },
            'Stage': {
                'Ref': 'api1DeploymentStageprod362746F6',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'mydomain592C948B',
            },
            'BasePath': 'api2',
            'RestApiId': {
                'Ref': 'api2C4850CEA',
            },
            'Stage': {
                'Ref': 'api2DeploymentStageprod4120D74E',
            },
        }));
        test.done();
    },
    'a domain name can be defined with the API'(test) {
        // GIVEN
        const domainName = 'my.domain.com';
        const stack = new core_1.Stack();
        const certificate = new acm.Certificate(stack, 'cert', { domainName: 'my.domain.com' });
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            domainName: { domainName, certificate },
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'my.domain.com',
            'EndpointConfiguration': {
                'Types': [
                    'REGIONAL',
                ],
            },
            'RegionalCertificateArn': {
                'Ref': 'cert56CA94EB',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'apiCustomDomain64773C4F',
            },
            'RestApiId': {
                'Ref': 'apiC8550315',
            },
            'Stage': {
                'Ref': 'apiDeploymentStageprod896C8101',
            },
        }));
        test.done();
    },
    'a domain name can be added later'(test) {
        // GIVEN
        const domainName = 'my.domain.com';
        const stack = new core_1.Stack();
        const certificate = new acm.Certificate(stack, 'cert', { domainName: 'my.domain.com' });
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {});
        api.root.addMethod('GET');
        api.addDomainName('domainId', { domainName, certificate });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': domainName,
            'EndpointConfiguration': {
                'Types': [
                    'REGIONAL',
                ],
            },
            'RegionalCertificateArn': {
                'Ref': 'cert56CA94EB',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'apidomainId102F8DAA',
            },
            'RestApiId': {
                'Ref': 'apiC8550315',
            },
            'Stage': {
                'Ref': 'apiDeploymentStageprod896C8101',
            },
        }));
        test.done();
    },
    'domain name cannot contain uppercase letters'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const certificate = new acm.Certificate(stack, 'cert', { domainName: 'someDomainWithUpercase.domain.com' });
        // WHEN
        test.throws(() => {
            new apigw.DomainName(stack, 'someDomain', { domainName: 'someDomainWithUpercase.domain.com', certificate });
        }, /uppercase/);
        // THEN
        test.done();
    },
    'multiple domain names can be added'(test) {
        // GIVEN
        const domainName = 'my.domain.com';
        const stack = new core_1.Stack();
        const certificate = new acm.Certificate(stack, 'cert', { domainName: 'my.domain.com' });
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {});
        api.root.addMethod('GET');
        const domainName1 = api.addDomainName('domainId', { domainName, certificate });
        api.addDomainName('domainId1', { domainName: 'your.domain.com', certificate });
        api.addDomainName('domainId2', { domainName: 'our.domain.com', certificate });
        test.deepEqual(api.domainName, domainName1);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'my.domain.com',
            'EndpointConfiguration': {
                'Types': [
                    'REGIONAL',
                ],
            },
            'RegionalCertificateArn': {
                'Ref': 'cert56CA94EB',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'your.domain.com',
            'EndpointConfiguration': {
                'Types': [
                    'REGIONAL',
                ],
            },
            'RegionalCertificateArn': {
                'Ref': 'cert56CA94EB',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            'DomainName': 'our.domain.com',
            'EndpointConfiguration': {
                'Types': [
                    'REGIONAL',
                ],
            },
            'RegionalCertificateArn': {
                'Ref': 'cert56CA94EB',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'apidomainId102F8DAA',
            },
            'RestApiId': {
                'Ref': 'apiC8550315',
            },
            'Stage': {
                'Ref': 'apiDeploymentStageprod896C8101',
            },
        }));
        test.done();
    },
    '"addBasePathMapping" can be used to add base path mapping to the domain with specific stage'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api1 = new apigw.RestApi(stack, 'api1');
        const api2 = new apigw.RestApi(stack, 'api2');
        const domain = new apigw.DomainName(stack, 'my-domain', {
            domainName: 'example.com',
            certificate: acm.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d'),
            endpointType: apigw.EndpointType.REGIONAL,
        });
        api1.root.addMethod('GET');
        api2.root.addMethod('GET');
        const testDeploy = new apigw.Deployment(stack, 'test-deployment', {
            api: api1,
        });
        const testStage = new apigw.Stage(stack, 'test-stage', {
            deployment: testDeploy,
        });
        // WHEN
        domain.addBasePathMapping(api1, { basePath: 'api1', stage: testStage });
        domain.addBasePathMapping(api2, { basePath: 'api2' });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'mydomain592C948B',
            },
            'BasePath': 'api1',
            'RestApiId': {
                'Ref': 'api1A91238E2',
            },
            'Stage': stack.resolve(testStage.stageName),
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            'DomainName': {
                'Ref': 'mydomain592C948B',
            },
            'BasePath': 'api2',
            'RestApiId': {
                'Ref': 'api2C4850CEA',
            },
            'Stage': {
                'Ref': 'api2DeploymentStageprod4120D74E',
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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