import * as iam from '@aws-cdk/aws-iam';
import * as cdk from '@aws-cdk/core';
import { Construct } from 'constructs';
import { ScalingSchedule, ServiceNamespace } from './scalable-target';
import { BasicStepScalingPolicyProps } from './step-scaling-policy';
import { BasicTargetTrackingScalingPolicyProps } from './target-tracking-scaling-policy';
/**
 * Properties for a ScalableTableAttribute.
 *
 * @stability stable
 */
export interface BaseScalableAttributeProps extends EnableScalingProps {
    /**
     * Service namespace of the scalable attribute.
     *
     * @stability stable
     */
    readonly serviceNamespace: ServiceNamespace;
    /**
     * Resource ID of the attribute.
     *
     * @stability stable
     */
    readonly resourceId: string;
    /**
     * Scalable dimension of the attribute.
     *
     * @stability stable
     */
    readonly dimension: string;
    /**
     * Role to use for scaling.
     *
     * @stability stable
     */
    readonly role: iam.IRole;
}
/**
 * Represent an attribute for which autoscaling can be configured.
 *
 * This class is basically a light wrapper around ScalableTarget, but with
 * all methods protected instead of public so they can be selectively
 * exposed and/or more specific versions of them can be exposed by derived
 * classes for individual services support autoscaling.
 *
 * Typical use cases:
 *
 * - Hide away the PredefinedMetric enum for target tracking policies.
 * - Don't expose all scaling methods (for example Dynamo tables don't support
 *    Step Scaling, so the Dynamo subclass won't expose this method).
 *
 * @stability stable
 */
export declare abstract class BaseScalableAttribute extends cdk.Construct {
    protected readonly props: BaseScalableAttributeProps;
    private target;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: BaseScalableAttributeProps);
    /**
     * Scale out or in based on time.
     *
     * @stability stable
     */
    protected doScaleOnSchedule(id: string, props: ScalingSchedule): void;
    /**
     * Scale out or in based on a metric value.
     *
     * @stability stable
     */
    protected doScaleOnMetric(id: string, props: BasicStepScalingPolicyProps): void;
    /**
     * Scale out or in in order to keep a metric around a target value.
     *
     * @stability stable
     */
    protected doScaleToTrackMetric(id: string, props: BasicTargetTrackingScalingPolicyProps): void;
}
/**
 * Properties for enabling DynamoDB capacity scaling.
 *
 * @stability stable
 */
export interface EnableScalingProps {
    /**
     * Minimum capacity to scale to.
     *
     * @default 1
     * @stability stable
     */
    readonly minCapacity?: number;
    /**
     * Maximum capacity to scale to.
     *
     * @stability stable
     */
    readonly maxCapacity: number;
}
