"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepScalingPolicy = void 0;
const aws_autoscaling_common_1 = require("@aws-cdk/aws-autoscaling-common");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk = require("@aws-cdk/core");
const step_scaling_action_1 = require("./step-scaling-action");
/**
 * Define a scaling strategy which scales depending on absolute values of some metric.
 *
 * You can specify the scaling behavior for various values of the metric.
 *
 * Implemented using one or more CloudWatch alarms and Step Scaling Policies.
 *
 * @stability stable
 */
class StepScalingPolicy extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (props.scalingSteps.length < 2) {
            throw new Error('You must supply at least 2 intervals for autoscaling');
        }
        const adjustmentType = props.adjustmentType || step_scaling_action_1.AdjustmentType.CHANGE_IN_CAPACITY;
        const changesAreAbsolute = adjustmentType === step_scaling_action_1.AdjustmentType.EXACT_CAPACITY;
        const intervals = aws_autoscaling_common_1.normalizeIntervals(props.scalingSteps, changesAreAbsolute);
        const alarms = aws_autoscaling_common_1.findAlarmThresholds(intervals);
        if (alarms.lowerAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.lowerAlarmIntervalIndex].upper;
            this.lowerAction = new step_scaling_action_1.StepScalingAction(this, 'LowerPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.lowerAlarmIntervalIndex; i >= 0; i--) {
                this.lowerAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: i !== 0 ? intervals[i].lower - threshold : undefined,
                    upperBound: intervals[i].upper - threshold,
                });
            }
            this.lowerAlarm = new cloudwatch.Alarm(this, 'LowerAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Lower threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
                threshold,
            });
            this.lowerAlarm.addAlarmAction(new StepScalingAlarmAction(this.lowerAction));
        }
        if (alarms.upperAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.upperAlarmIntervalIndex].lower;
            this.upperAction = new step_scaling_action_1.StepScalingAction(this, 'UpperPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.upperAlarmIntervalIndex; i < intervals.length; i++) {
                this.upperAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: intervals[i].lower - threshold,
                    upperBound: i !== intervals.length - 1 ? intervals[i].upper - threshold : undefined,
                });
            }
            this.upperAlarm = new cloudwatch.Alarm(this, 'UpperAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Upper threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
                threshold,
            });
            this.upperAlarm.addAlarmAction(new StepScalingAlarmAction(this.upperAction));
        }
    }
}
exports.StepScalingPolicy = StepScalingPolicy;
function aggregationTypeFromMetric(metric) {
    var _a;
    const statistic = (_a = metric.toMetricConfig().metricStat) === null || _a === void 0 ? void 0 : _a.statistic;
    if (statistic == null) {
        return undefined;
    } // Math expression, don't know aggregation, leave default
    switch (statistic) {
        case 'Average':
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
        case 'Minimum':
            return step_scaling_action_1.MetricAggregationType.MINIMUM;
        case 'Maximum':
            return step_scaling_action_1.MetricAggregationType.MAXIMUM;
        default:
            throw new Error(`Cannot only scale on 'Minimum', 'Maximum', 'Average' metrics, got ${statistic}`);
    }
}
/**
 * Use a StepScalingAction as an Alarm Action
 *
 * This class is here and not in aws-cloudwatch-actions because this library
 * needs to use the class, and otherwise we'd have a circular dependency:
 *
 * aws-autoscaling -> aws-cloudwatch-actions (for using the Action)
 * aws-cloudwatch-actions -> aws-autoscaling (for the definition of IStepScalingAction)
 */
class StepScalingAlarmAction {
    constructor(stepScalingAction) {
        this.stepScalingAction = stepScalingAction;
    }
    bind(_scope, _alarm) {
        return { alarmActionArn: this.stepScalingAction.scalingPolicyArn };
    }
}
//# sourceMappingURL=data:application/json;base64,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