# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/12_training.cnn.ipynb.

# %% auto 0
__all__ = ['Model', 'weights_init', 'Learner']

# %% ../nbs/12_training.cnn.ipynb 4
#nbdev_comment from __future__ import annotations

# Python utils
from collections import OrderedDict
from tqdm.auto import tqdm
from functools import partial, reduce
import operator
from pathlib import Path
import math

import numpy as np
import pandas as pd

from sklearn.model_selection import train_test_split

from ..data.loading import load_kssl
from ..data.selection import (select_y, select_tax_order, select_X)
from ..data.transform import log_transform_y
from ..data.torch import DataLoaders, SNV_transform
from .metrics import eval_reg
from .core import is_plateau

# Deep Learning stack
import torch
from torch import nn
from torchvision import transforms
from torch.utils.data import DataLoader, Dataset
from torch.optim import Adam
from torch.optim.lr_scheduler import CyclicLR
from torch.nn import MSELoss

from fastcore.test import *
from fastcore.basics import store_attr
from fastcore.transform import compose

import warnings
warnings.filterwarnings('ignore')

# %% ../nbs/12_training.cnn.ipynb 5
# https://datascience.stackexchange.com/questions/40906/determining-size-of-fc-layer-after-conv-layer-in-pytorch
class Model(nn.Module):
    def __init__(self, input_dim, in_channel=1, out_channel=16, is_classifier=False,
                 dropout=0.4):
        super(Model, self).__init__()
        # Build the neural network
        self.feature_extractor = nn.Sequential(
            self.make_convpool_block(in_channel, out_channel),
            self.make_convpool_block(out_channel, out_channel*2),
            self.make_convpool_block(out_channel*2, out_channel*4),
            self.make_convpool_block(out_channel*4, out_channel*8),
            self.make_convpool_block(out_channel*8, out_channel*16))

        num_features_before_fcnn = reduce(operator.mul,
                                          self.feature_extractor(torch.rand(1, in_channel, input_dim)).shape)

        output_layers = [nn.Dropout(dropout),
                         nn.Linear(in_features=num_features_before_fcnn, out_features=20),
                         nn.BatchNorm1d(20),
                         nn.LeakyReLU(0.1),
                         nn.Linear(in_features=20, out_features=1)];

        if is_classifier:
            output_layers.append(nn.Sigmoid())

        self.output_block = nn.Sequential(*output_layers)

    def make_convpool_block(self, input_channels, output_channels, kernel_size=3, stride=1):
        return nn.Sequential(
            nn.Conv1d(input_channels, output_channels, kernel_size, bias=False),
            nn.BatchNorm1d(output_channels),
            nn.LeakyReLU(0.1),
            nn.AvgPool1d(3))

    def forward(self, x):
        batch_size = x.size(0)
        out = self.feature_extractor(x)
        out = out.view(batch_size, -1)  # flatten the vector
        out = self.output_block(out)
        return out

# %% ../nbs/12_training.cnn.ipynb 6
def weights_init(m):
    if isinstance(m, nn.Conv1d) or isinstance(m, nn.Linear):
        torch.nn.init.kaiming_uniform_(m.weight)
    if isinstance(m, nn.BatchNorm1d):
        torch.nn.init.normal_(m.weight, 0.0, 0.02)
        torch.nn.init.constant_(m.bias, 0)

# %% ../nbs/12_training.cnn.ipynb 7
class Learner():
    def __init__(self, 
                 model, 
                 criterion=nn.MSELoss(), 
                 opt=torch.optim.Adam, 
                 n_epochs=50,
                 scheduler=None, 
                 early_stopper= None, 
                 tax_lookup=range(10), 
                 verbose=True
                ):
        store_attr() # see https://fastpages.fast.ai/fastcore
        use_cuda = torch.cuda.is_available()
        self.device = torch.device('cuda:0' if use_cuda else 'cpu')
        self._init_losses()

    def fit(self, 
            training_generator, 
            validation_generator,
            losses_append=None
           ):
        if losses_append is None: self._init_losses()
            
        for epoch in range(self.n_epochs):
            print(30*'-')
            print(f'Epoch: {epoch}')
            loss_train = []
            loss_valid = []
            
            loss_valid_tax = [[] for i in range(len(self.tax_lookup))]
            
            # Training loop
            self.model.train()
            for batches in training_generator:
                local_batch, local_labels, _ = self._to_device(batches)
                self.opt.zero_grad() 
                y_hat, loss = self.eval(local_batch, local_labels)
                loss.backward(retain_graph=True) 
                self.opt.step()
                loss_train.append(loss.item())


            # Validation loop
            self.model.eval()
            with torch.no_grad():
                for batches in validation_generator:
                    local_batch, local_labels, local_tax = self._to_device(batches)
                    y_hat, loss = self.eval(local_batch, local_labels)
                    loss_valid.append(loss.item())
                    
                    # Eval by tax order
                    for tax in self.tax_lookup:
                        mask = (local_tax == tax).squeeze()
                        if torch.sum(mask).item() == 0:
                            loss_valid_tax[tax].append(float('nan'))
                        else:
                            _, loss = self.eval(local_batch[mask, :, :], local_labels[mask, :])
                            loss_valid_tax[tax].append(loss.item())
                            
            # mean losses after an epoch
            mean_loss_train = np.mean(np.array(loss_train))
            mean_loss_valid = np.mean(np.array(loss_valid))
            mean_loss_valid_tax = np.nanmean(np.array(loss_valid_tax), axis=1)
            
            if self.scheduler is not None:
                self.scheduler.step()

            if self.verbose:
                print(f'Training loss: {mean_loss_train} | Validation loss: {mean_loss_valid}')

            self.losses['train'].append(mean_loss_train)
            self.losses['valid'].append(mean_loss_valid)
            self.losses['valid_tax'].append(mean_loss_valid_tax)

            # At the end of a Cyclic learning rate cycle check if levelling
            losses_end_cycle = np.array(
                [el 
                 for i, el in enumerate(self.losses['valid']) 
                 if not i % self.scheduler.total_size])
            print(f'Validation loss (ends of cycles): {losses_end_cycle}')
            if self.early_stopper(losses_end_cycle):
                print('Early stopping!')
                
                break
            
        return self.model, self.losses

    def lr_finder(self,
                  training_generator,
                  start=1e-6,
                  end=1e-1,
                  n_epochs=8
                 ):
        opt = torch.optim.Adam(self.model.parameters(), lr=start)
        lr_lambda = lambda x: math.exp(x * math.log(end / start) / (n_epochs * len(training_generator)))
        scheduler = torch.optim.lr_scheduler.LambdaLR(opt, lr_lambda)
        criterion = nn.MSELoss()
        
        lrs = []
        losses = []
        i = 0
        smoothing = 0.01
        for epoch in range(n_epochs):
            print(f'Epoch: {epoch}')
            self.model.train()
            for batches in training_generator:
                local_batch, local_labels, local_tax = self._to_device(batches)
                opt.zero_grad()
                y_hat = self.model(local_batch)
                loss = criterion(y_hat, local_labels)
                loss.backward()
                opt.step()

                # update lr
                scheduler.step()
                lr_step = opt.state_dict()["param_groups"][0]["lr"]
                lrs.append(lr_step)

                # smooth loss
                if i == 0:
                    losses.append(loss.item())
                else:
                    loss = smoothing  * loss + (1 - smoothing) * losses[-1]
                    losses.append(loss.item())
                i += 1
        return lrs, losses
        
    def eval(self, 
             X, 
             y
            ):
        y_hat = self.model(X)
        loss = self.criterion(y_hat, y)
        return y_hat, loss

    def predict(self, 
                data_generator,
                sample=None
               ):
        self.model.eval() if sample is None else self._MCDropoutOn()
        ys_hat, ys = [], []
        if sample is not None: raise Exception(f'MCDropout needs refactoring')
        with torch.no_grad():
            for batches in data_generator:
                local_batch, local_labels, _ = self._to_device(batches)
                if sample is None:
                    ys_hat.append(self.model(local_batch))
                else:
                    # TO BE FIXED: MCDROPOUT
                    y_hat = np.stack([self.model(local_batch).cpu() for s in tqdm(range(sample))]).squeeze()
                ys.append(local_labels)
        return [self._npify(torch.cat(y)) for y in (ys_hat, ys)]

    def _to_device(self, 
                   batches
                  ):
        if self.device.type == 'cpu': return batches
        return (batch.to(self.device) for batch in batches)

    def _init_losses(self):
        self.losses = {'train': [], 'valid': [], 'valid_tax': []}

    def _MCDropoutOn(self):
         # Check https://discuss.pytorch.org/t/turn-off-batch-norm-but-leave-dropout-on/14815
        # to deactivate BatchNorm during inference (maybe not critical when predicting all set)
        # and
        # for m in model.modules():
        #   if isinstance(m, nn.BatchNorm1d):
        #     m.eval()
        self.model.train()
        for m in self.model.modules():
            if isinstance(m, nn.BatchNorm1d):
                m.eval()
    def _npify(self, 
               tensor
              ):
        if self.device.type == 'cpu': return tensor.numpy()
        return tensor.cpu().numpy()
