import { aws_ec2 as ec2, aws_iam as iam } from 'aws-cdk-lib';
import { PolicyStatement, AddToPrincipalPolicyResult } from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
/**
 * @stability stable
 */
export interface ECRDeploymentProps {
    /**
     * Image to use to build Golang lambda for custom resource, if download fails or is not wanted.
     *
     * Might be needed for local build if all images need to come from own registry.
     *
     * Note that image should use yum as a package manager and have golang available.
     *
     * @default public.ecr.aws/sam/build-go1.x:latest
     * @stability stable
     */
    readonly buildImage?: string;
    /**
     * The source of the docker image.
     *
     * @stability stable
     */
    readonly src: IImageName;
    /**
     * The destination of the docker image.
     *
     * @stability stable
     */
    readonly dest: IImageName;
    /**
     * The amount of memory (in MiB) to allocate to the AWS Lambda function which replicates the files from the CDK bucket to the destination bucket.
     *
     * If you are deploying large files, you will need to increase this number
     * accordingly.
     *
     * @default 512
     * @stability stable
     */
    readonly memoryLimit?: number;
    /**
     * Execution role associated with this function.
     *
     * @default - A role is automatically created
     * @stability stable
     */
    readonly role?: iam.IRole;
    /**
     * The VPC network to place the deployment lambda handler in.
     *
     * @default None
     * @stability stable
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Where in the VPC to place the deployment lambda handler.
     *
     * Only used if 'vpc' is supplied.
     *
     * @default - the Vpc default strategy if not specified
     * @stability stable
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * The environment variable to set.
     *
     * @stability stable
     */
    readonly environment?: {
        [key: string]: string;
    };
}
/**
 * @stability stable
 */
export interface IImageName {
    /**
     * The uri of the docker image.
     *
     * The uri spec follows https://github.com/containers/skopeo
     *
     * @stability stable
     */
    readonly uri: string;
    /**
     * The credentials of the docker image.
     *
     * Format `user:password` or `AWS Secrets Manager secret arn` or `AWS Secrets Manager secret name`
     *
     * @stability stable
     */
    creds?: string;
}
/**
 * @stability stable
 */
export declare class DockerImageName implements IImageName {
    private name;
    creds?: string | undefined;
    /**
     * @stability stable
     */
    constructor(name: string, creds?: string | undefined);
    /**
     * The uri of the docker image.
     *
     * The uri spec follows https://github.com/containers/skopeo
     *
     * @stability stable
     */
    get uri(): string;
}
/**
 * @stability stable
 */
export declare class S3ArchiveName implements IImageName {
    creds?: string | undefined;
    private name;
    /**
     * @stability stable
     */
    constructor(p: string, ref?: string, creds?: string | undefined);
    /**
     * The uri of the docker image.
     *
     * The uri spec follows https://github.com/containers/skopeo
     *
     * @stability stable
     */
    get uri(): string;
}
/**
 * @stability stable
 */
export declare class ECRDeployment extends Construct {
    private handler;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: ECRDeploymentProps);
    /**
     * @stability stable
     */
    addToPrincipalPolicy(statement: PolicyStatement): AddToPrincipalPolicyResult;
    private renderSingletonUuid;
}
