"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ECRDeployment = exports.S3ArchiveName = exports.DockerImageName = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const child_process = require("child_process");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const config_1 = require("./config");
function getCode(buildImage) {
    if (config_1.shouldUsePrebuiltLambda()) {
        try {
            console.log('Try to get prebuilt lambda');
            const installScript = path.join(__dirname, '../lambda/install.js');
            const prebuiltPath = path.join(__dirname, '../lambda/out');
            child_process.execSync(`${process.argv0} ${installScript} ${prebuiltPath}`);
            return aws_cdk_lib_1.aws_lambda.Code.fromAsset(prebuiltPath);
        }
        catch (err) {
            console.warn(`Can not get prebuilt lambda: ${err}`);
        }
    }
    console.log('Build lambda from scratch');
    return aws_cdk_lib_1.aws_lambda.Code.fromDockerBuild(path.join(__dirname, '../lambda'), {
        buildArgs: {
            buildImage,
        },
    });
}
/**
 * @stability stable
 */
class DockerImageName {
    /**
     * @stability stable
     */
    constructor(name, creds) {
        this.name = name;
        this.creds = creds;
    }
    /**
     * The uri of the docker image.
     *
     * The uri spec follows https://github.com/containers/skopeo
     *
     * @stability stable
     */
    get uri() { return `docker://${this.name}`; }
}
exports.DockerImageName = DockerImageName;
_a = JSII_RTTI_SYMBOL_1;
DockerImageName[_a] = { fqn: "cdk-ecr-deployment.DockerImageName", version: "2.5.6" };
/**
 * @stability stable
 */
class S3ArchiveName {
    /**
     * @stability stable
     */
    constructor(p, ref, creds) {
        this.creds = creds;
        this.name = p;
        if (ref) {
            this.name += ':' + ref;
        }
    }
    /**
     * The uri of the docker image.
     *
     * The uri spec follows https://github.com/containers/skopeo
     *
     * @stability stable
     */
    get uri() { return `s3://${this.name}`; }
}
exports.S3ArchiveName = S3ArchiveName;
_b = JSII_RTTI_SYMBOL_1;
S3ArchiveName[_b] = { fqn: "cdk-ecr-deployment.S3ArchiveName", version: "2.5.6" };
/**
 * @stability stable
 */
class ECRDeployment extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d, _e;
        super(scope, id);
        const memoryLimit = (_d = props.memoryLimit) !== null && _d !== void 0 ? _d : 512;
        this.handler = new aws_cdk_lib_1.aws_lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(memoryLimit),
            code: getCode((_e = props.buildImage) !== null && _e !== void 0 ? _e : 'public.ecr.aws/sam/build-go1.x:latest'),
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.GO_1_X,
            handler: 'main',
            environment: props.environment,
            lambdaPurpose: 'Custom::CDKECRDeployment',
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: props.role,
            memorySize: memoryLimit,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
        });
        const handlerRole = this.handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        handlerRole.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                'ecr:GetAuthorizationToken',
                'ecr:BatchCheckLayerAvailability',
                'ecr:GetDownloadUrlForLayer',
                'ecr:GetRepositoryPolicy',
                'ecr:DescribeRepositories',
                'ecr:ListImages',
                'ecr:DescribeImages',
                'ecr:BatchGetImage',
                'ecr:ListTagsForResource',
                'ecr:DescribeImageScanFindings',
                'ecr:InitiateLayerUpload',
                'ecr:UploadLayerPart',
                'ecr:CompleteLayerUpload',
                'ecr:PutImage',
            ],
            resources: ['*'],
        }));
        handlerRole.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                's3:GetObject',
            ],
            resources: ['*'],
        }));
        new aws_cdk_lib_1.CustomResource(this, 'CustomResource', {
            serviceToken: this.handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SrcImage: props.src.uri,
                SrcCreds: props.src.creds,
                DestImage: props.dest.uri,
                DestCreds: props.dest.creds,
            },
        });
    }
    /**
     * @stability stable
     */
    addToPrincipalPolicy(statement) {
        const handlerRole = this.handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        return handlerRole.addToPrincipalPolicy(statement);
    }
    renderSingletonUuid(memoryLimit) {
        let uuid = 'bd07c930-edb9-4112-a20f-03f096f53666';
        // if user specify a custom memory limit, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (aws_cdk_lib_1.Token.isUnresolved(memoryLimit)) {
                throw new Error('Can\'t use tokens when specifying "memoryLimit" since we use it to identify the singleton custom resource handler');
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        return uuid;
    }
}
exports.ECRDeployment = ECRDeployment;
_c = JSII_RTTI_SYMBOL_1;
ECRDeployment[_c] = { fqn: "cdk-ecr-deployment.ECRDeployment", version: "2.5.6" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxxRUFBcUU7QUFDckUsc0NBQXNDO0FBR3RDLCtDQUErQztBQUMvQyw2QkFBNkI7QUFDN0IsNkNBQW9IO0FBRXBILDJDQUF1QztBQUN2QyxxQ0FBbUQ7QUFvQ25ELFNBQVMsT0FBTyxDQUFDLFVBQWtCO0lBQ2pDLElBQUksZ0NBQXVCLEVBQUUsRUFBRTtRQUM3QixJQUFJO1lBQ0YsT0FBTyxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1lBRTFDLE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHNCQUFzQixDQUFDLENBQUM7WUFDbkUsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFDM0QsYUFBYSxDQUFDLFFBQVEsQ0FBQyxHQUFHLE9BQU8sQ0FBQyxLQUFLLElBQUksYUFBYSxJQUFJLFlBQVksRUFBRSxDQUFDLENBQUM7WUFFNUUsT0FBTyx3QkFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDNUM7UUFBQyxPQUFPLEdBQUcsRUFBRTtZQUNaLE9BQU8sQ0FBQyxJQUFJLENBQUMsZ0NBQWdDLEdBQUcsRUFBRSxDQUFDLENBQUM7U0FDckQ7S0FDRjtJQUVELE9BQU8sQ0FBQyxHQUFHLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUV6QyxPQUFPLHdCQUFNLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxXQUFXLENBQUMsRUFBRTtRQUNwRSxTQUFTLEVBQUU7WUFDVCxVQUFVO1NBQ1g7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDOzs7O0FBRUQsTUFBYSxlQUFlOzs7O0lBQzFCLFlBQTJCLElBQVksRUFBUyxLQUFjO1FBQW5DLFNBQUksR0FBSixJQUFJLENBQVE7UUFBUyxVQUFLLEdBQUwsS0FBSyxDQUFTO0lBQUksQ0FBQzs7Ozs7Ozs7SUFDbkUsSUFBVyxHQUFHLEtBQWEsT0FBTyxZQUFZLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUM7O0FBRjlELDBDQUdDOzs7Ozs7QUFFRCxNQUFhLGFBQWE7Ozs7SUFFeEIsWUFBbUIsQ0FBUyxFQUFFLEdBQVksRUFBUyxLQUFjO1FBQWQsVUFBSyxHQUFMLEtBQUssQ0FBUztRQUMvRCxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNkLElBQUksR0FBRyxFQUFFO1lBQ1AsSUFBSSxDQUFDLElBQUksSUFBSSxHQUFHLEdBQUcsR0FBRyxDQUFDO1NBQ3hCO0lBQ0gsQ0FBQzs7Ozs7Ozs7SUFDRCxJQUFXLEdBQUcsS0FBYSxPQUFPLFFBQVEsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQzs7QUFSMUQsc0NBU0M7Ozs7OztBQUVELE1BQWEsYUFBYyxTQUFRLHNCQUFTOzs7O0lBRzFDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBeUI7O1FBQ2pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDakIsTUFBTSxXQUFXLFNBQUcsS0FBSyxDQUFDLFdBQVcsbUNBQUksR0FBRyxDQUFDO1FBQzdDLElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSx3QkFBTSxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRTtZQUN6RSxJQUFJLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFdBQVcsQ0FBQztZQUMzQyxJQUFJLEVBQUUsT0FBTyxPQUFDLEtBQUssQ0FBQyxVQUFVLG1DQUFJLHVDQUF1QyxDQUFDO1lBQzFFLE9BQU8sRUFBRSx3QkFBTSxDQUFDLE9BQU8sQ0FBQyxNQUFNO1lBQzlCLE9BQU8sRUFBRSxNQUFNO1lBQ2YsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzlCLGFBQWEsRUFBRSwwQkFBMEI7WUFDekMsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUk7WUFDaEIsVUFBVSxFQUFFLFdBQVc7WUFDdkIsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO1lBQ2QsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO1NBQzdCLENBQUMsQ0FBQztRQUVILE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDO1FBQ3RDLElBQUksQ0FBQyxXQUFXLEVBQUU7WUFBRSxNQUFNLElBQUksS0FBSyxDQUFDLHFEQUFxRCxDQUFDLENBQUM7U0FBRTtRQUU3RixXQUFXLENBQUMsb0JBQW9CLENBQzlCLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsTUFBTSxFQUFFLHFCQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLDJCQUEyQjtnQkFDM0IsaUNBQWlDO2dCQUNqQyw0QkFBNEI7Z0JBQzVCLHlCQUF5QjtnQkFDekIsMEJBQTBCO2dCQUMxQixnQkFBZ0I7Z0JBQ2hCLG9CQUFvQjtnQkFDcEIsbUJBQW1CO2dCQUNuQix5QkFBeUI7Z0JBQ3pCLCtCQUErQjtnQkFDL0IseUJBQXlCO2dCQUN6QixxQkFBcUI7Z0JBQ3JCLHlCQUF5QjtnQkFDekIsY0FBYzthQUNmO1lBQ0QsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1NBQ2pCLENBQUMsQ0FBQyxDQUFDO1FBQ04sV0FBVyxDQUFDLG9CQUFvQixDQUFDLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdkQsTUFBTSxFQUFFLHFCQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLGNBQWM7YUFDZjtZQUNELFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNqQixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksNEJBQWMsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUU7WUFDekMsWUFBWSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVztZQUN0QyxZQUFZLEVBQUUsNkJBQTZCO1lBQzNDLFVBQVUsRUFBRTtnQkFDVixRQUFRLEVBQUUsS0FBSyxDQUFDLEdBQUcsQ0FBQyxHQUFHO2dCQUN2QixRQUFRLEVBQUUsS0FBSyxDQUFDLEdBQUcsQ0FBQyxLQUFLO2dCQUN6QixTQUFTLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHO2dCQUN6QixTQUFTLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxLQUFLO2FBQzVCO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7OztJQUVNLG9CQUFvQixDQUFDLFNBQTBCO1FBQ3BELE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDO1FBQ3RDLElBQUksQ0FBQyxXQUFXLEVBQUU7WUFBRSxNQUFNLElBQUksS0FBSyxDQUFDLHFEQUFxRCxDQUFDLENBQUM7U0FBRTtRQUU3RixPQUFPLFdBQVcsQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUNyRCxDQUFDO0lBRU8sbUJBQW1CLENBQUMsV0FBb0I7UUFDOUMsSUFBSSxJQUFJLEdBQUcsc0NBQXNDLENBQUM7UUFFbEQsMEVBQTBFO1FBQzFFLDJFQUEyRTtRQUMzRSw0Q0FBNEM7UUFDNUMsSUFBSSxXQUFXLEVBQUU7WUFDZixJQUFJLG1CQUFLLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxFQUFFO2dCQUNuQyxNQUFNLElBQUksS0FBSyxDQUFDLG1IQUFtSCxDQUFDLENBQUM7YUFDdEk7WUFFRCxJQUFJLElBQUksSUFBSSxXQUFXLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQztTQUN6QztRQUVELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQzs7QUF0Rkgsc0NBdUZDIiwic291cmNlc0NvbnRlbnQiOlsiLy8gQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4vLyBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuXG5cbmltcG9ydCAqIGFzIGNoaWxkX3Byb2Nlc3MgZnJvbSAnY2hpbGRfcHJvY2Vzcyc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgYXdzX2VjMiBhcyBlYzIsIGF3c19pYW0gYXMgaWFtLCBhd3NfbGFtYmRhIGFzIGxhbWJkYSwgRHVyYXRpb24sIEN1c3RvbVJlc291cmNlLCBUb2tlbiB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCwgQWRkVG9QcmluY2lwYWxQb2xpY3lSZXN1bHQgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgc2hvdWxkVXNlUHJlYnVpbHRMYW1iZGEgfSBmcm9tICcuL2NvbmZpZyc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgRUNSRGVwbG95bWVudFByb3BzIHtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBidWlsZEltYWdlPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNyYzogSUltYWdlTmFtZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVzdDogSUltYWdlTmFtZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWVtb3J5TGltaXQ/OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdnBjPzogZWMyLklWcGM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZwY1N1Ym5ldHM/OiBlYzIuU3VibmV0U2VsZWN0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW52aXJvbm1lbnQ/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIElJbWFnZU5hbWUge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB1cmk6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgY3JlZHM/OiBzdHJpbmc7XG59XG5cbmZ1bmN0aW9uIGdldENvZGUoYnVpbGRJbWFnZTogc3RyaW5nKTogbGFtYmRhLkFzc2V0Q29kZSB7XG4gIGlmIChzaG91bGRVc2VQcmVidWlsdExhbWJkYSgpKSB7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnNvbGUubG9nKCdUcnkgdG8gZ2V0IHByZWJ1aWx0IGxhbWJkYScpO1xuXG4gICAgICBjb25zdCBpbnN0YWxsU2NyaXB0ID0gcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL2xhbWJkYS9pbnN0YWxsLmpzJyk7XG4gICAgICBjb25zdCBwcmVidWlsdFBhdGggPSBwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vbGFtYmRhL291dCcpO1xuICAgICAgY2hpbGRfcHJvY2Vzcy5leGVjU3luYyhgJHtwcm9jZXNzLmFyZ3YwfSAke2luc3RhbGxTY3JpcHR9ICR7cHJlYnVpbHRQYXRofWApO1xuXG4gICAgICByZXR1cm4gbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHByZWJ1aWx0UGF0aCk7XG4gICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICBjb25zb2xlLndhcm4oYENhbiBub3QgZ2V0IHByZWJ1aWx0IGxhbWJkYTogJHtlcnJ9YCk7XG4gICAgfVxuICB9XG5cbiAgY29uc29sZS5sb2coJ0J1aWxkIGxhbWJkYSBmcm9tIHNjcmF0Y2gnKTtcblxuICByZXR1cm4gbGFtYmRhLkNvZGUuZnJvbURvY2tlckJ1aWxkKHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9sYW1iZGEnKSwge1xuICAgIGJ1aWxkQXJnczoge1xuICAgICAgYnVpbGRJbWFnZSxcbiAgICB9LFxuICB9KTtcbn1cblxuZXhwb3J0IGNsYXNzIERvY2tlckltYWdlTmFtZSBpbXBsZW1lbnRzIElJbWFnZU5hbWUge1xuICBwdWJsaWMgY29uc3RydWN0b3IocHJpdmF0ZSBuYW1lOiBzdHJpbmcsIHB1YmxpYyBjcmVkcz86IHN0cmluZykgeyB9XG4gIHB1YmxpYyBnZXQgdXJpKCk6IHN0cmluZyB7IHJldHVybiBgZG9ja2VyOi8vJHt0aGlzLm5hbWV9YDsgfVxufVxuXG5leHBvcnQgY2xhc3MgUzNBcmNoaXZlTmFtZSBpbXBsZW1lbnRzIElJbWFnZU5hbWUge1xuICBwcml2YXRlIG5hbWU6IHN0cmluZztcbiAgcHVibGljIGNvbnN0cnVjdG9yKHA6IHN0cmluZywgcmVmPzogc3RyaW5nLCBwdWJsaWMgY3JlZHM/OiBzdHJpbmcpIHtcbiAgICB0aGlzLm5hbWUgPSBwO1xuICAgIGlmIChyZWYpIHtcbiAgICAgIHRoaXMubmFtZSArPSAnOicgKyByZWY7XG4gICAgfVxuICB9XG4gIHB1YmxpYyBnZXQgdXJpKCk6IHN0cmluZyB7IHJldHVybiBgczM6Ly8ke3RoaXMubmFtZX1gOyB9XG59XG5cbmV4cG9ydCBjbGFzcyBFQ1JEZXBsb3ltZW50IGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgcHJpdmF0ZSBoYW5kbGVyOiBsYW1iZGEuU2luZ2xldG9uRnVuY3Rpb247XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEVDUkRlcGxveW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgY29uc3QgbWVtb3J5TGltaXQgPSBwcm9wcy5tZW1vcnlMaW1pdCA/PyA1MTI7XG4gICAgdGhpcy5oYW5kbGVyID0gbmV3IGxhbWJkYS5TaW5nbGV0b25GdW5jdGlvbih0aGlzLCAnQ3VzdG9tUmVzb3VyY2VIYW5kbGVyJywge1xuICAgICAgdXVpZDogdGhpcy5yZW5kZXJTaW5nbGV0b25VdWlkKG1lbW9yeUxpbWl0KSxcbiAgICAgIGNvZGU6IGdldENvZGUocHJvcHMuYnVpbGRJbWFnZSA/PyAncHVibGljLmVjci5hd3Mvc2FtL2J1aWxkLWdvMS54OmxhdGVzdCcpLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuR09fMV9YLFxuICAgICAgaGFuZGxlcjogJ21haW4nLFxuICAgICAgZW52aXJvbm1lbnQ6IHByb3BzLmVudmlyb25tZW50LFxuICAgICAgbGFtYmRhUHVycG9zZTogJ0N1c3RvbTo6Q0RLRUNSRGVwbG95bWVudCcsXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDE1KSxcbiAgICAgIHJvbGU6IHByb3BzLnJvbGUsXG4gICAgICBtZW1vcnlTaXplOiBtZW1vcnlMaW1pdCxcbiAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgICAgdnBjU3VibmV0czogcHJvcHMudnBjU3VibmV0cyxcbiAgICB9KTtcblxuICAgIGNvbnN0IGhhbmRsZXJSb2xlID0gdGhpcy5oYW5kbGVyLnJvbGU7XG4gICAgaWYgKCFoYW5kbGVyUm9sZSkgeyB0aHJvdyBuZXcgRXJyb3IoJ2xhbWJkYS5TaW5nbGV0b25GdW5jdGlvbiBzaG91bGQgaGF2ZSBjcmVhdGVkIGEgUm9sZScpOyB9XG5cbiAgICBoYW5kbGVyUm9sZS5hZGRUb1ByaW5jaXBhbFBvbGljeShcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgJ2VjcjpHZXRBdXRob3JpemF0aW9uVG9rZW4nLFxuICAgICAgICAgICdlY3I6QmF0Y2hDaGVja0xheWVyQXZhaWxhYmlsaXR5JyxcbiAgICAgICAgICAnZWNyOkdldERvd25sb2FkVXJsRm9yTGF5ZXInLFxuICAgICAgICAgICdlY3I6R2V0UmVwb3NpdG9yeVBvbGljeScsXG4gICAgICAgICAgJ2VjcjpEZXNjcmliZVJlcG9zaXRvcmllcycsXG4gICAgICAgICAgJ2VjcjpMaXN0SW1hZ2VzJyxcbiAgICAgICAgICAnZWNyOkRlc2NyaWJlSW1hZ2VzJyxcbiAgICAgICAgICAnZWNyOkJhdGNoR2V0SW1hZ2UnLFxuICAgICAgICAgICdlY3I6TGlzdFRhZ3NGb3JSZXNvdXJjZScsXG4gICAgICAgICAgJ2VjcjpEZXNjcmliZUltYWdlU2NhbkZpbmRpbmdzJyxcbiAgICAgICAgICAnZWNyOkluaXRpYXRlTGF5ZXJVcGxvYWQnLFxuICAgICAgICAgICdlY3I6VXBsb2FkTGF5ZXJQYXJ0JyxcbiAgICAgICAgICAnZWNyOkNvbXBsZXRlTGF5ZXJVcGxvYWQnLFxuICAgICAgICAgICdlY3I6UHV0SW1hZ2UnLFxuICAgICAgICBdLFxuICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgfSkpO1xuICAgIGhhbmRsZXJSb2xlLmFkZFRvUHJpbmNpcGFsUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ3MzOkdldE9iamVjdCcsXG4gICAgICBdLFxuICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICB9KSk7XG5cbiAgICBuZXcgQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0N1c3RvbVJlc291cmNlJywge1xuICAgICAgc2VydmljZVRva2VuOiB0aGlzLmhhbmRsZXIuZnVuY3Rpb25Bcm4sXG4gICAgICByZXNvdXJjZVR5cGU6ICdDdXN0b206OkNES0J1Y2tldERlcGxveW1lbnQnLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBTcmNJbWFnZTogcHJvcHMuc3JjLnVyaSxcbiAgICAgICAgU3JjQ3JlZHM6IHByb3BzLnNyYy5jcmVkcyxcbiAgICAgICAgRGVzdEltYWdlOiBwcm9wcy5kZXN0LnVyaSxcbiAgICAgICAgRGVzdENyZWRzOiBwcm9wcy5kZXN0LmNyZWRzLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBhZGRUb1ByaW5jaXBhbFBvbGljeShzdGF0ZW1lbnQ6IFBvbGljeVN0YXRlbWVudCk6IEFkZFRvUHJpbmNpcGFsUG9saWN5UmVzdWx0IHtcbiAgICBjb25zdCBoYW5kbGVyUm9sZSA9IHRoaXMuaGFuZGxlci5yb2xlO1xuICAgIGlmICghaGFuZGxlclJvbGUpIHsgdGhyb3cgbmV3IEVycm9yKCdsYW1iZGEuU2luZ2xldG9uRnVuY3Rpb24gc2hvdWxkIGhhdmUgY3JlYXRlZCBhIFJvbGUnKTsgfVxuXG4gICAgcmV0dXJuIGhhbmRsZXJSb2xlLmFkZFRvUHJpbmNpcGFsUG9saWN5KHN0YXRlbWVudCk7XG4gIH1cblxuICBwcml2YXRlIHJlbmRlclNpbmdsZXRvblV1aWQobWVtb3J5TGltaXQ/OiBudW1iZXIpIHtcbiAgICBsZXQgdXVpZCA9ICdiZDA3YzkzMC1lZGI5LTQxMTItYTIwZi0wM2YwOTZmNTM2NjYnO1xuXG4gICAgLy8gaWYgdXNlciBzcGVjaWZ5IGEgY3VzdG9tIG1lbW9yeSBsaW1pdCwgZGVmaW5lIGFub3RoZXIgc2luZ2xldG9uIGhhbmRsZXJcbiAgICAvLyB3aXRoIHRoaXMgY29uZmlndXJhdGlvbi4gb3RoZXJ3aXNlLCBpdCB3b24ndCBiZSBwb3NzaWJsZSB0byB1c2UgbXVsdGlwbGVcbiAgICAvLyBjb25maWd1cmF0aW9ucyBzaW5jZSB3ZSBoYXZlIGEgc2luZ2xldG9uLlxuICAgIGlmIChtZW1vcnlMaW1pdCkge1xuICAgICAgaWYgKFRva2VuLmlzVW5yZXNvbHZlZChtZW1vcnlMaW1pdCkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5cXCd0IHVzZSB0b2tlbnMgd2hlbiBzcGVjaWZ5aW5nIFwibWVtb3J5TGltaXRcIiBzaW5jZSB3ZSB1c2UgaXQgdG8gaWRlbnRpZnkgdGhlIHNpbmdsZXRvbiBjdXN0b20gcmVzb3VyY2UgaGFuZGxlcicpO1xuICAgICAgfVxuXG4gICAgICB1dWlkICs9IGAtJHttZW1vcnlMaW1pdC50b1N0cmluZygpfU1pQmA7XG4gICAgfVxuXG4gICAgcmV0dXJuIHV1aWQ7XG4gIH1cbn1cbiJdfQ==