# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['soda']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'soda-svg',
    'version': '1.1.2',
    'description': 'Fast SVG generation tool',
    'long_description': '# soda - a fast SVG generation tool\n\nHere\'s some basic usage:\n\n```python\nfrom soda import Tag, Root\n\n# root is a custom svg tag\nroot = Root(viewBox="0 0 10 10")(\n    Tag.rect(width=10, height=10, fill="#ff5"),\n    Tag.circle(cx=5, cy=5, r=4, fill="#222")\n)\n\nprint(root.render(pretty=True))\n```\n\n## Installation\n\nJust use `python setup.py` or `python -m pip install soda-svg`\n\n## Tag construction\n\nThe main class of the module is `Tag`. You can create it with a constructor:\n\n```python\nTag("g")\n```\n\nor with a shorthand:\n\n```python\nTag.g\n```\n\nYou can also pass children and attributes into the constructor:\n\n```python\nTag("g", child1, child2, attr1=1, attr2=2)\n```\n\nor as call arguments (this would change tag in place, no additional copies):\n\n```python\nTag("g")(child1, attr1=1)(child2, child3, attr2=10, attr3=3)\n# or\nTag.g(child1, attr1=1)(child2, child3, attr2=10, attr3=3)\n\n# \'_\' to \'-\' conversion\nTag.g(child1, attr_1=1)(child2, child3, attr_2=10, attr_3=3) # <g attr-1="1" attr-2="10" attr-3="3" >child1child2child3</g>\n```\n\n## Float rounding\n\nAs floats can be used as attribute values (even though in resulting SVG every value is a string), module will round floats to 3 digits after the decimal point:\n\n```python\nfrom soda import Tag\n\ng = Tag.g(x=1/3)\nprint(g.render()) # \'<g x="0.333"/>\'\n\n```\n\nTo change this behaviour, edit `soda.config.decimal_length` before value is assigned:\n\n```python\nfrom soda import Tag, config as soda_config\n\nsoda_config.decimal_length = 4\n\ng = Tag.g(x=1/3)\nprint(g.render()) # \'<g x="0.3333"/>\'\n\nsoda_config.decimal_length = 2\ng(y=1/3)\n\nprint(g.render()) # \'<g x="0.3333" y="0.33"/>\'\n\n```\n\n## Attribute conversion\n\nFor convenience, leading and trailing underscores are removed by default, and underscores in the middle of words are replaced with hyphens:\n\n```python\nfrom soda import Tag\n\ng = Tag.g(cla_ss_="test") \n\nprint(g.render()) # <g cla-ss="test"/>\n\n```\n\nTo disable replacing behavior, use `config` class:\n\n```python\nfrom soda import Tag, config as soda_config\n\nsoda_config.replace_underscores = False\n\ng = Tag.g(cla_ss_="test") \n\nprint(g.render()) # <g cla_ss="test"/>\n\n```\n\n...and to disable stripping of leading/traililng underscores:\n\n```python\nfrom soda import Tag, config as soda_config\n\nsoda_config.strip_underscores = False\n\ng = Tag.g(cla_ss_="test") \n\nprint(g.render()) # <g cla-ss_="test"/>\n```\n\nIt\'s important to do that before tag creation, as all conversions are happening at the tag creation time:\n\n```python\n\nfrom soda import Tag, config as soda_config\n\ng1 = Tag.g(cla_ss_="test") # g.attributes == {"cla-ss": "test"}\n\nsoda_config.replace_underscores = False\n\ng2 = Tag.g(cla_ss_="test") # g.attributes == {"cla_ss_": "test"}\n\nprint(g1.render()) # <g cla-ss="test"/>\nprint(g2.render()) # <g cla_ss_="test"/>\n\n```\n\n## Creating a Tag from XML string\n\n*new in 1.1.0*\n\nYou can use `Tag.from_str(xml_string)` to parse an XML document in that string.    \nNote that currently this doesn\'t preserve any comments or declarations of original document.\n\n```python\nfrom soda import Tag, Root\n\nroot = Root(viewBox="0 0 10 10")(\n    Tag.rect(width=10, height=10, fill="#ff5"),\n    Tag.circle(cx=5, cy=5, r=4, fill="#222")\n)\n\nrendered_root = root.render(pretty=True)\nnew_root = Tag.from_str(rendered_root)\n\nassert rendered_root == new_root.render(pretty=True)\n```\n\n### Text\n\nBasic text handling is pretty straightforward:\n\n```python\nfrom soda import Tag\n\nTag.text("Hello, World") # just pass a string as a children\n```\n\nThis code is roughly equivalent to:\n\n```python\nfrom soda import Tag, Literal\n\nTag.text(Literal("Hello, World"))\n```\n\n...except that first piece doesn\'t create a `Literal` object.\n\nIf you need to add unescaped text (such as prerendered XML), you should pass `escape=False` to a `Literal` constructor:\n\n```python\nfrom soda import Tag, Literal\n\nTag.g(Literal(\'<path d="M0 0 L10 0 Z"/>\', escape=False))\n```\n\n## Accessing data\n\n`tag[attr]` syntax can be used to manage tag attributes (where `attr` should be a string).\n\n```python\nfrom soda import Tag\n\ntag = Tag.g\ntag["id"] = "yes-thats-an-id" # sets attribute\ntag["cool"] = None # deletes attribute if exists, otherwise does nothing\nprint(tag["id"]) # prints attribute\nprint(tag["non-existent-attribute"]) # prints None\n```\n\n`tag[index]` syntax can be used to manage tag children (where `index` should be either integer or slice).\n\n```python\nfrom soda import Tag\n\ntag = Tag.g(Tag.a)\ntag[0]["href"] = "https://github.com/evtn/soda"\nprint(tag[1]) # IndexError\nprint(tag[0]) # prints <a href="https://github.com/evtn/soda" />\n```\n\nChildren can also be accessed directly through `tag.children` attribute.\n\n## Fragments\n\nFragments use concept similar to React\'s fragment. It renders just it\'s children:\n\n```python\nfrom soda import Tag, Fragment\n\ntag = Tag.g(\n    Fragment(Tag.a, Tag.a)\n)\nprint(tag) # <g><a/><a/></g>\n\n```\n\n## Paths\n\n*new in 0.1.7*\n\nThere is a builder for SVG path commands in soda:\n\n<svg viewBox="0 0 100 100">\n    <rect width="100%" height="100%" fill="white"/>\n    <path d="M 10,30\n           A 20,20 0,0,1 50,30\n           A 20,20 0,0,1 90,30\n           Q 90,60 50,90\n           Q 10,60 10,30 z"\n    />\n</svg>\n\nYou can build a list of path commands using descriptive command names:\n\n```python\nfrom soda import Tag, Root, Path\n\ncommands = (\n    Path.moveto(x=10, y=30),\n    Path.arc(\n        radius_x=20,\n        radius_y=20,\n        # for convenience, omitted arguments\n        # (here: x_axis_rotation and large_arc_flag) are set to 0\n        sweep_flag=1,\n        x=50,\n        y=30,\n    ),\n    Path.arc(\n        radius_x=20,\n        radius_y=20,\n        sweep_flag=1,\n        x=90,\n        y=30,\n    ),\n    Path.quadratic(\n        x1=90,\n        y1=60,\n        x=50,\n        y=90,\n    ),\n    Path.quadratic(\n        x1=10,\n        y1=60,\n        x=10,\n        y=30,\n    ),\n    Path.close()\n)\n\n\n```\n\n...or using common SVG command names (letter case signifies if command is relative):\n\n```python\n\n# or\n\ncommands = (\n    Path.M(10, 30),\n    Path.A(20, 20, 0, 0, 1, 50, 30),\n    Path.A(20, 20, 0, 0, 1, 50, 30),\n    Path.Q(90, 60, 50, 90),\n    Path.Q(10, 60, 10, 30),\n    Path.Z()\n)\n\n```\n\n...and render it with `Path.build(*commands, compact=False)` method\n\n```python\n\nroot = Root(\n    viewBox="0 0 100 100",\n    use_namespace=True,\n)(\n    Tag.rect(width="100%", height="100%", fill="white"),\n    Tag.path()(\n        d=Path.build(*commands)\n    )\n)\n\nprint(root.render(pretty=True))\n\n"""\nyields:\n\n<svg\n  viewBox="0 0 100 100"\n  version="2.0"\n  xmlns="http://www.w3.org/2000/svg"\n  xmlns:xlink="http://www.w3.org/1999/xlink"\n>\n  <rect\n    width="100%"\n    height="100%"\n    fill="white"\n  />\n  <path\n    d="M 10 30 A 20 20 0 0 1 50 30 A 20 20 0 0 1 50 30 Q 90 60 50 90 Q 10 60 10 30 Z"\n  />\n</svg>\n"""\n```\n\nYou can also optimize resulting path with `compact` argument:\n\n```python\nprint(Path.build(*commands, compact=True))\n# prints M10 30A20 20 0 0 1 50 30A20 20 0 0 1 50 30Q90 60 50 90Q10 60 10 30Z\n```\n\n## Points\n\n*new in 1.1.0*\n\nTo work with coordinates, you can use `Point` class:\n\n```python\nfrom soda import Point\n\na = Point(1, 2)\nb = Point(4, 5)\n```\n\nIn any context where a point could be used, "point-like" values can be used:\n\n-   `(1, 2)` <-> `Point(1, 2)`\n-   `[1, 2]` <-> `Point(1, 2)`\n-   `1` <-> `Point(1, 1)`\n\nYou can use coordinates of a point:\n\n```python\nprint(a) # Point[1, 2]\nprint(a.x) # 1\nprint(a.coords) # (1, 2)\nprint([*a]) # [1, 2] (same as [*a.coords])\n```\n\n...perform mathematical operations on points:\n\n```python\nprint(a + b) # Point[5, 7]\nprint(a - b) # Point[-3, -3]\nprint(a * b) # Point[4, 10] (a.x * b.x, a.y * b.y)\nprint(a / b) # Point[0.25, 0.4]\nprint(a % b) # Point[1, 2]\n```\n\n...and any point-like values:\n\n```python\nprint(a + 10) # Point[11, 12]\nprint(a * 2) # Point[2, 4]\n```\n\nYou also can calculate distance between points and rotate a point around some point:\n\n```python\nfrom math import pi\n\nprint(a.distance(b)) # 4.242640687119285\nprint(a.distance()) # 2.23606797749979 (distance between a and (0, 0), basically the length of a vector)\nprint(a.rotate(degrees=90)) # Point[-2, 1]\nprint(a.rotate((10, 10), degrees=90))\nprint(a.rotate((10, 10), radians=pi / 2)) # Point[18, 1]\n```\n\n...and get a normalized vector:\n\n```python\nprint(a.normalized()) # Point[0.4472135954999579, 0.8944271909999159]\n```\n\nYou also can get an angle (in radians) between two vectors (with specified starting point):\n\n```python\nprint(a.angle(b)) # 0.21109333322274684\nprint(a.angle(b, (10, 10))) # 0.03190406448501816 (second argument specifies starting point (0, 0) by default)\nprint(a.angle()) # 1.1071487177940904 (angle between `a` and (1, 0) vector)\n\nprint(\n    a.angle(\n        a.rotate(radians=2)\n    )\n) # 2\n```\n\n### Converting angles\n\nTo convert between radians and degrees, use `radians_to_degrees` and `degrees_to_radians`:\n\n```python\nfrom soda.point import radians_to_degrees, degrees_to_radians\nfrom math import pi\n\nprint(degrees_to_radians(90) / pi) # 0.5\nprint(radians_to_degrees(degrees_to_radians(90))) # 90\n```\n\n### Using as attributes\n\n`Point.as_` provides a convenient way of using points as tag attributes:\n\n```python\nfrom soda import Tag, Point\n\nsize = Point(100, 200)\n\nprint(size.as_()) # {"x": 100, "y": 200}\nprint(size.as_("width", "height")) # {"width": 100, "height": 200}\n\n\nprint(\n    Tag.rect(\n        **size.as_("width", "height")\n    )\n) # <rect width="100" height="200"/>\n\n```\n\n### PointPath\n\nA version of [`Path`](#paths) accepting `Point` instead of some arguments.\nWhere Path.something(...) accepts coordinates (as two arguments) or some size (like `radius_x` and `radius_y` in `arc`), `PointPath` accepts a point-like object instead.\n\n## Custom components\n\nYou can build custom components, using different approaches:\n\n### Building a tree on init\n\nBuilds a tree on every component creation\n\n```python\nfrom soda import Tag, Fragment\n\nclass CustomComponent(Fragment):\n    def __init__(self):\n        children = Tag.g(\n            Tag.anythingother,\n            Tag.lalala(\n                Tag.yes,\n                Tag.no\n            )\n        )\n        super().__init__(*children)\n\nCustomComponent().render()\n```\n\n### Functional approach\n\nBuilds a tree on every call\n\n```python\nfrom soda import Tag\n\ndef custom_component():\n    return Tag.g(\n        Tag.anythingother,\n        Tag.lalala(\n            Tag.yes,\n            Tag.no\n        )\n    )\n\ncustom_component().render()\n```\n\n## Speed\n\nsoda is able to render tens of thousands tags per second, but if you wanna optimize your execution, there are some tips:\n\n### Building a tree efficiently\n\nIf you using the same structure many times (especially if it\'s a heavy one), avoid rebuilds. Rather than building a new tree every time, consider changing specific parts of it when needed. It won\'t speed up the render time, though\n\n### Prerendering\n\nIf you have some static tags, you can use `tag.prerender()` to get a prerendered `Literal`.\nThis could speed up your render significantly in some cases.\n\n### Pretty or not?\n\nPretty render gives a nice formatted output, which is very readable.  \n~~But using `pretty=True` in rendering would make renders 3-5x slower than default `pretty=False`.~~  \nStarting with 0.1.5 version, pretty rendering is roughly the same in speed as default one.\n',
    'author': 'Dmitry Gritsenko',
    'author_email': 'soda@evtn.ru',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/evtn/soda',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
