"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ListenerCondition = void 0;
/**
 * ListenerCondition providers definition.
 */
class ListenerCondition {
    /**
     * Create a host-header listener rule condition.
     *
     * @param values Hosts for host headers.
     */
    static hostHeaders(values) {
        return new HostHeaderListenerCondition(values);
    }
    /**
     * Create a http-header listener rule condition.
     *
     * @param name HTTP header name.
     * @param values HTTP header values.
     */
    static httpHeader(name, values) {
        return new HttpHeaderListenerCondition(name, values);
    }
    /**
     * Create a http-request-method listener rule condition.
     *
     * @param values HTTP request methods.
     */
    static httpRequestMethods(values) {
        return new HttpRequestMethodListenerCondition(values);
    }
    /**
     * Create a path-pattern listener rule condition.
     *
     * @param values Path patterns.
     */
    static pathPatterns(values) {
        return new PathPatternListenerCondition(values);
    }
    /**
     * Create a query-string listener rule condition.
     *
     * @param values Query string key/value pairs.
     */
    static queryStrings(values) {
        return new QueryStringListenerCondition(values);
    }
    /**
     * Create a source-ip listener rule condition.
     *
     * @param values Source ips.
     */
    static sourceIps(values) {
        return new SourceIpListenerCondition(values);
    }
}
exports.ListenerCondition = ListenerCondition;
/**
 * Host header config of the listener rule condition
 */
class HostHeaderListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'host-header',
            hostHeaderConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * HTTP header config of the listener rule condition
 */
class HttpHeaderListenerCondition extends ListenerCondition {
    constructor(name, values) {
        super();
        this.name = name;
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'http-header',
            httpHeaderConfig: {
                httpHeaderName: this.name,
                values: this.values,
            },
        };
    }
}
/**
 * HTTP reqeust method config of the listener rule condition
 */
class HttpRequestMethodListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'http-request-method',
            httpRequestMethodConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * Path pattern config of the listener rule condition
 */
class PathPatternListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
        if (values && values.length > 5) {
            throw new Error("A rule can only have '5' condition values");
        }
    }
    renderRawCondition() {
        return {
            field: 'path-pattern',
            pathPatternConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * Query string config of the listener rule condition
 */
class QueryStringListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'query-string',
            queryStringConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * Source ip config of the listener rule condition
 */
class SourceIpListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'source-ip',
            sourceIpConfig: {
                values: this.values,
            },
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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