"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseListener = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const util_1 = require("./util");
/**
 * Base class for listeners.
 */
class BaseListener extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, additionalProps) {
        super(scope, id);
        const resource = new elasticloadbalancingv2_generated_1.CfnListener(this, 'Resource', {
            ...additionalProps,
            defaultActions: core_1.Lazy.any({ produce: () => this.defaultAction ? this.defaultAction.renderActions() : [] }),
        });
        this.listenerArn = resource.ref;
    }
    /**
     * Queries the load balancer listener context provider for load balancer
     * listener info.
     * @internal
     */
    static _queryContextProvider(scope, options) {
        var _a;
        if (core_1.Token.isUnresolved(options.userOptions.loadBalancerArn)
            || Object.values((_a = options.userOptions.loadBalancerTags) !== null && _a !== void 0 ? _a : {}).some(core_1.Token.isUnresolved)
            || core_1.Token.isUnresolved(options.userOptions.listenerPort)) {
            throw new Error('All arguments to look up a load balancer listener must be concrete (no Tokens)');
        }
        let cxschemaTags;
        if (options.userOptions.loadBalancerTags) {
            cxschemaTags = util_1.mapTagMapToCxschema(options.userOptions.loadBalancerTags);
        }
        const props = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.LOAD_BALANCER_LISTENER_PROVIDER,
            props: {
                listenerArn: options.listenerArn,
                listenerPort: options.userOptions.listenerPort,
                listenerProtocol: options.listenerProtocol,
                loadBalancerArn: options.userOptions.loadBalancerArn,
                loadBalancerTags: cxschemaTags,
                loadBalancerType: options.loadBalancerType,
            },
            dummyValue: {
                listenerArn: `arn:aws:elasticloadbalancing:us-west-2:123456789012:listener/${options.loadBalancerType}/my-load-balancer/50dc6c495c0c9188/f2f7dc8efc522ab2`,
                listenerPort: 80,
                securityGroupIds: ['sg-123456789012'],
            },
        }).value;
        return props;
    }
    /**
     * Validate this listener.
     */
    validate() {
        if (!this.defaultAction) {
            return ['Listener needs at least one default action or target group (call addTargetGroups or addAction)'];
        }
        return [];
    }
    /**
     * Configure the default action
     *
     * @internal
     */
    _setDefaultAction(action) {
        // It might make sense to 'throw' here.
        //
        // However, programs may already exist out there which configured an action twice,
        // in which case the second action accidentally overwrite the initial action, and in some
        // way ended up with a program that did what the author intended. If we were to add throw now,
        // the previously working program would be broken.
        //
        // Instead, signal this through a warning.
        // @deprecate: upon the next major version bump, replace this with a `throw`
        if (this.defaultAction) {
            core_1.Annotations.of(this).addWarning('A default Action already existed on this Listener and was replaced. Configure exactly one default Action.');
        }
        this.defaultAction = action;
    }
}
exports.BaseListener = BaseListener;
//# sourceMappingURL=data:application/json;base64,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