"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseLoadBalancer = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const region_info_1 = require("@aws-cdk/region-info");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const util_1 = require("./util");
/**
 * Base class for both Application and Network Load Balancers.
 */
class BaseLoadBalancer extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id, {
            physicalName: baseProps.loadBalancerName,
        });
        /**
         * Attributes set on this load balancer
         */
        this.attributes = {};
        const internetFacing = util_1.ifUndefined(baseProps.internetFacing, false);
        const vpcSubnets = util_1.ifUndefined(baseProps.vpcSubnets, (internetFacing ? { subnetType: ec2.SubnetType.PUBLIC } : {}));
        const { subnetIds, internetConnectivityEstablished } = baseProps.vpc.selectSubnets(vpcSubnets);
        this.vpc = baseProps.vpc;
        const resource = new elasticloadbalancingv2_generated_1.CfnLoadBalancer(this, 'Resource', {
            name: this.physicalName,
            subnets: subnetIds,
            scheme: internetFacing ? 'internet-facing' : 'internal',
            loadBalancerAttributes: core_1.Lazy.any({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            ...additionalProps,
        });
        if (internetFacing) {
            resource.node.addDependency(internetConnectivityEstablished);
        }
        this.setAttribute('deletion_protection.enabled', baseProps.deletionProtection ? 'true' : 'false');
        this.loadBalancerCanonicalHostedZoneId = resource.attrCanonicalHostedZoneId;
        this.loadBalancerDnsName = resource.attrDnsName;
        this.loadBalancerFullName = resource.attrLoadBalancerFullName;
        this.loadBalancerName = resource.attrLoadBalancerName;
        this.loadBalancerArn = resource.ref;
        this.loadBalancerSecurityGroups = resource.attrSecurityGroups;
    }
    /**
     * Queries the load balancer context provider for load balancer info.
     * @internal
     */
    static _queryContextProvider(scope, options) {
        var _a;
        if (core_1.Token.isUnresolved(options.userOptions.loadBalancerArn)
            || Object.values((_a = options.userOptions.loadBalancerTags) !== null && _a !== void 0 ? _a : {}).some(core_1.Token.isUnresolved)) {
            throw new Error('All arguments to look up a load balancer must be concrete (no Tokens)');
        }
        let cxschemaTags;
        if (options.userOptions.loadBalancerTags) {
            cxschemaTags = util_1.mapTagMapToCxschema(options.userOptions.loadBalancerTags);
        }
        const props = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.LOAD_BALANCER_PROVIDER,
            props: {
                loadBalancerArn: options.userOptions.loadBalancerArn,
                loadBalancerTags: cxschemaTags,
                loadBalancerType: options.loadBalancerType,
            },
            dummyValue: {
                ipAddressType: cxapi.LoadBalancerIpAddressType.DUAL_STACK,
                loadBalancerArn: `arn:aws:elasticloadbalancing:us-west-2:123456789012:loadbalancer/${options.loadBalancerType}/my-load-balancer/50dc6c495c0c9188`,
                loadBalancerCanonicalHostedZoneId: 'Z3DZXE0EXAMPLE',
                loadBalancerDnsName: 'my-load-balancer-1234567890.us-west-2.elb.amazonaws.com',
                securityGroupIds: ['sg-1234'],
                vpcId: 'vpc-12345',
            },
        }).value;
        return props;
    }
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     */
    logAccessLogs(bucket, prefix) {
        this.setAttribute('access_logs.s3.enabled', 'true');
        this.setAttribute('access_logs.s3.bucket', bucket.bucketName.toString());
        this.setAttribute('access_logs.s3.prefix', prefix);
        const region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Region is required to enable ELBv2 access logging');
        }
        const account = region_info_1.RegionInfo.get(region).elbv2Account;
        if (!account) {
            throw new Error(`Cannot enable access logging; don't know ELBv2 account for region ${region}`);
        }
        prefix = prefix || '';
        bucket.grantPut(new iam.AccountPrincipal(account), `${(prefix ? prefix + '/' : '')}AWSLogs/${core_1.Stack.of(this).account}/*`);
        // make sure the bucket's policy is created before the ALB (see https://github.com/aws/aws-cdk/issues/1633)
        this.node.addDependency(bucket);
    }
    /**
     * Set a non-standard attribute on the load balancer.
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/application-load-balancers.html#load-balancer-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Remove an attribute from the load balancer.
     */
    removeAttribute(key) {
        this.setAttribute(key, undefined);
    }
}
exports.BaseLoadBalancer = BaseLoadBalancer;
//# sourceMappingURL=data:application/json;base64,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