"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadBalancerNameFromListenerArn = exports.TargetGroupBase = void 0;
const cdk = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const enums_1 = require("./enums");
const util_1 = require("./util");
/**
 * Define the target of a load balancer.
 */
class TargetGroupBase extends cdk.Construct {
    /**
     *
     */
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id);
        /**
         * Configurable dependable with all resources that lead to load balancer attachment.
         */
        this.loadBalancerAttachedDependencies = new cdk.ConcreteDependable();
        /**
         * Attributes of this target group
         */
        this.attributes = {};
        /**
         * The JSON objects returned by the directly registered members of this target group
         */
        this.targetsJson = new Array();
        if (baseProps.deregistrationDelay !== undefined) {
            this.setAttribute('deregistration_delay.timeout_seconds', baseProps.deregistrationDelay.toSeconds().toString());
        }
        this.healthCheck = baseProps.healthCheck || {};
        this.vpc = baseProps.vpc;
        this.targetType = baseProps.targetType;
        this.resource = new elasticloadbalancingv2_generated_1.CfnTargetGroup(this, 'Resource', {
            name: baseProps.targetGroupName,
            targetGroupAttributes: cdk.Lazy.any({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            targetType: cdk.Lazy.string({ produce: () => this.targetType }),
            targets: cdk.Lazy.any({ produce: () => this.targetsJson }, { omitEmptyArray: true }),
            vpcId: cdk.Lazy.string({ produce: () => this.vpc && this.targetType !== enums_1.TargetType.LAMBDA ? this.vpc.vpcId : undefined }),
            // HEALTH CHECK
            healthCheckEnabled: cdk.Lazy.any({ produce: () => this.healthCheck && this.healthCheck.enabled }),
            healthCheckIntervalSeconds: cdk.Lazy.number({
                produce: () => this.healthCheck && this.healthCheck.interval && this.healthCheck.interval.toSeconds(),
            }),
            healthCheckPath: cdk.Lazy.string({ produce: () => this.healthCheck && this.healthCheck.path }),
            healthCheckPort: cdk.Lazy.string({ produce: () => this.healthCheck && this.healthCheck.port }),
            healthCheckProtocol: cdk.Lazy.string({ produce: () => this.healthCheck && this.healthCheck.protocol }),
            healthCheckTimeoutSeconds: cdk.Lazy.number({
                produce: () => this.healthCheck && this.healthCheck.timeout && this.healthCheck.timeout.toSeconds(),
            }),
            healthyThresholdCount: cdk.Lazy.number({ produce: () => this.healthCheck && this.healthCheck.healthyThresholdCount }),
            unhealthyThresholdCount: cdk.Lazy.number({ produce: () => this.healthCheck && this.healthCheck.unhealthyThresholdCount }),
            matcher: cdk.Lazy.any({
                produce: () => this.healthCheck && this.healthCheck.healthyHttpCodes !== undefined ? {
                    httpCode: this.healthCheck.healthyHttpCodes,
                } : undefined,
            }),
            ...additionalProps,
        });
        this.targetGroupLoadBalancerArns = this.resource.attrLoadBalancerArns;
        this.targetGroupArn = this.resource.ref;
        this.targetGroupFullName = this.resource.attrTargetGroupFullName;
        this.loadBalancerArns = this.resource.attrLoadBalancerArns.toString();
        this.targetGroupName = this.resource.attrTargetGroupName;
        this.defaultPort = additionalProps.port;
    }
    /**
     * List of constructs that need to be depended on to ensure the TargetGroup is associated to a load balancer.
     */
    get loadBalancerAttached() {
        return this.loadBalancerAttachedDependencies;
    }
    /**
     * Set/replace the target group's health check.
     */
    configureHealthCheck(healthCheck) {
        this.healthCheck = healthCheck;
    }
    /**
     * Set a non-standard attribute on the target group.
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-target-groups.html#target-group-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Register the given load balancing target as part of this group.
     */
    addLoadBalancerTarget(props) {
        if (this.targetType !== undefined && this.targetType !== props.targetType) {
            throw new Error(`Already have a of type '${this.targetType}', adding '${props.targetType}'; make all targets the same type.`);
        }
        this.targetType = props.targetType;
        if (this.targetType === enums_1.TargetType.LAMBDA && this.targetsJson.length >= 1) {
            throw new Error('TargetGroup can only contain one LAMBDA target. Create a new TargetGroup.');
        }
        if (props.targetJson) {
            this.targetsJson.push(props.targetJson);
        }
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     */
    validate() {
        const ret = super.validate();
        if (this.targetType === undefined && this.targetsJson.length === 0) {
            cdk.Annotations.of(this).addWarning("When creating an empty TargetGroup, you should specify a 'targetType' (this warning may become an error in the future).");
        }
        if (this.targetType !== enums_1.TargetType.LAMBDA && this.vpc === undefined) {
            ret.push("'vpc' is required for a non-Lambda TargetGroup");
        }
        return ret;
    }
}
exports.TargetGroupBase = TargetGroupBase;
/**
 * Extract the full load balancer name (used for metrics) from the listener ARN:
 *
 * Turns
 *
 *     arn:aws:elasticloadbalancing:us-west-2:123456789012:listener/app/my-load-balancer/50dc6c495c0c9188/f2f7dc8efc522ab2
 *
 * Into
 *
 *     app/my-load-balancer/50dc6c495c0c9188
 */
function loadBalancerNameFromListenerArn(listenerArn) {
    const arnParts = cdk.Fn.split('/', listenerArn);
    return `${cdk.Fn.select(1, arnParts)}/${cdk.Fn.select(2, arnParts)}/${cdk.Fn.select(3, arnParts)}`;
}
exports.loadBalancerNameFromListenerArn = loadBalancerNameFromListenerArn;
//# sourceMappingURL=data:application/json;base64,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