"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const https = require("https");
const URL = require("url");
/**
 * Custom CloudFormation resource helper
 */
class CustomResource {
    constructor(context, callback, logger) {
        /**
         * Stores functions executed when resource creation is requested
         */
        this.createFunctions = [];
        /**
         * Stores functions executed when resource update is requested
         */
        this.updateFunctions = [];
        /**
         * Stores functions executed when resource deletion is requested
         */
        this.deleteFunctions = [];
        /**
         * Stores values returned to CloudFormation
         */
        this.ResponseData = {};
        this.context = context;
        this.callback = callback;
        this.logger = logger || new StandardLogger();
    }
    /**
     * Adds a function to the CREATE queue
     */
    onCreate(func) {
        this.createFunctions.push(func);
        return this;
    }
    /**
     * Adds a function to the UPDATE queue
     */
    onUpdate(func) {
        this.updateFunctions.push(func);
        return this;
    }
    /**
     * Adds a function to the DELETE queue
     */
    onDelete(func) {
        this.deleteFunctions.push(func);
        return this;
    }
    /**
     * Handles the Lambda event
     */
    handle(event) {
        const lambdaEvent = event;
        const self = this;
        if (typeof lambdaEvent.ResponseURL === 'undefined') {
            throw new Error('ResponseURL missing');
        }
        this.logger.debug(`REQUEST RECEIVED:\n${JSON.stringify(lambdaEvent)}`);
        this.timeout(lambdaEvent);
        event.addResponseValue = (key, value) => {
            self.ResponseData[key] = value;
        };
        event.setPhysicalResourceId = (value) => {
            self.PhysicalResourceId = value;
        };
        try {
            let queue;
            if (lambdaEvent.RequestType == 'Create')
                queue = this.createFunctions;
            else if (lambdaEvent.RequestType == 'Update')
                queue = this.updateFunctions;
            else if (lambdaEvent.RequestType == 'Delete')
                queue = this.deleteFunctions;
            else {
                this.sendResponse(lambdaEvent, 'FAILED', `Unexpected request type: ${lambdaEvent.RequestType}`);
                return this;
            }
            let result = queue.reduce((current, next) => {
                return current.then((value) => {
                    return next(value);
                });
            }, Promise.resolve(event));
            result
                .then(function (event) {
                self.logger.debug(event);
                self.sendResponse(lambdaEvent, 'SUCCESS', `${lambdaEvent.RequestType} completed successfully`);
            })
                .catch(function (err) {
                self.logger.error(err, err.stack);
                self.sendResponse(lambdaEvent, 'FAILED', err.message || err.code);
            });
        }
        catch (err) {
            this.sendResponse(lambdaEvent, 'FAILED', err.message || err.code);
        }
        return this;
    }
    /**
     * Sends CloudFormation response just before the Lambda times out
     */
    timeout(event) {
        const self = this;
        const handler = () => {
            self.logger.error('Timeout FAILURE!');
            new Promise(() => self.sendResponse(event, 'FAILED', 'Function timed out')).then(() => self.callback(new Error('Function timed out')));
        };
        setTimeout(handler, this.context.getRemainingTimeInMillis() - 1000);
    }
    /**
     * Sends CloudFormation response
     */
    sendResponse(event, responseStatus, responseData) {
        const self = this;
        this.logger.debug(`Sending response ${responseStatus}:\n${JSON.stringify(responseData)}`);
        const data = this.ResponseData;
        data['Message'] = responseData;
        const body = JSON.stringify({
            Status: responseStatus,
            Reason: `${responseData} | Full error in CloudWatch ${this.context.logStreamName}`,
            PhysicalResourceId: self.PhysicalResourceId || event.ResourceProperties.Name,
            StackId: event.StackId,
            RequestId: event.RequestId,
            LogicalResourceId: event.LogicalResourceId,
            Data: data,
        });
        this.logger.debug('RESPONSE BODY:\n', body);
        const url = URL.parse(event.ResponseURL);
        const options = {
            hostname: url.hostname,
            port: 443,
            path: url.path,
            method: 'PUT',
            headers: {
                'content-type': '',
                'content-length': body.length,
            },
        };
        this.logger.info('SENDING RESPONSE...\n');
        const request = https.request(options, function (response) {
            self.logger.debug(`STATUS: ${response.statusCode}`);
            self.logger.debug(`HEADERS: ${JSON.stringify(response.headers)}`);
            self.context.done();
        });
        request.on('error', function (error) {
            self.logger.error(`sendResponse Error: ${error}`);
            self.context.done();
        });
        request.write(body);
        request.end();
    }
}
exports.CustomResource = CustomResource;
/**
 * Standard logger class
 */
class StandardLogger {
    constructor(level) {
        this.level = level || 1 /* WARN */;
    }
    /**
     * Logs message with level ERROR
     */
    error(message, ...optionalParams) {
        if (this.level < 0 /* ERROR */)
            return;
        console.error(message, ...optionalParams);
    }
    /**
     * Logs message with level WARN
     */
    warn(message, ...optionalParams) {
        if (this.level < 1 /* WARN */)
            return;
        console.warn(message, ...optionalParams);
    }
    /**
     * Logs message with level INFO
     */
    info(message, ...optionalParams) {
        if (this.level < 2 /* INFO */)
            return;
        console.info(message, ...optionalParams);
    }
    /**
     * Logs message with level DEBUG
     */
    debug(message, ...optionalParams) {
        if (this.level < 3 /* DEBUG */)
            return;
        console.debug(message, ...optionalParams);
    }
    /**
     * Alias for info
     */
    log(message, ...optionalParams) {
        this.info(message, ...optionalParams);
    }
}
exports.StandardLogger = StandardLogger;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9pbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUVBLCtCQUFnQztBQUNoQywyQkFBNEI7QUE4QjVCOztHQUVHO0FBQ0gsTUFBYSxjQUFjO0lBMkN6QixZQUFZLE9BQWdCLEVBQUUsUUFBa0IsRUFBRSxNQUFlO1FBMUNqRTs7V0FFRztRQUNILG9CQUFlLEdBQVcsRUFBRSxDQUFDO1FBRTdCOztXQUVHO1FBQ0gsb0JBQWUsR0FBVyxFQUFFLENBQUM7UUFFN0I7O1dBRUc7UUFDSCxvQkFBZSxHQUFXLEVBQUUsQ0FBQztRQVk3Qjs7V0FFRztRQUNILGlCQUFZLEdBRVIsRUFBRSxDQUFDO1FBYUwsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7UUFDdkIsSUFBSSxDQUFDLFFBQVEsR0FBRyxRQUFRLENBQUM7UUFDekIsSUFBSSxDQUFDLE1BQU0sR0FBRyxNQUFNLElBQUksSUFBSSxjQUFjLEVBQUUsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7O09BRUc7SUFDSCxRQUFRLENBQUMsSUFBVTtRQUNqQixJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7T0FFRztJQUNILFFBQVEsQ0FBQyxJQUFVO1FBQ2pCLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOztPQUVHO0lBQ0gsUUFBUSxDQUFDLElBQVU7UUFDakIsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7O09BRUc7SUFDSCxNQUFNLENBQUMsS0FBa0I7UUFDdkIsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDO1FBQzFCLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQztRQUVsQixJQUFJLE9BQU8sV0FBVyxDQUFDLFdBQVcsS0FBSyxXQUFXLEVBQUU7WUFDbEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1NBQ3hDO1FBRUQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsc0JBQXNCLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ3ZFLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLENBQUM7UUFFMUIsS0FBSyxDQUFDLGdCQUFnQixHQUFHLENBQUMsR0FBVyxFQUFFLEtBQVUsRUFBRSxFQUFFO1lBQ25ELElBQUksQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO1FBQ2pDLENBQUMsQ0FBQztRQUVGLEtBQUssQ0FBQyxxQkFBcUIsR0FBRyxDQUFDLEtBQWEsRUFBRSxFQUFFO1lBQzlDLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxLQUFLLENBQUM7UUFDbEMsQ0FBQyxDQUFDO1FBRUYsSUFBSTtZQUNGLElBQUksS0FBYSxDQUFDO1lBQ2xCLElBQUksV0FBVyxDQUFDLFdBQVcsSUFBSSxRQUFRO2dCQUFFLEtBQUssR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDO2lCQUNqRSxJQUFJLFdBQVcsQ0FBQyxXQUFXLElBQUksUUFBUTtnQkFDMUMsS0FBSyxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUM7aUJBQzFCLElBQUksV0FBVyxDQUFDLFdBQVcsSUFBSSxRQUFRO2dCQUMxQyxLQUFLLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQztpQkFDMUI7Z0JBQ0gsSUFBSSxDQUFDLFlBQVksQ0FDZixXQUFXLEVBQ1gsUUFBUSxFQUNSLDRCQUE0QixXQUFXLENBQUMsV0FBVyxFQUFFLENBQ3RELENBQUM7Z0JBQ0YsT0FBTyxJQUFJLENBQUM7YUFDYjtZQUVELElBQUksTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQ3ZCLENBQUMsT0FBNkMsRUFBRSxJQUFVLEVBQUUsRUFBRTtnQkFDNUQsT0FBUSxPQUEwQixDQUFDLElBQUksQ0FBQyxDQUFDLEtBQVksRUFBRSxFQUFFO29CQUN2RCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDckIsQ0FBQyxDQUFDLENBQUM7WUFDTCxDQUFDLEVBQ0QsT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFjLENBQUMsQ0FDaEMsQ0FBQztZQUVGLE1BQU07aUJBQ0gsSUFBSSxDQUFDLFVBQVUsS0FBMkI7Z0JBQ3pDLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUN6QixJQUFJLENBQUMsWUFBWSxDQUNmLFdBQVcsRUFDWCxTQUFTLEVBQ1QsR0FBRyxXQUFXLENBQUMsV0FBVyx5QkFBeUIsQ0FDcEQsQ0FBQztZQUNKLENBQUMsQ0FBQztpQkFDRCxLQUFLLENBQUMsVUFBVSxHQUFpQjtnQkFDaEMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDbEMsSUFBSSxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUUsUUFBUSxFQUFFLEdBQUcsQ0FBQyxPQUFPLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3BFLENBQUMsQ0FBQyxDQUFDO1NBQ047UUFBQyxPQUFPLEdBQUcsRUFBRTtZQUNaLElBQUksQ0FBQyxZQUFZLENBQUMsV0FBVyxFQUFFLFFBQVEsRUFBRSxHQUFHLENBQUMsT0FBTyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNuRTtRQUNELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOztPQUVHO0lBQ0gsT0FBTyxDQUFDLEtBQWtCO1FBQ3hCLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQztRQUNsQixNQUFNLE9BQU8sR0FBRyxHQUFHLEVBQUU7WUFDbkIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUMsQ0FBQztZQUN0QyxJQUFJLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FDZixJQUFJLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsb0JBQW9CLENBQUMsQ0FDekQsQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUMvRCxDQUFDLENBQUM7UUFDRixVQUFVLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsd0JBQXdCLEVBQUUsR0FBRyxJQUFJLENBQUMsQ0FBQztJQUN0RSxDQUFDO0lBRUQ7O09BRUc7SUFDSCxZQUFZLENBQ1YsS0FBa0IsRUFDbEIsY0FBc0IsRUFDdEIsWUFBb0I7UUFFcEIsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2xCLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUNmLG9CQUFvQixjQUFjLE1BQU0sSUFBSSxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUN2RSxDQUFDO1FBRUYsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQztRQUMvQixJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsWUFBWSxDQUFDO1FBRS9CLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDMUIsTUFBTSxFQUFFLGNBQWM7WUFDdEIsTUFBTSxFQUFFLEdBQUcsWUFBWSwrQkFBK0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLEVBQUU7WUFDbEYsa0JBQWtCLEVBQ2hCLElBQUksQ0FBQyxrQkFBa0IsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsSUFBSTtZQUMxRCxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87WUFDdEIsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQzFCLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUI7WUFDMUMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUU1QyxNQUFNLEdBQUcsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUN6QyxNQUFNLE9BQU8sR0FBRztZQUNkLFFBQVEsRUFBRSxHQUFHLENBQUMsUUFBUTtZQUN0QixJQUFJLEVBQUUsR0FBRztZQUNULElBQUksRUFBRSxHQUFHLENBQUMsSUFBSTtZQUNkLE1BQU0sRUFBRSxLQUFLO1lBQ2IsT0FBTyxFQUFFO2dCQUNQLGNBQWMsRUFBRSxFQUFFO2dCQUNsQixnQkFBZ0IsRUFBRSxJQUFJLENBQUMsTUFBTTthQUM5QjtTQUNGLENBQUM7UUFFRixJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBRTFDLE1BQU0sT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLFVBQVUsUUFBYTtZQUM1RCxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxXQUFXLFFBQVEsQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDO1lBQ3BELElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLFlBQVksSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQ2xFLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDdEIsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxVQUFVLEtBQVk7WUFDeEMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsdUJBQXVCLEtBQUssRUFBRSxDQUFDLENBQUM7WUFDbEQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUN0QixDQUFDLENBQUMsQ0FBQztRQUVILE9BQU8sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDcEIsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO0lBQ2hCLENBQUM7Q0FDRjtBQWxORCx3Q0FrTkM7QUF1QkQ7O0dBRUc7QUFDSCxNQUFhLGNBQWM7SUFRekIsWUFBWSxLQUFnQjtRQUMxQixJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssZ0JBQWlCLENBQUM7SUFDdEMsQ0FBQztJQUVEOztPQUVHO0lBQ0gsS0FBSyxDQUFDLE9BQVksRUFBRSxHQUFHLGNBQXFCO1FBQzFDLElBQUksSUFBSSxDQUFDLEtBQUssZ0JBQWlCO1lBQUUsT0FBTztRQUN4QyxPQUFPLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxHQUFHLGNBQWMsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7T0FFRztJQUNILElBQUksQ0FBQyxPQUFZLEVBQUUsR0FBRyxjQUFxQjtRQUN6QyxJQUFJLElBQUksQ0FBQyxLQUFLLGVBQWdCO1lBQUUsT0FBTztRQUN2QyxPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxHQUFHLGNBQWMsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7T0FFRztJQUNILElBQUksQ0FBQyxPQUFZLEVBQUUsR0FBRyxjQUFxQjtRQUN6QyxJQUFJLElBQUksQ0FBQyxLQUFLLGVBQWdCO1lBQUUsT0FBTztRQUN2QyxPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxHQUFHLGNBQWMsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7T0FFRztJQUNILEtBQUssQ0FBQyxPQUFZLEVBQUUsR0FBRyxjQUFxQjtRQUMxQyxJQUFJLElBQUksQ0FBQyxLQUFLLGdCQUFpQjtZQUFFLE9BQU87UUFDeEMsT0FBTyxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsR0FBRyxjQUFjLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7O09BRUc7SUFDSCxHQUFHLENBQUMsT0FBWSxFQUFFLEdBQUcsY0FBcUI7UUFDeEMsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsR0FBRyxjQUFjLENBQUMsQ0FBQztJQUN4QyxDQUFDO0NBQ0Y7QUFsREQsd0NBa0RDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ2FsbGJhY2ssIENvbnRleHQgfSBmcm9tICdhd3MtbGFtYmRhJztcbmltcG9ydCBBV1MgPSByZXF1aXJlKCdhd3Mtc2RrJyk7XG5pbXBvcnQgaHR0cHMgPSByZXF1aXJlKCdodHRwcycpO1xuaW1wb3J0IFVSTCA9IHJlcXVpcmUoJ3VybCcpO1xuXG4vKipcbiAqIFRoZSBldmVudCBwYXNzZWQgdG8gdGhlIExhbWJkYSBoYW5kbGVyXG4gKi9cblxuZXhwb3J0IGludGVyZmFjZSBMYW1iZGFFdmVudCB7XG4gIFtrZXk6IHN0cmluZ106IGFueTtcbn1cblxuLyoqXG4gKiBUaGUgZXZlbnQgcGFzc2VkIHRocm91Z2ggdGhlIHByb21pc2VzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEV2ZW50IGV4dGVuZHMgTGFtYmRhRXZlbnQge1xuICAvKipcbiAgICogQWRkcyB2YWx1ZXMgdG8gdGhlIHJlc3BvbnNlIHJldHVybmVkIHRvIENsb3VkRm9ybWF0aW9uXG4gICAqL1xuICBhZGRSZXNwb25zZVZhbHVlOiAoa2V5OiBzdHJpbmcsIHZhbHVlOiBhbnkpID0+IHZvaWQ7XG5cbiAgLyoqXG4gICAqIFNldCB0aGUgcGh5c2ljYWwgSUQgb2YgdGhlIHJlc291cmNlXG4gICAqL1xuICBzZXRQaHlzaWNhbFJlc291cmNlSWQ6ICh2YWx1ZTogYW55KSA9PiB2b2lkO1xufVxuXG4vKipcbiAqIEZ1bmN0aW9uIHNpZ25hdHVyZVxuICovXG5leHBvcnQgdHlwZSBmdW5jID0gKGV2ZW50OiBFdmVudCkgPT4gUHJvbWlzZTxFdmVudCB8IEFXUy5BV1NFcnJvcj47XG5cbi8qKlxuICogQ3VzdG9tIENsb3VkRm9ybWF0aW9uIHJlc291cmNlIGhlbHBlclxuICovXG5leHBvcnQgY2xhc3MgQ3VzdG9tUmVzb3VyY2Uge1xuICAvKipcbiAgICogU3RvcmVzIGZ1bmN0aW9ucyBleGVjdXRlZCB3aGVuIHJlc291cmNlIGNyZWF0aW9uIGlzIHJlcXVlc3RlZFxuICAgKi9cbiAgY3JlYXRlRnVuY3Rpb25zOiBmdW5jW10gPSBbXTtcblxuICAvKipcbiAgICogU3RvcmVzIGZ1bmN0aW9ucyBleGVjdXRlZCB3aGVuIHJlc291cmNlIHVwZGF0ZSBpcyByZXF1ZXN0ZWRcbiAgICovXG4gIHVwZGF0ZUZ1bmN0aW9uczogZnVuY1tdID0gW107XG5cbiAgLyoqXG4gICAqIFN0b3JlcyBmdW5jdGlvbnMgZXhlY3V0ZWQgd2hlbiByZXNvdXJjZSBkZWxldGlvbiBpcyByZXF1ZXN0ZWRcbiAgICovXG4gIGRlbGV0ZUZ1bmN0aW9uczogZnVuY1tdID0gW107XG5cbiAgLyoqXG4gICAqIFRoZSBjb250ZXh0IHBhc3NlZCB0byB0aGUgTGFtYmRhIGhhbmRsZXJcbiAgICovXG4gIGNvbnRleHQ6IENvbnRleHQ7XG5cbiAgLyoqXG4gICAqIFRoZSBjYWxsYmFjayBmdW5jdGlvbiBwYXNzZWQgdG8gdGhlIExhbWJkYSBoYW5kbGVyXG4gICAqL1xuICBjYWxsYmFjazogQ2FsbGJhY2s7XG5cbiAgLyoqXG4gICAqIFN0b3JlcyB2YWx1ZXMgcmV0dXJuZWQgdG8gQ2xvdWRGb3JtYXRpb25cbiAgICovXG4gIFJlc3BvbnNlRGF0YToge1xuICAgIFtrZXk6IHN0cmluZ106IGFueTtcbiAgfSA9IHt9O1xuXG4gIC8qKlxuICAgKiBTdG9yZXMgdmFsdWVzIHBoeXNpY2FsIElEIG9mIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgUGh5c2ljYWxSZXNvdXJjZUlkPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBMb2dnZXIgY2xhc3NcbiAgICovXG4gIGxvZ2dlcjogTG9nZ2VyO1xuXG4gIGNvbnN0cnVjdG9yKGNvbnRleHQ6IENvbnRleHQsIGNhbGxiYWNrOiBDYWxsYmFjaywgbG9nZ2VyPzogTG9nZ2VyKSB7XG4gICAgdGhpcy5jb250ZXh0ID0gY29udGV4dDtcbiAgICB0aGlzLmNhbGxiYWNrID0gY2FsbGJhY2s7XG4gICAgdGhpcy5sb2dnZXIgPSBsb2dnZXIgfHwgbmV3IFN0YW5kYXJkTG9nZ2VyKCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIGZ1bmN0aW9uIHRvIHRoZSBDUkVBVEUgcXVldWVcbiAgICovXG4gIG9uQ3JlYXRlKGZ1bmM6IGZ1bmMpOiB0aGlzIHtcbiAgICB0aGlzLmNyZWF0ZUZ1bmN0aW9ucy5wdXNoKGZ1bmMpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBmdW5jdGlvbiB0byB0aGUgVVBEQVRFIHF1ZXVlXG4gICAqL1xuICBvblVwZGF0ZShmdW5jOiBmdW5jKTogdGhpcyB7XG4gICAgdGhpcy51cGRhdGVGdW5jdGlvbnMucHVzaChmdW5jKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgZnVuY3Rpb24gdG8gdGhlIERFTEVURSBxdWV1ZVxuICAgKi9cbiAgb25EZWxldGUoZnVuYzogZnVuYyk6IHRoaXMge1xuICAgIHRoaXMuZGVsZXRlRnVuY3Rpb25zLnB1c2goZnVuYyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogSGFuZGxlcyB0aGUgTGFtYmRhIGV2ZW50XG4gICAqL1xuICBoYW5kbGUoZXZlbnQ6IExhbWJkYUV2ZW50KTogdGhpcyB7XG4gICAgY29uc3QgbGFtYmRhRXZlbnQgPSBldmVudDtcbiAgICBjb25zdCBzZWxmID0gdGhpcztcblxuICAgIGlmICh0eXBlb2YgbGFtYmRhRXZlbnQuUmVzcG9uc2VVUkwgPT09ICd1bmRlZmluZWQnKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Jlc3BvbnNlVVJMIG1pc3NpbmcnKTtcbiAgICB9XG5cbiAgICB0aGlzLmxvZ2dlci5kZWJ1ZyhgUkVRVUVTVCBSRUNFSVZFRDpcXG4ke0pTT04uc3RyaW5naWZ5KGxhbWJkYUV2ZW50KX1gKTtcbiAgICB0aGlzLnRpbWVvdXQobGFtYmRhRXZlbnQpO1xuXG4gICAgZXZlbnQuYWRkUmVzcG9uc2VWYWx1ZSA9IChrZXk6IHN0cmluZywgdmFsdWU6IGFueSkgPT4ge1xuICAgICAgc2VsZi5SZXNwb25zZURhdGFba2V5XSA9IHZhbHVlO1xuICAgIH07XG5cbiAgICBldmVudC5zZXRQaHlzaWNhbFJlc291cmNlSWQgPSAodmFsdWU6IHN0cmluZykgPT4ge1xuICAgICAgc2VsZi5QaHlzaWNhbFJlc291cmNlSWQgPSB2YWx1ZTtcbiAgICB9O1xuXG4gICAgdHJ5IHtcbiAgICAgIGxldCBxdWV1ZTogZnVuY1tdO1xuICAgICAgaWYgKGxhbWJkYUV2ZW50LlJlcXVlc3RUeXBlID09ICdDcmVhdGUnKSBxdWV1ZSA9IHRoaXMuY3JlYXRlRnVuY3Rpb25zO1xuICAgICAgZWxzZSBpZiAobGFtYmRhRXZlbnQuUmVxdWVzdFR5cGUgPT0gJ1VwZGF0ZScpXG4gICAgICAgIHF1ZXVlID0gdGhpcy51cGRhdGVGdW5jdGlvbnM7XG4gICAgICBlbHNlIGlmIChsYW1iZGFFdmVudC5SZXF1ZXN0VHlwZSA9PSAnRGVsZXRlJylcbiAgICAgICAgcXVldWUgPSB0aGlzLmRlbGV0ZUZ1bmN0aW9ucztcbiAgICAgIGVsc2Uge1xuICAgICAgICB0aGlzLnNlbmRSZXNwb25zZShcbiAgICAgICAgICBsYW1iZGFFdmVudCxcbiAgICAgICAgICAnRkFJTEVEJyxcbiAgICAgICAgICBgVW5leHBlY3RlZCByZXF1ZXN0IHR5cGU6ICR7bGFtYmRhRXZlbnQuUmVxdWVzdFR5cGV9YFxuICAgICAgICApO1xuICAgICAgICByZXR1cm4gdGhpcztcbiAgICAgIH1cblxuICAgICAgbGV0IHJlc3VsdCA9IHF1ZXVlLnJlZHVjZShcbiAgICAgICAgKGN1cnJlbnQ6IFByb21pc2U8RXZlbnQgfCBBV1MuQVdTRXJyb3I+IHwgZnVuYywgbmV4dDogZnVuYykgPT4ge1xuICAgICAgICAgIHJldHVybiAoY3VycmVudCBhcyBQcm9taXNlPEV2ZW50PikudGhlbigodmFsdWU6IEV2ZW50KSA9PiB7XG4gICAgICAgICAgICByZXR1cm4gbmV4dCh2YWx1ZSk7XG4gICAgICAgICAgfSk7XG4gICAgICAgIH0sXG4gICAgICAgIFByb21pc2UucmVzb2x2ZShldmVudCBhcyBFdmVudClcbiAgICAgICk7XG5cbiAgICAgIHJlc3VsdFxuICAgICAgICAudGhlbihmdW5jdGlvbiAoZXZlbnQ6IEV2ZW50IHwgQVdTLkFXU0Vycm9yKSB7XG4gICAgICAgICAgc2VsZi5sb2dnZXIuZGVidWcoZXZlbnQpO1xuICAgICAgICAgIHNlbGYuc2VuZFJlc3BvbnNlKFxuICAgICAgICAgICAgbGFtYmRhRXZlbnQsXG4gICAgICAgICAgICAnU1VDQ0VTUycsXG4gICAgICAgICAgICBgJHtsYW1iZGFFdmVudC5SZXF1ZXN0VHlwZX0gY29tcGxldGVkIHN1Y2Nlc3NmdWxseWBcbiAgICAgICAgICApO1xuICAgICAgICB9KVxuICAgICAgICAuY2F0Y2goZnVuY3Rpb24gKGVycjogQVdTLkFXU0Vycm9yKSB7XG4gICAgICAgICAgc2VsZi5sb2dnZXIuZXJyb3IoZXJyLCBlcnIuc3RhY2spO1xuICAgICAgICAgIHNlbGYuc2VuZFJlc3BvbnNlKGxhbWJkYUV2ZW50LCAnRkFJTEVEJywgZXJyLm1lc3NhZ2UgfHwgZXJyLmNvZGUpO1xuICAgICAgICB9KTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIHRoaXMuc2VuZFJlc3BvbnNlKGxhbWJkYUV2ZW50LCAnRkFJTEVEJywgZXJyLm1lc3NhZ2UgfHwgZXJyLmNvZGUpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTZW5kcyBDbG91ZEZvcm1hdGlvbiByZXNwb25zZSBqdXN0IGJlZm9yZSB0aGUgTGFtYmRhIHRpbWVzIG91dFxuICAgKi9cbiAgdGltZW91dChldmVudDogTGFtYmRhRXZlbnQpIHtcbiAgICBjb25zdCBzZWxmID0gdGhpcztcbiAgICBjb25zdCBoYW5kbGVyID0gKCkgPT4ge1xuICAgICAgc2VsZi5sb2dnZXIuZXJyb3IoJ1RpbWVvdXQgRkFJTFVSRSEnKTtcbiAgICAgIG5ldyBQcm9taXNlKCgpID0+XG4gICAgICAgIHNlbGYuc2VuZFJlc3BvbnNlKGV2ZW50LCAnRkFJTEVEJywgJ0Z1bmN0aW9uIHRpbWVkIG91dCcpXG4gICAgICApLnRoZW4oKCkgPT4gc2VsZi5jYWxsYmFjayhuZXcgRXJyb3IoJ0Z1bmN0aW9uIHRpbWVkIG91dCcpKSk7XG4gICAgfTtcbiAgICBzZXRUaW1lb3V0KGhhbmRsZXIsIHRoaXMuY29udGV4dC5nZXRSZW1haW5pbmdUaW1lSW5NaWxsaXMoKSAtIDEwMDApO1xuICB9XG5cbiAgLyoqXG4gICAqIFNlbmRzIENsb3VkRm9ybWF0aW9uIHJlc3BvbnNlXG4gICAqL1xuICBzZW5kUmVzcG9uc2UoXG4gICAgZXZlbnQ6IExhbWJkYUV2ZW50LFxuICAgIHJlc3BvbnNlU3RhdHVzOiBzdHJpbmcsXG4gICAgcmVzcG9uc2VEYXRhOiBzdHJpbmdcbiAgKSB7XG4gICAgY29uc3Qgc2VsZiA9IHRoaXM7XG4gICAgdGhpcy5sb2dnZXIuZGVidWcoXG4gICAgICBgU2VuZGluZyByZXNwb25zZSAke3Jlc3BvbnNlU3RhdHVzfTpcXG4ke0pTT04uc3RyaW5naWZ5KHJlc3BvbnNlRGF0YSl9YFxuICAgICk7XG5cbiAgICBjb25zdCBkYXRhID0gdGhpcy5SZXNwb25zZURhdGE7XG4gICAgZGF0YVsnTWVzc2FnZSddID0gcmVzcG9uc2VEYXRhO1xuXG4gICAgY29uc3QgYm9keSA9IEpTT04uc3RyaW5naWZ5KHtcbiAgICAgIFN0YXR1czogcmVzcG9uc2VTdGF0dXMsXG4gICAgICBSZWFzb246IGAke3Jlc3BvbnNlRGF0YX0gfCBGdWxsIGVycm9yIGluIENsb3VkV2F0Y2ggJHt0aGlzLmNvbnRleHQubG9nU3RyZWFtTmFtZX1gLFxuICAgICAgUGh5c2ljYWxSZXNvdXJjZUlkOlxuICAgICAgICBzZWxmLlBoeXNpY2FsUmVzb3VyY2VJZCB8fCBldmVudC5SZXNvdXJjZVByb3BlcnRpZXMuTmFtZSxcbiAgICAgIFN0YWNrSWQ6IGV2ZW50LlN0YWNrSWQsXG4gICAgICBSZXF1ZXN0SWQ6IGV2ZW50LlJlcXVlc3RJZCxcbiAgICAgIExvZ2ljYWxSZXNvdXJjZUlkOiBldmVudC5Mb2dpY2FsUmVzb3VyY2VJZCxcbiAgICAgIERhdGE6IGRhdGEsXG4gICAgfSk7XG5cbiAgICB0aGlzLmxvZ2dlci5kZWJ1ZygnUkVTUE9OU0UgQk9EWTpcXG4nLCBib2R5KTtcblxuICAgIGNvbnN0IHVybCA9IFVSTC5wYXJzZShldmVudC5SZXNwb25zZVVSTCk7XG4gICAgY29uc3Qgb3B0aW9ucyA9IHtcbiAgICAgIGhvc3RuYW1lOiB1cmwuaG9zdG5hbWUsXG4gICAgICBwb3J0OiA0NDMsXG4gICAgICBwYXRoOiB1cmwucGF0aCxcbiAgICAgIG1ldGhvZDogJ1BVVCcsXG4gICAgICBoZWFkZXJzOiB7XG4gICAgICAgICdjb250ZW50LXR5cGUnOiAnJyxcbiAgICAgICAgJ2NvbnRlbnQtbGVuZ3RoJzogYm9keS5sZW5ndGgsXG4gICAgICB9LFxuICAgIH07XG5cbiAgICB0aGlzLmxvZ2dlci5pbmZvKCdTRU5ESU5HIFJFU1BPTlNFLi4uXFxuJyk7XG5cbiAgICBjb25zdCByZXF1ZXN0ID0gaHR0cHMucmVxdWVzdChvcHRpb25zLCBmdW5jdGlvbiAocmVzcG9uc2U6IGFueSkge1xuICAgICAgc2VsZi5sb2dnZXIuZGVidWcoYFNUQVRVUzogJHtyZXNwb25zZS5zdGF0dXNDb2RlfWApO1xuICAgICAgc2VsZi5sb2dnZXIuZGVidWcoYEhFQURFUlM6ICR7SlNPTi5zdHJpbmdpZnkocmVzcG9uc2UuaGVhZGVycyl9YCk7XG4gICAgICBzZWxmLmNvbnRleHQuZG9uZSgpO1xuICAgIH0pO1xuXG4gICAgcmVxdWVzdC5vbignZXJyb3InLCBmdW5jdGlvbiAoZXJyb3I6IEVycm9yKSB7XG4gICAgICBzZWxmLmxvZ2dlci5lcnJvcihgc2VuZFJlc3BvbnNlIEVycm9yOiAke2Vycm9yfWApO1xuICAgICAgc2VsZi5jb250ZXh0LmRvbmUoKTtcbiAgICB9KTtcblxuICAgIHJlcXVlc3Qud3JpdGUoYm9keSk7XG4gICAgcmVxdWVzdC5lbmQoKTtcbiAgfVxufVxuXG4vKipcbiAqIExvZ2dlciBjbGFzc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIExvZ2dlciB7XG4gIGxvZyhtZXNzYWdlOiBhbnksIC4uLm9wdGlvbmFsUGFyYW1zOiBhbnlbXSk6IHZvaWQ7XG4gIGluZm8obWVzc2FnZTogYW55LCAuLi5vcHRpb25hbFBhcmFtczogYW55W10pOiB2b2lkO1xuICBkZWJ1ZyhtZXNzYWdlOiBhbnksIC4uLm9wdGlvbmFsUGFyYW1zOiBhbnlbXSk6IHZvaWQ7XG4gIHdhcm4obWVzc2FnZTogYW55LCAuLi5vcHRpb25hbFBhcmFtczogYW55W10pOiB2b2lkO1xuICBlcnJvcihtZXNzYWdlOiBhbnksIC4uLm9wdGlvbmFsUGFyYW1zOiBhbnlbXSk6IHZvaWQ7XG59XG5cbi8qKlxuICogTG9nTGV2ZWxzIHN1cHBvcnRlZCBieSB0aGUgbG9nZ2VyXG4gKi9cbmV4cG9ydCBjb25zdCBlbnVtIExvZ0xldmVsIHtcbiAgRVJST1IsXG4gIFdBUk4sXG4gIElORk8sXG4gIERFQlVHLFxufVxuXG4vKipcbiAqIFN0YW5kYXJkIGxvZ2dlciBjbGFzc1xuICovXG5leHBvcnQgY2xhc3MgU3RhbmRhcmRMb2dnZXIge1xuICAvKipcbiAgICogVGhlIGxvZyBsZXZlbFxuICAgKlxuICAgKiBAZGVmYXVsdCBMb2dMZXZlbC5XQVJOXG4gICAqL1xuICBsZXZlbDogTG9nTGV2ZWw7XG5cbiAgY29uc3RydWN0b3IobGV2ZWw/OiBMb2dMZXZlbCkge1xuICAgIHRoaXMubGV2ZWwgPSBsZXZlbCB8fCBMb2dMZXZlbC5XQVJOO1xuICB9XG5cbiAgLyoqXG4gICAqIExvZ3MgbWVzc2FnZSB3aXRoIGxldmVsIEVSUk9SXG4gICAqL1xuICBlcnJvcihtZXNzYWdlOiBhbnksIC4uLm9wdGlvbmFsUGFyYW1zOiBhbnlbXSkge1xuICAgIGlmICh0aGlzLmxldmVsIDwgTG9nTGV2ZWwuRVJST1IpIHJldHVybjtcbiAgICBjb25zb2xlLmVycm9yKG1lc3NhZ2UsIC4uLm9wdGlvbmFsUGFyYW1zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBMb2dzIG1lc3NhZ2Ugd2l0aCBsZXZlbCBXQVJOXG4gICAqL1xuICB3YXJuKG1lc3NhZ2U6IGFueSwgLi4ub3B0aW9uYWxQYXJhbXM6IGFueVtdKSB7XG4gICAgaWYgKHRoaXMubGV2ZWwgPCBMb2dMZXZlbC5XQVJOKSByZXR1cm47XG4gICAgY29uc29sZS53YXJuKG1lc3NhZ2UsIC4uLm9wdGlvbmFsUGFyYW1zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBMb2dzIG1lc3NhZ2Ugd2l0aCBsZXZlbCBJTkZPXG4gICAqL1xuICBpbmZvKG1lc3NhZ2U6IGFueSwgLi4ub3B0aW9uYWxQYXJhbXM6IGFueVtdKSB7XG4gICAgaWYgKHRoaXMubGV2ZWwgPCBMb2dMZXZlbC5JTkZPKSByZXR1cm47XG4gICAgY29uc29sZS5pbmZvKG1lc3NhZ2UsIC4uLm9wdGlvbmFsUGFyYW1zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBMb2dzIG1lc3NhZ2Ugd2l0aCBsZXZlbCBERUJVR1xuICAgKi9cbiAgZGVidWcobWVzc2FnZTogYW55LCAuLi5vcHRpb25hbFBhcmFtczogYW55W10pIHtcbiAgICBpZiAodGhpcy5sZXZlbCA8IExvZ0xldmVsLkRFQlVHKSByZXR1cm47XG4gICAgY29uc29sZS5kZWJ1ZyhtZXNzYWdlLCAuLi5vcHRpb25hbFBhcmFtcyk7XG4gIH1cblxuICAvKipcbiAgICogQWxpYXMgZm9yIGluZm9cbiAgICovXG4gIGxvZyhtZXNzYWdlOiBhbnksIC4uLm9wdGlvbmFsUGFyYW1zOiBhbnlbXSkge1xuICAgIHRoaXMuaW5mbyhtZXNzYWdlLCAuLi5vcHRpb25hbFBhcmFtcyk7XG4gIH1cbn1cbiJdfQ==