"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeyPair = exports.PublicKeyFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const statement = require("cdk-iam-floyd");
const constructs_1 = require("constructs");
const path = require("path");
const resourceType = 'Custom::EC2-Key-Pair';
const ID = `CFN::Resource::${resourceType}`;
const createdByTag = 'CreatedByCfnCustomResource';
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
/**
 * @stability stable
 */
var PublicKeyFormat;
(function (PublicKeyFormat) {
    PublicKeyFormat["OPENSSH"] = "OPENSSH";
    PublicKeyFormat["PEM"] = "PEM";
})(PublicKeyFormat = exports.PublicKeyFormat || (exports.PublicKeyFormat = {}));
/**
 * An EC2 Key Pair.
 *
 * @stability stable
 */
class KeyPair extends constructs_1.Construct {
    /**
     * Defines a new EC2 Key Pair.
     *
     * The private key will be stored in AWS Secrets Manager
     *
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        /**
         * ARN of the private key in AWS Secrets Manager.
         *
         * @stability stable
         */
        this.privateKeyArn = '';
        /**
         * ARN of the public key in AWS Secrets Manager.
         *
         * @stability stable
         */
        this.publicKeyArn = '';
        /**
         * The public key.
         *
         * Only filled, when `exposePublicKey = true`
         *
         * @stability stable
         */
        this.publicKeyValue = '';
        /**
         * Name of the Key Pair.
         *
         * @stability stable
         */
        this.keyPairName = '';
        /**
         * ID of the Key Pair.
         *
         * @stability stable
         */
        this.keyPairID = '';
        /**
         * @stability stable
         */
        this.prefix = '';
        if (props.removeKeySecretsAfterDays &&
            (props.removeKeySecretsAfterDays < 0 ||
                (props.removeKeySecretsAfterDays > 0 &&
                    props.removeKeySecretsAfterDays < 7) ||
                props.removeKeySecretsAfterDays > 30)) {
            cdk.Annotations.of(this).addError(`Parameter removeKeySecretsAfterDays must be 0 or between 7 and 30. Got ${props.removeKeySecretsAfterDays}`);
        }
        if (((_b = props.publicKey) === null || _b === void 0 ? void 0 : _b.length) &&
            props.publicKeyFormat !== undefined &&
            props.publicKeyFormat !== PublicKeyFormat.OPENSSH) {
            cdk.Annotations.of(this).addError('When importing a key, the format has to be of type OpenSSH');
        }
        const stack = cdk.Stack.of(this).stackName;
        this.prefix = props.resourcePrefix || stack;
        if (this.prefix.length + cleanID.length > 62)
            // Cloudformation limits names to 63 characters.
            cdk.Annotations.of(this).addError(`Cloudformation limits names to 63 characters.
         Prefix ${this.prefix} is too long to be used as a prefix for your roleName. Define parameter resourcePrefix?:`);
        this.lambda = this.ensureLambda();
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'Custom::EC2-Key-Pair');
        this.tags.setTag(createdByTag, ID);
        const kmsPrivate = props.kmsPrivateKey || props.kms;
        const kmsPublic = props.kmsPublicKey || props.kms;
        const key = new cdk.CustomResource(this, `EC2-Key-Pair-${props.name}`, {
            serviceToken: this.lambda.functionArn,
            resourceType: resourceType,
            properties: {
                Name: props.name,
                Description: props.description || '',
                KmsPrivate: (kmsPrivate === null || kmsPrivate === void 0 ? void 0 : kmsPrivate.keyArn) || 'alias/aws/secretsmanager',
                KmsPublic: (kmsPublic === null || kmsPublic === void 0 ? void 0 : kmsPublic.keyArn) || 'alias/aws/secretsmanager',
                PublicKey: props.publicKey || '',
                StorePublicKey: props.storePublicKey || false,
                ExposePublicKey: props.exposePublicKey || false,
                PublicKeyFormat: props.publicKeyFormat || PublicKeyFormat.OPENSSH,
                RemoveKeySecretsAfterDays: props.removeKeySecretsAfterDays || 0,
                SecretPrefix: props.secretPrefix || 'ec2-ssh-key/',
                StackName: stack,
                Tags: cdk.Lazy.any({
                    produce: () => this.tags.renderTags(),
                }),
            },
        });
        if (typeof props.kms !== 'undefined') {
            props.kms.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kms);
            key.node.addDependency(this.lambda.role);
        }
        if (typeof props.kmsPrivateKey !== 'undefined') {
            props.kmsPrivateKey.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kmsPrivateKey);
            key.node.addDependency(this.lambda.role);
        }
        if (typeof props.kmsPublicKey !== 'undefined') {
            props.kmsPublicKey.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kmsPublicKey);
            key.node.addDependency(this.lambda.role);
        }
        this.privateKeyArn = key.getAttString('PrivateKeyARN');
        this.publicKeyArn = key.getAttString('PublicKeyARN');
        this.publicKeyValue = key.getAttString('PublicKeyValue');
        this.keyPairName = key.getAttString('KeyPairName');
        this.keyPairID = key.getAttString('KeyPairID');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'EC2-Key-Name-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'EC2-Key-Pair-Manager-Policy', {
            managedPolicyName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            statements: [
                new statement.Ec2() // generally allow to inspect key pairs
                    .allow()
                    .toDescribeKeyPairs(),
                new statement.Ec2() // allow creation/import, only if createdByTag is set
                    .allow()
                    .toCreateKeyPair()
                    .toImportKeyPair()
                    .toCreateTags()
                    .onKeyPair('*', undefined, undefined, stack.partition)
                    .ifAwsRequestTag(createdByTag, ID),
                new statement.Ec2() // allow delete/update, only if createdByTag is set
                    .allow()
                    .toDeleteKeyPair()
                    .toCreateTags()
                    .toDeleteTags()
                    .onKeyPair('*', undefined, undefined, stack.partition)
                    .ifResourceTag(createdByTag, ID),
                new statement.Secretsmanager() // generally allow to list secrets. we need this to check if a secret exists before attempting to delete it
                    .allow()
                    .toListSecrets(),
                new statement.Secretsmanager() // allow creation, only if createdByTag is set
                    .allow()
                    .toCreateSecret()
                    .toTagResource()
                    .ifAwsRequestTag(createdByTag, ID),
                new statement.Secretsmanager() // allow delete/update, only if createdByTag is set
                    .allow()
                    .allMatchingActions('/^(Describe|Delete|Put|Update)/')
                    .toGetSecretValue()
                    .toGetResourcePolicy()
                    .toRestoreSecret()
                    .toListSecretVersionIds()
                    .toUntagResource()
                    .ifResourceTag(createdByTag, ID),
            ],
        });
        const role = new iam.Role(stack, 'EC2-Key-Pair-Manager-Role', {
            roleName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${this.prefix}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage EC2 Key Pairs',
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout),
        });
        return fn;
    }
    /**
     * Grants read access to the private key in AWS Secrets Manager.
     *
     * @stability stable
     */
    grantReadOnPrivateKey(grantee) {
        return this.grantRead(this.privateKeyArn, grantee);
    }
    /**
     * Grants read access to the public key in AWS Secrets Manager.
     *
     * @stability stable
     */
    grantReadOnPublicKey(grantee) {
        return this.grantRead(this.publicKeyArn, grantee);
    }
    grantRead(arn, grantee) {
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'secretsmanager:DescribeSecret',
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:ListSecretVersionIds',
            ],
            resourceArns: [arn],
            scope: this,
        });
        return result;
    }
}
exports.KeyPair = KeyPair;
_a = JSII_RTTI_SYMBOL_1;
KeyPair[_a] = { fqn: "cdk-ec2-key-pair.KeyPair", version: "3.2.0" };
//# sourceMappingURL=data:application/json;base64,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