"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
/**
 * The input to send to the event target
 */
class RuleTargetInput {
    /**
     * Pass text to the event target
     *
     * May contain strings returned by EventField.from() to substitute in parts of the
     * matched event.
     */
    static fromText(text) {
        return new FieldAwareEventInput(text, InputType.Text);
    }
    /**
     * Pass text to the event target, splitting on newlines.
     *
     * This is only useful when passing to a target that does not
     * take a single argument.
     *
     * May contain strings returned by EventField.from() to substitute in parts
     * of the matched event.
     */
    static fromMultilineText(text) {
        return new FieldAwareEventInput(text, InputType.Multiline);
    }
    /**
     * Pass a JSON object to the event target
     *
     * May contain strings returned by EventField.from() to substitute in parts of the
     * matched event.
     */
    static fromObject(obj) {
        return new FieldAwareEventInput(obj, InputType.Object);
    }
    /**
     * Take the event target input from a path in the event JSON
     */
    static fromEventPath(path) {
        return new LiteralEventInput({ inputPath: path });
    }
    constructor() {
    }
}
exports.RuleTargetInput = RuleTargetInput;
/**
 * Event Input that is directly derived from the construct
 */
class LiteralEventInput extends RuleTargetInput {
    constructor(props) {
        super();
        this.props = props;
    }
    /**
     * Return the input properties for this input object
     */
    bind(_rule) {
        return this.props;
    }
}
/**
 * Input object that can contain field replacements
 *
 * Evaluation is done in the bind() method because token resolution
 * requires access to the construct tree.
 *
 * Multiple tokens that use the same path will use the same substitution
 * key.
 *
 * One weird exception: if we're in object context, we MUST skip the quotes
 * around the placeholder. I assume this is so once a trivial string replace is
 * done later on by CWE, numbers are still numbers.
 *
 * So in string context:
 *
 *    "this is a string with a <field>"
 *
 * But in object context:
 *
 *    "{ \"this is the\": <field> }"
 *
 * To achieve the latter, we postprocess the JSON string to remove the surrounding
 * quotes by using a string replace.
 */
class FieldAwareEventInput extends RuleTargetInput {
    constructor(input, inputType) {
        super();
        this.input = input;
        this.inputType = inputType;
    }
    bind(rule) {
        let fieldCounter = 0;
        const pathToKey = new Map();
        const inputPathsMap = {};
        function keyForField(f) {
            const existing = pathToKey.get(f.path);
            if (existing !== undefined) {
                return existing;
            }
            fieldCounter += 1;
            const key = f.displayHint || `f${fieldCounter}`;
            pathToKey.set(f.path, key);
            return key;
        }
        const self = this;
        class EventFieldReplacer extends core_1.DefaultTokenResolver {
            constructor() {
                super(new core_1.StringConcat());
            }
            resolveToken(t, _context) {
                if (!isEventField(t)) {
                    return core_1.Token.asString(t);
                }
                const key = keyForField(t);
                if (inputPathsMap[key] && inputPathsMap[key] !== t.path) {
                    throw new Error(`Single key '${key}' is used for two different JSON paths: '${t.path}' and '${inputPathsMap[key]}'`);
                }
                inputPathsMap[key] = t.path;
                return self.keyPlaceholder(key);
            }
        }
        const stack = core_1.Stack.of(rule);
        let resolved;
        if (this.inputType === InputType.Multiline) {
            // JSONify individual lines
            resolved = core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer()
            });
            resolved = resolved.split('\n').map(stack.toJsonString).join('\n');
        }
        else {
            resolved = stack.toJsonString(core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer()
            }));
        }
        if (Object.keys(inputPathsMap).length === 0) {
            // Nothing special, just return 'input'
            return { input: resolved };
        }
        return {
            inputTemplate: this.unquoteKeyPlaceholders(resolved),
            inputPathsMap
        };
    }
    /**
     * Return a template placeholder for the given key
     *
     * In object scope we'll need to get rid of surrounding quotes later on, so
     * return a bracing that's unlikely to occur naturally (like tokens).
     */
    keyPlaceholder(key) {
        if (this.inputType !== InputType.Object) {
            return `<${key}>`;
        }
        return UNLIKELY_OPENING_STRING + key + UNLIKELY_CLOSING_STRING;
    }
    /**
     * Removing surrounding quotes from any object placeholders
     *
     * Those have been put there by JSON.stringify(), but we need to
     * remove them.
     */
    unquoteKeyPlaceholders(sub) {
        if (this.inputType !== InputType.Object) {
            return sub;
        }
        return core_1.Lazy.stringValue({ produce: (ctx) => core_1.Token.asString(deepUnquote(ctx.resolve(sub))) });
        function deepUnquote(resolved) {
            if (Array.isArray(resolved)) {
                return resolved.map(deepUnquote);
            }
            else if (typeof (resolved) === 'object' && resolved !== null) {
                for (const [key, value] of Object.entries(resolved)) {
                    resolved[key] = deepUnquote(value);
                }
                return resolved;
            }
            else if (typeof (resolved) === 'string') {
                return resolved.replace(OPENING_STRING_REGEX, '<').replace(CLOSING_STRING_REGEX, '>');
            }
            return resolved;
        }
    }
}
const UNLIKELY_OPENING_STRING = '<<${';
const UNLIKELY_CLOSING_STRING = '}>>';
const OPENING_STRING_REGEX = new RegExp(regexQuote('"' + UNLIKELY_OPENING_STRING), 'g');
const CLOSING_STRING_REGEX = new RegExp(regexQuote(UNLIKELY_CLOSING_STRING + '"'), 'g');
/**
 * Represents a field in the event pattern
 */
class EventField {
    constructor(path) {
        this.path = path;
        this.displayHint = this.path.replace(/^[^a-zA-Z0-9_-]+/, '').replace(/[^a-zA-Z0-9_-]/g, '-');
        Object.defineProperty(this, EVENT_FIELD_SYMBOL, { value: true });
        this.creationStack = core_1.captureStackTrace();
    }
    /**
     * Extract the event ID from the event
     */
    static get eventId() {
        return this.fromPath('$.id');
    }
    /**
     * Extract the detail type from the event
     */
    static get detailType() {
        return this.fromPath('$.detail-type');
    }
    /**
     * Extract the source from the event
     */
    static get source() {
        return this.fromPath('$.source');
    }
    /**
     * Extract the account from the event
     */
    static get account() {
        return this.fromPath('$.account');
    }
    /**
     * Extract the time from the event
     */
    static get time() {
        return this.fromPath('$.time');
    }
    /**
     * Extract the region from the event
     */
    static get region() {
        return this.fromPath('$.region');
    }
    /**
     * Extract a custom JSON path from the event
     */
    static fromPath(path) {
        return new EventField(path).toString();
    }
    resolve(_ctx) {
        return this.path;
    }
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.EventField = EventField;
var InputType;
(function (InputType) {
    InputType[InputType["Object"] = 0] = "Object";
    InputType[InputType["Text"] = 1] = "Text";
    InputType[InputType["Multiline"] = 2] = "Multiline";
})(InputType || (InputType = {}));
function isEventField(x) {
    return EVENT_FIELD_SYMBOL in x;
}
const EVENT_FIELD_SYMBOL = Symbol.for('@aws-cdk/aws-events.EventField');
/**
 * Quote a string for use in a regex
 */
function regexQuote(s) {
    return s.replace(/[.?*+^$[\]\\(){}|-]/g, '\\$&');
}
//# sourceMappingURL=data:application/json;base64,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