"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const crypto = require("crypto");
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
/**
 * A Deployment of a REST API.
 *
 * An immutable representation of a RestApi resource that can be called by users
 * using Stages. A deployment must be associated with a Stage for it to be
 * callable over the Internet.
 *
 * Normally, you don't need to define deployments manually. The RestApi
 * construct manages a Deployment resource that represents the latest model. It
 * can be accessed through `restApi.latestDeployment` (unless `deploy: false` is
 * set when defining the `RestApi`).
 *
 * If you manually define this resource, you will need to know that since
 * deployments are immutable, as long as the resource's logical ID doesn't
 * change, the deployment will represent the snapshot in time in which the
 * resource was created. This means that if you modify the RestApi model (i.e.
 * add methods or resources), these changes will not be reflected unless a new
 * deployment resource is created.
 *
 * To achieve this behavior, the method `addToLogicalId(data)` can be used to
 * augment the logical ID generated for the deployment resource such that it
 * will include arbitrary data. This is done automatically for the
 * `restApi.latestDeployment` deployment.
 *
 * Furthermore, since a deployment does not reference any of the REST API
 * resources and methods, CloudFormation will likely provision it before these
 * resources are created, which means that it will represent a "half-baked"
 * model. Use the `node.addDependency(dep)` method to circumvent that. This is done
 * automatically for the `restApi.latestDeployment` deployment.
 */
class Deployment extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new LatestDeploymentResource(this, 'Resource', {
            description: props.description,
            restApiId: props.api.restApiId,
        });
        if (props.retainDeployments) {
            this.resource.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        this.api = props.api;
        this.deploymentId = core_1.Lazy.stringValue({ produce: () => this.resource.ref });
    }
    /**
     * Adds a component to the hash that determines this Deployment resource's
     * logical ID.
     *
     * This should be called by constructs of the API Gateway model that want to
     * invalidate the deployment when their settings change. The component will
     * be resolve()ed during synthesis so tokens are welcome.
     */
    addToLogicalId(data) {
        this.resource.addToLogicalId(data);
    }
    /**
     * Hook into synthesis before it occurs and make any final adjustments.
     */
    prepare() {
        if (this.api instanceof restapi_1.RestApi) {
            // Ignore IRestApi that are imported
            /*
             * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
             * Quoting from CloudFormation's docs - "If you create an AWS::ApiGateway::RestApi resource and its methods (using AWS::ApiGateway::Method) in
             * the same template as your deployment, the deployment must depend on the RestApi's methods. To create a dependency, add a DependsOn attribute
             * to the deployment. If you don't, AWS CloudFormation creates the deployment right after it creates the RestApi resource that doesn't contain
             * any methods, and AWS CloudFormation encounters the following error: The REST API doesn't contain any methods."
             */
            /*
             * Adding a dependency between LatestDeployment and Method construct, using ConstructNode.addDependencies(), creates additional dependencies
             * between AWS::ApiGateway::Deployment and the AWS::Lambda::Permission nodes (children under Method), causing cyclic dependency errors. Hence,
             * falling back to declaring dependencies between the underlying CfnResources.
             */
            this.api.methods.map(m => m.node.defaultChild).forEach(m => this.resource.addDependsOn(m));
        }
    }
}
exports.Deployment = Deployment;
class LatestDeploymentResource extends apigateway_generated_1.CfnDeployment {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.hashComponents = new Array();
        this.originalLogicalId = core_1.Stack.of(this).getLogicalId(this);
    }
    /**
     * Allows adding arbitrary data to the hashed logical ID of this deployment.
     * This can be used to couple the deployment to the API Gateway model.
     */
    addToLogicalId(data) {
        // if the construct is locked, it means we are already synthesizing and then
        // we can't modify the hash because we might have already calculated it.
        if (this.node.locked) {
            throw new Error('Cannot modify the logical ID when the construct is locked');
        }
        this.hashComponents.push(data);
    }
    /**
     * Hooks into synthesis to calculate a logical ID that hashes all the components
     * add via `addToLogicalId`.
     */
    prepare() {
        const stack = core_1.Stack.of(this);
        // if hash components were added to the deployment, we use them to calculate
        // a logical ID for the deployment resource.
        if (this.hashComponents.length > 0) {
            const md5 = crypto.createHash('md5');
            this.hashComponents.map(x => stack.resolve(x)).forEach(c => md5.update(JSON.stringify(c)));
            this.overrideLogicalId(this.originalLogicalId + md5.digest('hex'));
        }
        super.prepare();
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVwbG95bWVudC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImRlcGxveW1lbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSx3Q0FBNkY7QUFDN0YsaUNBQWlDO0FBQ2pDLGlFQUEyRTtBQUMzRSx1Q0FBOEM7QUF5QjlDOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQTZCRztBQUNILE1BQWEsVUFBVyxTQUFRLGVBQVE7SUFPdEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFzQjtRQUM5RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSx3QkFBd0IsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzdELFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztZQUM5QixTQUFTLEVBQUUsS0FBSyxDQUFDLEdBQUcsQ0FBQyxTQUFTO1NBQy9CLENBQUMsQ0FBQztRQUVILElBQUksS0FBSyxDQUFDLGlCQUFpQixFQUFFO1lBQzNCLElBQUksQ0FBQyxRQUFRLENBQUMsa0JBQWtCLENBQUMsb0JBQWEsQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUN4RDtRQUVELElBQUksQ0FBQyxHQUFHLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQztRQUNyQixJQUFJLENBQUMsWUFBWSxHQUFHLFdBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO0lBQzdFLENBQUM7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksY0FBYyxDQUFDLElBQVM7UUFDN0IsSUFBSSxDQUFDLFFBQVEsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOztPQUVHO0lBQ08sT0FBTztRQUNmLElBQUksSUFBSSxDQUFDLEdBQUcsWUFBWSxpQkFBTyxFQUFFO1lBQy9CLG9DQUFvQztZQUVwQzs7Ozs7O2VBTUc7WUFFSDs7OztlQUlHO1lBQ0gsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUEyQixDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUMzRztJQUNILENBQUM7Q0FDRjtBQTFERCxnQ0EwREM7QUFFRCxNQUFNLHdCQUF5QixTQUFRLG9DQUFhO0lBSWxELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBeUI7UUFDakUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFKbEIsbUJBQWMsR0FBRyxJQUFJLEtBQUssRUFBTyxDQUFDO1FBTXhDLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksY0FBYyxDQUFDLElBQWE7UUFDakMsNEVBQTRFO1FBQzVFLHdFQUF3RTtRQUN4RSxJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQ3BCLE1BQU0sSUFBSSxLQUFLLENBQUMsMkRBQTJELENBQUMsQ0FBQztTQUM5RTtRQUVELElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7O09BR0c7SUFDTyxPQUFPO1FBQ2YsTUFBTSxLQUFLLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU3Qiw0RUFBNEU7UUFDNUUsNENBQTRDO1FBQzVDLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ2xDLE1BQU0sR0FBRyxHQUFHLE1BQU0sQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDckMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUMzRixJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLGlCQUFpQixHQUFHLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztTQUNwRTtRQUNELEtBQUssQ0FBQyxPQUFPLEVBQUUsQ0FBQztJQUNsQixDQUFDO0NBQ0YiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDZm5SZXNvdXJjZSwgQ29uc3RydWN0LCBMYXp5LCBSZW1vdmFsUG9saWN5LCBSZXNvdXJjZSwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGNyeXB0byBmcm9tICdjcnlwdG8nO1xuaW1wb3J0IHsgQ2ZuRGVwbG95bWVudCwgQ2ZuRGVwbG95bWVudFByb3BzIH0gZnJvbSAnLi9hcGlnYXRld2F5LmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBJUmVzdEFwaSwgUmVzdEFwaSB9IGZyb20gJy4vcmVzdGFwaSc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgRGVwbG95bWVudFByb3BzICB7XG4gIC8qKlxuICAgKiBUaGUgUmVzdCBBUEkgdG8gZGVwbG95LlxuICAgKi9cbiAgcmVhZG9ubHkgYXBpOiBJUmVzdEFwaTtcblxuICAvKipcbiAgICogQSBkZXNjcmlwdGlvbiBvZiB0aGUgcHVycG9zZSBvZiB0aGUgQVBJIEdhdGV3YXkgZGVwbG95bWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBkZXNjcmlwdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBXaGVuIGFuIEFQSSBHYXRld2F5IG1vZGVsIGlzIHVwZGF0ZWQsIGEgbmV3IGRlcGxveW1lbnQgd2lsbCBhdXRvbWF0aWNhbGx5IGJlIGNyZWF0ZWQuXG4gICAqIElmIHRoaXMgaXMgdHJ1ZSwgdGhlIG9sZCBBUEkgR2F0ZXdheSBEZXBsb3ltZW50IHJlc291cmNlIHdpbGwgbm90IGJlIGRlbGV0ZWQuXG4gICAqIFRoaXMgd2lsbCBhbGxvdyBtYW51YWxseSByZXZlcnRpbmcgYmFjayB0byBhIHByZXZpb3VzIGRlcGxveW1lbnQgaW4gY2FzZSBmb3IgZXhhbXBsZVxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcmV0YWluRGVwbG95bWVudHM/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIEEgRGVwbG95bWVudCBvZiBhIFJFU1QgQVBJLlxuICpcbiAqIEFuIGltbXV0YWJsZSByZXByZXNlbnRhdGlvbiBvZiBhIFJlc3RBcGkgcmVzb3VyY2UgdGhhdCBjYW4gYmUgY2FsbGVkIGJ5IHVzZXJzXG4gKiB1c2luZyBTdGFnZXMuIEEgZGVwbG95bWVudCBtdXN0IGJlIGFzc29jaWF0ZWQgd2l0aCBhIFN0YWdlIGZvciBpdCB0byBiZVxuICogY2FsbGFibGUgb3ZlciB0aGUgSW50ZXJuZXQuXG4gKlxuICogTm9ybWFsbHksIHlvdSBkb24ndCBuZWVkIHRvIGRlZmluZSBkZXBsb3ltZW50cyBtYW51YWxseS4gVGhlIFJlc3RBcGlcbiAqIGNvbnN0cnVjdCBtYW5hZ2VzIGEgRGVwbG95bWVudCByZXNvdXJjZSB0aGF0IHJlcHJlc2VudHMgdGhlIGxhdGVzdCBtb2RlbC4gSXRcbiAqIGNhbiBiZSBhY2Nlc3NlZCB0aHJvdWdoIGByZXN0QXBpLmxhdGVzdERlcGxveW1lbnRgICh1bmxlc3MgYGRlcGxveTogZmFsc2VgIGlzXG4gKiBzZXQgd2hlbiBkZWZpbmluZyB0aGUgYFJlc3RBcGlgKS5cbiAqXG4gKiBJZiB5b3UgbWFudWFsbHkgZGVmaW5lIHRoaXMgcmVzb3VyY2UsIHlvdSB3aWxsIG5lZWQgdG8ga25vdyB0aGF0IHNpbmNlXG4gKiBkZXBsb3ltZW50cyBhcmUgaW1tdXRhYmxlLCBhcyBsb25nIGFzIHRoZSByZXNvdXJjZSdzIGxvZ2ljYWwgSUQgZG9lc24ndFxuICogY2hhbmdlLCB0aGUgZGVwbG95bWVudCB3aWxsIHJlcHJlc2VudCB0aGUgc25hcHNob3QgaW4gdGltZSBpbiB3aGljaCB0aGVcbiAqIHJlc291cmNlIHdhcyBjcmVhdGVkLiBUaGlzIG1lYW5zIHRoYXQgaWYgeW91IG1vZGlmeSB0aGUgUmVzdEFwaSBtb2RlbCAoaS5lLlxuICogYWRkIG1ldGhvZHMgb3IgcmVzb3VyY2VzKSwgdGhlc2UgY2hhbmdlcyB3aWxsIG5vdCBiZSByZWZsZWN0ZWQgdW5sZXNzIGEgbmV3XG4gKiBkZXBsb3ltZW50IHJlc291cmNlIGlzIGNyZWF0ZWQuXG4gKlxuICogVG8gYWNoaWV2ZSB0aGlzIGJlaGF2aW9yLCB0aGUgbWV0aG9kIGBhZGRUb0xvZ2ljYWxJZChkYXRhKWAgY2FuIGJlIHVzZWQgdG9cbiAqIGF1Z21lbnQgdGhlIGxvZ2ljYWwgSUQgZ2VuZXJhdGVkIGZvciB0aGUgZGVwbG95bWVudCByZXNvdXJjZSBzdWNoIHRoYXQgaXRcbiAqIHdpbGwgaW5jbHVkZSBhcmJpdHJhcnkgZGF0YS4gVGhpcyBpcyBkb25lIGF1dG9tYXRpY2FsbHkgZm9yIHRoZVxuICogYHJlc3RBcGkubGF0ZXN0RGVwbG95bWVudGAgZGVwbG95bWVudC5cbiAqXG4gKiBGdXJ0aGVybW9yZSwgc2luY2UgYSBkZXBsb3ltZW50IGRvZXMgbm90IHJlZmVyZW5jZSBhbnkgb2YgdGhlIFJFU1QgQVBJXG4gKiByZXNvdXJjZXMgYW5kIG1ldGhvZHMsIENsb3VkRm9ybWF0aW9uIHdpbGwgbGlrZWx5IHByb3Zpc2lvbiBpdCBiZWZvcmUgdGhlc2VcbiAqIHJlc291cmNlcyBhcmUgY3JlYXRlZCwgd2hpY2ggbWVhbnMgdGhhdCBpdCB3aWxsIHJlcHJlc2VudCBhIFwiaGFsZi1iYWtlZFwiXG4gKiBtb2RlbC4gVXNlIHRoZSBgbm9kZS5hZGREZXBlbmRlbmN5KGRlcClgIG1ldGhvZCB0byBjaXJjdW12ZW50IHRoYXQuIFRoaXMgaXMgZG9uZVxuICogYXV0b21hdGljYWxseSBmb3IgdGhlIGByZXN0QXBpLmxhdGVzdERlcGxveW1lbnRgIGRlcGxveW1lbnQuXG4gKi9cbmV4cG9ydCBjbGFzcyBEZXBsb3ltZW50IGV4dGVuZHMgUmVzb3VyY2Uge1xuICAvKiogQGF0dHJpYnV0ZSAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGVwbG95bWVudElkOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBhcGk6IElSZXN0QXBpO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgcmVzb3VyY2U6IExhdGVzdERlcGxveW1lbnRSZXNvdXJjZTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRGVwbG95bWVudFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMucmVzb3VyY2UgPSBuZXcgTGF0ZXN0RGVwbG95bWVudFJlc291cmNlKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGRlc2NyaXB0aW9uOiBwcm9wcy5kZXNjcmlwdGlvbixcbiAgICAgIHJlc3RBcGlJZDogcHJvcHMuYXBpLnJlc3RBcGlJZCxcbiAgICB9KTtcblxuICAgIGlmIChwcm9wcy5yZXRhaW5EZXBsb3ltZW50cykge1xuICAgICAgdGhpcy5yZXNvdXJjZS5hcHBseVJlbW92YWxQb2xpY3koUmVtb3ZhbFBvbGljeS5SRVRBSU4pO1xuICAgIH1cblxuICAgIHRoaXMuYXBpID0gcHJvcHMuYXBpO1xuICAgIHRoaXMuZGVwbG95bWVudElkID0gTGF6eS5zdHJpbmdWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucmVzb3VyY2UucmVmIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBjb21wb25lbnQgdG8gdGhlIGhhc2ggdGhhdCBkZXRlcm1pbmVzIHRoaXMgRGVwbG95bWVudCByZXNvdXJjZSdzXG4gICAqIGxvZ2ljYWwgSUQuXG4gICAqXG4gICAqIFRoaXMgc2hvdWxkIGJlIGNhbGxlZCBieSBjb25zdHJ1Y3RzIG9mIHRoZSBBUEkgR2F0ZXdheSBtb2RlbCB0aGF0IHdhbnQgdG9cbiAgICogaW52YWxpZGF0ZSB0aGUgZGVwbG95bWVudCB3aGVuIHRoZWlyIHNldHRpbmdzIGNoYW5nZS4gVGhlIGNvbXBvbmVudCB3aWxsXG4gICAqIGJlIHJlc29sdmUoKWVkIGR1cmluZyBzeW50aGVzaXMgc28gdG9rZW5zIGFyZSB3ZWxjb21lLlxuICAgKi9cbiAgcHVibGljIGFkZFRvTG9naWNhbElkKGRhdGE6IGFueSkge1xuICAgIHRoaXMucmVzb3VyY2UuYWRkVG9Mb2dpY2FsSWQoZGF0YSk7XG4gIH1cblxuICAvKipcbiAgICogSG9vayBpbnRvIHN5bnRoZXNpcyBiZWZvcmUgaXQgb2NjdXJzIGFuZCBtYWtlIGFueSBmaW5hbCBhZGp1c3RtZW50cy5cbiAgICovXG4gIHByb3RlY3RlZCBwcmVwYXJlKCkge1xuICAgIGlmICh0aGlzLmFwaSBpbnN0YW5jZW9mIFJlc3RBcGkpIHtcbiAgICAgIC8vIElnbm9yZSBJUmVzdEFwaSB0aGF0IGFyZSBpbXBvcnRlZFxuXG4gICAgICAvKlxuICAgICAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWFwaWdhdGV3YXktZGVwbG95bWVudC5odG1sXG4gICAgICAgKiBRdW90aW5nIGZyb20gQ2xvdWRGb3JtYXRpb24ncyBkb2NzIC0gXCJJZiB5b3UgY3JlYXRlIGFuIEFXUzo6QXBpR2F0ZXdheTo6UmVzdEFwaSByZXNvdXJjZSBhbmQgaXRzIG1ldGhvZHMgKHVzaW5nIEFXUzo6QXBpR2F0ZXdheTo6TWV0aG9kKSBpblxuICAgICAgICogdGhlIHNhbWUgdGVtcGxhdGUgYXMgeW91ciBkZXBsb3ltZW50LCB0aGUgZGVwbG95bWVudCBtdXN0IGRlcGVuZCBvbiB0aGUgUmVzdEFwaSdzIG1ldGhvZHMuIFRvIGNyZWF0ZSBhIGRlcGVuZGVuY3ksIGFkZCBhIERlcGVuZHNPbiBhdHRyaWJ1dGVcbiAgICAgICAqIHRvIHRoZSBkZXBsb3ltZW50LiBJZiB5b3UgZG9uJ3QsIEFXUyBDbG91ZEZvcm1hdGlvbiBjcmVhdGVzIHRoZSBkZXBsb3ltZW50IHJpZ2h0IGFmdGVyIGl0IGNyZWF0ZXMgdGhlIFJlc3RBcGkgcmVzb3VyY2UgdGhhdCBkb2Vzbid0IGNvbnRhaW5cbiAgICAgICAqIGFueSBtZXRob2RzLCBhbmQgQVdTIENsb3VkRm9ybWF0aW9uIGVuY291bnRlcnMgdGhlIGZvbGxvd2luZyBlcnJvcjogVGhlIFJFU1QgQVBJIGRvZXNuJ3QgY29udGFpbiBhbnkgbWV0aG9kcy5cIlxuICAgICAgICovXG5cbiAgICAgIC8qXG4gICAgICAgKiBBZGRpbmcgYSBkZXBlbmRlbmN5IGJldHdlZW4gTGF0ZXN0RGVwbG95bWVudCBhbmQgTWV0aG9kIGNvbnN0cnVjdCwgdXNpbmcgQ29uc3RydWN0Tm9kZS5hZGREZXBlbmRlbmNpZXMoKSwgY3JlYXRlcyBhZGRpdGlvbmFsIGRlcGVuZGVuY2llc1xuICAgICAgICogYmV0d2VlbiBBV1M6OkFwaUdhdGV3YXk6OkRlcGxveW1lbnQgYW5kIHRoZSBBV1M6OkxhbWJkYTo6UGVybWlzc2lvbiBub2RlcyAoY2hpbGRyZW4gdW5kZXIgTWV0aG9kKSwgY2F1c2luZyBjeWNsaWMgZGVwZW5kZW5jeSBlcnJvcnMuIEhlbmNlLFxuICAgICAgICogZmFsbGluZyBiYWNrIHRvIGRlY2xhcmluZyBkZXBlbmRlbmNpZXMgYmV0d2VlbiB0aGUgdW5kZXJseWluZyBDZm5SZXNvdXJjZXMuXG4gICAgICAgKi9cbiAgICAgIHRoaXMuYXBpLm1ldGhvZHMubWFwKG0gPT4gbS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5SZXNvdXJjZSkuZm9yRWFjaChtID0+IHRoaXMucmVzb3VyY2UuYWRkRGVwZW5kc09uKG0pKTtcbiAgICB9XG4gIH1cbn1cblxuY2xhc3MgTGF0ZXN0RGVwbG95bWVudFJlc291cmNlIGV4dGVuZHMgQ2ZuRGVwbG95bWVudCB7XG4gIHByaXZhdGUgaGFzaENvbXBvbmVudHMgPSBuZXcgQXJyYXk8YW55PigpO1xuICBwcml2YXRlIG9yaWdpbmFsTG9naWNhbElkOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENmbkRlcGxveW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuXG4gICAgdGhpcy5vcmlnaW5hbExvZ2ljYWxJZCA9IFN0YWNrLm9mKHRoaXMpLmdldExvZ2ljYWxJZCh0aGlzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBbGxvd3MgYWRkaW5nIGFyYml0cmFyeSBkYXRhIHRvIHRoZSBoYXNoZWQgbG9naWNhbCBJRCBvZiB0aGlzIGRlcGxveW1lbnQuXG4gICAqIFRoaXMgY2FuIGJlIHVzZWQgdG8gY291cGxlIHRoZSBkZXBsb3ltZW50IHRvIHRoZSBBUEkgR2F0ZXdheSBtb2RlbC5cbiAgICovXG4gIHB1YmxpYyBhZGRUb0xvZ2ljYWxJZChkYXRhOiB1bmtub3duKSB7XG4gICAgLy8gaWYgdGhlIGNvbnN0cnVjdCBpcyBsb2NrZWQsIGl0IG1lYW5zIHdlIGFyZSBhbHJlYWR5IHN5bnRoZXNpemluZyBhbmQgdGhlblxuICAgIC8vIHdlIGNhbid0IG1vZGlmeSB0aGUgaGFzaCBiZWNhdXNlIHdlIG1pZ2h0IGhhdmUgYWxyZWFkeSBjYWxjdWxhdGVkIGl0LlxuICAgIGlmICh0aGlzLm5vZGUubG9ja2VkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBtb2RpZnkgdGhlIGxvZ2ljYWwgSUQgd2hlbiB0aGUgY29uc3RydWN0IGlzIGxvY2tlZCcpO1xuICAgIH1cblxuICAgIHRoaXMuaGFzaENvbXBvbmVudHMucHVzaChkYXRhKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBIb29rcyBpbnRvIHN5bnRoZXNpcyB0byBjYWxjdWxhdGUgYSBsb2dpY2FsIElEIHRoYXQgaGFzaGVzIGFsbCB0aGUgY29tcG9uZW50c1xuICAgKiBhZGQgdmlhIGBhZGRUb0xvZ2ljYWxJZGAuXG4gICAqL1xuICBwcm90ZWN0ZWQgcHJlcGFyZSgpIHtcbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHRoaXMpO1xuXG4gICAgLy8gaWYgaGFzaCBjb21wb25lbnRzIHdlcmUgYWRkZWQgdG8gdGhlIGRlcGxveW1lbnQsIHdlIHVzZSB0aGVtIHRvIGNhbGN1bGF0ZVxuICAgIC8vIGEgbG9naWNhbCBJRCBmb3IgdGhlIGRlcGxveW1lbnQgcmVzb3VyY2UuXG4gICAgaWYgKHRoaXMuaGFzaENvbXBvbmVudHMubGVuZ3RoID4gMCkge1xuICAgICAgY29uc3QgbWQ1ID0gY3J5cHRvLmNyZWF0ZUhhc2goJ21kNScpO1xuICAgICAgdGhpcy5oYXNoQ29tcG9uZW50cy5tYXAoeCA9PiBzdGFjay5yZXNvbHZlKHgpKS5mb3JFYWNoKGMgPT4gbWQ1LnVwZGF0ZShKU09OLnN0cmluZ2lmeShjKSkpO1xuICAgICAgdGhpcy5vdmVycmlkZUxvZ2ljYWxJZCh0aGlzLm9yaWdpbmFsTG9naWNhbElkICsgbWQ1LmRpZ2VzdCgnaGV4JykpO1xuICAgIH1cbiAgICBzdXBlci5wcmVwYXJlKCk7XG4gIH1cbn1cbiJdfQ==