"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Base class for backend integrations for an API Gateway method.
 *
 * Use one of the concrete classes such as `MockIntegration`, `AwsIntegration`, `LambdaIntegration`
 * or implement on your own by specifying the set of props.
 */
class Integration {
    constructor(props) {
        this.props = props;
    }
    /**
     * Allows `Method` to access the integration props.
     *
     * @internal
     */
    get _props() {
        return this.props;
    }
    /**
     * Can be overridden by subclasses to allow the integration to interact with the method
     * being integrated, access the REST API object, method ARNs, etc.
     */
    bind(_method) {
        return;
    }
}
exports.Integration = Integration;
var ContentHandling;
(function (ContentHandling) {
    /**
     * Converts a request payload from a base64-encoded string to a binary blob.
     */
    ContentHandling["CONVERT_TO_BINARY"] = "CONVERT_TO_BINARY";
    /**
     * Converts a request payload from a binary blob to a base64-encoded string.
     */
    ContentHandling["CONVERT_TO_TEXT"] = "CONVERT_TO_TEXT";
})(ContentHandling = exports.ContentHandling || (exports.ContentHandling = {}));
var IntegrationType;
(function (IntegrationType) {
    /**
     * For integrating the API method request with an AWS service action,
     * including the Lambda function-invoking action. With the Lambda
     * function-invoking action, this is referred to as the Lambda custom
     * integration. With any other AWS service action, this is known as AWS
     * integration.
     */
    IntegrationType["AWS"] = "AWS";
    /**
     * For integrating the API method request with the Lambda function-invoking
     * action with the client request passed through as-is. This integration is
     * also referred to as the Lambda proxy integration
     */
    IntegrationType["AWS_PROXY"] = "AWS_PROXY";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC. This integration is also referred to
     * as the HTTP custom integration.
     */
    IntegrationType["HTTP"] = "HTTP";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC, with the client request passed
     * through as-is. This is also referred to as the HTTP proxy integration
     */
    IntegrationType["HTTP_PROXY"] = "HTTP_PROXY";
    /**
     * For integrating the API method request with API Gateway as a "loop-back"
     * endpoint without invoking any backend.
     */
    IntegrationType["MOCK"] = "MOCK";
})(IntegrationType = exports.IntegrationType || (exports.IntegrationType = {}));
var PassthroughBehavior;
(function (PassthroughBehavior) {
    /**
     * Passes the request body for unmapped content types through to the
     * integration back end without transformation.
     */
    PassthroughBehavior["WHEN_NO_MATCH"] = "WHEN_NO_MATCH";
    /**
     * Rejects unmapped content types with an HTTP 415 'Unsupported Media Type'
     * response
     */
    PassthroughBehavior["NEVER"] = "NEVER";
    /**
     * Allows pass-through when the integration has NO content types mapped to
     * templates. However if there is at least one content type defined,
     * unmapped content types will be rejected with the same 415 response.
     */
    PassthroughBehavior["WHEN_NO_TEMPLATES"] = "WHEN_NO_TEMPLATES";
})(PassthroughBehavior = exports.PassthroughBehavior || (exports.PassthroughBehavior = {}));
var ConnectionType;
(function (ConnectionType) {
    /**
     * For connections through the public routable internet
     */
    ConnectionType["INTERNET"] = "INTERNET";
    /**
     * For private connections between API Gateway and a network load balancer in a VPC
     */
    ConnectionType["VPC_LINK"] = "VPC_LINK";
})(ConnectionType = exports.ConnectionType || (exports.ConnectionType = {}));
//# sourceMappingURL=data:application/json;base64,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