"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const apigw = require("../lib");
const DUMMY_AUTHORIZER = {
    authorizerId: 'dummyauthorizer',
    authorizationType: apigw.AuthorizationType.CUSTOM
};
module.exports = {
    'default setup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'POST',
            AuthorizationType: 'NONE',
            Integration: {
                Type: 'MOCK'
            }
        }));
        test.done();
    },
    'method options can be specified'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                apiKeyRequired: true,
                operationName: 'MyOperation',
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            ApiKeyRequired: true,
            OperationName: 'MyOperation'
        }));
        test.done();
    },
    'integration can be set via a property'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            integration: new apigw.AwsIntegration({ service: 's3', path: 'bucket/key' })
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            Integration: {
                IntegrationHttpMethod: 'POST',
                Type: 'AWS',
                Uri: {
                    'Fn::Join': [
                        '',
                        [
                            'arn:', { Ref: 'AWS::Partition' }, ':apigateway:',
                            { Ref: 'AWS::Region' }, ':s3:path/bucket/key'
                        ]
                    ]
                }
            }
        }));
        test.done();
    },
    'integration with a custom http method can be set via a property'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            integration: new apigw.AwsIntegration({ service: 's3', path: 'bucket/key', integrationHttpMethod: 'GET' })
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            Integration: {
                IntegrationHttpMethod: 'GET'
            }
        }));
        test.done();
    },
    'use default integration from api'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const defaultIntegration = new apigw.Integration({ type: apigw.IntegrationType.HTTP_PROXY, uri: 'https://amazon.com' });
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
            defaultIntegration
        });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            Integration: {
                Type: 'HTTP_PROXY',
                Uri: 'https://amazon.com'
            }
        }));
        test.done();
    },
    '"methodArn" returns the ARN execute-api ARN for this method in the current stage'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api');
        // WHEN
        const method = new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        test.deepEqual(stack.resolve(method.methodArn), {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'testapiD6451F70' },
                    '/',
                    { Ref: 'testapiDeploymentStageprod5C9E92A4' },
                    '/POST/'
                ]
            ]
        });
        test.done();
    },
    '"testMethodArn" returns the ARN of the "test-invoke-stage" stage (console UI)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api');
        // WHEN
        const method = new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
        });
        // THEN
        test.deepEqual(stack.resolve(method.testMethodArn), {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'testapiD6451F70' },
                    '/test-invoke-stage/POST/'
                ]
            ]
        });
        test.done();
    },
    '"methodArn" fails if the API does not have a deployment stage'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const method = new apigw.Method(stack, 'my-method', { httpMethod: 'POST', resource: api.root });
        // WHEN + THEN
        test.throws(() => method.methodArn, /Unable to determine ARN for method "my-method" since there is no stage associated with this API./);
        test.done();
    },
    'integration "credentialsRole" can be used to assume a role when calling backend'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const role = new iam.Role(stack, 'MyRole', { assumedBy: new iam.ServicePrincipal('foo') });
        // WHEN
        api.root.addMethod('GET', new apigw.Integration({
            type: apigw.IntegrationType.AWS_PROXY,
            options: {
                credentialsRole: role
            }
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            Integration: {
                Credentials: { 'Fn::GetAtt': ['MyRoleF48FFE04', 'Arn'] }
            }
        }));
        test.done();
    },
    'integration "credentialsPassthrough" can be used to passthrough user credentials to backend'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        api.root.addMethod('GET', new apigw.Integration({
            type: apigw.IntegrationType.AWS_PROXY,
            options: {
                credentialsPassthrough: true
            }
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            Integration: {
                Credentials: { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::*:user/*']] }
            }
        }));
        test.done();
    },
    'integration "credentialsRole" and "credentialsPassthrough" are mutually exclusive'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const role = new iam.Role(stack, 'MyRole', { assumedBy: new iam.ServicePrincipal('foo') });
        // WHEN
        const integration = new apigw.Integration({
            type: apigw.IntegrationType.AWS_PROXY,
            options: {
                credentialsPassthrough: true,
                credentialsRole: role
            }
        });
        // THEN
        test.throws(() => api.root.addMethod('GET', integration), /'credentialsPassthrough' and 'credentialsRole' are mutually exclusive/);
        test.done();
    },
    'integration connectionType VpcLink requires vpcLink to be set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        const integration = new apigw.Integration({
            type: apigw.IntegrationType.HTTP_PROXY,
            integrationHttpMethod: 'ANY',
            options: {
                connectionType: apigw.ConnectionType.VPC_LINK,
            }
        });
        // THEN
        test.throws(() => api.root.addMethod('GET', integration), /'connectionType' of VPC_LINK requires 'vpcLink' prop to be set/);
        test.done();
    },
    'connectionType of INTERNET and vpcLink are mutually exclusive'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const vpc = new ec2.Vpc(stack, 'VPC');
        const nlb = new elbv2.NetworkLoadBalancer(stack, 'NLB', {
            vpc
        });
        const link = new apigw.VpcLink(stack, 'link', {
            targets: [nlb]
        });
        // WHEN
        const integration = new apigw.Integration({
            type: apigw.IntegrationType.HTTP_PROXY,
            integrationHttpMethod: 'ANY',
            options: {
                connectionType: apigw.ConnectionType.INTERNET,
                vpcLink: link
            }
        });
        // THEN
        test.throws(() => api.root.addMethod('GET', integration), /cannot set 'vpcLink' where 'connectionType' is INTERNET/);
        test.done();
    },
    'methodResponse set one or more method responses via options'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                methodResponses: [{
                        statusCode: '200'
                    }, {
                        statusCode: '400',
                        responseParameters: {
                            'method.response.header.killerbees': false
                        }
                    }, {
                        statusCode: '500',
                        responseParameters: {
                            'method.response.header.errthing': true
                        },
                        responseModels: {
                            'application/json': apigw.Model.EMPTY_MODEL,
                            'text/plain': apigw.Model.ERROR_MODEL
                        }
                    }
                ]
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            MethodResponses: [{
                    StatusCode: '200'
                }, {
                    StatusCode: '400',
                    ResponseParameters: {
                        'method.response.header.killerbees': false
                    }
                }, {
                    StatusCode: '500',
                    ResponseParameters: {
                        'method.response.header.errthing': true
                    },
                    ResponseModels: {
                        'application/json': 'Empty',
                        'text/plain': 'Error'
                    }
                }
            ]
        }));
        test.done();
    },
    'multiple integration responses can be used'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        api.root.addMethod('GET', new apigw.AwsIntegration({
            service: 'foo-service',
            action: 'BarAction',
            options: {
                integrationResponses: [
                    {
                        statusCode: '200',
                        responseTemplates: { 'application/json': JSON.stringify({ success: true }) },
                    },
                    {
                        selectionPattern: 'Invalid',
                        statusCode: '503',
                        responseTemplates: { 'application/json': JSON.stringify({ success: false, message: 'Invalid Request' }) },
                    }
                ],
            }
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            Integration: {
                IntegrationHttpMethod: 'POST',
                IntegrationResponses: [
                    {
                        ResponseTemplates: { 'application/json': '{"success":true}' },
                        StatusCode: '200',
                    },
                    {
                        ResponseTemplates: { 'application/json': '{"success":false,"message":"Invalid Request"}' },
                        SelectionPattern: 'Invalid',
                        StatusCode: '503',
                    }
                ],
                Type: 'AWS',
                Uri: { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':apigateway:', { Ref: 'AWS::Region' }, ':foo-service:action/BarAction']] }
            }
        }));
        test.done();
    },
    'method is always set as uppercase'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'api');
        // WHEN
        api.root.addMethod('get');
        api.root.addMethod('PoSt');
        api.root.addMethod('PUT');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', { HttpMethod: 'POST' }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', { HttpMethod: 'GET' }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', { HttpMethod: 'PUT' }));
        test.done();
    },
    'requestModel can be set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const model = api.addModel('test-model', {
            contentType: 'application/json',
            modelName: 'test-model',
            schema: {
                title: 'test',
                type: apigw.JsonSchemaType.OBJECT,
                properties: { message: { type: apigw.JsonSchemaType.STRING } }
            }
        });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                requestModels: {
                    'application/json': model
                }
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            RequestModels: {
                'application/json': { Ref: stack.getLogicalId(model.node.findChild('Resource')) }
            }
        }));
        test.done();
    },
    'methodResponse has a mix of response modes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const htmlModel = api.addModel('my-model', {
            schema: {
                schema: apigw.JsonSchemaVersion.DRAFT4,
                title: 'test',
                type: apigw.JsonSchemaType.OBJECT,
                properties: { message: { type: apigw.JsonSchemaType.STRING } }
            }
        });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                methodResponses: [{
                        statusCode: '200'
                    }, {
                        statusCode: '400',
                        responseParameters: {
                            'method.response.header.killerbees': false
                        }
                    }, {
                        statusCode: '500',
                        responseParameters: {
                            'method.response.header.errthing': true
                        },
                        responseModels: {
                            'application/json': apigw.Model.EMPTY_MODEL,
                            'text/plain': apigw.Model.ERROR_MODEL,
                            'text/html': htmlModel
                        }
                    }
                ]
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            MethodResponses: [{
                    StatusCode: '200'
                }, {
                    StatusCode: '400',
                    ResponseParameters: {
                        'method.response.header.killerbees': false
                    }
                }, {
                    StatusCode: '500',
                    ResponseParameters: {
                        'method.response.header.errthing': true
                    },
                    ResponseModels: {
                        'application/json': 'Empty',
                        'text/plain': 'Error',
                        'text/html': { Ref: stack.getLogicalId(htmlModel.node.findChild('Resource')) }
                    }
                }
            ]
        }));
        test.done();
    },
    'method has a request validator'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const validator = api.addRequestValidator('validator', {
            validateRequestBody: true,
            validateRequestParameters: false
        });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                requestValidator: validator
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            RequestValidatorId: { Ref: stack.getLogicalId(validator.node.findChild('Resource')) }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            ValidateRequestBody: true,
            ValidateRequestParameters: false
        }));
        test.done();
    },
    'use default requestParameters'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
            defaultMethodOptions: {
                requestParameters: { 'method.request.path.proxy': true }
            }
        });
        // WHEN
        new apigw.Method(stack, 'defaultRequestParameters', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                operationName: 'defaultRequestParameters'
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            OperationName: 'defaultRequestParameters',
            RequestParameters: {
                'method.request.path.proxy': true
            }
        }));
        test.done();
    },
    'authorizer is bound correctly'(test) {
        const stack = new cdk.Stack();
        const restApi = new apigw.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: DUMMY_AUTHORIZER
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'ANY',
            AuthorizationType: 'CUSTOM',
            AuthorizerId: DUMMY_AUTHORIZER.authorizerId,
        }));
        test.done();
    },
    'authorizer via default method options'(test) {
        const stack = new cdk.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const auth = new apigw.TokenAuthorizer(stack, 'myauthorizer1', {
            authorizerName: 'myauthorizer1',
            handler: func
        });
        const restApi = new apigw.RestApi(stack, 'myrestapi', {
            defaultMethodOptions: {
                authorizer: auth
            }
        });
        restApi.root.addMethod('ANY');
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Name: 'myauthorizer1',
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId)
        }));
        test.done();
    },
    'fails when authorization type does not match the authorizer'(test) {
        const stack = new cdk.Stack();
        const restApi = new apigw.RestApi(stack, 'myrestapi');
        test.throws(() => {
            restApi.root.addMethod('ANY', undefined, {
                authorizationType: apigw.AuthorizationType.IAM,
                authorizer: DUMMY_AUTHORIZER
            });
        }, /Authorization type is set to AWS_IAM which is different from what is required by the authorizer/);
        test.done();
    },
    'fails when authorization type does not match the authorizer in default method options'(test) {
        const stack = new cdk.Stack();
        const restApi = new apigw.RestApi(stack, 'myrestapi', {
            defaultMethodOptions: {
                authorizer: DUMMY_AUTHORIZER
            }
        });
        test.throws(() => {
            restApi.root.addMethod('ANY', undefined, {
                authorizationType: apigw.AuthorizationType.NONE,
            });
        }, /Authorization type is set to NONE which is different from what is required by the authorizer/);
        test.done();
    },
    'method has Auth Scopes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: false });
        // WHEN
        new apigw.Method(stack, 'my-method', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                apiKeyRequired: true,
                authorizationScopes: ['AuthScope1', 'AuthScope2'],
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            ApiKeyRequired: true,
            AuthorizationScopes: ['AuthScope1', 'AuthScope2']
        }));
        test.done();
    },
    'use default Auth Scopes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
            defaultMethodOptions: {
                authorizationScopes: ['DefaultAuth']
            }
        });
        // WHEN
        new apigw.Method(stack, 'defaultAuthScopes', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                operationName: 'defaultAuthScopes'
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            OperationName: 'defaultAuthScopes',
            AuthorizationScopes: ['DefaultAuth']
        }));
        test.done();
    },
    'Method options Auth Scopes is picked up'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false,
            defaultMethodOptions: {
                authorizationScopes: ['DefaultAuth']
            }
        });
        // WHEN
        new apigw.Method(stack, 'MethodAuthScopeUsed', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                apiKeyRequired: true,
                authorizationScopes: ['MethodAuthScope'],
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            ApiKeyRequired: true,
            AuthorizationScopes: ['MethodAuthScope']
        }));
        test.done();
    },
    'Auth Scopes absent'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', {
            cloudWatchRole: false,
            deploy: false
        });
        // WHEN
        new apigw.Method(stack, 'authScopesAbsent', {
            httpMethod: 'POST',
            resource: api.root,
            options: {
                operationName: 'authScopesAbsent'
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            OperationName: 'authScopesAbsent',
            AuthorizationScopes: assert_1.ABSENT
        }));
        test.done();
    },
    'method has a request validator with provided properties'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                requestValidatorOptions: {
                    requestValidatorName: 'test-validator',
                    validateRequestBody: true,
                    validateRequestParameters: false
                }
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: stack.resolve(api.restApiId),
            ValidateRequestBody: true,
            ValidateRequestParameters: false,
            Name: 'test-validator'
        }));
        test.done();
    },
    'method does not have a request validator'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        // WHEN
        new apigw.Method(stack, 'method-man', {
            httpMethod: 'GET',
            resource: api.root
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            RequestValidatorId: assert_1.ABSENT
        }));
        test.done();
    },
    'method does not support both request validator and request validator options'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigw.RestApi(stack, 'test-api', { deploy: false });
        const validator = api.addRequestValidator('test-validator1', {
            validateRequestBody: true,
            validateRequestParameters: false
        });
        // WHEN
        const methodProps = {
            httpMethod: 'GET',
            resource: api.root,
            options: {
                requestValidatorOptions: {
                    requestValidatorName: 'test-validator2',
                    validateRequestBody: true,
                    validateRequestParameters: false
                },
                requestValidator: validator
            }
        };
        // THEN
        test.throws(() => new apigw.Method(stack, 'method', methodProps), /Only one of 'requestValidator' or 'requestValidatorOptions' must be specified./);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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