# **************************************************************************
# *
# * Authors:    David Herreros Calero (dherreros@cnb.csic.es)
# *             Scipion Team (scipion@cnb.csic.es)
# *
# *  BCU, Centro Nacional de Biotecnologia, CSIC
# *
# * This program is free software; you can redistribute it and/or modify
# * it under the terms of the GNU General Public License as published by
# * the Free Software Foundation; either version 2 of the License, or
# * (at your option) any later version.
# *
# * This program is distributed in the hope that it will be useful,
# * but WITHOUT ANY WARRANTY; without even the implied warranty of
# * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# * GNU General Public License for more details.
# *
# * You should have received a copy of the GNU General Public License
# * along with this program; if not, write to the Free Software
# * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
# * 02111-1307  USA
# *
# *  All comments concerning this program package may be sent to the
# *  e-mail address 'scipion@cnb.csic.es'
# *
# **************************************************************************
from enum import Enum
from os import remove
from os.path import abspath

from pwem.protocols import EMProtocol
from pyworkflow import BETA
from pyworkflow.protocol import params
from tomo.objects import SetOfCoordinates3D
from tomo.protocols import ProtTomoBase
from dynamo import Plugin, M_GENERAL_DES, M_GENERAL_WITH_BOXES_DES, M_GENERAL_NAME, M_SURFACE_NAME, \
    M_ELLIPSOIDAL_VESICLE_NAME, M_GENERAL_WITH_BOXES_NAME, M_SURFACE_DES, \
    M_SPH_VESICLE_NAME, M_VESICLE_DES, M_ELLIPSOIDAL_VESICLE_DES, M_MARKED_ELLIP_VESICLE_NAME, \
    M_MARKED_ELLIP_VESICLE_DES, \
    MB_BY_LEVELS, MB_ELLIPSOIDAL, MB_GENERAL, MB_GENERAL_BOXES, MB_VESICLE, MB_ELLIPSOIDAL_MARKED, \
    MODELS_NOT_PROCESSED_IN_MW, MODELS_ALLOWED_IN_MW_NAMES, M_VESICLE_NAME
from ..utils import genMCode4ReadAndSaveData, dynamoCroppingResults2Scipion, createSetOfOutputCoords, getCroppedFile


# Model types mapping
MODEL_CHOICES = [M_ELLIPSOIDAL_VESICLE_NAME, M_SURFACE_NAME, M_GENERAL_NAME]

# Model types encoding
M_VESICLE = 0
M_SURFACE = 1
M_GENERAL = 2

# Dynamo model names mapping
dynModelsDict = {
    MB_BY_LEVELS: M_SURFACE,
    MB_VESICLE: M_VESICLE,
    MB_ELLIPSOIDAL: M_VESICLE,
    MB_ELLIPSOIDAL_MARKED: M_VESICLE,
    MB_GENERAL: M_GENERAL,
    MB_GENERAL_BOXES: M_GENERAL,
}


class OutputsModelWf(Enum):
    coordinates = SetOfCoordinates3D


class DynamoModelWorkflow(EMProtocol, ProtTomoBase):
    """Apply a model workflow to a SetOfMeshes generated by Dynamo Boxing protocol. This workflow will use the
    models created by the user to create the corresponding cropping meshes needed to extract the crop points"""

    _label = 'model workflow'
    _devStatus = BETA
    _possibleOutputs = OutputsModelWf

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

    # --------------------------- DEFINE param functions ----------------------
    def _defineParams(self, form):
        form.addSection(label='Input')
        form.addParam('inputMeshes', params.PointerParam,
                      pointerClass='SetOfMeshes',
                      label="Input Meshes",
                      important=True,
                      help="Input Meshes that will be used to create the cropping geometry and "
                           "to extract the crop points")
        form.addParam('boxSize', params.IntParam,
                      default=0,
                      label='Box size',
                      important=True)

        form.addSection('Model parameters')
        form.addLine('Available models: description in help --> ', help=self._genModelsDescriptionMsg())
        form.addLine('Notation: %s' % self._genModelsNotationMsg(),
                     help='The meshes for the general models will be calculated treating them as surface models')
        group = form.addGroup('Common to multiple models')
        group.addParam('meshParameter', params.IntParam,
                       default=5,
                       label='Mesh parameter [G][S][V]',
                       help='Intended mesh parameter for the "mesh" that supports the depiction of the model')
        group.addParam('maxTr', params.IntParam,
                       default=100000,
                       label="Maximun number of triangles [G][S][V]",
                       help='Maximum number of triangles allowed during generation of a depiction mesh')
        group.addParam('cropping', params.IntParam,
                       default=10,
                       label="Cropping parameter [G][S][V]",
                       help='Intended mesh parameter for the "crop_mesh" that defined a cropping '
                            'geometry on a surface')
        group.addParam('subDivision', params.IntParam,
                       default=2,
                       label="Number of Subdivision steps [G][S]",
                       help="Specifiy the number of times the Mesh geometry will be subdivided. This will increase the "
                            "number of triangles in the mesh, making it smoother. However, it will also increase the "
                            "number of cropping points")

    # --------------------------- INSERT steps functions ----------------------
    def _insertAllSteps(self):
        modelsDict = self.inputMeshes.get().getUniqueValues(['_tomoId', '_dynModelName', '_dynModelFile', '_groupId'])
        for tomoId, modelName, modelFile in zip(modelsDict['_tomoId'], modelsDict['_dynModelName'],
                                                modelsDict['_dynModelFile']):
            self._insertFunctionStep(self.applyWorkflowStep, tomoId, modelName, modelFile)
        self._insertFunctionStep(self.createOutputStep)

    # --------------------------- STEPS functions -----------------------------
    def applyWorkflowStep(self, tomoId, modelName, modelFile):
        commandsFile = self.writeMatlabFile(tomoId, modelName, modelFile)
        args = ' %s' % commandsFile
        self.runJob(Plugin.getDynamoProgram(), args, env=Plugin.getEnviron())

    def createOutputStep(self):
        precedentsPointer = self.inputMeshes.get()._precedentsPointer
        precedents = precedentsPointer.get()
        tomoList = [tomo.clone() for tomo in precedents]
        tomoFileDict = {abspath(tomo.getFileName()): tomo for tomo in tomoList}
        croppedFile = getCroppedFile(self._getTmpPath())
        outCoords = createSetOfOutputCoords(self._getPath(), self._getExtraPath(), precedentsPointer, boxSize=self.boxSize.get())
        dynamoCroppingResults2Scipion(outCoords, croppedFile, tomoFileDict)
        # Remove previous file to avoid data repetition because of the append mode
        remove(croppedFile)
        # Define outputs and relations
        self._defineOutputs(**{self._possibleOutputs.coordinates.name: outCoords})
        self._defineSourceRelation(self.inputMeshes, outCoords)
        self._updateOutputSet(self._possibleOutputs.coordinates.name, outCoords, state=outCoords.STREAM_CLOSED)

    # --------------------------- DEFINE utils functions ----------------------
    def writeMatlabFile(self, tomoId, modelName, modelFile):
        content = ''
        codeFilePath = self._getExtraPath('modelWf_%s_%s.m' % (tomoId, modelName))
        modelType = self._getModelType(modelName)
        if modelType == M_VESICLE:
            content = self.genVesicleCmdFileContents(modelName)
        elif modelType == M_SURFACE:
            content = self.genSCmdFileContents()
        elif modelType == M_GENERAL:
            # Change its type to surface model and process it
            content = self.genGen2SurfCmdFileContents()
        content = genMCode4ReadAndSaveData(self._getTmpPath(), modelFile, savePicked=False, saveCropped=True, modelWfCode=content)
        with open(codeFilePath, 'w') as codeFid:
            codeFid.write(content)
        return codeFilePath

    def genVesicleCmdFileContents(self, modelName):
        # Let Dynamo approximate the geometry based on the points annotated in the boxing protocol
        contentMWf = "m.approximateGeometryFromPoints()\n"
        # Mesh creation steps (some of which are specific for each sub-model of type vesicle)
        contentMWf += "m.mesh_parameter = %i\n" % self.meshParameter.get()
        contentMWf += "m.crop_mesh_parameter = %i\n" % self.cropping.get()
        contentMWf += "m.mesh_maximum_triangles = %i\n" % self.maxTr.get()
        if modelName == MB_ELLIPSOIDAL:
            contentMWf += "m.createMesh()\n"
        contentMWf += "m.updateCrop()\n"
        contentMWf += "m.grepTable()\n"
        # Format and write the output data in a text file that will be read in the step create output
        return contentMWf

    def genSCmdFileContents(self):
        # Mesh creation steps
        contentMWf = self._genSurfaceModelMeshSteps()
        # Format and write the output data in a text file that will be read in the step create output
        return contentMWf

    def genGen2SurfCmdFileContents(self):
        # Change model type from general to surface
        contentMWf = "m = model.changeType(m, '%s')\n" % 'membraneByLevels'
        contentMWf += "zCoords = m.points(:, 3)\n"
        contentMWf += "zUVals = unique(zCoords)\n"
        contentMWf += "nParticles = length(zCoords)\n"
        contentMWf += "groupLabels = zeros(1, nParticles)\n"
        contentMWf += "for j=1:length(zUVals)\n"
        contentMWf += "currentZ = zUVals(i)\n"
        contentMWf += "groupLabels(zCoords == currentZ) = j\n"
        contentMWf += "end\n"
        contentMWf += "m.group_labels = groupLabels\n"
        contentMWf += "m.last_group_label = i\n"
        # Mesh creation steps
        contentMWf += self._genSurfaceModelMeshSteps()
        # Format and write the output data in a text file that will be read in the step create output
        return contentMWf

    @staticmethod
    def _genModelsNotationMsg():
        modelsHelp = '[G]: %s,  ' % M_GENERAL_NAME
        modelsHelp += '[S]: %s,  ' % M_SURFACE_NAME
        modelsHelp += '[V]: %s' % M_VESICLE_NAME
        return modelsHelp

    @staticmethod
    def _genModelsDescriptionMsg():
        modelsHelp = '*%s* (NOTE: they will be converted into surface model to generate the mesh):\n\n' % M_GENERAL_NAME.upper()
        modelsHelp += '\t1) *%s*: %s\n\n' % (M_GENERAL_NAME, M_GENERAL_DES)
        modelsHelp += '\t2) *%s*: %s\n\n\n' % (M_GENERAL_WITH_BOXES_NAME, M_GENERAL_WITH_BOXES_DES)
        modelsHelp += '*%s*:\n%s\n\n\n' % (M_SURFACE_NAME.upper(), M_SURFACE_DES)
        modelsHelp += '*%s*:\n\n' % M_VESICLE_NAME.upper()
        modelsHelp += '\t1) *%s*: %s\n\n' % (M_SPH_VESICLE_NAME, M_VESICLE_DES)
        modelsHelp += '\t2) *%s*: %s\n\n' % (M_ELLIPSOIDAL_VESICLE_NAME, M_ELLIPSOIDAL_VESICLE_DES)
        modelsHelp += '\t3) *%s*: %s\n\n\n' % (M_MARKED_ELLIP_VESICLE_NAME, M_MARKED_ELLIP_VESICLE_DES)
        return modelsHelp

    @staticmethod
    def _getModelType(modelName):
        # Map the Dynamo model names into the protocol encoding model values
        return dynModelsDict[modelName]

    def getMeshResultFile(self, tomoId):
        return abspath(self._getExtraPath('%s.txt' % tomoId))

    def _genSurfaceModelMeshSteps(self):
        # Mesh creation steps
        content = "m.mesh_parameter=%i\n" % self.meshParameter.get()
        content += "m.crop_mesh_parameter=%i\n" % self.cropping.get()
        content += "m.mesh_maximum_triangles=%i\n" % self.maxTr.get()
        content += "m.subdivision_iterations=%i\n" % self.subDivision.get()
        content += "m.createMesh()\n"
        content += "m.refineMesh()\n"
        content += "m.createCropMesh()\n"
        content += "m.refineCropMesh()\n"
        content += "m.updateCrop()\n"
        content += "m.grepTable()\n"
        return content

    # --------------------------- DEFINE INFO functions ----------------------
    def _summary(self):
        summary = []
        if self.getOutputsSize() >= 1:
            for _, outCoords in self.iterOutputAttributes():
                summary.append("Output *%s*:" % outCoords.getNameId().split('.')[1])
                summary.append("    * Particle box size: *%s*" % self.boxSize.get())
                summary.append("    * Coordinates defined by geometry: *%s*" %
                               outCoords.getSize())
        else:
            summary.append("Output coordinates not ready yet.")
        return summary

    def _validate(self):
        errorMsg = []
        # Only sets of meshes generated using the Dynamo picking protocol are accepted (they must contain
        # an attribute named '_dynCatalogue')
        if not getattr(self.inputMeshes.get(), '_dynCatalogue', None):
            errorMsg.append('Only sets of meshes generated using the Dynamo picking protocol are accepted')
        return errorMsg

    def _warnings(self):
        warnMsg = []
        presentModelList = self.inputMeshes.get().getUniqueValues(['_dynModelName'])
        for presentModel in presentModelList:
            if presentModel in MODELS_NOT_PROCESSED_IN_MW:
                warnMsg.append('Some of the models provided are not allowed in this protocol. Allowed models are:\n%s'
                               % '\n - '.join(MODELS_ALLOWED_IN_MW_NAMES.insert(0, '')))
                break
        return warnMsg
