import os, json, uuid, jsonschema, datetime

"""
Job tracker file management

Parameters
----------
job_ticket_id: job ticket ID generated by the ticket service
job_name: name of the job in the master task list
"""
class JobTracker():
  def __init__(self):
    self.hostname = os.uname().nodename
    # Absolute path to this script
    self.current_script_dir = os.path.dirname(__file__)
    self.schemas_dir = os.path.join(self.current_script_dir, '..', 'schemas')
    self.JOB_DATA_BASEDIR = os.getenv('JOB_DATA_BASEDIR', '/home/appuser/jobs/')

  # Get timestamp now in UTC ISO8601 format with timezone (always "+00:00" for UTC timezone) and truncate microseconds
  def get_timestamp_utc_now(self):
    return datetime.datetime.utcnow().replace(tzinfo=datetime.timezone.utc, microsecond=0).isoformat()

  """
  Create a new job ticket id

  Create a uuid4 unique identifier
  Prepend the uuid with the hostname of the container
  separated by an underscore character {HOSTNAME}_{UUID}
  e.g. e226a430b959_815985bc-b927-428b-953d-77fd408f8a18
  """
  def create_job_ticket_id(self):
    return '{}_{}'.format(self.hostname, str(uuid.uuid4()))

  """
  Filter an element in an array of objects by property name
  Return a list that contains the matching element

  Parameters
  ----------
  arr: array of objects, required
  prop: str, property name
  val: str, value to match
  """
  def filter_element_by_property(self, arr, prop, val):
    return list( filter((lambda x: x[prop] == val), arr))

  def get_main_step_list(self):
    print('get_main_step_list')
    main_step_list = None
    # Get master task list JSON file
    try:
      with open('{}/{}.json'.format(self.schemas_dir, 'main_step_list')) as json_file:
        main_step_list = json.load(json_file)
    except Exception as e:
      error_msg = "Unable to get main step list json file."
      print(error_msg)
      print(e)
      raise Exception(e)

    # Validate main step list file
    is_main_step_list_file_valid = None
    try:
      is_main_step_list_file_valid = self.validate_json_schema('main_step_list_schema', main_step_list)
    except Exception as e:
      error_msg = "Unable to validate main step list file."
      print(error_msg)
      print(e)
      raise Exception(e)

    if is_main_step_list_file_valid is not True:
        raise Exception('Invalid main step list file in lts-mpsjobtracker module schemas/main_step_list_schema.json')

    return main_step_list

  def create_initial_tracker_file(self):
    initial_tracker_file = None
    # Get tracker init JSON file
    try:
      with open('{}/{}.json'.format(self.schemas_dir, 'job_tracker_init')) as json_file:
        initial_tracker_file = json.load(json_file)
    except Exception as e:
      error_msg = "Unable to get tracker init json file."
      print(error_msg)
      print(e)
      raise Exception(e)
    return initial_tracker_file

  """
  Validate JSON schema

  Parameters
  ----------
  schema_filename: name of the json file in the /schemas directory without the file extension
  file_to_validate: a json file to validate
  """
  def validate_json_schema(self, schema_filename, file_to_validate):

    is_file_valid = False

    # Get JSON validation schema
    validation_schema = None
    try:
      with open('{}/{}.json'.format(self.schemas_dir, schema_filename)) as json_file:
        validation_schema = json.load(json_file)
    except Exception as e:
      error_msg = "Unable to get json schema file."
      print(error_msg)
      print(e)
      raise Exception(e)

    # Validate JSON schema 
    try:
      jsonschema.validate(file_to_validate, validation_schema)
    except json.decoder.JSONDecodeError as e:
      error_msg = "Invalid JSON format: {}".format(e)
      print(e)
      print(error_msg)
      raise Exception(e)
    except jsonschema.exceptions.ValidationError as e:
      error_msg = "Invalid JSON schema: {}".format(e)
      print(e)
      print(error_msg)
      raise Exception(e)
    except Exception as e:
      error_msg = "Unable to validate json model."
      print(e)
      print(error_msg)
      raise Exception(e)
    else:
      is_file_valid = True
    return is_file_valid

  """
  Create new or update existing tracker file
  A generic method to write the tracker file to the filesystem with schema validation
  This method is called any time a tracker file is created or updated to ensure
  the schema is always valid

  Parameters
  ----------
  tracker_file: the new tracker file to be written to the filesystem
  """
  def write_tracker_file(self, tracker_file):

    # Validate tracker file
    try:
      is_tracker_file_valid = self.validate_json_schema('job_tracker_schema', tracker_file)
    except Exception as e:
      error_msg = "Unable to validate job tracker file."
      print(error_msg)
      print(e)
      raise Exception(e)

    job_ticket_id = tracker_file['job_ticket_id']

    # Append updated date to tracker file
    timestamp_utc_now = self.get_timestamp_utc_now()
    tracker_file['last_modified_date'] = timestamp_utc_now

    # Create job tracker file sub directory with the ticket id as the directory name
    job_tracker_directory = os.path.join(self.JOB_DATA_BASEDIR, job_ticket_id)
    # Check if tracker file directory exists
    if os.path.isdir(job_tracker_directory) != True:
      # Create directory if it does not exist
      try:
        os.mkdir(job_tracker_directory)
      except Exception as e:
        error_msg = "Unable to create job tracker file directory."
        print(error_msg)
        print(e)
        raise Exception(e)

    # Create job tracker file with the ticket id as the file name
    try:
      with open('{}/{}.json'.format(job_tracker_directory, job_ticket_id), 'w') as fp:
        json.dump(tracker_file, fp)
    except Exception as e:
      error_msg = "Unable to write job tracker file."
      print(error_msg)
      print(e)
      raise Exception(e)
    return tracker_file

  def get_tracker_file(self, job_ticket_id):
    tracker_file = None
    # Get job tracker directory path
    job_tracker_directory = os.path.join(self.JOB_DATA_BASEDIR, job_ticket_id)
    # Get tracker file
    if job_ticket_id:

      # Validate job ticket id
      try:
        is_job_ticket_id_valid = self.validate_job_ticket_id(job_ticket_id)
      except Exception as e:
        error_msg = "Unable to validate job ticket id."
        print(error_msg)
        print(e)
        raise Exception(e)
      
      if not is_job_ticket_id_valid:
        raise Exception("Invalid job ticket id")

      elif is_job_ticket_id_valid:
        try:
          with open('{}/{}.json'.format(job_tracker_directory, job_ticket_id)) as json_file:
            tracker_file = json.load(json_file)
        except Exception as e:
          error_msg = "Unable to get job tracker file."
          print(error_msg)
          print(e)
          raise Exception(e)

        if tracker_file:
          # Validate tracker file
          try:
            is_tracker_file_valid = self.validate_json_schema('job_tracker_schema', tracker_file)
          except Exception as e:
            error_msg = "Unable to validate job tracker file."
            print(error_msg)
            print(e)
            raise Exception(e)

    return tracker_file

  """
  Initialize a new tracker file by job name
  """
  def init_tracker_file(self, job_name, job_context = None):
  
    # Get initial tracker file
    try:
      initial_tracker_file = self.create_initial_tracker_file()
    except Exception as e:
      error_msg = "Unable to get initial tracker file."
      print(error_msg)
      print(e)
      raise Exception(e)

    # Get main step list
    main_step_list = self.get_main_step_list()
    # Get steps in main step list by job name
    step_list_for_this_job = self.filter_element_by_property(main_step_list, 'job_name', job_name)
    
    steps_error_msg = "Unable to get steps from main step list job_name {}".format(job_name)
    if not step_list_for_this_job or len(step_list_for_this_job) < 1:
      raise Exception(steps_error_msg)

    steps = step_list_for_this_job[0].get('steps')
    job_name = step_list_for_this_job[0].get('job_name')

    if steps is None or job_name is None:
      raise Exception(steps_error_msg)
    # Get job ticket id
    job_ticket_id = self.create_job_ticket_id()
    # Add list of job steps to tracker file
    initial_tracker_file['job_management']['steps'] = steps
    initial_tracker_file['job_name'] = job_name
    # Add job ID to tracker file
    initial_tracker_file['job_ticket_id'] = job_ticket_id
    # Add created and updated dates to tracker file
    timestamp_utc_now = self.get_timestamp_utc_now()
    initial_tracker_file['creation_date'] = timestamp_utc_now
    initial_tracker_file['last_modified_date'] = timestamp_utc_now

    # Add job context to tracker file
    if job_context is not None:
      initial_tracker_file['context'] = job_context

    # Write initial tracker file
    try:
      self.write_tracker_file(initial_tracker_file)
    except Exception as e:
      error_msg = "Unable to write initial tracker file."
      print(error_msg)
      print(e)
      raise Exception(e)

    return json.loads(json.dumps(initial_tracker_file))

  """
  Update job timestamp file with current timestamp
  """
  def update_timestamp_file(self, job_ticket_id):

    timestamp_file = None

    if job_ticket_id:
      timestamp_file = {}
      timestamp_utc_now = self.get_timestamp_utc_now()
      timestamp_file['job_ticket_id'] = job_ticket_id
      timestamp_file['last_modified_date'] = timestamp_utc_now
      # Get job tracker directory path
      job_tracker_directory = os.path.join(self.JOB_DATA_BASEDIR, job_ticket_id)

      # Create job timestamp file with the current timestamp as the last modified date
      try:
        with open('{}/{}_TIMESTAMP.json'.format(job_tracker_directory, job_ticket_id), 'w') as fp:
          json.dump(timestamp_file, fp)
      except Exception as e:
        error_msg = "Unable to write job timestamp file."
        print(error_msg)
        print(e)
        raise Exception(e)

    return timestamp_file

  def get_timestamp_file(self, job_ticket_id):
    timestamp_file = None
    # Get job tracker directory path
    job_tracker_directory = os.path.join(self.JOB_DATA_BASEDIR, job_ticket_id)
    # Get tracker file
    if job_ticket_id:
       
      # Validate job ticket id
      is_job_ticket_id_valid = None
      try:
        is_job_ticket_id_valid = self.validate_job_ticket_id(job_ticket_id)
      except Exception as e:
        error_msg = "Unable to validate job ticket id."
        print(error_msg)
        print(e)
        raise Exception(e)
      
      if is_job_ticket_id_valid:
        try:
          with open('{}/{}_TIMESTAMP.json'.format(job_tracker_directory, job_ticket_id)) as json_file:
            timestamp_file = json.load(json_file)
        except Exception as e:
          error_msg = "Unable to get job timestamp file."
          print(error_msg)
          print(e)
          raise Exception(e)

        # Validate timestamp file
        is_timestamp_file_valid = None
        try:
          is_timestamp_file_valid = self.validate_json_schema('job_timestamp_schema', timestamp_file)
        except Exception as e:
          error_msg = "Unable to validate job timestamp file."
          print(error_msg)
          print(e)
          raise Exception(e)

          if is_timestamp_file_valid is not True:
            raise Exception('Invalid job timestamp file job_ticket_id {}'.format(job_ticket_id))
    
    return timestamp_file

  """
  Validate a uuid4 string
  """
  def validate_uuid4(self, uuid4_string):
    val = None
    uuid_string = str(uuid.uuid4())
    try:
      val = uuid.UUID(uuid4_string, version=4)
    except ValueError as verr:
      return False
    return str(val) == uuid4_string

  """
  Validate a job ticket id
  Job ticket should be a hostname and a uuid separated by an underscore character {HOST}_{UUID}
  """
  def validate_job_ticket_id(self, job_ticket_id):
    is_job_ticket_id_valid = False
    if '_' in job_ticket_id:
      # Split job ticket into a list
      ticket_split = job_ticket_id.split('_')
      # Get second element in the list
      ticket_uuid = ticket_split[1]
      if ticket_uuid:
        is_job_ticket_id_valid = self.validate_uuid4(ticket_uuid)
    return is_job_ticket_id_valid
