"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CheckAlbProps = exports.GetActiveListener = exports.AddFargateTarget = exports.AddLambdaTarget = exports.AddListener = exports.ObtainAlb = void 0;
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const elbt = require("aws-cdk-lib/aws-elasticloadbalancingv2-targets");
const utils_1 = require("./utils");
const alb_defaults_1 = require("./alb-defaults");
const s3_bucket_helper_1 = require("./s3-bucket-helper");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
//  Returns the correct ALB Load Balancer to use in this construct, either an existing
//  one provided as an argument or create  new one otherwise.
function ObtainAlb(scope, id, vpc, publicApi, existingLoadBalancerObj, loadBalancerProps, logAccessLogs, loggingBucketProps) {
    let loadBalancer;
    if (existingLoadBalancerObj) {
        loadBalancer = existingLoadBalancerObj;
    }
    else {
        const consolidatedProps = utils_1.consolidateProps({}, loadBalancerProps, { vpc, internetFacing: publicApi });
        loadBalancer = new elb.ApplicationLoadBalancer(scope, `${id}-alb`, consolidatedProps);
        if (logAccessLogs === undefined || logAccessLogs === true) {
            const consolidatedLoggingBucketProps = utils_1.consolidateProps(s3_bucket_defaults_1.DefaultLoggingBucketProps(), loggingBucketProps);
            const loggingBucket = s3_bucket_helper_1.createAlbLoggingBucket(scope, id, consolidatedLoggingBucketProps);
            loadBalancer.logAccessLogs(loggingBucket);
        }
    }
    return loadBalancer;
}
exports.ObtainAlb = ObtainAlb;
function AddListener(scope, id, loadBalancer, listenerProps) {
    let consolidatedListenerProps;
    consolidatedListenerProps = utils_1.consolidateProps(alb_defaults_1.DefaultListenerProps(loadBalancer), listenerProps);
    //  create the listener
    const listener = new elb.ApplicationListener(scope, `${id}-listener`, consolidatedListenerProps);
    loadBalancer.listeners.push(listener);
    if (consolidatedListenerProps.protocol === elb.ApplicationProtocol.HTTP) {
        // This will use core.printWarning in the actual construct
        utils_1.printWarning("AWS recommends encrypting traffic to an Application Load Balancer using HTTPS.");
        if (listenerProps.certificates?.length > 0) {
            throw new Error("HTTP listeners cannot use a certificate");
        }
    }
    else {
        if (!listenerProps.certificates || listenerProps.certificates.length === 0) {
            throw new Error("A listener using HTTPS protocol requires a certificate");
        }
        listener.addCertificates(`${id}-cert`, listenerProps.certificates);
    }
    if (consolidatedListenerProps.protocol === elb.ApplicationProtocol.HTTPS) {
        const opt = {
            port: "443",
            protocol: "HTTPS",
        };
        const httpListener = new elb.ApplicationListener(scope, `${id}-redirect`, {
            loadBalancer,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            defaultAction: aws_elasticloadbalancingv2_1.ListenerAction.redirect(opt),
        });
        loadBalancer.listeners.push(httpListener);
    }
    return listener;
}
exports.AddListener = AddListener;
// Creates a Target Group for Lambda functions and adds the
// provided functions as a target to that group. Then adds
// the new Target Group to the provided Listener.
function AddLambdaTarget(scope, id, currentListener, lambdaFunction, ruleProps, targetProps) {
    //  Create the target and assign it to a new target group
    const lambdaTarget = new elbt.LambdaTarget(lambdaFunction);
    const newTargetGroup = new elb.ApplicationTargetGroup(scope, `${id}-tg`, {
        targets: [lambdaTarget],
        targetGroupName: targetProps ? targetProps.targetGroupName : undefined,
        healthCheck: targetProps ? targetProps.healthCheck : undefined
    });
    // The interface AddRuleProps includes conditions and priority, combine that
    // with targetGroups and we can assemble AddApplicationTargetGroupProps
    const consolidatedTargetProps = utils_1.consolidateProps({}, ruleProps, { targetGroups: [newTargetGroup] });
    currentListener.addTargetGroups(`${scope.node.id}-targets`, consolidatedTargetProps);
    newTargetGroup.setAttribute('stickiness.enabled', undefined);
    return newTargetGroup;
}
exports.AddLambdaTarget = AddLambdaTarget;
function AddFargateTarget(scope, id, currentListener, fargateService, ruleProps, targetProps) {
    if (targetProps?.protocol !== elb.ApplicationProtocol.HTTPS) {
        utils_1.printWarning('AWS recommends using HTTPS protocol for Target Groups in production applications');
    }
    const newTargetGroup = new elb.ApplicationTargetGroup(scope, `${id}-tg`, targetProps);
    // The interface AddRuleProps includes conditions and priority, combine that
    // with targetGroups and we can assemble an AddApplicationTargetGroupProps object
    const consolidatedTargetProps = utils_1.consolidateProps({ targetGroups: [newTargetGroup] }, ruleProps);
    currentListener.addTargetGroups(`${scope.node.id}-targets`, consolidatedTargetProps);
    newTargetGroup.addTarget(fargateService);
    return newTargetGroup;
}
exports.AddFargateTarget = AddFargateTarget;
// Looks for the listener associated with Target Groups
// If there is a single listener, this returns it whether it is HTTP or HTTPS
// If there are 2 listeners, it finds the HTTPS listener (we assume the HTTP listener redirects to HTTPS)
function GetActiveListener(listeners) {
    let listener;
    if (listeners.length === 0) {
        throw new Error(`There are no listeners in the ALB`);
    }
    if (listeners.length === 1) {
        listener = listeners[0];
    }
    else {
        listener = listeners.find(i => i.node.children[0].protocol === "HTTPS");
    }
    return listener;
}
exports.GetActiveListener = GetActiveListener;
function CheckAlbProps(props) {
    let errorMessages = '';
    let errorFound = false;
    if (props.listenerProps?.certificateArns) {
        errorMessages += "certificateArns is deprecated. Please supply certificates using props.listenerProps.certificates\n";
        errorFound = true;
    }
    if (((props.existingLoadBalancerObj &&
        props.existingLoadBalancerObj.listeners.length === 0) ||
        !props.existingLoadBalancerObj) &&
        !props.listenerProps) {
        errorMessages += "When adding the first listener and target to a load balancer, listenerProps must be specified and include at least a certificate or protocol: HTTP\n";
        errorFound = true;
    }
    if (props.existingLoadBalancerObj &&
        props.existingLoadBalancerObj.listeners.length > 0 &&
        props.listenerProps) {
        errorFound = true;
        errorMessages += "This load balancer already has a listener, listenerProps may not be specified\n";
    }
    if (props.existingLoadBalancerObj &&
        props.existingLoadBalancerObj.listeners.length > 0 &&
        !props.ruleProps) {
        errorFound = true;
        errorMessages += "When adding a second target to an existing listener, there must be rules provided\n";
    }
    // Check construct specific invalid inputs
    if (props.existingLoadBalancerObj && !props.existingVpc) {
        errorFound = true;
        errorMessages += "An existing ALB is already in a VPC, that VPC must be provided in props.existingVpc for the rest of the construct to use.\n";
    }
    if (props.loadBalancerProps?.vpc) {
        errorFound = true;
        errorMessages += 'Specify any existing VPC at the construct level, not within loadBalancerProps.\n';
    }
    if (props.existingLoadBalancerObj) {
        utils_1.printWarning("The public/private property of an existing ALB must match the props.publicApi setting provided.");
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.CheckAlbProps = CheckAlbProps;
//# sourceMappingURL=data:application/json;base64,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