"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.consolidateProps = exports.addCfnSuppressRules = exports.generateResourceName = exports.printWarning = exports.overrideProps = void 0;
const deepmerge = require("deepmerge");
const override_warning_service_1 = require("./override-warning-service");
const log = require("npmlog");
const crypto = require("crypto");
const cdk = require("aws-cdk-lib");
function isObject(val) {
    return val != null && typeof val === 'object'
        && Object.prototype.toString.call(val) === '[object Object]';
}
function isPlainObject(o) {
    if (Array.isArray(o) === true) {
        return true;
    }
    if (isObject(o) === false) {
        return false;
    }
    // If has modified constructor
    const ctor = o.constructor;
    if (typeof ctor !== 'function') {
        return false;
    }
    // If has modified prototype
    const prot = ctor.prototype;
    if (isObject(prot) === false) {
        return false;
    }
    // If constructor does not have an Object-specific method
    if (prot.hasOwnProperty('isPrototypeOf') === false) {
        return false;
    }
    // Most likely a plain Object
    return true;
}
function combineMerge(target, source) {
    return target.concat(source);
}
function overwriteMerge(target, source) {
    target = source;
    return target;
}
function overrideProps(DefaultProps, userProps, concatArray = false) {
    // Notify the user via console output if defaults are overridden
    const overrideWarningsEnabled = (process.env.overrideWarningsEnabled !== 'false');
    if (overrideWarningsEnabled) {
        override_warning_service_1.flagOverriddenDefaults(DefaultProps, userProps);
    }
    // Override the sensible defaults with user provided props
    if (concatArray) {
        return deepmerge(DefaultProps, userProps, {
            arrayMerge: combineMerge,
            isMergeableObject: isPlainObject
        });
    }
    else {
        return deepmerge(DefaultProps, userProps, {
            arrayMerge: overwriteMerge,
            isMergeableObject: isPlainObject
        });
    }
}
exports.overrideProps = overrideProps;
function printWarning(message) {
    // Style the log output
    log.prefixStyle.bold = true;
    log.prefixStyle.fg = 'red';
    log.enableColor();
    log.warn('AWS_SOLUTIONS_CONSTRUCTS_WARNING: ', message);
}
exports.printWarning = printWarning;
/**
 * @summary Creates a resource name in the style of the CDK (string+hash)
 * @param {string[]} parts - the various string components of the name (eg - stackName, solutions construct ID, L2 construct ID)
 * @param {number} maxLength - the longest string that can be returned
 * @returns {string} - a string with concatenated parts (truncated if neccessary) + a hash of the full concatenated parts
 *
 * This is based upon this discussion - https://github.com/aws/aws-cdk/issues/1424
 */
function generateResourceName(parts, maxLength) {
    const hashLength = 12;
    const maxPartLength = Math.floor((maxLength - hashLength) / parts.length);
    const sha256 = crypto.createHash("sha256");
    let finalName = '';
    parts.forEach((part) => {
        sha256.update(part);
        finalName += removeNonAlphanumeric(part.slice(0, maxPartLength));
    });
    const hash = sha256.digest("hex").slice(0, hashLength);
    finalName += hash;
    return finalName.toLowerCase();
}
exports.generateResourceName = generateResourceName;
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, '');
}
/**
 * Adds CFN NAG suppress rules to the CDK resource.
 * @param resource The CDK resource
 * @param rules The CFN NAG suppress rules
 */
function addCfnSuppressRules(resource, rules) {
    if (resource instanceof cdk.Resource) {
        resource = resource.node.defaultChild;
    }
    if (resource.cfnOptions.metadata?.cfn_nag?.rules_to_suppress) {
        resource.cfnOptions.metadata?.cfn_nag.rules_to_suppress.push(...rules);
    }
    else {
        resource.addMetadata('cfn_nag', {
            rules_to_suppress: rules
        });
    }
}
exports.addCfnSuppressRules = addCfnSuppressRules;
/**
 * Creates the props to be used to instantiate a CDK L2 construct within a Solutions Construct
 *
 * @param defaultProps The default props to be used by the construct
 * @param clientProps Optional properties passed in from the client in the props object
 * @param constructProps Optional properties required by the construct for the construct to work (override any other values)
 * @returns The properties to use - all values prioritized:
 *  1) constructProps value
 *  2) clientProps value
 *  3) defaultProps value
 */
function consolidateProps(defaultProps, clientProps, constructProps, concatArray = false) {
    let result = defaultProps;
    if (clientProps) {
        result = overrideProps(result, clientProps, concatArray);
    }
    if (constructProps) {
        result = overrideProps(result, constructProps, concatArray);
    }
    return result;
}
exports.consolidateProps = consolidateProps;
//# sourceMappingURL=data:application/json;base64,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