"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
test('Test override for buildUserPool', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const userpoolProps = {
        userPoolName: 'test',
        signInAliases: { username: false, email: true, phone: true }
    };
    defaults.buildUserPool(stack, userpoolProps);
    expect(stack).toHaveResource('AWS::Cognito::UserPool', {
        UsernameAttributes: [
            "email",
            "phone_number"
        ],
        UserPoolAddOns: {
            AdvancedSecurityMode: "ENFORCED"
        },
        UserPoolName: "test"
    });
});
test('Test override for buildUserPoolClient', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclientProps = {
        userPoolClientName: 'test',
        userPool: userpool
    };
    defaults.buildUserPoolClient(stack, userpool, userpoolclientProps);
    expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
        UserPoolId: {
            Ref: "CognitoUserPool53E37E69"
        },
        ClientName: "test"
    });
});
test('Test override for buildIdentityPool', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    defaults.buildIdentityPool(stack, userpool, userpoolclient, {
        allowUnauthenticatedIdentities: true
    });
    expect(stack).toHaveResource('AWS::Cognito::IdentityPool', {
        AllowUnauthenticatedIdentities: true,
        CognitoIdentityProviders: [
            {
                ClientId: {
                    Ref: "CognitoUserPoolClient5AB59AE4"
                },
                ProviderName: {
                    "Fn::GetAtt": [
                        "CognitoUserPool53E37E69",
                        "ProviderName"
                    ]
                },
                ServerSideTokenCheck: true
            }
        ]
    });
});
test('Test setupCognitoForElasticSearch', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    const identitypool = defaults.buildIdentityPool(stack, userpool, userpoolclient);
    defaults.setupCognitoForElasticSearch(stack, 'test-domain', {
        userpool,
        userpoolclient,
        identitypool
    });
    expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
        Domain: "test-domain"
    });
    expect(stack).toHaveResource('AWS::Cognito::IdentityPoolRoleAttachment', {
        IdentityPoolId: {
            Ref: "CognitoIdentityPool"
        },
        Roles: {
            authenticated: {
                "Fn::GetAtt": [
                    "CognitoAuthorizedRole14E74FE0",
                    "Arn"
                ]
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRoleWithWebIdentity",
                    Condition: {
                        "StringEquals": {
                            "cognito-identity.amazonaws.com:aud": {
                                Ref: "CognitoIdentityPool"
                            }
                        },
                        "ForAnyValue:StringLike": {
                            "cognito-identity.amazonaws.com:amr": "authenticated"
                        }
                    },
                    Effect: "Allow",
                    Principal: {
                        Federated: "cognito-identity.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "es:ESHttp*",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":es:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":domain/test-domain/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "CognitoAccessPolicy"
            }
        ]
    });
});
//# sourceMappingURL=data:application/json;base64,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