"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
const ec2 = require("aws-cdk-lib/aws-ec2");
function deployES(stack, domainName, clientDomainProps, lambdaRoleARN, vpc) {
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    const identitypool = defaults.buildIdentityPool(stack, userpool, userpoolclient);
    const cognitoAuthorizedRole = defaults.setupCognitoForElasticSearch(stack, 'test-domain', {
        userpool,
        userpoolclient,
        identitypool
    });
    if (lambdaRoleARN) {
        return defaults.buildElasticSearch(stack, {
            userpool,
            identitypool,
            cognitoAuthorizedRoleARN: cognitoAuthorizedRole.roleArn,
            serviceRoleARN: lambdaRoleARN,
            vpc,
            domainName,
            clientDomainProps
        });
    }
    else {
        return defaults.buildElasticSearch(stack, {
            userpool,
            identitypool,
            cognitoAuthorizedRoleARN: cognitoAuthorizedRole.roleArn,
            vpc,
            domainName,
            clientDomainProps
        });
    }
}
test('Test override SnapshotOptions for buildElasticSearch', () => {
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    deployES(stack, 'test-domain', {
        snapshotOptions: {
            automatedSnapshotStartHour: 5
        }
    });
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::GetAtt": [
                                "CognitoAuthorizedRole14E74FE0",
                                "Arn"
                            ]
                        }
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoKibanaConfigureRole62CCE76A",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        ElasticsearchVersion: "6.3",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 5
        }
    });
});
test('Test VPC with 1 AZ, Zone Awareness Disabled', () => {
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const vpc = defaults.getTestVpc(stack, false);
    deployES(stack, 'test-domain', {
        elasticsearchClusterConfig: {
            dedicatedMasterEnabled: true,
            dedicatedMasterCount: 3,
            instanceCount: 3,
            zoneAwarenessEnabled: false
        }
    }, undefined, vpc);
    expect(stack).toHaveResourceLike('AWS::Elasticsearch::Domain', {
        DomainName: "test-domain",
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessEnabled: false
        }
    });
});
test('Test VPC with 2 AZ, Zone Awareness Enabled', () => {
    // If no environment is specified, a VPC will use 2 AZs by default.
    // If an environment is specified, a VPC will use 3 AZs by default.
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {});
    const vpc = defaults.getTestVpc(stack, false);
    deployES(stack, 'test-domain', {}, undefined, vpc);
    expect(stack).toHaveResourceLike('AWS::Elasticsearch::Domain', {
        DomainName: "test-domain",
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 2,
            ZoneAwarenessEnabled: true
        }
    });
});
test('Test VPC with 3 AZ, Zone Awareness Enabled', () => {
    // If no environment is specified, a VPC will use 2 AZs by default.
    // If an environment is specified, a VPC will use 3 AZs by default.
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const vpc = defaults.getTestVpc(stack);
    deployES(stack, 'test-domain', {}, undefined, vpc);
    expect(stack).toHaveResourceLike('AWS::Elasticsearch::Domain', {
        DomainName: "test-domain",
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessEnabled: true
        }
    });
});
test('Test deployment with an existing private VPC', () => {
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const vpc = new ec2.Vpc(stack, 'existing-private-vpc-test', {
        natGateways: 1,
        subnetConfiguration: [
            {
                cidrMask: 24,
                name: 'application',
                subnetType: ec2.SubnetType.PRIVATE_WITH_NAT,
            },
            {
                cidrMask: 24,
                name: "public",
                subnetType: ec2.SubnetType.PUBLIC,
            }
        ]
    });
    deployES(stack, 'test-domain', {}, undefined, vpc);
    expect(stack).toHaveResourceLike('AWS::Elasticsearch::Domain', {
        DomainName: "test-domain",
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessEnabled: true
        }
    });
});
test('Test error thrown with no private subnet configurations', () => {
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: {
            subnetConfiguration: [
                {
                    cidrMask: 18,
                    name: "public",
                    subnetType: ec2.SubnetType.PUBLIC,
                }
            ]
        }
    });
    const app = () => {
        deployES(stack, 'test-domain', {}, undefined, vpc);
    };
    expect(app).toThrowError('Error - ElasticSearch Domains can only be deployed in Isolated or Private subnets');
});
test('Test override ES version for buildElasticSearch', () => {
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    deployES(stack, 'test-domain', {
        elasticsearchVersion: '7.0'
    });
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::GetAtt": [
                                "CognitoAuthorizedRole14E74FE0",
                                "Arn"
                            ]
                        }
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoKibanaConfigureRole62CCE76A",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        ElasticsearchVersion: "7.0",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 1
        }
    });
});
test('Test ES with lambdaRoleARN', () => {
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    deployES(stack, 'test-domain', {}, 'arn:aws:us-east-1:mylambdaRoleARN');
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: [
                            {
                                "Fn::GetAtt": [
                                    "CognitoAuthorizedRole14E74FE0",
                                    "Arn"
                                ]
                            },
                            "arn:aws:us-east-1:mylambdaRoleARN"
                        ]
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoKibanaConfigureRole62CCE76A",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        ElasticsearchVersion: "6.3",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 1
        }
    });
});
test('Count ES CW Alarms', () => {
    const stack = new aws_cdk_lib_1.Stack();
    deployES(stack, 'test-domain');
    const cwList = defaults.buildElasticSearchCWAlarms(stack);
    expect(cwList.length).toEqual(9);
});
//# sourceMappingURL=data:application/json;base64,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