"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk = require("aws-cdk-lib");
const kinesisanalytics = require("aws-cdk-lib/aws-kinesisanalytics");
const kinesisFirehose = require("aws-cdk-lib/aws-kinesisfirehose");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const defaults = require("../index");
const utils_1 = require("../lib/utils");
require("@aws-cdk/assert/jest");
test("test kinesisanalytics override inputProperty", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const inputProperty = {
        inputSchema: {
            recordColumns: [{ name: "x", sqlType: "y" }],
            recordFormat: { recordFormatType: "csv" },
        },
        namePrefix: "zzz",
    };
    const defaultProps = defaults.DefaultCfnApplicationProps;
    const inProps = {
        inputs: [inputProperty],
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new kinesisanalytics.CfnApplication(stack, "KinesisAnalytics", outProps);
    expect(stack).toHaveResource("AWS::KinesisAnalytics::Application", {
        Inputs: [
            {
                InputSchema: {
                    RecordColumns: [
                        {
                            Name: "x",
                            SqlType: "y",
                        },
                    ],
                    RecordFormat: {
                        RecordFormatType: "csv",
                    },
                },
                NamePrefix: "zzz",
            },
        ],
    });
});
test("Test default implementation", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const newFirehose = CreateFirehose(stack);
    const kinesisProps = {
        kinesisFirehose: newFirehose,
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        },
    };
    defaults.buildKinesisAnalyticsApp(stack, kinesisProps);
    expect(stack).toHaveResourceLike("AWS::KinesisAnalytics::Application", {
        Inputs: [{
                InputSchema: {
                    RecordColumns: [{
                            Name: 'ts',
                            SqlType: 'TIMESTAMP',
                            Mapping: '$.timestamp'
                        }, {
                            Name: 'trip_id',
                            SqlType: 'VARCHAR(64)',
                            Mapping: '$.trip_id'
                        }],
                    RecordFormat: {
                        RecordFormatType: 'JSON'
                    },
                    RecordEncoding: 'UTF-8'
                },
                NamePrefix: 'SOURCE_SQL_STREAM'
            }]
    });
});
// test('Test for customer overrides', {
// test('Check policy created', {
function CreateFirehose(stack) {
    // Creating the Firehose is kind of a big deal. FirehoseToS3 is not readily available here in core,
    // so this routine pretty much replicates it. If this function ceases to work correctly, look at
    // FirehoseToS3 and see if that changed.
    const destinationBucket = defaults.CreateScrapBucket(stack, {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true,
    });
    const kinesisFirehoseLogGroup = defaults.buildLogGroup(stack, "firehose-log-group", {});
    const cwLogStream = kinesisFirehoseLogGroup.addStream("firehose-log-stream");
    const firehoseRole = new iam.Role(stack, "test-role", {
        assumedBy: new iam.ServicePrincipal("firehose.amazonaws.com"),
    });
    // Setup the IAM policy for Kinesis Firehose
    const firehosePolicy = new iam.Policy(stack, "KinesisFirehosePolicy", {
        statements: [
            new iam.PolicyStatement({
                actions: [
                    "s3:AbortMultipartUpload",
                    "s3:GetBucketLocation",
                    "s3:GetObject",
                    "s3:ListBucket",
                    "s3:ListBucketMultipartUploads",
                    "s3:PutObject",
                ],
                resources: [
                    `${destinationBucket.bucketArn}`,
                    `${destinationBucket.bucketArn}/*`,
                ],
            }),
            new iam.PolicyStatement({
                actions: ["logs:PutLogEvents"],
                resources: [
                    `arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:${kinesisFirehoseLogGroup.logGroupName}:log-stream:${cwLogStream.logStreamName}`,
                ],
            }),
        ],
    });
    // Attach policy to role
    firehosePolicy.attachToRole(firehoseRole);
    const awsManagedKey = kms.Alias.fromAliasName(stack, "aws-managed-key", "alias/aws/s3");
    const defaultKinesisFirehoseProps = defaults.DefaultCfnDeliveryStreamProps(destinationBucket.bucketArn, firehoseRole.roleArn, kinesisFirehoseLogGroup.logGroupName, cwLogStream.logStreamName, awsManagedKey);
    destinationBucket.grantPut(firehoseRole);
    const firehose = new kinesisFirehose.CfnDeliveryStream(stack, "KinesisFirehose", defaultKinesisFirehoseProps);
    return firehose;
}
//# sourceMappingURL=data:application/json;base64,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