"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const sqs = require("aws-cdk-lib/aws-sqs");
const defaults = require("../");
require("@aws-cdk/assert/jest");
const sqs_helper_1 = require("../lib/sqs-helper");
// --------------------------------------------------------------
// Test deployment w/ imported encryption key
// --------------------------------------------------------------
test('Test deployment w/ imported encryption key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    defaults.buildQueue(stack, 'existing-queue', {
        queueProps: {
            queueName: 'existing-queue'
        },
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKey: defaults.buildEncryptionKey(stack)
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "existing-queue"
    });
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
});
// --------------------------------------------------------------
// Test deployment without imported encryption key
// --------------------------------------------------------------
test('Test deployment without imported encryption key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    defaults.buildQueue(stack, 'existing-queue', {
        queueProps: {
            queueName: 'existing-queue'
        }
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "existing-queue",
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
// --------------------------------------------------------------
// Test deployment w/ construct created encryption key
// --------------------------------------------------------------
test('Test deployment w/ construct created encryption key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const [queue, key] = defaults.buildQueue(stack, 'existing-queue', {
        queueProps: {
            queueName: 'existing-queue'
        },
        enableEncryptionWithCustomerManagedKey: true,
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "existing-queue"
    });
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
    expect(queue).toBeDefined();
    expect(key).toBeDefined();
});
test('Test DLQ when existing Queue Provided', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingQueue = new sqs.Queue(stack, 'test-queue');
    const buildDlqProps = {
        existingQueueObj: existingQueue,
    };
    const returnedQueueu = defaults.buildDeadLetterQueue(stack, buildDlqProps);
    expect(returnedQueueu).toBeUndefined();
    expect(stack).toCountResources("AWS::SQS::Queue", 1);
});
test('Test DLQ with all defaults', () => {
    const stack = new aws_cdk_lib_1.Stack();
    sqs_helper_1.buildDeadLetterQueue(stack, {});
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
test("Test DLQ with a provided properties", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testQueueName = "test-unique252";
    const returnedQueue = sqs_helper_1.buildDeadLetterQueue(stack, {
        deadLetterQueueProps: {
            queueName: testQueueName,
        },
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        QueueName: testQueueName,
    });
    expect(returnedQueue).toBeDefined();
});
test('Test DLQ with a provided maxReceiveCount', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testMaxReceiveCount = 31;
    const dlqInterface = sqs_helper_1.buildDeadLetterQueue(stack, {
        maxReceiveCount: testMaxReceiveCount
    });
    expect(dlqInterface?.maxReceiveCount).toEqual(testMaxReceiveCount);
});
test('Test returning an existing Queue', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testQueueName = 'existing-queue';
    const existingQueue = new sqs.Queue(stack, 'test-queue', {
        queueName: testQueueName
    });
    const [returnedQueue] = defaults.buildQueue(stack, 'newQueue', {
        existingQueueObj: existingQueue
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        QueueName: testQueueName,
    });
    expect(existingQueue.queueName).toEqual(returnedQueue.queueName);
});
test('Test creating a queue with a DLQ', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const dlqInterface = sqs_helper_1.buildDeadLetterQueue(stack, {});
    const [newQueue] = sqs_helper_1.buildQueue(stack, 'new-queue', {
        deadLetterQueue: dlqInterface
    });
    expect(stack).toCountResources("AWS::SQS::Queue", 2);
    expect(newQueue).toBeDefined();
    expect(newQueue.deadLetterQueue).toBeDefined();
});
test('Test creating a FIFO queue', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const [newFifoQueue] = sqs_helper_1.buildQueue(stack, 'new-queue', {
        queueProps: {
            fifo: true
        }
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        FifoQueue: true
    });
    expect(newFifoQueue.fifo).toBe(true);
});
//# sourceMappingURL=data:application/json;base64,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